<# 
.SYNOPSIS
  Updates the Broker Config file with end points required by the Linux VDA
   
.DESCRIPTION
  Updates the Broker Config file with end points required by the Linux VDA. A copy  
  of the original config file is taken and placed alongside the updated version, but with
  '.prelinux' appended to the file name.  
 
.PARAMETER Path
  Full path to the Broker Config file 
  
.EXAMPLE
  .\Update-BrokerServiceConfig.ps1"
  
  Updates the Broker Config file with end points required by the Linux VDA.
  The location of the Config file is determined dynamically.
#>

Param 
(
    [Parameter(Mandatory=$false)] [string] $path = (Get-ItemProperty -Path "HKLM:\SOFTWARE\Citrix\Broker\Service" -Name "ConfigTool").ConfigTool + ".config"
)


$ErrorActionPreference = "Stop"

Write-Host $("Loading '" + $path + "'.")

# Load config file.
$file = gi $path
$xml = [xml](gc $file)

Write-Host $("Saving backup to '" + $path + ".prelinux'.")

# Backup original config file.
Copy-Item -Path $path -Destination $($path + ".prelinux")

# List of endpoints to insert if necessary.
$xml_endpoints = $xml.ImportNode(([xml]"
<xml>
	<endpoint
		name='WsHttpBindingIQueryAgentEndpointLinux'
		binding='wsHttpBinding'
		bindingConfiguration='WorkerWsHttpBindingWithLargeMessageSizeLinux'
		behaviorConfiguration='WorkerEndpointBehaviour'
		contract='Citrix.Cds.Protocol.Worker.IQueryAgent' />
	<endpoint
		name='WsHttpBindingILaunchEndpointLinux'
		binding='wsHttpBinding'
		bindingConfiguration='WorkerWsHttpBindingLinux'
		behaviorConfiguration='WorkerEndpointBehaviour'
		contract='Citrix.Cds.Protocol.Worker.ILaunch' />
	<endpoint
		name='WsHttpBindingIDynamicDataQueryEndpointLinux'
		binding='wsHttpBinding'
		bindingConfiguration='WorkerWsHttpBindingLinux'
		behaviorConfiguration='WorkerEndpointBehaviour'
		contract='Citrix.Cds.Protocol.Worker.IDynamicDataQuery' />
	<endpoint
		name='WsHttpBindingIConfigurationEndpointLinux'
		binding='wsHttpBinding'
		bindingConfiguration='WorkerWsHttpBindingLinux'
		behaviorConfiguration='WorkerEndpointBehaviour'
		contract='Citrix.Cds.Protocol.Worker.IConfiguration' />
	<endpoint
		name='WsHttpBindingISessionManagerEndpointLinux'
		binding='wsHttpBinding'
		bindingConfiguration='WorkerWsHttpBindingLinux'
		behaviorConfiguration='WorkerEndpointBehaviour'
		contract='Citrix.Cds.Protocol.Worker.ISessionManager' />
</xml>
").xml, "true")
	
# List of bindings to insert if necessary.
$xml_bindings = $xml.ImportNode(([xml]"
<xml>
	<binding
		name='WorkerWsHttpBindingLinux'
		maxReceivedMessageSize='2000000'
		useDefaultWebProxy='false'
		bypassProxyOnLocal='true'>
		<readerQuotas maxArrayLength='2000000' maxStringContentLength='2000000' />
		<security mode='Message'>
			<message clientCredentialType='Windows' negotiateServiceCredential='false' establishSecurityContext='false' />
		</security>
	</binding>
	<binding
		name='WorkerWsHttpBindingWithLargeMessageSizeLinux'
		maxReceivedMessageSize='50000000'
		useDefaultWebProxy='false'
		bypassProxyOnLocal='true'>
		<readerQuotas maxArrayLength='2000000' maxStringContentLength='2000000' />
		<security mode='Message'>
			<message clientCredentialType='Windows' negotiateServiceCredential='false' establishSecurityContext='false' />
		</security>
	</binding>
</xml>
").xml, "true")

$changes_made = 0

Write-Host "Adding endpoints."

# Check if an endpoint already exists.
function EndpointExists($xml, $name)
{
	foreach ($endpoint in $xml.configuration.'system.serviceModel'.client.endpoint)	{
		if ($endpoint.name -eq $name) {
			return $true
		}
	}
	return $false
}

# Insert endpoints.
foreach ($endpoint in $xml_endpoints.endpoint) {
	if (!(EndpointExists -xml $xml -name $endpoint.name)) {
		$xml.configuration.'system.serviceModel'.client.AppendChild($endpoint)
		$changes_made += 1
	}
}

Write-Host "Adding bindings."

# Check if a binding already exists.
function BindingExists($xml, $name)
{
	foreach ($binding in $xml.configuration.'system.serviceModel'.bindings.wsHttpBinding.binding) {
		if ($binding.name -eq $name) {
			return $true
		}
	}
	return $false
}

# Insert bindings.
foreach ($binding in $xml_bindings.binding) {
	if (!(BindingExists -xml $xml -name $binding.name)) {
		$xml.configuration.'system.serviceModel'.bindings.wsHttpBinding.AppendChild($binding)
		$changes_made += 1
	}
}

Write-Host $("Saving '" + $path + "'.")

# Save updated config file.
$xml.Save($path)

if ($changes_made -gt 0)
{
	Write-Host "Restarting broker service."
	Restart-Service -name CitrixBrokerService
}

Write-Host "Done."

