﻿# General function for execution of the plugins. 
Function Invoke-CTXOEPlugin ([String]$PluginName, [System.Xml.XmlElement]$Params, [String]$Mode) {

    [String]$m_FunctionName = "Invoke-CTXOE$($PluginName)$($Mode.ToString())"

    # First test if the required plugin and function is available 
    If ($(Get-Command "$m_FunctionName" -Module CTXOEP_$($PluginName) -ErrorAction SilentlyContinue) -isnot [System.Management.Automation.FunctionInfo]) {
        Throw "Failed to load the required plugin or required function has not been implemented.
        Module: CTXOEP_$($PluginName)
        Function: $m_FunctionName"
    }

    If ($Params -isnot [Object]) {
        Throw "<params /> element is invalid for current entry. Review the definition XML file."
    }

    # Run the plugin with arguments
    & (Get-ChildItem "Function:$m_FunctionName") -Params $Params

}

# Test if registry key (Key + Name) has the required value (Value). Returns a dictionary with two values - [Bool]Result and [String]Details. 
Function Test-CTXOERegistryValue ([String]$Key, [String]$Name, [String]$Value) {
    # Initialize $return object and always assume failure
    [Hashtable]$Return = @{}
    $Return.Result = $False

    [Boolean]$m_RegKeyExists = Test-Path Registry::$($Key)

    # If value is CTXOE_DeleteKey, check if key itself exists. We need to process this first, because DeleteKey does not require 'Name' parameter and next section would fail
    If ($Value -eq "CTXOE_DeleteKey") {
        $Return.Result = $m_RegKeyExists -eq $False;
        If ($Return.Result -eq $True) {
            $Return.Details = "Registry key does not exist";
        } Else {
            $Return.Details = "Registry key exists";
        }
        Return $Return;
    }

    # If value name ('name') is not defined, Optimizer will only test if key exists. This is used in scenarios where you only need to create registry key, but without any values.
    If ($Name.Length -eq 0) {
        $Return.Result = $m_RegKeyExists;
        If ($Return.Result -eq $True) {
            $Return.Details = "Registry key exists";
        } Else {
            $Return.Details = "Registry key does not exist";
        }
        Return $Return;
    }

    # Retrieve the registry item
    $m_RegObject = Get-ItemProperty -Path Registry::$($Key) -Name $Name -ErrorAction SilentlyContinue;

    # If value is CTXOE_DeleteValue (or legacy CTXOE_NoValue), check if value exists. This code doesn't care what is the actual value data, only if it exists or not.
    If (($Value -eq "CTXOE_NoValue") -or ($Value -eq "CTXOE_DeleteValue")) {
        $Return.Result = $m_RegObject -isnot [System.Management.Automation.PSCustomObject];
        If ($Return.Result -eq $True) {
            $Return.Details = "Registry value does not exist";
        } Else {
            $Return.Details = "Registry value exists";
        }
        Return $Return;
    }

    # Return false if registry value was not found
    If ($m_RegObject -isnot [System.Management.Automation.PSCustomObject]) {
        $Return.Details = "Registry value does not exists"
        Return $Return;
    }

    # Registry value can be different object types, for example byte array or integer. The problem is that PowerShell does not properly compare some object types, for example you cannot compare two byte arrays. 
    # When we force $m_Value to always be [String], we have more predictable comparison operation. For example [String]$([Byte[]]@(1,1,1)) -eq $([Byte[]]@(1,1,1)) will work as expected, but $([Byte[]]@(1,1,1)) -eq $([Byte[]]@(1,1,1)) will not
    [string]$m_Value = $m_RegObject.$Name; 

    # If value is binary array, we need to convert it to string first
    If ($m_RegObject.$Name -is [System.Byte[]]) {
        [Byte[]]$Value = $Value.Split(",");
    }

    # If value type is DWORD or QWORD, registry object returns decimal value, while template can use both decimal and hexadecimal. If hexa is used in template, convert to decimal before comparison
    If ($Value -like "0x*") {
        # $m_RegObject.$Name can be different types (Int32, UInt32, Int64, UInt64...). Instead of handling multiple If...Else..., just use convert as to make sure that we are comparing apples to apples
        $Value = $Value -as $m_RegObject.$Name.GetType();
    }
    
    # $m_Value is always [String], $Value can be [String] or [Byte[]] array
    If ($m_value -ne $Value) {
        $Return.Details = "Different value ($m_value instead of $Value)"
    } Else {
        $Return.Result = $True
        $Return.Details = "Requested value $Value is configured"
    }
    Return $Return
}

# Set value of a specified registry key. Returns a dictionary with two values - [Bool]Result and [String]Details.
# There are few special values - CTXOE_DeleteKey (delete whole registry key if present), CTXOE_DeleteValue (delete registry value if present) and LEGACY CTXOE_NoValue (use CTXOE_DeleteValue instead, this was original name)
Function Set-CTXOERegistryValue ([String]$Key, [String]$Name, [String]$Value, [String]$ValueType) {
    
    [Hashtable]$Return = @{"Result" = $False; "Details" = "Internal error in function"}; 

    [Boolean]$m_RegKeyExists = Test-Path Registry::$Key;

    # First we need to handle scenario where whole key should be deleted
    If ($Value -eq "CTXOE_DeleteKey") {
        If ($m_RegKeyExists -eq $True) {
            Remove-Item -Path Registry::$Key -Force -ErrorAction SilentlyContinue | Out-Null
        }

        # Test if registry key exists or not. We need to pass value, so test function understands that we do NOT expect to find anything at target location
        [Hashtable]$Return = Test-CTXOERegistryValue -Key $Key -Value $Value;

        # When we delete whole registry key, we cannot restore it (unless we completely export it before, which is not supported yet)
        $Return.OriginalValue = "CTXOE_DeleteKey";

        Return $Return;

    }
    
    # If parent registry key does not exists, create it
    If ($m_RegKeyExists -eq $False) {
        New-Item Registry::$Key -Force | Out-Null;
        $Return.OriginalValue = "CTXOE_DeleteKey";
    }

    # If 'Name' is not defined, we need to only create a key and not any values
    If ($Name.Length -eq 0) {
        [Hashtable]$Return = Test-CTXOERegistryValue -Key $Key;
        # We need to re-assign this value again - $Return is overwritten by function Test-CTXOERegistryValue
        If ($m_RegKeyExists -eq $False) {
            $Return.OriginalValue = "CTXOE_DeleteKey";
        }
        Return $Return;
    }

    # Now change the value
    $m_ExistingValue = Get-ItemProperty -Path Registry::$Key -Name $Name -ErrorAction SilentlyContinue
    Try {
        If (($Value -eq "CTXOE_NoValue") -or ($Value -eq "CTXOE_DeleteValue")) {
            Remove-ItemProperty -Path Registry::$Key -Name $Name -Force -ErrorAction SilentlyContinue | Out-Null
        } Else {
            # If value type is binary, we need to convert string to byte array first. If this method is used directly with -Value argument (one line instead of two), it fails with error "You cannot call a method on a null-valued expression."
            If ($ValueType -eq "Binary") {
                [Byte[]]$m_ByteArray = $Value.Split(","); #[System.Text.Encoding]::Unicode.GetBytes($Value);
                New-ItemProperty -Path Registry::$Key -Name $Name -PropertyType $ValueType -Value $m_ByteArray -Force | Out-Null
            } Else {
                New-ItemProperty -Path Registry::$Key -Name $Name -PropertyType $ValueType -Value $Value -Force | Out-Null
            }
        }
    } Catch {
        $Return.Details = $($_.Exception.Message); 
        $Return.OriginalValue = "CTXOE_DeleteValue";
        Return $Return; 
    }

    # Re-run the validation test again
    [Hashtable]$Return = Test-CTXOERegistryValue -Key $Key -Name $Name -Value $Value
    
    # Save previous value for rollback functionality
    If ($m_RegKeyExists -eq $True) {
        If ($m_ExistingValue -is [Object]) {
            $Return.OriginalValue = $m_ExistingValue.$Name
        } Else {
            $Return.OriginalValue = "CTXOE_DeleteValue"
        }
    } Else {
        # We need to set this again, since $Return is overwritten by Test-CTXOERegistryValue function
        $Return.OriginalValue = "CTXOE_DeleteKey";
    }
    
    Return $Return
}
Function ConvertTo-CTXOERollbackElement ([Xml.XmlElement]$Element) {
    # Convert the element to XmlDocument. 
    [Xml]$m_TempXmlDocument = New-Object Xml.XmlDocument

    # Change the <params /> (or <executeparams /> to <rollbackparams />. 
    [Xml.XmlElement]$m_TempRootElement = $m_TempXmlDocument.CreateElement("rollbackparams")
    $m_TempRootElement.InnerXml = $Element.InnerXml
    $m_TempXmlDocument.AppendChild($m_TempRootElement) | Out-Null

    # Rollback is based on <value /> element. If this element doesn't exist already (in $Element), create an empty one. If we don't create this empty element, other functions that are trying to assign data to property .value will fail
    If ($m_TempRootElement.Item("value") -isnot [Xml.XmlElement]) {
        $m_TempRootElement.AppendChild($m_TempXmlDocument.CreateElement("value")) | Out-Null; 
    }

    # Return object
    Return $m_TempXmlDocument
}
Function New-CTXOEHistoryElement ([Xml.XmlElement]$Element, [Boolean]$SystemChanged, [DateTime]$StartTime, [Boolean]$Result, [String]$Details, [Xml.XmlDocument]$RollbackInstructions) {
    # Delete any previous <history /> from $Element
    If ($Element.History -is [Object]) {
        $Element.RemoveChild($Element.History) | Out-Null; 
    }

    # Get the parente XML document of the target element
    [Xml.XmlDocument]$SourceXML = $Element.OwnerDocument

    # Generate new temporary XML document. This is easiest way how to construct more complex XML structures with minimal performance impact. 
    [Xml]$m_TempXmlDoc = "<history><systemchanged>$([Int]$SystemChanged)</systemchanged><starttime>$($StartTime.ToString())</starttime><endtime>$([DateTime]::Now.ToString())</endtime><return><result>$([Int]$Result)</result><details>$Details</details></return></history>"

    # Import temporary XML document (standalone) as an XML element to our existing document
    $m_TempNode = $SourceXML.ImportNode($m_TempXmlDoc.DocumentElement, $true)
    $Element.AppendChild($m_TempNode) | Out-Null; 

    # If $RollbackInstructions is provided, save it as a <rollackparams /> element
    If ($RollbackInstructions -is [Object]) {
        $Element.Action.AppendChild($SourceXML.ImportNode($RollbackInstructions.DocumentElement, $true)) | Out-Null
    }
}

# Function to validate conditions. Returns hashtable object with two properties - Result (boolean) and Details. Result should be $True
Function Test-CTXOECondition([Xml.XmlElement]$Element) {

    [Hashtable]$m_Result = @{}; 

    # Always assume that script will fail
    $m_Result.Result = $False;
    $m_Result.Details = "No condition message defined"

    # $CTXOE_Condition is variable that should be returned by code. Because it is global, we want to reset it first. Do NOT assign $Null to variable - it will not delete it, just create variable with $null value
    Remove-Variable -Force -Name CTXOE_Condition -ErrorAction SilentlyContinue -Scope Global;
    Remove-Variable -Force -Name CTXOE_ConditionMessage -ErrorAction SilentlyContinue -Scope Global;

    # Check if condition has all required information (code is most important)
    If ($Element.conditioncode -isnot [object]) {
        $m_Result.Details = "Invalid or missing condition code. Condition cannot be processed";
        Return $m_Result;
    }

    # Execute code. This code should always return $Global:CTXOE_Condition variable (required) and $Global:CTXOE_ConditionMessage (optional)
    Try {
        Invoke-Expression -Command $Element.conditioncode;
    } Catch {
        $m_Result.Details = "Unexpected failure while processing condition: $($_.Exception.Message)";
        Return $m_Result;
    }
    

    # Validate output

    # Test if variable exists
    If (-not $(Test-Path Variable:Global:CTXOE_Condition)) {
        $m_Result.Details = "Required variable (CTXOE_Condition) NOT returned by condition. Condition cannot be processed";
        Return $m_Result;
    }

    # Test if variable is boolean
    If ($Global:CTXOE_Condition -isnot [Boolean]) {
        $m_Result.Details = "Required variable (CTXOE_Condition) is NOT boolean ($True or $False), but $($Global:CTXOE_Condition.GetType().FullName). Condition cannot be processed";
        Return $m_Result;
    }

    # Assign value to variable
    $m_Result.Result = $Global:CTXOE_Condition;

    # If condition failed and failed message is specified in XML section for condition, assign it
    If ($Element.conditionfailedmessage -is [Object] -and $m_Result.Result -eq $False) {
        $m_Result.Details = $Element.conditionfailedmessage;
    }

    # If $CTXOE_ConditionMessage is returned by code, use it to override the failed message
    If ((Test-Path Variable:Global:CTXOE_ConditionMessage)) {
        $m_Result.Details = $Global:CTXOE_ConditionMessage
    }

    # Return object
    Return $m_Result;

}