﻿# In older versions of PowerShell (before Windows 10), Get-Service did not return StartType. To support these older operating systems, we have to switch to WMI when we detect that class doesn't have this custom property. 
# We cannot use WMI call for all operating systems though, because it doesn't see some of the services (most notably kernel driver services like 'CSC')
[Boolean]$m_IsServicesLegacyMode = $(Get-Service | Select-Object -First 1).StartType -isnot [Object];

# Function to test is service with specified name exists. 
Function Test-CTXOEServicesServiceExist ([String]$Name) {
    Return $(Get-Service -Name $Name -ErrorAction SilentlyContinue) -is [System.ServiceProcess.ServiceController]
}

# Function to test if specified service has desider startup type configured. 
Function Test-CTXOEServicesStartupType ([String]$Name, [String]$StartType) {
    [Hashtable]$Return = @{}

    $Return.Result = $False

    # Test if service exists. 
    If ($(Test-CTXOEServicesServiceExist -Name $Name) -eq $False) {
        $Return.Details = "Service $($Params.Name) was not found"
        $Return.NotFound = $true;
        $Return.Result = $StartType -eq "Disabled";
        Return $Return
    } Else {
        If ($m_IsServicesLegacyMode) {
            # If legacy service mode is detected, Get-Service does not support start mode and we have to switch to WMI mode instead. However, WMI is not using 'Automatic', but shortened version 'Auto'. If 'Auto' is detected in output, we need to change it right variant ("Automatic")
            [String]$m_LegacyStartMode = (Get-WmiObject Win32_Service -filter "Name='$Name'").StartMode;
            If ($m_LegacyStartMode -eq "Auto") {
                [String]$m_CurrentStartMode = "Automatic";
            } Else {
                [String]$m_CurrentStartMode = $m_LegacyStartMode;
            }
        } Else {
            [String]$m_CurrentStartMode = Get-Service -Name $Name | Select-Object -ExpandProperty StartType;
        }

        $Return.Result = $m_CurrentStartMode -eq $StartType

        If ($m_CurrentStartMode -eq $StartType) {
            $Return.Details = "Startup type is $StartType"
        } Else {
            $Return.Details = "Desired startup type $($StartType), current type $($m_CurrentStartMode)"
        }
    }

    Return $Return
}

Function Invoke-CTXOEServicesExecuteInternal ([Xml.XmlElement]$Params, [Boolean]$RollbackSupported = $False) {
    [Hashtable]$m_Result = Test-CTXOEServicesStartupType -Name $Params.Name -StartType $Params.Value

    # If service is already configured, no action need to be taken.
    If ($m_Result.Result -eq $true) {
        $Global:CTXOE_Result = $m_Result.Result
        $Global:CTXOE_Details = $m_Result.Details
        Return
    }

    # If service was not found, return
    If ($m_Result.NotFound) {
        # If service should be disabled AND does not exists, that's OK. But if other value was configured than DISABLED, it's not OK. 
        $Global:CTXOE_Result = $Params.Value -eq "Disabled"
        $Global:CTXOE_Details = "Service $($Params.Name) was not found"
        Return
    } Else {
        # Save the current startup mode. 
        If ($m_IsServicesLegacyMode) {
            # If legacy service mode is detected, Get-Service does not support start mode and we have to switch to WMI mode instead. However, WMI is not using 'Automatic', but shortened version 'Auto'. If 'Auto' is detected in output, we need to change it right variant ("Automatic")
            [String]$m_PreviousLegacyStartMode = (Get-WmiObject Win32_Service -filter "Name='$($Params.Name)'").StartMode;
            If ($m_PreviousLegacyStartMode -eq "Auto") {
                [String]$m_PreviousStartMode = "Automatic";
            } Else {
                [String]$m_PreviousStartMode = $m_PreviousLegacyStartMode;
            }
        } Else {
            [String]$m_PreviousStartMode = Get-Service -Name $($Params.Name) | Select-Object -ExpandProperty StartType;
        }

        # Set service to required type, re-run the test for startup type
        Try {
            # There is a bug in Set-Service - it cannot properly set start type SYSTEM (used for kernel driver services). Instead of using cmdlet, we will change it directly in registry
            If ($Params.Value -eq "System") {
                New-ItemProperty -Path Registry::HKEY_LOCAL_MACHINE\SYSTEM\CurrentControlSet\Services\$($Params.Name) -Name Start -PropertyType DWORD -Value 1 -Force | Out-Null
            } Else {
                If ($Params.Value -eq "Auto") { # WMI is using 'Auto', Set-Service is using "Automatic"
                    Set-Service -Name $Params.Name -StartupType Automatic
                } Else {
                    Set-Service -Name $Params.Name -StartupType $Params.Value
                }
            }

            # Try to stop service. This is not critical action, only nice to have, so we don't need to check if it was successful or not (or even if service exists)
            If ($Params.Value -eq "Disabled") {Stop-Service $Params.Name -ErrorAction SilentlyContinue;}
        } Catch {
            $Global:CTXOE_Result = $False; 
            $Global:CTXOE_Details = "Failed to change service state with following error: $($_.Exception.Message)"; 
            Return
        }

        # Same the last known startup type for rollback instructions
        $Global:CTXOE_SystemChanged = $true;
        If ($RollbackSupported) {
            [Xml.XmlDocument]$m_RollbackElement = CTXOE\ConvertTo-CTXOERollbackElement -Element $Params
            $m_RollbackElement.rollbackparams.value = $m_PreviousStartMode.ToString();
            $Global:CTXOE_ChangeRollbackParams = $m_RollbackElement
        }

        [Hashtable]$m_Result = Test-CTXOEServicesStartupType -Name $Params.Name -StartType $Params.Value
        $Global:CTXOE_Result = $m_Result.Result
        $Global:CTXOE_Details = $m_Result.Details
        # There is a bug in Set-Service - it cannot properly set start type SYSTEM (used for kernel driver services). We change it in registry instead, but this is reflected after reboot, so we cannot use normal status reporting.
        If ($Params.Value -eq "System") {
            $Global:CTXOE_Result = $True;
            $Global:CTXOE_Details = "Startup type is System"
        }
    }

    Return
}

Function Invoke-CTXOEServicesAnalyze ([Xml.XmlElement]$Params) {
    [Hashtable]$m_Result = Test-CTXOEServicesStartupType -Name $Params.Name -StartType $Params.Value

    $Global:CTXOE_Result = $m_Result.Result
    $Global:CTXOE_Details = $m_Result.Details

    Return

}

Function Invoke-CTXOEServicesExecute ([Xml.XmlElement]$Params) {
    Invoke-CTXOEServicesExecuteInternal -Params $Params -RollbackSupported $True
}

Function Invoke-CTXOEServicesRollback ([Xml.XmlElement]$Params) {
    Invoke-CTXOEServicesExecuteInternal -Params $Params -RollbackSupported $False
}