#!/usr/bin/bash

################################################################################
#
# Citrix Virtual Apps & Desktops For Linux Script: Machine Creation Service
# Copyright (c) Citrix Systems, Inc. All Rights Reserved.
#

source /var/xdl/mcs/mcs_util.sh

fname=$(basename $0)
scriptName=${fname%.*}  # Script name without extension
logFile="/var/log/$scriptName.log"


function conf_hostname()
{
    log "Debug: Enter conf_hostname"

    hosts_file="/etc/hosts"
    hostname_file="/etc/hostname"
    content="127.0.0.1 ${NEW_HOSTNAME}.${realm} $NEW_HOSTNAME localhost localhost.localdomain localhost4 localhost4.localdomain4"

    sed -i "/^$content/d" "${hosts_file}"
    echo "$content" >> "$hosts_file"
    echo "$NEW_HOSTNAME" > "$hostname_file"
    hostname "$NEW_HOSTNAME"

    sysctl -w kernel.hostname="$NEW_HOSTNAME"  >> "$logFile" 2>&1
    log "Debug: Exit conf_hostname"
}

function conf_ntp()
{
    log "Debug: Enter conf_ntp"

    ntp_file_path="/etc/chrony.conf"
    remove_default_ntp_server $ntp_file_path
    echo "server $NTP_SERVER iburst" >> "$ntp_file_path"

    # sync time with ntp server
    log_echo "restart chronyd daemon"
    /usr/bin/systemctl restart chronyd.service  >> "$logFile" 2>&1

    sync_with_ntp_server $NTP_SERVER

    log "Debug: Exit conf_ntp"
}

function conf_winbind_krb5()
{
    log "Debug: Enter function conf_winbind_krb5"
    krbFile="/etc/krb5.conf"
    winbind_krbFile="/etc/xdl/mcs/winbind_krb5.conf.tmpl"

    # Customize /etc/krb5.conf
    REALM=$REALM AD_FQDN=$AD_FQDN realm=$realm envsubst < "$winbind_krbFile" > "$krbFile"

    log "Debug: Exit function conf_winbind_krb5"
}

function conf_sssd_krb5()
{
    log "Debug: Enter function conf_sssd_krb5"
    krbFile="/etc/krb5.conf"
    sssd_krbFile="/etc/xdl/mcs/sssd_krb5.conf.tmpl"

    # Customize /etc/krb5.conf
    REALM=$REALM AD_FQDN=$AD_FQDN realm=$realm envsubst < "$sssd_krbFile" > "$krbFile"

    log "Debug: Exit function conf_sssd_krb5"
}

function conf_winbind()
{
    log "Debug: Enter conf_winbind"

    # stop winbind daemon
    log "Stopping winbind service"
    /usr/bin/systemctl stop winbind  >> "$logFile" 2>&1

    # configure winbind authentication
    str="authselect select winbind with-mkhomedir --force"
    log "Execute command: \" ${str} \" "
    authselect select winbind with-mkhomedir --force
    [[ "$?" -ne "0" ]] && log_echo "failed to execute command: \" ${str} \""

    winbind_smbFile="/etc/xdl/mcs/winbind_smb.conf.tmpl"
    smbFile="/etc/samba/smb.conf"
    pamFile="/etc/security/pam_winbind.conf"

    # Customize /etc/samba/smb.conf
    WORKGROUP=$WORKGROUP REALM=$REALM envsubst < "$winbind_smbFile" > "$smbFile"
    `sed -i 's/.*winbind offline logon =.*$/winbind offline logon = no/g' "$smbFile"`

    conf_winbind_krb5             #Customize /etc/krb5.conf in case user execute the script multi times

    if [[ -f "$pamFile" ]]; then
        `sed -i 's/.*krb5_auth =.*$/krb5_auth = yes/g' "$pamFile"`
        `sed -i 's/.*krb5_ccache_type =.*$/krb5_ccache_type = FILE/g' "$pamFile"`
        `sed -i 's/.*mkhomedir =.*$/mkhomedir = yes/g' "$pamFile"`
    else
    echo "[global]
    krb5_auth = yes
    krb5_ccache_type = FILE
    mkhomedir = yes">"$pamFile"
    fi

    log "Debug: Exit conf_winbind"
}


function conf_sssd()
{
    log "Debug: Enter conf_sssd"

    # configure sssd authentication
    str="authselect select sssd with-mkhomedir --force"
    log "Execute command: \" ${str} \" "
    authselect select sssd with-mkhomedir --force
    [[ "$?" -ne "0" ]] && log_echo "failed to execute command: \" ${str} \""


    sssd_smbFile="/var/xdl/mcs/sssd_smb.conf.tmpl"
    smbFile="/etc/samba/smb.conf"

    # Customize /etc/samba/smb.conf
    WORKGROUP=$WORKGROUP REALM=$REALM envsubst < "$sssd_smbFile" > "$smbFile"
    `sed -i 's/.*winbind offline logon =.*$/winbind offline logon = no/g' "$smbFile"`

    conf_sssd_krb5             #Customize /etc/krb5.conf in case user execute the script multi times

    # Customize /etc/sssd/sssd.conf
    conf_sssd_sssd

    log "Debug: Exit conf_sssd"
}

function conf_sssd_sssd()
{
    log "Debug: Enter function conf_sssd_sssd"
    sssdFile="/etc/sssd/sssd.conf"
    sssdFile_tmpl="/etc/xdl/mcs/sssd.conf.tmpl"

    # Customize /etc/sssd/sssd.conf
    AD_FQDN=$AD_FQDN realm=$realm envsubst < "$sssdFile_tmpl" > "$sssdFile"

    # Set the file ownership and permissions on sssd.conf
    chown root:root "$sssdFile"
    chmod 0600 "$sssdFile"
    restorecon  "$sssdFile"

    #enable sssd service
    str="sudo systemctl enable sssd"
    log "Execute command: \" ${str} \""
    sudo /usr/bin/systemctl enable sssd
    [[ "$?" -ne "0" ]] && log_echo "failed to execute command: \" ${str} \""

    #start sssd service to create secret file
#    /usr/bin/systemctl start sssd  2>&1 >> "$logFile"
#    sleep 2

    log "Debug: Exit conf_sssd_sssd"
}


## dbus needs to know the auth method.
function conf_winbind_nsswitch()
{
    log "Debug: Enter conf_winbind_nsswitch"

    # Customize /etc/nsswitch.conf
    nsswitchFile="/etc/nsswitch.conf"
    passwdLine="$(cat $nsswitchFile |grep passwd |grep -v sss | grep winbind 2>&1)"
    shadowLine="$(cat $nsswitchFile |grep shadow |grep -v sss | grep winbind 2>&1)"
    groupLine="$(cat $nsswitchFile |grep group | grep -v netgroup | grep -v sss | grep winbind 2>&1)"
    netgroupLine="$(cat $nsswitchFile |grep netgroup |grep sss 2>&1)"
    serviceLine="$(cat $nsswitchFile |grep services |grep sss 2>&1)"
    automountLine="$(cat $nsswitchFile |grep automount |grep sss 2>&1)"
    [[ -z "$passwdLine" ]] && `sed -i 's/^passwd:.*$/passwd:     files winbind/g' "$nsswitchFile"`
    [[ -z "$shadowLine" ]] && `sed -i 's/^shadow:.*$/shadow:     files winbind/g' "$nsswitchFile"`
    [[ -z "$groupLine" ]] && `sed -i 's/^group:.*$/group:      files winbind/g' "$nsswitchFile"`
    [[ ! -z "$serviceLine" ]] && `sed -i 's/^services:.*$/services:   files/g' "$nsswitchFile"`
    [[ ! -z "$netgroupLine" ]] && `sed -i 's/^netgroup:.*$/netgroup:   files/g' "$nsswitchFile"`
    [[ ! -z "$automountLine" ]] && `sed -i 's/^automount:.*$/automount:  files/g' "$nsswitchFile"`

    log "Debug: Exit conf_winbindnsswitch"
}

function conf_sssd_nsswitch()
{
    log "Debug: Enter conf_sssd_nsswitch"

    # Customize /etc/nsswitch.conf
    nsswitchFile="/etc/nsswitch.conf"
    passwdLine="$(cat $nsswitchFile |grep passwd |grep sss 2>&1)"
    shadowLine="$(cat $nsswitchFile |grep shadow |grep sss 2>&1)"
    groupLine="$(cat $nsswitchFile |grep group | grep -v netgroup | grep sss 2>&1)"
    netgroupLine="$(cat $nsswitchFile |grep netgroup |grep sss 2>&1)"
    serviceLine="$(cat $nsswitchFile |grep services |grep sss 2>&1)"
    automountLine="$(cat $nsswitchFile |grep automount |grep sss 2>&1)"
    [[ -z "$passwdLine" ]] && `sed -i 's/^passwd:.*$/passwd:     files sss/g' "$nsswitchFile"`
    [[ -z "$shadowLine" ]] && `sed -i 's/^shadow:.*$/shadow:     files sss/g' "$nsswitchFile"`
    [[ -z "$groupLine" ]] && `sed -i 's/^group:.*$/group:      files sss/g' "$nsswitchFile"`
    [[ -z "$serviceLine" ]] && `sed -i 's/^services:.*$/services:   files sss/g' "$nsswitchFile"`
    [[ -z "$netgroupLine" ]] && `sed -i 's/^netgroup:.*$/netgroup:   files sss/g' "$nsswitchFile"`
    [[ -z "$automountLine" ]] && `sed -i 's/^automount:.*$/automount:  files sss/g' "$nsswitchFile"`

    log "Debug: Exit conf_sssd_nsswitch"
}

function conf_centrify_nss_pam()
{
    log "Debug: Enter conf_centrify_nss_pam"

    # Customize /etc/nsswitch.conf
    nsswitchFile="/etc/nsswitch.conf"
    passwdLine="$(cat $nsswitchFile |grep passwd |grep centrifydc 2>&1)"
    shadowLine="$(cat $nsswitchFile |grep shadow |grep centrifydc 2>&1)"
    groupLine="$(cat $nsswitchFile |grep group | grep -v netgroup | grep centrifydc 2>&1)"
    [[ -z "$passwdLine" ]] && `sed -i 's/^passwd:.*$/passwd: centrifydc sss files systemd/g' "$nsswitchFile"`
    [[ -z "$shadowLine" ]] && `sed -i 's/^shadow:.*$/shadow: centrifydc files sss/g' "$nsswitchFile"`
    [[ -z "$groupLine" ]] && `sed -i 's/^group:.*$/group: centrifydc sss files systemd/g' "$nsswitchFile"`

    # Customize /etc/pam.d
    system_auth_pam_pre="/etc/pam.d/system-auth.pre"
    system_auth_pam="/etc/pam.d/system-auth"
    password_auth_pam_pre="/etc/pam.d/password-auth.pre"
    password_auth_pam="/etc/pam.d/password-auth"
    if [ -f $system_auth_pam_pre ]; then
        cp -f $system_auth_pam_pre $system_auth_pam
    else
        cp $system_auth_pam $system_auth_pam_pre
    fi
    if [ -f $password_auth_pam_pre ]; then
        cp -f $password_auth_pam_pre $password_auth_pam
    else
        cp $password_auth_pam $password_auth_pam_pre
    fi

    sed -i '1i auth    sufficient      pam_centrifydc.so\
auth    requisite       pam_centrifydc.so deny\
account sufficient      pam_centrifydc.so\
account requisite       pam_centrifydc.so deny\
session required        pam_centrifydc.so homedir\
password        sufficient      pam_centrifydc.so try_first_pass\
password        requisite       pam_centrifydc.so deny' "$system_auth_pam"

    sed -i '1i auth    sufficient      pam_centrifydc.so\
auth    requisite       pam_centrifydc.so deny\
account sufficient      pam_centrifydc.so\
account requisite       pam_centrifydc.so deny\
session required        pam_centrifydc.so homedir\
password        sufficient      pam_centrifydc.so try_first_pass\
password        requisite       pam_centrifydc.so deny' "$password_auth_pam"

    system_auth_authselect="/etc/authselect/system-auth"
    password_auth_authselect="/etc/authselect/password-auth"
    cp $system_auth_pam $system_auth_authselect
    cp $password_auth_pam $password_auth_authselect
    rm -rf $system_auth_pam
    rm -rf $password_auth_pam

    ln -s $system_auth_authselect $system_auth_pam
    ln -s $password_auth_authselect $password_auth_pam

    # Customize /etc/pam.d/su
    su_pam="/etc/pam.d/su"
    pam_centrifydc_auth_line="$(cat $su_pam |grep auth |grep pam_centrifydc| grep enable_dzpamgate 2>&1)"
    [[ -z "$pam_centrifydc_auth_line" ]] && `sed -i '/^auth.*sufficient.*pam_rootok.so$/a auth    sufficient      pam_centrifydc.so enable_dzpamgate' "$su_pam"`

    log "Debug: Exit conf_centrify_nss_pam"
}

function conf_centrify_krb5_smb()
{
    log "Debug: Enter conf_centrify_krb5_smb"

    # Configure adclient
    centrifydc_conf="/etc/centrifydc/centrifydc.conf"
    krb5_autoedit="$(cat $centrifydc_conf |grep adclient.krb5.autoedit | grep -v "^#" 2>&1)"
    if [ -z "$krb5_autoedit" ]; then
        sed -i '1i adclient.krb5.autoedit: false' "$centrifydc_conf"
    else
        sed -i 's/^adclient.krb5.autoedit:.*$/adclient.krb5.autoedit: false/g' "$centrifydc_conf"
    fi

    krbFile="/etc/krb5.conf"
    centrify_krbFile="/etc/xdl/mcs/centrify_krb5.conf.tmpl"

    # Customize /etc/krb5.conf
    REALM=$REALM AD_FQDN=$AD_FQDN realm=$realm NEW_HOSTNAME=$new_hostname envsubst < "$centrify_krbFile" > "$krbFile"

    # Customize  /var/centrify/samba/adbindproxy.pl.rsp
    adbindproxyFile="/var/centrify/samba/adbindproxy.pl.rsp"
    centrify_adbindproxyFile="/etc/xdl/mcs/adbindproxy.pl.rsp.tmpl"
    adbindproxy_version="$(/usr/share/centrifydc/sbin/adbindd -v | awk {'print $NF'} |tr -d ')')"
    [ ! -d "/var/centrify/samba" ] && mkdir -p /var/centrify/samba
    ADBINDPROXY_VERSION=$adbindproxy_version  envsubst < "$centrify_adbindproxyFile" > "$adbindproxyFile"

    # Configure samba
    smbFile="/etc/samba/smb.conf"
    centrify_smbFile="/etc/xdl/mcs/centrify_smb.conf.tmpl"

    adclient_samba_sync="$(cat $centrifydc_conf |grep adclient.samba.sync | grep -v "^#" 2>&1)"
    if [ -z "$adclient_samba_sync" ]; then
        sed -i '1i adclient.samba.sync: true' "$centrifydc_conf"
    else
        sed -i 's/^adclient.samba.sync:.*$/adclient.samba.sync: true/g' "$centrifydc_conf"
    fi
    samba_base_path="$(cat $centrifydc_conf |grep samba.base.path | grep -v "^#" 2>&1)"
    if [ -z "$samba_base_path" ]; then
        sed -i '1i samba.base.path: /opt/centrify/samba' "$centrifydc_conf"
    else
        sed -i 's/^samba.base.path:.*$/samba.base.path: /opt/centrify/samba/g' "$centrifydc_conf"
    fi
    samba_interop_uselibtdb="$(cat $centrifydc_conf |grep samba.interop.uselibtdb | grep -v "^#" 2>&1)"
    if [ -z "$samba_interop_uselibtdb" ]; then
        sed -i '1i samba.interop.uselibtdb: false' "$centrifydc_conf"
    else
        sed -i 's/^samba.interop.uselibtdb:.*$/samba.interop.uselibtdb: false/g' "$centrifydc_conf"
    fi

    REALM=$REALM WORKGROUP=$WORKGROUP new_hostname=$new_hostname envsubst < "$centrify_smbFile" > "$smbFile"

    log "Debug: Exit conf_centrify_krb5_smb"
}

# this is called by ad_join service
function join_domain_setup_vda()
{
    log "Debug: Enter join_domain_setup_vda"
    conf_ntp
    if [ "$Use_Existing_Configurations_Of_Current_VDA" == "Y" ]; then
        log "Keep running configuration files"
        if [ "$AD_INTEGRATION" == "winbind" ]; then
                join_domain_samba
                restart_service winbind /usr/bin/systemctl
                write_EncryptionTypes_attribute
        elif [ "$AD_INTEGRATION" == "sssd" ]; then
                join_domain_samba
                restart_service sssd /usr/bin/systemctl
                write_EncryptionTypes_attribute
        elif [ "$AD_INTEGRATION" == "pbis" ]; then
                join_domain_pbis
        elif [ "$AD_INTEGRATION" == "centrify" ]; then
            join_domain_centrify
            restart_service postgresql /usr/bin/systemctl
        fi
    else
        if [ "$AD_INTEGRATION" == "winbind" ]; then
            conf_winbind
            join_domain_samba
            restart_service winbind /usr/bin/systemctl
            write_EncryptionTypes_attribute
        elif [ "$AD_INTEGRATION" == "sssd" ]; then
            conf_sssd
            join_domain_samba
            restart_service sssd /usr/bin/systemctl
            write_EncryptionTypes_attribute
        elif [ "$AD_INTEGRATION" == "pbis" ]; then
            join_domain_pbis
        elif [ "$AD_INTEGRATION" == "centrify" ]; then
            join_domain_centrify
        fi
    fi
    setup_vda
    log "Debug: Exit join_domain_setup_vda"
}

# this is called before dbus service start
function config_file()
{
    log "Debug: Enter config_file"
    # If its NDJ mode, this config is not needed
    if [ "$NON_DOMAIN_JOINED" == "Y" ]; then
        log "NDJ mode, config_file return"
        return 0
    fi
    # If DJ mode, continue the configuration
    conf_hostname
    if [ "$AD_INTEGRATION" == "winbind" ]; then
        conf_winbind_nsswitch
    elif [ "$AD_INTEGRATION" == "sssd" ]; then
        conf_sssd_nsswitch
    elif [ "$AD_INTEGRATION" == "centrify" ]; then
        if [ "$Use_Existing_Configurations_Of_Current_VDA" != "Y" ]; then
            conf_centrify_nss_pam
        fi
        conf_centrify_krb5_smb
        /usr/bin/systemctl enable centrifydc >> "$logFile" 2>&1
    fi
    log "Debug: Exit config_file"
}

if [ "$1" == "--update-key" ] ; then
    update_private_key_timestamp
    exit 0
fi

if [ "$1" == "--backup-keydata" ] ; then
    backup_keydata_json_file
    exit 0
fi

check_user_and_vda_configured
check_ntfs_3g
read_id_disk

if [ "$1" == "--config" ] ; then
    config_file
    exit 0
elif [ "$1" == "--setup" ] ; then
     # If its NDJ mode, run NDJ setup
    if [ "$NON_DOMAIN_JOINED" == "Y" ]; then
        non_domain_joined_setup /bin/systemctl
    else
        join_domain_setup_vda
    fi

    # Set flag to indicate MCS setup is configured.
    set_mcs_setting_flag

    exit 0
else
    exit 1
fi
