#!/bin/bash
################################################################################
#
# Citrix Virtual Apps & Desktops For Linux Setup Script
# ctxsetup.sh
#
# This script must be run at least once, but can be run multiple times
# if it is necessary to change settings.
#
# Any changes to the system made by this script must be reverted by the
# cleanup script.
#

source /var/xdl/configure_common.sh
source /var/xdl/configure_platform.sh
source /var/xdl/configure_utilities.sh

function usage {
    get_str SHELL_SETUP_USAGE $0 "$short_name" "$product_name"
}

if [ $EUID -ne 0 ]; then
    get_str SHELL_SETUP_MUST_ROOT
    exit 1
fi

if [ "$1" == "--help" ] || [ "$1" == "-h" ]; then
    get_str SHELL_SETUP_SETUP_SCRIPT "$product_name"
    echo
    usage
    exit 0
elif [ "$1" == "--silent" ]; then
    get_str SHELL_SETUP_WELCOME "$product_name" "$linux_virtual_desktop"
    echo
    get_str SHELL_SETUP_RUN_SILENT
    echo
    silent="y"
elif [ ! -z "$1" ]; then
    get_str SHELL_SETUP_WELCOME "$product_name" "$linux_virtual_desktop"
    echo
    get_str SHELL_SETUP_UNKNOWN_OPT $1
    echo
    get_str SHELL_SETUP_TYPE_HELP $0
    exit 1
else
    silent="n"
    get_str SHELL_SETUP_WELCOME "$product_name" "$linux_virtual_desktop"
    echo
fi
failed="n"

get_str SHELL_SETUP_GATHER_INFO
echo

if [ -f /var/xdl/.setup.updated ]; then
    rm -f /var/xdl/.setup.updated
    if [ -f /var/xdl/.setup.postgres ] &&
       [ -f /var/xdl/.setup.confdb ] &&
       [ -f /var/xdl/.setup.confdb.populate ]; then
        get_str SHELL_SETUP_PACKAGE_UPGRADE
        echo
        cmdlog_start
        service_stop
        service_register 
        ldconfig_update
        postgresql_start
        # For upgrade from non-dotnet vda to dotnet vda, the password should not be changed
        if [[ ! -f /etc/xdl/ctx-jproxy.conf && -f /etc/xdl/ctx-vda.conf ]]; then
            confdb_update_conf_file
        fi        
        confdb_update_database
        firewall_update_rules
        # there are 2 scenarios:
        # 1. upgrade from an env that does not have dotnet runtime,
        # 2. upgrade from an env that has dotnet runtime.
        opt_dotnet_runtime_path=`/opt/Citrix/VDA/bin/ctxreg read -k "HKLM\Software\Citrix\VirtualDesktopAgent" -v "DotNetRuntimePath" 2>/dev/null`
        if [[ -z $opt_dotnet_runtime_path ]]; then
            get_valve CTX_XDL_DOTNET_RUNTIME_PATH \
            "$(get_str SHELL_SETUP_DOTNET_RUNTIME "/usr/bin")" \
            "/usr/bin" \
            dotnet_rutnime_path_validate
            opt_dotnet_runtime_path="$result"
        fi
        # write dotnet runtime path to database
        set_dotnet_runtime_path $opt_dotnet_runtime_path
        selinux_environment_configure
        desktop_environment_configure
        create_anonymous_group
        audio_remove_config
        hdx_3d_pro_configure
        set_installation_type "upgrade"
        service_start
        cmdlog_stop
        exit 0
    fi
fi

get_valve CTX_XDL_DOTNET_RUNTIME_PATH \
        "$(get_str SHELL_SETUP_DOTNET_RUNTIME "/usr/bin")" \
        "/usr/bin" \
        dotnet_rutnime_path_validate
opt_dotnet_runtime_path="$result"

get_valve CTX_XDL_DESKTOP_ENVIRONMENT \
        "$(get_str SHELL_SETUP_DESKTOP_ENVIRONMENT "$virtual_delivery_agent")" \
        "gnome" \
        desktop_environment_validate
opt_desktop_environment="$result"

get_ynd CTX_XDL_SUPPORT_DDC_AS_CNAME \
        "$(get_str SHELL_SETUP_DDC_VIA_CNAME "$virtual_delivery_agent")" \
        "n"
opt_support_ddc_as_cname="$result"

get_valv CTX_XDL_DDC_LIST \
        "$(get_str SHELL_SETUP_PROVIDE_FQDN "$virtual_delivery_agent")" \
        ddc_list_validate
opt_ddc_list="$result"

get_numd_port CTX_XDL_VDA_PORT \
        "$(get_str SHELL_SETUP_DDC_ADDRESS "$virtual_delivery_agent")" \
        "80"
opt_vda_port="$result"

get_numd_port CTX_XDL_TELEMETRY_SOCKET_PORT \
        "$(get_str SHELL_SETUP_TELEMETRY_SOCKET_ADDRESS "$virtual_delivery_agent")" \
        "7503"
opt_telemetry_socket_port="$result"

get_numd_port CTX_XDL_TELEMETRY_PORT \
        "$(get_str SHELL_SETUP_TELEMETRY_ADDRESS "$virtual_delivery_agent")" \
        "7502"
opt_telemetry_port="$result"

get_ynd CTX_XDL_REGISTER_SERVICE \
        "$(get_str SHELL_SETUP_REG_SERVICES "$linux_virtual_desktop")" \
        "y"
opt_register_service="$result"

get_ynd CTX_XDL_ADD_FIREWALL_RULES \
        "$(get_str SHELL_SETUP_CFG_FIREWALL "$linux_virtual_desktop")" \
        "y"
opt_add_firewall_rules="$result"

get_numd CTX_XDL_AD_INTEGRATION \
        "$(get_str SHELL_SETUP_WHAT_AD "$virtual_delivery_agent")" \
        "5" \
        "1" \
        "Winbind" \
        "Quest" \
        "Centrify" \
        "SSSD" \
        "PBIS"
opt_ad_integration="$result"

get_ynd CTX_XDL_HDX_3D_PRO \
        "$(get_str SHELL_SETUP_HDX_3D_PRO "$linux_virtual_desktop" "$virtual_delivery_agent")" \
        "n"
opt_hdx_3d_pro="$result"

if [ "$opt_hdx_3d_pro" == "y" ]; then
    opt_vdi_mode="y"
else
    get_ynd CTX_XDL_VDI_MODE \
            "$(get_str SHELL_SETUP_VDI_MODE "$linux_virtual_desktop")" \
            "n"
    opt_vdi_mode="$result"
fi

get_vale CTX_XDL_SITE_NAME \
        "$(get_str SHELL_SETUP_SITE_NAME "$virtual_delivery_agent")"
opt_site_name="$result"

get_vale CTX_XDL_LDAP_LIST \
        "$(get_str SHELL_SETUP_LDAP_LIST "$virtual_delivery_agent")"
opt_ldap_list="$result"

get_vale CTX_XDL_SEARCH_BASE \
        "$(get_str SHELL_SETUP_SEARCH_BASE "$virtual_delivery_agent")"
opt_search_base="$result"

get_vale CTX_XDL_FAS_LIST \
         "$(get_str SHELL_SETUP_FAS_LIST "$virtual_delivery_agent")"
opt_fas_list="$result"

#get_ynd CTX_XDL_SMART_CARD \
#        "$(get_str SHELL_SETUP_SMART_CARD "$linux_virtual_desktop")" \
#        "n"
#opt_smart_card="$result"

get_ynd CTX_XDL_START_SERVICE \
        "$(get_str SHELL_SETUP_TO_START_SERVICES "$linux_virtual_desktop")" \
        "y"
opt_start_service="$result"

if [ "$failed" == "y" ]; then
    get_str SHELL_SETUP_NOT_ALL_INFOR "$product_name"
    echo
    get_str SHELL_SETUP_TYPE_HELP $0
    exit 1
fi

cmdlog_start

service_stop

ldconfig_update

postgresql_start

if [ ! -f /var/xdl/.setup.postgres ]; then
    postgresql_update_config
    postgresql_restart
    touch /var/xdl/.setup.postgres
fi

# sleep to wait for postgresql ready.
sleep 5

if [ ! -f /var/xdl/.setup.confdb ]; then
    confdb_generate_passwd
    confdb_create_database
    touch /var/xdl/.setup.confdb
fi

if [ ! -f /var/xdl/.setup.confdb.populate ]; then
    confdb_populate_database
    touch /var/xdl/.setup.confdb.populate
fi

ddc_cname_lookup_allow "$opt_support_ddc_as_cname"

ddc_list_update "$opt_ddc_list"

# Remove any existing XDL firewall rules before setting the VDA port.
# This is to prevent an existing VDA port remaining open.
firewall_remove_rules

vda_port_update "$opt_vda_port"

telemetry_socket_port_update "$opt_telemetry_socket_port"

telemetry_port_update "$opt_telemetry_port"

if [ "$opt_register_service" == "y" ]; then
    service_register
else
    service_unregister
fi

if [ "$opt_add_firewall_rules" == "y" ]; then
    firewall_add_rules $opt_vda_port $opt_telemetry_port $opt_telemetry_socket_port
else
    firewall_remove_rules
fi
firewall_try_restart

if [ "$opt_ad_integration" == "1" ]; then
    ad_integration_use_winbind
fi
if [ "$opt_ad_integration" == "2" ]; then
    ad_integration_use_quest
fi
if [ "$opt_ad_integration" == "3" ]; then
    ad_integration_use_centrify
fi
if [ "$opt_ad_integration" == "4" ]; then
    ad_integration_use_sssd
fi
if [ "$opt_ad_integration" == "5" ]; then
    ad_integration_use_pbis
fi

selinux_environment_configure

if [[ "$opt_desktop_environment" == "<none>" || "$opt_desktop_environment" == "gnome" ]]; then
    mate_session_restore
    set_desktop_environment "gnome"
elif [ "$opt_desktop_environment" == "mate" ]; then
    if [ ! -f "/usr/share/xsessions/mate.desktop" ]; then
        echo -e "Mate desktop environment not installed\n" >> $configure_log
        set_desktop_environment "gnome"
    else
        mate_session_configure
        set_desktop_environment "mate"
    fi
fi

desktop_environment_configure

create_anonymous_group

hdx_3d_pro_configure

set_installation_type "install"

if [ "$opt_hdx_3d_pro" == "y" ]; then
    set_hdx_3d_pro 0
else
    set_hdx_3d_pro 1
fi

if [ "$opt_vdi_mode" == "y" ]; then
    set_vdi_mode 0
    set_single_session_mode 1
else
    set_vdi_mode 1
    set_single_session_mode 0
fi

if [ ! "$opt_site_name" == "<none>" ]; then
    ad_site_name "$opt_site_name"
fi

if [ ! "$opt_ldap_list" == "<none>" ]; then
    ldap_server_list "$opt_ldap_list"
fi

if [ ! "$opt_search_base" == "<none>" ]; then
    ldap_search_base "$opt_search_base"
fi

if [ ! "$opt_fas_list" == "<none>" ]; then
    fas_server_list "$opt_fas_list"
fi

# write dotnet runtime path to database
set_dotnet_runtime_path $opt_dotnet_runtime_path

#if [ "$opt_smart_card" == "y" ]; then
#    set_smart_card 1
#else
#    set_smart_card 0
#fi

if [ "$opt_start_service" == "y" ]; then
    service_start
fi

cmdlog_stop

