#!/bin/bash

#******************************************************************************
# ctxwcamcfg.sh
#
# Copyright 1998-2021, 2021 Citrix Systems, Inc. All Rights Reserved.
# 
# Authors:
#   Long Xu, 27 April. 2021
#
# Description:
#   This script is used to manager (dkms) v4l2loopback ko for HDX Optimized Webcam, please run with superuser privileges.
# Package installed:
#   dkms, kernel-devel
# Note:
#   Run this script at least once
#*******************************************************************************

source /var/xdl/configure_common.sh
source /var/xdl/configure_platform.sh
source /var/xdl/configure_utilities.sh

# https://github.com/umlaeute/v4l2loopback/tree/81b8df79107d1fbf392fdcbaa051bd227a9c94c1
commit="81b8df79107d1fbf392fdcbaa051bd227a9c94c1"
version="1.${commit}"
ctxlib64="/opt/Citrix/VDA/lib64"
kernelsrc=""
kernelmodule=""
osType="" # UB_DB_PD/RH_CT/SS

function checkUser()
{
    if [ "$(id -u)" != 0 ]; then
        exit 1
    fi
}

function checkOS()
{
    # get version
    getOSInfo

    case "$osPlatform" in
        *ubuntu* | *debian* | *pardus* )
            osType="UB_DB_PD"
            ;;
        *centos* | *red* | *amazon* | *rocky* )
            osType="RH_CT"
            ;;
        *suse*)
            osType="SS"
            ;;
        *)
            get_str CTXFASCFG_UNSUPPORT_PLATFORM_PRECHECK "${osPlatform}"
            exit 1
            ;;
    esac
}

function confOS()
{
    # get version
    getOSInfo

    case "$osPlatform" in
        *ubuntu* | *debian* | *pardus* )
            kernelsrc=/usr/src/linux-headers-$(uname -r)
            kernelmodule=/lib/modules/$(uname -r)

            apt-get -y install linux-headers-$(uname -r)
            apt-get -y install dkms
            ;;
        *centos* | *red* | *amazon* | *rocky* )
            kernelsrc=/usr/src/kernels/$(uname -r)
            kernelmodule=/lib/modules/$(uname -r)

            #https://fedoraproject.org/wiki/EPEL
            case "$osVersion" in
                *7.*)
                    # dkms
                    yum -y install https://dl.fedoraproject.org/pub/epel/epel-release-latest-7.noarch.rpm
                    subscription-manager repos --enable rhel-7-server-optional-rpms --enable rhel-7-server-extras-rpms
                    ;;
                2 )
                    # dkms
                    amazon-linux-extras install epel
                    ;;
                *8.*)
                    ARCH=$( /bin/arch )
                    # dkms
                    dnf install -y --nogpgcheck https://dl.fedoraproject.org/pub/epel/epel-release-latest-8.noarch.rpm
                    subscription-manager repos --enable "codeready-builder-for-rhel-8-${ARCH}-rpms"
                    # centos 8
                    dnf config-manager --set-enabled powertools
                    ;;
                *)
                    exit 1
                    ;;
            esac

            yum -y install "kernel-devel-uname-r == $(uname -r)"
            yum -y install dkms
            ;;
        *suse*)
            zypper -n install kernel-default-devel
            zypper -n install curl
            ;;
        *)
            get_str CTXFASCFG_UNSUPPORT_PLATFORM_PRECHECK "${osPlatform}"
            exit 1
            ;;
    esac

    # link kernel module with kernel src
    if [ "${osType}" != "SS" ]; then
        rm ${kernelmodule}/build
        ln ${kernelsrc} -s ${kernelmodule}/build
    fi
}


function confSuseV4l2()
{
    # download and extract the latest tested commit v4l2loopback source code
    curl -L  https://github.com/umlaeute/v4l2loopback/archive/${commit}.zip --output v4l2loopback-${commit}.zip
    unzip v4l2loopback-${commit}.zip -d /usr/src
    mv /usr/src/v4l2loopback-${commit} /usr/src/v4l2loopback-${version}
    cd /usr/src/v4l2loopback-${version}
    make
    cd -
}

function confDkms()
{
    # remove DKMS-module
    dkms remove -m v4l2loopback -v ${version} --all
    rm -rf /usr/src/v4l2loopback-${version}

    # download and extract the latest tested commit v4l2loopback source code
    curl -L  https://github.com/umlaeute/v4l2loopback/archive/${commit}.zip --output v4l2loopback-${commit}.zip
    unzip v4l2loopback-${commit}.zip -d /usr/src
    mv /usr/src/v4l2loopback-${commit} /usr/src/v4l2loopback-${version}

    # build and install DKMS-module
    dkms add -m v4l2loopback -v ${version}
    dkms build -m v4l2loopback -v ${version}
    dkms install -m v4l2loopback -v ${version}
}

function confWCam()
{
    # RHEL/CentOS
    if [ -f /var/lib/dkms/v4l2loopback/${version}/$(uname -r)/x86_64/module/v4l2loopback.ko.xz ]; then
        xz -d /var/lib/dkms/v4l2loopback/${version}/$(uname -r)/x86_64/module/v4l2loopback.ko.xz
    fi

    if [ -f ${ctxlib64}/v4l2loopback.ko ]; then
        rm ${ctxlib64}/v4l2loopback.ko
    fi

    if [ "${osType}" == "SS" ]; then
        cp /usr/src/v4l2loopback-${version}/v4l2loopback.ko ${ctxlib64}/v4l2loopback.ko
    else
        cp /var/lib/dkms/v4l2loopback/${version}/$(uname -r)/x86_64/module/v4l2loopback.ko ${ctxlib64}/v4l2loopback.ko
    fi
}

function confWCamSrv()
{
    service ctxwcamsd restart
}

########################### main process block #################################
main()
{
    checkUser

    checkOS

    if [ "${osType}" == "SS" ]; then
        if [ ! -f /usr/src/v4l2loopback-${version}/v4l2loopback.ko ]; then
            confOS
            confSuseV4l2
        fi
    else
        if [ ! -f /var/lib/dkms/v4l2loopback/${version}/$(uname -r)/x86_64/module/v4l2loopback.ko ]; then
            confOS
            confDkms
        fi
    fi

    confWCam

    confWCamSrv
}

main "$@"
