#!/bin/bash

###############################################################################
#
# Citrix Virtual Apps & Desktops For Linux Script: Create a VNC server for session shadowing
# Copyright (c) Citrix Systems, Inc. All Rights Reserved.
#
source /var/xdl/configure_utilities.sh

export LC_ALL="en_US.UTF-8"
CTX_USER="ctxsrvr"
CTX_BIN="/opt/Citrix/VDA/bin"
CERT="/etc/xdl/shadowingssl/shadowingcert.*"
KEY="/etc/xdl/shadowingssl/shadowingkey.*"
logFile="/dev/null"
osPlatform=""

function usage {
    echo "Usage: "
    echo "  Pop up a query window:"
    echo "           $(basename $0) --popupquery [session id] [msg number] "
    echo "  Start a VNC server:"
    echo "           $(basename $0) --startserver [session id] [port number] [password] [password file path] [enable ssl]"
    echo "  Kill a VNC server:"
    echo "           $(basename $0) --killserver [process number]"
}

function get_str {
    /opt/Citrix/VDA/bin/getstr "$@"
}

function myLog
{
    echo -e "$1">>"$logFile" 2>&1
}

function pop_up_query_window {
    case $2 in
        1)
            msg="$(get_str UI_SHADOW_NEW_REQ)"
            ;;
        2)
            msg="$(get_str UI_SHADOW_SWITCH_REQ)"
            ;;
        *)
            exit 1
            ;;
    esac
    # By default, wait for 20 seconds
    result1=$(runuser -l ${CTX_USER} -s /bin/bash -c "${CTX_BIN}/ctxmsg -w -b \"yes_no\" -i \"question\" $1 \"$msg\" 20")
    read -a result2 <<< ${result1}
    result=${result2#PopupResult=}
    myLog "Pop up result: ${result}"
    case "${result}" in
        yes)
            exit 2
            ;;
        no|timeout)
            exit 3
            ;;
        *)
            exit 1
            ;;
    esac
}

function trim {
    echo $(echo "$1" | sed -r 's/^[[:space:]]+|[[:space:]]+$//g')
}

# Kill vnc server given its process id and all the child processes
function kill_vnc_server {
    pid=$1
    myLog "kill process: $pid and its child processes"
    for i in `ps -ef | awk '$3 == '$pid' {print $2}'`
    do
        myLog "kill process: kill process $pid's child process: $i"
        kill -9 $i >> "${logFile}" 2>&1
    done
    # kill the server process it self
    myLog "kill process: kill process $pid"
    kill -9 "${pid}" >> "${logFile}" 2>&1
    exit 0
}

# Start a vnc server
function start_vnc_server {
    SID=$1
    port_number=$2
    password=$3
    passwdfile=$4
    use_ssl=$5

    getOSInfo

    gfxpid=$(ps h -C ctxgfx -o pid,cmd | grep -e "-session $SID[[:space:]]*\$" | awk '{print $1}')
    if [ -z "$gfxpid" ]; then
        myLog "Session $SID is not found."
        exit 1
    fi
    dpy=$(ps h -C Xorg,ctxvfb -o ppid,cmd | grep -e "^[[:space:]]*$gfxpid" | sed -r 's/.*(:[[:digit:]]+).*/\1/')
    dpy=$(trim "$dpy")
    if [ -z "$dpy" ]; then
        myLog "Failed to find display for session $SID, ppid $gfxpid."
        exit 1
    fi
    username=$(ps h -C ctxlogin -o ruser:25,cmd | grep -e "display $dpy" | awk '{print $1}')
    if [ -z "$username" ]; then
        myLog "Failed to find user of dedicated display $dpy"
        exit 1
    fi

    myLog "dpy=$dpy,SID=$SID,ppid=$gfxpid,username=$username"
    # Generate a passwd-file first
    runuser -l "${username}" -s /bin/bash -c "x11vnc -storepasswd $password $passwdfile" >> "${logFile}" 2>&1

    websocpath=$(which websockify)
    websocpath=${websocpath%websockify}
    if [ -z "$websocpath" ]; then
        myLog "Websockify is not found."
        exit 1
    fi
    cp -n /usr/bin/rebind.so "${websocpath}" >> "${logFile}" 2>&1
    
    # If using ssl, we need to check cert files and use ssl-only for vnc server
    if [ "$use_ssl" == "True" ]; then
        # Check whether the cert and files exist
        if [ ! -f ${CERT} ] || [ ! -f ${KEY} ]; then
            myLog "Failed to find ${CERT} or ${KEY}."
            exit 1
        fi

        # Create a vnc server using $port_number and $dpy, this server will exit in 60s if there is no connection.
        # And it can only be used once.
        # TODO: error handling
        if [ "${osPlatform}" == "amazon" ]; then
            runuser -l "${username}" -s /bin/bash -c "websockify -D $port_number --ssl-only --cert ${CERT} --key ${KEY} -- x11vnc -viewonly -rfbauth $passwdfile -rfbport $port_number -display $dpy -timeout 60 -o /dev/null" >> "${logFile}" 2>&1
        else 
            runuser -l "${username}" -s /bin/bash -c "websockify -D $port_number --ssl-only --cert ${CERT} --key ${KEY} -- x11vnc -viewonly -rfbauth $passwdfile -rfbport $port_number -display $dpy -timeout 60" >> "${logFile}" 2>&1
        fi
    else
        if [ "${osPlatform}" == "amazon" ]; then
            runuser -l "${username}" -s /bin/bash -c "websockify -D $port_number -- x11vnc -viewonly -rfbauth $passwdfile -rfbport $port_number -display $dpy -timeout 60 -o /dev/null" >> "${logFile}" 2>&1
        else
            runuser -l "${username}" -s /bin/bash -c "websockify -D $port_number -- x11vnc -viewonly -rfbauth $passwdfile -rfbport $port_number -display $dpy -timeout 60" >> "${logFile}" 2>&1
        fi
    fi

    # Check if the port is ready
    retry_max=20
    for ((retry_num=1; retry_num<=$retry_max; retry_num++))
    do
        myLog "Retry time: $retry_num, sleeping 1 second before trying again..."
        sleep 1
        netstat -tuplen |grep "x11vnc" |grep "$port_number"
        if [[ "$?" -ne "0" ]]; then
            myLog "VNC server on port $port_number not ready"
        else
            myLog "VNC server ready on port $port_number"
            break
        fi
    done
    
    # Check if max retry time reached
   if [ $retry_num -gt $retry_max ]; then
        myLog "WARNING: setup VNC server timeout, the server is not ready for connection!"
   fi


    # Get the pid of the vnc server
    pid=$(ps h -C websockify -o pid,cmd |grep -e "$dpy[[:space:]]" | grep "$port_number" | awk '{print $1}')

    if [ -z "$pid" ]; then
        myLog "VNC server pid not found."
        exit 1
    fi
    echo "VNCPID:$pid"
}

if [ "$1" == "--help" ] ; then
    usage
    exit 0
elif [ "$1" == "--popupquery" ] ; then
    if [ "$#" -ne 3 ]; then
        usage
        exit 1
    fi
    pop_up_query_window $2 $3
    exit 0
elif [ "$1" == "--startserver" ] ; then
    if [ "$#" -ne 6 ]; then
        usage
        exit 1
    fi
    start_vnc_server $2 $3 $4 $5 $6
    exit 0
elif [ "$1" == "--killserver" ] ; then
    if [ "$#" -ne 2 ]; then
        usage
        exit 1
    fi
    kill_vnc_server $2
    exit 0
else
    usage
    exit 1
fi
