#!/bin/bash
#set -x
#######################################################################
#                                    #                                #
#  Script to collect logs and configuration of XDT Linux VDA          #
#                                    #                                #
#                                    #                                #
#  Change History:                                                    #
#    Initial version  Valerio Fuoglio  (C)Citrix EMEA Escalation Team #
#    10 Nov. 2017     Guoyi Zhou       Linux VDA LCM team             #
#    20 Mar. 2018     Qianqian Zu      Linux VDA LCM team             #
#                                    #                                #
#######################################################################

if [[ $EUID -ne 0 ]]; then
    echo "This script must be run as root"
    exit 1
fi

# ======== CONFIG
fname=$(basename $0)
APPNAME=${fname%.*}  # Script name without extension
#DEST_DIR=`pwd`
TMPDIR="/tmp"
DATE0="$(date +%F_%H%M%S)"
DATE="`date "+%Y-%m-%d %H:%M:%S"`"

# Global TMP directory for the application
APP_GLOBAL_TMP="${TMPDIR}/${APPNAME}"
# TMP directory for this instance
APP_TMP="${APP_GLOBAL_TMP}/${APPNAME}_${DATE0}"

LOG_TIMEFORMAT="+%F_%H:%M:%S"
LOG_FILENAME="${APPNAME}.log"
LOG_COMPLETEPATH="${APP_TMP}/${LOG_FILENAME}"

FINAL_FILE="${APPNAME}_${DATE0}.tar.bz2"

TOTAL_STEPS=5
CUR_STEP=0

# ========== /CONFIG


DISTRO=""
systemType="$(uname  -a |grep -iE "Ubuntu|Debian" 2>&1)"

#if grep -F "Red Hat" /etc/redhat-release
if [ -f /etc/redhat-release ]
then
    alias netcat="nc"
    type nc >/dev/null 2>&1 || { echo >&2 "Netcat is required but not installed, please install it and run this script again."; echo "Please run: yum install nc"; exit 1; }
    DISTRO="RHEL"
elif [ -f /etc/SuSE-release ]
then
    DISTRO="SLE"
    type netcat >/dev/null 2>&1 || { echo >&2 "Netcat is required but not installed, please install it and run this script again."; echo "Please run: zypper in netcat"; exit 1; }
elif [ -n "$systemType" ]
then
        alias netcat="nc"
        type nc >/dev/null 2>&1 || { echo >&2 "Netcat is required but not installed, please install it and run this script again."; echo "Please run: apt-get install netcat-openbsd"; exit 1; }
        DISTRO="UBUNTU or Debian"
        if [ "$1" == "-silent" ]
        then
            echo "Skip mailutils installation for silent mode."
        else
            dpkg -l | grep mailutils > /dev/null
            if [ "$?" -ne 0 ]
            then
                apt install -y mailutils
            fi
        fi
else
    os_platform="$(< /etc/os-release grep ^ID= |cut -d '=' -f2 |tr '[:upper:]' '[:lower:]' |tr -d '"'  2>&1)"
    if [[ $os_platform == "sles" || $os_platform == "sled" ]]; then
        DISTRO="SLE"
        type netcat >/dev/null 2>&1 || { echo >&2 "Netcat is required but not installed, please install it and run this script again."; echo "Please run: zypper in netcat"; exit 1; }
    else
        ##echo "Installed Linux distribution is currently unsupported"
        alias netcat="nc"
        type nc >/dev/null 2>&1 || { echo >&2 "Netcat is required but not installed, please install it and run this script again."; echo "Please run: yum install nc"; exit 1; }
        DISTRO="RHEL"
    fi
fi

mkdir -p ${APP_TMP}

function log {
    echo  "$@" | tee -a "${LOG_COMPLETEPATH}"
}

function coll_system {

    MOD_DIR="${APP_TMP}/system"
    mkdir -p ${MOD_DIR}


    {
        echo "Release file"
        echo ""
        cat /etc/*elease
        echo ""
        echo "LSB"
        lsb_release -d
        echo ""
    } >> ${MOD_DIR}/release 2>&1

    {
        uname -a
        echo ""
        echo "Hostname: $(hostname)"
        echo "Hostname - Full: $(hostname -f)"
        echo ""
        uptime
        echo ""
        free -m
        echo ""
        df -h -all
        echo ""
        mount
    } >> ${MOD_DIR}/sysinfo 2>&1
        
    {
        echo ""
        fdisk -l
        echo ""
        lsblk
        echo ""
        blkid
        echo ""
    } >> ${MOD_DIR}/diskinfo 2>&1

    lsmod >> ${MOD_DIR}/lsmod 2>&1

    lspci -v >> ${MOD_DIR}/lspci 2>&1

    {
        lsusb
        echo ""
        lsusb -v
    } >> ${MOD_DIR}/lsusb 2>&1

    top -n 1 >> ${MOD_DIR}/top 2>&1

    ps aux >> ${MOD_DIR}/ps 2>&1

    cat /proc/meminfo >> ${MOD_DIR}/meminfo 2>&1
    cat /proc/cpuinfo >> ${MOD_DIR}/cpuinfo 2>&1

        if [ "${DISTRO}" != "UBUNTU or Debian" ]
        then
        chkconfig --list >> ${MOD_DIR}/services 2>&1
        fi
    cat /etc/resolv.conf >> ${MOD_DIR}/resolv.conf 2>&1

    cp /etc/hosts* ${MOD_DIR} > /dev/null 2>&1

    cat /etc/services >> ${MOD_DIR}/services 2>&1

        if [ "${DISTRO}" = "UBUNTU or Debian" ]
        then
            dpkg -l | sort >> ${MOD_DIR}/RPMs
        else
            rpm -qa | sort >> ${MOD_DIR}/RPMs
        fi
    dmesg >> ${MOD_DIR}/dmesg 2>&1

        if [ "${DISTRO}" = "UBUNTU or Debian" ]
        then
            cp /var/log/syslog ${MOD_DIR}/ > /dev/null 2>&1
        else
            cp /var/log/messages ${MOD_DIR}/ > /dev/null 2>&1
        fi

    #Xorg
    mkdir ${MOD_DIR}/Xorg
    cp /var/log/Xorg.* ${MOD_DIR}/Xorg > /dev/null 2>&1

    if [ "${DISTRO}" = "SLE" ]
    then
        cp /var/log/boot.msg ${MOD_DIR}/ > /dev/null 2>&1
    elif [ "${DISTRO}" = "RHEL" ]
    then
        cp /var/log/boot.log ${MOD_DIR}/ > /dev/null 2>&1
    elif [ "${DISTRO}" = "UBUNTU or Debian" ]
    then
        cp /var/log/bootstrap.log ${MOD_DIR}/ > /dev/null 2>&1
    fi

    #postgresql configuration
    mkdir ${MOD_DIR}/Postgresql

    if [ "${DISTRO}" = "UBUNTU or Debian" ]
    then
            #PGHBADIR=$(dirname $(su postgres -c "psql -v ON_ERROR_STOP=1 -c \"show hba_file;\" | grep \"^[ ]*/\" | tr -d ' '"))
            PGHBADIR="/etc/postgresql/9.*/main" # default dir
            cp ${PGHBADIR}/postgresql.conf ${MOD_DIR}/Postgresql > /dev/null 2>&1
    else
            #PGHBADIR=$(dirname $(su postgres -c "psql -v ON_ERROR_STOP=1 -c \"show hba_file;\" | grep \"^[ ]*/\" | tr -d ' '"))
            PGHBADIR="/var/lib/pgsql/data"  # default dir
            cp ${PGHBADIR}/postgresql.conf ${MOD_DIR}/Postgresql > /dev/null 2>&1
            cp ${PGHBADIR}/pg_log/*.log ${MOD_DIR}/Postgresql > /dev/null 2>&1
    fi
}


function coll_network {

    MOD_DIR="${APP_TMP}/network"
    mkdir -p ${MOD_DIR}

    ifconfig -a >> ${MOD_DIR}/ifconfig_a 2>&1

    iptables -L >> ${MOD_DIR}/iptables_list 2>&1

    iptables-save >> ${MOD_DIR}/iptables_save 2>&1

    route > ${MOD_DIR}/route 2>&1

    netstat -apvee >> ${MOD_DIR}/netstat 2>&1

    {
        for nameserver in $(cat /etc/resolv.conf |grep ^nameserver|cut -d " " -f 2)
        do
            echo "Nameserver: $nameserver"
            echo "PING:"
            ping -c 10 $nameserver
            echo "NETCAT:"
            netcat -zvu $nameserver 53
            echo ""
        done
    } >> ${MOD_DIR}/nameservers 2>&1
}

function coll_ad_winbind {

    MOD_DIR="${APP_TMP}/ad"
    mkdir -p ${MOD_DIR}

    MOD_DIR_LOGS="${MOD_DIR}/logs"
    mkdir -p ${MOD_DIR_LOGS}


    if [ "${DISTRO}" = "SLE" ]
    then
        cp -r /var/log/krb5 ${MOD_DIR_LOGS} > /dev/null 2>&1
    fi

    cp -r /var/log/samba ${MOD_DIR_LOGS} > /dev/null 2>&1

    MOD_DIR_CONF="${MOD_DIR}/conf"
    mkdir -p ${MOD_DIR_CONF}

    cp /etc/krb5.conf ${MOD_DIR_CONF} > /dev/null 2>&1
    cp /etc/krb5.keytab ${MOD_DIR_CONF} > /dev/null 2>&1
    cp -r /etc/samba ${MOD_DIR_CONF} > /dev/null 2>&1

    SEC_DDR_CONF="${MOD_DIR}/security"
    mkdir -p ${SEC_DDR_CONF}

    cp -r /etc/security/ ${SEC_DDR_CONF} > /dev/null 2>&1

    {
        echo "== net ads info =="
        net ads info > /dev/null
        echo ""
        echo "== net ads testjoin =="
        net ads testjoin > /dev/null
        echo "== net ads workgroup =="
        net ads workgroup > /dev/null
        echo ""
        echo "== net ads keytab list =="
        net ads keytab list > /dev/null
        echo ""
        echo "== net ads lookup =="
        net ads lookup > /dev/null
        echo ""
    } >> ${MOD_DIR}/ads 2>&1

    {
        echo "== wbinfo --own-domain =="
        wbinfo --own-domain
        echo "" >>  ${MOD_DIR}/wbinfo
        echo "== wbinfo -p =="
        wbinfo -p
        echo "" >>  ${MOD_DIR}/wbinfo
        echo "== wbinfo --online-status =="
        wbinfo --online-status
        echo "" >>  ${MOD_DIR}/wbinfo
        echo "== wbinfo -D `wbinfo --own-domain` =="
        wbinfo -D `wbinfo --own-domain`
        echo "" >>  ${MOD_DIR}/wbinfo
    } >> ${MOD_DIR}/wbinfo 2>&1

    {
        echo "== klist =="
        klist
        echo ""
        echo "== klist -ke =="
        klist -ke
        echo ""
    } >> ${MOD_DIR}/klist 2>&1

    KDC_SERVER=`net ads info | grep "KDC server"|cut -d ":" -f 2`
    LDAP_SERVER=`net ads info | grep "LDAP server:"|cut -d ":" -f 2`
    LDAP_SERVER_NAME=`net ads info | grep "LDAP server name:"|cut -d ":" -f 2`
    LDAP_PORT=`net ads info|grep "LDAP port"|cut -d ":" -f 2`

    {
        echo "KDC Server: $KDC_SERVER"
        echo "PING:"
        ping -c 10 $KDC_SERVER
        echo "NETCAT - $LDAP_PORT":
        netcat -zv $nameserver ${LDAP_PORT}
        echo ""

        echo "LDAP Server: $LDAP_SERVER"
        echo "PING:"
        ping -c 10 $LDAP_SERVER
        echo "NETCAT - $LDAP_PORT":
        netcat -zv $LDAP_SERVER ${LDAP_PORT}
        echo ""

        echo "NSLookup for LDAP server name - $LDAP_SERVER_NAME"
        nslookup $(net ads info | grep "LDAP server name"|cut -d ":" -f 2)
    } >> ${MOD_DIR}/ad_net 2>&1
}

function coll_xdl {

    if [ "${DISTRO}" = "UBUNTU or Debian" ]
    then
        export VDA_BIN_PATH=`dpkg -L xenDesktopVDA | grep /bin/ctxreg|xargs dirname`
    else
    export VDA_BIN_PATH=`rpm -qa|grep XenDesktopVDA|xargs rpm -ql|grep /bin/ctxreg|xargs dirname`
    fi
    export PATH=$VDA_BIN_PATH:$PATH

    MOD_DIR="${APP_TMP}/xdl"
    mkdir -p ${MOD_DIR}

    MOD_DIR_LOGS="${MOD_DIR}/logs"
    mkdir -p ${MOD_DIR_LOGS}

    mkdir ${MOD_DIR_LOGS}/tmp

    {
        # XDL 1.0-1.2
        cp /tmp/hdx.* ${MOD_DIR_LOGS}/tmp >> ${MOD_DIR_LOGS}/tmp/info.log 2>&1
        cp /tmp/xdl.* ${MOD_DIR_LOGS}/tmp >> ${MOD_DIR_LOGS}/tmp/info.log 2>&1
        cp /tmp/citrix.dbg ${MOD_DIR_LOGS}/tmp >> ${MOD_DIR_LOGS}/tmp/info.log 2>&1

        # XDL 1.3+
        cp /tmp/.ctxlog ${MOD_DIR_LOGS}/tmp >> ${MOD_DIR_LOGS}/tmp/info.log 2>&1

        cp -r /var/log/xdl ${MOD_DIR_LOGS}/var_xdl >> ${MOD_DIR_LOGS}/tmp/info.log 2>&1
        cp -r /etc/xdl ${MOD_DIR}/etc_xdl >> ${MOD_DIR_LOGS}/tmp/info.log 2>&1

        # easy install log
        cp /var/log/ctxinstall.log  ${MOD_DIR} >> ${MOD_DIR_LOGS}/tmp/info.log 2>&1

        # mcs log
        cp /var/log/deploymcs.log  ${MOD_DIR} >> ${MOD_DIR_LOGS}/tmp/info.log 2>&1
        cp /var/log/ad_join.log  ${MOD_DIR} >> ${MOD_DIR_LOGS}/tmp/info.log 2>&1
        cp /var/log/update_machine_secret.log  "${MOD_DIR}" >> "${MOD_DIR_LOGS}"/tmp/info.log 2>&1

    }

    ctxreg list -kHKLM > ${MOD_DIR}/ctxreg.list 2>&1
    ctxreg dump -kHKLM > ${MOD_DIR}/ctxreg.dump 2>&1

    {
        echo "Sevices"
        echo "ctxvda: "
        service ctxvda status
        echo ""
        echo "ctxhdx: "
        service ctxhdx status
        echo ""
        echo "postgresql: "
        service postgresql status
        echo ""
    } >> ${MOD_DIR}/ctxservices 2>&1

    CFX_PortHex=$(ctxreg read -kHKLM\\Software\\Citrix\\VirtualDesktopAgent\\ -v VdaCxfServicesPort|sed 's/x//'|sed 's/^0*//')
    CFX_PortDec=$((16#${CFX_PortHex}))
    DDC_PortHex=$(ctxreg read -kHKLM\\Software\\Citrix\\VirtualDesktopAgent\\ -v ControllerRegistrarPort|sed 's/x//'|sed 's/^0*//')
    DDC_PortDec=$((16#${DDC_PortHex}))
    ListOfDDCs="$(ctxreg read -kHKLM\\Software\\Citrix\\VirtualDesktopAgent\\ -v ListOfDDCs)"

    {
        echo "Testing DDCs"
        echo ""

        for ddc in ${ListOfDDCs}
        do
            echo "DDC: $ddc"
            echo "PING:"
            ping -c 10 $ddc
            echo "NETCAT - $DDC_PortDec":
            netcat -zv $ddc ${DDC_PortDec}
            echo ""
            echo "NSLOOKUP - $ddc"
            nslookup $ddc
            echo ""
        done
    } >> ${MOD_DIR}/ddcs 2>&1

    {
        echo "XDL NET"
        echo "NETCAT - $CFX_PortDec":
        netcat -zv 127.0.0.1 ${DDC_PortDec}
        echo ""
    } >> ${MOD_DIR}/xdl_net 2>&1

    # generate core dump for ctxhdx service twice for better analysis
    for ((count=1; count<=2; count++))
    do
        gcore -a -o "${MOD_DIR}/ctxhdx.core.${count}" `pgrep ctxhdx` > /dev/null 2>&1
        sleep 5
    done

}

function compress_xdl {
    tar cjf ${APP_GLOBAL_TMP}/${FINAL_FILE} -C ${APP_GLOBAL_TMP} ${APPNAME}_${DATE0} > /dev/null
    chmod 640 ${APP_GLOBAL_TMP}/${FINAL_FILE} > /dev/null
    rm -rf ${APP_TMP}
}

rm ${APP_GLOBAL_TMP}/xdlcollect_*.tar.bz2 > /dev/null 2>&1
rm ${APP_GLOBAL_TMP}/id.txt > /dev/null 2>&1

echo "#####################################################Copyright####################################################"
echo "The tool collects information of this machine and send it to Citrix for debugging issues."
echo "Citrix may use this information to provide support."
if [ "$1" != "-silent" ]
then
    echo -n "Do you want to continue with using this tool to collect information? (Y/N): "
    read str
    lowerStr=$(echo "$str" | tr '[:upper:]' '[:lower:]')
    if [[ "${lowerStr}" == "n" || "${lowerStr}" == "no" ]]; then
        exit 0
    fi
fi

log '################################################################'
log "Starting ${APPNAME} - Date: ${DATE}                             "
log "Total Steps: ${TOTAL_STEPS}               "
CUR_STEP=`expr $CUR_STEP + 1`
log "    Step $CUR_STEP - Collecting System info"

CUR_STEP=`expr $CUR_STEP + 1`
coll_system
log "    Step $CUR_STEP - Collecting Network info"

CUR_STEP=`expr $CUR_STEP + 1`
coll_network
log  "    Step $CUR_STEP - Collecting XDL logs and configurations"

CUR_STEP=`expr $CUR_STEP + 1`
coll_xdl
if [[ $(ps -ef | grep -c winbindd)  -ne 1 ]]; then
    log "    Step $CUR_STEP - Collecting Active Directory client logs and configurations"
    coll_ad_winbind
else
    log "    Step $CUR_STEP - Collecting Active Directory client logs and configurations -- Ignore"
fi
CUR_STEP=`expr $CUR_STEP + 1`

log "    Step $CUR_STEP - Compressing ${APPNAME} collection"
compress_xdl

DATE="`date "+%Y-%m-%d %H:%M:%S"`"
echo "DONE - Date: ${DATE}"
echo '################################################################'
if [ "$1" == "-silent" ]
then
    echo "The result file(${FINAL_FILE}) is under directory($APP_GLOBAL_TMP). "
else
    echo ""
    echo -n "Do you want to upload this file on https://cis.citrix.com/? (Y/N): "
    read str
    lowerStr=$(echo "$str" | tr '[:upper:]' '[:lower:]')

    hostname=$(hostname -f)
    if [[ "${lowerStr}" == "y" || "${lowerStr}" == "yes" ]]; then
        Upload_File="${APP_GLOBAL_TMP}/${FINAL_FILE}"
        python3 /var/xdl/upload_cis.py "$Upload_File" >> "${APP_GLOBAL_TMP}/id.txt"
	Upload_Id="$(cat "${APP_GLOBAL_TMP}/id.txt")"
        echo "The result file(${FINAL_FILE}) is under directory($APP_GLOBAL_TMP).
You can get it either from that or from https://cis.citrix.com/ using ID ${Upload_Id}. "
    else
        echo "The result file(${FINAL_FILE}) is under directory($APP_GLOBAL_TMP). "
    fi
fi
