#!/usr/bin/env python
import json
import base64
import os.path
import sys
import copy
import time
import zipfile
import urllib.request, urllib.parse, urllib.error
from urllib.error import HTTPError

ZipFileName = "/var/xdl/ceip/linuxvda_ceip.zip"
MAX_RETRY_CNT = 10
LOG_FILE = os.devnull


class UploadError(Exception):
    def __init__(self, error_code, value):
        self.error_code = error_code
        self.value = value
    def __str__(self):
        return "Error code: {0}, {1}".format(self.error_code, self.value)

def post(url, params, headers, data):
    if params is not None:
        url = url + "?" + urllib.parse.urlencode(params)
    request = urllib.request.Request(url, b'data', headers)
    try:
        res = urllib.request.urlopen(request)
        return res
    except HTTPError as e:
        print(e.reason)
        raise UploadError(e.code, e.reason)

def upload_files(pk_key, url, *args):
    log = open(LOG_FILE, 'w+')
    log.write(time.ctime()+'\n')
    failed = []
    failed.extend(args)
    response = "-1"
    # compress all data to a zip file
    try:
        f = zipfile.ZipFile(ZipFileName, 'w', zipfile.ZIP_DEFLATED)
        for filename in failed:
            log.write("Add " + filename + " to zip file.\n")
            f.write(filename, os.path.basename(filename))
        f.close()
    except:
        return "Compress Error!"

    retry_count = 0
    retry = True

    # upload the file and print the progress
    while retry and retry_count <= MAX_RETRY_CNT: 
        try:
            response = _upload_whole_file(pk_key, url, ZipFileName)
            retry = False
        except UploadError as err: 
            response = "Upload error: {0}".format(str(err))
        except IOError as err1:
            response = "IO error: {0}".format(str(err1))
        except Exception as err2:
            response = "Exception: {0}".format(str(err2))
        retry_count += 1
        #retry uploading  after 1 seconds
        if retry_count <= MAX_RETRY_CNT: 
            time.sleep(1)
            log.write("retry...\n")

    #delete zip file
    try:
        os.remove(ZipFileName)
    except:
        log.write("Delete " + ZipFileName + " failed." + "\n")
        pass
    log.write("response: " + response + "\n")
    log.close()
    return response


########################################
#Upload file
#
########################################
def _upload_whole_file(pd_key, url, filename):
    auth_header = {"content-type": "application/json", "Authorization": "PK "+ pd_key}
    with open(filename, "rb") as f:
        fc = f.read()

    bundle_params = {
        "name": os.path.basename(filename)
    }

    bundle_params["size"] = len(fc)

    payload = {}
    ret = post(url + "feeds/api/bundle/", params=bundle_params, headers=auth_header, data=json.dumps(payload))
    upload_id = json.loads(ret.read())["id"]

    CHUNK_SIZE = 5 * 1024 * 1024
    offset = 0
    while offset < len(fc):
        this_chunk_length = CHUNK_SIZE if len(fc) >= (offset + CHUNK_SIZE) else len(fc) - offset
        this_chunk = fc[offset:(offset+this_chunk_length)]
        ret = post(url + "feeds/api/upload_chunk/", None, headers=auth_header, data=json.dumps({"id": upload_id, "offset": offset, "size": this_chunk_length, "data": base64.b64encode(this_chunk)}))
        offset += this_chunk_length
    return upload_id

if __name__ =="__main__":
    print(upload_files("pd_key", "https://rttf-staging.citrix.com/", "upload_cis.py", "upload_ga.py"))
