#!/bin/bash

###############################################################################
#
# Citrix Virtual Apps & Desktops For Linux Script: Enumerate applications
#
# Copyright (c) Citrix Systems, Inc. All Rights Reserved.
#

export LC_ALL="en_US.UTF-8"

function usage {
    echo "Usage: $(basename $0) [session id]"
}

function trim {
    echo $(echo "$1" | sed -r 's/^[[:space:]]+|[[:space:]]+$//g')
}

if [ "$#" -ne 1 ]; then
    usage
    exit 1
fi

if [ "$1" == "--help" ] ; then
    usage
    exit 0
fi

SID=$1

gfxpid=$(ps h -C ctxgfx -o pid,cmd | grep "-session $SID" | awk '{print $1}')
if [ -z "$gfxpid" ]; then
    echo "Session $SID is not found."
    exit 1
fi
dpy=$(ps h -C ctxvfb,Xorg -o ppid,cmd | grep -e "^[[:space:]]*$gfxpid" | sed -r 's/.*(:[[:digit:]]+).*/\1/')
dpy=$(trim "$dpy")
if [ -z "$dpy" ]; then
    echo "Failed to find display for session $SID, ppid $gfxpid."
    exit 1
fi

windows=$(xprop -display "$dpy" -root | grep _NET_CLIENT_LIST\(WINDOW\) | sed -r -e 's/.*# (.*)/\1/' -e 's/,//g')
declare -A pids

for w in $windows; do
    proplist=$(xprop -display "$dpy" -id "$w")
    pid=$(echo "$proplist" | sed -nr 's/_NET_WM_PID\(CARDINAL\) = (.*)/\1/p')
    title=$(echo "$proplist" | sed -nr 's/WM_NAME\(STRING\) = "(.*)"/\1/p')
    if [ -z "$title" ] ; then
        # In some instances this attribute is not a quoted string, but is instead
        # a comma separated list of UTF8 bytes (in hex). In those instances this
        # match will fail, and a subsequent match attempt is made using
        # the WM_NAME(COMPOUND_TEXT) attribute.
	    title=$(echo "$proplist" | sed -nr 's/_NET_WM_NAME\(UTF8_STRING\) = "(.*)"/\1/p')
    fi
    if [ -z "$title" ] ; then
        title=$(echo "$proplist" | sed -nr 's/WM_NAME\(COMPOUND_TEXT\) = "(.*)"/\1/p')
    fi
    wtype=$(echo "$proplist" | sed -nr 's/_NET_WM_WINDOW_TYPE\(ATOM\) = (.*)/\1/p')
    if [ "$wtype" == "_NET_WM_WINDOW_TYPE_NORMAL" ] ; then
        if [ ! -z "$pid" ]; then
            pids[$pid]=1
        fi
        echo -en "$pid\t$(printf '%lld' $w)\t"
        echo -E "$title"
    fi
done

# Workaround: some processes only have dialog window, we also treat them as applications
for w in $windows; do
    proplist=$(xprop -display "$dpy" -id "$w")
    pid=$(echo "$proplist" | sed -nr 's/_NET_WM_PID\(CARDINAL\) = (.*)/\1/p')
    if [ ! -z "$pid" -a "${pids[$pid]}" != "1" ]; then
        title=$(echo "$proplist" | sed -nr 's/WM_NAME\(STRING\) = "(.*)"/\1/p')
        if [ -z "$title" ] ; then
            title=$(echo "$proplist" | sed -nr 's/_NET_WM_NAME\(UTF8_STRING\) = "(.*)"/\1/p')
        fi
        if [ -z "$title" ] ; then
            title=$(echo "$proplist" | sed -nr 's/WM_NAME\(COMPOUND_TEXT\) = "(.*)"/\1/p')
        fi
        wtype=$(echo "$proplist" | sed -nr 's/_NET_WM_WINDOW_TYPE\(ATOM\) = (.*)/\1/p')
        if [ "$wtype" == "_NET_WM_WINDOW_TYPE_DIALOG" ] ; then
            if [ ! -z "$pid" ]; then
                pids[$pid]=1
            fi
            echo -en "$pid\t$(printf '%lld' $w)\t"
            echo -E "$title"
        fi
    fi
done
