#!/bin/bash

#******************************************************************************
# ctxfascfg.sh 
#
# Copyright 1998-2018, 2018 Citrix Systems, Inc. All Rights Reserved.
# 
# Authors: Long Xu, 2 Feb. 2018
#
# Citrix Virtual Apps & Desktops For Linux VDA Federated Authentication Setup Script. This script should 
# be run at least once if FAS support is enabled on StoreFront.
#
# All changes by this script can be manually reset by administrator, including:
#    a. /etc/krb5.conf
#    b. /etc/samba/smb.conf
#    c. /etc/pam.d/ctxfas
#    d. /etc/sssd/sssd.conf   
#    e. newly installed packages: krb5-pkinit, libpam-krb5
#
# Return value:
#    0 - [success] Normally return
#    1 - [failed]  None root users
#    2 - [failed]  Unsupported platform
#    3 - [failed]  Unsupported integrator way
#    4 - [failed]  File nonexistent
#
# Environment variables
#    - CTX_FAS_ADINTEGRATIONWAY (CTX_EASYINSTALL_ADINTEGRATIONWAY)
#    - CTX_FAS_CERT_PATH
#    - CTX_FAS_KDC_HOSTNAME
#    - CTX_FAS_PKINIT_KDC_HOSTNAME (Same as CTX_FAS_KDC_HOSTNAME by default)
#
# Change History:
#    2 Feb. 2018	Long Xu			Initial version
#    9 Jul. 2018    Miya Zhao		Add environment variables
#
#*******************************************************************************

source /var/xdl/configure_common.sh
source /var/xdl/configure_platform.sh
source /var/xdl/configure_utilities.sh

# global variables
fname=$(basename $0)
osPlatform=""           # OS platform:  Redhat, CentOS, SUSE, Ubuntu
osVersion=""            #
joinDomainWay=""        # legal value:  winbind, Quest, centrify, sssd

# file 
krb5File="/etc/krb5.conf"
pamFile="/etc/pam.d/ctxfas"
smbFile="/etc/samba/smb.conf"    # Ubuntu winbind
sssdFile="/etc/sssd/sssd.conf"   # Ubuntu sssd
kcmFile="/etc/krb5.conf.d/kcm_default_ccache"

# krb5.conf
default_ccache_name=""
kdc_hostname=""
default_realm=""

# pkcs11 auth
pkinit_kdc_hostname=""
pkinit_identities=""
pkinit_anchors=""
pkinit_eku_checking=""
pkcs11Lib=""
certPath="" # the path that store root CA and all the intermediate CA

# backup & log
backupDir="/var/ctxinstall"

# check user
function checkUser()
{
    if [ "$(id -u)" != 0 ]; then
        get_str CTXFASCFG_MUST_ROOT
        exit 1
    fi
}

function createLogFile()
{
    # make sure dir is exist
    if [[ ! -d "$backupDir" ]]; then
        mkdir -p "$backupDir"
    else
        # remove all the files
        rm -f "$backupDir""/*"
    fi
}

# check os distro/platform
function checkOS()
{
    # get version
    getOSInfo

    case "$osPlatform" in
        *ubuntu*)
            osPlatform="Ubuntu"
            ;;
        *debian*)
            osPlatform="Debian"
            ;;
        *centos*)
            osPlatform="CentOS"
            ;;
        *rocky*)
            osPlatform="Rocky"
            ;;
        *red*)
            osPlatform="Redhat"
            ;;
        *amazon*)
            osPlatform="Redhat"
            ;;
        *suse*)
            osPlatform="SUSE"
            ;;
        *)
            get_str CTXFASCFG_UNSUPPORT_PLATFORM_PRECHECK "${osPlatform}"
            exit 2
            ;;
    esac

    if [[ "${osPlatform}" == "Redhat" || "${osPlatform}" == "CentOS" || "${osPlatform}" == "Rocky" ]]; then
        case "$osVersion" in
            *7.*)
                osVersion="R/C7"
                /opt/Citrix/VDA/bin/ctxreg create -k "HKLM\Software\Citrix\VirtualDesktopAgent\Authentication\UserCredentialService" -t "REG_SZ" -v "DigestMethod" -d "SHA256" --force
                ;;
            2)
                osVersion="R/C7"
                ;;
            *8.*)
                # redhat/centos 8 abandon pam_krb5 package
                osVersion="R/C8"
                ;;
            *)
                get_str CTXFASCFG_UNSUPPORT_PLATFORM_PRECHECK "${osPlatform}"
                exit 2
                ;;
        esac
    fi

    get_str CTXFASCFG_SUPPORT_PLATFORM_PRECHECK "${osPlatform}"
}

# get ad integrator style
function getADJoinWay()
{
    if [[ -n "${CTX_EASYINSTALL_ADINTEGRATIONWAY}" ]]; then
        CTX_FAS_ADINTEGRATIONWAY="${CTX_EASYINSTALL_ADINTEGRATIONWAY}"
    fi

    if [[ "${osPlatform}" == "SUSE" || "${osPlatform}" == "Ubuntu" || "${osPlatform}" == "Debian" ]]; then
        if [[ "${CTX_FAS_ADINTEGRATIONWAY}" == "winbind" || "${CTX_FAS_ADINTEGRATIONWAY}" == "centrify" ]]; then
            joinDomainWay="${CTX_FAS_ADINTEGRATIONWAY}"
        else
            str0="$(get_str CTXFASCFG_SELECT_AD_WAY_CONFIGURATION_PARTIAL)"
            while true ; do
                read -p "$str0" optADJoinNum
                case "${optADJoinNum}" in
                    ""|1 )
                        joinDomainWay="winbind"
                        break;
                        ;;
                    2 ) 
                        joinDomainWay="centrify"
                        break;
                        ;;
                    * )
                        get_str CTXINSTALL_INPUT_CORRECT_AD_WAY_SELECTION 
                        continue
                        ;;
                esac
            done
        fi
    elif [[ "${osPlatform}" == "Redhat" || "${osPlatform}" == "CentOS" || "${osPlatform}" == "Rocky" ]]; then
        if [[ "${CTX_FAS_ADINTEGRATIONWAY}" == "winbind" || "${CTX_FAS_ADINTEGRATIONWAY}" == "sssd" || "${CTX_FAS_ADINTEGRATIONWAY}" == "centrify" || "${CTX_FAS_ADINTEGRATIONWAY}" == "pbis" ]]; then
            joinDomainWay="${CTX_FAS_ADINTEGRATIONWAY}"
        else
            str0="$(get_str CTXFASCFG_SELECT_AD_WAY_CONFIGURATION)"
            while true ; do
                read -p "$str0" optADJoinNum
                case "${optADJoinNum}" in
                    ""|1 )
                        joinDomainWay="winbind"
                        break;
                        ;;
                    2 ) 
                        joinDomainWay="sssd"
                        break;
                        ;;                       
                    3 ) 
                        joinDomainWay="centrify"
                        break;
                        ;;
                    4 ) 
                        joinDomainWay="pbis"
                        break;
                        ;;                          
                    * )
                        get_str CTXFASCFG_INPUT_CORRECT_AD_WAY_SELECTION 
                        continue
                        ;;
                esac
            done
        fi
    else
        get_str CTXFASCFG_UNSUPPORT_PLATFORM_PRECHECK "${osPlatform}"
        exit 2
    fi
}

# install packages
function installPkgs()
{
    get_str CTXFASCFG_PKG_INSTALLATION
    case "$osPlatform" in
        "Ubuntu" | "Debian")
            if [ ! -f /usr/lib/x86_64-linux-gnu/krb5/plugins/preauth/pkinit.so ] || [ ! -f /lib/x86_64-linux-gnu/security/pam_krb5.so ]; then
                `apt-get -y install krb5-pkinit libpam-krb5 >/dev/null 2>&1`
            fi

            [ -f /usr/lib/x86_64-linux-gnu/krb5/plugins/preauth/pkinit.so ] && [ -f /lib/x86_64-linux-gnu/security/pam_krb5.so ] && \
            get_str CTXFASCFG_SUCCESS || get_str CTXFASCFG_FAILED
            ;;
        "Redhat" | "CentOS" | "Rocky")
            if [[ "${osVersion}" == "R/C7" ]]; then
                if [ ! -f /usr/lib64/krb5/plugins/preauth/pkinit.so ] || [ ! -f /usr/lib64/security/pam_krb5.so ]; then
                    `yum -y install krb5-pkinit pam_krb5 >/dev/null 2>&1`
                fi

                [ -f /usr/lib64/krb5/plugins/preauth/pkinit.so ] && [ -f /usr/lib64/security/pam_krb5.so ] && \
                get_str CTXFASCFG_SUCCESS || get_str CTXFASCFG_FAILED
            elif [[ "${osVersion}" == "R/C8" ]]; then
                if [ ! -f /usr/lib64/krb5/plugins/preauth/pkinit.so ]; then
                    `yum -y install krb5-pkinit-1.17-18.el8.x86_64 >/dev/null 2>&1`
                fi

                # For Redhat/CentOS 8.x, please refer to Citrix official document to install pam_krb5 package

                [ -f /usr/lib64/krb5/plugins/preauth/pkinit.so ] && \
                get_str CTXFASCFG_SUCCESS || get_str CTXFASCFG_FAILED
            fi
            ;;
        "SUSE")
            if [ ! -f /usr/lib64/krb5/plugins/preauth/pkinit.so ] || [ ! -f /lib64/security/pam_krb5.so ]; then
                `zypper install krb5-plugin-preauth-pkinit pam_krb5 >/dev/null 2>&1`
            fi

            [ -f /usr/lib64/krb5/plugins/preauth/pkinit.so ] && [ -f /lib64/security/pam_krb5.so ] && \
            get_str CTXFASCFG_SUCCESS || get_str CTXFASCFG_FAILED
            ;;
        *)
           get_str CTXFASCFG_UNSUPPORTED_PLATFORM
           exit 2
            ;;
    esac
}

# configure /etc/krb5.conf
function configKrb5()
{
    if [ "$osVersion" == "R/C8" ] && [ -f "$kcmFile" ]; then
        # common out the kcm
        sed -i "/^.*default_ccache_name*/d" $kcmFile
        sed -i "/\[libdefaults\]/a \  #default_ccache_name = KCM:" $kcmFile
    fi

    if [ -f "$krb5File" ]; then
        # backup the conf file
        cp -f "$krb5File" "$backupDir"

        # centrify should unlock krb5file first
        chattr -i $krb5File
    
        #1. set the default_ccache_name    
        confCCacheName

        #2. add rules about auth_to_local
        auto_to_local="auth_to_local = RULE:[1:\$1@\$0]"

        #3. conf pkinit_kdc_hostname
        confPKinitKDC
        if [ "$joinDomainWay" == "pbis" ]; then
            kdc_name="kdc = $kdc_hostname"
        fi

        #4. Get pkcs11 lib path
        getIdDir

        #5. Get certificate path
        getCertPath

        #6. conf pkinit_eku_checking
        pkinit_eku_checking="pkinit_eku_checking = kpServerAuth"
        #echo "  set $pkinit_eku_checking"

        #7. conf pkinit_cert_match
        pkinit_cert_match="pkinit_cert_match = ||<EKU>msScLogin,<KU>digitalSignature"

        # delete the older and write the rew
        sed -i "/^ *auth_to_local*/d" $krb5File
        sed -i "/^ *pkinit*/d" $krb5File
        
        if [ "$osPlatform" == "Ubuntu" ] || [ "$osPlatform" == "Debian" ] || [ "$osPlatform" == "SUSE" ]; then
            sed -i "/^\s*kdc\s*=\s*$kdc_hostname/a \ $auto_to_local\n $pkinit_kdc_hostname\n $pkinit_identities\n $pkinit_anchors\n $pkinit_eku_checking\n $pkinit_cert_match" $krb5File
        else # Redhat && CentOS
	        if [ "$joinDomainWay" == "pbis" ]; then
                sed -i "/^\s*$default_realm\s*=\s*{/a \ $kdc_name\n $auto_to_local\n $pkinit_kdc_hostname\n $pkinit_anchors\n $pkinit_eku_checking\n $pkinit_cert_match" $krb5File
	        else
                sed -i "/^\s*kdc\s*=\s*$kdc_hostname/a \ $auto_to_local\n $pkinit_kdc_hostname\n $pkinit_anchors\n $pkinit_eku_checking\n $pkinit_cert_match" $krb5File
	        fi
        fi

        # centrify should lock krb5file after configuration
        if [ "$joinDomainWay" == "centrify" ]; then
            chattr +i $krb5File
        fi
    else
        get_str CTXFASCFG_NOT_EXIST "${krb5File}"
        exit 4
    fi

    get_str CTXFASCFG_CFG_FINISH "${krb5File}"
}

function confCCacheName()
{
    default_ccache_name="default_ccache_name = FILE:/tmp/krb5cc_%{uid}"
    
    # winbind integration change the default_ccache_name
    if [ "$joinDomainWay" == "winbind" ]; then
        default_ccache_name="default_ccache_name = FILE:/tmp/krb5cc_%{uid}_XXXXXX"
    fi

    sed -i '/^\s*default_ccache_name*/d' $krb5File
    sed -i "/\[libdefaults\]/a \ $default_ccache_name" $krb5File

    #echo "  set $default_ccache_name"
}

function confPKinitKDC()
{
    if [ "$joinDomainWay" == "pbis" ]; then
        # Get default realm
        default_realm=`cat $krb5File | grep '^\s*default_realm\s*='`
        default_realm=${default_realm/*default_realm}
        default_realm=${default_realm/*=}
        default_realm=${default_realm/* }
        # Get KDC hostname
        if [[ -n "${CTX_FAS_KDC_HOSTNAME}" ]]; then
            kdc_hostname="${CTX_FAS_KDC_HOSTNAME}"
        else
            get_str CTXFASCFG_CFG_KDC_HOSTNAME
            read -r kdc_hostname
        fi
    else
        # Get default realm
        default_realm=`cat $krb5File | grep '^\s*default_realm\s*='`
        default_realm=${default_realm/*=} # remove the prefix
        default_realm=${default_realm/* } # remove the blank prefix

        # Get realm block
        realm_block=`sed -n "/^\s*$default_realm\s*=\s*{/,/kdc\s*=/p" $krb5File`

        # Get kdc host name
        if [[ -n "${CTX_FAS_KDC_HOSTNAME}" ]]; then
            kdc_hostname="${CTX_FAS_KDC_HOSTNAME}"
        else
            kdc_hostname=${realm_block/*kdc}    
            kdc_hostname=${kdc_hostname/*=} # remove the prefix
            kdc_hostname=${kdc_hostname/* } # remove the blank suffix
            kdc_hostname=${kdc_hostname/:*} # remove the suffix
        fi

    fi

    # Get pkinit KDC hostname
    if [[ -n "${CTX_FAS_PKINIT_KDC_HOSTNAME}" ]]; then
        pkinit_kdc_hostname="${CTX_FAS_PKINIT_KDC_HOSTNAME}"
    else
        pkinit_kdc_hostname=$kdc_hostname
        # SUSE pkinit_kdc_hostname = HOST.domain.local
        if [ "$osPlatform" == "SUSE" ]; then
            hostname=${pkinit_kdc_hostname%%\.*}     # the first '.' before
            hostname=$(echo "$hostname" | tr '[:lower:]' '[:upper:]') # change the realm to upper case
            realm=${pkinit_kdc_hostname#*\.}         # the first '.' after
            realm=$(echo "$realm" | tr '[:upper:]' '[:lower:]') # change the realm to lower case
            pkinit_kdc_hostname=$(printf "%s%s%s" "$hostname" "." "$realm") #  strcat
        fi
    fi

    pkinit_kdc_hostname="pkinit_kdc_hostname = $pkinit_kdc_hostname"

    #echo "  set $pkinit_kdc_hostname"
}

# get pkcs11 lib dir
function getIdDir()
{
    case "$osPlatform" in
        "Ubuntu"|"Debian")
            pkcs11Lib=/usr/lib/x86_64-linux-gnu/libctxpkcs11.so
            ;;
        "CentOS"|"Redhat"|"Rocky")
            pkcs11Lib=/usr/lib64/libctxpkcs11.so
            ;;
        "SUSE")
            pkcs11Lib=/usr/lib64/libctxpkcs11.so
            ;;
        *)
            get_str CTXFASCFG_UNSUPPORTED_PLATFORM
            exit 2
            ;;    
    esac

    pkinit_identities="pkinit_identities = PKCS11:$pkcs11Lib"

    #echo "  set $pkinit_identities"
}

# get certificates path
function getCertPath()
{
    local certPathExp="/etc/pki/CA/certs/"
    if [[ -n "${CTX_FAS_CERT_PATH}" ]]; then
        certPath="${CTX_FAS_CERT_PATH}"
    else
        get_str CTXFASCFG_SPECIFY_CERT_PATH ${certPathExp}
        read certPath
    fi
        while [[ ! -d "${certPath}" ]]; do
            get_str CTXFASCFG_PATH_ERR
            get_str CTXFASCFG_SPECIFY_CERT_PATH ${certPathExp}
            read certPath
        done

    pkinit_anchors="pkinit_anchors = DIR:${certPath}"
        
    #echo "  set $pkinit_anchors"
}

# configure pam conf
function configPAM()
{
    if [ -f "$pamFile" ]; then
        # backup the conf file
        cp -f "$pamFile" "$backupDir"

        case "$osPlatform" in
            "Ubuntu")
                confUbuntuPam
                ;;
            "Debian")
                confDebianPam
                ;;
            "Redhat")
                confRhelPam
                ;;
            "CentOS")
                confCentOSPam
                ;;
            "Rocky")
                confRockyPam
                ;;
            "SUSE")
                confSusePam
                ;;
            *)
                get_str CTXFASCFG_UNSUPPORTED_PLATFORM
                exit 2
                ;;
        esac
    else
        get_str CTXFASCFG_FAILED
        get_str CTXFASCFG_NOT_EXIST "${pamFile}"
        exit 4
    fi

    get_str CTXFASCFG_SUCCESS
}

function confUbuntuPam()
{
    # winbind
    if [ "$joinDomainWay" == "winbind" ]; then
        if [ -f "$smbFile" ]; then
            winbind_use_default_domain=`cat "$smbFile" | grep "winbind use default domain"`
            if [[ "$winbind_use_default_domain" == *"yes"* || "$winbind_use_default_domain" == *"true"* ]]; then
                sed -i "/^ *$winbind_use_default_domain*/d" $smbFile     
            fi                
        else
            get_str CTXFASCFG_NOT_EXIST "${smbFile}"
            exit 4
        fi
    # sssd
    elif [ "$joinDomainWay" == "sssd" ]; then
        if [ -f "$sssdFile" ]; then
            ad_gpo_map_remote_interactive=`cat "$sssdFile" | grep "ad_gpo_map_remote_interactive"`
            if [ -n "$ad_gpo_map_remote_interactive" ]; then
                nonexist=`echo "$ad_gpo_map_remote_interactive" | grep "ctxfas"`
                if [ -z "$nonexist" ]; then
                    sed -i "/$ad_gpo_map_remote_interactive/ s/$/,+ctxfas/" $sssdFile
                fi
            fi
        else
            get_str CTXFASCFG_NOT_EXIST "${sssdFile}"
            exit 4
        fi
    fi

    # conf the ctxfas
    cat /dev/null > $pamFile
    echo "#Linux VDA Federated Authentication#">>"$pamFile" 
    echo "#%PAM-1.0">>"$pamFile" 
    echo "#pam auth">>"$pamFile"
    echo "auth        sufficient    pam_krb5.so">>"$pamFile"
    echo "@include    common-auth">>"$pamFile"
    echo "#pam account">>"$pamFile"
    echo "account     sufficient    pam_krb5.so">>"$pamFile"
    echo "@include    common-account">>"$pamFile"
    echo "#pam password">>"$pamFile"
    echo "password    sufficient    pam_krb5.so">>"$pamFile"
    echo "@include    common-password">>"$pamFile"
    echo "#pam session">>"$pamFile"
    echo "session     optional      pam_krb5.so">>"$pamFile"
    echo "@include    common-session">>"$pamFile"
}

function confDebianPam()
{
    confUbuntuPam;
}

function confRhelPam()
{
    # centrify
    if [ "$joinDomainWay" == "centrify" ]; then
        cat /dev/null > $pamFile
        echo "#Linux VDA Federated Authentication#">>"$pamFile" 
        echo "#%PAM-1.0">>"$pamFile" 
        echo "#pam auth">>"$pamFile"
        echo "auth        sufficient     pam_krb5.so preauth_options=X509_user_identity=PKCS11:/usr/lib64/libctxpkcs11.so">>"$pamFile"
        echo "auth        include        remote">>"$pamFile"
        echo "#pam account">>"$pamFile"
        echo "account     required       pam_krb5.so">>"$pamFile"
        echo "account     required       pam_nologin.so">>"$pamFile"
        echo "#pam password">>"$pamFile"
        echo "password    sufficient     pam_krb5.so">>"$pamFile"
        echo "password    include        remote">>"$pamFile"
        echo "#pam session">>"$pamFile"
        echo "session     optional       pam_krb5.so">>"$pamFile"
        echo "session     include        remote">>"$pamFile"
    else
        cat /dev/null > $pamFile
        echo "#Linux VDA Federated Authentication#">>"$pamFile" 
        echo "#%PAM-1.0">>"$pamFile" 
        echo "#pam auth">>"$pamFile"
        echo "auth        sufficient     pam_krb5.so preauth_options=X509_user_identity=PKCS11:/usr/lib64/libctxpkcs11.so">>"$pamFile"
        echo "auth        include        remote">>"$pamFile"
        echo "#pam account">>"$pamFile"
        echo "account     sufficient     pam_krb5.so">>"$pamFile"
        echo "account     include        remote">>"$pamFile"
        echo "#pam password">>"$pamFile"
        echo "password    sufficient     pam_krb5.so">>"$pamFile"
        echo "password    include        remote">>"$pamFile"
        echo "#pam session">>"$pamFile"
        echo "session     optional       pam_krb5.so">>"$pamFile"
        echo "session     include        remote">>"$pamFile"
    fi
}

function confCentOSPam()
{
    confRhelPam;
}

function confRockyPam()
{
    confRhelPam;
}

function confSusePam()
{
    # centrify should unlock krb5file first
    if [ "$joinDomainWay" == "centrify" ]; then
        chattr -i $krb5File
    fi

    # change the default ccname template
    appdefaults=`cat $krb5File | grep '^\s*\[appdefaults\]*'`
    if [ -z "$appdefaults" ]; then
        echo "[appdefaults]
pam = {
        ccname_template = FILE:/tmp/krb5cc_%U
}">>"$krb5File"
    else
        ccname_template="ccname_template = FILE:/tmp/krb5cc_%U"
        sed -i "/^ *ccname_template*/d" $krb5File
        sed -i "/^\s*pam\s*=\s*{/a \        $ccname_template" $krb5File
    fi

    # centrify should lock krb5file after configuration
    if [ "$joinDomainWay" == "centrify" ]; then
        chattr +i $krb5File
    fi

    # config the ctxfas
    cat /dev/null > $pamFile
    echo "#Linux VDA Federated Authentication#">>"$pamFile" 
    echo "#%PAM-1.0">>"$pamFile" 
    echo "#pam auth">>"$pamFile"
    echo "auth        sufficient     pam_krb5.so">>"$pamFile"
    echo "auth        include        commmon-auth">>"$pamFile"
    echo "#pam account">>"$pamFile"
    echo "account     sufficient     pam_krb5.so">>"$pamFile"
    echo "account     include        common-account">>"$pamFile"
    echo "#pam password">>"$pamFile"
    echo "password    sufficient     pam_krb5.so">>"$pamFile"
    echo "password    include        common-password">>"$pamFile"
    echo "#pam session">>"$pamFile"
    echo "session     optional       pam_krb5.so">>"$pamFile"
    echo "session     include        common-session">>"$pamFile"
}

# restore the bakcup files
function backout()
{
    get_str CTXFASCFG_RESTORE_CONFIGURATION

    for file in ${backupDir}/*; do
        targetDir=""
        case "$file" in
            krb5.conf)
                targetDir="/etc/"
                ;;
            ctxfas)
                targetDir="/etc/pam.d/"
                ;;
            sssd.conf)
                targetDir="/etc/sssd/"
                ;;
            smb.conf)
                targetDir="/etc/samba/"
                ;;
            *)
                get_str CTXFASCFG_SHOULD_NOT_RESTORE ${file} 
                ;;
        esac

        # cp file 
        cp -f "${backupDir}""/""$file" "$targetDir"
        if [[ "$?" -ne "0" ]]; then
            get_str CTXFASCFG_RESTORE_FAIL ${file}
        fi
    done
}

########################### main process block #################################
main()
{
    get_str CTXFASCFG_BEGIN_INFO "${fname}"

    # Check if the user is an root
    checkUser

    # backup the file 
    createLogFile    
    
    #1. check platform
    get_str CTXFASCFG_PLATFORM_PRECHECK 1
    checkOS

    #2. get the ad join way
    get_str CTXFASCFG_AD_WAY_GET 2
    getADJoinWay

    #3. install dependent packages
    get_str CTXFASCFG_INSTALL_PACKAGES 3
    installPkgs

    #4. Configure krb5
    get_str CTXFASCFG_CFG_KRB5 4
    configKrb5

    #5. Configure pam conf
    get_str CTXFASCFG_CFG_PAM 5
    configPAM    

    get_str CTXFASCFG_FINISH_SUCCESS_INFO "${fname}"
}

main "$@"
