#!/bin/bash

################################################################################
#
# Citrix Virtual Apps & Desktops For Linux Script: Machine Creation Service
# Copyright (c) Citrix Systems, Inc. All Rights Reserved.
#

source /var/xdl/mcs/mcs_util.sh

fname=$(basename $0)
scriptName=${fname%.*}  # Script name without extension
logFile="/var/log/$scriptName.log"

dbus_service_file_path="/usr/lib/systemd/system/dbus.service"

## These pkgs need to be installed in order for LVDA pkg to work
## ntp and winbind related
common_pkgList=("redhat-lsb-core"\
             "nautilus"\
             "totem-nautilus"\
             "brasero-nautilus"\
             "authconfig-gtk"\
             "pulseaudio"\
             "pulseaudio-module-x11"\
             "pulseaudio-gdm-hooks"\
             "pulseaudio-module-bluetooth"\
             "alsa-plugins-pulseaudio"\
             "pciutils"\
             "openssh"\
             "openssh-clients"\
             "java-11-openjdk"\
             "chrony"\
             "krb5-workstation"\
             "authconfig"\
             "oddjob-mkhomedir")
common_pkgListNum=${#common_pkgList[@]}

winbind_pkgList=("samba-winbind"\
             "samba-winbind-clients")
winbind_pkgListNum=${#winbind_pkgList[@]}

sssd_pkgList=("sssd"\
             "samba-common-tools")
sssd_pkgListNum=${#sssd_pkgList[@]}

centrify_pkgList=("samba"\
             "samba-winbind")
centrify_pkgListNum=${#centrify_pkgList[@]}

pkgMcs=("tdb-tools"\
         "ntfs-3g")
pkgMcsNum=${#pkgMcs[@]}

function install_lvda_common_pkgs()
{
    log "Debug: Enter install_lvda_common_pkgs"
    #Install the common pakages
    info="Installing Linux VDA dependency packages ..."
    log_echo "$info"
    for((i=0;i<common_pkgListNum;i++)); do
        info="Installing package ${common_pkgList[$i]}"
        log_echo "$info"
        yum -y install "${common_pkgList[$i]}"  >> "$logFile" 2>&1
        if [[ "$?" -ne "0" ]]; then
            info="Failed to install ${common_pkgList[$i]}"
            yum info "${common_pkgList[$i]}"  >> "$logFile" 2>&1
            [[ "$?" -ne "0" ]] && log_echo "$info"
        fi
    done

    # start PostgreSQL
    info="Starting PostgreSQL database ..."
    log_echo "$info"

    postgresql-setup initdb  >> "$logFile" 2>&1
    /usr/bin/systemctl enable postgresql.service   >> "$logFile" 2>&1
    /usr/bin/systemctl start postgresql   >> "$logFile" 2>&1

    sudo -u postgres psql -c 'show data_directory'  >> "$logFile" 2>&1
    log "Debug: Exit install_lvda_common_pkgs"

}

function install_lvda_winbind_dependency_pkgs()
{
    log "Debug: Enter install_lvda_winbind_dependency_pkgs"
    # Install winbind dependency packages
    info="Installing winbind dependency packages ..."
    log_echo "$info"
    for((i=0;i<winbind_pkgListNum;i++)); do
        info="Installing package ${winbind_pkgList[$i]}"
        log_echo "$info"
        yum -y install "${winbind_pkgList[$i]}"  >> "$logFile" 2>&1
        if [[ "$?" -ne "0" ]]; then
            info="Failed to install ${winbind_pkgList[$i]}"
            yum info "${winbind_pkgList[$i]}"  >> "$logFile" 2>&1
            [[ "$?" -ne "0" ]] && log_echo "$info"
        fi
    done

    # enable winbind service
    /usr/bin/systemctl enable winbind.service  >> "$logFile" 2>&1

    log "Debug: Exit install_lvda_winbind_dependency_pkgs"
}

function install_lvda_sssd_dependency_pkgs()
{
    log "Debug: Enter install_lvda_sssd_dependency_pkgs"
    # Install the common packages
    info="Installing Linux VDA sssd dependency packages ..."
    log_echo "$info"
    for((i=0;i<sssd_pkgListNum;i++)); do
        info="Installing package ${sssd_pkgList[$i]}"
        log_echo "$info"
        yum -y install "${sssd_pkgList[$i]}"  >> "$logFile" 2>&1
        if [[ "$?" -ne "0" ]]; then
            info="Failed to install ${sssd_pkgList[$i]}"
            yum info "${sssd_pkgList[$i]}"  >> "$logFile" 2>&1
            [[ "$?" -ne "0" ]] && log_echo "$info"
        fi
    done

    # enable sssd service
    /usr/bin/systemctl enable sssd.service  >> "$logFile" 2>&1

    log "Debug: Exit install_lvda_dependency_pkgs"
}


function install_lvda_centrify_samba_dependency_pkgs() 
{
    log "Debug: Enter install_lvda_centrify_samba_dependency_pkgs"

    for((i=0;i<centrify_pkgListNum;i++)); do
        info="Installing package ${centrify_pkgList[$i]}"
        log_echo "$info"
        yum -y install "${centrify_pkgList[$i]}"  >> "$logFile" 2>&1
        if [[ "$?" -ne "0" ]]; then
            info="Failed to install ${centrify_pkgList[$i]}"
            yum info "${centrify_pkgList[$i]}"  >> "$logFile" 2>&1
            [[ "$?" -ne "0" ]] && log_echo "$info"
        fi
    done

    log "Debug: Exit install_lvda_centrify_samba_dependency_pkgs"
}

function install_mcs_pkgs()
{
    log "Debug: Enter install_mcs_pkgs"
    for ((i=0;i<pkgMcsNum;i++)); do
        info="Installing package ${pkgMcs[$i]}"
        log_echo "$info"
        yum -y install "${pkgMcs[$i]}"  >> "$logFile" 2>&1

        if [[ "$?" -ne "0" ]]; then
            log_echo "Error: install ${pkgMcs[$i]} failed."
        fi
    done

    log "Debug: Exit install_mcs_pkgs"
}

#
# Configure DNS
#
function conf_dns()
{
    log "Debug: Enter funtion conf_dns"
    [[ -z "$dns1" && -z "$dns2" && -z "$dns3" && -z "$dns4" ]] && \
            log_echo "ERROR: Exit funtion conf_dns, dns not configured, please do it manually." && \
            return

    conf_resolv_conf

    # Disable Multicast DNS
    dir="/etc/sysconfig/network-scripts"
    FILELIST=`ls $dir/ifcfg-eth* 2>>"$logFile" | grep -v "\."`
    #In case of Predictable Network Interface Names(from Systemd v197 )
    [[ -z $FILELIST ]] && FILELIST=`ls $dir/ifcfg-en* 2>>"$logFile" | grep -v "\."`
    [[ -z $FILELIST ]] && log_echo "ERROR: Network Interface configuration file does not exist!" && \
                          log_echo "ERROR: Exit funtion conf_dns, dns not configured, please do it manually." && \
                          return
    for file in $FILELIST
    do
        `sed -i '/^PEERDNS=*/s/^/#/' $file`
        `sed -i '/^DNS./s/^/#/' "$file"`
        if [[ -f "$file" ]]; then
            echo "PEERDNS=no">>"$file"           
            [[ -n "$dns1" ]] && echo "DNS1=$dns1">>"$file"
            [[ -n "$dns2" ]] && echo "DNS2=$dns2">>"$file"
            [[ -n "$dns3" ]] && echo "DNS3=$dns3">>"$file"
            [[ -n "$dns4" ]] && echo "DNS4=$dns4">>"$file"
        fi
    done 

    # on RHEL workstation, service NetworkManager will automatically fetch DHCP DNS 
    # and insert into /etc/resolv.conf, this will block DNS search, in order to solve this issue, 
    # we create a new /etc/resolv.conf.custom file and put the correct DNS entries to it, when
    # NetworkManager restarted, we will use this file to overwrite /etc/resolv.conf.
    customFile="/etc/resolv.conf.custom"           
    [[ -n "$dns1" ]] && echo "nameserver $dns1">"$customFile"
    [[ -n "$dns2" ]] && echo "nameserver $dns2">>"$customFile"
    [[ -n "$dns3" ]] && echo "nameserver $dns3">>"$customFile"
    [[ -n "$dns4" ]] && echo "nameserver $dns4">>"$customFile"
    copyFile="/etc/NetworkManager/dispatcher.d/15-resolv"
    echo "#!/bin/bash
    #
    # Description : script to override default resolv.conf file
    # with customized file.
    cp -f /etc/resolv.conf.custom /etc/resolv.conf">"$copyFile"
    chmod +x "$copyFile"

    log "Debug: Exit funtion conf_dns"
}

function main()
{
    log "Debug: Enter main"
    check_user
    create_log_file
    install_lvda_common_pkgs
    install_mcs_pkgs
    install_mcs_scripts
    # For NDJ mode, we need to create local mapped account group
    create_localuser_group

    hack_dbus_service_file $dbus_service_file_path
    if [ "$AD_INTEGRATION" == "winbind" ]; then
        install_lvda_winbind_dependency_pkgs
    elif [ "$AD_INTEGRATION" == "sssd" ]; then
        install_lvda_sssd_dependency_pkgs
    elif [ "$AD_INTEGRATION" == "centrify" ]; then
        install_lvda_centrify_dependency_pkgs
        install_lvda_centrify_samba_dependency_pkgs
        install_lvda_centrify_samba_integration_pkgs
    elif [ "$AD_INTEGRATION" == "pbis" ]; then
        install_lvda_pbis_dependency_pkgs
    fi
    conf_dns
    log "Debug: Exit main"
}

main "$@"
