#!/bin/bash

###############################################################################
#
# Citrix Virtual Apps & Desktops For Linux Power Script: Power action callbacks
#
# Copyright (c) Citrix Systems, Inc. All Rights Reserved.
#

# Timeout period in seconds to wait before returning from this script
VDA_WAIT_TIMEOUT=30
# Sleep interval to wait before checking vda_registered() status again
SLEEP_INTERVAL=1

# Function: notifyCtxVDA
#  Sends the requested power action to the VDA
function notifyCtxVDA {
PM_STATE="$1" /usr/bin/python - <<END
#!/usr/bin/python
import os
import socket
import sys

def sendMessage(message):
    try:
        pmSock = socket.socket(socket.AF_UNIX, socket.SOCK_STREAM)
    except socket.error, msg:
        print >>sys.stderr, 'Error: Unable to create socket. Details: %s' % msg
        pmSock = None
        sys.exit(1)
    ctxvdapm_addr = '/var/xdl/.ctxvdapm'
    
    try:
        pmSock.connect(ctxvdapm_addr)
    except socket.error, msg:
        print >>sys.stderr, 'Error: Unable to connect to socket. Details: %s' % msg
        pmSock.close()
        pmSock = None
        sys.exit(1)

    try:
        pmDataTotal = 0
        while pmDataTotal < len(message):
            pmDataSent = pmSock.send(message[pmDataTotal:])
            if pmDataSent == 0:
                raise RuntimeError('Error: Broken socket to ctxvda')
            pmDataTotal = pmDataTotal + pmDataSent
    finally:
        pmSock.close()
        pmSock = None

def invalidMessage():
    print >>sys.stderr, 'Error: Unsupported PM-Utils state: %s' % os.environ['PM_STATE']

ctxvdapm_action = {
    'suspend'           : 'ctxvdapmsleep\0',
    'hibernate'         : 'ctxvdapmhibernate\0',
    'resume'            : 'ctxvdapmpoweron\0',
    'thaw'              : 'ctxvdapmpoweron\0',
    'pre/suspend'       : 'ctxvdapmsleep\0',
    'pre/hibernate'     : 'ctxvdapmhibernate\0',
    'pre/hybrid-sleep'  : 'ctxvdapmhibernate\0',
    'post/suspend'      : 'ctxvdapmpoweron\0',
    'post/hibernate'    : 'ctxvdapmpoweron\0',
    'post/hybrid-sleep' : 'ctxvdapmpoweron\0'
};

try: 
  sendMessage(ctxvdapm_action[os.environ['PM_STATE']]);
except KeyError:
  invalidMessage;
                   
END
}

# Function: vda_registered
#  Determines if the VDA is in a "Registered" state
function vda_registered() {
    [[ $(/opt/Citrix/VDA/bin/ctxreg list -k "HKLM\Software\Citrix\VirtualDesktopAgent\State" | grep "Registered") ]]
}

# Function: wait_for_vda
#  Spin until the power action sleep/hibernate is complete
#  or the timeout occurs
function wait_for_vda() {
    let elapsed=0
    while vda_registered && [ $elapsed -lt $VDA_WAIT_TIMEOUT ]; do
        let elapsed=$(($elapsed + $SLEEP_INTERVAL))
        sleep $SLEEP_INTERVAL;
    done
    if [ $elapsed -eq $VDA_WAIT_TIMEOUT ]; then
        echo "$0 - Timed out waiting for power action to complete. `date`" 
    fi
}

# Log start of power action
echo "$0 - Begin Power Action: `date`"

# Process requested power action
if [ "$1" == "pre" ] || [ "$1" == "post" ]; then
    echo "$0 - '$1/$2 initiated for ctxvda: `date`"
    notifyCtxVDA $1/$2
else
    echo "$0 - '$1' initiated for ctxvda: `date`"
    notifyCtxVDA $1
fi

# Determine if this script should wait for the power action to complete
case $1 in
    pre|suspend|hibernate)
        wait_for_vda
        ;;
esac

echo "$0 - Completed Power Action: `date`"
# End of script
