#!/bin/bash

function postgresql_update_config {
    function inner {
        export PGHBADIR=$(dirname $(su - postgres -c "psql -v ON_ERROR_STOP=1 -c \"show hba_file;\" | grep \"^[ ]*/\" | tr -d ' '")) || return 1

        su - postgres -c "cp $PGHBADIR/pg_hba.conf $PGHBADIR/pg_hba.conf.old" || return 1
        su - postgres -c "cp /var/xdl/pg_hba.conf  $PGHBADIR/pg_hba.conf" || return 1
    }
    cmdlog_exec inner $FUNCNAME "update PostgreSQL configuration"
}

function postgresql_restore_config {
    function inner {
        export PGHBADIR=$(dirname $(su - postgres -c "psql -v ON_ERROR_STOP=1 -c \"show hba_file;\" | grep \"^[ ]*/\" | tr -d ' '")) || return 1

        su - postgres -c "cp $PGHBADIR/pg_hba.conf.old $PGHBADIR/pg_hba.conf" || return 1
        su - postgres -c "rm $PGHBADIR/pg_hba.conf.old" || return 1
    }
    cmdlog_exec inner $FUNCNAME "restore original PostgreSQL configuration"
}

function confdb_generate_passwd {
    function inner {
        # Remove VDA configuration and create database scripts which contain old password
        rm -f /etc/xdl/ctx-jproxy.conf || return 1
        umask 0077 || return 1
        touch /etc/xdl/ctx-jproxy.conf || return 1
        chmod 0600 /etc/xdl/ctx-jproxy.conf || return 1

        rm -f /var/xdl/create-database.sql || return 1
        touch /var/xdl/create-database.sql || return 1
        chown postgres:postgres /var/xdl/create-database.sql || return 1
        chmod 0600 /var/xdl/create-database.sql || return 1

        # Generate a new password for PostgreSQL ctxvda user
        confDB_PWD=0
        generate_time=1000
        while [ ${#confDB_PWD} != 32 ] && [ "$generate_time" -gt "0" ]
        do 
            export confDB_PWD=$(head -c 1000 < /dev/urandom | LC_ALL=C tr -dc '_A-Za-z0-9' | head -c32)
            generate_time=$((generate_time -= 1))
        done

        if [ ${#confDB_PWD} != 32 ]; then
            echo "Generate more than 1000 times, still cannot get valid password, please retry the command."
            return 1;
        fi

        # Generate new VDA configuration and create database scripts with new password. Ensure least privilege.
        sed -r '/ConfigDbPasswd=/s/".*"/"'"$confDB_PWD"'"/' /etc/xdl/ctx-jproxy.conf.tmpl > /etc/xdl/ctx-jproxy.conf || return 1
        sed 's/{CONFDB_PWD}/'"$confDB_PWD"'/g' /var/xdl/create-database.sql.tmpl > /var/xdl/create-database.sql || return 1
        export confDB_PWD= || return 1
        chmod 0400 /var/xdl/create-database.sql || return 1
    }
    cmdlog_exec inner $FUNCNAME "generate ConfDB password"
}

function confdb_update_conf_file
 {
    function inner {
        confDB_PWD=`awk '/^ConfigDbPasswd/' /etc/xdl/ctx-vda.conf |awk -F '"' '{print $2}'`
        sed -r '/ConfigDbPasswd=/s/".*"/"'"$confDB_PWD"'"/' /etc/xdl/ctx-jproxy.conf.tmpl > /etc/xdl/ctx-jproxy.conf || return 1
        rm -f /etc/xdl/ctx-vda.conf
    }
    cmdlog_exec inner $FUNCNAME "copy ConfDB password to new jproxy conf file"
}

function confdb_create_database {
    function inner {
        export confDB_Exists=$(su - postgres -c "psql -l | grep -w citrix-confdb | wc -l") || return 1
        if [ "$confDB_Exists" != "0" ] ; then echo "Database already exists." ; return 1 ; fi || return 1
        su - postgres -c "psql -v ON_ERROR_STOP=1 -q --file=/var/xdl/create-database.sql" || return 1
    }
    cmdlog_exec inner $FUNCNAME "create ConfDB database"
}

function confdb_populate_database {
    function inner {
        /opt/Citrix/VDA/bin/ctxreg load -i /var/xdl/ctxhdx.reg || return 1
        /opt/Citrix/VDA/bin/ctxreg load -i /var/xdl/ctxvda.reg || return 1
    }
    cmdlog_exec inner $FUNCNAME "populate ConfDB database"
}

function confdb_drop_database {
    function inner {
        su - postgres -c "psql -v ON_ERROR_STOP=1 -q --file=/var/xdl/drop-database.sql" || return 1
    }
    cmdlog_exec inner $FUNCNAME "drop ConfDB database"
}

function confdb_update_database {
    function inner {
        /opt/Citrix/VDA/bin/ctxreg load -i /var/xdl/ctxhdx_upgrade.reg || return 1
        /opt/Citrix/VDA/bin/ctxreg load -i /var/xdl/ctxvda_upgrade.reg || return 1
    }
    cmdlog_exec inner $FUNCNAME "update ConfDB database from V11 to V12"
}

function cleanup_temp_files {
    function inner {
        rm -f /etc/xdl/ctx-jproxy.conf || return 1
        rm -f /var/xdl/create-database.sql || return 1

        rm -f /tmp/citrix.dbg || return 1
        rm -f /dev/shm/ctx_hdx_xorg_x11_sync* || return 1

        rm -f /tmp/ctxaudioctl* || return 1
        rm -f /dev/shm/ctx_hdx_audio_* || return 1

        rm -f /tmp/hdx.*.log*

        find /tmp -maxdepth 1 -name ".X*-lock" -user root -group ctxadm -print0 | xargs -0 rm -f || return 1
        find /tmp/.X11-unix -name "X*" -user root -group ctxadm -print0 | xargs -0 rm -f || return 1
    }
    cmdlog_exec inner $FUNCNAME "clean up temporary files"
}


# Matches a valid hostname.
regex_hostname="(([a-zA-Z](\-*[a-zA-Z0-9]+)*)|([0-9]{1,14}[a-zA-Z]+((\-*[a-zA-Z0-9]+)*)))"

# Matches a valid domain name. 
regex_domainname="([a-zA-Z0-9]+(\-*[a-zA-Z0-9]+)*)"

# Matches a valid FQDN
regex_fqdn="($regex_hostname((\.$regex_domainname)*))"

# Matches a space separated list of valid FQDNs.
regex_fqdn_list="($regex_fqdn(\s+$regex_fqdn)*)"

# Matches a list of FQDNs grouped by parentheses.
regex_fqdn_group="((\(\s*$regex_fqdn_list\s*\))*)"

# Matches a list of grouped FQDNs.
regex_fqdn_groups="($regex_fqdn_group(\s*$regex_fqdn_group)*)"

function ddc_cname_lookup_allow {
    function inner {
        /opt/Citrix/VDA/bin/ctxreg create -k "HKLM\Software\Citrix\VirtualDesktopAgent" -t "REG_DWORD" -v "UseCnameLookup" -d "$1" --force || return 1
    }
    if [ "$1" == "y" ]; then
        cmdlog_exec inner $FUNCNAME "apply registry changes" "1"
    else
        # Although the VDA will default to disallow CName DDCs
        # when the UseCnameLoolup valuename is missing, we
        # have to either explicitly set UseCnameLookup to 0
        # or delete an existing valuename to ensure that any
        # pre-existing value is overwritten.
        cmdlog_exec inner $FUNCNAME "apply registry changes" "0"
    fi
}

# Validate the dotnet runtime path use input is valid
# and dotnet runtime is in the path
function dotnet_rutnime_path_validate {
    if [[ -f "$1/dotnet" ]]; then
        valid=true
    else
        get_str SHELL_COMMON_DOTENT_RUNTIME_PATH_VALIDATE_ERROR
        echo
        valid=false
    fi
}

# Validate the desktop evironment input is valid, if mate is installed
function desktop_environment_validate {
    if [[ $1 == "mate" && ! -f "/usr/share/xsessions/mate.desktop" ]]; then
        get_str SHELL_COMMON_DESKTOP_ENVIRONMENT_VALIDATE_ERROR
        echo
        valid=false
    elif [[ $1 != "mate" && $1 != "gnome" ]]; then
        get_str SHELL_COMMON_DESKTOP_ENVIRONMENT_VALIDATE_ERROR_TWO
        echo
        valid=false
    else
        valid=true
    fi
}

# Validate that DDC list is either a list of valid FQDNs
# or multiple lists of valid FQDNs grouped by parentheses.
function ddc_list_validate {
    if [[ $1 =~ ^$regex_fqdn_groups$ ]]; then
        valid=true
    else
        if [[ $1 =~ ^$regex_fqdn_list$ ]]; then
            valid=true
        else
            get_str SHELL_COMMON_DDC_LIST_VALIDATE_ERROR
            echo
            valid=false
        fi
    fi
}

function ddc_list_update {
    function inner {
        /opt/Citrix/VDA/bin/ctxreg create -k "HKLM\Software\Citrix\VirtualDesktopAgent" -t "REG_SZ" -v "ListOfDDCs" -d "$1" --force || return 1
    }
    cmdlog_exec inner $FUNCNAME "update DDC list" "$1"
}

function vda_port_update {
    function inner {
        /opt/Citrix/VDA/bin/ctxreg create -k "HKLM\Software\Citrix\VirtualDesktopAgent" -t "REG_DWORD" -v "ControllerRegistrarPort" -d "$1" --force || return 1
        /opt/Citrix/VDA/bin/ctxreg create -k "HKLM\Software\Citrix\VirtualDesktopAgent" -t "REG_DWORD" -v "VdaCxfServicesPort" -d "$1" --force || return 1
    }
    cmdlog_exec inner $FUNCNAME "update VDA port" "$1"
}
function telemetry_socket_port_update {
    function inner {
         FILE=/etc/systemd/system/ctxtelemetry.socket
         if [ -f $FILE ]; then
             telemetry_socket_port=$(grep -oP '(?<=ListenStream=)\d+' $FILE)
             sed -i "s/$telemetry_socket_port/$1/g" $FILE|| return 1
         fi
    }
    cmdlog_exec inner $FUNCNAME "update TELEMETRY socket port" "$1"
}
function telemetry_port_update {
    function inner {
        /opt/Citrix/VDA/bin/ctxreg create -k "HKLM\Software\Citrix\VirtualDesktopAgent" -t "REG_DWORD" -v "TelemetryServicePort" -d "$1" --force || return 1
    }
    cmdlog_exec inner $FUNCNAME "update TELEMETRY port" "$1"
}

function ad_integration_use_winbind {
    function inner {
        /opt/Citrix/VDA/bin/ctxreg create -k "HKLM\Software\Citrix\VirtualDesktopAgent" -t "REG_SZ" -v "Krb5Keytab" -d "/etc/krb5.keytab" --force || return 1
        /opt/Citrix/VDA/bin/ctxreg create -k "HKLM\Software\Citrix\VirtualDesktopAgent" -t "REG_SZ" -v "Krb5Conf" -d "/etc/krb5.conf" --force || return 1
        /opt/Citrix/VDA/bin/ctxreg create -k "HKLM\Software\Citrix\VirtualDesktopAgent" -t "REG_SZ" -v "DomainJoinMethod" -d "Winbind" --force || return 1
    }
    cmdlog_exec inner $FUNCNAME "apply registry changes"
}

function ad_integration_use_quest {
    function inner {
        /opt/Citrix/VDA/bin/ctxreg create -k "HKLM\Software\Citrix\VirtualDesktopAgent" -t "REG_SZ" -v "Krb5Keytab" -d "/etc/opt/quest/vas/host.keytab" --force || return 1
        /opt/Citrix/VDA/bin/ctxreg create -k "HKLM\Software\Citrix\VirtualDesktopAgent" -t "REG_SZ" -v "Krb5Conf" -d "/etc/opt/quest/vas/vas.conf" --force || return 1
        /opt/Citrix/VDA/bin/ctxreg create -k "HKLM\Software\Citrix\VirtualDesktopAgent" -t "REG_SZ" -v "DomainJoinMethod" -d "Quest" --force || return 1
    }
    cmdlog_exec inner $FUNCNAME "apply registry changes"
}

function ad_integration_use_centrify {
    function inner {
        /opt/Citrix/VDA/bin/ctxreg create -k "HKLM\Software\Citrix\VirtualDesktopAgent" -t "REG_SZ" -v "Krb5Keytab" -d "/etc/krb5.keytab" --force || return 1
        /opt/Citrix/VDA/bin/ctxreg create -k "HKLM\Software\Citrix\VirtualDesktopAgent" -t "REG_SZ" -v "Krb5Conf" -d "/etc/krb5.conf" --force || return 1
        /opt/Citrix/VDA/bin/ctxreg create -k "HKLM\Software\Citrix\VirtualDesktopAgent" -t "REG_SZ" -v "DomainJoinMethod" -d "Centrify" --force || return 1
    }
    cmdlog_exec inner $FUNCNAME "apply registry changes"
}

function ad_integration_use_sssd {
    function inner {
        /opt/Citrix/VDA/bin/ctxreg create -k "HKLM\Software\Citrix\VirtualDesktopAgent" -t "REG_SZ" -v "Krb5Keytab" -d "/etc/krb5.keytab" --force || return 1
        /opt/Citrix/VDA/bin/ctxreg create -k "HKLM\Software\Citrix\VirtualDesktopAgent" -t "REG_SZ" -v "Krb5Conf" -d "/etc/krb5.conf" --force || return 1
        /opt/Citrix/VDA/bin/ctxreg create -k "HKLM\Software\Citrix\VirtualDesktopAgent" -t "REG_SZ" -v "DomainJoinMethod" -d "SSSD" --force || return 1
    }
    cmdlog_exec inner $FUNCNAME "apply registry changes"
}

function ad_integration_use_pbis {
    function inner {
        /opt/Citrix/VDA/bin/ctxreg create -k "HKLM\Software\Citrix\VirtualDesktopAgent" -t "REG_SZ" -v "Krb5Keytab" -d "/etc/krb5.keytab" --force || return 1
        /opt/Citrix/VDA/bin/ctxreg create -k "HKLM\Software\Citrix\VirtualDesktopAgent" -t "REG_SZ" -v "Krb5Conf" -d "/etc/krb5.conf" --force || return 1
        /opt/Citrix/VDA/bin/ctxreg create -k "HKLM\Software\Citrix\VirtualDesktopAgent" -t "REG_SZ" -v "DomainJoinMethod" -d "PBIS" --force || return 1
    }
    cmdlog_exec inner $FUNCNAME "apply registry changes"
}

function selinux_environment_configure {
    function inner {
        if [ -x /usr/sbin/semodule -a -r /etc/xdl/XenDesktopVDA.pp ]
        then
            /usr/sbin/semodule -n -i /etc/xdl/XenDesktopVDA.pp || return 1
            if /usr/sbin/selinuxenabled
            then
                /usr/sbin/load_policy || return 1
                /sbin/restorecon -F -v /opt/Citrix/VDA/bin/ctxlogin || return 1
                /sbin/restorecon -F -v /usr/lib/cups/backend/cups-ctx
            fi
        fi
    }
    cmdlog_exec inner $FUNCNAME "configure selinux environment"
}

function selinux_environment_restore {
    function inner {
        if [ -x /usr/sbin/semodule ]
        then
            /usr/sbin/semodule -n -r XenDesktopVDA
            if /usr/sbin/selinuxenabled
            then
                /usr/sbin/load_policy
                /sbin/restorecon -F -v /opt/Citrix/VDA/bin/ctxlogin
                /sbin/restorecon -F -v /usr/lib/cups/backend/cups-ctx
            fi
        fi
        return 0 # ignore missing policy module on uninstall
     }
    cmdlog_exec inner $FUNCNAME "configure selinux environment"
}

function audio_remove_config {
    function inner {
        config=`pulseaudio --dump-conf | sed -n 's/^default-script-file\s*=\s*//pg'`
        [ -f "$config" ] && sed -i -e '/^###Citrix Audio Support/d' -e '/^load-module module-ctx-ctl/d' $config || return 1
    }
    cmdlog_exec inner $FUNCNAME "remove audio config"
}

function hdx_3d_pro_configure {
    function inner {
        if [ -f /etc/X11/ctx-nvidia.sh ]; then
            /etc/X11/ctx-nvidia.sh || return 1
        fi
    }
    cmdlog_exec inner $FUNCNAME "configure HDX 3D Pro"
}

function set_hdx_3d_pro {
    function inner {
        /opt/Citrix/VDA/bin/ctxreg create -k "HKLM\System\CurrentControlSet\Control\Citrix" -t "REG_DWORD" -v "3DPro" -d "$1" --force || return 1
    }
    cmdlog_exec inner $FUNCNAME "apply registry changes" "$1"
}

function set_vdi_mode {
    function inner {
        /opt/Citrix/VDA/bin/ctxreg create -k "HKLM\System\CurrentControlSet\Control\Citrix\WinStations\tcp" -t "REG_DWORD" -v "StackSessionMode" -d "$1" --force || return 1
    }
    cmdlog_exec inner $FUNCNAME "apply registry changes" "$1"
}

function set_single_session_mode {
    function inner {
        /opt/Citrix/VDA/bin/ctxreg create -k "HKLM\Software\Citrix\VirtualDesktopAgent" -t "REG_DWORD" -v "ForceSingleSession" -d "$1" --force || return 1
    }
    cmdlog_exec inner $FUNCNAME "apply registry changes" "$1"
}

function set_dotnet_runtime_path {
    function inner {
        /opt/Citrix/VDA/bin/ctxreg create -k "HKLM\Software\Citrix\VirtualDesktopAgent" -t "REG_SZ" -v "DotNetRuntimePath" -d "$1" --force || return 1
    }
    cmdlog_exec inner $FUNCNAME "apply registry changes" "$1"
}

function set_desktop_environment {
    function inner {
        /opt/Citrix/VDA/bin/ctxreg create -k "HKLM\Software\Citrix\VirtualDesktopAgent" -t "REG_SZ" -v "DesktopEnvironment" -d "$1" --force || return 1
    }
    cmdlog_exec inner $FUNCNAME "apply registry changes" "$1"
}

function set_installation_type {
    function inner {
        /opt/Citrix/VDA/bin/ctxreg create -k "HKLM\System\CurrentControlSet\Control\Citrix" -t "REG_SZ" -v "InstallationType" -d "$1" --force || return 1
    }
    cmdlog_exec inner $FUNCNAME "apply registry changes" "$1"
}

function ad_site_name {
    function inner {
        /opt/Citrix/VDA/bin/ctxreg create -k "HKLM\Software\Citrix\VirtualDesktopAgent" -t "REG_SZ" -v "ActiveDirectorySiteName" -d "$1" --force || return 1
    }
    cmdlog_exec inner $FUNCNAME "apply registry changes" "$1"
}

function ldap_server_list {
    function inner {
        /opt/Citrix/VDA/bin/ctxreg create -k "HKLM\Software\Citrix\VirtualDesktopAgent" -t "REG_SZ" -v "ListOfLDAPServers" -d "$1" --force || return 1
    }
    cmdlog_exec inner $FUNCNAME "apply registry changes" "$1"
}

function ldap_search_base {
    function inner {
        /opt/Citrix/VDA/bin/ctxreg create -k "HKLM\Software\Citrix\VirtualDesktopAgent" -t "REG_SZ" -v "LDAPComputerSearchBase" -d "$1" --force || return 1
    }
    cmdlog_exec inner $FUNCNAME "apply registry changes" "$1"
}

function fas_server_list {
    function inner {
        /opt/Citrix/VDA/bin/ctxreg create -k "HKLM\Software\Citrix\VirtualDesktopAgent\Authentication\UserCredentialService" -t "REG_SZ" -v "Addresses" -d "$1" --force || return 1
    }
    cmdlog_exec inner $FUNCNAME "apply registry changes" "$1"
}

function set_smart_card {
    function inner {
        /opt/Citrix/VDA/bin/ctxreg create -k "HKLM\Software\Citrix\VirtualDesktopAgent\SmartLogon" -t "REG_DWORD" -v "SmartCard" -d "$1" --force || return 1
        if [ $1 -eq 1 ]; then
            if [ -f /etc/systemd/system/ctxscardsd.service ]; then
                 /bin/systemctl start ctxscardsd || return 1
            fi
        else
            if [ -f /etc/systemd/system/ctxscardsd.service ]; then
                 /bin/systemctl stop ctxscardsd || return 1
            fi
        fi
    }
    cmdlog_exec inner $FUNCNAME "apply registry changes" "$1"
}

function create_anonymous_group {
    function inner {
        if [ ! $(getent group CtxAnonymous) ]; then
            /usr/sbin/groupadd CtxAnonymous || return 1
        fi
    }
    cmdlog_exec inner $FUNCNAME "create a group for anonymous users: ctxanonymous"
}

function delete_anonymous_group {
    function inner {
        if [ $(getent group CtxAnonymous) ]; then
            for i in {0..99}
            do
                username=$(printf "Anon%03d" $i)
                if [ $(getent passwd $username) ]; then
                    find /tmp -user $username -exec rm -rf {} \;
                    find /var/tmp -user $username -exec rm -rf {} \;
                    /usr/sbin/userdel -rf $username || return 1
                fi
            done
            /usr/sbin/groupdel CtxAnonymous || return 1
        fi
    }
    cmdlog_exec inner $FUNCNAME "delete the group ctxanonymous"
}

function delete_localuser_group {
    function inner {
        if [ $(getent group CitrixLocalUsersGroup) ]; then
            # do we need to remove users first?
            /usr/sbin/groupdel CitrixLocalUsersGroup || return 1
        fi
    }
    cmdlog_exec inner $FUNCNAME "delete the group CitrixLocalUsersGroup"
}

