#!/bin/bash

product_name="Citrix Linux VDA"
short_name="XDL"
linux_virtual_desktop="Linux VDA"
virtual_delivery_agent="Virtual Delivery Agent"

configure_log="/tmp/xdl.configure.log"
lastcmd_log="/tmp/xdl.lastcmd.log"

coreFileRhel="/etc/redhat-release"  
coreFileCent="/etc/centos-release" 
coreFileRocky="/etc/rocky-release"   
coreFileSuse="/etc/SuSE-release"  
coreFileUbuntu="/etc/lsb-release"
coreFilePardus="/etc/lsb-release"
coreFileOS="/etc/os-release"

PYTHON3_PATH="/opt/Citrix/VDA/sbin/ctxpython3"

function get_str {
    /opt/Citrix/VDA/bin/getstr "$@"
}

# Prompt user for a yes or no response.
#   $1: Question to ask.
function prompt_yn {
    result=""
    if [ "$silent" == "y" ]; then
        failed="y"
        return
    fi
    while true; do
        read -p "$1 (y/n): " yn
        case $yn in
            [Yy]* ) result="y"; break;;
            [Nn]* ) result="n"; break;;
            *     ) /opt/Citrix/VDA/bin/getstr SHELL_UTIL_YES_NO;;
        esac
    done
}

# Prompt user for a yes or no response, accepting
# blank answers as a default response.
#   $1: Question to ask.
#   $2: Default response.
function prompt_ynd {
    result=""
    if [ "$silent" == "y" ]; then
        failed="y"
        return
    fi
    while true; do
        read -p "$1 (y/n) [$2]: " yn
        case $yn in
            [Yy]* ) result="y"; break;;
            [Nn]* ) result="n"; break;;
            ""    ) result=$2;  break;;
            *     ) /opt/Citrix/VDA/bin/getstr SHELL_UTIL_YES_NO;;
        esac
    done
}

# Prompt user for a string response.
#   $1: Question to ask.
function prompt_val {
    result=""
    if [ "$silent" == "y" ]; then
        failed="y"
        return
    fi
    while true; do
        read -p "$1 " val
        case "$val" in
            "" ) /opt/Citrix/VDA/bin/getstr SHELL_UTIL_PLS_ENTER_VALUE;;
            *  ) result="$val"; break;;
        esac
    done
}

# Prompt user for a string response, accepting
# blank answers as a default response.
#   $1: Question to ask.
#   $2: Default response.
function prompt_vald {
    result=""
    if [ "$silent" == "y" ]; then
        failed="y"
        return
    fi
    while true; do
        read -p "$1 [$2]: " val
        case "$val" in
            "" ) result="$2";   break;;
            *  ) result="$val"; break;;
        esac
    done
}

# Prompt user for a string response then
# pass the result to a callback to validate.
#   $1: Question to ask.
#   $2: Validatation callback.
function prompt_valv {
    result=""
    if [ "$silent" == "y" ]; then
        failed="y"
        return
    fi
    while true; do
        read -p "$1 " val
        if [ ! -z "$val" ]; then
            $2 "$val"
            if [ "$valid" == true ]; then
                result="$val"
                break
            fi
        fi
        get_str SHELL_UTIL_PLS_ENTER_VALUE
    done
}

# Prompt user for a string response, accepting
# empty answers as valid.
#   $1: Question to ask.
function prompt_vale {
    result=""
    if [ "$silent" == "y" ]; then
        return
    fi
    read -p "$1 [<none>]: " result
}

# Prompt user for a string response 
# pass empty answers as valid. If not empty,
# pass the result to a callback to validate.
#   $1: Question to ask.
#   $2: Default value if empty
#   $3: Validatation callback.
function prompt_valve {
    result=""
    if [ "$silent" == "y" ]; then
        return
    fi
    while true; do
        read -p "$1 " val
        case $val in
            ""    ) result=$2;;
            *     ) result=$val;;
        esac
        $3 "$result"
        if [ "$valid" == true ]; then
            break
        fi
        get_str SHELL_UTIL_PLS_ENTER_VALUE
    done
}

# Prompt user for a numbered response.
#   $1:     Question to ask.
#   $2:     Number of options.
#   $3..$n: Description of options.
function prompt_num {
    result=""
    if [ "$silent" == "y" ]; then
        failed="y"
        return
    fi
    while true; do
        echo "$1"
        for i in `seq 1 $2`; do
            let opt=i+2
            eval opt=\$$opt
            echo "  $i: $opt"
        done
        promp_msg=`/opt/Citrix/VDA/bin/getstr SHELL_UTIL_CHOOSE_FROM_ABOVE $2`
        read -p "$promp_msg" val
        if [[ $val =~ ^[0-9]+$ ]] && [ $val -le $2 ] && [ $val -ge 1 ]; then
            result=$val
            break;
        fi
        /opt/Citrix/VDA/bin/getstr SHELL_UTIL_PLEASE_SPECIFY $2
    done
}

# Prompt user for a numbered response, accepting
# blank answers as a default response.
#   $1:     Question to ask.
#   $2:     Number of options.
#   $3:     Default response.
#   $4..$n: Description of options.
function prompt_numd {
    result=""
    if [ "$silent" == "y" ]; then
        failed="y"
        return
    fi
    while true; do
        echo "$1"
        for i in `seq 1 $2`; do
            let opt=i+3
            eval opt=\$$opt
            echo "  $i: $opt"
        done
        promp_msg=`/opt/Citrix/VDA/bin/getstr SHELL_UTIL_CHOOSE_FROM_ABOVE_D $2 $3`
        read -p "$promp_msg" val
        if [ -z $val ]; then
            result=$3
            return
        fi
        if [[ $val =~ ^[0-9]+$ ]] && [ $val -le $2 ] && [ $val -ge 1 ]; then
            result=$val
            return
        fi
        /opt/Citrix/VDA/bin/getstr SHELL_UTIL_PLEASE_SPECIFY $2
    done
}

# Prompt user for a numbered port response, accepting
# blank answer as a default response.
#   $1:     Question to ask.
#   $2:     Default response.
function prompt_numd_port {
    result=""
    if [ "$silent" == "y" ]; then
        failed="y"
        return
    fi
    while true; do
        read -p "$1 [$2]: " val
        if [ -z $val ]; then
            result=$2
            return
        fi
        if [[ $val =~ ^[0-9]+$ ]] && [ $val -le 65535 ] && [ $val -ge 0 ]; then
            result=$val
            return
        fi
        /opt/Citrix/VDA/bin/getstr SHELL_UTIL_PLEASE_SPECIFY_PORT 0 65535
    done
}

# Get desired value either from environment var
# if it exists, or from the user otherwise.
#   $1: Environment variable.
#   $2: Question to ask.
function get_yn {
    eval val=\$$1
    /opt/Citrix/VDA/bin/getstr SHELL_UTIL_CHECKING_STH $1
    case $val in
        [Yy]* ) echo "$1 = '$val'"; result="y";;
        [Nn]* ) echo "$1 = '$val'"; result="n";;
        ""    ) /opt/Citrix/VDA/bin/getstr SHELL_UTIL_NOT_SET $1; prompt_yn "$2";;
        *     ) /opt/Citrix/VDA/bin/getstr SHELL_UTIL_NOT_CONTAIN_VALID; prompt_yn "$2";;
    esac
    echo
}

# Get desired value either from environment var
# if it exists, or from the user otherwise.
#   $1: Environment variable.
#   $2: Question to ask.
#   $3: Default response.
function get_ynd {
    eval val=\$$1
    /opt/Citrix/VDA/bin/getstr SHELL_UTIL_CHECKING_STH $1
    case $val in
        [Yy]* ) echo "$1 = '$val'"; result="y";;
        [Nn]* ) echo "$1 = '$val'"; result="n";;
        ""    ) /opt/Citrix/VDA/bin/getstr SHELL_UTIL_NOT_SET $1; prompt_ynd "$2" "$3";;
        *     ) /opt/Citrix/VDA/bin/getstr SHELL_UTIL_NOT_CONTAIN_VALID $1; prompt_ynd "$2" "$3";;
    esac
    echo
}

# Get desired value either from environment var
# if it exists, or from the user otherwise.
#   $1: Environment variable.
#   $2: Question to ask.
function get_val {
    eval val=\$$1
    /opt/Citrix/VDA/bin/getstr SHELL_UTIL_CHECKING_STH $1
    case $val in
        "" ) /opt/Citrix/VDA/bin/getstr SHELL_UTIL_NOT_SET $1; prompt_val "$2";;
        *  ) echo "$1 = '$val'"; result="$val";;
    esac
    echo
}

# Get desired value either from environment var
# if it exists, or from the user otherwise.
#   $1: Environment variable.
#   $2: Question to ask.
#   $3: Default response.
function get_vald {
    eval val=\$$1
    /opt/Citrix/VDA/bin/getstr SHELL_UTIL_CHECKING_STH $1
    case $val in
        "" ) /opt/Citrix/VDA/bin/getstr SHELL_UTIL_NOT_SET $1; prompt_vald "$2" "$3";;
        *  ) echo "$1 = '$val'"; result="$val";;
    esac
    echo
}

# Get desired value either from environment var
# if it exists, or from the user otherwise.
# Checks user responses against a validate callback.
#   $1: Environment variable.
#   $2: Question to ask.
#   $3: Validation callback.
function get_valv {
    eval val=\$$1
    /opt/Citrix/VDA/bin/getstr SHELL_UTIL_CHECKING_STH $1
    case $val in
        "" ) /opt/Citrix/VDA/bin/getstr SHELL_UTIL_NOT_SET $1; prompt_valv "$2" "$3";;
        *  ) echo "$1 = '$val'"; result="$val";;
    esac
    echo
}

# Get desired value either from environment var
# if it exists, or from the user otherwise.
# Accept empty environment vars if running in silent mode.
#   $1: Environment variable.
#   $2: Question to ask.
function get_vale {
    eval val=\$$1
    /opt/Citrix/VDA/bin/getstr SHELL_UTIL_CHECKING_STH $1
    if [ "$silent" == "y" ]; then
        echo "$1 = '$val'";
        result="$val";
    else
        case $val in
            "" ) /opt/Citrix/VDA/bin/getstr SHELL_UTIL_NOT_SET $1; prompt_vale "$2";;
            *  ) echo "$1 = '$val'"; result="$val";;
        esac
    fi
    echo
}

# Get desired value either from environment var
# if it exists, or from the user otherwise.
# Accept empty environment vars if running in silent mode.
# Checks user responses against a validate callback.
#   $1: Environment variable.
#   $2: Question to ask.
#   $3: Validation callback.
function get_valve {
    eval val=\$$1
    /opt/Citrix/VDA/bin/getstr SHELL_UTIL_CHECKING_STH $1
    if [ "$silent" == "y" ]; then
        echo "$1 = '$val'";
        result="$val";
    else
        case $val in
            "" ) /opt/Citrix/VDA/bin/getstr SHELL_UTIL_NOT_SET $1; prompt_valve "$2" "$3" "$4";;
            *  ) echo "$1 = '$val'"; result="$val";;
        esac
    fi
    echo
}

# Get desired value either from environment var
# if it exists, or from the user otherwise.
#   $1:     Environment variable.
#   $2:     Question to ask.
#   $3:     Number of options.
#   $4..$n: Description of options.
function get_num {
    eval val=\$$1
    /opt/Citrix/VDA/bin/getstr SHELL_UTIL_CHECKING_STH $1
    case $val in
        "" ) /opt/Citrix/VDA/bin/getstr SHELL_UTIL_NOT_SET $1; prompt_num "${@:2}";;
        *  ) echo "$1 = '$val'"; result=$val;;
    esac
    echo
}

# Get desired value either from environment var
# if it exists, or from the user otherwise.
#   $1:     Environment variable.
#   $2:     Question to ask.
#   $3:     Number of options.
#   $4:     Default response.
#   $5..$n: Description of options.
function get_numd {
    eval val=\$$1
    /opt/Citrix/VDA/bin/getstr SHELL_UTIL_CHECKING_STH $1
    case $val in
        "" ) /opt/Citrix/VDA/bin/getstr SHELL_UTIL_NOT_SET $1; prompt_numd "${@:2}";;
        *  ) echo "$1 = '$val'"; result=$val;;
    esac
    echo
}

# Get desired port value either from environment var
# if it exists, or from the user otherwise.
#   $1:     Environment variable.
#   $2:     Question to ask.
#   $3:     Default response.
function get_numd_port {
    eval val=\$$1
    /opt/Citrix/VDA/bin/getstr SHELL_UTIL_CHECKING_STH $1
    case $val in
        "" ) /opt/Citrix/VDA/bin/getstr SHELL_UTIL_NOT_SET $1; prompt_numd_port "${@:2}";;
        *  ) echo "$1 = '$val'"; result=$val;;
    esac
    echo
}

# Initialise command logging.
#   $1: Is this a cleanup operation?
function cmdlog_start {
    if [ -f "$configure_log" ]; then
        echo >> $configure_log
    fi
    cmdlog_mode=$1
    if [ "$cmdlog_mode" == "cleanup" ]; then
        echo "$product_name Cleanup Log" >> $configure_log
# When come to clean up step, the '/opt/Citrix/VDA/bin/getstr' is not avialible
        /opt/Citrix/VDA/bin/getstr SHELL_UTIL_CLEANING "$product_name"
#        echo "Cleaning up $product_name..."
        echo
    else
        echo "$product_name Configuration Log" >> $configure_log
        /opt/Citrix/VDA/bin/getstr SHELL_UTIL_CONFIGURING "$product_name"
        echo
    fi
    date >> $configure_log
    echo >> $configure_log
}

# Stop command logging.
function cmdlog_stop {
    if [ "$cmdlog_mode" == "cleanup" ]; then
        get_str SHELL_UTIL_CLEANUP_COMPLETE
        echo "Cleanup complete." >> $configure_log
    else
        /opt/Citrix/VDA/bin/getstr SHELL_UTIL_CONFIG_COMPLETE
        echo "Configuration complete." >> $configure_log
    fi
}

# Execute a function, save its output, and then print
# an error message and abort if the command fails.
#   $1: Function.
#   $2: Name of calling function.
#   $3: Description of action attempted by function.
#   $4: Any parameters to be passed to the function.
function cmdlog_exec {
    ( $1 "${@:4}" ) 1>$lastcmd_log 2>$lastcmd_log
    result=$?
    echo "$2()" >> $configure_log
    echo "$3" >> $configure_log
    cat $lastcmd_log >> $configure_log
    echo >> $configure_log
    if [ ! $result -eq 0 ]; then
        /opt/Citrix/VDA/bin/getstr SHELL_UTIL_EXEC_ERROR $3
        cat $lastcmd_log
        echo
        /opt/Citrix/VDA/bin/getstr SHELL_UTIL_EXEC_LOG_LOCATION $configure_log
        if [ "$cmdlog_mode" == "cleanup" ]; then
            /opt/Citrix/VDA/bin/getstr SHELL_UTIL_CLEANUP_FAILED
            echo "Cleanup failed." >> $configure_log
        else
            /opt/Citrix/VDA/bin/getstr SHELL_UTIL_CONFIG_FAILED
            echo "Configuration failed." >> $configure_log
        fi
        exit 1
    fi
}


# output:
#   osPlatform: OS platform, suse, centos, red, ubuntu, pardus, debian
#   osVersion:
# return:
#   0: success
#   1: fail
function getOSInfo()
{
    systemKernel="$(uname -v |grep -i Ubuntu 2>&1)"

    #Check Core file, checking sequence is suse, centos, red, ubuntu, pardus, debian
    if [[ -f "$coreFileSuse" ]]; then
        osPlatform="suse"
        osVersion="$(cat "$coreFileSuse" |grep "SUSE Linux Enterprise" |cut -d" " -f5|tr A-Z a-z  2>&1)"
        if [[ "$?" -ne "0" || -z "$osVersion" ]]; then           
            return 1
        fi
    elif [[ -f "$coreFileRocky" ]]; then 
        osPlatform="rocky"
        osVersion="$(cat $coreFileRocky |cut -d ' ' -f4 |cut -d '.' -f1-2  2>&1)"    
        if [[ "$?" -ne "0" || -z "$osVersion" ]]; then           
            return 1
        fi
    elif [[ -f "$coreFileCent" ]]; then 
        osPlatform="$(cat $coreFileCent |cut -d ' ' -f1 |tr A-Z a-z  2>&1)"
        if [[ "$?" -ne "0" || "$osPlatform" -ne "centos" ]]; then           
            return 1
        fi
        
        num="$(cat $coreFileCent |wc -w 2>&1)"
        if [[ $num -gt 4 ]]; then
            num=`expr $num - 1`
        fi
        osVersion="$(cat $coreFileCent |cut -d ' ' -f$num |cut -d '.' -f1-2  2>&1)"
        if [[ "$?" -ne "0" || -z "$osVersion" ]]; then           
            return 1
        fi
        
    elif [[ -f "$coreFileRhel" ]]; then
        osPlatform="$(cat $coreFileRhel |cut -d ' ' -f1 |tr A-Z a-z  2>&1)"
        if [[ "$?" -ne "0" || "$osPlatform" -ne "red" ]]; then           
            return 1
        fi
        osVersion="$(cat $coreFileRhel |cut -d ' ' -f7  2>&1)"

        if [[ ! $osVersion =~ ^[0-9.]*$ ]]; then
            osVersion="$(cat $coreFileRhel |cut -d ' ' -f6  2>&1)"
        fi

        if [[ "$?" -ne "0" || -z "$osVersion" ]]; then           
            return 1
        fi

    elif [[ -f "$coreFileUbuntu"  && -n "$systemKernel" ]]; then
        osPlatform="$(cat $coreFileUbuntu |grep DISTRIB_ID |cut -d '=' -f2 |tr A-Z a-z  2>&1)"
        if [[ "$?" -ne "0" || "$osPlatform" -ne "ubuntu" ]]; then           
            return 1
        fi
        osVersion="$(cat $coreFileUbuntu |grep DISTRIB_RELEASE |cut -d '=' -f2 |tr A-Z a-z  2>&1)" 
        if [[ "$?" -ne "0" || -z "${osVersion}" ]]; then           
            return 1
        fi  
    # MUST follow Ubuntu, because Pardus utilizes the same file(lsb-release) as Ubuntu.
    elif [[ -f "$coreFilePardus" ]]; then
        osPlatform="$(cat $coreFilePardus |grep DISTRIB_ID |cut -d '=' -f2 |tr A-Z a-z  2>&1)"
        if [[ "$?" -ne "0" || "$osPlatform" -ne "pardus" ]]; then           
            return 1
        fi
        osVersion="$(cat $coreFilePardus |grep DISTRIB_RELEASE |cut -d '=' -f2 |tr A-Z a-z  2>&1)" 
        if [[ "$?" -ne "0" || -z "${osVersion}" ]]; then           
            return 1
        fi

    elif [[ -f "$coreFileOS" ]]; then
        osPlatform="$(< $coreFileOS grep ^ID= |cut -d '=' -f2 |tr '[:upper:]' '[:lower:]' |tr -d '"' 2>&1)"
        if [[ "$?" -ne "0" || -z "$osPlatform" ]]; then           
            return 1
        fi

        # map sles and sled to suse
        if [[ $osPlatform == "sles" || $osPlatform == "sled" ]]; then
            osPlatform="suse"
        fi

        if [[ $osPlatform == "amzn" ]]; then
            osPlatform="amazon"
        fi

        osVersion="$(< $coreFileOS grep VERSION_ID |cut -d '=' -f2 |tr '[:upper:]' '[:lower:]' |tr -d '"' 2>&1)"
        if [[ "$?" -ne "0" || -z "${osVersion}" ]]; then           
            return 1
        fi  
    else
        return 1
    fi

    # Change all strings to be lower case
    osPlatform=`tr '[A-Z]' '[a-z]' <<<"$osPlatform"`
    osVersion=`tr '[A-Z]' '[a-z]' <<<"$osVersion"`

    return 0

}

# install python3 xdping
# return:
#   0: success
#   1: fail
function python3_install_xdping {
    osPlatform=""
    osVersion=""
    if ! getOSInfo; then
		echo "Failed to get system info"
		echo "osPlatform=$osPlatform"
		echo "osVersion=$osVersion"
		return 1
	fi

    if [[ ! -f "${PYTHON3_PATH}/bin/python3" ]]; then
        echo "Command ${PYTHON3_PATH}/bin/python3 Does Not Exist!"
        return 1
    fi

    if [[ "${osPlatform}" == "suse" ]]; then
        ${PYTHON3_PATH}/bin/python3 -m easy_install $PYTHON3_PATH/packages/xdping-*.whl &> /dev/null
    else
        ${PYTHON3_PATH}/bin/python3 -m pip install $PYTHON3_PATH/packages/xdping-*.whl &> /dev/null
    fi

    if [ "$?" -ne "0" ]; then
        echo "xdping installation failed!"
        return 1
    fi

    return 0
}

# install python3 ctxldaps dependencies
# return:
#   0: success
#   1: fail
function python3_install_ctxldaps {
    osPlatform=""
    osVersion=""
    if ! getOSInfo; then
		echo "Failed to get system info"
		echo "osPlatform=$osPlatform"
		echo "osVersion=$osVersion"
		return 1
	fi

    if [[ ! -f "${PYTHON3_PATH}/bin/python3" ]]; then
        echo "Command ${PYTHON3_PATH}/bin/python3 Does Not Exist!"
        return 1
    fi

    if [[ "${osPlatform}" == "suse" ]]; then
        ${PYTHON3_PATH}/bin/python3 -m easy_install "cffi==1.14.2" "cryptography==3.1" "decorator==4.4.2" "gssapi==1.6.2" "ldap3==2.8.1" "pyasn1==0.4.8" "pycparser==2.20" "six==1.15.0"  &> /dev/null
    else
        ${PYTHON3_PATH}/bin/python3 -m pip install -r $PYTHON3_PATH/ctxldaps.txt  &> /dev/null
    fi

    if [ "$?" -ne "0" ]; then
        echo "ctxldaps installation failed!"
        return 1
    fi

    return 0
}

# Configure python3 virtual environment
# return:
#   0: success
#   1: fail
function python3_configure_venv()
{
	if ! getOSInfo; then
		echo "Failed to get system info"
		echo "osPlatform=$osPlatform"
		echo "osVersion=$osVersion"
		return 1
	fi

    case "$osPlatform" in
        "red" | "centos" | "rocky")
            yum -y install python36-devel krb5-devel gcc &> /dev/null
            ;;
        "ubuntu" | "debian")
            export DEBIAN_FRONTEND=noninteractive
            apt-get -y install python3-dev python3-venv python3-pip libkrb5-dev  &> /dev/null
            ;;
        "suse")
            zypper -n install lsb-release python3-devel python3-setuptools krb5-devel gcc libffi-devel libopenssl-devel  &> /dev/null
            ;;
        "amazon")
            yum -y install python3 python3-devel krb5-devel gcc &> /dev/null
            ;;
        *)
            echo "OS paltform not supported"
            return 1
            ;;
    esac

    if [[ "$?" -ne "0" ]]; then    
        echo "Failed to install Python3 dependencies"      
        return 1
    fi

    if [[ ! -d "$PYTHON3_PATH" ]]; then
        echo "Folder Path ${PYTHON3_PATH} Does Not Exist!"
        return 1
    fi

    if [[ "${osPlatform}" == "suse" ]]; then
        if ! command -v 'krb5-config' &> /dev/null
        then
            export PATH=$PATH:/usr/lib/mit/bin:/usr/lib/mit/sbin
            if ! command -v 'krb5-config' &> /dev/null
            then
                echo "command krb5-config could not be found"
                return 1
            fi
        fi
        if [[ ! -f  "/usr/lib/mit/include/gssapi/gssapi_ext.h" ]]; then
            mkdir -p /usr/lib/mit/include/gssapi/
            ln -s /usr/include/gssapi/gssapi_ext.h /usr/lib/mit/include/gssapi/gssapi_ext.h
        fi
        python3 -m venv $PYTHON3_PATH  &> /dev/null
        ${PYTHON3_PATH}/bin/python3 -m pip install setuptools==40.6.2  &> /dev/null
    else
        python3 -m venv $PYTHON3_PATH  &> /dev/null
    fi

    if [[ "$?" -ne "0" ]]; then    
        echo "Failed to create virtual environment"      
        return 1
    fi

    return 0
}
