#!/bin/bash
#
# Session launcher
#
# Copyright 1999-2014 Citrix Systems, Inc. All rights reserved.
#

# This script is invoked after a user has logged in, to launch a session. Both
# desktop and published-application sessions come through here.
#
# It is defined that this process embodies a session : when this process exits,
# the system considers this to mean the end of the session, and cleans up.
# Typically, this script should start up some apps, then exec a window manager.
#
# It is intended that this script be customised for local preferences.
#
# Parameters are passed into this script using environment variables, rather
# than positional parameters. This is to ensure that they are safe from shell
# munging.
#
# Note that this script is invoked via a login shell, so a user's .profile might
# modify them for some reason. In particular, DISPLAY seems to be prone to being
# overwritten : see CITRIX_DISPLAY below
#
# CITRIX_SESSION_ID     is the numeric session id (as output by ctxqsession)
#                       This can be used in invocations of ctxmsg to pop up
#                       messages, for example.
#
# DISPLAY               is as expected.
#
# CITRIX_DISPLAY        is a copy of DISPLAY, to detect hardcoding of
#                       DISPLAY=... in user's profile (see below).
#
# CITRIX_REMOTE_DISPLAY is of the form hostname:x.0 : it can be used when
#                       invoking an application on a remote server. eg
#                       rsh server xterm -display $CITRIX_REMOTE_DISPLAY.

####### Diagnosing problems  #########

# Uncomment this line to allow problems with this script to be recorded. (need
# to make sure /dev/console is world writable)

# exec 1> /dev/console 2>&1

# Can send them to a file instead, but remember that this script runs as the
# logged-in user, so need to be careful with file permissions.

# exec 1>> /tmp/ctxsession.log 2>&1

# If the following line is un-commented then each command executed in this shell
# will also be echo'd to the target of the previous "exec".

# set -x
# exec >/dev/null 2>&1

SessionLog='disable'

if [ "$SessionLog" = "enable" ]
then
    if [ -f /etc/X11/xinit/Xclients ]
    then
        result=`grep 'set -x' /etc/X11/xinit/Xclients`
        if [ -z "$result" ]
        then
            sed -i '1 aset -x' /etc/X11/xinit/Xclients
        fi
    fi
    exec 1>> /tmp/ctxsession.log 2>&1
    set -x
else
    if [ -f /etc/X11/xinit/Xclients ]
    then
        result=`grep 'set -x' /etc/X11/xinit/Xclients`
        if [ -n "$result" ]
        then
            sed -i '2d' /etc/X11/xinit/Xclients
        fi
    fi
    exec >/dev/null 2>&1
fi

#
# Pre session
#

# Login execution sequence for bash
if [ -f /etc/profile ]
then
    . /etc/profile
fi

if [ -f ~/.bash_profile ]
then
    . ~/.bash_profile
elif [ -f ~/.bash_login ]
then
    . ~/.bash_login
elif [ -f ~/.profile ]
then
    . ~/.profile
fi

# System Commands
XHOST=/usr/bin/xhost

# Citrix environment variables
CTXBASEDIR=/opt/Citrix/VDA
CTXBINDIR=$CTXBASEDIR/bin
CTXMANDIR=$CTXBASEDIR/share/man

if [ -d ${CTXMANDIR} ]
then
    MANPATH="${MANPATH:-/usr/share/man}:${CTXMANDIR}"
    export MANPATH
fi

PATH="${PATH:-/usr/bin}:$CTXBINDIR"
export PATH

#
# Session
#

# Escape POSIX basic regular expression characters .[\*^$ as the
# separator in username format be one of these characters
escape()
{
    if [ "$1" = "-" ]
    then
        while read -r line
        do
            echo $line | sed 's/[\.\[\\\*\^\$]/\\&/g'
        done
    else
        echo $1 | sed 's/[\.\[\\\*\^\$]/\\&/g'
    fi
}

USER_ESCAPED=$(escape $USER)
USER_ID=$(id -u $USER)

(
for name in $("$XHOST" 2> /dev/null | cut -f1 -d\  | sed -E -n '/'"$USER_ESCAPED|${USER_ID-NONE}"'/!p')
do
    $XHOST -$name 2> /dev/null
done
ip addr | awk -v xhost_="$XHOST" '/inet6? / { gsub(/\/[0-9]+/, "", $2); system(xhost_" -"$1":"$2) }'
)&
$XHOST "+SI:localuser:$USER"
$XHOST "+SI:localuser:#$USER_ID"
$XHOST SI:localuser:ctxsrvr
$XHOST SI:localuser:root
$XHOST "-LOCAL:"

# Install randr hook so disable resolution change from WM
XSESSION="$HOME/.xsession"
[ -f $XSESSION ] && ! grep -q "libctxXrandrhook\.so" $XSESSION && sed -r -i.orig '/^#!/a\
export LD_PRELOAD='\''/usr/$LIB/libctxXrandrhook.so'\' $XSESSION

# Changes for Gnome 2 desktop
if [ -x /usr/bin/gconftool-2 ]
then
    # Disable fast user switching
    applets=`/usr/bin/gconftool-2 --get /apps/panel/general/applet_id_list |
             sed -r -e s/[][]*//g -e s/,/\ /g |
             awk 'BEGIN { printf "[" } { for (i=1; i<=NF; i++) { if ($i !~ "fast_user_switch_applet") { printf $i ","; } } } END { printf "]" }' |
             sed 's/,]/]/'`
    /usr/bin/gconftool-2 --set -t list --list-type=string /apps/panel/general/applet_id_list ${applets}

    # Disable screensaver lock
    /usr/bin/gconftool-2 --set /apps/gnome-screensaver/lock_enabled --type bool false

    # Disable user switching
    /usr/bin/gconftool-2 --set /desktop/gnome/lockdown/disable_user_switching --type boolean true

    # For anonymous session
    if [ "$CITRIX_ANONYMOUS" ]
    then
        # Disable the lock screen button/menu/shortcut
        /usr/bin/gconftool-2 --set /apps/panel/global/disable_lock_screen --type boolean true
        /usr/bin/gconftool-2 --set /desktop/gnome/lockdown/disable_lock_screen --type boolean true
        /usr/bin/gconftool-2 --type string --set /apps/gnome_settings_daemon/keybindings/screensaver ''
        # Disable activating screensaver when idle
        /usr/bin/gconftool-2 --set /apps/gnome-screensaver/idle_activation_enabled --type boolean false
    fi

    # Disable xrandr plugin of gnome-setting-daemon
    /usr/bin/gconftool-2 --set /apps/gnome_settings_daemon/plugins/xrandr/active --type bool false

    # Replace Ctrl-Alt-Delete shortcut since it generally raises the power-off prompt
    POWER_SHORTCUT=$(/usr/bin/gconftool-2 -g /apps/gnome_settings_daemon/keybindings/power)
    if [ "$POWER_SHORTCUT" = "<Primary><Alt>Delete" ] || [ "$POWER_SHORTCUT" = "<Control><Alt>Delete" ] || [ "$POWER_SHORTCUT" = "<Ctrl><Alt>Delete" ]
    then
        /usr/bin/gconftool-2 -s /apps/gnome_settings_daemon/keybindings/power -t string ""

        EXISTING_CUSTOM_BINDINGS=$(/usr/bin/gconftool-2 --all-dirs /desktop/gnome/keybindings | sed -n '/.*\/custom[0-9]*/p')
        if [ -z "$EXISTING_CUSTOM_BINDINGS" ]
        then
            CUSTOM_BINDING_PATH="/desktop/gnome/keybindings/custom0"
        else
            CUSTOM_BINDING_NUMBER=$(echo "$EXISTING_CUSTOM_BINDINGS" | sed 's/.*\/custom\([0-9]*\).*/\1/' | sort -n | tail -n1)
            CUSTOM_BINDING_NUMBER=$(expr "$CUSTOM_BINDING_NUMBER" + 1)
            CUSTOM_BINDING_PATH="/desktop/gnome/keybindings/custom$CUSTOM_BINDING_NUMBER"
        fi

        /usr/bin/gconftool-2 -s $CUSTOM_BINDING_PATH/name -t string "Log out HDX session"
        /usr/bin/gconftool-2 -s $CUSTOM_BINDING_PATH/action -t string "gnome-session-save --logout-dialog"
        /usr/bin/gconftool-2 -s $CUSTOM_BINDING_PATH/binding -t string "$POWER_SHORTCUT"
    fi
fi

if [ -n "$CITRIX_PARAMS_FD" ] && [ -n "$CITRIX_PUBLISHED_APP" ]
then
    #start dbus-daemon explicitily so that could set DBUS_SESSION_BUS_ADDRESS env
    #gsettings will also start a new dbus-daemon if no dbus-daemon running.
    #So, to avoid additinal dbus-daemon, this command need to be run earlier.
    eval `dbus-launch --sh-syntax --exit-with-session`
fi

# Changes for Gnome 3 desktop
if [ -x /usr/bin/gsettings ]
then
    # For seamless application session
    if [ "$CITRIX_PUBLISHED_APP" ] && [ "$CITRIX_TWI_ENABLED" ]
    then
        # Backup the gtk.css if it already exists for reverting later
        GTK_CSS_DIR=~/.config/gtk-3.0
        GTK_CSS_FILE=$GTK_CSS_DIR/gtk.css
        GTK_CSS_BAK_FILE=$GTK_CSS_DIR/gtk.css.bak

        if [ -f $GTK_CSS_FILE ]
        then
            mv -f $GTK_CSS_FILE $GTK_CSS_BAK_FILE
        else
            if [ ! -f $GTK_CSS_DIR ]
            then
                mkdir -p $GTK_CSS_DIR
            fi
        fi

        # Set gtk styles to avoid showing shadow around the window
        echo ".window-frame, .window-frame:backdrop {box-shadow:0 0 0 1px #c3c3c3;border-style:none;margin:0px;}" > $GTK_CSS_FILE
        # For GTK 3.20 and later
        echo "decoration, decoration:backdrop {box-shadow:0 0 0 1px #c3c3c3;border-style:none;margin:0px;}" >> $GTK_CSS_FILE

        # Set draggable border width to 0 to avoid showing shadow, keep the old configuration for reverting later
        OLD_BORDER_WIDTH=$(/usr/bin/gsettings get org.gnome.mutter draggable-border-width)
        /usr/bin/gsettings set org.gnome.mutter draggable-border-width 0

        # Add the minimize, maximize and close buttons, keep the old configuration for reverting later
        OLD_BUTTON_LAYOUT=$(/usr/bin/gsettings get org.gnome.desktop.wm.preferences button-layout)
        /usr/bin/gsettings set org.gnome.desktop.wm.preferences button-layout ':minimize,maximize,close'
    fi

    # Disable fast user switching
    /usr/bin/gsettings set org.gnome.desktop.lockdown disable-user-switching true
    /usr/bin/gsettings set org.gnome.desktop.screensaver user-switch-enabled false

    #disable smartcard utility
    /usr/bin/gsettings set org.gnome.settings-daemon.plugins.smartcard active false

    # Disable screensaver lock
    /usr/bin/gsettings set org.gnome.desktop.screensaver lock-enabled false

    SVERSION=$(lsb_release -d | awk -F":" '/Description/ {print $2}')
    if [[ "$SVERSION" = *"Ubuntu 18.04"* || "$SVERSION" = *"Ubuntu 20.04"* ]];
    then

        # Enable gnome extensions
        if [[ ! -f ~/.config/autostart/gnome-extensions.desktop ]];
        then
            if [[ "$SVERSION" = *"Ubuntu 18.04"* ]];
            then
                /usr/bin/gsettings set org.gnome.shell enabled-extensions "['ubuntu-appindicators@ubuntu.com', 'ubuntu-dock@ubuntu.com']"
            else
                /usr/bin/gsettings set org.gnome.shell enabled-extensions "['ubuntu-appindicators@ubuntu.com', 'ubuntu-dock@ubuntu.com', 'desktop-icons@csoriano']"
            fi
        fi

        # Disable lock screen
        /usr/bin/gsettings set org.gnome.desktop.lockdown disable-lock-screen true
        /usr/bin/gsettings set org.gnome.settings-daemon.plugins.media-keys screensaver "[]"
        /usr/bin/gsettings set org.gnome.desktop.session idle-delay 0

    fi

    # For anonymous session
    if [ "$CITRIX_ANONYMOUS" ]
    then
       # Disable lock screen button/menu/shortcut
       /usr/bin/gsettings set org.gnome.desktop.lockdown disable-lock-screen true
       /usr/bin/gsettings set org.gnome.settings-daemon.plugins.media-keys screensaver "[]"
       # Disable activating screensaver when idle
       /usr/bin/gsettings set org.gnome.desktop.session idle-delay 0
    fi

    # Disable xrandr plugin of gnome-settings-daemon
    /usr/bin/gsettings set org.gnome.settings-daemon.plugins.xrandr active false

    # Replace Ctrl-Alt-Delete shortcut since it generally raises the power-off prompt
    LOGOUT_SHORTCUT=$(/usr/bin/gsettings get org.gnome.settings-daemon.plugins.media-keys logout)
    if [ "$LOGOUT_SHORTCUT" = "'<Primary><Alt>Delete'" ] || [ "$LOGOUT_SHORTCUT" = "'<Control><Alt>Delete'" ] || [ "$LOGOUT_SHORTCUT" = "'<Ctrl><Alt>Delete'" ]
    then
        /usr/bin/gsettings set org.gnome.settings-daemon.plugins.media-keys logout ''

        EXISTING_CUSTOM_BINDINGS=$(/usr/bin/gsettings get org.gnome.settings-daemon.plugins.media-keys custom-keybindings | sed 's/\(.*\)]/\1/')
        if [ $(echo "$EXISTING_CUSTOM_BINDINGS" | tr -d -c "'" | wc -c) -eq 0 ]
        then
            CUSTOM_BINDING_PATH="/org/gnome/settings-daemon/plugins/media-keys/custom-keybindings/custom0/"
            /usr/bin/gsettings set org.gnome.settings-daemon.plugins.media-keys custom-keybindings "['$CUSTOM_BINDING_PATH']"
        else
            CUSTOM_BINDING_NUMBER=$(echo "$EXISTING_CUSTOM_BINDINGS" | sed 's/,/\n/' | sed 's/.*\/custom\([0-9]*\).*/\1/' | sort -n | tail -n1)
            CUSTOM_BINDING_NUMBER=$(expr "$CUSTOM_BINDING_NUMBER" + 1)
            CUSTOM_BINDING_PATH="/org/gnome/settings-daemon/plugins/media-keys/custom-keybindings/custom$CUSTOM_BINDING_NUMBER/"
            EXISTING_CUSTOM_BINDINGS="$EXISTING_CUSTOM_BINDINGS, '$CUSTOM_BINDING_PATH']"
            /usr/bin/gsettings set org.gnome.settings-daemon.plugins.media-keys custom-keybindings "$EXISTING_CUSTOM_BINDINGS"
        fi

    /usr/bin/gsettings set org.gnome.settings-daemon.plugins.media-keys.custom-keybinding:$CUSTOM_BINDING_PATH name "Log out HDX session"
    /usr/bin/gsettings set org.gnome.settings-daemon.plugins.media-keys.custom-keybinding:$CUSTOM_BINDING_PATH command "/usr/bin/gnome-session-quit"
    /usr/bin/gsettings set org.gnome.settings-daemon.plugins.media-keys.custom-keybinding:$CUSTOM_BINDING_PATH binding "$LOGOUT_SHORTCUT"
    fi

    # Change font antialiasing to grayscale for cleanning the halos around the text
    /usr/bin/gsettings set org.gnome.settings-daemon.plugins.xsettings antialiasing grayscale

    # Change desktop background to 'zoom' if set to'scaled'
    if [[ "$(/usr/bin/gsettings get org.gnome.desktop.background picture-options)" == *scaled* ]]
    then
        /usr/bin/gsettings set org.gnome.desktop.background picture-options 'zoom'
        /usr/bin/gsettings set org.gnome.desktop.screensaver picture-options 'zoom'
    fi

    # Changes for Mate Desktop
    MSESSION="$(type -p mate-session)"
    if [ -n "$MSESSION" ]; then
        # Disable fast user switching
        /usr/bin/gsettings set org.mate.lockdown disable-user-switching true
        /usr/bin/gsettings set org.mate.screensaver user-switch-enabled false

        #disable smartcard utility
        /usr/bin/gsettings set org.mate.SettingsDaemon.plugins.smartcard active false

        # Disable screensaver lock
        /usr/bin/gsettings set org.mate.screensaver lock-enabled false

        SVERSION=$(lsb_release -d | awk -F":" '/Description/ {print $2}')
        if [[ "$SVERSION" = *"Ubuntu 18.04"* || "$SVERSION" = *"Ubuntu 20.04"* || "$CITRIX_ANONYMOUS" ]];
        then
            # Disable lock screen
            /usr/bin/gsettings set org.mate.lockdown disable-lock-screen true
            /usr/bin/gsettings set org.mate.SettingsDaemon.plugins.media-keys screensaver "[]"
            /usr/bin/gsettings set org.mate.session idle-delay 0
        fi

        # Disable xrandr plugin of mate-settings-daemon
        /usr/bin/gsettings set org.mate.SettingsDaemon.plugins.xrandr active false

        # Replace Ctrl-Alt-Delete shortcut since it generally raises the power-off prompt
        /usr/bin/gsettings set org.mate.SettingsDaemon.plugins.media-keys logout logout ''

        EXISTING_CUSTOM_BINDINGS=$(dconf dump /org/mate/desktop/keybindings/ |  grep "^\[custom"  | sed 's/[][]//g')
        if [ -z "$EXISTING_CUSTOM_BINDINGS" ]
        then
            CUSTOM_BINDING_PATH="/org/mate/desktop/keybindings/custom0/"
        else
            CUSTOM_BINDING_NUMBER=$(echo "$EXISTING_CUSTOM_BINDINGS" | tr -cd "[0-9]\n" | sort -n | tail -n1)
            CUSTOM_BINDING_NUMBER=$(expr "$CUSTOM_BINDING_NUMBER" + 1)
            CUSTOM_BINDING_PATH="/org/mate/desktop/keybindings/custom$CUSTOM_BINDING_NUMBER/"
        fi

        /usr/bin/gsettings set org.mate.control-center.keybinding:$CUSTOM_BINDING_PATH name "Log out HDX Mate session"
        /usr/bin/gsettings set org.mate.control-center.keybinding:$CUSTOM_BINDING_PATH action "/usr/bin/mate-session-save --logout-dialog"
        /usr/bin/gsettings set org.mate.control-center.keybinding:$CUSTOM_BINDING_PATH binding "'<Ctrl><Alt>Delete'"

        # Change desktop background to 'zoom' if set to'scaled'
        if [[ "$(/usr/bin/gsettings get org.mate.background picture-options)" == *scaled* ]]
        then
            /usr/bin/gsettings set org.mate.background picture-options 'zoom'
            /usr/bin/gsettings set org.mate.screensaver picture-options 'zoom'
        fi
    fi
fi

# Disable the lock screen button/menu/shortcut for anonymous session with KDE desktop
if [ "$CITRIX_ANONYMOUS" ] && [ -x /usr/bin/kwriteconfig ]
then
    /usr/bin/kwriteconfig --file kdeglobals --group "KDE Action Restrictions" --key "action/lock_screen" --type bool false
    /usr/bin/kwriteconfig --file kglobalshortcutsrc --group ksmserver --key "Lock Session" "none,Ctrl+Alt+L,Lock Session"
    /usr/bin/kwriteconfig --file kglobalshortcutsrc --group krunner --key "Lock Session" "none,Ctrl+Alt+L,Lock Session"
fi

function desktop_session() {
    # RHEL
    if [ -r /etc/X11/xinit/Xsession ]
    then
        /etc/X11/xinit/Xsession
    # SUSE
    elif [ -r /etc/X11/xdm/Xsession ]
    then
        VERSION=`cat /etc/SuSE-release 2>&1 | sed -n 's/VERSION = \(.*\)/\1/p'`
        if [ $VERSION -le 11 -a -x /usr/bin/ck-launch-session ]
        then
            /usr/bin/ck-launch-session /etc/X11/xdm/Xsession
        else
            /etc/X11/xdm/Xsession
        fi
    # Ubuntu, Debian and Pardus
    elif [ -r /etc/X11/Xsession ]
    then
        DISTRO=$(lsb_release -is | tr "[:upper:]" "[:lower:]")
        # Pardus
        if [ "${DISTRO}" = "pardus" ]
        then
            #export XDG_CURRENT_DESKTOP=KDE
            export PATH=$PATH:/sbin
            /etc/X11/Xsession
        # Debian
        elif [ "${DISTRO}" = "debian" ]; then
            export PATH=$PATH:/sbin
            /etc/X11/Xsession
        # Ubuntu
        else
            export XDG_SESSION_DESKTOP=ubuntu
            export DESKTOP_SESSION=ubuntu
            export GDMSESSION=ubuntu
            SVERSION=$(lsb_release -rs)
            if [[ "$SVERSION" == "16.04" ]]
            then
                export XDG_CURRENT_DESKTOP=Unity
                export SESSIONTYPE=gnome-session
            elif [[ "$SVERSION" == "18.04" ]]
            then
                export XDG_CURRENT_DESKTOP=ubuntu:GNOME
                export SESSIONTYPE=gnome-session
            elif [[ "$SVERSION" == "20.04" ]]
            then
                export XDG_CURRENT_DESKTOP=ubuntu:GNOME
                export XDG_SESSION_TYPE=x11
            fi
            export PATH=$PATH:/sbin
            #Clear 3 variables in order to not affect /etc/X11/Xsession.d/70im-config_launch
            unset GTK_IM_MODULE
            unset XMODIFIERS
            unset QT_IM_MODULE
            /etc/X11/Xsession
        fi
    # Unknown Linux distribution
    else
        logger "ctxsession.sh: Your Linux distribution is not supported by the Citrix Linux VDA."
    fi
}

function published_application() {
    MUTTER=/usr/bin/mutter
    METACITY=/usr/bin/metacity
    KWIN=/usr/bin/kwin
    COMPIZ=/usr/bin/compiz
    WM=

    # Check the user configuration file
    if [ -r $HOME/.icawm ]; then
       . $HOME/.icawm
       WM=$ICAWM

    # RHEL
    elif [ -r /etc/X11/xinit/Xsession ]
    then
        if [ -f /etc/sysconfig/desktop ]; then
            . /etc/sysconfig/desktop
            if [ "$DESKTOP" = "GNOME" ]; then
                WM=$MUTTER
            elif [ "$DESKTOP" = "KDE" ]; then
                WM=$KWIN
            else
                WM=$MUTTER
        fi
        else
            WM=$MUTTER
        fi

    # SUSE
    elif [ -r /etc/X11/xdm/Xsession ]
    then
        if [ -f /etc/sysconfig/windowmanager ]; then
            . /etc/sysconfig/windowmanager
            if [ "$DEFAULT_WM" = "gnome" ]; then
               WM=$MUTTER
            elif [[ $DEFAULT_WM == kde* ]] || [[ $DEFAULT_WM == startkde* ]]; then
               WM=$KWIN
            else
               WM=$MUTTER
            fi
        else
            WM=$MUTTER
        fi
    # Ubuntu, Debian and Pardus
    elif [ -r /etc/X11/Xsession ]
    then
        DISTRO=$(lsb_release -is | tr "[:upper:]" "[:lower:]")
        osVersion=$(lsb_release -rs)
        # Pardus
        if [ "${DISTRO}" = "pardus" ]
        then
          WM=$KWIN
        # Debian
        elif [ "${DISTRO}" = "debian" ]; then
          WM=$MUTTER
        # Ubuntu
        elif [ $osVersion = "16.04" ]; then
          WM=$COMPIZ
        elif [[ $osVersion == "18.04" || $osVersion == "20.04" ]]; then
          WM=$MUTTER
        fi
    fi

    if [ ! -x "$WM" ]
    then
        WM=$METACITY
    fi

    # Start ICA control virtual channel
    if [ -x $CTXBINDIR/ctxctl ]
    then
       $CTXBINDIR/ctxctl &
    fi

    # Start clipboard virtual channel
    if [ -x $CTXBINDIR/ctxclipboard ]
    then
       $CTXBINDIR/ctxclipboard &
    fi

    # Start EUEM virtual channel
    if [ -x $CTXBINDIR/ctxeuem ]
    then
       $CTXBINDIR/ctxeuem &
    fi

    # Start File transfer virtual channel
    if [ -x $CTXBINDIR/ctxfiletransfer ]
    then
       $CTXBINDIR/ctxfiletransfer &
    fi

    # Start mobile receiver virtual channel
    if [ -x $CTXBINDIR/ctxmrvc ]
    then
       $CTXBINDIR/ctxmrvc &
    fi

    # Start NSAP virtual channel
    if [ -x $CTXBINDIR/ctxnsap ]
    then
       $CTXBINDIR/ctxnsap &
    fi

    # Start audio virtual channel
    if [ -x $CTXBINDIR/ctxaudio -a -x /usr/bin/pulseaudio ]
    then
        /usr/bin/pulseaudio --start --log-target=syslog &
        $CTXBINDIR/ctxaudio &
    fi

    # Start printing virtual channel
    if [ -x $CTXBINDIR/ctxlpmngt ]
    then
        $CTXBINDIR/ctxlpmngt &
    fi

    # Start all third party virtual channels
    if [ -x $CTXBINDIR/ctxvdload ]
    then
        $CTXBINDIR/ctxvdload
    fi

    # Record session ready time
    if [ -x $CTXBINDIR/ctxtimerecord ]
    then
        $CTXBINDIR/ctxtimerecord
    fi

    if [ -n "$CITRIX_PARAMS_FD" ]
    then
        # Pass DBUS_SESSION_BUS_ADDRESS back to login process
        echo "DBUS_SESSION_BUS_ADDRESS=$DBUS_SESSION_BUS_ADDRESS" >&$CITRIX_PARAMS_FD
        # DO NOT miss this line, or ctxlogin will wait forever.
        echo "" >&$CITRIX_PARAMS_FD
        eval "exec $CITRIX_PARAMS_FD>&-"
    fi

    # Start the window manager
    if [ -x "$WM" ]
    then
        #Compiz configuration
        if [ "$WM" == "$COMPIZ" ]
        then
            COMPIZ_CONFIG_PATH=$HOME/.config/compiz-1/compizconfig/
            # Make a backup of the existing config file
            if [ -f $COMPIZ_CONFIG_PATH/Default.ini ] && [ ! -f $COMPIZ_CONFIG_PATH/Default.ini.bak ]
            then
                mv -f $COMPIZ_CONFIG_PATH/Default.ini $COMPIZ_CONFIG_PATH/Default.ini.bak
            fi

            if [ ! -f $COMPIZ_CONFIG_PATH ]
            then
                mkdir -p $COMPIZ_CONFIG_PATH
            fi

            echo "[core]" > $COMPIZ_CONFIG_PATH/Default.ini
            echo "s0_active_plugins = core;composite;opengl;decor;move;resize;" >> $COMPIZ_CONFIG_PATH/Default.ini
        fi

        export LD_PRELOAD='/usr/$LIB/libctxXrandrhook.so'
        $WM &
        WM_PID=$!
        if [[ "$SUSE_LINUX_11" == "1" && $WM == *metacity ]]; then
            unset SUSE_LINUX_11
            sleep infinity &
            WM_PID=$!
        fi

        #Make sure window is decorated by Compiz
        if  [ "$WM" == "$COMPIZ" ]
        then
            /usr/bin/gtk-window-decorator &
        fi

        #Start seamless virtual channel
        if [ "$CITRIX_TWI_ENABLED" ] && [ -x $CTXBINDIR/ctxtwi ]
        then
            $CTXBINDIR/ctxtwi $CITRIX_TWI_FD $WM_PID &
        else
            if [ -x $CTXBINDIR/ctxtwi ]
            then
                $CTXBINDIR/ctxtwi 1 &
            fi
            echo $WM_PID >&$CITRIX_TWI_FD
        fi

        # Start cdm virtual channel
        if [ -x $CTXBINDIR/ctxcdmmount ]
        then
            sleep 5
            $CTXBINDIR/ctxcdmmount &
        fi

        eval "exec $CITRIX_TWI_FD>&-"
        wait $WM_PID
    else
        logger "ctxsession.sh: Failed to find a window manager"
    fi

}

# BCR remove old cert and insert new cert to user nss datebase
certutil -d sql:$HOME/.pki/nssdb -D -n ctxbcr-ca.pem
certutil -d sql:$HOME/.pki/nssdb -A -t TC -n ctxbcr-ca.pem -i /etc/xdl/sslkeystore-bcr/cacerts/ctxbcr-ca.pem

# Start session
if [ -z "$CITRIX_PUBLISHED_APP" ]
then
  desktop_session
else
  published_application
fi

#
# Post session
#

if [ -x /usr/bin/gsettings ]
then
    # Store Gnome extensions
    SVERSION=$(lsb_release -d | awk -F":" '/Description/ {print $2}')
    if [[ "$SVERSION" = *"Ubuntu 18.04"* || "$SVERSION" = *"Ubuntu 20.04"* ]];
    then
        if [[ ! -f ~/.config/autostart/gnome-extensions.desktop ]];
        then
            if [[ ! -d ~/.config/autostart ]];
            then
                mkdir -p ~/.config/autostart
            fi
            cp /etc/xdl/desktop/gnome-extensions-template.desktop ~/.config/autostart/gnome-extensions.desktop
        fi
        GNOME_EXTENSIONS="$(/usr/bin/gsettings get org.gnome.shell enabled-extensions)"
        sed -i "s#Exec=/usr/bin/gsettings set org.gnome.shell enabled-extensions.*#Exec=/usr/bin/gsettings set org.gnome.shell enabled-extensions \"${GNOME_EXTENSIONS}\"#g" ~/.config/autostart/gnome-extensions.desktop
    fi

    #Revert changes in seamless application sessions
    if [ "$CITRIX_PUBLISHED_APP" ] && [ "$CITRIX_TWI_ENABLED" ]
    then
        #Revert gtk.css
        if [ -f $GTK_CSS_BAK_FILE ]
        then
            mv -f $GTK_CSS_BAK_FILE $GTK_CSS_FILE
        else
            rm -f $GTK_CSS_FILE
        fi

        #Revert draggable border width
        /usr/bin/gsettings set org.gnome.mutter draggable-border-width $OLD_BORDER_WIDTH

        #Revert button layout
        /usr/bin/gsettings set org.gnome.desktop.wm.preferences button-layout $OLD_BUTTON_LAYOUT
    fi
fi

# Logout execution sequence for bash
if [ -f ~/.bash_logout ]
then
    . ~/.bash_logout
fi

# Unload sink and source before killing pulseaudio to avoid aborting
pacmd unload-module module-ctx-sink
pacmd unload-module module-ctx-source
pulseaudio -k

# Terminate session processes
(
for i in $(ls -d /proc/*/); do
    PROCPID=$(echo ${i%%/} | awk 'BEGIN { FS="/" } { print $3 }')
    if [[ $PROCPID == $BASHPID || $PROCPID == "self" ]]; then
        continue
    fi;
    if [[ $PROCPID == $PPID ]]; then
        continue;
    fi;
    KILL=$(awk -v session_id="$CITRIX_SESSION_ID" -F= '
            BEGIN { RS="\0"; found=0 }
            { if ($1 == "CITRIX_SESSION_ID" && $2 == session_id) { found=1} }
            END { print found }' ${i%%/}/environ)
    if [ "$KILL" = 1 ]; then
        kill -TERM $PROCPID
    fi;
done
)&
