#!/bin/bash

################################################################################
#
# Citrix Virtual Apps & Desktops For Linux Script: Machine Creation Service
# Copyright (c) Citrix Systems, Inc. All Rights Reserved.
#

source /var/xdl/mcs/mcs_util.sh

fname=$(basename "$0")
scriptName=${fname%.*}  # Script name without extension
logFile="/var/log/$scriptName.log"


#
# Configure host name for SUSE
#
function conf_hostname()
{
    log "Debug: Enter function conf_hostname"
    local confFile1="/etc/hostname"
    local contents1="${NEW_HOSTNAME}"
    local confFile2="/etc/hosts"
    local contents2IPV4="127.0.0.2  ${NEW_HOSTNAME}.${realm} $NEW_HOSTNAME"
    local contents3IPV4="127.0.0.1  ${NEW_HOSTNAME}.${realm} $NEW_HOSTNAME localhost"
    local contents2IPV6="::1  ${NEW_HOSTNAME}.${realm} $NEW_HOSTNAME localhost"
    # update the contents
    # configure hostname
    echo "$contents1">"$confFile1"

    # configure hosts
#    sed -i "s/^127.0.0.1/#127.0.0.1/" "${confFile2}"
#    sed -i "s/^127.0.0.2/#127.0.0.2/" "${confFile2}"
#    sed -i "s/^::1/#::1/" "${confFile2}"
    echo "$contents2IPV4">>"$confFile2"
    sed -i "/^$contents3IPV4/d" "${confFile2}"
    echo "$contents3IPV4">>"$confFile2"
    echo "$contents2IPV6">>"$confFile2"
    sysctl -w kernel.hostname="${NEW_HOSTNAME}" >> "$logFile"

    log_echo "Debug: $confFile1: $contents1"
    log_echo "Debug: $confFile2: $contents1"

    confFile="/etc/sysctl.conf"

    echo "net.ipv6.conf.all.disable_ipv6 = 1">>$confFile

    confFile="/etc/sysconfig/network/dhcp"
    sed -i 's/^DHCLIENT_SET_HOSTNAME.*$/DHCLIENT_SET_HOSTNAME=\"no\"/g' "$confFile"
    sed -i 's/^WRITE_HOSTNAME_TO_HOSTS.*$/WRITE_HOSTNAME_TO_HOSTS=\"yes\"/g' "$confFile"

    log "Debug: Exit function conf_hostname"
}

# Configure chrony
# common function for SUSE platforms
#
function conf_chrony()
{
    log "Debug: Enter function conf_chrony"
    local conf="/etc/chrony.conf"

    # remove any include configuration
    sed -i 's/^include.*$/#&/g' "$conf"
    # remove any server
    sed -i 's/^server.*$/#&/g' "$conf"
    # remove any pool
    sed -i 's/^pool.*$/#&/g' "$conf"

    [[ -n "$NTP_SERVER" ]] && echo "server $NTP_SERVER iburst" >> "$conf"

    # sync time with ntp server
    log_echo "enable and restart chronyd daemon"
    systemctl enable chronyd >> "$logFile" 2>&1
    systemctl restart chronyd >> "$logFile" 2>&1
    chronyc -a makestep >> "$logFile" 2>&1

    log "Debug: Exit function conf_chrony"
}

function conf_winbind_krb5()
{
    log "Debug: Enter function conf_winbind_krb5"
    krbFile="/etc/krb5.conf"
    winbind_krbFile="/etc/xdl/mcs/winbind_krb5.conf.tmpl"

    # Customize /etc/krb5.conf
    REALM=$REALM AD_FQDN=$AD_FQDN realm=$realm envsubst < "$winbind_krbFile" > "$krbFile"

    log_echo "Debug: Exit function conf_winbind_krb5"
}

function conf_sssd_krb5()
{
    log "Debug: Enter function conf_sssd_krb5"
    krbFile="/etc/krb5.conf"
    sssd_krbFile="/etc/xdl/mcs/sssd_krb5.conf.tmpl"

    # Customize /etc/krb5.conf
    REALM=$REALM AD_FQDN=$AD_FQDN realm=$realm envsubst < "$sssd_krbFile" > "$krbFile"

    log "Debug: Exit function conf_sssd_krb5"
}

function conf_winbind_smb()
{
    log "Debug: Enter function conf_winbind_smb"
    winbind_smbFile="/etc/xdl/mcs/winbind_smb.conf.tmpl"
    smbFile="/etc/samba/smb.conf"

    # Customize /etc/samba/smb.conf
    WORKGROUP=$WORKGROUP REALM=$REALM envsubst < "$winbind_smbFile" > "$smbFile"

    log "Debug: Exit function conf_winbind_smb"
}

function conf_sssd_smb()
{
    log "Debug: Enter function conf_sssd_smb"
    sssd_smbFile="/etc/xdl/mcs/sssd_smb.conf.tmpl"
    smbFile="/etc/samba/smb.conf"

    # Customize /etc/samba/smb.conf
    WORKGROUP=$WORKGROUP REALM=$REALM envsubst < "$sssd_smbFile" > "$smbFile"

    log "Debug: Exit function conf_sssd_smb"
}

function conf_sssd_sssd()
{
    log "Debug: Enter function conf_sssd_sssd"
    sssdFile="/etc/sssd/sssd.conf"
    sssdFile_tmpl="/etc/xdl/mcs/sssd.conf.tmpl"

    # Customize /etc/sssd/sssd.conf
    AD_FQDN=$AD_FQDN realm=$realm envsubst < "$sssdFile_tmpl" > "$sssdFile"

    # Set the file ownership and permissions on sssd.conf
    chmod 0600 "$sssdFile"

    log "Debug: Exit conf_sssd_sssd"
}

#
# Customize configuration for Winbind AD Integration
# common function for SUSE platform
#
function conf_winbind()
{
    log "Debug: Enter function conf_winbind"
    displaymanagerFile="/etc/sysconfig/displaymanager"
    accountFile="/etc/pam.d/common-account-pc"
    passwordFile="/etc/pam.d/common-password-pc"
    sessionFile="/etc/pam.d/common-session-pc"
    authFile="/etc/pam.d/common-auth-pc"
    sshFile="/etc/ssh/ssh_config"
    sshdFile="/etc/ssh/sshd_config"

    pamFile="/etc/security/pam_winbind.conf"
    nsswitchFile="/etc/nsswitch.conf"

    # Customize /etc/sysconfig/displaymanager
    sed -i 's/^DISPLAYMANAGER_AD_INTEGRATION.*$/DISPLAYMANAGER_AD_INTEGRATION="yes"/g' "$displaymanagerFile"

    # Customize /etc/pam.d/common-account-pc
    sed -i 's/^account.*required.*pam_unix.so.*try_first_pass/account requisite       pam_unix.so     try_first_pass/g' "$accountFile"
    sed -i '/^account *sufficient *pam_localuser.so.*$/d' "$accountFile"
    sed -i '/^account *required *pam_winbind.so *use_first_pass.*$/d' "$accountFile"

    echo "account sufficient      pam_localuser.so
account required        pam_winbind.so  use_first_pass">>"$accountFile"

    # Customize /etc/pam.d/common-password-pc
    sed -i '/^password *sufficient *pam_winbind.so.*$/d' "$passwordFile"
    echo "password        sufficient      pam_winbind.so">>"$passwordFile"

    # Customize /etc/pam.d/common-session-pc
    sed -i '/^session *optional *pam_mkhomedir.so.*$/d' "$sessionFile"
    sed -i '/^session *required *pam_winbind.so.*$/d' "$sessionFile"
    echo "session optional        pam_mkhomedir.so">>"$sessionFile"
    echo "session required        pam_winbind.so">>"$sessionFile"

    # Customize /etc/pam.d/common-auth-pc
    sed -i '/^auth.*required.*pam_unix.so.*try_first_pass.*$/d' "$authFile"
    sed -i '/^auth *sufficient *pam_unix.so *try_first_pass.*$/d' "$authFile"
    sed -i '/^auth *required *pam_winbind.so *use_first_pass.*$/d' "$authFile"
    sed -i '/^auth *sufficient *pam_winbind.so *use_first_pass.*$/d' "$authFile"
    echo "auth    sufficient      pam_unix.so     try_first_pass">>"$authFile"
    echo "auth    required        pam_winbind.so  use_first_pass">>"$authFile"

    # Customize /etc/ssh/ssh_config
    sed -i '/^SendEnv *LANG *LC_CTYPE *LC_NUMERIC *LC_TIME *LC_COLLATE *LC_MONETARY *LC_MESSAGES.*$/d' "$sshFile"
    sed -i '/^SendEnv *LC_PAPER *LC_NAME *LC_ADDRESS *LC_TELEPHONE *LC_MEASUREMENT.*$/d' "$sshFile"
    sed -i '/^SendEnv *LC_IDENTIFICATION *LC_ALL.*$/d' "$sshFile"
    sed -i '/^Protocol *2.*$/d' "$sshFile"
    sed -i '/^SendEnv *LC_IDENTIFICATION *LC_ALL.*$/d' "$sshFile"
    sed -i '/^GSSAPIAuthentication *yes.*$/d' "$sshFile"
    sed -i '/^GSSAPIDelegateCredentials *yes.*$/d' "$sshFile"
    echo "Protocol 2
SendEnv LC_IDENTIFICATION LC_ALL
GSSAPIAuthentication yes
GSSAPIDelegateCredentials yes">>"$sshFile"


    # Customize /etc/ssh/sshd_config
    sed -i '/^GSSAPIAuthentication *yes.*$/d' "$sshdFile"
    sed -i '/^GSSAPICleanupCredentials *yes.*$/d' "$sshdFile"
    sed -i '/^ChallengeResponseAuthentication *yes.*$/d' "$sshdFile"
    echo "GSSAPIAuthentication yes
GSSAPICleanupCredentials yes
ChallengeResponseAuthentication yes">>"$sshdFile"

    # Customize /etc/samba/smb.conf
    #`sed -i '/kerberos method =.*$/d' "$smbFile"`
    #`sed -i '/winbind refresh tickets =.*$/d' "$smbFile"`      # del line in case user execute the script multi times
    #`sed -i '/\[global\]/a winbind refresh tickets = true' "$smbFile"`
    #`sed -i '/\[global\]/a kerberos method = secrets and keytab' "$smbFile"`
    conf_winbind_smb

    # Customize /etc/krb5.conf
    #`sed -i '/default_ccache_name.*$/d' "$krbFile"`
    #`sed -i '/\[libdefaults\]/a default_ccache_name = FILE:/tmp/krb5cc_%{uid}' "$krbFile"`
    conf_winbind_krb5


    # Customize /etc/nsswitch.conf
    passwdLine="$(< $nsswitchFile grep passwd |grep winbind 2>&1)"
    groupLine="$(< $nsswitchFile grep group |grep winbind 2>&1)"
    [[ -z "$passwdLine" ]] && sed -i 's/passwd:.*$/& winbind/g' "$nsswitchFile"
    [[ -z "$groupLine" ]] && sed -i 's/group:.*$/& winbind/g' "$nsswitchFile"


    # Customize /etc/security/pam_winbind.conf
    if [[ -f "$pamFile" ]]; then
        sed -i 's/.*krb5_auth =.*$/krb5_auth = yes/g' "$pamFile"
        sed -i 's/.*krb5_ccache_type =.*$/krb5_ccache_type = FILE/g' "$pamFile"
        sed -i 's/.*mkhomedir =.*$/mkhomedir = yes/g' "$pamFile"
    else
    echo "[global]
    krb5_auth = yes
    krb5_ccache_type = FILE
    mkhomedir = yes">"$pamFile"
    fi

    log "Debug: Exit function conf_winbind"
}

#
# Customize configuration for SSSD AD Integration
# common function for SUSE platform
#
function conf_sssd()
{
    log "Debug: Enter function conf_sssd"
    nsswitchFile="/etc/nsswitch.conf"

    # Disable nscd service
    log_echo "Disable nscd service"
    /usr/bin/systemctl stop nscd  >>"$logFile"  2>&1
    /usr/bin/systemctl disable nscd  >> "$logFile" 2>&1

    # Configure /etc/krb5.conf
    conf_sssd_krb5

    #Configure /etc/samba/smb.conf
    conf_sssd_smb

    # Configure /etc/nsswitch.conf
    sed -i 's/^passwd:.*$/passwd: compat sss/g' "$nsswitchFile"
    sed -i 's/^group:.*$/group:  compat sss/g' "$nsswitchFile"

    # Configure PAM
    pam-config --add --sss >> "$logFile" 2>&1
    pam-config --add --mkhomedir >> "$logFile" 2>&1

    conf_sssd_sssd

    log "Debug: Exit function conf_winbind"
}

function conf_usb_module()
{
    log "Debug: Enter conf_usb_module"
    #install system usb module
    #check usb_common.ko installed
    kernel_version=$(uname -r)
    lsmod | grep usb_common
    [[ "$?" -ne "0" ]] && insmod /lib/modules/"$kernel_version"/kernel/drivers/usb/common/usb-common.ko
    #check usbcore.ko installed
    lsmod | grep usbcore
    [[ "$?" -ne "0" ]] && insmod /lib/modules/"$kernel_version"/kernel/drivers/usb/core/usbcore.ko
    log "Debug: Exit conf_usb_module"
}

# this is called by ad_join service
function join_domain_setup_vda()
{
    log "Debug: Enter join_domain_setup_vda"
    conf_chrony
    if [ "$AD_INTEGRATION" == "winbind" ]; then
        join_domain_samba
        restart_service winbind /usr/bin/systemctl
        write_EncryptionTypes_attribute
    elif [ "$AD_INTEGRATION" == "sssd" ]; then
        join_domain_samba
        restart_service sssd /usr/bin/systemctl
        write_EncryptionTypes_attribute
    elif [ "$AD_INTEGRATION" == "pbis" ]; then
        join_domain_pbis
    fi
    conf_usb_module
    setup_vda
    log "Debug: Exit join_domain_setup_vda"
}

# this is called before dbus service start
function config_file()
{
    log "Debug: Enter config_file"
    # If its NDJ mode, this config is not needed
    if [ "$NON_DOMAIN_JOINED" == "Y" ]; then
        log "NDJ mode, config_file return"
        return 0
    fi
    # If DJ mode, continue the configuration
    conf_hostname
    if [ "$Use_Existing_Configurations_Of_Current_VDA" == "Y" ]; then
        log "Keep running configuration files"
    elif [ "$AD_INTEGRATION" == "winbind" ]; then
        conf_winbind
    elif [ "$AD_INTEGRATION" == "sssd" ]; then
        conf_sssd
    fi
    log "Debug: Exit config_file"
}

if [ "$1" == "--update-key" ] ; then
    update_private_key_timestamp
    exit 0
fi

if [ "$1" == "--backup-keydata" ] ; then
    backup_keydata_json_file
    exit 0
fi

check_user_and_vda_configured
check_ntfs_3g
read_id_disk

if [ "$1" == "--config" ] ; then
    config_file
    exit 0
elif [ "$1" == "--setup" ] ; then
    # If its NDJ mode, run NDJ setup
    if [ "$NON_DOMAIN_JOINED" == "Y" ]; then
        non_domain_joined_setup /bin/systemctl
    else
        join_domain_setup_vda
    fi

    # Set flag to indicate MCS setup is configured.
    set_mcs_setting_flag

    exit 0
else
    exit 1
fi
