#!/bin/bash

################################################################################
#
# Citrix Virtual Apps & Desktops For Linux Script: Machine Creation Service
# Copyright (c) Citrix Systems, Inc. All Rights Reserved.
#

source /var/xdl/mcs/mcs_util.sh

fname=$(basename "$0")
scriptName=${fname%.*}  # Script name without extension
logFile="/var/log/$scriptName.log"

dbus_service_file_path="/usr/lib/systemd/system/dbus.service"

## These pkgs need to be installed in order for LVDA pkg to work
## ntp and winbind related
pkgList=("krb5-client" \
         "krb5" \
         "bind" \
         "java-11-openjdk" \
         "libcap-progs")
pkgListNum=${#pkgList[@]}

winbind_pkgList=("samba-winbind")
winbind_pkgListNum=${#winbind_pkgList[@]}

sssd_pkgList=("samba-client" \
              "sssd" \
              "sssd-ad")
sssd_pkgListNum=${#sssd_pkgList[@]}

pkgMcs=("tdb-tools" \
         "libntfs-3g84")
pkgMcsNum=${#pkgMcs[@]}


function install_lvda_dependency_pkgs()
{
    log "Debug: Enter install_lvda_dependency_pkgs"
    # Install the common packages
    info="Installing Linux VDA dependency packages ..."
    log_echo "$info"
    for((i=0;i<pkgListNum;i++)); do
        info="Installing package ${pkgList[$i]}"
        log_echo "$info"
        zypper -n install "${pkgList[$i]}"  >> "$logFile" 2>&1
        if [[ "$?" -ne "0" ]]; then
            info="Error: Failed to install ${pkgList[$i]}"
            zypper info "${pkgList[$i]}" >> "$logFile" 2>&1
            [[ "$?" -ne "0" ]] && log_echo "$info"
        fi
    done

    # start PostgreSQL
    info="Starting PostgreSQL database ..."
    log_echo "$info"

    /usr/bin/systemctl enable postgresql.service  >> "$logFile" 2>&1
    /usr/bin/systemctl restart postgresql   >> "$logFile" 2>&1

    postgres psql -c 'show data_directory' >> "$logFile" 2>&1

    log "Debug: Exit install_lvda_dependency_pkgs"
}

function install_lvda_winbind_dependency_pkgs () {
    log "Debug: Enter install_lvda_winbind_dependency_pkgs"
    # Install winbind dependency packages
    info="Installing winbind dependency packages ..."
    log_echo "$info"
    for((i=0;i<winbind_pkgListNum;i++)); do
        info="Installing package ${winbind_pkgList[$i]}"
        log_echo "$info"
        zypper -n install "${winbind_pkgList[$i]}" >> "$logFile" 2>&1
        if [[ "$?" -ne "0" ]]; then
            info="Failed to install ${winbind_pkgList[$i]}"
            zypper info "${winbind_pkgList[$i]}" >> "$logFile" 2>&1
            [[ "$?" -ne "0" ]] && log_echo "$info"
        fi
    done

    # enable chrony and winbind service
    log_echo "enable winbind and chrony service"
    /usr/bin/systemctl enable winbind.service >> "$logFile" 2>&1
    /usr/bin/systemctl enable chronyd.service >> "$logFile" 2>&1

    log "Debug: Exit install_lvda_winbind_dependency_pkgs"
}

function install_lvda_sssd_dependency_pkgs () {
    log "Debug: Enter install_lvda_sssd_dependency_pkgs"
    # Install the common packages
    info="Installing Linux VDA sssd dependency packages ..."
    log_echo "$info"
    for((i=0;i<sssd_pkgListNum;i++)); do
        info="Installing package ${sssd_pkgList[$i]}"
        log_echo "$info"
        zypper -n install "${sssd_pkgList[$i]}" >> "$logFile" 2>&1
        if [[ "$?" -ne "0" ]]; then
            info="Failed to install ${sssd_pkgList[$i]}"
            zypper info "${sssd_pkgList[$i]}" >> "$logFile" 2>&1
            [[ "$?" -ne "0" ]] && log_echo "$info"
        fi
    done

    # enable ntp and sssd service
    log_echo "enable sssd and chronyd service"
    /usr/bin/systemctl enable sssd.service >> "$logFile" 2>&1
    /usr/bin/systemctl enable chronyd.service >> "$logFile" 2>&1

    log "Debug: Exit install_lvda_dependency_pkgs"
}

function install_mcs_pkgs()
{
    log "Debug: Enter install_mcs_pkgs"
    for ((i=0;i<pkgMcsNum;i++)); do
        info="Installing package ${pkgMcs[$i]}"
        log_echo "$info"
        zypper -n install "${pkgMcs[$i]}" >> "$logFile" 2>&1

        if [[ "$?" -ne "0" ]]; then
            log_echo "Error: install ${pkgMcs[$i]} failed."
        fi
    done

    log "Debug: Exit install_mcs_pkgs"
}

#
# Configure DNS
#
function conf_dns()
{
    log "Debug: Enter funtion conf_dns"
    [[ -z "$dns1" && -z "$dns2" && -z "$dns3" && -z "$dns4" ]] && \
            log_echo "ERROR: Exit funtion conf_dns, dns not configured, please do it manually." && \
            return

    conf_resolv_conf

    # Disable Multicast DNS
    if [ "$AD_INTEGRATION" != "pbis" ]; then
        configFile="/etc/nsswitch.conf"
        [[ ! -f "${configFile}" ]] && log_echo "ERROR: file $configFile does not exist!" && \
                                    log_echo "ERROR: Exit funtion conf_dns, dns not configured, please do it manually." && \
                                    return
        sed -i 's/^.*mdns_minimal.*$/hosts:       files dns/g' "$configFile"
        [[ "$?" -ne "0" ]] && log_echo "WARNING: Failed to disable Multicast DNS[Warning]."
        sed -i 's/^passwd:.*$/passwd: compat winbind/g' "$configFile"
        sed -i 's/^group:.*$/group:  compat winbind/g' "$configFile"
    fi
  
    # Name Server - Add the IP address of the DNS server. This is typically the IP address of the AD Domain Controller.
    # Domain Search list - Add the DNS domain name.awqw
    configFile="/etc/sysconfig/network/config"
    [[ ! -f "${configFile}" ]] && log_echo "ERROR: file $configFile does not exist!" && \
                                  log_echo "ERROR: Exit funtion conf_dns, dns not configured, please do it manually." && \
                                  return

    sed -i 's/^NETCONFIG_DNS_POLICY.*$/NETCONFIG_DNS_POLICY=\"STATIC\"/g' "$configFile"
    sed -i "s/^NETCONFIG_DNS_STATIC_SEARCHLIST.*$/NETCONFIG_DNS_STATIC_SEARCHLIST=\"$realm\"/g" $configFile
    sed -i 's/^NETCONFIG_DNS_FORWARDER.*$/NETCONFIG_DNS_FORWARDER=\"bind\"/g' "$configFile"
    sed -i '/NETCONFIG_DNS_STATIC_SERVERS.*$/d' "$configFile"

    [[ -n "$dns1" ]] && echo "NETCONFIG_DNS_STATIC_SERVERS=$dns1">>"$configFile"
    [[ -n "$dns2" ]] && echo "NETCONFIG_DNS_STATIC_SERVERS=$dns2">>"$configFile"
    [[ -n "$dns3" ]] && echo "NETCONFIG_DNS_STATIC_SERVERS=$dns3">>"$configFile"
    [[ -n "$dns4" ]] && echo "NETCONFIG_DNS_STATIC_SERVERS=$dns4">>"$configFile"

    log "Debug: Exit funtion conf_dns"
}


function main()
{
    log "Debug: Enter main"
    check_user
    create_log_file
    install_lvda_dependency_pkgs
    install_mcs_pkgs
    install_mcs_scripts
    # For NDJ mode, we need to create local mapped account group
    create_localuser_group

    hack_dbus_service_file $dbus_service_file_path
    if [ "$AD_INTEGRATION" == "winbind" ]; then
        install_lvda_winbind_dependency_pkgs 
    elif [ "$AD_INTEGRATION" == "sssd" ]; then
        install_lvda_sssd_dependency_pkgs
    elif [ "$AD_INTEGRATION" == "pbis" ]; then
        install_lvda_pbis_dependency_pkgs
    fi
    conf_dns
    log "Debug: Exit main"
}

main "$@"
