#!/bin/bash

#****************************************************************
# * Copyright (c) Citrix Systems, Inc. All Rights Reserved.
# *
# * This script is to help Linux VDA automatically self update.
# *
# * Return value: 
# *     0 - Normally return
# *     1 - Invalid parameters
# *     2 - Got no newer update than current version
# *     3 - Failed to get system info
# *     4 - Failed to check update
# *     5 - Failed to get Linux VDA package
# *     6 - Failed to schedule update
# *     7 - Failed to execute update
# *     8 - Failed to roll back
# *     100-599 - Http response status codes
# *
#****************************************************************

export LC_ALL="en_US.UTF-8"
localFolder="/opt/CitrixUpdate"
fname=$(basename $0)
scriptName=${fname%.*}  # Script name without extension
logFile="${localFolder}/${scriptName}.log"
updateInfoFile="UpdateInfo.json"
url=""
cert=""
certOption=""
os=""
currentVdaVer=""
newVdaVer=""
currentVdaName=""
newPkgName=""
osPlatform=""           # OS platform,  red, centos, suse, ubuntu
osVersion=""            # OS version
targetOs=""
pkgMng=""        # which kind of command will be used (apt, yum)


function usage {
    echo "Usage: "
    echo "  Get Linux VDA update info and schedule self update."
    echo "  Currently only Rhel and Ubuntu are supported."
    echo "  Check info and download package if it is newer:"
    echo "           ${fname} --checkupdate [url]"
    echo "  Execute update now:"
    echo "           ${fname} --updatenow"
}

function myPrint()
{
    echo -e "$1" | tee -a "$logFile"
}

function myLog
{
    time=$(date +"%Y-%m-%d %H:%M:%S.%3N")
    echo -e "[${time}] $1">>"${logFile}" 2>&1
}

#
# Get current system information
# populate the following variables:
#    osPlatform
#    osVersion
#
function getSysInfo()
{
    myLog "Enter getSysInfo()"

    # check if current user is root
    # if [ $id!=0 ]; then
    #     myPrint "MUST BE RUN BY ROOT!"
    #     exit 3
    # fi

    errInfoOS="Failed to get OS platform"
    errInfoVersion="Failed to get OS version"

    coreFileRhel="/etc/redhat-release"  
    coreFileCent="/etc/centos-release"  
    coreFileRocky="/etc/rocky-release"  
    coreFileSuse="/etc/SuSE-release"  
    coreFileUbuntu="/etc/lsb-release"
    coreFilePardus="/etc/lsb-release"
    coreFileOS="/etc/os-release"

    systemKernel=$(uname -v |grep -i Ubuntu 2>&1)

    #Check Core file, checking sequence is suse, cent, rhel, ubuntu, pardus
    if [[ -f "$coreFileSuse" ]]; then
        osPlatform="suse"
        osVersion=$(cat "$coreFileSuse" |grep "SUSE Linux Enterprise" |cut -d" " -f5|tr A-Z a-z  2>&1)
        if [[ "$?" -ne "0" || -z "$osVersion" ]]; then           
           myPrint "$errInfoVersion"
           exit 3
        fi
        
        if [[ "$osVersion" == "12" ]]; then
           osPatchVersion=$(cat "$coreFileSuse" |grep "PATCHLEVEL" |cut -d" " -f3 2>&1)           
           if [[ "$?" -ne "0" || -z "$osVersion" ]]; then           
               myPrint "$errInfoVersion"
               exit 3
           fi
        fi
    elif [[ -f "$coreFileRocky" ]]; then 
        osPlatform="rocky"
        osVersion="$(cat $coreFileRocky |cut -d ' ' -f4 |cut -d '.' -f1-2  2>&1)"    
        if [[ "$?" -ne "0" || -z "$osVersion" ]]; then           
            return 1
        fi
    elif [[ -f $coreFileCent ]]; then 
       osPlatform=$(cat $coreFileCent |cut -d ' ' -f1 |tr A-Z a-z  2>&1)
       if [[ "$?" -ne "0" || "$osPlatform" -ne "centos" ]]; then           
           myPrint "$errInfoOS"
           exit 3
       fi
      
       # the contants of $coreFileCent may be 
       # "CentOS release 6.8 (Final)"
       # or "CentOS Linux release 7.2.1511 (Core)"
       # we need to adjust the field  
       num=$(cat $coreFileCent |wc -w 2>&1)
       if [[ $num -lt 5 ]]; then
           num=`expr $num`
       else
           num=`expr $num - 1`
       fi

       osVersion=$(cat $coreFileCent |cut -d ' ' -f$num |cut -d '.' -f1-2  2>&1)
       if [[ "$?" -ne "0" || -z "$osVersion" ]]; then           
           myPrint "$errInfoVersion"
           exit 3
       fi
      
    elif [[ -f $coreFileRhel ]]; then
       osPlatform=$(cat $coreFileRhel |cut -d ' ' -f1 |tr A-Z a-z  2>&1)
       if [[ "$?" -ne "0" || "$osPlatform" -ne "red" ]]; then           
           myPrint "$errInfoOS"
           exit 3
       fi
       osVersion=$(cat $coreFileRhel |cut -d ' ' -f7  2>&1)

       if [[ $osVersion =~ ^[0-9.]*$ ]]; then
           myPrint "$osVersion"
       else
           osVersion=$(cat $coreFileRhel |cut -d ' ' -f6  2>&1)
           myPrint "$osVersion"
       fi

       if [[ "$?" -ne "0" || -z "$osVersion" ]]; then           
           myPrint "$errInfoVersion"
           exit 3
       fi

    elif [[ -f $coreFileUbuntu  && -n "$systemKernel" ]]; then
       osPlatform=$(cat $coreFileUbuntu |grep DISTRIB_ID |cut -d '=' -f2 |tr A-Z a-z  2>&1)
       if [[ "$?" -ne "0" || "$osPlatform" -ne "ubuntu" ]]; then           
           myPrint "$errInfoOS"
           exit 3
       fi
       osVersion=$(cat $coreFileUbuntu |grep DISTRIB_RELEASE |cut -d '=' -f2 |tr A-Z a-z  2>&1)
       if [[ "$?" -ne "0" || -z "${osVersion}" ]]; then           
           myPrint "$errInfoVersion"
           exit 3
       fi  
    # MUST follow Ubuntu, because Pardus utilizes the same file(lsb-release) as Ubuntu.
    elif [[ -f $coreFilePardus ]]; then
       osPlatform=$(cat $coreFilePardus |grep DISTRIB_ID |cut -d '=' -f2 |tr A-Z a-z  2>&1)
       if [[ "$?" -ne "0" || "$osPlatform" -ne "pardus" ]]; then           
           myPrint "$errInfoOS"
           exit 3
       fi
       osVersion=$(cat $coreFilePardus |grep DISTRIB_RELEASE |cut -d '=' -f2 |tr A-Z a-z  2>&1)
       if [[ "$?" -ne "0" || -z "${osVersion}" ]]; then           
           myPrint "$errInfoVersion"
           exit 3
       fi
    elif [[ -f "$coreFileOS" ]]; then
        osPlatform="$(< $coreFileOS grep ^ID= |cut -d '=' -f2 |tr '[:upper:]' '[:lower:]' |tr -d '"'  2>&1)"
        if [[ "$?" -ne "0" || -z "$osPlatform" ]]; then           
            myPrint "$errInfoOS"
            exit 3
        fi

        if [[ $osPlatform == "sles" || $osPlatform == "sled" ]]; then
            osPlatform="suse"
        fi

        osVersion=$(< $coreFileOS grep VERSION_ID |cut -d '=' -f2 |tr '[:upper:]' '[:lower:]' |tr -d '"' 2>&1)
        if [[ "$?" -ne "0" || -z "${osVersion}" ]]; then           
            myPrint "$errInfoVersion"
            exit 3
        fi  
    fi 
   
    # Change all strings to be lower case
    osPlatform=`tr '[A-Z]' '[a-z]' <<<"$osPlatform"`
    osVersion=`tr '[A-Z]' '[a-z]' <<<"$osVersion"`

    myLog "Info: osPlatform=$osPlatform"
    myLog "Info: osVersion=$osVersion"

    #distributions=["RHEL7_9","RHEL8_3","SUSE12_5","UBUNTU16_04","UBUNTU18_04","UBUNTU20_04","PARDUS17_1","DEBIAN10"]

    # Check the Linux platform and version to set and 'targetOs' and 'pkgMng'
    if [[ "$osPlatform" == "red" || "$osPlatform" = "centos" || "$osPlatform" = "amzn" ]]; then
        if [[ "${osVersion}" == "7.8" || "${osVersion}" == "7.9" || "${osVersion}" == "2" ]]; then
            targetOs="RHEL7_9"
        elif [[ "${osVersion}" == "8.1" || "${osVersion}" == "8.2" || "${osVersion}" == "8.3" ]]; then
            targetOs="RHEL8_3"
        else
            myPrint "Unsupported OS"
            exit 3
        fi
        pkgMng="yum"
    elif [[ "${osPlatform}" == "ubuntu" ]]; then
        if [[ "${osVersion}" == "16.04" ]]; then
            targetOs="UBUNTU16_04"
        elif [[ "${osVersion}" == "18.04" ]]; then
            targetOs="UBUNTU18_04"
        elif [[ "${osVersion}" == "20.04" ]]; then
            targetOs="UBUNTU20_04"
        else
            myPrint "Unsupported OS"
            exit 3
        fi
        pkgMng="apt"
    # elif [[ "${osPlatform}" == "suse" ]]; then
    #     if [[ "${osVersion}" == "12" ]]; then
    #         targetOs="SUSE12_5"
    #     else
    #         myPrint "Unsupported OS"
    #         exit 3
    #     fi
    #     pkgMng="zypper"
    # elif [[ "${osPlatform}" == "pardus" ]]; then
    #     if [[ "${osVersion}" == "17.5" ]]; then
    #         targetOs="PARDUS17_1"
    #     else
    #         myPrint "Unsupported OS"
    #         exit 3
    #     fi
    #     pkgMng="apt"
    # elif [[ "${osPlatform}" == "debian" ]]; then
    #     if [[ "${osVersion}" == "10" ]]; then
    #         targetOs="DEBIAN10"
    #     else
    #         myPrint "Unsupported OS"
    #         exit 3
    #     fi
    #     pkgMng="apt"
    else
        myPrint "Unsupported OS"
        exit 3
    fi

    myLog "Info: targetOs=${targetOs}, pkgMng=$pkgMng"

    myLog "Debug: Exit function getSysInfo"
}

function getCurrentVdaVer()
{
    # get current vda version
    if [[ ${pkgMng} == "apt" || ${newPkgName} == *".deb" ]]; then
        currentVdaVer=$(dpkg -l | grep xendesktopvda | tr -s ' ' | cut -d ' ' -f 3 | cut -d '-' -f1)
    else
        currentVdaVer=$(rpm -qa | grep XenDesktopVDA | cut -d '-' -f2)
    fi
    ret=$?
    if [[ $ret != 0 || -z "${currentVdaVer}" ]]; then
        myLog "Failed to get current vda version"
        exit 4
    else
        myLog "Got current vda version: ${currentVdaVer}"
    fi
}

function installJQ()
{
    myLog "Entry installJQ()"

    local ret=0

    which jq
    ret=$?
    if [ ${ret} == 0 ]; then
        myLog "jq is already installed"
        return 0
    fi

    myLog "Will install jq..."
    if [[ ${pkgMng} == "apt" ]]; then
        apt-get install -y jq >> "$logFile" 2>&1
    elif [[ ${pkgMng} == "yum" ]]; then
        if [[ ${targetOs} == "RHEL7_9" ]]; then
            rpm -ivh https://dl.fedoraproject.org/pub/epel/epel-release-latest-7.noarch.rpm >> "$logFile" 2>&1
        fi
        yum install -y jq >> "$logFile" 2>&1
    # elif [[ ${pkgMng} == "zypper" ]]; then
    #     zypper install -y jq >> "$logFile" 2>&1
    fi

    ret=$?
    if [ $ret != 0 ]; then
        myLog "Failed to install jq"
        exit 4
    fi
}

function checkUpdate()
{
    myLog "Entry checkUpdate()"

    local ret=0

    # query update info
    url="$1"
    myLog "Got url: ${url}"

    if [ ! -d "${localFolder}" ]; then
        myLog "Directory ${localFolder} DOES NOT exists."
        exit 4
    fi

    cert=$(/opt/Citrix/VDA/bin/ctxreg dump | grep SelfUpdate |grep CaCertificate |cut -d' ' -f9)
    cert=$(echo "${cert//\"}")  # remove the quotes
    myLog "Got certificate: ${cert}"
    if [[ ${cert} == *".pem" ]]; then
        myLog "Got certificate ${cert}."
        certOption=" --cacert ${cert}"
    else
        myLog "Got no or wrong-format certificate ${cert}. Won't use certificate verification."
    fi

    ret=$(curl -s -w '%{http_code}%{ssl_verify_result}' ${url}/${updateInfoFile} -o ${localFolder}/${updateInfoFile} ${certOption})
    if [ ${ret} != 2000 ]; then
        myLog "Failed to execute curl with ret ${ret}"
        exit $ret
    fi
    myLog "Successfully queried update info and saved it locally."

    getCurrentVdaVer

    installJQ

    # get new vda version
    newVdaVer=$(cat ${localFolder}/${updateInfoFile} | jq '.Version')
    ret=$?
    newVdaVer=$(echo "${newVdaVer//\"}")    # remove the quotes
    if [[ $ret != 0 || -z "${newVdaVer}" ]]; then
        myLog "Failed to retrieve newVdaVer from json"
        exit 4
    else
        myLog "Got update version: ${newVdaVer}"
    fi

    # compare if it is needed to update
    if [ $(printf '%s\n' ${newVdaVer} ${currentVdaVer} | sort -V | head -n1) != "$newVdaVer" ]; then
        myLog "Update version is newer than current one. Need to update."
    else
        myLog "Update version is not newer than current one. Ignore."
        exit 2
    fi

    return 0
}

function getNewLinuxVdaPkg()
{
    myLog "Entry getNewLinuxVdaPkg()"

    local ret=0
    # get update package name
    newPkgName=$(jq '.Distributions[] | select(.TargetOS=="'${targetOs}'") | .PackageName' ${localFolder}/${updateInfoFile})
    ret=$?
    newPkgName=$(echo "${newPkgName//\"}")  # remove the quotes
    if [[ $ret != 0 || -z "${newPkgName}" ]]; then
        myLog "Failed to retrieve package name from json"
        exit 5
    fi

    # download the update package
    myLog "Downloading the new update package..."
    ret=$(curl -s -w '%{http_code}%{ssl_verify_result}' ${url}/${newPkgName} -o ${localFolder}/${newPkgName} ${certOption})
    if [ ${ret} != 2000 ]; then
        myLog "Failed to execute curl to get package with ret: $ret"
        exit $ret
    fi
    myLog "Successfully got Linux VDA package and saved it locally."

    # get update package sha-256 value
    local sha256=$(jq '.Distributions[] | select(.TargetOS=="'${targetOs}'") | .PackageHash' ${localFolder}/${updateInfoFile})
    ret=$?
    sha256=$(echo "${sha256//\"}")  # remove the quotes
    if [[ $ret != 0 || -z "${sha256}" ]]; then
        myLog "Failed to get package hash from json"
        exit 5
    else
        myLog "Got package hash ${sha256} from json"
    fi
    # verify sha-256 value with downloaded package
    echo "${sha256} ${localFolder}/${newPkgName}" | sha256sum -c >> "$logFile" 2>&1
    ret=$?
    if [ $ret != 0 ]; then
        myLog "Failed to verify package hash"
        exit 5
    else
        myLog "Successfully verified package hash"
    fi

    # set ctxreg 'State'to 'UpdateReady', and 'PkgName' to {newPkgName}
	/opt/Citrix/VDA/bin/ctxreg create -k "HKLM\System\CurrentControlSet\Control\Citrix\SelfUpdate" -t "REG_SZ" -v "State" -d "UpdateReady" --force
	/opt/Citrix/VDA/bin/ctxreg create -k "HKLM\System\CurrentControlSet\Control\Citrix\SelfUpdate" -t "REG_SZ" -v "PkgName" -d ${newPkgName} --force

    return 0
}

function scheduleUpdate()
{
    myLog "Entry scheduleUpdate()"

    scheduledTime=$(/opt/Citrix/VDA/bin/ctxreg dump | grep SelfUpdate |grep ScheduledTime |cut -d' ' -f9)
    scheduledTime=$(echo "${scheduledTime//\"}")    #remove the quotes

    if [[ ${scheduledTime} == "NextStart" ]]; then
        myLog "Will execute update upon next start of ctxmonitor"
    elif [[ ${scheduledTime} == "" ]]; then
        myLog "Failed to get reg 'ScheduledTime'"
        exit 6
    elif [[ ${scheduledTime} == "Immediately" ]]; then
        myLog "Will execute update now"
        ${localFolder}/${fname} --updatenow
    fi

    return 0
}

function deleteTempRegs()
{
    myLog "Clear registry 'State' and 'PkgName'. Exit."
    /opt/Citrix/VDA/bin/ctxreg delete -k "HKLM\System\CurrentControlSet\Control\Citrix\SelfUpdate" -v "State" --force >> "$logFile" 2>&1
    /opt/Citrix/VDA/bin/ctxreg delete -k "HKLM\System\CurrentControlSet\Control\Citrix\SelfUpdate" -v "PkgName" --force >> "$logFile" 2>&1
}

function rollBack()
{
    myLog "Entry rollBack()"

    deleteTempRegs

    if [[ ${newPkgName} == *".deb" ]]; then
        dpkg -i ${localFolder}/${currentVdaName} >> "$logFile" 2>&1 && apt-get install -f >> "$logFile" 2>&1
    elif [[ ${newPkgName} == *".el"* ]]; then
        yum -y localinstall ${localFolder}/${currentVdaName} >> "$logFile" 2>&1
    # elif [[ ${newPkgName} == *".sle"* ]]; then
    #     zypper -i --oldpackage install ${localFolder}/${currentVdaName} >> "$logFile" 2>&1
    else
        myLog "Unknown OS."
        exit 8
    fi

    ret=$?
    if [[ $ret != 0 ]]; then
        myLog "Failed to rollback to ${currentVdaName}."
        exit 8
    else
        myLog "Rollback succeeeded."
    fi
}

function executeUpdate()
{
    myLog "Entry executeUpdate()"

    local newPkgName=$(/opt/Citrix/VDA/bin/ctxreg dump | grep SelfUpdate |grep PkgName |cut -d' ' -f9)
    newPkgName=$(echo "${newPkgName//\"}")    #remove the quotes
    if [[ ${newPkgName} == "" || ! -f ${localFolder}/${newPkgName} ]]; then
        myLog "Failed to get local update package."
        deleteTempRegs
        exit 7
    fi

    # get current vda pkg name in case it need rollback
    getCurrentVdaVer
    currentVdaName=$(ls ${localFolder} | grep ${currentVdaVer} )

    # do update Linux VDA
    myLog "Will update now..."
    if [[ ${newPkgName} == *".deb" ]]; then
        dpkg -i ${localFolder}/${newPkgName} >> "$logFile" 2>&1 && apt-get install -f >> "$logFile" 2>&1
    elif [[ ${newPkgName} == *".el"* ]]; then
        yum -y localinstall ${localFolder}/${newPkgName} >> "$logFile" 2>&1
    # elif [[ ${newPkgName} == *".sle"* ]]; then
    #     zypper -i install ${localFolder}/${newPkgName} >> "$logFile" 2>&1
    else
        myLog "Unknown OS of the update package."
        deleteTempRegs
        exit 7
    fi

    # check result
    ret=$?
    systemctl -q is-active ctxmonitorservice.service >> "$logFile" 2>&1
    local status=$?
    if [[ $ret != 0 || status == 0 ]]; then
        myLog "Failed to update to ${newPkgName}"
        if [ ! -z ${currentVdaName} ]; then
            myLog "Will ROLL BACK to ${currentVdaName}!"
            rollBack
        fi
        exit 7
    else
        deleteTempRegs
        myLog "Update succeeded."
    fi

    return 0
}


if [ "$1" == "--help" ]; then
    usage
    exit 0
elif [ "$1" == "--checkupdate" ]; then
    if [ "$#" -ne 2 ]; then
        usage
        exit 1
    fi
    getSysInfo
    checkUpdate $2
    getNewLinuxVdaPkg
    scheduleUpdate
    exit 0
elif [ "$1" == "--updatenow" ]; then
    if [ "$#" -ne 1 ]; then
        usage
        exit 1
    fi
    executeUpdate
    exit 0
else
    usage
    exit 0
fi
