#!/bin/sh

ldconfig_update() {
    inner() {
        /sbin/ldconfig /opt/Citrix/VDA/lib64 || return 1
        /sbin/ldconfig /opt/Citrix/VDA/lib32 || return 1
    }
    cmdlog_exec inner "${FUNCNAME%% *}" "update ldconfig"
    result=$?
    unset -f inner
    [ "$result" = "0" ] || return 1
}

postgresql_start() {
    inner() {
        /bin/systemctl start postgresql || return 1
    }
    cmdlog_exec inner "${FUNCNAME%% *}" "start PostgreSQL"
    result=$?
    unset -f inner
    [ "$result" = "0" ] || return 1
}

postgresql_restart() {
    inner() {
        /bin/systemctl restart postgresql || return 1
    }
    cmdlog_exec inner "${FUNCNAME%% *}" "restart PostgreSQL"
    result=$?
    unset -f inner
    [ "$result" = "0" ] || return 1
}

service_start() {
    inner() {
        /bin/systemctl daemon-reload || return 1
        if [ -f /etc/systemd/system/ctxlogd.service ]; then
            /bin/systemctl start ctxlogd || return 1
        fi
        /bin/systemctl start ctxhdx || return 1
        /bin/systemctl start ctxpolicyd || return 1
        /bin/systemctl start ctxjproxy || return 1
        /bin/systemctl start ctxvda || return 1
        /bin/systemctl start ctxcups
        /bin/systemctl start cups
        /bin/systemctl start ctxcdm
        /bin/systemctl start ctxceip
        /bin/systemctl start ctxvhcid
        /bin/systemctl start ctxusbsd
        /bin/systemctl start ctxtelemetry.socket
        /bin/systemctl start ctxbcrd
        /bin/systemctl start ctxmonitorservice
        /bin/systemctl start ctxgdtd
        /bin/systemctl start ctxwcamsd
    }
    cmdlog_exec inner "${FUNCNAME%% *}" "start $short_name"
    result=$?
    unset -f inner
    [ "$result" = "0" ] || return 1
}

service_stop() {
    inner() {
        /bin/systemctl daemon-reload || return 1
        /bin/systemctl stop ctxwcamsd
        /bin/systemctl stop ctxgdtd
        /bin/systemctl stop ctxmonitorservice
        /bin/systemctl stop ctxbcrd
        pkill -f /opt/Citrix/VDA/bin/ctxfm
        /bin/systemctl stop ctxcdm
        /bin/systemctl stop ctxceip
        /bin/systemctl stop ctxcups
        /bin/systemctl stop cups
        /bin/systemctl stop ctxusbsd
        /bin/systemctl stop ctxvhcid
        /bin/systemctl stop ctxtelemetry.socket
        /bin/systemctl stop ctxtelemetry@*.service
        /bin/systemctl stop ctxpolicyd || return 1

        /bin/systemctl stop ctxvda || return 1
        /bin/systemctl stop ctxjproxy || return 1
        /bin/systemctl stop ctxhdx || return 1
        if [ -f /etc/systemd/system/ctxlogd.service ]; then
            /bin/systemctl stop ctxlogd || return 1
        fi
    }
    cmdlog_exec inner "${FUNCNAME%% *}" "stop $short_name"
    result=$?
    unset -f inner
    [ "$result" = "0" ] || return 1
}

service_kill() {
    inner() {
        /bin/systemctl is-active -q ctxwcamsd && /bin/systemctl kill ctxwcamsd
        /bin/systemctl is-active -q ctxgdtd && /bin/systemctl kill ctxgdtd
        /bin/systemctl is-active -q ctxmonitorservice && /bin/systemctl kill ctxmonitorservice
        /bin/systemctl is-active -q ctxbcrd && /bin/systemctl kill ctxbcrd
        /bin/systemctl is-active -q ctxvda && /bin/systemctl kill ctxvda
        /bin/systemctl is-active -q ctxjproxy && /bin/systemctl kill ctxjproxy
        /bin/systemctl is-active -q ctxtelemetry && /bin/systemctl kill ctxtelemetry


        /bin/systemctl is-active -q ctxhdx && /bin/systemctl kill ctxhdx
        if [ -f /etc/systemd/system/ctxlogd.service ]; then
            /bin/systemctl is-active -q ctxlogd && /bin/systemctl kill ctxlogd
        fi
        if [ -f /etc/systemd/system/ad_join.service ]; then
            /bin/systemctl is-active -q ad_join && /bin/systemctl kill ad_join
        fi

        return 0
    }
    cmdlog_exec inner "${FUNCNAME%% *}" "kill $short_name"
    result=$?
    unset -f inner
    [ "$result" = "0" ] || return 1
}

service_register() {
    inner() {
        /bin/systemctl daemon-reload || return 1
        if [ -f /etc/systemd/system/ctxlogd.service ]; then
            /bin/systemctl enable ctxlogd || return 1
        fi
        /bin/systemctl enable ctxhdx || return 1
        /bin/systemctl enable ctxvda || return 1
        /bin/systemctl enable ctxjproxy || return 1

        /bin/systemctl enable ctxpolicyd || return 1
        /bin/systemctl enable ctxcups
        /bin/systemctl enable cups
        /bin/systemctl enable ctxcdm
        /bin/systemctl enable ctxceip
        /bin/systemctl enable ctxvhcid
        /bin/systemctl enable ctxusbsd
        /bin/systemctl enable ctxtelemetry.socket
        /bin/systemctl enable ctxbcrd
        /bin/systemctl enable ctxmonitorservice
        /bin/systemctl enable ctxgdtd
        /bin/systemctl enable ctxwcamsd
        /bin/systemctl enable ctxrunatboot
    }
    cmdlog_exec inner "${FUNCNAME%% *}" "register $short_name services"
    result=$?
    unset -f inner
    [ "$result" = "0" ] || return 1
}

service_unregister() {
    inner() {
        /bin/systemctl daemon-reload || return 1
        /bin/systemctl disable ctxwcamsd
        /bin/systemctl disable ctxgdtd
        /bin/systemctl disable ctxmonitorservice
        /bin/systemctl disable ctxbcrd
        /bin/systemctl disable ctxcups
        /bin/systemctl disable cups
        /bin/systemctl disable ctxusbsd
        /bin/systemctl disable ctxvhcid
        /bin/systemctl disable ctxpolicyd || return 1
        /bin/systemctl disable ctxcdm
        /bin/systemctl disable ctxceip
        /bin/systemctl disable ctxtelemetry.socket

        /bin/systemctl disable ctxvda || return 1
        /bin/systemctl disable ctxjproxy || return 1
        /bin/systemctl disable ctxhdx || return 1
        if [ -f /etc/systemd/system/ctxlogd.service ]; then
            /bin/systemctl disable ctxlogd || return 1
        fi
        if [ -f /etc/systemd/system/ad_join.service ]; then
            /bin/systemctl disable ad_join || return 1
        fi
    }
    cmdlog_exec inner "${FUNCNAME%% *}" "unregister $short_name services"
    result=$?
    unset -f inner
    [ "$result" = "0" ] || return 1
}

firewall_add_rules() {
    inner() {
        if /usr/bin/firewall-cmd --state; then
            /usr/bin/firewall-cmd --permanent --zone=public --add-port="$1"/tcp || return 1
            /usr/bin/firewall-cmd --permanent --zone=public --add-port="$2"/tcp || return 1
            /usr/bin/firewall-cmd --permanent --zone=public --add-port="$3"/tcp || return 1
            /usr/bin/firewall-cmd --permanent --zone=public --add-port=1494/tcp || return 1
            /usr/bin/firewall-cmd --permanent --zone=public --add-port=2598/tcp || return 1
            /usr/bin/firewall-cmd --permanent --zone=public --add-port=1494/udp || return 1
            /usr/bin/firewall-cmd --permanent --zone=public --add-port=2598/udp || return 1
            /usr/bin/firewall-cmd --permanent --zone=public --add-port=8008/tcp || return 1
            /usr/bin/firewall-cmd --permanent --zone=public --add-port=6001-6099/tcp || return 1
        else
            return 0
        fi
    }
    cmdlog_exec inner "${FUNCNAME%% *}" "add firewall rules for $short_name" "$1" "$2" "$3"
    result=$?
    unset -f inner
    [ "$result" = "0" ] || return 1
}

firewall_update_rules() {
    inner() {
        if /usr/bin/firewall-cmd --state; then
            /usr/bin/firewall-cmd --permanent --zone=public --add-port=2598/tcp || return 1
            /usr/bin/firewall-cmd --permanent --zone=public --add-port=1494/udp || return 1
            /usr/bin/firewall-cmd --permanent --zone=public --add-port=2598/udp || return 1
            /usr/bin/firewall-cmd --permanent --zone=public --add-port=8008/tcp || return 1
            /usr/bin/firewall-cmd --permanent --zone=public --add-port=6001-6099/tcp || return 1
        else
            return 0
        fi
    }
    cmdlog_exec inner "${FUNCNAME%% *}" "update firewall rules for ${short_name}" "$1"
    result=$?
    unset -f inner
    [ "$result" = "0" ] || return 1
}

firewall_remove_rules() {
    inner() {
        if /usr/bin/firewall-cmd --state; then
            output="$(/opt/Citrix/VDA/bin/ctxreg read -k 'HKLM\Software\Citrix\VirtualDesktopAgent' -v 'ControllerRegistrarPort')" || return 1
            vda_port="$(printf "%d" "$output")" || return 1
            export vda_port

            telemetry_socket_port="$(grep -oP '(?<=ListenStream=)\d+' /etc/systemd/system/ctxtelemetry.socket)" || return 1
            export telemetry_socket_port

            # since we will try to remove the firewall first before set the telemetery port, so here we may fail to query the 'TelemetryServicePort'
            output="$(/opt/Citrix/VDA/bin/ctxreg read -k 'HKLM\Software\Citrix\VirtualDesktopAgent' -v 'TelemetryServicePort')" || output="7502"
            telemetry_port=$(printf "%d" "$output") || return 1
            export telemetry_port

            /usr/bin/firewall-cmd --permanent --zone=public --remove-port="$telemetry_socket_port"/tcp || return 1
            /usr/bin/firewall-cmd --permanent --zone=public --remove-port="$telemetry_port"/tcp || return 1
            /usr/bin/firewall-cmd --permanent --zone=public --remove-port="$vda_port"/tcp || return 1
            /usr/bin/firewall-cmd --permanent --zone=public --remove-port=1494/tcp || return 1
            /usr/bin/firewall-cmd --permanent --zone=public --remove-port=2598/tcp || return 1
            /usr/bin/firewall-cmd --permanent --zone=public --remove-port=1494/udp || return 1
            /usr/bin/firewall-cmd --permanent --zone=public --remove-port=2598/udp || return 1
            /usr/bin/firewall-cmd --permanent --zone=public --remove-port=8008/tcp || return 1
            /usr/bin/firewall-cmd --permanent --zone=public --remove-port=6001-6099/tcp || return 1
        else
            return 0
        fi
    }
    cmdlog_exec inner "${FUNCNAME%% *}" "remove firewall rules for $short_name"
    result=$?
    unset -f inner
    [ "$result" = "0" ] || return 1
}

firewall_try_restart() {
    inner() {
        if /usr/bin/firewall-cmd --state; then
            /usr/bin/firewall-cmd --reload || return 1
        else
            return 0
        fi
    }
    cmdlog_exec inner "${FUNCNAME%% *}" "restart the firewall"
    result=$?
    unset -f inner
    [ "$result" = "0" ] || return 1
}

ctxXrandrhook() {
    BEGIN_PATTERN="# Citrix configuration begin"
    MAIN_CONTENT="export LD_PRELOAD='/usr/\$LIB/libctxXrandrhook.so'"
    END_PATTERN="# Citrix congiguration end"
    XCLIENTS="/etc/X11/xinit/xinitrc"
    XCLIENTS_ORIG="/etc/X11/xinit/xinitrc.orig"
    [ "$#" -ne "1" ] && return 1
    case $1 in
        "configure")
            if [ -f $XCLIENTS ]; then
                test -f $XCLIENTS_ORIG || cp $XCLIENTS $XCLIENTS_ORIG
                sed -i -e "/$BEGIN_PATTERN/,/$END_PATTERN/d" $XCLIENTS
                awk -v "txt=$BEGIN_PATTERN\n$MAIN_CONTENT\n$END_PATTERN" '/^[^#]/ && !x {print txt; x=1} 1' "$XCLIENTS" > "${XCLIENTS}.tmp" && mv -f "${XCLIENTS}.tmp" "$XCLIENTS" && chmod a+x+r "$XCLIENTS"
            fi
            ;;
        "restore")
            sed -i -e "/$BEGIN_PATTERN/,/$END_PATTERN/d" $XCLIENTS
            ;;
        *)
            ;;
    esac
}

mate_session() {
    action=$1
    DEFAULT_WM="default"
    windowmanager_conf="/etc/sysconfig/windowmanager"

    if [ -f ${windowmanager_conf} ]; then
        WM_OUTPUT="$(< ${windowmanager_conf} grep ^DEFAULT_WM=)"
        if [ -n "${WM_OUTPUT}" ]; then
            DEFAULT_WM="$(echo "${WM_OUTPUT}" |cut -d '=' -f2|tr '[:upper:]' '[:lower:]' |tr -d '"')"
        fi
    fi

    case "${DEFAULT_WM}" in
        "gnome")
            if [ "$action" = "configure" ]; then
                MATE_SESSION="$(update-alternatives --list default-xsession.desktop | grep mate.desktop)"
                if [ -n "${MATE_SESSION}" ]; then
                    sed -i 's/^DEFAULT_WM=.*$/DEFAULT_WM="mate"/g' "${windowmanager_conf}"
                fi
            fi
            ;;
        "mate")
            if [ "$action" = "restore" ]; then
                GNOME_SESSION="$(update-alternatives --list default-xsession.desktop | grep gnome.desktop)"
                if [ -n "${GNOME_SESSION}" ]; then
                    sed -i 's/^DEFAULT_WM=.*$/DEFAULT_WM="gnome"/g' "${windowmanager_conf}"
                fi
            fi
            ;;
        "default")
            default_xsession_desktop_conf="/etc/alternatives/default-xsession.desktop"
            if [ -L "${default_xsession_desktop_conf}" ]; then
                DEEAFULT_XSESSION="$(update-alternatives --display default-xsession.desktop | grep 'link currently points to')"
                DEEAFULT_XSESSION="${DEEAFULT_XSESSION##*/}"
                case "${DEEAFULT_XSESSION%%.*}" in
                    "gnome")
                        if [ "$action" = "configure" ]; then
                            MATE_SESSION="$(update-alternatives --list default-xsession.desktop | grep mate.desktop)"
                            if [ -n "${MATE_SESSION}" ]; then
                                update-alternatives --set default-xsession.desktop "${MATE_SESSION}"
                            fi
                        fi
                        ;;
                    "mate")
                        if [ "$action" = "restore" ]; then
                            GNOME_SESSION="$(update-alternatives --list default-xsession.desktop | grep gnome.desktop)"
                            if [ -n "${GNOME_SESSION}" ]; then
                                update-alternatives --set default-xsession.desktop "${GNOME_SESSION}"
                            fi
                        fi
                        ;;
                    *)
                        ;;
                esac
            fi
            ;;
        *)
            ;;
    esac
}

mate_session_configure() {
    inner() {
        mate_session configure
    }

    cmdlog_exec inner "${FUNCNAME%% *}" "mate session configured"
    result=$?
    unset -f inner
    [ "$result" = "0" ] || return 1
}

mate_session_restore() {
    inner() {
        mate_session restore
    }

    cmdlog_exec inner "${FUNCNAME%% *}" "mate session restored"
    result=$?
    unset -f inner
    [ "$result" = "0" ] || return 1
}

desktop_environment_configure() {
    inner() {
        ctxXrandrhook configure
        cp -f /var/xdl/10-ctx-media-automount.rules /etc/udev/rules.d
        cp -f /var/xdl/50-ctx-persistent-wcam.rules /etc/udev/rules.d
        cp -f /var/xdl/51-ctx-persistent-v4l.rules  /etc/udev/rules.d
        cp -f /var/xdl/10-ctx-usb-disk-mount.rules /usr/share/polkit-1/rules.d
        cp -f /usr/lib/systemd/system/systemd-udevd.service /etc/systemd/system/systemd-udevd.service
        sed -i "s/MountFlags/#MountFlags/g" /etc/systemd/system/systemd-udevd.service
        systemctl daemon-reload
        systemctl restart systemd-udevd.service
        cp -f /var/xdl/10-ctxhdx-disable-power-management.rules /usr/share/polkit-1/rules.d || return 1
    }
    cmdlog_exec inner "${FUNCNAME%% *}" "desktop environment configured"
    result=$?
    unset -f inner
    [ "$result" = "0" ] || return 1
}

desktop_environment_restore() {
    inner() {
        ctxXrandrhook restore
        rm -f /etc/udev/rules.d/10-ctx-media-automount.rules
        rm -f /etc/udev/rules.d/50-ctx-persistent-wcam.ruless
        rm -f /etc/udev/rules.d/51-ctx-persistent-v4l.ruless
        rm -f /usr/share/polkit-1/rules.d/10-ctx-usb-disk-mount.rules
        rm -f /etc/systemd/system/systemd-udevd.service
        systemctl daemon-reload
        systemctl restart systemd-udevd.service
        rm -f /usr/share/polkit-1/rules.d/10-ctxhdx-disable-power-management.rules || return 1
    }
    cmdlog_exec inner "${FUNCNAME%% *}" "desktop environment restored"
    result=$?
    unset -f inner
    [ "$result" = "0" ] || return 1
}

remove_mcs_config() {
    inner() {
        dbus_service_file_path="/usr/lib/systemd/system/dbus.service"
        sed -i -e '/^ExecStartPre=.*$/ d' "$dbus_service_file_path"
        cron_job="/etc/cron.d/mcs_update_password_cronjob"
        if [ -f $cron_job ]; then
            rm -f $cron_job
        fi
    }
    cmdlog_exec inner "${FUNCNAME%% *}" "Remove mcs configurations"
    result=$?
    unset -f inner
    [ "$result" = "0" ] || return 1
}
