#!/bin/bash

################################################################################
#
# Citrix Virtual Apps & Desktops For Linux Script: Machine Creation Service
# Copyright (c) Citrix Systems, Inc. All Rights Reserved.
#

source /var/xdl/mcs/mcs_util.sh

fname=$(basename $0)
scriptName=${fname%.*}  # Script name without extension
logFile="/var/log/$scriptName.log"


function conf_hostname()
{
    log "Debug: Enter conf_hostname"

    cloud_cfg_file="/etc/cloud/cloud.cfg"
    if [[ -f "$cloud_cfg_file" ]]; then
        sed -i "s/preserve_hostname:.*/preserve_hostname: true/g" "$cloud_cfg_file"
    fi

    hosts_file="/etc/hosts"
    hostname_file="/etc/hostname"
    content_dns="127.0.0.1 ${NEW_HOSTNAME}.${realm} $NEW_HOSTNAME localhost"
    content="127.0.1.1 $NEW_HOSTNAME"

    sed -i "1i$content_dns" "${hosts_file}"
    sed -i "1a$content" "${hosts_file}"
    echo "$content" >> "$hosts_file"
    echo "$NEW_HOSTNAME" > "$hostname_file"
    hostname "$NEW_HOSTNAME"

    sysctl -w kernel.hostname="$NEW_HOSTNAME"  >> "$logFile" 2>&1
    log "Debug: Exit conf_hostname"
}

function conf_ntp()
{
    log "Debug: Enter conf_ntp"

    ntp_file_path="/etc/chrony/chrony.conf"
    remove_default_ntp_server $ntp_file_path
    echo "server $NTP_SERVER iburst" >> "$ntp_file_path"

    # sync time with ntp server
    log_echo "restart chronyd daemon"
    /bin/systemctl restart chrony.service  >> "$logFile" 2>&1

    sync_with_ntp_server $NTP_SERVER

    log "Debug: Exit conf_ntp"
}

function conf_winbind()
{
    log "Debug: Enter conf_winbind"

    # cutomize configuration files
    winbind_smbFile="/etc/xdl/mcs/winbind_smb.conf.tmpl"
    smbFile="/etc/samba/smb.conf"
    winbind_krbFile="/etc/xdl/mcs/winbind_krb5.conf.tmpl"
    krbFile="/etc/krb5.conf"
    nsswitchFile="/etc/nsswitch.conf"
    pamFile="/etc/security/pam_winbind.conf"

    # Customize /etc/samba/smb.conf
    WORKGROUP=$WORKGROUP REALM=$REALM envsubst < "$winbind_smbFile" > "$smbFile"


    # Customize /etc/krb5.conf
    REALM=$REALM AD_FQDN=$AD_FQDN realm=$realm envsubst < "$winbind_krbFile" > "$krbFile"


    # Customize /etc/nsswitch.conf
    passwdLine="$(cat $nsswitchFile |grep passwd |grep winbind 2>&1)"
    groupLine="$(cat $nsswitchFile |grep group |grep winbind 2>&1)"
    hostsLine="$(cat $nsswitchFile |grep hosts |grep  mdns4_minimal 2>&1)"
    [[ -z "$passwdLine" ]] && `sed -i 's/passwd:.*$/& winbind/g' "$nsswitchFile"`
    [[ -z "$groupLine" ]] && `sed -i 's/group:.*$/& winbind/g' "$nsswitchFile"`
    [[ ! -z "$hostsLine" ]] && `sed -i 's/hosts:.*$/hosts:          files dns/g' "$nsswitchFile"`

    # command will pop-up a window and wait for usr to input, in order to reduce user interaction,
    # we use the following code lines to change the configuration directly.
    #sudo pam-auth-update
    local configFile="/etc/pam.d/common-session"
    mkhomedirLine="$(cat $configFile |grep session |grep pam_mkhomedir 2>&1)"
    [[ -z "$mkhomedirLine" ]] && echo "session optional            pam_mkhomedir.so" >> "${configFile}"

    log "Debug: Exit conf_winbind"
}

function conf_sssd()
{
    log "Debug: Enter conf_sssd"

    # cutomize configuration files
    sssd_smbFile="/etc/xdl/mcs/sssd_smb.conf.tmpl"
    smbFile="/etc/samba/smb.conf"
    sssd_krbFile="/etc/xdl/mcs/sssd_krb5.conf.tmpl"
    krbFile="/etc/krb5.conf"
    nsswitchFile="/etc/nsswitch.conf"
    pamFile="/etc/security/pam_winbind.conf"
    sssdFile_tmpl="/etc/xdl/mcs/sssd.conf.tmpl"
    sssdFile="/etc/sssd/sssd.conf"

    # Customize /etc/krb5.conf
    REALM=$REALM AD_FQDN=$AD_FQDN realm=$realm envsubst < "$sssd_krbFile" > "$krbFile"

    # Customize /etc/samba/smb.conf
    WORKGROUP=$WORKGROUP REALM=$REALM envsubst < "$sssd_smbFile" > "$smbFile"

    # Customize /etc/nsswitch.conf
    passwdLine="$(cat $nsswitchFile |grep passwd |grep sss 2>&1)"
    shadowLine="$(cat $nsswitchFile |grep shadow |grep sss 2>&1)"
    groupLine="$(cat $nsswitchFile |grep group | grep -v netgroup | grep sss 2>&1)"
    netgroupLine="$(cat $nsswitchFile |grep netgroup |grep sss 2>&1)"
    serviceLine="$(cat $nsswitchFile |grep services |grep sss 2>&1)"
    hostsLine="$(cat $nsswitchFile |grep hosts |grep  mdns4_minimal 2>&1)"
    [[ -z "$passwdLine" ]] && `sed -i 's/^passwd:.*$/passwd:     files sss/g' "$nsswitchFile"`
    [[ -z "$shadowLine" ]] && `sed -i 's/^shadow:.*$/shadow:     files sss/g' "$nsswitchFile"`
    [[ -z "$groupLine" ]] && `sed -i 's/^group:.*$/group:      files sss/g' "$nsswitchFile"`
    [[ -z "$serviceLine" ]] && `sed -i 's/^services:.*$/services:   files sss/g' "$nsswitchFile"`
    [[ -z "$netgroupLine" ]] && `sed -i 's/^netgroup:.*$/netgroup:   files sss/g' "$nsswitchFile"`
    [[ ! -z "$hostsLine" ]] && `sed -i 's/hosts:.*$/hosts:          files dns/g' "$nsswitchFile"`

    # under certain case, some lines are not commented out, we need to remove them
    # Customize /etc/security/pam_winbind.conf
    if [[ -f "$pamFile" ]]; then
        `sed -i 's/.*krb5_auth =.*$/krb5_auth = yes/g' "$pamFile"`
        `sed -i 's/.*krb5_ccache_type =.*$/krb5_ccache_type = FILE/g' "$pamFile"`
        `sed -i 's/.*mkhomedir =.*$/mkhomedir = yes/g' "$pamFile"`
    else
        echo "[global]
    krb5_auth = yes
    krb5_ccache_type = FILE
    mkhomedir = yes">"$pamFile"
    fi

    # Customize /etc/sssd/sssd.conf
    REALM=$REALM envsubst < "$sssdFile_tmpl" > "$sssdFile"


    # Set the file ownership and permissions on sssd.conf

    chmod 0600 "$sssdFile"
    # command will pop-up a window and wait for usr to input, in order to reduce user interaction,
    # we use the following code lines to change the configuration directly.
    #sudo pam-auth-update
    local configFile="/etc/pam.d/common-session"
    mkhomedirLine="$(cat $configFile |grep session |grep pam_mkhomedir 2>&1)"
    [[ -z "$mkhomedirLine" ]] && echo "session optional            pam_mkhomedir.so" >> "${configFile}"

    log "Debug: Exit conf_sssd"
}

# this is called by ad_join service
function join_domain_setup_vda()
{
    log "Debug: Enter join_domain_setup_vda"
    conf_ntp
    if [ "$AD_INTEGRATION" == "winbind" ]; then
        join_domain_samba
        restart_service winbind /bin/systemctl
        write_EncryptionTypes_attribute
    elif [ "$AD_INTEGRATION" == "sssd" ]; then
        join_domain_samba
        restart_service sssd /bin/systemctl
        write_EncryptionTypes_attribute
    elif [ "$AD_INTEGRATION" == "pbis" ]; then
        join_domain_pbis
    fi
    setup_vda
    log "Debug: Exit join_domain_setup_vda"
}

# this is called before dbus service start
function config_file()
{
    log "Debug: Enter config_file"
    # If its NDJ mode, this config is not needed
    if [ "$NON_DOMAIN_JOINED" == "Y" ]; then
        log "NDJ mode, config_file return"
        return 0
    fi
    # If DJ mode, continue configuration
    conf_hostname
    if [ "$Use_Existing_Configurations_Of_Current_VDA" == "Y" ]; then
        log "Keep running configuration files"
        # if [ "$AD_INTEGRATION" == "centrify" ]; then
        #     conf_centrify_exist_domain
        #     conf_centrify_krb5_smb
        #     /bin/systemctl enable centrifydc 2>&1 >> "$logFile"
        # fi
    else
        if [ "$AD_INTEGRATION" == "winbind" ]; then
            conf_winbind
        elif [ "$AD_INTEGRATION" == "sssd" ]; then
            conf_sssd
        # elif [ "$AD_INTEGRATION" == "centrify" ]; then
        #     conf_centrify_nss_pam
        #     conf_centrify_krb5_smb
        #     /bin/systemctl enable centrifydc 2>&1 >> "$logFile"
        fi
    fi
    log "Debug: Exit config_file"
}

if [ "$1" == "--update-key" ] ; then
    update_private_key_timestamp
    exit 0
fi

if [ "$1" == "--backup-keydata" ] ; then
    backup_keydata_json_file
    exit 0
fi

check_user_and_vda_configured
check_ntfs_3g
read_id_disk

if [ "$1" == "--config" ] ; then
    config_file
    exit 0
elif [ "$1" == "--setup" ] ; then
    # If its NDJ mode, run NDJ setup
    if [ "$NON_DOMAIN_JOINED" == "Y" ]; then
        non_domain_joined_setup /bin/systemctl
    else
        join_domain_setup_vda
    fi

    # Set flag to indicate MCS setup is configured.
    set_mcs_setting_flag

    exit 0
else
    exit 1
fi
