#!/bin/bash

#******************************************************************************
# ctxsmartlogon.sh
#
# Copyright 1998-2018, 2018 Citrix Systems, Inc. All Rights Reserved.
#
# Authors: Long Xu, 6 Nov. 2018
#
# Citrix Virtual Apps & Desktops For Linux VDA Smart Card Logon Setup Script.
#
# All changes by this script can be manually reset by administrator, including:
#    a. /etc/krb5.conf
#    b. /etc/pam.d/ctxfsc
#    c. /etc/pam.d/smartcard-auth
#    d. /etc/samba/smb.conf
#    e. /etc/sssd/sssd.conf
#    f. newly installed packages: krb5-pkinit, libpam-krb5, pcsc-lite, pcsc-lite-ccid, pcsc-lite-libs, coolkey
#    g. /opt/Citrix/VDA/bin/ctxreg update -k "HKLM\System\CurrentControlSet\Control\Citrix\VirtualChannels\Scard" -v "PKCS11LibPath" -d "PATH"
#
#    FSC : Fast Smart Card
#    CSC : Common Smart Card
# Return value:
#    0 - [success] Normally return
#    1 - [failed]  None root users
#    2 - [failed]  Unsupported platform
#    3 - [failed]  Unsupported integrator way
#    4 - [failed]  File nonexistent
#    5 - [failed]  Service start failed
#    6 - [failed]  Failed to write
#
#*******************************************************************************

source /var/xdl/configure_common.sh
source /var/xdl/configure_platform.sh
source /var/xdl/configure_utilities.sh

# global variables
fname=$(basename $0)
osPlatform=""           # OS platform:  Redhat
osVersion=""            # Redhat 7 & 8
joinDomainWay=""        # legal value:  winbind, sssd, centrify
smartLogonType=""       # legal value:  None, SmartCard

# file
krb5File="/etc/krb5.conf"
pamFile=""
smbFile="/etc/samba/smb.conf"    # Ubuntu winbind
sssdFile="/etc/sssd/sssd.conf"   # Ubuntu sssd
kcmFile="/etc/krb5.conf.d/kcm_default_ccache"

# krb5.conf
default_ccache_name=""
kdc_hostname=""

# pkcs11 auth
pkinit_kdc_hostname=""
pkinit_identities=""
pkinit_anchors=""
pkinit_eku_checking=""
pkcs11Lib=""
preauth_options=""
rootCA=""
pkcs11Path=""

# pkcs11 library
pklibName="opensc-pkcs11.so"

# backup & log
backupDir="/var/ctxinstall"

#*******************************************************************************
# check user
function checkUser()
{
    if [ "$(id -u)" != 0 ]; then
        get_str CTXSMARTLOGON_PRECHECK_MUST_ROOT
        exit 1
    fi
}

function checkOS()
{
    # get version
    getOSInfo

    case "$osPlatform" in
        *ubuntu*)
            osPlatform="Ubuntu"
            ;;
        *red* | *centos* | *amazon* | *rocky* )
            osPlatform="Redhat/CentOS"
            ;;
        *debian*)
            osPlatform="Debian"
            ;;
        *)
            get_str CTXSMARTLOGON_PRECHECK_UNSUPPORT_PLATFORM "${osPlatform}"
            exit 2
            ;;
    esac

    if [ "${osPlatform}" == "Redhat/CentOS" ]; then
        case "$osVersion" in
            *7.*)
                osVersion="R/C7"
                ;;
            2)
                osVersion="R/C7"
                ;;
            *8.*)
                osVersion="R/C8"
                ;;
            *)
                get_str CTXSMARTLOGON_PRECHECK_UNSUPPORT_PLATFORM "${osPlatform}"
                exit 2
                ;;
        esac
    fi
}

function createLogFile()
{
    # make sure dir is exist
    if [[ ! -d "$backupDir" ]]; then
        mkdir -p "$backupDir"
    fi
}


#**************************************step 1***************************************
# TODO: The vda host will enable Fast Smart Card Logon by default.
function enableSmartLogon()
{
    str="$(get_str CTXSMARTLOGON_PREFACE_ENABLE_SMARTCARD)"
    while true ; do
        read -p "$str" opt
        case "${opt}" in
            ""|[yY] )
                smartLogonType="SmartCard"
                smartCardConf
                break;
                ;;
            [nN] )
                smartLogonType="None"
                expectToExit
                break;
                ;;
            * )
                get_str CTXSMARTLOGON_PREFACE_ENABLE_SMARTCARD
                continue
                ;;
        esac
    done
}

function smartCardConf()
{
    # restart smart card service
    if [ -f /etc/systemd/system/ctxscardsd.service ]; then
        /bin/systemctl restart ctxscardsd || return 5
        /bin/systemctl enable ctxscardsd || return 5
    fi
}

function expectToExit()
{
    # stop smart card service
    if [ -f /etc/systemd/system/ctxscardsd.service ]; then
        /bin/systemctl stop ctxscardsd || return 5
        /bin/systemctl disable ctxscardsd || return 5
    fi

    pcsclibhook restore

    echo "${fname} exit."

    exit 0
}

#**************************************step 2***************************************
# get ad integrator style
function getADJoinWay()
{
    if [[ "${osPlatform}" == "Redhat/CentOS" || "${osPlatform}" == "Ubuntu" || "${osPlatform}" == "Debian" ]]; then
        str="$(get_str CTXSMARTLOGON_PREFACE_SELECT_AD_WAY_CONFIGURATION)"
        while true ; do
            read -p "$str" optADJoinNum
            case "${optADJoinNum}" in
                ""|1 )
                    joinDomainWay="winbind"
                    break;
                    ;;
                2 )
                    joinDomainWay="sssd"
                    break;
                    ;;
                3 ) 
                    joinDomainWay="centrify"
                    break;
                    ;;
                * )
                    get_str CTXSMARTLOGON_PREFACE_SELECT_AD_WAY_CONFIGURATION
                    continue
                    ;;
            esac
        done
    fi
}

#**************************************step 3***************************************
function installPkgs()
{
    get_str CTXSMARTLOGON_PKG_INSTALL

    if [ "$smartLogonType" == "SmartCard" ]; then
        # install krb5 pkgs
        installKrb5Pkgs

        # install common smart card pkgs
        installCSCPkgs

        checkInstallPkgs
    fi
}

function checkInstallPkgs()
{
    case "$osPlatform" in
        "Redhat/CentOS")
            if [ ! -f /usr/lib64/libpcsclite.so.1 ] || \
               [ ! -f /usr/lib64/pkcs11/opensc-pkcs11.so ] || \
               [ ! -f /usr/lib64/krb5/plugins/preauth/pkinit.so ] || \
               [ ! -f /usr/lib64/security/pam_krb5.so ]; then
                get_str CTXSMARTLOGON_PKG_INSTALL_FAILED
            else
                get_str CTXSMARTLOGON_PKG_INSTALL_SUCCESS
            fi
            ;;
        "Ubuntu" | "Debian")
            if ([ ! -f /usr/lib/x86_64-linux-gnu/libpcsclite.so.1 ] && \
                [ ! -f /lib/x86_64-linux-gnu/libpcsclite.so.1 ]) || \
               [ ! -f /usr/lib/pcsc/drivers/ifd-ccid.bundle/Contents/Linux/libccid.so ] || \
               [ ! -f /usr/lib/x86_64-linux-gnu/opensc-pkcs11.so ] || \
               [ ! -f /usr/lib/x86_64-linux-gnu/krb5/plugins/preauth/pkinit.so ] || \
               [ ! -f /lib/x86_64-linux-gnu/security/pam_krb5.so ]; then
                get_str CTXSMARTLOGON_PKG_INSTALL_FAILED
            else
                get_str CTXSMARTLOGON_PKG_INSTALL_SUCCESS
            fi
            ;;

        *)
            get_str CTXSMARTLOGON_UNSUPPORTED_PLATFORM
            exit 2
            ;;
    esac
}


#
#   1. pcsc-lite pcsc-lite-ccid pcsc-lite-libs
#
function installCSCPkgs()
{
    case "$osPlatform" in
        "Redhat/CentOS")
            if [ -f /usr/lib64/libpcsclite.so.1 ] && \
               [ -f /usr/lib64/pkcs11/opensc-pkcs11.so ]; then
                get_str CTXSMARTLOGON_PKG_ALREADY_INSTALL "[pcsc-lite][pcsc-lite-ccid][pcsc-lite-libs][opensc]"
            else
                yum -y install pcsc-lite pcsc-lite-ccid pcsc-lite-libs opensc >/dev/null 2>&1
            fi
            ;;
        "Ubuntu" | "Debian")
            if ([ -f /usr/lib/x86_64-linux-gnu/libpcsclite.so.1 ] || \
                [ -f /lib/x86_64-linux-gnu/libpcsclite.so.1 ]) && \
               [ -f /usr/lib/pcsc/drivers/ifd-ccid.bundle/Contents/Linux/libccid.so ] && \
               [ -f /usr/lib/x86_64-linux-gnu/opensc-pkcs11.so ]; then
                get_str CTXSMARTLOGON_PKG_ALREADY_INSTALL "[pcsclite1][ccid][opensc]"
            else
                apt-get install -y libpcsclite1 libccid  opensc >/dev/null 2>&1
            fi
            ;;
        *)
            get_str CTXSMARTLOGON_UNSUPPORTED_PLATFORM
            exit 2
            ;;
    esac
}

#
#   1. pam_krb5
#   2. krb5-pkinit  
#
function installKrb5Pkgs()
{
    case "$osPlatform" in
        "Redhat/CentOS")
            if [ -f /usr/lib64/krb5/plugins/preauth/pkinit.so ]; then
                get_str CTXSMARTLOGON_PKG_ALREADY_INSTALL "[krb5-pkinit]"
            else
                yum -y install krb5-pkinit >/dev/null 2>&1
            fi
            ;;
        "Ubuntu" | "Debian")
            if [ -f /usr/lib/x86_64-linux-gnu/krb5/plugins/preauth/pkinit.so ] && \
               [ -f /lib/x86_64-linux-gnu/security/pam_krb5.so ]; then
                get_str CTXSMARTLOGON_PKG_ALREADY_INSTALL "[krb5-pkinit]"
            else
                apt-get install -y libpam-krb5 krb5-pkinit >/dev/null 2>&1
            fi
            ;;
        *)
            get_str CTXSMARTLOGON_UNSUPPORTED_PLATFORM
            exit 2
            ;;
    esac

    if [ "$osVersion" == "R/C7" ]; then
        if [ -f /usr/lib64/security/pam_krb5.so ]; then
            get_str CTXSMARTLOGON_PKG_ALREADY_INSTALL "[pam_krb5]"
        else
            yum -y install pam_krb5 >/dev/null 2>&1
        fi
    fi
}


#**************************************step 4***************************************
# configure /etc/krb5.conf
function configKrb5()
{
    if [[ $osVersion == "R/C8" ]] && [ -f "$kcmFile" ]; then
        # common out the kcm
        sed -i "/^.*default_ccache_name*/d" $kcmFile
        sed -i "/\[libdefaults\]/a \  #default_ccache_name = KCM:" $kcmFile
    fi

    if [ -f "$krb5File" ]; then
        # backup the conf file
        cp -f "$krb5File" "$backupDir"

        # centrify will lock krb5file, here unlock it
        chattr -i $krb5File

        #1. set the default_ccache_name
        if [ "$osPlatform" == "Redhat/CentOS" ]; then
            confCCacheName
        fi

        #2. add rules about auth_to_local
        auto_to_local="auth_to_local = RULE:[1:\$1@\$0]"
        #echo "  set $auto_to_local"

        #3. conf pkinit_kdc_hostname
        confPKinitKDC

        #4. Get pkcs11 lib path
        getIdDir

        #5. Get root CA path
        getRootCADir

        #6. conf pkinit_eku_checking
        pkinit_eku_checking="pkinit_eku_checking = kpServerAuth"
        #echo "  set $pkinit_eku_checking"

        #7. conf pkinit_cert_match
        pkinit_cert_match="pkinit_cert_match = ||<EKU>msScLogin,<KU>digitalSignature"

        # delete the older and write the rew
        sed -i "/^ *auth_to_local*/d" $krb5File
        sed -i "/^ *pkinit*/d" $krb5File

        if [[ "$osPlatform" == "Ubuntu" || "$osPlatform" == "SUSE" || "$osPlatform" == "Debian" ]]; then
            sed -i "/^\s*kdc\s*=\s*$kdc_hostname/a \ $auto_to_local\n $pkinit_kdc_hostname\n $pkinit_identities\n $pkinit_anchors\n $pkinit_eku_checking\n $pkinit_cert_match" $krb5File
        else # Redhat && CentOS
            sed -i "/^\s*kdc\s*=\s*$kdc_hostname/a \ $auto_to_local\n $pkinit_kdc_hostname\n $pkinit_anchors\n $pkinit_eku_checking\n $pkinit_cert_match" $krb5File
        fi

        # centrify should lock krb5file after configuration
        if [ "$joinDomainWay" == "centrify" ]; then
            chattr +i $krb5File
        fi

        get_str CTXSMARTLOGON_CFG_SUCCESS $krb5File
    else
        get_str CTXSMARTLOGON_NOT_EXIST $krb5File
        exit 4
    fi
}

function confCCacheName()
{
    default_ccache_name="default_ccache_name = FILE:/tmp/krb5cc_%{uid}"

    # winbind integration change the default_ccache_name
    if [ "$joinDomainWay" == "winbind" ]; then
        default_ccache_name="default_ccache_name = FILE:/tmp/krb5cc_%{uid}_XXXXXX"
    fi

    sed -i '/^\s*default_ccache_name*/d' $krb5File
    sed -i "/\[libdefaults\]/a \ $default_ccache_name" $krb5File

    #echo "  set $default_ccache_name"
}

function confPKinitKDC()
{
    # Get default realm
    default_realm=$(cat $krb5File | grep '^\s*default_realm\s*=')
    default_realm=${default_realm/*=} # remove the prefix
    default_realm=${default_realm/* } # remove the blank prefix

    # Get realm block
    realm_block=$(sed -n "/^\s*$default_realm\s*=\s*{/,/kdc\s*=/p" $krb5File)

    # Get kdc host name
    pkinit_kdc_hostname=${realm_block/*kdc}
    pkinit_kdc_hostname=${pkinit_kdc_hostname/*=} # remove the prefix
    pkinit_kdc_hostname=${pkinit_kdc_hostname/* } # remove the blank suffix
    pkinit_kdc_hostname=${pkinit_kdc_hostname/:*} # remove the suffix
    kdc_hostname=$pkinit_kdc_hostname

    # SUSE pkinit_kdc_hostname = HOST.domain.local
    if [ "$osPlatform" == "SUSE" ]; then
        hostname=${pkinit_kdc_hostname%%\.*}     # the first '.' before
        hostname=$(echo "$hostname" | tr '[:lower:]' '[:upper:]') # change the realm to upper case
        realm=${pkinit_kdc_hostname#*\.}         # the first '.' after
        realm=$(echo "$realm" | tr '[:upper:]' '[:lower:]') # change the realm to lower case
        pkinit_kdc_hostname=$(printf "%s%s%s" "$hostname" "." "$realm") #  strcat
    fi

    pkinit_kdc_hostname="pkinit_kdc_hostname = $pkinit_kdc_hostname"

    #echo "  set $pkinit_kdc_hostname"
}

# get pkcs11 lib dir
function getIdDir()
{
    case "$osPlatform" in
        "Redhat/CentOS")
            pkcs11Lib=/usr/lib64/$pklibName
            ;;
        "Ubuntu" | "Debian")
            pkcs11Lib=/usr/lib/x86_64-linux-gnu/$pklibName
            ;;
        *)
            get_str CTXSMARTLOGON_UNSUPPORTED_PLATFORM
            exit 2
            ;;
    esac

    pkinit_identities="pkinit_identities = PKCS11:$pkcs11Lib"
}

# get root CA dir
function getRootCADir()
{
    local rootCAexample="/etc/pki/CA/certs/root.pem"

    get_str CTXSMARTLOGON_SPECIFY_ROOT_CA ${rootCAexample}
    read rootCA

    while [[ ! -f "${rootCA}" ]]; do
        get_str CTXSMARTLOGON_PATH_ERR
        get_str CTXSMARTLOGON_SPECIFY_ROOT_CA ${rootCAexample}
        read rootCA
    done

    pkinit_anchors="pkinit_anchors = FILE:${rootCA}"

    #echo "  set $pkinit_anchors"
}

#**************************************step 5***************************************
function getPKCS11Path()
{
    # set the pkcs11 path
    local pkcs11PathExample="/usr/lib64/pkcs11/opensc-pkcs11.so"
    if [[ "$osPlatform" == "Ubuntu" || "$osPlatform" == "Debian" ]]; then
        pkcs11PathExample="/usr/lib/x86_64-linux-gnu/pkcs11/opensc-pkcs11.so"
    fi
    get_str CTXSMARTLOGON_SPECIFY_PKCS11_PATH ${pkcs11PathExample}
    read pkcs11Path

    #while [ -z "${pkcs11Path}" ]; do
    while [[ ! -f "${pkcs11Path}" ]]; do
        get_str CTXSMARTLOGON_PATH_ERR
        get_str CTXSMARTLOGON_SPECIFY_PKCS11_PATH ${pkcs11PathExample}
        read pkcs11Path
    done
}

function configPAM()
{
    if [ "$smartLogonType" == "SmartCard" ]; then
        getPKCS11Path

        configFSCPAM
        configCSCPAM

        if ([ "$osPlatform" == "Ubuntu" ] || [ "$osPlatform" == "Debian" ]) && [ "$joinDomainWay" == "sssd" ]; then
            configureSssd
        fi
    fi
}

# configure pam conf
function configFSCPAM()
{
    # set pamFile
    pamFile="/etc/pam.d/ctxfsc"

    if [ -f "$pamFile" ]; then
        # backup the conf file
        cp -f "$pamFile" "$backupDir"
    fi

    if [[ "$osPlatform" == "Redhat/CentOS" || "$osPlatform" == "Ubuntu" || "$osPlatform" == "Debian" ]]; then
        confFSCPam
    else 
        get_str CTXSMARTLOGON_UNSUPPORTED_PLATFORM
        exit 2
    fi

    get_str CTXSMARTLOGON_CFG_SUCCESS $pamFile
}

# configure pam conf
function configCSCPAM()
{
    # set pamFile
    pamFile="/etc/pam.d/smartcard-auth"

    if [ -f "$pamFile" ]; then
        # backup the conf file
        cp -f "$pamFile" "$backupDir"
    fi

    if [[ "$osPlatform" == "Redhat/CentOS" || "$osPlatform" == "Ubuntu" || "$osPlatform" == "Debian" ]]; then
        confCSCPam
    else 
        get_str CTXSMARTLOGON_UNSUPPORTED_PLATFORM
        exit 2
    fi

    get_str CTXSMARTLOGON_CFG_SUCCESS $pamFile
}

function confFSCPam()
{
    # set pamFile
    pamFile="/etc/pam.d/ctxfsc"

    # set pkcs11 library
    preauth_options="preauth_options=X509_user_identity=PKCS11:/usr/lib64/libctxpk11.so"
    if [[ "$osPlatform" == "Ubuntu" || "$osPlatform" == "Debian" ]]; then
        preauth_options="preauth_opt=X509_user_identity=PKCS11:/usr/lib/x86_64-linux-gnu/libctxpk11.so"
    fi
    # centrify
    if [ "$joinDomainWay" == "centrify" ]; then
        cat /dev/null > $pamFile
        echo "#%PAM-1.0">>"$pamFile"
        echo "#pam auth">>"$pamFile"
        echo "auth        sufficient     pam_krb5.so $preauth_options">>"$pamFile"
        echo "auth        include        remote">>"$pamFile"
        echo "#pam account">>"$pamFile"
        echo "account     required       pam_krb5.so">>"$pamFile"
        echo "account     required       pam_nologin.so">>"$pamFile"
        echo "#pam password">>"$pamFile"
        echo "password    sufficient     pam_krb5.so">>"$pamFile"
        echo "password    include        remote">>"$pamFile"
        echo "#pam session">>"$pamFile"
        echo "session     optional       pam_krb5.so">>"$pamFile"
        echo "session     include        remote">>"$pamFile"
    else
        cat /dev/null > $pamFile
        echo "#%PAM-1.0">>"$pamFile"
        echo "#pam auth">>"$pamFile"
        echo "auth        sufficient     pam_krb5.so $preauth_options">>"$pamFile"
        echo "auth        include        remote">>"$pamFile"
        echo "#pam account">>"$pamFile"
        echo "account     sufficient     pam_krb5.so">>"$pamFile"
        echo "account     include        remote">>"$pamFile"
        echo "#pam password">>"$pamFile"
        echo "password    sufficient     pam_krb5.so">>"$pamFile"
        echo "password    include        remote">>"$pamFile"
        echo "#pam session">>"$pamFile"
        echo "session     optional       pam_krb5.so">>"$pamFile"
        echo "session     include        remote">>"$pamFile"
    fi
}

function setReconnectPKCS11Path()
{
    /opt/Citrix/VDA/bin/ctxreg update -k "HKLM\System\CurrentControlSet\Control\Citrix\VirtualChannels\Scard" -v "PKCS11LibPath" -d "$pkcs11Path"
}

function confCSCPam()
{
    # set pamFile
    pamFile="/etc/pam.d/smartcard-auth"

    # set pkcs11 library
    # here we set coolkey by default
    preauth_options="preauth_options=X509_user_identity=PKCS11:${pkcs11Path}"
    if [[ "$osPlatform" == "Ubuntu" || "$osPlatform" == "Debian" ]]; then
        preauth_options="preauth_opt=X509_user_identity=PKCS11:${pkcs11Path}"
    fi

    if [ "$joinDomainWay" == "winbind" ]; then
        cat /dev/null > $pamFile
        echo "#%PAM-1.0">>"$pamFile"
        echo "# This file is auto-generated.">>"$pamFile"
        echo "# User changes will be destroyed the next time authconfig is run.">>"$pamFile"
        echo "auth        required      pam_env.so">>"$pamFile"
        echo "auth        [success=done ignore=ignore default=die]  pam_krb5.so $preauth_options">>"$pamFile"
        echo "auth        sufficient    pam_permit.so">>"$pamFile"
        echo "auth        required      pam_deny.so">>"$pamFile"
        echo "">>"$pamFile"
        echo "account     required      pam_unix.so broken_shadow">>"$pamFile"
        echo "account     sufficient    pam_localuser.so">>"$pamFile"
        echo "account     sufficient    pam_succeed_if.so uid < 1000 quiet">>"$pamFile"
        echo "account     [default=bad success=ok user_unknown=ignore] pam_winbind.so">>"$pamFile"
        echo "account     [default=bad success=ok auth_err=ignore user_unknown=ignore ignore=ignore]  pam_krb5.so">>"$pamFile"
        echo "account     required      pam_permit.so">>"$pamFile"
        echo "">>"$pamFile"
        echo "session     optional      pam_keyinit.so revoke">>"$pamFile"
        echo "session     required      pam_limits.so">>"$pamFile"
        echo "-session     optional      pam_systemd.so">>"$pamFile"
        echo "#session     optional      pam_oddjob_mkhomedir.so umask=0077">>"$pamFile"
        echo "session     optional      pam_mkhomedir.so umask=0077">>"$pamFile"
        echo "session     [success=1 default=ignore] pam_succeed_if.so service in crond quiet use_uid">>"$pamFile"
        echo "session     required      pam_unix.so">>"$pamFile"
        echo "session     optional      pam_winbind.so">>"$pamFile"
        echo "session     optional      pam_krb5.so">>"$pamFile"
    elif [ "$joinDomainWay" == "sssd" ]; then
        cat /dev/null > $pamFile
        echo "#%PAM-1.0">>"$pamFile"
        echo "# This file is auto-generated.">>"$pamFile"
        echo "# User changes will be destroyed the next time authconfig is run.">>"$pamFile"
        echo "auth        required      pam_env.so">>"$pamFile"
        echo "auth        [success=done ignore=ignore default=die]  pam_krb5.so $preauth_options">>"$pamFile"
        echo "auth        sufficient    pam_permit.so">>"$pamFile"
        echo "auth        required      pam_deny.so">>"$pamFile"
        echo "">>"$pamFile"
        echo "account     required      pam_unix.so">>"$pamFile"
        echo "account     sufficient    pam_localuser.so">>"$pamFile"
        echo "account     sufficient    pam_succeed_if.so uid < 1000 quiet">>"$pamFile"
        echo "account     [default=bad success=ok user_unknown=ignore] pam_sss.so">>"$pamFile"
        echo "account     [default=bad success=ok auth_err=ignore user_unknown=ignore ignore=ignore]  pam_krb5.so">>"$pamFile"
        echo "account     required      pam_permit.so">>"$pamFile"
        echo "">>"$pamFile"
        echo "session     optional      pam_keyinit.so revoke">>"$pamFile"
        echo "session     required      pam_limits.so">>"$pamFile"
        echo "-session     optional      pam_systemd.so">>"$pamFile"
        echo "#session     optional      pam_oddjob_mkhomedir.so umask=0077">>"$pamFile"
        echo "session     optional      pam_mkhomedir.so umask=0077">>"$pamFile"
        echo "session     [success=1 default=ignore] pam_succeed_if.so service in crond quiet use_uid">>"$pamFile"
        echo "session     required      pam_unix.so">>"$pamFile"
        echo "session     optional      pam_sss.so">>"$pamFile"
        echo "session     optional      pam_krb5.so">>"$pamFile"
    elif [ "$joinDomainWay" == "centrify" ]; then
        cat /dev/null > $pamFile
        echo "#%PAM-1.0">>"$pamFile"
        echo "# This file is auto-generated.">>"$pamFile"
        echo "# User changes will be destroyed the next time authconfig is run.">>"$pamFile"
        echo "auth        required      pam_env.so">>"$pamFile"
        echo "auth        [success=done ignore=ignore default=die]  pam_krb5.so $preauth_options">>"$pamFile"
        echo "auth        sufficient    pam_permit.so">>"$pamFile"
        echo "auth        required      pam_deny.so">>"$pamFile"
        echo "">>"$pamFile"
        echo "account     required      pam_nologin.so">>"$pamFile"
        echo "account     required      pam_krb5.so">>"$pamFile"
        echo "account     required      pam_permit.so">>"$pamFile"
        echo "">>"$pamFile"
        echo "session     optional      pam_keyinit.so revoke">>"$pamFile"
        echo "session     required      pam_limits.so">>"$pamFile"
        echo "-session     optional      pam_systemd.so">>"$pamFile"
        echo "session     optional      pam_mkhomedir.so umask=0077">>"$pamFile"
        echo "session     [success=1 default=ignore] pam_succeed_if.so service in crond quiet use_uid">>"$pamFile"
        echo "session     required      pam_unix.so">>"$pamFile"
        echo "session     optional      pam_krb5.so">>"$pamFile"
    fi

    setReconnectPKCS11Path
}

function configureSssd()
{
    gpo_permit="ad_gpo_map_permit = +ctxscard"
    if [ -f $sssdFile ]; then
        sed -i "/^\s*access_provider\s*=\s*ad/a $gpo_permit" $sssdFile
    else
        get_str CTXSMARTLOGON_FAILED
        get_str CTXSMARTLOGON_NOT_EXIST "${sssdFile}"
        exit 5
    fi

}

#**************************************step 6***************************************
# restore the bakcup files
function backout()
{
    get_str CTXSMARTLOGON_RESTORE_CONFIGURATION

    for file in ${backupDir}/*; do
        targetDir=""
        case "$file" in
            krb5.conf)
                targetDir="/etc/"
                ;;
            ctxfsc|smartcard-auth)
                targetDir="/etc/pam.d/"
                ;;
            sssd.conf)
                targetDir="/etc/sssd/"
                ;;
            smb.conf)
                targetDir="/etc/samba/"
                ;;
            *)
                get_str CTXSMARTLOGON_SHOULD_NOT_RESTORE ${file}
                ;;
        esac

        # cp file
        cp -f "${backupDir}""/""$file" "$targetDir"
        if [[ "$?" -ne "0" ]]; then
            get_str CTXSMARTLOGON_RESTORE_FAIL ${file}
        fi
    done
}

########################### main process block #################################
main()
{
    get_str CTXSMARTLOGON_BEGIN_INFO "${fname}"

    # Check if the user is an root
    checkUser

    # check platform
    checkOS

    # backup the file
    createLogFile

    #1. Check if the user want to enable Smart Card
    get_str CTXSMARTLOGON_STEP_ENABLE_SMARTCARD 1
    enableSmartLogon

    #2. get the ad join way
    get_str CTXSMARTLOGON_STEP_GET_AD_WAY 2
    getADJoinWay

    #3. install dependent packages
    get_str CTXSMARTLOGON_STEP_INSTALL_PACKAGES 3
    installPkgs

    #4. Configure krb5
    get_str CTXSMARTLOGON_STEP_CFG_KRB5 4
    configKrb5

    #5. Configure pam conf
    get_str CTXSMARTLOGON_STEP_CFG_PAM 5
    configPAM

    pcsclibhook configure

    get_str CTXSMARTLOGON_END_INFO "${fname}" "$smartLogonType"
}

main "$@"
