#!/bin/bash

#################################################################################
#
# Citrix Virtual Apps & Desktops For Linux Script: Machine Creation Service
# Copyright (c) Citrix Systems, Inc. All Rights Reserved.
#

source /var/xdl/mcs/mcs_util.sh

fname=$(basename $0)
scriptName=${fname%.*}  # Script name without extension
logFile="/var/log/$scriptName.log"

dbus_service_file_path="/lib/systemd/system/dbus.service"

## These pkgs need to be installed in order for LVDA pkg to work
## ntp and winbind related
common_pkgList=("chrony"\
             "ntpdate"\
             "openjdk-11-jdk"\
             "libgtk2.0-0"\
             "libsasl2-2"\
             "samba"\
             "krb5-user")

common_pkgListNum=${#common_pkgList[@]}


winbind_pkgList=("winbind"\
             "libnss-winbind"\
             "libpam-winbind"\
             "krb5-config"\
             "krb5-locales"\
             "oddjob-mkhomedir")

winbind_pkgListNum=${#winbind_pkgList[@]}

sssd_pkgList=("sssd")

sssd_pkgListNum=${#sssd_pkgList[@]}

pkgMcs=("tdb-tools"\
         "ntfs-3g")
pkgMcsNum=${#pkgMcs[@]}

#this is to prevent interaction during krb5-user installation
export DEBIAN_FRONTEND=noninteractive

function install_lvda_common_dependency_pkgs()
{
    log "Debug: Enter install_lvda_common_dependency_pkgs"
    # Install the common packages
    info="Installing Linux VDA common dependency packages ..."
    log_echo "$info"
    for((i=0;i<common_pkgListNum;i++)); do
        info="Installing package ${common_pkgList[$i]}"
        log_echo "$info"
        apt-get -y install ${common_pkgList[$i]}  >> "$logFile" 2>&1
        if [[ "$?" -ne "0" ]]; then
            info="Error: Failed to install ${common_pkgList[$i]}"
            apt-cache show ${common_pkgList[$i]}  >> "$logFile" 2>&1
            [[ "$?" -ne "0" ]] && log_echo "$info"
        fi
    done

    # start PostgreSQL
    info="Starting PostgreSQL database ..."
    log_echo "$info"

    /bin/systemctl enable postgresql.service  >> "$logFile" 2>&1
    /bin/systemctl start postgresql   >> "$logFile" 2>&1

    sudo -u postgres psql -c 'show data_directory'  >> "$logFile" 2>&1

    log "Debug: Exit install_lvda_common_dependency_pkgs"
}

function install_lvda_winbind_dependency_pkgs()
{
    log "Debug: Enter install_lvda_winbind_dependency_pkgs"
    # Install the common packages
    info="Installing Linux VDA winbind dependency packages ..."
    log_echo "$info"
    for((i=0;i<winbind_pkgListNum;i++)); do
        info="Installing package ${winbind_pkgList[$i]}"
        log_echo "$info"
        apt-get -y install ${winbind_pkgList[$i]}  >> "$logFile" 2>&1
        if [[ "$?" -ne "0" ]]; then
            info="Error: Failed to install ${winbind_pkgList[$i]}"
            apt-cache show ${winbind_pkgList[$i]}  >> "$logFile" 2>&1
            [[ "$?" -ne "0" ]] && log_echo "$info"
        fi
    done

    # enable winbind service
    /bin/systemctl enable winbind.service  >> "$logFile" 2>&1

    log "Debug: Exit install_lvda_winbind_dependency_pkgs"
}


function install_lvda_sssd_dependency_pkgs()
{
    log "Debug: Enter install_lvda_sssd_dependency_pkgs"
    # Install the common packages
    info="Installing Linux VDA sssd dependency packages ..."
    log_echo "$info"
    for((i=0;i<sssd_pkgListNum;i++)); do
        info="Installing package ${sssd_pkgList[$i]}"
        log_echo "$info"
        apt-get -y install ${sssd_pkgList[$i]}  >> "$logFile" 2>&1
        if [[ "$?" -ne "0" ]]; then
            info="Error: Failed to install ${sssd_pkgList[$i]}"
            apt-cache show "${sssd_pkgList[$i]}"  >> "$logFile" 2>&1
            [[ "$?" -ne "0" ]] && log_echo "$info"
        fi
    done

     # enable sssd service
    /bin/systemctl enable sssd.service  >> "$logFile" 2>&1

    log "Debug: Exit install_lvda_sssd_dependency_pkgs"
}

function install_mcs_pkgs()
{
    log "Debug: Enter install_mcs_pkgs"
    for ((i=0;i<pkgMcsNum;i++)); do
        info="Installing package: ${pkgMcs[$i]}"
        log_echo "$info"
        apt-get -y install ${pkgMcs[$i]}  >> "$logFile" 2>&1

        if [[ "$?" -ne "0" ]]; then
            log_echo "Error: install ${pkgMcs[$i]} failed."
        fi
    done

    log "Debug: Exit install_mcs_pkgs"
}

function conf_dns()
{
    log "Debug: Enter funtion conf_dns"
    [[ -z "$dns1" && -z "$dns2" && -z "$dns3" && -z "$dns4" ]] && \
            log_echo "ERROR: Exit funtion conf_dns, dns not configured, please do it manually." && \
            return

    conf_resolv_conf

	override_resolv_by_network_manager

    if [[ -d "/etc/cloud" || -f "/etc/google_instance_id" ]]; then
        # Cloud platforms
        cloudCfg="/etc/cloud/cloud.cfg"
        if [[ -f "$cloudCfg" ]]; then
            `sed -i '/manage_etc_hosts.*$/d' "$cloudCfg"` 
            echo "manage_etc_hosts:false">>"$cloudCfg"
        fi
        override_resolv_by_dhclient
    else
        override_resolv_by_interface
    fi

    stop_DHCP_changing_resolv

    log "Debug: Exit funtion conf_dns"
}

function main()
{
    log "Debug: Enter main"
    check_user
    create_log_file
    install_lvda_common_dependency_pkgs
    install_mcs_pkgs
    install_mcs_scripts
    # For NDJ mode, we need to create local mapped account group
    create_localuser_group

    hack_dbus_service_file $dbus_service_file_path
    if [ "$AD_INTEGRATION" == "winbind" ]; then
        install_lvda_winbind_dependency_pkgs
    elif [ "$AD_INTEGRATION" == "sssd" ]; then
        install_lvda_sssd_dependency_pkgs
    elif [ "$AD_INTEGRATION" == "pbis" ]; then
        install_lvda_pbis_dependency_pkgs
    fi
    conf_dns
    log "Debug: Exit main"
}

main "$@"
