#!/bin/bash
#***********************************************************************
# *	enable_ldaps.sh
# *
# * 	Copyright 1998-2017, 2017 Citrix Systems, Inc. All Rights Reserved.
# * 
# *	Enable ldaps by set the registry flag and configure certificate
# *	
# *	@author joeym , Aug 2016
# *
# *************************************************************************

source /var/xdl/configure_common.sh
source /var/xdl/configure_platform.sh
source /var/xdl/configure_utilities.sh

PATH_BAK=$PATH
export PATH=/usr/lib/jvm/java-8-openjdk-amd64/jre/bin:${PATH}

###Global variables###
ctxreg="/opt/Citrix/VDA/bin/ctxreg"
REGVDA="HKLM\Software\Citrix\VirtualDesktopAgent"
#REGLDAPSKEY="ListOfLDAPServersForPolicy"
REGLDAPSKEY="ListOfLDAPServers"
REGPOLVDA="HKLM\Software\Policies\Citrix\VirtualDesktopAgent"
REGFLAGKEY="UseLDAPS"
REGKEYSTOREKEY="Keystore"
REGCB="EnableChannelBinding"
STOREPATH="/etc/xdl/.keystore"
CERTPATH="/etc/xdl/.ldapcert"
ldapServers=""
ldapServer=
CAcert=
fEnable=true
osPlatform=""
osVersion=""
CTX_LDAPS_KEYSTORE_PASSWORD=""
CTX_LDAPS_LDAP_SERVERS=""

# script usage
function usage()
{
	get_str SHELL_ENABLE_LDAPS_USAGE
: <<'REMOVE'
	echo "Command: enable_ldaps"
	echo "Usage: -Enable [CAcert file]"
	echo "	Enable LDAPS for the LDAP Search Security with CA Certificate provided."
	echo "Usage: -Enablecb [CAcert file in pem format]"
	echo "	Enable LDAPS with Channel Binding support (Lower LDAP Search efficiency but support Channel Binding)."
	echo "Usage: -Disable"
	echo "	Fallback to LDAP without SSL."
	echo "Warning: Only administrator can execute this command!"
	echo "	 This command may impact the service, suggest to run it at "
	echo "   maintenance mode!"
	get_str SHELL_ENABLE_LDAPS_USAGE
REMOVE
}

# check user
function checkUser()
{
	if [ "$(id -u)" != 0 ]; then
		#echo "This script must be run as root."
		get_str SHELL_ENABLE_LDAPS_MUST_ROOT
		exit 1
	fi
}



# import the Certificate to keystore
function setKeystore()
{
	#echo -n "Set Keystore Password: [changeit] : "
	echo -n `get_str SHELL_ENABLE_LDAPS_SET_PASSWD`

	if [[ -z "$CTX_LDAPS_KEYSTORE_PASSWORD" ]]; then
		read -s pass
	else
		pass="$CTX_LDAPS_KEYSTORE_PASSWORD"
	fi
	if [ -z "$pass" ]; then
		pass="changeit"
	fi

	`rm -f $STOREPATH` # clear the keystore first

	echo ""
	#echo "Setting Kestore to $STOREPATH"
	get_str SHELL_ENABLE_LDAPS_SET_KEYSTORE $STOREPATH

	local str="keytool -import -keystore ${STOREPATH} -storetype JKS -storepass ${pass} -noprompt -file ${CAcert}"
	if ! eval "${str}"; then
		#echo "Error: Failed to set keystore."
		get_str SHELL_ENABLE_LDAPS_SET_KEYSTORE_ERROR
		exit 1
	fi
	
}

# copy the Certificate to CERTPATH
function storeCert()
{
	#echo "Store Certificate in $CERTPATH"
	get_str SHELL_ENABLE_LDAPS_STORE_CERT $CERTPATH
	cp -f "$CAcert" $CERTPATH
	chmod 400 $CERTPATH
}

# set LDAP servers
function setLDAPServers()
{

	local tmp=`$ctxreg read -k $REGVDA -v $REGLDAPSKEY`
	if [[ "$?" -ne "0" ]]; then
		#echo "Error: Failed to read ldapServer."
		get_str SHELL_ENABLE_LDAPS_READ_LDAPSERVERS_ERROR
		exit 1
	fi
	
	#echo $tmp
	local port="389"
	if [[ "$fEnable" = true ]]; then
		port="636"
	fi

	for ldaps in $tmp
	do
		ldapServer=`echo $ldaps | sed 's/:.*//'`
		#echo $ldaps

		ldapServers="$ldapServers $ldapServer:$port"
		#echo $ldapServers
	done
	
	#echo -n "Set ldap server with port#: [$ldapServer:636] : "
	echo -n `get_str SHELL_ENABLE_LDAPS_SERVER_IN`
	echo -n ": [$ldapServers ] : "

	if [[ -z "$CTX_LDAPS_LDAP_SERVERS" ]]; then
		read -e ldapServerIn
	else
		ldapServerIn="$CTX_LDAPS_LDAP_SERVERS"
	fi

	if [ -z "$ldapServerIn" ]; then
		ldapServer="$ldapServers"
	else
		ldapServer="$ldapServerIn"
	fi
	#echo "Using Ldap Server $ldapServer"
	echo -n `get_str SHELL_ENABLE_LDAPS_SERVER_OUT`
	echo ": [$ldapServer ] "
}

# set reg. 
function setRegEnable()
{
	#echo "Updating Reg. Settings.."
	get_str SHELL_ENABLE_LDAPS_UPDATE_REG

	local flag=0

	if [[ "$fEnable" = true ]]; then

		`$ctxreg update -k $REGVDA -v $REGKEYSTOREKEY -d $STOREPATH`
		if [[ "$?" -ne "0" ]]; then
			#echo "Error: Failed to update Registry."
			get_str SHELL_ENABLE_LDAPS_UPDATE_REG_ERROR
			exit 1
		fi

		flag=1
	fi 

	if [[ "$cbEnable" = true ]]; then
		`$ctxreg update -k $REGVDA -v $REGCB -d 1`
		if [[ "$?" -ne "0" ]]; then
			#echo "Error: Failed to update Registry."
			get_str SHELL_ENABLE_LDAPS_UPDATE_REG_ERROR
			exit 1
		fi
	else
		`$ctxreg update -k $REGVDA -v $REGCB -d 0`
		if [[ "$?" -ne "0" ]]; then
			#echo "Error: Failed to update Registry."
			get_str SHELL_ENABLE_LDAPS_UPDATE_REG_ERROR
			exit 1
		fi
	fi 

	#echo $ldapServer

	`$ctxreg update -k $REGVDA -v $REGLDAPSKEY -d "$ldapServer"`
	if [[ "$?" -ne "0" ]]; then
		#echo "Error: Failed to update Registry."
		get_str SHELL_ENABLE_LDAPS_UPDATE_REG_ERROR
		exit 1
	fi

	`$ctxreg update -k $REGVDA -v $REGFLAGKEY -d $flag`
	if [[ "$?" -ne "0" ]]; then
		#echo "Error: Failed to update Registry."
		get_str SHELL_ENABLE_LDAPS_UPDATE_REG_ERROR
		exit 1
	fi
}


# restart ctxvda service
function restartVDA()
{
	#echo "Restarting Desktop Service.."
	get_str SHELL_ENABLE_LDAPS_RESTART_VDA

	#`systemctl restart ctxvda`
	service_stop
	service_start
}

function python3LdapsConfigure()
{
	if [[ ! -f "$PYTHON3_PATH/bin/python3" ]]; then
		echo "Configure Python3 Virtual environment..."
		python3_configure_venv
	fi
	if [[ -f "$PYTHON3_PATH/bin/python3" ]]; then
		echo "Install Python3 packages..."
		python3_install_ctxldaps
	fi
}

######################### main process block #################################
main()
{
	# Check if the user is an root
	checkUser

	# Check the parameter
	if [[ $# = 0 ]]; then
		usage
		exit 1
	fi

	# Enable LDAS
	if [[ "$1" = "-Enable" && -f "$2" ]];then
		echo "=============="

		CAcert="$2"
		fEnable=true
		cbEnable=false

		#echo "Enabling LDAP using SSL.."
		get_str SHELL_ENABLE_LDAPS
		
		setLDAPServers

		setKeystore
		
		setRegEnable	

	# Enable LDAS with Channel Binding
	elif [[ "$1" = "-Enablecb" && -f "$2" ]];then
		echo "=============="

		CAcert="$2"
		fEnable=true
		cbEnable=true

		#echo "Enabling LDAP using SSL.."
		get_str SHELL_ENABLE_LDAPS

		python3LdapsConfigure

		setLDAPServers

		setKeystore
		
		storeCert

		setRegEnable	

	# Disable LDAPS
	elif [[ "$1" = "-Disable" ]];then
		echo "=============="
		#echo "Fallback to LDAP without SSL.."
		get_str SHELL_DISABLE_LDAPS
		
		fEnable=false

		setLDAPServers

		setRegEnable
	else
		usage
		exit 1
	fi

	#restart service
	restartVDA

	echo "==============="
}


main "$@"

export PATH=${PATH_BAK}
