#!/bin/bash
set +x
##
#
# enable/disable SSL for linux VDA.
#
##
source /var/xdl/configure_common.sh
source /var/xdl/configure_platform.sh
source /var/xdl/configure_utilities.sh

######################Global variables######################
Disable=false 
Enable=false 
SSLPort=443                  # Default SSL port 
SSLMinVersion="TLS_1.2"        # ValidateSet("TLS_1.0", "TLS_1.1", "TLS_1.2", "TLS_1.3")
SSLCipherSuite="ALL"           # ValidateSet("GOV", "COM", "ALL")
Certificate=""               # Certificate Name
KeyStore="/etc/xdl/.sslkeystore" # Certificate keystore
SubKeyStores=("cacerts" "intcerts" "crls" "certs") #sub-directories under keystore
CertificateFile=""           # Certificate file, specified by user.
CertificateFileNamePfx=""    # Certificate file name, inferred from $CertificateFile
CertificateFileNamePem=""    # Certificate file name, inferred from $CertificateFile
CertificateFilePath=""       # Certificate file path, inferred from $CertificateFile
RootCertificateFile=""       # Root Certificate file, specified by user.
IsPfx=1                      # Indicate the type of Certificate file(inputted by user)
                             # 1: pfx, 2: pem, other:  error                           
IsQuiet=0                    # 1: quiet model, needs user interaction. 
                             # 0: non-quiet model. 
#Parameters=('SSLPort' 'SSLMinVersion' 'SSLCipherSuite' 'Certificate' 'KeyStore')
OsType=""                    # OS Type: redhat, centos, suse, ubuntu, amzn
OsVersion=""                 # OS Version: 2, 6, 7, 11, 12, 16
args=("$@")

# declare constants

# Registry path
ICA_LISTENER_PATH="HKLM\\System\\CurrentControlSet\\Control\\Citrix\\WinStations\\tcp"
SSL_LISTENER_PATH="HKLM\\System\\CurrentControlSet\\Control\\Citrix\\WinStations\\ssl"
ICA_LISTENER_PORT_KEY_LOCAL="PortNumber"
ICA_INPUT_BUFFER_LENGTH="InputBufferLength"
ICA_INPUT_BUFFER_LENGTH_VALUE="0x00000800" # Default value, without SSL enable. 
SSL_INPUT_BUFFER_LENGTH_VALUE="0x00001000" # New value, with SSL enable.
ENABLE_SSL_KEY="fEnableWinStation"         #tag SSLEnabled, keep align with ICA flag.
SSL_KEYSTORE_KEY="SSLKeyStore"
SSL_CERT_KEY="SSLCertName"
SSL_PORT_KEY="PortNumber" # tag SSLPort, generalize the port number for winstations
SSL_MINVERSION_KEY="SSLMinVersion"
SSL_CIPHERSUITE_KEY="SSLCipherSuite"

POLICIES_PATH="HKLM\\Software\\Policies\\Citrix\\ICAPolicies"
ICA_LISTENER_PORT_KEY="IcaListenerPortNumber"
SESSION_RELIABILITY_PORT_KEY='SessionReliabilityPort'
WEBSOCKET_PORT_KEY='WebSocketPort'

######################system commands######################
OPENSSL="/usr/bin/openssl"
MKDIR="/usr/bin/mkdir"
MKDIR_Ubuntu="/bin/mkdir"
FIND="usr/bin/find"
SED="/usr/bin/sed"
SED_Ubuntu="/bin/sed"
CP="/usr/bin/cp"
CP_Ubuntu="/bin/cp"
RM="/usr/bin/rm"
RM_Ubuntu="/bin/rm"
CHMOD="/usr/bin/chmod"
CHMOD_Ubuntu="/bin/chmod"

#VDA commands
CTXREG="/opt/Citrix/VDA/bin/ctxreg"
######################Functions definition######################

# print script usage
function usage()
{ 
    get_str SHELL_ENABLE_VDASSL_USAGE0
    get_str SHELL_ENABLE_VDASSL_USAGE1
    get_str SHELL_ENABLE_VDASSL_USAGE2
    get_str SHELL_ENABLE_VDASSL_USAGE3
    get_str SHELL_ENABLE_VDASSL_USAGE4
}

# recognize arguments
function argsMatch()
{
    case "${args[0]}" in
         -Disable)
          Disable=true
         
          if [[ ${#args[@]} == 3 ]]; then
              if [[ "${args[1]}" = "-Quiet" || "${args[1]}" = "-quiet" || "${args[1]}" = "-Q" || "${args[1]}" = "-q" ]]; then
                  IsQuiet=${args[2]}
              else
                  get_str SHELL_ENABLE_VDASSL_INVALID_OPT "${args[1]}"
                  usage
                  exit 1
              fi
          elif [[ ${#args[@]} != 1 ]]; then
             get_str SHELL_ENABLE_VDASSL_INVALID_OPT "${args[1]}"
             usage
             exit 1
          fi
          get_str SHELL_DISABLE_VDASSL
          ;;
         -Enable)
          Enable=true

          i=1
          while(( $i<${#args[@]} )); do
              case "${args[$i]}" in
                   -Certificate)
                    CertificateFile=${args[$i+1]}
                    ;;
                   -RootCertificate)
                    RootCertificateFile=${args[$i+1]}
                    ;;
                   -SSLCipherSuite)
                    SSLCipherSuite=${args[$i+1]}
                    ;;
                   -SSLMinVersion)
                    SSLMinVersion=${args[$i+1]}
                    ;;
                   -SSLPort)
                    SSLPort=${args[$i+1]}
                    ;;
                   -Quiet|-quiet|-Q|-q)
                    IsQuiet=${args[$i+1]}
                    ;;
                   *)
                    get_str SHELL_ENABLE_VDASSL_INVALID_OPT "${args[$i]}"
                    usage
                    exit 1
                    ;;
              esac
              i=`expr $i + 2` 
        done
        get_str SHELL_ENABLE_VDASSL
        ;;
       -help|--help|-HELP|?)
        usage
        exit 1
        ;;
       *)
        get_str SHELL_ENABLE_VDASSL_INVALID_OPT "${args[0]}"
        usage
        exit 1
        ;;
    esac
}

#
#Validate arguments
#Following validation will be done:
# 1. If Certificate file was specified, the file should be existing, and the format 
#    should be PKCS#12 or PEM(with priate key)
# 2. If Certificate file was not specified, $KeyStore and $SubKeyStores should be 
#    existing and at least one file suffixed with pem is under $KeyStore/$SubKeyStores[3]
#
function validateArgs()
{
    #Certificate file was specified
    if [ -n  "$CertificateFile" ]; then
        if [ ! -f "$CertificateFile" ]; then
            get_str SHELL_ENABLE_VDASSL_FILE_NONEXIST_OPT "$CertificateFile"
            exit 1
        fi
        CertificateFilePath=`dirname "$CertificateFile"`
        TmpFileName=`basename "$CertificateFile"`
        Extension="${TmpFileName##*.}"

        #Convert the extention to be lower case
        NewExtension=`echo $Extension | tr '[A-Z' '[a-z]'`
        if [ "$NewExtension" = "pfx" ]; then
            CertificateFileNamePfx=$TmpFileName
            CertificateFileNamePem=${TmpFileName%.*}".pem"
            IsPfx=1
            get_str SHELL_ENABLE_VDASSL_UNSUPPORTED_CERTIFICATE_FILE_OPT "$NewExtension"
            exit 1
        elif [ "$NewExtension" = "pem" ]; then
            CertificateFileNamePem=$TmpFileName
            IsPfx=2
        else
            get_str SHELL_ENABLE_VDASSL_UNSUPPORTED_CERTIFICATE_FILE_OPT "$NewExtension" 
            exit 1
        fi

    else   #Certificate file was not specified
        if [[ ! -d "$KeyStore"  ||
              ! -d "$KeyStore"/"${SubKeyStores[0]}" ||
              ! -d "$KeyStore"/"${SubKeyStores[1]}" ||
              ! -d "$KeyStore"/"${SubKeyStores[2]}" ||
              ! -d "$KeyStore"/"${SubKeyStores[3]}" ]]; then
            get_str SHELL_ENABLE_VDASSL_KEYSTORE_DAMAGED_OPT "$KeyStore"
            exit 1
        fi

        fileCount=`find "$KeyStore"/"${SubKeyStores[3]}" -type f -name '*.pem' | wc -l `
        if [ "$fileCount" -lt "0" ]; then
            get_str SHELL_ENABLE_VDASSL_NO_CERTIFICATE_OPT "$KeyStore/${SubKeyStores[3]}"
            exit 1
        fi
    fi

    # Validate RootCertificateFile
    if [ -n  "$RootCertificateFile" ]; then
        if [ ! -f "$RootCertificateFile" ]; then
            get_str SHELL_ENABLE_VDASSL_FILE_NONEXIST_OPT "$RootCertificateFile"
            exit 1
        fi
    fi

    if [ "$IsQuiet" -ne "0" ]; then
        IsQuiet=1
    fi

}

#
# Get the OS type
#
function getOsType()
{
    coreFileRhel="/etc/redhat-release"  
    coreFileCent="/etc/centos-release"   
    coreFileRocky="/etc/rocky-release" 
    coreFileSuse="/etc/SuSE-release"  
    coreFileUbuntu="/etc/lsb-release"
    coreFileOS="/etc/os-release"

    if [ -f "/etc/rocky-release" ] ; then
        OsType="rocky"
        OsVersion="$(cat $coreFileRocky |cut -d ' ' -f4 |cut -d '.' -f1  2>&1)"
    elif [ -f /etc/centos-release ] ; then
        OsType="centos"
        # the contants of $coreFileCent may be 
        # "CentOS release 6.8 (Final)"
        # or "CentOS Linux release 7.2.1511 (Core)"
        # we need to adjust the field  
        num="$(cat $coreFileCent |wc -w 2>&1)"
        num=`expr $num - 1` 
        OsVersion="$(cat $coreFileCent |cut -d ' ' -f$num |cut -d '.' -f1 2>&1)"
      
    elif [ -f /etc/redhat-release ] ; then        
        OsType="redhat"
        OsVersion="$(cat $coreFileRhel |cut -d ' ' -f7 |cut -d '.' -f1 2>&1)"
        if [[ $OsVersion != 7 ]]; then
           OsVersion="$(cat $coreFileRhel |cut -d ' ' -f6 |cut -d '.' -f1  2>&1)"
        fi

    elif [ -f /etc/SuSE-release ] ; then
        OsType="suse" 
        OsVersion="$(cat "$coreFileSuse" |grep "SUSE Linux Enterprise" |cut -d" " -f5|tr A-Z a-z  2>&1)"   
    elif [ -f /etc/debian_version ] ; then
        if [ -f "$coreFileUbuntu" ]; then
            OsType="ubuntu"
            OsVersion="$(cat $coreFileUbuntu |grep DISTRIB_RELEASE |cut -d '=' -f2 |tr A-Z a-z  2>&1)"
        else
            OsType="debian"
            OsVersion="$(cat $coreFileOS |grep VERSION_ID |cut -d '=' -f2 |tr A-Z a-z |sed 's/\"//g' 2>&1)" 
        fi
    else
        if [ -f $coreFileOS ] ; then
            local os_platform=""
            os_platform="$(< $coreFileOS grep ^ID= |cut -d '=' -f2 |tr '[:upper:]' '[:lower:]' |tr -d '"'  2>&1)"
            if [[ $os_platform == "sles" || $os_platform == "sled" ]]; then
                OsType="suse"
                OsVersion="$(< $coreFileOS grep VERSION_ID |cut -d '=' -f2 |tr '[:upper:]' '[:lower:]' |sed 's/\"//g' 2>&1)"
            elif [[ $os_platform == "amzn" ]]; then
                OsType="amzn"
                OsVersion="$(< $coreFileOS grep VERSION_ID |cut -d '=' -f2 |tr '[:upper:]' '[:lower:]' |sed 's/\"//g' 2>&1)"
            else
                get_str SHELL_ENABLE_VDASSL_CANNOT_SUPPORT_OS
                exit 1
            fi
        else
            get_str SHELL_ENABLE_VDASSL_CANNOT_SUPPORT_OS
            exit 1
        fi
    fi
}

#
# Create Keystore
# Create the following directories:
# 1. $KeyStore
# 2. $KeyStore/$SubKeyStores[0]
# 3. $KeyStore/$SubKeyStores[1]
# 4. $KeyStore/$SubKeyStores[2]
# 5. $KeyStore/$SubKeyStores[3]
#
function createKeystore()
{
    if [ ! -d "$KeyStore" ]; then
        mkdir "$KeyStore"  
        if [ "$?" -ne "0" ]; then
            get_str SHELL_ENABLE_VDASSL_CREATE_DIR_FAIL "$KeyStore"
            exit 1
        fi

        #Change the permission of dir and make it readable/writeable for root only
        chmod 500 "$KeyStore"
        if [ "$?" -ne "0" ]; then
           get_str SHELL_ENABLE_VDASSL_MOD_PERMISS_DIR_FAIL "$KeyStore"
           exit 1
       fi
    fi
    num=${#SubKeyStores[@]}
    for((i=0;i<num;i++)); do
        if [ ! -d "$KeyStore"/"${SubKeyStores[$i]}" ]; then
            mkdir "$KeyStore"/"${SubKeyStores[$i]}"  
            if [ "$?" -ne "0" ]; then
                get_str SHELL_ENABLE_VDASSL_CREATE_DIR_FAIL "$KeyStore/${SubKeyStores[$i]}"
                exit 1
            fi
 
            #Change the permission of dir and make it readable/writeable for root only
            chmod 500 "$KeyStore"/"${SubKeyStores[$i]}"
            if [ "$?" -ne "0" ]; then
                get_str SHELL_ENABLE_VDASSL_MOD_PERMISS_DIR_FAIL "${SubKeyStores[$i]}"
                exit 1
            fi
        fi
    done 
}

#
#Convert Certificate file from pfx to pem format
#
function converCertFormat()
{
    # Conver the certificate file to pem
    if [[ -f "$CertificateFile" ]]; then
        openssl pkcs12 -in $CertificateFile -nodes -out  $CertificateFilePath/$CertificateFileNamePem
        if [ "$?" -ne "0" ]; then
            get_str SHELL_ENABLE_VDASSL_CONVERT_FORMAT_FAIL "$CertificateFilePath/$CertificateFileNamePfx" 
            exit 1
        fi
    fi
}

#
#Remove password from the pem file and generate a new pem file 
#
function removePassword()
{
    tmpFile="$CertificateFilePath"/tmp.pem
     
    if [ -f "$CertificateFilePath/$CertificateFileNamePem" ]; then
        get_str SHELL_ENABLE_VDASSL_REMOVE_PASSPHRASE_INFO
        openssl rsa -in $CertificateFilePath/$CertificateFileNamePem -out $tmpFile
        if [ "$?" -ne "0" ]; then
            get_str SHELL_ENABLE_VDASSL_WRITE_RSA_FAIL "$tmpFile"
            exit 1
        fi

        beginLine="-----BEGIN PRIVATE KEY-----"
        endLine="-----END PRIVATE KEY-----"
        sed "/${beginLine}/,/${endLine}/d" $CertificateFilePath/$CertificateFileNamePem>>$tmpFile
        if [ "$?" -ne "0" ]; then
            get_str SHELL_ENABLE_VDASSL_REMOVE_RST_FAIL "$CertificateFilePath/$CertificateFileNamePem"
            exit 1
        fi
   
        cp  $tmpFile $CertificateFilePath/$CertificateFileNamePem
        if [ "$?" -ne "0" ]; then
           get_str SHELL_ENABLE_VDASSL_MOVE_RST_FAIL "$tmpFile" "$CertificateFilePath/$CertificateFileNamePem"
           exit 1
        fi
   
        rm -f $tmpFile
    fi
}

#
#Put the Certificate file into keystore
#
function putCertToKeystore()
{
    if [[ "$OsType" = "redhat" || "$OsType" = "centos" || "$OsType" = "amzn" || "$OsType" = "rocky" ]]; then
        caDir="/etc/ssl/certs"
        caName="ca-bundle.crt"
        caFile="$caDir/$caName"
    elif [[ "$OsType" = "ubuntu" || "$OsType" = "debian" ]]; then        
        caDir="/etc/ssl/certs"
        caName="ca-certificates.crt"
        caFile="$caDir/$caName"
    else  #SUSE      
        caDir="/etc/ssl"
        caName="ca-bundle.pem"
        caFile="$caDir/$caName"
    fi
    
    if [[ -f "$caFile" && -f $KeyStore/${SubKeyStores[0]}/"$caName" ]]; then
        # Overwrite file
        Title="$(get_str SHELL_ENABLE_VDASSL_OVERWRITE_FILE__TITLE)" 
        # Overwrite $KeyStore/${SubKeyStores[0]}/ca-bundle.crt
        Content="$(get_str SHELL_EANBLE_VDASSL_OVERWRITE_FILE_WARN $KeyStore/${SubKeyStores[0]}/$caName)" 
        shouldProcess #?  
    
        if [[ "$Config" = "true" ]]; then
            cp "$caFile" $KeyStore/${SubKeyStores[0]}
            if [[ "$?" -ne "0" ]]; then
                get_str SHELL_EANBLE_VDASSL_COPY_FILE_FAIL "$caFile" "$KeyStore/${SubKeyStores[0]}"
                exit 1
            fi
        fi
    elif [ -f "$caFile" ]; then
        cp "$caFile" $KeyStore/${SubKeyStores[0]} 
    fi

    if [[ -f $CertificateFilePath/$CertificateFileNamePem && 
          -f $KeyStore/${SubKeyStores[3]}/$CertificateFileNamePem ]]; then
        # Overwrite file
        Title="$(get_str SHELL_ENABLE_VDASSL_OVERWRITE_FILE__TITLE)"
        # Overwrite  $KeyStore/${SubKeyStores[3]}/$CertificateFileNamePem 
        Content="$(get_str SHELL_EANBLE_VDASSL_OVERWRITE_FILE_WARN $KeyStore/${SubKeyStores[3]}/$CertificateFileNamePem)"
        shouldProcess #?

        if [[ "$Config" = true ]]; then
            cp $CertificateFilePath/$CertificateFileNamePem $KeyStore/${SubKeyStores[3]}/$CertificateFileNamePem
            if [[ "$?" -ne "0" ]]; then
                get_str SHELL_EANBLE_VDASSL_COPY_FILE_FAIL "$CertificateFilePath/$CertificateFileNamePem" "$KeyStore/${SubKeyStores[3]}/$CertificateFileNamePem"
                exit 1
            fi
        fi

    elif [ -f $CertificateFilePath/$CertificateFileNamePem ]; then
        cp $CertificateFilePath/$CertificateFileNamePem $KeyStore/${SubKeyStores[3]}/$CertificateFileNamePem
    fi 

    #Put root certificate into keystore
    if [ -f  "$RootCertificateFile" ]; then
        TmpFileName=`basename "$RootCertificateFile"`
        if [ -f "$KeyStore/${SubKeyStores[0]}/$TmpFileName" ]; then
             # read answer
             # Overwrite file
             Title="$(get_str SHELL_ENABLE_VDASSL_OVERWRITE_FILE__TITLE)"
             # Overwrite  $KeyStore/${SubKeyStores[3]}/$CertificateFileNamePem
             Content="$(get_str SHELL_EANBLE_VDASSL_OVERWRITE_FILE_WARN $KeyStore/${SubKeyStores[0]}/$TmpFileName)"
             shouldProcess #?
             if [[ "$Config" = false ]]; then
                 return   
             fi
        fi
        cp $RootCertificateFile $KeyStore/${SubKeyStores[0]} 
        if [[ "$?" -ne "0" ]]; then
            get_str SHELL_EANBLE_VDASSL_COPY_FILE_FAIL "$RootCertificateFile" "$KeyStore/${SubKeyStores[0]}/$TmpFileName"
            exit 1
        fi      
    fi  
}

#
# Check user: only root user has the permission to execute this script
#
function checkUser()
{
    if [ "$(id -u)" != 0 ]; then
        get_str SHELL_ENABLE_VDASSL_MUST_ROOT
        exit 1
    fi
}

#
# get data by reg key
#
function getData()
{
    ret="$($CTXREG read -k $Key -v $Value 2>&1)" 
    if [[ "$ret" = *"Error"* ]]; then
        get_str SHELL_ENABLE_VDASSL_READ_REG_FAIL "$Key" "$Value"
    fi
}

#
# set data by reg key
#
function setData()
{
    ret="$($CTXREG update -k $Key -v $Value -d $Data 2>&1)"
    if [[ "$ret" = *"Error"* || "$ret" = *"not found in key"* ]]; then
        get_str SHELL_ENABLE_VDASSL_SET_REG_FAIL "$Key" "$Value"
        exit 1
    fi
}

# should process switch
Title=""
Content=""
Confirm="$(get_str SHELL_ENABLE_VDASSL_CONFIRM)"
Config=false
function shouldProcess()
{
     if [ "$IsQuiet" -eq "1" ]; then
        Config="true"
        return
     fi
     echo ""
     echo $Title
     echo $Content
     echo -n "$Confirm, (Y|Yes, N|No) [Y] "

     read ask
     case "$ask" in
         Y|y|Yes|yes)
             Config="true"
             ;;
         "")
             Config="true"  
             ;;
         *)
             Config="false"
             ;;
    esac
}

#
# delete firewall rules for SSL
# enable firewall rules for ICA, CGP and HTML5 ports
#
function firewall_rules_disablessl()
{
    # Delete any existing rules for Citrix SSL Service 
    # Enable any existing rules for ICA, CGP and HTML5 ports
    if [[ "$OsType" = "redhat" || "$OsType" = "centos" || "$OsType" = "amzn" || "$OsType" = "rocky" ]]; then
        if [[ "$OsVersion" = "6" ]]; then
             chmod +w /etc/xdl/firewall/ctxhdx 
             echo "-I INPUT 1 -p tcp -m tcp --dport 1494 -j ACCEPT" > /etc/xdl/firewall/ctxhdx 
             echo "-I INPUT 1 -p tcp -m tcp --dport 2598 -j ACCEPT" >> /etc/xdl/firewall/ctxhdx
             echo "-I INPUT 1 -p tcp -m tcp --dport 8008 -j ACCEPT" >> /etc/xdl/firewall/ctxhdx
             echo "-I INPUT 1 -p udp -m udp --dport 1494 -j ACCEPT" >> /etc/xdl/firewall/ctxhdx 
             echo "-I INPUT 1 -p udp -m udp --dport 2598 -j ACCEPT" >> /etc/xdl/firewall/ctxhdx

             if [ -f /etc/sysconfig/system-config-firewall ]; then
                 /usr/sbin/lokkit --custom-rules=ipv4:filter:/etc/xdl/firewall/ctxhdx 
                 /usr/sbin/lokkit --custom-rules=ipv6:filter:/etc/xdl/firewall/ctxhdx 
             else
                 /usr/sbin/lokkit --disabled --custom-rules=ipv4:filter:/etc/xdl/firewall/ctxhdx 
                 /usr/sbin/lokkit --disabled --custom-rules=ipv6:filter:/etc/xdl/firewall/ctxhdx 
             fi
             /usr/sbin/lokkit --update 
        elif [[ "$OsVersion" = "7" || "$OsVersion" = "8" || "$OsVersion" = "2" ]]; then 
            result="$(firewall-cmd --state|grep  not)"
            if [[ -z "$result" ]]; then            
                firewall-cmd --permanent --zone=public --remove-port=$SSLPort/tcp
                firewall-cmd --permanent --zone=public --remove-port=$SSLPort/udp
                firewall-cmd --permanent --zone=public --add-port=$IcaPort/tcp 
                firewall-cmd --permanent --zone=public --add-port=2598/tcp 
                firewall-cmd --permanent --zone=public --add-port=8008/tcp
                firewall-cmd --permanent --zone=public --add-port=$IcaPort/udp
                firewall-cmd --permanent --zone=public --add-port=2598/udp
                firewall-cmd --permanent --zone=public --add-port=6001-6099/tcp
                firewall-cmd --reload 
            else
                get_str SHELL_ENABLE_VDASSL_CFG_FIREWALL_DISABLE_FAIL
            fi    
        else
            get_str SHELL_ENABLE_VDASSL_CFG_FIREWALL_DISABLE_FAIL
        fi
    elif [[ "$OsType" = "suse" ]]; then
        local major_verion=${OsVersion%\.*}
        if [[ "$major_verion" == "15" ]]; then
            result="$(firewall-cmd --state|grep  not)"
            if [[ -z "$result" ]]; then
                firewall-cmd --permanent --zone=public --remove-port="$SSLPort"/tcp
                firewall-cmd --permanent --zone=public --remove-port="$SSLPort"/udp
                firewall-cmd --permanent --zone=public --add-port="$IcaPort"/tcp
                firewall-cmd --permanent --zone=public --add-port=2598/tcp
                firewall-cmd --permanent --zone=public --add-port=8008/tcp
                firewall-cmd --permanent --zone=public --add-port="$IcaPort"/udp
                firewall-cmd --permanent --zone=public --add-port=2598/udp
                firewall-cmd --permanent --zone=public --add-port=6001-6099/tcp
                firewall-cmd --reload
            else
                get_str SHELL_ENABLE_VDASSL_CFG_FIREWALL_DISABLE_FAIL
            fi
        else
            chmod +w /etc/sysconfig/SuSEfirewall2.d/services/ctxhdx 
            echo "TCP=\"1494 2598 8008\"" > /etc/sysconfig/SuSEfirewall2.d/services/ctxhdx 
            echo "UDP=\"1494 2598\"" >> /etc/sysconfig/SuSEfirewall2.d/services/ctxhdx
            /sbin/yast2 firewall services add zone=EXT service=service:ctxhdx 
            if [ `/sbin/SuSEfirewall2 status | wc -l` -gt "1" ]; then
                /sbin/SuSEfirewall2 stop 
                /sbin/SuSEfirewall2 start 
            fi
        fi
    elif [[ "$OsType" = "ubuntu" || "$OsType" = "debian" ]]; then 
        /usr/sbin/ufw allow proto tcp from any to any port $IcaPort 
        /usr/sbin/ufw allow proto tcp from any to any port 2598
        /usr/sbin/ufw allow proto tcp from any to any port 8008
        /usr/sbin/ufw allow proto udp from any to any port $IcaPort
        /usr/sbin/ufw allow proto udp from any to any port 2598
        /usr/sbin/ufw allow proto tcp from any to any port 6001:6099
        /usr/sbin/ufw delete allow proto tcp from any to any port $SSLPort
        /usr/sbin/ufw delete allow proto udp from any to any port $SSLPort
        /usr/sbin/ufw reload 
    fi
}

#
# create firewall rules for SSL
# disable firewall rules for ICA, CGP and HTML5 ports
#
function firewall_rules_enablessl()
{
    # Delete any existing rules for the SSLPort  
    # Disable any existing rules for ICA, CGP and HTML5 ports
    if [[ "$OsType" = "redhat" || "$OsType" = "centos" || "$OsType" = "amzn" || "$OsType" = "rocky" ]]; then
        if [[ "$OsVersion" = "6" ]]; then
             chmod +w /etc/xdl/firewall/ctxhdx 
             echo "-I INPUT 1 -p tcp -m tcp --dport $SSLPort -j ACCEPT" > /etc/xdl/firewall/ctxhdx
             echo "-I INPUT 1 -p udp -m udp --dport $SSLPort -j ACCEPT" >> /etc/xdl/firewall/ctxhdx 
             if [ -f /etc/sysconfig/system-config-firewall ]; then
                 /usr/sbin/lokkit --custom-rules=ipv4:filter:/etc/xdl/firewall/ctxhdx 
                 /usr/sbin/lokkit --custom-rules=ipv6:filter:/etc/xdl/firewall/ctxhdx 
             else
                 /usr/sbin/lokkit --disabled --custom-rules=ipv4:filter:/etc/xdl/firewall/ctxhdx 
                 /usr/sbin/lokkit --disabled --custom-rules=ipv6:filter:/etc/xdl/firewall/ctxhdx 
             fi
             /usr/sbin/lokkit --update 
        elif [[ "$OsVersion" = "7" || "$OsVersion" = "8" || "$OsVersion" = "2" ]]; then
             result="$(firewall-cmd --state|grep  not)"
             if [[ -z "$result" ]]; then
                 firewall-cmd --permanent --zone=public --add-port=$SSLPort/tcp
                 firewall-cmd --permanent --zone=public --add-port=$SSLPort/udp
                 firewall-cmd --permanent --zone=public --remove-port=$IcaPort/tcp
                 firewall-cmd --permanent --zone=public --remove-port=2598/tcp
                 firewall-cmd --permanent --zone=public --remove-port=8008/tcp
                 firewall-cmd --permanent --zone=public --remove-port=$IcaPort/udp
                 firewall-cmd --permanent --zone=public --remove-port=2598/udp
                 firewall-cmd --permanent --zone=public --add-port=6001-6099/tcp
                 firewall-cmd --reload 
             else
                 get_str SHELL_ENABLE_VDASSL_CFG_FIREWALL_ENABLE_FAIL
             fi
        else
             get_str SHELL_ENABLE_VDASSL_CFG_FIREWALL_ENABLE_FAIL
        fi
    elif [[ "$OsType" = "suse" ]]; then
        local major_verion=${OsVersion%\.*}
        if [[ "$major_verion" == "15" ]]; then
            result="$(firewall-cmd --state|grep  not)"
            if [[ -z "$result" ]]; then
                firewall-cmd --permanent --zone=public --add-port="$SSLPort"/tcp
                firewall-cmd --permanent --zone=public --add-port="$SSLPort"/udp
                firewall-cmd --permanent --zone=public --remove-port="$IcaPort"/tcp
                firewall-cmd --permanent --zone=public --remove-port=2598/tcp
                firewall-cmd --permanent --zone=public --remove-port=8008/tcp
                firewall-cmd --permanent --zone=public --remove-port="$IcaPort"/udp
                firewall-cmd --permanent --zone=public --remove-port=2598/udp
                firewall-cmd --permanent --zone=public --add-port=6001-6099/tcp
                firewall-cmd --reload
            else
                get_str SHELL_ENABLE_VDASSL_CFG_FIREWALL_ENABLE_FAIL
            fi
        else
            chmod +w /etc/sysconfig/SuSEfirewall2.d/services/ctxhdx 
            echo "TCP=\"$SSLPort\"" > /etc/sysconfig/SuSEfirewall2.d/services/ctxhdx 
            echo "UDP=\"$SSLPort\"" >> /etc/sysconfig/SuSEfirewall2.d/services/ctxhdx
            /sbin/yast2 firewall services add zone=EXT service=service:ctxhdx 
            if [ `/sbin/SuSEfirewall2 status | wc -l` -gt "1" ]; then
                /sbin/SuSEfirewall2 stop 
                /sbin/SuSEfirewall2 start 
            fi
        fi
    elif [[ "$OsType" = "ubuntu" || "$OsType" = "debian" ]]; then
        /usr/sbin/ufw allow proto tcp from any to any port $SSLPort 
        /usr/sbin/ufw allow proto udp from any to any port $SSLPort
        /usr/sbin/ufw allow proto tcp from any to any port 6001:6099
        /usr/sbin/ufw delete allow proto tcp from any to any port $IcaPort 
        /usr/sbin/ufw delete allow proto tcp from any to any port 2598
        /usr/sbin/ufw delete allow proto tcp from any to any port 8008
        /usr/sbin/ufw delete allow proto udp from any to any port $IcaPort
        /usr/sbin/ufw delete allow proto udp from any to any port 2598
        /usr/sbin/ufw reload 
    fi
}

# retrive ssl certificate
function getCert()
{
    local let fileno=0
    local found=false
    local file=""
    for cert in $KeyStore/certs/*; do
        ((fileno+=1))
        file=$cert

        if [[ -f $cert && $(basename "$cert") =  $Certificate ]]; then
            get_str SHELL_ENABLE_VDASSL_GET_CERT "$Certificate"
            found=true
            break
        fi
    done
  
    if [[ $fileno = 0 ]]; then
        get_str SHELL_ENABLE_VDASSL_NO_CERT
        get_str SHELL_ENABLE_VDASSL_FAIL
        exit 1
    elif [[ $fileno = 1 && $found = false ]]; then
        if [[ $Certificate != "" ]]; then
            get_str SHELL_ENABLE_VDASSL_NO_VALID_CERT "$Certificate"
            get_str SHELL_ENABLE_VDASSL_FAIL
            exit 1
        else
            Certificate=$file
        fi
    elif [[ $fileno > 1 && $found = false ]]; then
        if [[ $Certificate = "" ]]; then
            get_str SHELL_ENABLE_VDASSL_MORE_CERT
            get_str SHELL_ENABLE_VDASSL_FAIL
            exit 1  
        else
            get_str SHELL_ENABLE_VDASSL_NO_VALID_CERT "$Certificate"
            get_str SHELL_ENABLE_VDASSL_FAIL
            exit 1
        fi  
    else
        Certificate=$file
    fi
}

#
# add capability support for binding port less than 1024
#
function addBindCap()
{
    `setcap CAP_NET_BIND_SERVICE=+eip /opt/Citrix/VDA/bin/ctxhdx`
}

# remove capabiltiy support for binding port less than 1024
function removeBindCap()
{
    `setcap CAP_NET_BIND_SERVICE=-eip /opt/Citrix/VDA/bin/ctxhdx`
}

# validate ssl certificate
function validateCert()
{
    # Validate expiration date
    local expire="$(openssl x509 -noout -in $Certificate -enddate 2>&1)"
    expire=${expire##*=}
    expire_format=$(date --utc --date="$expire" +"%Y-%m-%d %H:%M:%S")
    local utcNow=$(date --utc --date="$(date)" +"%Y-%m-%d %H:%M:%S")
    if [[ $expire_format < $utcNow ]]; then
        get_str SHELL_ENABLE_VDASSL_EXPIRE_CERT "$Certificate"
        get_str SHELL_ENABLE_VDASSL_FAIL
        exit 1
    fi  

    # Check certificate trust
    if [[ ! -d $KeyStore/tmp ]]; then
        `mkdir $KeyStore/tmp`
    fi
    if [[ $(ls -A $KeyStore/cacerts | wc -l) != '0' ]]; then
        `cat $KeyStore/cacerts/*.* &> $KeyStore/tmp/root.pem` 
    fi
    if [[ $(ls -A $KeyStore/intcerts | wc -l) != '0' ]]; then
        `cat $KeyStore/intcerts/*.* &> $KeyStore/tmp/int.pem`
    fi
    if [[ $(ls -A $KeyStore/tmp | wc -l) != '0' ]]; then
        `cat $KeyStore/tmp/*.pem &> $KeyStore/tmp/concat.pem`
    fi

    local verify="$(openssl verify -CAfile $KeyStore/tmp/concat.pem -purpose sslserver $Certificate)"

    if [[ "$verify" = *"error"* ]]; then
        get_str SHELL_ENABLE_VDASSL_VERIFY_CERT_FAIL "$Certificate"
        get_str SHELL_ENABLE_VDASSL_FAIL
        exit 1
    else
        get_str SHELL_ENABLE_VDASSL_VERIFY_CERT "$Certificate"
    fi
  
    if [[ ! -d $KeyStore/tmp ]]; then
        `rm -rf $KeyStore/tmp`
    fi

    # Check private key availability
    local privateKey="$(openssl rsa -noout -modulus -in $Certificate | openssl sha1)"
    #local checkRSA="$(openssl rsa -in $Certificate -check)"
    #refer to https://kb.wisc.edu/middleware/page.php?id=4064
    #local privateKey="$(openssl x509 -noout -modulus -in $Certificate | openssl md5)"
    if [[ $privateKey = *"error"* ]]; then
        get_str SHELL_ENABLE_VDASSL_INVALID_PRI_KEY
        get_str SHELL_ENABLE_VDASSL_FAIL
        exit 1
    fi  
}

# set registry key/value for SSL
function setReg()
{
    #Key=$ICA_LISTENER_PATH
    keys=($SSL_KEYSTORE_KEY $SSL_CERT_KEY $ENABLE_SSL_KEY $SSL_PORT_KEY $SSL_MINVERSION_KEY $SSL_CIPHERSUITE_KEY)
    # SSLSDK just need the name in API SSLPSetCertificate(). so we trim the path here.
    Certificate=$(basename "$Certificate")
    values=($KeyStore $Certificate $EnableSSL $SSLPort $SSLVer $SSLCS)
    if [[ "${#keys[@]}" -eq "${#values[@]}" ]]; then
        for ((index=0; index<${#keys[@]}; index++)) ; do
            Value="${keys[$index]}"
            Data=${values[$index]}
            setData
            get_str SHELL_ENABLE_VDASSL_SET_REG "$Value" "$Data"
        done
    else
        # should never go here
        get_str SHELL_ENABLE_VDASSL_KEYVALUE_NOMATCH
        exit 1
    fi
}

# set ACLs for certificate
function setACL()
{
    # assign KeyStore rx access
    `setfacl -R -m g:ctxadm:rx $KeyStore`
    # crls need write access for sslsdk
    `setfacl -m g:ctxadm:wx $KeyStore/crls`
}

# restart ctxhdx/ctxvda services to take SSL change into effect. 
function serviceStart()
{
    Title="$(get_str SHELL_ENABLE_VDASSL_RESTART_SERVICE)" 
    Content="$(get_str SHELL_ENABLE_VDASSL_RESTART_ENABLE_SSL_SERVICE 'ctxhdx')" 
    shouldProcess #?
    
    if [[ "$Config" = "true" ]]; then
        service ctxhdx restart
    fi
}

################## the main process block ################
main()
{
    # Check if the user is an root
    checkUser

    if [[ $# = 0 ]]; then
        usage
        exit 1
    fi

    # handle parameters
    if [[ $# > 0 ]]; then
        argsMatch
        # validate parameters
        validateArgs
    fi

    # Get OS Type
    getOsType
    
    # Read ICAListener, SessionReliability and WebSocket ports from the registry
    Key=$POLICIES_PATH
    Value=$ICA_LISTENER_PORT_KEY
    IcaPort=$(getData)
    
    if [[ -z "$IcaPort" ||  "$IcaPort" = *"Error"* || "$IcaPort" = *"Failed"* ]]; then
        IcaPort=1494
    fi

    #we don't support session reliability and html5 at this moment.
: <<'REMOVE'
    Value=$SESSION_RELIABILITY_PORT_KEY
    SessionReliabilityPort=$(getData)
    if [[ -z "$SessinReliabilityPort" || "$SessinReliabilityPort" = *"Error"* ]]; then
        SessionReliabilityPort=2598
    fi
    #echo "SessionReliabilityPort=$SessionReliabilityPort"

    Value=$WEBSOCKET_PORT_KEY
    WebSocketPort=$(getData)
    if [[ -z "$WebSocketPort" ||  "$WebSocketPort" = *"Error"* ]]; then
        WebSocketPort=8008
    fi
    #echo "WebSocketPort=$WebSocketPort"
REMOVE

    # Handle the Enable/Disable Mode Process
    if [[ "$Disable" = "true" && "$Enable" = "false" ]]; then

        Key=$SSL_LISTENER_PATH
        Value=$ENABLE_SSL_KEY
        Data=0
        setData
        
        # Restore the value of InputBufferLength to be 2048
        Key=$SSL_LISTENER_PATH
        Value=$ICA_INPUT_BUFFER_LENGTH
        Data=$ICA_INPUT_BUFFER_LENGTH_VALUE
        setData
 
        Title=" " 
        Content="$(get_str SHELL_ENABLE_VDASSL_DISABLE_SSL)"
        shouldProcess #?

        if [[ "$Config" = "false" ]]; then
            get_str SHELL_ENABLE_VDASSL_CANCEL                
            exit 1        
        fi

        # Firewall configuration
        firewall_rules_disablessl
        
        removeBindCap # allow ctxhdx bind to SSL port

        
        #Restart ctxhdx service to take SSL change into effect.
        serviceStart

        get_str SHELL_VDASSL_DISABLED

    elif [[ "$Disable" = "false" && "$Enable" = "true" ]]; then
                
        Title="" 
        Content="$(get_str SHELL_ENABLE_VDASSL_ENABLE_SSL)"
        shouldProcess #?

        if [[ "$Config" = "false" ]]; then
            get_str SHELL_ENABLE_VDASSL_CANCEL                
            exit 1        
        fi
        
        # 3 steps to enable ssl
        RegistryKeysSet=false
        ACLsSet=false
        FirewallConfigured=false

        # Create KeyStore
        createKeystore
        
        # when certificate file is specified and with PKCS#12 format(suffix is pfx)
        if [[ "$IsPfx" = 1 ]]; then
            # Convert certificate file from pfx to pem
            converCertFormat

            # Remove password from pem file and generate new pem file
            removePassword
        fi

        # Move Certificate file(pem) to the keystore
        putCertToKeystore

        Certificate=$CertificateFileNamePem

        # retrieve certificate
        getCert        

        # validate certificate
        validateCert

        # Create registry keys to enable SSL to Linux VDA        
        get_str SHELL_ENABLE_VDASSL_SETTING_REG
        EnableSSL=1
        case "$SSLMinVersion" in
             "TLS_1.0")
                 SSLVer=2
                 ;;
             "TLS_1.1")
                 SSLVer=3
                 ;;
             "TLS_1.2")
                 SSLVer=4
                 ;;
             "TLS_1.3")
                 SSLVer=5
                 ;;
        esac

        case "$SSLCipherSuite" in
             "GOV")
                SSLCS=1
                ;;
             "COM")
                SSLCS=2
                ;;
             "ALL")
                SSLCS=3
                ;;
        esac

        Key=$SSL_LISTENER_PATH
        setReg
        RegistryKeysSet="true"

        # Set ACLs
        serviceName="ctxhdx"
        Title="$(get_str SHELL_ENABLE_VDASSL_CFG_ACL_TITLE)" # "Configure ACLs"
        Content="$(get_str SHELL_ENABLE_VDASSL_CFG_ACL "$serviceName")" # "This will grant $serviceName read/write access to the certificate keystore."
        shouldProcess #?
        if [[ "$Config" = "true" ]]; then
             setACL
             ACLsSet="true"
        else
             get_str SHELL_ENABLE_VDASSL_CFG_ACL_SKIP
        fi

        # Configure firewall
        Title="$(get_str SHELL_ENABLE_VDASSL_CFG_FIREWALL_TITLE)" # "Configure Firewall"
        Content="$(get_str SHELL_ENABLE_VDASSL_FIREWALL_ENABLE_SSL $SSLPort)" # "This will remove any existing firewall rules for port $SSLPort and the rules for CGP, ICA and Websocket, if present."
        shouldProcess #?

        if [[ "$Config" = "true" ]]; then
             firewall_rules_enablessl
             FirewallConfigured="true"
        else
             get_str SHELL_ENABLE_VDASSL_CFG_FIREWALL_SKIP
        fi

        if [[ "$RegistryKeysSet" = "true" && "$ACLsSet" = "true" && "$FirewallConfigured" = "true" ]]; then
             addBindCap #add bind capability for ctxhdx
        else
             if [[ "$RegistryKeysSet" = "false" ]]; then
                 get_str SHELL_ENABLE_VDASSL_REG_SKIPPED
             fi
             if [[ "$ACLsSet" = "false" ]]; then
                 get_str SHELL_ENABLE_VDASSL_ACL_SKIPPED
             fi
             if [[ "$FirewallConfigured" = "false" ]]; then
                 get_str SHELL_ENABLE_VDASSL_FIREWALL_SKIPPED
             fi
         fi
     
         #Restart ctxhdx service to take SSL change into effect.
         serviceStart
         get_str SHELL_VDASSL_ENABLED

         #Remove temp directory. 
         if [[ -d $KeyStore/tmp ]]; then
            rm -r -f $KeyStore/tmp
         fi
    else
         get_str SHELL_ENABLE_VDASSL_INVALID_OPT2 
         usage
         exit 1
    fi # end  if [[ "$Disable" = true && "$Enable" = false ]]; then

}

main "$@"

