#!/bin/bash

function ldconfig_update {
    function inner {
        /sbin/ldconfig /opt/Citrix/VDA/lib64 || return 1
    }
    cmdlog_exec inner $FUNCNAME "update ldconfig"
}

function postgresql_start {
    function inner {
        /bin/systemctl start postgresql || return 1
    }
    cmdlog_exec inner $FUNCNAME "start PostgreSQL"
}

function postgresql_restart {
    function inner {
        /bin/systemctl restart postgresql || return 1
    }
    cmdlog_exec inner $FUNCNAME "restart PostgreSQL"
}

function service_start {
    function inner {
        /bin/systemctl daemon-reload || return 1
        if [ -f /etc/systemd/system/ctxlogd.service ]; then
            /bin/systemctl start ctxlogd || return 1
        fi
        /bin/systemctl start ctxhdx || return 1
        /bin/systemctl start ctxpolicyd || return 1
        /bin/systemctl start ctxjproxy || return 1
        /bin/systemctl start ctxvda || return 1
        /bin/systemctl start ctxcups
        /bin/systemctl start cups
        /bin/systemctl start ctxvhcid
        /bin/systemctl start ctxusbsd
        /bin/systemctl start ctxcdm
        /bin/systemctl start ctxceip
        /bin/systemctl start ctxtelemetry.socket
        /bin/systemctl start ctxbcrd
        /bin/systemctl start ctxmonitorservice
        /bin/systemctl start ctxgdtd
        /bin/systemctl start ctxwcamsd
        /bin/systemctl start ctxsdcd
    }
    cmdlog_exec inner $FUNCNAME "start $short_name"
}

function service_stop {
    function inner {
        /bin/systemctl daemon-reload || return 1
        /bin/systemctl stop ctxsdcd
        /bin/systemctl stop ctxwcamsd
        /bin/systemctl stop ctxgdtd
        /bin/systemctl stop ctxmonitorservice
        /bin/systemctl stop ctxbcrd
        /bin/systemctl stop ctxcdm
        /bin/systemctl stop ctxceip
        /bin/systemctl stop ctxcups
        /bin/systemctl stop cups
        /bin/systemctl stop ctxusbsd
        /bin/systemctl stop ctxvhcid
        /bin/systemctl stop ctxtelemetry.socket
        /bin/systemctl stop ctxtelemetry@*.service

        /bin/systemctl stop ctxpolicyd || return 1
        /bin/systemctl stop ctxvda || return 1
        /bin/systemctl stop ctxjproxy || return 1
        /bin/systemctl stop ctxhdx || return 1
        if [ -f /etc/systemd/system/ctxlogd.service ]; then
            /bin/systemctl stop ctxlogd || return 1
        fi
    }
    cmdlog_exec inner $FUNCNAME "stop $short_name"
}

function service_kill {
    function inner {
        /bin/systemctl is-active -q ctxsdcd && /bin/systemctl kill ctxsdcd
        /bin/systemctl is-active -q ctxwcamsd && /bin/systemctl kill ctxwcamsd
        /bin/systemctl is-active -q ctxgdtd && /bin/systemctl kill ctxgdtd
        /bin/systemctl is-active -q ctxmonitorservice && /bin/systemctl kill ctxmonitorservice
        /bin/systemctl is-active -q ctxbcrd && /bin/systemctl kill ctxbcrd
        /bin/systemctl is-active -q ctxcdm && /bin/systemctl kill ctxcdm
        /bin/systemctl is-active -q ctxceip && /bin/systemctl kill ctxceip
        /bin/systemctl is-active -q ctxcups && /bin/systemctl kill ctxcups
        /bin/systemctl is-active -q ctxusbsd && /bin/systemctl kill ctxusbsd
        /bin/systemctl is-active -q ctxvhcid && /bin/systemctl kill ctxvhcid
        /bin/systemctl is-active -q ctxtelemetry && /bin/systemctl kill ctxtelemetry

        /bin/systemctl is-active -q ctxpolicyd && /bin/systemctl kill ctxpolicyd
        /bin/systemctl is-active -q ctxvda && /bin/systemctl kill ctxvda
        /bin/systemctl is-active -q ctxjproxy && /bin/systemctl kill ctxjproxy
        /bin/systemctl is-active -q ctxhdx && /bin/systemctl kill ctxhdx
        if [ -f /etc/systemd/system/ctxlogd.service ]; then
            /bin/systemctl is-active -q ctxlogd && /bin/systemctl kill ctxlogd
        fi
        if [ -f /etc/systemd/system/ad_join.service ]; then
            /bin/systemctl is-active -q ad_join && /bin/systemctl kill ad_join
        fi
        return 0
    }
    cmdlog_exec inner $FUNCNAME "kill $short_name"
}

function service_register {
    function inner {
        /bin/systemctl daemon-reload || return 1
        if [ -f /etc/systemd/system/ctxlogd.service ]; then
            /bin/systemctl enable ctxlogd || return 1
        fi
        /bin/systemctl enable ctxhdx || return 1
        /bin/systemctl enable ctxvda || return 1
        /bin/systemctl enable ctxjproxy || return 1
        /bin/systemctl enable ctxpolicyd || return 1

        /bin/systemctl enable cups
        /bin/systemctl enable ctxcups
        /bin/systemctl enable ctxcdm
        /bin/systemctl enable ctxceip
        /bin/systemctl enable ctxvhcid
        /bin/systemctl enable ctxusbsd
        /bin/systemctl enable ctxtelemetry.socket

        /bin/systemctl enable ctxbcrd
        /bin/systemctl enable ctxmonitorservice
        /bin/systemctl enable ctxrunatboot
        /bin/systemctl enable ctxgdtd
        /bin/systemctl enable ctxwcamsd
        /bin/systemctl enable ctxsdcd
    }
    cmdlog_exec inner $FUNCNAME "register $short_name services"
}

function service_unregister {
    function inner {
        /bin/systemctl daemon-reload || return 1
        /bin/systemctl disable ctxsdcd
        /bin/systemctl disable ctxwcamsd
        /bin/systemctl disable ctxgdtd
        /bin/systemctl disable ctxmonitorservice
        /bin/systemctl disable ctxbcrd
        /bin/systemctl disable ctxusbsd
        /bin/systemctl disable ctxvhcid
        /bin/systemctl disable ctxcdm
        /bin/systemctl disable ctxceip
        /bin/systemctl disable ctxcups
        /bin/systemctl disable cups
        /bin/systemctl disable ctxtelemetry.socket

        /bin/systemctl disable ctxvda || return 1
        /bin/systemctl disable ctxjproxy || return 1
        /bin/systemctl disable ctxhdx || return 1
        if [ -f /etc/systemd/system/ctxlogd.service ]; then
            /bin/systemctl disable ctxlogd || return 1
        fi
        if [ -f /etc/systemd/system/ad_join.service ]; then
            /bin/systemctl disable ad_join || return 1
        fi
    }
    cmdlog_exec inner $FUNCNAME "unregister $short_name services"
}

function firewall_add_rules {
    function inner {
        /usr/sbin/ufw allow proto tcp from any to any port $1|| return 1
        /usr/sbin/ufw allow proto tcp from any to any port $2|| return 1
        /usr/sbin/ufw allow proto tcp from any to any port $3 || return 1
        /usr/sbin/ufw allow proto tcp from any to any port 1494 || return 1
        /usr/sbin/ufw allow proto tcp from any to any port 2598 || return 1
        /usr/sbin/ufw allow proto udp from any to any port 1494 || return 1
        /usr/sbin/ufw allow proto udp from any to any port 2598 || return 1
        /usr/sbin/ufw allow proto tcp from any to any port 8008 || return 1
        /usr/sbin/ufw allow proto tcp from any to any port 6001:6099
    }
    cmdlog_exec inner $FUNCNAME "add firewall rules for $short_name" $1 $2 $3
}

function firewall_update_rules {
    function inner {
        /usr/sbin/ufw allow proto tcp from any to any port 2598 || return 1
        /usr/sbin/ufw allow proto udp from any to any port 1494 || return 1
        /usr/sbin/ufw allow proto udp from any to any port 2598 || return 1
        /usr/sbin/ufw allow proto tcp from any to any port 8008 || return 1
        /usr/sbin/ufw allow proto tcp from any to any port 6001:6099
    }
    cmdlog_exec inner $FUNCNAME "update firewall rules for $short_name" $1
}

function firewall_remove_rules {
    function inner {
        export vda_port=$(printf "%d" $(/opt/Citrix/VDA/bin/ctxreg read -k "HKLM\Software\Citrix\VirtualDesktopAgent" -v "ControllerRegistrarPort"))
        export telemetry_socket_port=$(grep -oP '(?<=ListenStream=)\d+' /etc/systemd/system/ctxtelemetry.socket)
        export telemetry_port=$(printf "%d" $(/opt/Citrix/VDA/bin/ctxreg read -k "HKLM\Software\Citrix\VirtualDesktopAgent" -v "TelemetryServicePort"))
        /usr/sbin/ufw delete allow proto tcp from any to any port $telemetry_socket_port
        /usr/sbin/ufw delete allow proto tcp from any to any port $telemetry_port
        /usr/sbin/ufw delete allow proto tcp from any to any port $vda_port
        /usr/sbin/ufw delete allow proto tcp from any to any port 1494 || return 1
        /usr/sbin/ufw delete allow proto tcp from any to any port 2598 || return 1
        /usr/sbin/ufw delete allow proto udp from any to any port 1494 || return 1
        /usr/sbin/ufw delete allow proto udp from any to any port 2598 || return 1
        /usr/sbin/ufw delete allow proto tcp from any to any port 8008 || return 1
        /usr/sbin/ufw delete allow proto tcp from any to any port 6001:6099
    }
    cmdlog_exec inner $FUNCNAME "remove firewall rules for $short_name"
}

function firewall_try_restart {
    function inner {
        /usr/sbin/ufw reload
    }
    cmdlog_exec inner $FUNCNAME "restart the firewall"
}

function ctxXrandrhook {
    BEGIN_PATTERN="# Citrix configuration begin"
    MAIN_CONTENT="export LD_PRELOAD='/usr/\$LIB/libctxXrandrhook.so'"
    END_PATTERN="# Citrix congiguration end"
    XCLIENTS="/etc/X11/Xsession"
    XCLIENTS_ORIG="/etc/X11/Xsession.orig"
    [ "$#" -ne "1" ] && return 1
    case $1 in
        "configure")
            if [ -f $XCLIENTS ]; then
                [[ ! -f $XCLIENTS_ORIG ]] && cp $XCLIENTS $XCLIENTS_ORIG
                sed -i -e "/$BEGIN_PATTERN/,/$END_PATTERN/d" $XCLIENTS
                awk -v "txt=$BEGIN_PATTERN\n$MAIN_CONTENT\n$END_PATTERN" '/^[^#]/ && !x {print txt; x=1} 1' "$XCLIENTS" > "${XCLIENTS}.tmp" && mv -f "${XCLIENTS}.tmp" "$XCLIENTS" && chmod a+x+r "$XCLIENTS"
            fi
            ;;
        "restore")
            sed -i -e "/$BEGIN_PATTERN/,/$END_PATTERN/d" $XCLIENTS
            ;;
        *)
            ;;
    esac
}

function mate_session {
    [ "$#" -ne "1" ] && return 1
    case $1 in
        "configure")
            update-alternatives --set x-session-manager /usr/bin/mate-session
            ;;
        "restore")
            GNOME_SESSION=$(update-alternatives --list x-session-manager | grep -i gnome-session)
            if [ -n "$GNOME_SESSION" ]; then
                update-alternatives --set x-session-manager $GNOME_SESSION
            fi
            ;;
        *)
            ;;
    esac
}

function mate_session_configure {
    function inner {
        mate_session configure
    }
    cmdlog_exec inner $FUNCNAME "mate session configured"
}

function mate_session_restore {
    function inner {
        mate_session restore
    }
    cmdlog_exec inner $FUNCNAME "mate session restored"
}

function desktop_environment_configure {
    function inner {
		#sversion=`lsb_release -rs`
        ctxXrandrhook configure
        cp -f /var/xdl/10-ctx-media-automount.rules /etc/udev/rules.d
        cp -f /var/xdl/50-ctx-persistent-wcam.rules /etc/udev/rules.d
        cp -f /var/xdl/10-ctxhdx-disable-power-management.rules /usr/share/polkit-1/rules.d || return 1
        #In Debian10 the polkit version is 0.105<0.106 so only the old snytax .pkla configuration works
        cp -f /var/xdl/10-ctxhdx-disable-power-management-ubuntu16.pkla /var/lib/polkit-1/localauthority/10-vendor.d/10-ctxhdx-disable-power-management.pkla || return 1
        cp -f /var/xdl/20-no-show-color-dialog.pkla  /var/lib/polkit-1/localauthority/10-vendor.d/ || return 1
        cp -f /var/xdl/20-no-show-refresh-dialog.pkla  /var/lib/polkit-1/localauthority/10-vendor.d/ || return 1
        sed -i "s/MountFlags/#MountFlags/g" /lib/systemd/system/systemd-udevd.service
        /bin/systemctl daemon-reload
        systemctl restart systemd-udevd.service
    }
    cmdlog_exec inner $FUNCNAME "desktop environment configured"
}

function desktop_environment_restore {
    function inner {
		ctxXrandrhook restore
        mate_session restore
        rm -f /etc/udev/rules.d/10-ctx-media-automount.rules
        rm -f /etc/udev/rules.d/50-ctx-persistent-wcam.rules
        rm -f /usr/share/polkit-1/rules.d/10-ctxhdx-disable-power-management.rules || return 1
        rm -f /var/lib/polkit-1/localauthority/10-vendor.d/10-ctxhdx-disable-power-management.pkla || return 1
        rm -f /var/lib/polkit-1/localauthority/10-vendor.d/20-no-show-color-dialog.pkla || return 1
        rm -f /var/lib/polkit-1/localauthority/10-vendor.d/20-no-show-refresh-dialog.pkla || return 1
    }
    cmdlog_exec inner $FUNCNAME "desktop environment restored"
}

function remove_mcs_config {
    function inner {
        dbus_service_file_path="/lib/systemd/system/dbus.service"
        sed -i -e '/^ExecStartPre=.*$/ d' "$dbus_service_file_path"
        cron_job="/etc/cron.d/mcs_update_password_cronjob"
        if [ -f $cron_job ]; then
            rm -f $cron_job
        fi
    }
    cmdlog_exec inner $FUNCNAME "Remove mcs configurations"
}

function pcsclibhook {
    PCSC_LIB_NAME="libpcsclite.so.1.0.0"
    PCSC_HOOK_NAME="libctxpcsc.so"
    PCSC_LIB_LINK="libpcsclite.so.1"
    PCSC_LIB_DIR="/usr/lib/x86_64-linux-gnu/"
    PCSC_BAK_DIR="/var/xdl"
    PCSC_HOOK_DIR="/opt/Citrix/VDA/lib64"

    [ "$#" -ne "1" ] && return 1
    case $1 in
        "configure")
            if [ -e $PCSC_LIB_DIR/$PCSC_LIB_NAME ]; then
                mv $PCSC_LIB_DIR/$PCSC_LIB_NAME $PCSC_BAK_DIR/
                ldconfig

                if [ -e $PCSC_LIB_DIR/$PCSC_LIB_LINK ]; then
                    rm -f $PCSC_LIB_DIR/$PCSC_LIB_LINK
                fi

                ln $PCSC_HOOK_DIR/$PCSC_HOOK_NAME -s $PCSC_LIB_DIR/$PCSC_LIB_LINK

            fi
            ;;
        "restore")
            if [ -e $PCSC_BAK_DIR/$PCSC_LIB_NAME ]; then
                mv $PCSC_BAK_DIR/$PCSC_LIB_NAME $PCSC_LIB_DIR/
                ldconfig
            fi
            ;;
        *)
            ;;
    esac
}