#!/bin/bash

###############################################################################
#
# Citrix Virtual Apps & Desktops For Linux Script: Enumerate tasks
#
# Copyright (c) Citrix Systems, Inc. All Rights Reserved.
#

export LC_ALL="en_US.UTF-8"

function usage {
    echo "Usage: $(basename $0) [session id]"
}

function trim {
    echo $(echo "$1" | sed -r 's/^[[:space:]]+|[[:space:]]+$//g')
}

if [ "$#" -ne 1 ]; then
    usage
    exit 1
fi

if [ "$1" == "--help" ]; then
    usage
    exit 0
fi

sid=$1

gfxpid=$(ps h -C ctxgfx -o pid,cmd | grep "-session $sid" | awk '{print $1}')
if [ -z "$gfxpid" ]; then
    echo "Session $sid is not found."
    exit 1
fi

uid=$(ps h -C ctxlogin -o ppid,uid,cmd | grep "^[[:space:]]*$gfxpid" | sed -r 's/[[:digit:]]+[[:space:]]+([[:digit:]]+)[[:space:]]+.*/\1/')
uid=$(trim "$uid")

if [ -z "$uid" ]; then
    echo "Failed to lookup user id for session $sid."
    exit 1
fi

tasks=$(ps h -U "$uid" -o pid,pcpu,rss)

IFS='
'
for task_line in $tasks; do
    pid=$(echo "$task_line" | awk '{print $1}')
    psid=$(cat "/proc/$pid/environ" | strings -1 | grep ^CITRIX_SESSION_ID | awk -F = '{print $2}')
    if [ -z "$psid" ]; then continue; fi
    if [ "$psid" -ne "$sid" ]; then continue; fi
    pname=$(ps h -p "$pid" -o cmd)
    cpu=$(echo "$task_line" | awk '{print $2}')
    mem=$(echo "$task_line" | awk '{print $3}')
    uname=$(getent passwd "$uid" | awk -F : '{print $1}')
    echo -ne "$pid\t$sid\t"
    echo -nE "$uname"
    echo -e "\t$mem\t$cpu\t$pname"
done

