#!/bin/bash

###############################################################################
#
# Citrix Virtual Apps & Desktops For Linux Script: Machine Creation Service
# Copyright (c) Citrix Systems, Inc. All Rights Reserved.
#

source /etc/xdl/mcs/mcs.conf

build_dir="/opt/Citrix/VDA/lib64/mcs"
mcs_sysd_unit_file="ad_join.service"
mcs_boot_script="ad_join.sh"
mcs_script_folder="/var/xdl/mcs"
mcs_setting_flag_file="${mcs_script_folder}/.mcs_setting_flag"
filePath="/tmp/ctxmcs"

centrifyPath="${filePath}/Centrify"
centrifySambaPath="${centrifyPath}/Samba"
id_disk_mnt_point="/var/xdl/iddisk"
ad_info_file_path="/PvsVm/CTXSOSID.INI"
ceip_machine_id="/CEIPMachineGuid"
SAMBA_SECRETS="/var/lib/samba/private/secrets.tdb"
KEYTAB="/etc/krb5.keytab"
pbis_pkgList="pbis"
pbisPath="${filePath}/pbis"

persistent_data_location="/var/xdl/vda"
keydata_name="keydata.json"

####cron job####
cron_folder="/etc/cron.d"
cron_job_file="mcs_update_password_cronjob"

AD_INTEGRATION=`echo $AD_INTEGRATION | tr 'A-Z' 'a-z'`
UPDATE_MACHINE_PW=`echo $UPDATE_MACHINE_PW | tr 'A-Z' 'a-z'`
NON_DOMAIN_JOINED="N"
NEW_HOSTNAME=""
new_hostname=""
AD_FQDN=""
DDCS=""
domain=""
realm=""
PASSWORD=""
DNSDISID=""
DDC=""
REALM=""
retry_max=10

# Websocket related
Local_Mapped_Account="HKLM\\Software\\Citrix\VirtualDesktopAgent\\LocalMappedAccounts"
Websocket_Reg_Path="HKLM\\SYSTEM\\CurrentControlSet\\Services\\CitrixBrokerAgent\\WebSocket"
PrivateKey=""
PublicKey=""
PrivateKeyTimeStamp=""
VdaHostId=""
VirtualSiteId=""
SiteAddress=""
CustomerId=""
Websocket_Enabled_Str=""
WebsocketEnabled=""
NonDomainJoined=""
MachineName=""

osVersion=""
coreFileRhel="/etc/redhat-release"

if [[ -f $coreFileRhel ]];then
    osVersion="$(cat $coreFileRhel |cut -d ' ' -f6  2>&1)"
fi

function log()
{
    PREFIX=$(date +"%b %d %T")
    echo "${PREFIX} ${1}" >> $logFile
}

function log_echo()
{
    log "$1"
    echo "$1"
}

function create_log_file()
{
    if [[ ! -f "$logFile" ]]; then
        touch "$logFile"
        if [[ "$?" -ne "0" ]]; then
          log_echo "Create log file $logFile failed"
        fi
    fi

    echo "#### Begin $scriptName ####">>$logFile
    str="`date "+%Y-%m-%d %H:%M:%S"`"
    echo $str>>$logFile
}

#
# Check user, only root user has the permission to run this script
#
function check_user()
{
    if [ "$(id -u)" != 0 ]; then
        log_echo "Error: The script must be run by root user"
        exit 1
    fi
}

function check_string_empty()
{
    if [ -z "$2" ]; then
    log_echo "Error: $1 is empty, existing ..."
    exit 1
    fi
}

function check_ntfs_3g()
{
    log_debug "Enter check_ntfs_3g"

    # check whether ntfs-3g is available
    if ! mount.ntfs-3g --version >> "$logFile" 2>&1; then
        log_error "ntfs-3g is not available, pls check ntfs-3g installation, exit"
        exit 1
    else
        log_info "ntfs-3g is available."
    fi

    log_debug "Exit check_ntfs_3g"
}

function install_update_machine_secret_cronjob()
{
    pushd $build_dir
    if [ "$UPDATE_MACHINE_PW" == "enabled" ]; then
        log "installing the cron job file to /etc/cron.d"
        install -m644 $cron_job_file ${cron_folder}
    else
        log "update machine secret is disabled"
    fi
    popd
}

## install mcs related scripts and services
function install_mcs_scripts()
{
    log "Debug: Enter install_mcs_scripts"

    pushd $build_dir

    log_echo "installing mcs systemd unit file: ${mcs_sysd_unit_file} ... "

    install -o root -m 0644 ${mcs_sysd_unit_file} /etc/systemd/system/

    log "reloading systemd daemon..."
    systemctl daemon-reload

    log "setting ${mcs_sysd_unit_file} to start on boot"
    systemctl enable ${mcs_sysd_unit_file}

    log "installing MCS boot script ${mcs_boot_script}"
    install -o root -m 0544 ${mcs_boot_script} ${mcs_script_folder}

    popd

    install_update_machine_secret_cronjob

    log "Debug: Exit install_mcs_scripts"
}

function hack_dbus_service_file()
{
    log "Debug: Enter hack_dbus_service_file"

    service_path=$1
    log "dbus service file path is: $service_path"

    sed -i -e '/^ExecStartPre=.*$/ d' "$service_path"
    sed -i "/\[Service\]/a ExecStartPre=-/bin/bash ${mcs_script_folder}/${mcs_boot_script} --config" "$service_path"
    log "reloading systemd daemon..."
    systemctl daemon-reload

    log "Debug: Exit hack_dbus_service_file"
}

function conf_hostname_non_domain_joined()
{
    log "Debug: Enter conf_hostname_non_domain_joined"

    hostname_file="/etc/hostname"

    echo "$MachineName" > "$hostname_file"
    hostname "$MachineName"

    sysctl -w kernel.hostname="$MachineName" >> "$logFile" 2>&1
    log "Debug: Exit conf_hostname_non_domain_joined"
}

function remove_default_ntp_server()
{
    log "Debug: Enter remove_default_ntp_server"
    ntp_file=$1

    log "Remove default ntp server in $ntp_file"
    sed -i -e '/^server.*iburst$/ d' "$ntp_file"
    sed -i -e '/^pool.*iburst$/ d' "$ntp_file"

    log "Debug: Exit remove_default_ntp_server"
}

function sync_with_ntp_server()
{
    log "Debug: Enter sync_with_ntp_server"

    time_server=$1
    log_echo "sync time with server $time_server"
    sleep 2
    /usr/sbin/ntpdate -b "$time_server" >> "$logFile" 2>&1

    log "Debug: Exit sync_with_ntp_server"
}

function change_id_disk_setting()
{
    log "Debug: Enter change_id_disk_setting"

    setting_name=$1
    setting_value=$2
    log "Change $setting_name in identity disk to $setting_value"

    sed -i "s/^${setting_name}=.*$/${setting_name}=`echo ${setting_value}`/g" "${id_disk_mnt_point}${ad_info_file_path}"

    log "Debug: Exit change_id_disk_setting"
}

function read_id_disk_for_machine_pw()
{
    log "Debug: Enter read_id_disk_for_machine_pw"
    if [ ! -f ${id_disk_mnt_point}${ad_info_file_path} ]; then
        log " ERROR: no identity disk found, exit"
        exit 1
    fi
    realm=`cat ${id_disk_mnt_point}${ad_info_file_path} | grep 'DomainName' | cut -d'=' -f2 | sed 's/\s//g'`
    domain=`echo ${realm} | cut -d'.' -f1`
    DOMAIN=`echo ${domain} | tr 'a-z' 'A-Z'`

    # Get WORKGROUP info
    if [[ -z "$WORKGROUP" ]]; then
        log_echo "WORKGROUP is not set in config file, set to $DOMAIN"
        WORKGROUP=$DOMAIN
    else
        log_echo "Read WORKGROUP from config file: $WORKGROUP"
        WORKGROUP=`echo $WORKGROUP | tr 'a-z' 'A-Z'`
    fi

    log "Debug: Exit read_id_disk_for_machine_pw"
}

function update_private_key_timestamp()
{
    log "Debug: Enter update_private_key_timestamp"

    # first read private key and timestamp from registry
    updated_privatekey=`/opt/Citrix/VDA/bin/ctxreg read -k $Websocket_Reg_Path -v "PrivateKey"`
    updated_publickey=`/opt/Citrix/VDA/bin/ctxreg read -k $Websocket_Reg_Path -v "PublicKey"`
    updated_timestamp=`/opt/Citrix/VDA/bin/ctxreg read -k $Websocket_Reg_Path -v "PrivateKeyTimeStamp"`

    # update values to identity disk
    change_id_disk_setting "PrivateKey" $updated_privatekey
    change_id_disk_setting "PublicKey" $updated_publickey
    change_id_disk_setting "PrivateKeyTimeStamp" $updated_timestamp

    log "Debug: Exit update_private_key_timestamp"
}

function backup_keydata_json_file()
{
    log "Debug: Enter backup_keydata_json_file"
    if [[ -f $persistent_data_location/$keydata_name ]]; then
        cp $persistent_data_location/$keydata_name $id_disk_mnt_point/$keydata_name
    fi
    log "Debug: Exit backup_keydata_json_file"
}

function restore_keydata_json_file()
{
    log "Debug: Enter restore_keydata_json_file"
    if [[ -f $id_disk_mnt_point/$keydata_name ]]; then
        cp $id_disk_mnt_point/$keydata_name $persistent_data_location/$keydata_name
        chmod 600 $persistent_data_location/$keydata_name
    fi
    log "Debug: Exit restore_keydata_json_file"
}

function read_domain_joined_info()
{
    log "Debug: Enter read_domain_joined_info"
    # extract information from id disk
    NEW_HOSTNAME=`cat ${id_disk_mnt_point}${ad_info_file_path} | grep 'LongHostName' | cut -d'=' -f2 | sed 's/\s//g'`
    AD_FQDN=`cat ${id_disk_mnt_point}${ad_info_file_path} | grep 'DCName' | cut -d'=' -f2 | sed 's/\\\\//g' | sed 's/\s//g'`
    DDCS=`cat ${id_disk_mnt_point}${ad_info_file_path} | grep 'ListOfDDCs' | cut -d'=' -f2 | sed 's/\s//g'`
    realm=`cat ${id_disk_mnt_point}${ad_info_file_path} | grep 'DomainName' | cut -d'=' -f2 | sed 's/\s//g'`

    # password is base64 encoded, so it need to be decoded here
    PASSWORD=`cat ${id_disk_mnt_point}${ad_info_file_path} | grep 'MachinePassword' | cut -d'=' -f2 | sed 's/\s//g' | base64 -d`

    # DOMAIN SID
    DNSDISID=`cat ${id_disk_mnt_point}${ad_info_file_path} | grep 'DNSDISid' | cut -d'=' -f2 | sed 's/\s//g'`

    check_string_empty "NEW_HOSTNAME" $NEW_HOSTNAME
    check_string_empty "AD_FQDN" $AD_FQDN
    check_string_empty "DDCS" $DDCS
    check_string_empty "realm" $realm
    check_string_empty "PASSWORD" $PASSWORD
    check_string_empty "DNSDISID" $DNSDISID

    log_echo "NEW_HOSTNAME = $NEW_HOSTNAME"
    log_echo "AD_FQDN = $AD_FQDN"
    log_echo "DDCS = $DDCS"
    log_echo "realm = $realm"
    log_echo "DNSDISID = $DNSDISID"


    DDCS=`echo ${DDCS} | tr ',' ' '`
    REALM=`echo $realm | tr 'a-z' 'A-Z'`
    realm_lower=`echo $realm | tr 'A-Z' 'a-z'`
    new_hostname=`echo $NEW_HOSTNAME | tr 'A-Z' 'a-z'`
    domain=`echo ${realm} | cut -d'.' -f1`
    DOMAIN=`echo ${domain} | tr 'a-z' 'A-Z'`
    Use_Existing_Configurations_Of_Current_VDA=`echo ${Use_Existing_Configurations_Of_Current_VDA} | tr 'a-z' 'A-Z'`

    # Get configuration info
    if [[ -z "$Use_Existing_Configurations_Of_Current_VDA" ]]; then
        log_echo "Use_Existing_Configurations_Of_Current_VDA is not set in config file, set to "N""
        Use_Existing_Configurations_Of_Current_VDA="N"
    else
        log_echo "Read Use_Existing_Configurations_Of_Current_VDA from config file: $Use_Existing_Configurations_Of_Current_VDA"
    fi
    # Get WORKGROUP info
    if [[ -z "$WORKGROUP" ]]; then
        log_echo "WORKGROUP is not set in config file, set to $DOMAIN"
        WORKGROUP=$DOMAIN
    else
        log_echo "Read WORKGROUP from config file: $WORKGROUP"
        WORKGROUP=`echo $WORKGROUP | tr 'a-z' 'A-Z'`
    fi
    # Get NTP_SERVER info
    if [[ -z "$NTP_SERVER" ]]; then
        log_echo "NTP_SERVER is not set in config file, set to $AD_FQDN"
        NTP_SERVER=$AD_FQDN
    else
        log_echo "Read NTP_SERVER from config file: $NTP_SERVER"
    fi
    # Get LDAP_LIST info
    if [[ -z "$LDAP_LIST" ]]; then
        LDAP_LIST="${AD_FQDN}:389"
        log_echo "LDAP_LIST is not set in config file, set to $LDAP_LIST"
    else
        log_echo "Read LDAP_LIST from config file: $LDAP_LIST"
    fi

    log_echo "domain = $domain"

    # Write CEIPMachineGuid
    if [[ -f "${id_disk_mnt_point}${ceip_machine_id}" ]]; then
        log_echo "${id_disk_mnt_point}${ceip_machine_id} already exists"
    else
        UUID=$(cat /proc/sys/kernel/random/uuid)
        `echo ${UUID} > ${id_disk_mnt_point}${ceip_machine_id}`
        log_echo "Created ${id_disk_mnt_point}${ceip_machine_id}"
    fi

    log "Debug: Exit read_domain_joined_info"
}

function read_non_domain_joined_info()
{
    log "Debug: Enter read_non_domain_joined_info"
    # check if websocket enabled
    TrustIdentity=`cat ${id_disk_mnt_point}${ad_info_file_path} | grep '\[TrustIdentity\]' | sed 's/\s//g'`
    if [ "$TrustIdentity" == "[TrustIdentity]" ]; then
        NonDomainJoined=1
    fi
    # extract the STATUS file name
    # Websocket related info
    PrivateKey=`cat ${id_disk_mnt_point}${ad_info_file_path} | grep 'PrivateKey' | grep -v 'TimeStamp' | cut -d '=' -f 2- | sed 's/\s//g'`
    PublicKey=`cat ${id_disk_mnt_point}${ad_info_file_path} | grep 'PublicKey' | grep -v 'TimeStamp' | cut -d '=' -f 2- | sed 's/\s//g'`
    PrivateKeyTimeStamp=`cat ${id_disk_mnt_point}${ad_info_file_path} | grep 'PrivateKeyTimeStamp' | cut -d'=' -f2 | sed 's/\s//g'`
    VdaHostId=`cat ${id_disk_mnt_point}${ad_info_file_path} | grep 'VdaHostId' | cut -d'=' -f2 | sed 's/\s//g'`
    SiteAddress=`cat ${id_disk_mnt_point}${ad_info_file_path} | grep 'PublicEndpoint' | cut -d'=' -f2 | sed 's/\s//g'`
    VirtualSiteId=`cat ${id_disk_mnt_point}${ad_info_file_path} | grep 'VirtualSiteId' | cut -d'=' -f2 | sed 's/\s//g'`
    CustomerId=`cat ${id_disk_mnt_point}${ad_info_file_path} | grep 'CustomerId' | cut -d'=' -f2 | sed 's/\s//g'`
    Websocket_Enabled_Str=`cat ${id_disk_mnt_point}${ad_info_file_path} | grep 'WebSocketsEnabled' | cut -d'=' -f2 | sed 's/\s//g'`
    MachineName=`cat ${id_disk_mnt_point}${ad_info_file_path} | grep 'MachineName' | cut -d'=' -f2 | sed 's/\s//g'`

    # set a dummy ddc value
    DDCS="ddc.xd.local"

    if [ "$Websocket_Enabled_Str" == "True" ]; then
        WebsocketEnabled=1
    else
        WebsocketEnabled=0
    fi

    echo "PublicKey = $PrivateKeyTimeStamp"
    echo "PrivateKeyTimeStamp = $PrivateKeyTimeStamp"
    echo "VdaHostId = $VdaHostId"
    echo "SiteAddress = $SiteAddress"
    echo "VirtualSiteId = $VirtualSiteId"
    echo "CustomerId = $CustomerId"
    echo "Websocket_Enabled_Str = $Websocket_Enabled_Str"
    echo "MachineName = $MachineName"
    echo "WebsocketEnabled = $WebsocketEnabled"
    echo "NonDomainJoined = $NonDomainJoined"

    log "Debug: Exit read_domain_joined_info"
}

function write_non_domain_info_to_registry()
{
    log "Debug: Enter write_non_domain_info_to_registry"
    sudo /opt/Citrix/VDA/bin/ctxreg create -k $Websocket_Reg_Path -t "REG_SZ" -v "PrivateKey" -d $PrivateKey --force
    sudo /opt/Citrix/VDA/bin/ctxreg create -k $Websocket_Reg_Path -t "REG_SZ" -v "PublicKey" -d $PublicKey --force
    sudo /opt/Citrix/VDA/bin/ctxreg create -k $Websocket_Reg_Path -t "REG_SZ" -v "PrivateKeyTimeStamp" -d $PrivateKeyTimeStamp --force
    sudo /opt/Citrix/VDA/bin/ctxreg create -k $Websocket_Reg_Path -t "REG_SZ" -v "VdaHostId" -d $VdaHostId --force
    sudo /opt/Citrix/VDA/bin/ctxreg create -k $Websocket_Reg_Path -t "REG_SZ" -v "SiteAddress" -d $SiteAddress --force
    sudo /opt/Citrix/VDA/bin/ctxreg create -k $Websocket_Reg_Path -t "REG_SZ" -v "VirtualSiteId" -d $VirtualSiteId --force
    sudo /opt/Citrix/VDA/bin/ctxreg create -k $Websocket_Reg_Path -t "REG_SZ" -v "CustomerId" -d $CustomerId --force
    sudo /opt/Citrix/VDA/bin/ctxreg create -k $Websocket_Reg_Path -t "REG_DWORD" -v "Enabled" -d $((WebsocketEnabled)) --force
    sudo /opt/Citrix/VDA/bin/ctxreg create -k $Websocket_Reg_Path -t "REG_DWORD" -v "NonDomainJoinedMode" -d $((NonDomainJoined)) --force

    # write local mapped account key
    sudo /opt/Citrix/VDA/bin/ctxreg create -k $Local_Mapped_Account --force

    log "Debug: Exit write_non_domain_info_to_registry"
}

function locate_id_disk()
{
    log "Debug: Enter locate_id_disk"

    id_disk=$(blkid -o device -lt LABEL="MCS-IDDisk")

    # find id disk successfully
    if [[ -n "$id_disk" ]]; then
        if  mount | grep "$id_disk" | grep  ${id_disk_mnt_point}; then
            log_echo "$id_disk is an id disk and already mounted, umount it"
            umount "$id_disk" >> "$logFile" 2>&1
        fi
        log_echo "mount $id_disk to ${id_disk_mnt_point}"
        mount -t ntfs-3g -o umask=077 "$id_disk" "${id_disk_mnt_point}" >> "$logFile" 2>&1
        log "Debug: find id disk $id_disk, return"
        return
    fi

    # for compatiablity consideration, keep the process of locating by size to collaborate with outdated MCS server.
    # test whether id disk is attached
    iddisk_dev_name=$(lsblk | grep -E "13M|1021M" | sed 's/^[^[:alnum:]]*\([[:alnum:]]\+\).*$/\1/')
    disks=($iddisk_dev_name)
    if [ -z "$iddisk_dev_name" ]; then
        log_echo "Error: No id disk is attached to the machine. Exit!"
        exit 0
    fi

    # create the mount point folder if it does not exist
    [ ! -d ${id_disk_mnt_point} ] && mkdir -p ${id_disk_mnt_point}
    found=false

    # check whether the 13M/1021M disk is iddisk, check whether id disk is already mounted, if so, umount it
    # if correct id disk is found, mount it to ${id_disk_mnt_point}
    id_disk_device=""
    for iddisk_dev_name in "${disks[@]}"
    do
        if [ "ntfs" != "$(blkid -s TYPE /dev/$iddisk_dev_name -o value)" ]; then
            log_echo "/dev/$iddisk_dev_name is not an identity disk, continue"
            continue
        fi
        if [ "New Volume" != "$(blkid -s LABEL /dev/$iddisk_dev_name -o value)" ]; then
            log_echo "/dev/$iddisk_dev_name is not an identity disk, continue"
            continue
        fi
        #check whether the 13M/1021M disk is already mounted
        if mount | grep "/dev/$iddisk_dev_name" ; then
            # get the mount point and check if ${id_disk_mnt_point}/PvsVm/CTXSOSID.INI exists
            if  mount | grep "/dev/$iddisk_dev_name" | grep  ${id_disk_mnt_point}; then
                log_echo "/dev/$iddisk_dev_name is an id disk and already mounted, umount it"
                umount /dev/"$iddisk_dev_name" >> "$logFile" 2>&1
            else
                continue
            fi
        fi
        # mount id disk to ${id_disk_mnt_point}
        log_echo "mount /dev/$iddisk_dev_name to ${id_disk_mnt_point}"
        mount -t ntfs-3g -o umask=077 /dev/"$iddisk_dev_name" "${id_disk_mnt_point}" >> "$logFile" 2>&1
        if [[ "$?" -ne "0" ]]; then
            log_echo "/dev/$iddisk_dev_name is not an identity disk, continue"
            continue
        fi
        if [ -f ${id_disk_mnt_point}/PvsVm/CTXSOSID.INI ]; then
            log_echo "/dev/$iddisk_dev_name is an id disk"
            umount /dev/"$iddisk_dev_name"
            found=true
            break
        else
            log_echo "/dev/$iddisk_dev_name is not an identity disk, umount it and continue"
            umount /dev/"$iddisk_dev_name"
            continue
        fi
    done


    # if id disk is not found, exit
    if [ false = "$found" ]; then
        log_echo "Identity disk is not found, exit..."
        exit 0
    else
        # in case ntfslabel not installed
        if which ntfslabel >> "$logFile" 2>&1; then
            ntfslabel /dev/"$iddisk_dev_name" "MCS-IDDisk"
        fi
        mount -t ntfs-3g -o umask=077 /dev/"$iddisk_dev_name" "${id_disk_mnt_point}" >> "$logFile" 2>&1
        log_echo "/dev/$iddisk_dev_name is mounted"
    fi
    log "Debug: Exit locate_id_disk"
}

function read_id_disk()
{
    log "Debug: Enter read_id_disk"
    locate_id_disk
    # check whether the id disk is an image preparation disk or id disk
    img_prep_signature=`cat ${id_disk_mnt_point}/PvsVm/CTXSOSID.INI | grep '\[ImagePreparation\]' | sed 's/\s//g'`
    if [ "$img_prep_signature" == "[ImagePreparation]" ]; then
        # extract the STATUS file name
        status_filename=`cat ${id_disk_mnt_point}/PvsVm/CTXSOSID.INI | grep 'StatusFile' | cut -d'=' -f2 | sed 's/\s//g'`

        # write status file to the root of ${id_disk_mnt_point}
        cat <<-EOF > ${id_disk_mnt_point}/${status_filename}
<?xml version="1.0"?>
<PreparationResults>
  <PreparationResults>
    <Status>Complete</Status>
    <CycleCount>1</CycleCount>
    <StepStatus>
      <OfficeRearm>Success</OfficeRearm>
      <EnableDHCP>Success</EnableDHCP>
      <PrepareMSMQ>Success</PrepareMSMQ>
      <OsRearm>Success</OsRearm>
      <PrepJoinDomain>Success</PrepJoinDomain>
    </StepStatus>
    <ThirdPartyStatus />
    <Log>
</Log>
  </PreparationResults>
</PreparationResults>
EOF
        # flush the image preparation status file to disk
        log "flushing the status file to id disk..."
        sync ${id_disk_mnt_point}/${status_filename}
        #shutdown image prep machine
        sleep 5
        log_echo "shutting down the machine..."
        shutdown -h now

        log "exiting ..."
        exit 0
    fi

    # here we first need to determine whether its NDJ or DJ mode
    Domain_Join_Section=`cat ${id_disk_mnt_point}${ad_info_file_path} | grep '\[DomainJoin\]' | sed 's/\s//g'`
    if [ "$Domain_Join_Section" == "[DomainJoin]" ]; then
        read_domain_joined_info
    else
        NON_DOMAIN_JOINED="Y"
        read_non_domain_joined_info
    fi

    log "Debug: Exit read_id_disk"
}

function conf_resolv_conf()
{
    log "Debug: Enter conf_resolv_conf"

    if [[ -z "$dns1" && -z "$dns2" && -z "$dns3" && -z "$dns4" ]]; then
      return
    fi

    dnsFile="/etc/resolv.conf"

    `sed -i '/^nameserver.*$/d' "$dnsFile"`
    [[ -n "$dns1" ]] && str1="nameserver $dns1" && echo "$str1">"$dnsFile"
    [[ -n "$dns2" ]] && str1="nameserver $dns2" && echo "$str1">>"$dnsFile"
    [[ -n "$dns3" ]] && str1="nameserver $dns3" && echo "$str1">>"$dnsFile"
    [[ -n "$dns4" ]] && str1="nameserver $dns4" && echo "$str1">>"$dnsFile"

    log "Debug: exit conf_resolv_conf"
}

function override_resolv_by_network_manager()
{
    local nmDir="/etc/NetworkManager/dispatcher.d"
    local customFile="/etc/resolv.conf.custom"
    [[ -n "$dns1" ]] && echo "nameserver $dns1">"$customFile"
    [[ -n "$dns2" ]] && echo "nameserver $dns2">>"$customFile"
    [[ -n "$dns3" ]] && echo "nameserver $dns3">>"$customFile"
    [[ -n "$dns4" ]] && echo "nameserver $dns4">>"$customFile"

    local resolvFile="${nmDir}/15-resolv"
    echo "#!/bin/bash
    #
    # Description : script to override default resolv.conf file
    # with customized file.
    cp -f /etc/resolv.conf.custom /etc/resolv.conf">"$resolvFile"
    chmod +x "$resolvFile"
}

function override_resolv_by_dhclient()
{
    local dhcpClientFile="/etc/dhcp/dhclient.conf"
    [[ ! -f "${dhcpClientFile}" ]] && log_echo "Warning: file $dhcpClientFile does not exist!" && log_echo "Debug: Exit function overrideResolvByDhclient" && return
    `sed -i '/^prepend.*domain-name-servers.*$/d' "$dhcpClientFile"`
    [[ -n "$dns1" ]] && echo "prepend domain-name-servers $dns1">>"$dhcpClientFile"
    [[ -n "$dns2" ]] && echo "prepend domain-name-servers $dns2">>"$dhcpClientFile"
    [[ -n "$dns3" ]] && echo "prepend domain-name-servers $dns3">>"$dhcpClientFile"
    [[ -n "$dns4" ]] && echo "prepend domain-name-servers $dns4">>"$dhcpClientFile"
}

function override_resolv_by_interface()
{
    local interfaceConfigFile="/etc/network/interfaces"
    `sed -i '/dns-nameservers.*$/d' "$interfaceConfigFile"`
    [[ -n "$dns1" ]] && echo "dns-nameservers $dns1">>"$interfaceConfigFile"
    [[ -n "$dns2" ]] && echo "dns-nameservers $dns2">>"$interfaceConfigFile"
    [[ -n "$dns3" ]] && echo "dns-nameservers $dns3">>"$interfaceConfigFile"
    [[ -n "$dns4" ]] && echo "dns-nameservers $dns4">>"$interfaceConfigFile"
}

function stop_DHCP_changing_resolv()
{
    local dhcpHookDir="/etc/dhcp/dhclient-enter-hooks.d/"
    if [[ ! -d "${dhcpHookDir}" ]]; then
        mkdir -p "${dhcpHookDir}"
    fi
    local dhcpHook="/etc/dhcp/dhclient-enter-hooks.d/nodnsupdate"
    echo "#!/bin/bash
make_resolv_conf(){
    :
}" > $dhcpHook
}

function install_lvda_centrify_dependency_pkgs()
{
    log "Debug: Enter install_lvda_centrify_dependency_pkgs"

    # downloadPkg is used to indicate if the package should be download from network or not:
    #   0: don't download
    #   1: download(default value)
    downloadPkg="1"
    # validate pkg path and files under pkg path
    if [[ -d "$centrifyPath" ]]; then
        countAdcheck=$(find "$centrifyPath" -name adcheck*x86_64 |wc -l 2>&1)
        countInstall=$(find "$centrifyPath" -name install.sh |wc -l 2>&1)
        if [[ "$countAdcheck" -eq "0" ]]; then
            log_echo "No adcheck*x86_64 file under $centrifyPath."
            downloadPkg="1"
        elif [[ "$countAdcheck" -gt "1" ]]; then
            log_echo "More than one adcheck*x86_64 file under $centrifyPath."
            downloadPkg="1"
        fi
        if [[ "$countInstall" -eq "0" ]]; then
            log_echo "No install.sh file under $centrifyPath."
            downloadPkg="1"
        elif [[ "$countInstall" -gt "1" ]]; then
            log_echo "More than one install.sh file under $centrifyPath."
            downloadPkg="1"
        fi
        if [[ "$countAdcheck" -eq "1" && "$countInstall" -eq "1" ]]; then
            downloadPkg="0"
            log_echo "Info: Centrify package under $centrifyPath will be used."
        fi
    else  # directory $centrifyPath does not exit
        log_echo "Info: Directory($centrifyPath) does not exist, try to download Centrify pacakge from network!"
        downloadPkg="1"
       fi

    if [[ "$downloadPkg" -eq "1" ]]; then
        [[ ! -d "$centrifyPath" ]] && mkdir -p "$centrifyPath"
        [[ -d "$centrifyPath" ]] && rm -rf "$centrifyPath/*"
        if [[ -n "$CENTRIFY_DOWNLOAD_PATH" ]]; then
            centrifyDownloadPath=$CENTRIFY_DOWNLOAD_PATH
        else
            log_echo "Error: Please set CENTRIFY_DOWNLOAD_PATH value first in mcs.conf"
            return
        fi

        log_echo "Downloading Linux VDA centrify package from $centrifyDownloadPath..."
        wget -nc --directory-prefix="$centrifyPath" "$centrifyDownloadPath" >> "$logFile" 2>&1
        info="Error: Failed to download centrify package from $centrifyDownloadPath"
        tar zxvf "$centrifyPath"/centrify* -C "$centrifyPath" >> "$logFile" 2>&1
        [[ "$?" -ne "0" ]] && log_echo "$info"
    fi

    "$centrifyPath"/install.sh --express >> "$logFile" 2>&1
    if [[ "$?" -eq "1" || "$?" -gt "25" ]]; then
        log_echo "Error: Failed to install centrify package"
    fi

    log "Debug: Exit install_lvda_centrify_dependency_pkgs"
}

function install_lvda_centrify_samba_integration_pkgs()
{
    log "Debug: Enter install_lvda_centrify_samba_integration_pkgs"

    # downloadPkg is used to indicate if the package should be download from network or not:
    #   0: don't download
    #   1: download(default value)
    downloadPkg="1"
    # validate pkg path and files under pkg path
    if [[ -d "$centrifySambaPath" ]]; then
        countAdbindproxy=$(find "$centrifySambaPath" -name CentrifyDC-adbindproxy*x86_64.rpm |wc -l 2>&1)
        if [[ "$countAdbindproxy" -eq "0" ]]; then
            log_echo "No centrifydc-adbindproxy*-x86_64.rpm file under $centrifySambaPath."
            downloadPkg="1"
        else
            downloadPkg="0"
            log_echo "Info: Centrify package under $centrifySambaPath will be used."
        fi
    else  # directory $centrifyPath does not exit
        log_echo "Info: Directory($centrifySambaPath) does not exist, try to download Centrify Samba pacakge from network!"
        downloadPkg="1"
    fi

    if [[ "$downloadPkg" -eq "1" ]]; then
        [[ ! -d "$centrifySambaPath" ]] && mkdir -p "$centrifySambaPath"
        [[ -d "$centrifySambaPath" ]] && rm -rf "$centrifySambaPath/*"
        if [[ -n "$CENTRIFY_SAMBA_DOWNLOAD_PATH" ]]; then
            centrifySambaDownloadPath=$CENTRIFY_SAMBA_DOWNLOAD_PATH
        else
            log_echo "Error: Please set CENTRIFY_SAMBA_DOWNLOAD_PATH value first in mcs.conf"
            return
        fi

        log_echo "Downloading Linux VDA centrify package from $centrifySambaDownloadPath..."
        wget -nc --directory-prefix="$centrifySambaPath" "$centrifySambaDownloadPath" >> "$logFile" 2>&1
        info="Error: Failed to download centrify samba package from $centrifySambaDownloadPath"
        tar xvzf $centrifySambaPath/centrify-* -C $centrifySambaPath >> "$logFile" 2>&1
        [[ "$?" -ne "0" ]] && log_echo "$info"
    fi

    rpm -i $centrifySambaPath/CentrifyDC-adbindproxy*x86_64.rpm >> "$logFile" 2>&1

    if [[ "$?" -ne "0" ]]; then
        log_echo "Error: Failed to install centrify samba package"
    else
        log_echo "Install centrify samba packages successfully."
    fi

    log "Debug: Exit install_lvda_centrify_samba_integration_pkgs"
}

# In some cloud environment, msDS-SupportedEncryptionTypes attribute is 0 which cause registration failure
# Set this attribute to 56 to check all AES encryption type
# This issue only occured when using sssd and winbind
function write_EncryptionTypes_attribute() 
{
    log "Debug: Enter write_EncryptionTypes_attribute"
    if ! net ads enctypes set "$NEW_HOSTNAME$" 56 -U "$NEW_HOSTNAME$" -P >> "$logFile" 2>&1;
    then
        log_echo "Error: Set msDS-SupportedEncryptionTypes attribute of machine account failed!"
    else
        log_echo "Set msDS-SupportedEncryptionTypes attribute of machine account succeed!"
    fi

    log "Debug: Exit write_EncryptionTypes_attribute"
}

function join_domain_samba()
{
    log "Debug: Enter join_domain_samba"

    # update secret.tdb
    log "Updating secret.tdb to remove the easyinstall info"
    tdbtool ${SAMBA_SECRETS} delete "SECRETS/MACHINE_DOMAIN_INFO/${WORKGROUP}"

    log "Updating secret.tdb to remove last change time info"
    tdbtool ${SAMBA_SECRETS} delete SECRETS/MACHINE_LAST_CHANGE_TIME/${DOMAIN}
    if [ "$UPDATE_MACHINE_PW" == "enabled" ]; then
        # add a dummy MACHINE_LAST_CHANGE_TIME to make net ads changetrustpw work for newer versions(4.7 or higher) of samba
        # Note that for newer versions of samba, if we set the value, machine password will be updated by samba.
        log "Adding SECRETS/MACHINE_LAST_CHANGE_TIME/${DOMAIN}"
        tdbtool "${SAMBA_SECRETS}" store SECRETS/MACHINE_LAST_CHANGE_TIME/"${DOMAIN}" "00" >> "$logFile" 2>&1
    fi

    # update domain sid
    log "Updating SECRET/SID/${DOMAIN} to ${DNSDISID}"
    net setdomainsid "${DNSDISID}" >> "$logFile" 2>&1


    log "Changing SECRETS/SALTING_PRINCIPAL/DES/${REALM} to host/${NEW_HOSTNAME}.${domain}@${DOMAIN}"
    tdbtool "${SAMBA_SECRETS}" store SECRETS/SALTING_PRINCIPAL/DES/"${REALM}" "host/${new_hostname}.${realm_lower}@${REALM}\0" >> "$logFile" 2>&1


    # update machine password
    log "Setting SECRETS/MACHINE_PASSWORD/${WORKGROUP}"
    tdbtool ${SAMBA_SECRETS} store SECRETS/MACHINE_PASSWORD/${WORKGROUP} "${PASSWORD}\0" > /dev/null 2>&1

    # write second channel type
    log "Changing SECRETS/MACHINE_SEC_CHANNEL_TYPE/${WORKGROUP} to \02\00\00\00"
    tdbtool ${SAMBA_SECRETS} store "SECRETS/MACHINE_SEC_CHANNEL_TYPE/${WORKGROUP}" "\02\00\00\00"

    net cache flush

    # update keytab file
    log_echo "Updating $KEYTAB"
    if [ -f $KEYTAB ]; then
        rm -f $KEYTAB
    fi

    for ((retry_num=1; retry_num<=$retry_max; retry_num++))
    do
        log_echo "Retry time: $retry_num, sleeping 2 seconds before creating keytab file..."
        sleep 2
        net ads keytab create -P >> "$logFile" 2>&1
        if [[ "$?" -ne "0" ]]; then
            log_echo "Error: Kerboros keytab creation failed!"
        else
            log_echo "Kerboros keytab creation succeed!"
            break
        fi
    done

    sleep 1
    # update dns record
    log "updating DNS record ..."
    net ads dns register -P >> "$logFile" 2>&1

    sleep 1

    #
    # Check if password has been changed, update it to identity disk
    #
    SAMBA_PWORD=$(tdbdump $SAMBA_SECRETS -k "SECRETS/MACHINE_PASSWORD/${WORKGROUP}" | \
        sed -n "s/\(^\S\+\)\\\00$/\1/p")
    if [ "${PASSWORD}" != "${SAMBA_PWORD}" ]; then
        log "Password has been changed, update it to identity disk"
        PWORD_BASE64=$(echo -n $SAMBA_PWORD | base64 | sed 's/\//\\\//g')
        change_id_disk_setting "MachinePassword" "$PWORD_BASE64"
    fi

    STATUS=$(net ads testjoin > /dev/null; echo $?)
    if [ $STATUS -eq 0 ]; then
        log_echo "Machine join passed testing"
    else
        log_echo "Error: Machine join failed testing"
    fi

    log "Debug: Exit join_domain_samba"
}

function install_lvda_pbis_dependency_pkgs()
{
    log "Debug: Enter install_lvda_pbis_dependency_pkgs"
    # Check if pbis is installed
    info="Check if pbis is installed ..."
    log_echo "$info"
    which "$pbis_pkgList" >> "$logFile" 2>&1
    if [[ "$?" -ne "0" ]]; then
        info="PBIS is not installed, install PBIS dependency libnsl"
        log_echo "$info"
        # downloadPkg is used to indicate if the package should be download from network or not:
        #   0: don't download
        #   1: download(default value)
        downloadPkg="1"
        if [[ -n "$pbisPath" ]]; then
            if [[ -d "$pbisPath" ]]; then
                countInstall="$(ls $pbisPath/*x86_64*.sh |wc -l 2>&1)"
                if [[ "$countInstall" -eq "0" ]]; then
                    info="Info: No install.sh file under $pbisPath."
                    log_echo "$info"
                    downloadPkg="1"
                elif [[ "$countInstall" -gt "1" ]]; then
                    info="Info: More than one install.sh file under $pbisPath."
                    log_echo "$info"
                    downloadPkg="1"
                fi
                if [[ "$countInstall" -eq "1" ]]; then
                    downloadPkg="0"
                    info="Info: Pbis package under $pbispkgpath will be used."
                    log_echo "$info"
                fi
            fi
        else  # directory $pbisPath does not exit
            info="Info: directory($pbisPath) does not exist, try to download Pbis pacakge from network!"
            log_echo "$info"
            downloadPkg="1"
        fi
        if [[ "$downloadPkg" -eq "1" ]]; then
            pbisDownloadPath=$PBIS_DOWNLOAD_PATH
            info="Info: pbisDownloadPath=$pbisDownloadPath"
            log_echo "$info"
            [[ ! -d "$pbisPath" ]] && mkdir -p "$pbisPath"
            [[ -d "$pbisPath" ]] && sudo rm -rf $pbisPath/*
            sudo wget -nc --directory-prefix="$pbisPath" "$pbisDownloadPath"  >>"$logFile"  2>&1
            if [[ "$?" -ne "0" ]]; then
                info="Error: Download PBIS failed"
                log_echo "$info"
            fi
            sudo chmod +x "$pbisPath"/*x86_64*.sh  >>"$logFile"  2>&1
            info="Error: Change file permission failed"
            [[ "$?" -ne "0" ]] && log_echo "$info"
        fi
        # install Pbis script
        sudo sh "$pbisPath"/*x86_64*.sh  >>"$logFile"  2>&1
        info="Error: Failed to install PBIS package"
        [[ "$?" -ne "0" ]] && log_echo "$info"
    fi
    # Configure PBIS before join domain in no reboot mode
    /usr/bin/domainjoin-cli configure --enable nsswitch >> "$logFile" 2>&1

    log "Debug: Exit install_lvda_pbis_dependency_pkgs"
}

function enablePbisUpdateDNS {
    local service_name="pbis-update-dns"
    local service_conf="/etc/systemd/system/${service_name}.service"

    [ -f "${service_conf}" ] || echo "# System unit that update this host's Active Directory DNS record on boot
[Unit]
Description=Update Active Directory DNS with current IP address
Requires=lwsmd.service
After=lwsmd.service

[Service]
ExecStart=/opt/pbis/bin/update-dns --show
ExecReload=/opt/pbis/bin/update-dns --show

[Install]
WantedBy=multi-user.target" > "$service_conf"
    systemctl enable "$service_name"
}

function join_domain_pbis()
{
    log "Debug: Enter join_domain_pbis"

    # Run domain join command
    for ((retry_num=1; retry_num<=$retry_max; retry_num++))
    do
        log_echo "Retry time: $retry_num, sleeping 2 seconds before join domain..."
        sleep 2
        if [ "$Use_Existing_Configurations_Of_Current_VDA" == "Y" ]; then
            log "Join domain with pbis command: domainjoin-cli join --disable krb5 ${realm} ${new_hostname}$ password"
	    if ! /opt/pbis/bin/domainjoin-cli join --disable krb5 "${realm}" "${new_hostname}\$" "$PASSWORD" >> "$logFile" 2>&1;
            then
                myPrint "Error: Join domain failed!"
            else
                #Update machine password in iddisk if join domain success
                myPrint "Join domain succeed!"
                myLog "Get new password"
                password_line=$(/opt/pbis/bin/lsa ad-get-machine password |grep Password: | sed 's/\s//g') # gitleaks:allow
                new_pbis_pword=${password_line#*:}
                new_pbis_pword_base64=$(echo -n "${new_pbis_pword}" | base64 | sed 's/\//\\\//g')
                myLog "Change machine password inside identity disk"
                change_id_disk_setting "MachinePassword" "$new_pbis_pword_base64"

                /opt/pbis/bin/update-dns --show
                enable_pbis_update_dns
                break
            fi
        else
            log "Join domain with pbis command: domainjoin-cli join ${realm} ${new_hostname}$ password"
	    if ! /opt/pbis/bin/domainjoin-cli join "${realm}" "${new_hostname}\$" "$PASSWORD" >> "$logFile" 2>&1
            then
                myPrint "Error: Join domain failed!"
            else
                #Update machine password in iddisk if join domain success
                myPrint "Join domain succeed!"
                myLog "Get new password"
                password_line=$(/opt/pbis/bin/lsa ad-get-machine password |grep Password: | sed 's/\s//g') # gitleaks:allow 
                new_pbis_pword=${password_line#*:}
                new_pbis_pword_base64=$(echo -n "${new_pbis_pword}" | base64 | sed 's/\//\\\//g')
                myLog "Change machine password inside identity disk"
                change_id_disk_setting "MachinePassword" "$new_pbis_pword_base64"

                /opt/pbis/bin/update-dns --show
                enable_pbis_update_dns
                break
            fi
        fi
    done

     log "Debug: Exit join_domain_pbis"
}

function conf_centrify_exist_domain()
{
    log "Debug: Enter conf_centrify_exist_domain"

    joinedAsName=$(adinfo -n)
    if [ -z $(echo "$joinedAsName" | grep -i "${new_hostname}") ]; then
        log_echo "Remove incorrect hosts under /etc/hosts"
        sed -i "/${joinedAsName}/Id" /etc/hosts

        log_echo "Machine joined name is incorrect, need to re-join"
        echo "${NEW_HOSTNAME}.${realm}" > /var/centrifydc/kset.host
        echo "${NEW_HOSTNAME}" > /var/centrifydc/kset.prew2k.host
    fi

    log "Debug: Exit conf_centrify_exist_domain"
}


function join_domain_centrify()
{
    local centrify_conf="/etc/centrifydc/centrifydc.conf"

    log "Debug: Enter join_domain_centrify"

    log_echo "Joining ${NEW_HOSTNAME} to ${REALM} domain"

    joinFlag="0"
    adinfo -d
    if [[ "$?" -eq 0 ]]; then
        runningMode=$(adinfo -m)
        if [ "$runningMode" != "connected" ]; then
            log_echo "Machine joined the domain with incorrect mode: $runningMode, need to update keytab"
            adkeytab -r -u "${NEW_HOSTNAME}\$" -p "${PASSWORD}" --verbose >> "$logFile" 2>&1
            sleep 5
            adleave --reset >> "$logFile" 2>&1
            adclient_pid=$(pidof /usr/sbin/adclient)
            if test "$adclient_pid"; then
                kill -9 "$adclient_pid"
            fi

            sed -i '/^adclient.dynamic.dns.enabled: */d' "$centrify_conf"
            echo 'adclient.dynamic.dns.enabled: true' >>"$centrify_conf"

            adjoin -w "${REALM}" -u "${NEW_HOSTNAME}\$" -p "${new_hostname}" --verbose >> "$logFile" 2>&1
            joinFlag="1"
        else
            log_echo "Machine is already joined to domain"
        fi
    else
        sed -i '/^adclient.dynamic.dns.enabled: */d' "$centrify_conf"
        echo 'adclient.dynamic.dns.enabled: true' >>"$centrify_conf"

        log_echo "Join domain with: adjoin -w "${REALM}" -u "${NEW_HOSTNAME}\$" -p PASSWORD -C --verbose"
        sleep 5
        adjoin -w "${REALM}" -u "${NEW_HOSTNAME}\$" -p "${PASSWORD}" --force --verbose >> "$logFile" 2>&1
        perl /usr/share/centrifydc/bin/adbindproxy.pl -n  >> "$logFile" 2>&1
        PWD=$(date +%s | md5sum | head -c 32)
        adkeytab -C -w "$PWD"  >> "$logFile" 2>&1
        centrify_validate_pwd_chg
        joinFlag="1"
    fi

    if [ "$joinFlag" -eq "1" ]; then
        adinfo -d
        if [[ "$?" -eq 0 ]]; then
            log_echo "Machine joined the domain successfully"
            PWORD=$(tdbdump $SAMBA_SECRETS -k "SECRETS/MACHINE_PASSWORD/${WORKGROUP}" | sed -n "s/\(^\S\+\)\\\00$/\1/p")
            PWORD_BASE64=$(echo "$PWORD" | base64 | sed 's/\//\\\//g')

            log "Change machine password inside identity disk"
            if [ -n "$PWORD_BASE64" ]; then
                change_id_disk_setting "MachinePassword" "$PWORD_BASE64"
            else
                log_echo "No SECRETS/MACHINE_PASSWORD under $SAMBA_SECRETS"
            fi
        else
            log_echo "Error: Machine joined the domain failed"
        fi
    fi

    log "Debug: Exit join_domain_centrify"
}

function centrify_validate_pwd_chg()
{
    log "Debug: Enter centrify_validate_pwd_chg"
    time=1
    for ((retry_num=1; retry_num<=$retry_max; retry_num++))
    do
        log_echo "Retry time: $retry_num, centrify validate password..."
        time=$[$time * 2]
        sleep "$time"
        PWORD=$(tdbdump $SAMBA_SECRETS -k "SECRETS/MACHINE_PASSWORD/${WORKGROUP}" | sed -n "s/\(^\S\+\)\\\00$/\1/p")
        if [[ -z "$PWORD" ||  "$PWORD" == "none" ]]; then
            PWD=$(date +%s | md5sum | head -c 32)
            adkeytab -C -w "$PWD" --verbose  >> "$logFile" 2>&1
            log_echo "Error: No password stored, Centrify try to change password!"
        else
            log_echo "Centrify validate password succeed!"
            break
        fi
    done

    log "Debug: Exit centrify_validate_pwd_chg"
}

function setup_vda()
{
    log "Debug: Enter setup_vda"

    # restore keydata.json file
    restore_keydata_json_file

    log_echo "updating VDA configurations ..."
    if [ "$AD_INTEGRATION" == "winbind" ]; then
        CTX_AD_INTEGRATION=1
    elif [ "$AD_INTEGRATION" == "centrify" ]; then
        CTX_AD_INTEGRATION=3
    elif [ "$AD_INTEGRATION" == "sssd" ]; then
        CTX_AD_INTEGRATION=4
    elif [ "$AD_INTEGRATION" == "pbis" ]; then
        CTX_AD_INTEGRATION=5
    fi
    export CTX_XDL_DOTNET_RUNTIME_PATH=${DOTNET_RUNTIME_PATH}
    export CTX_XDL_DESKTOP_ENVIRONMENT=${DESKTOP_ENVIRONMENT}
    export CTX_XDL_SUPPORT_DDC_AS_CNAME=${SUPPORT_DDC_AS_CNAME}
    export CTX_XDL_DDC_LIST=${DDCS}
    export CTX_XDL_VDA_PORT=${VDA_PORT}
    export CTX_XDL_REGISTER_SERVICE=${REGISTER_SERVICE}
    export CTX_XDL_ADD_FIREWALL_RULES=${ADD_FIREWALL_RULES}
    export CTX_XDL_AD_INTEGRATION=${CTX_AD_INTEGRATION}
    export CTX_XDL_HDX_3D_PRO=${HDX_3D_PRO}
    export CTX_XDL_VDI_MODE=${VDI_MODE}
    export CTX_XDL_SITE_NAME=${SITE_NAME}
    export CTX_XDL_LDAP_LIST=${LDAP_LIST}
    export CTX_XDL_SEARCH_BASE=${SEARCH_BASE}
    export CTX_XDL_FAS_LIST=${FAS_LIST}
    export CTX_XDL_START_SERVICE=${START_SERVICE}
    export CTX_XDL_TELEMETRY_SOCKET_PORT=${TELEMETRY_SOCKET_PORT}
    export CTX_XDL_TELEMETRY_PORT=${TELEMETRY_PORT}

    log_echo "CTX_XDL_DOTNET_RUNTIME_PATH=${CTX_XDL_DOTNET_RUNTIME_PATH}"
    log_echo "CTX_XDL_DESKTOP_ENVIRONMENT=${CTX_XDL_DESKTOP_ENVIRONMENT}"
    log_echo "CTX_XDL_SUPPORT_DDC_AS_CNAME=${CTX_XDL_SUPPORT_DDC_AS_CNAME}"
    log_echo "CTX_XDL_DDC_LIST=${CTX_XDL_DDC_LIST}"
    log_echo "CTX_XDL_VDA_PORT=${CTX_XDL_VDA_PORT}"
    log_echo "CTX_XDL_REGISTER_SERVICE=${CTX_XDL_REGISTER_SERVICE}"
    log_echo "CTX_XDL_ADD_FIREWALL_RULES=${CTX_XDL_ADD_FIREWALL_RULES}"
    log_echo "CTX_XDL_AD_INTEGRATION=${CTX_XDL_AD_INTEGRATION}"
    log_echo "CTX_XDL_HDX_3D_PRO=${CTX_XDL_HDX_3D_PRO}"
    log_echo "CTX_XDL_VDI_MODE=${CTX_XDL_VDI_MODE}"
    log_echo "CTX_XDL_SITE_NAME=${CTX_XDL_SITE_NAME}"
    log_echo "CTX_XDL_LDAP_LIST=${CTX_XDL_LDAP_LIST}"
    log_echo "CTX_XDL_SEARCH_BASE=${CTX_XDL_SEARCH_BASE}"
    log_echo "CTX_XDL_FAS_LIST=${CTX_XDL_FAS_LIST}"
    log_echo "CTX_XDL_START_SERVICE=${CTX_XDL_START_SERVICE}"
    log_echo "CTX_XDL_TELEMETRY_SOCKET_PORT=${CTX_XDL_TELEMETRY_SOCKET_PORT}"
    log_echo "CTX_XDL_TELEMETRY_PORT=${CTX_XDL_TELEMETRY_PORT}"

    for ((retry_num=1; retry_num<=$retry_max; retry_num++))
    do
        log_echo "Retry time: $retry_num, sleeping 2 seconds before running ctxsetup.sh..."
        sleep 2
        /opt/Citrix/VDA/sbin/ctxsetup.sh  >> "$logFile" 2>&1
        if [[ "$?" -ne "0" ]]; then
            log_echo "Error: Run ctxsetup.sh failed!"
        else
            log_echo "Run ctxsetup.sh succeed!"
            break
        fi
    done

    # configure local settings
    configure_local_settings

    log "Debug: Exit setup_vda"
}


function configure_local_settings()
{
    log "Debug: Enter configure_local_settings"
    /opt/Citrix/VDA/bin/ctxreg load -i /etc/xdl/mcs/mcs_local_setting.reg || return 1
    log "Debug: Exit configure_local_settings"
}

function create_localuser_group()
{
    log "Debug: Enter create_localuser_group"
    if [ ! $(getent group CitrixLocalUsersGroup) ]; then
        /usr/sbin/groupadd CitrixLocalUsersGroup || return 1
    fi
    log "Debug: Enter create_localuser_group"
}

function restart_service()
{
    log "Debug: Enter restart_service"

    log_echo "Restarting $1 service using $2..."
    "$2" restart "$1" >> "$logFile" 2>&1

    log "Debug: Exit restart_service"
}

function set_mcs_setting_flag()
{
    log "Debug: Enter set_mcs_setting_flag"

    touch ${mcs_setting_flag_file}

    log "Debug: Exit set_mcs_setting_flag"
}

#
# Check vda, read flag in registry to see if already configured
#
function vda_already_set()
{
    if [[ -f ${mcs_setting_flag_file} ]]; then
        log "MCS Setting Flag is set, skipping configuration"
        locate_id_disk
        exit 0
    fi
}

function check_user_and_vda_configured()
{
    check_user
    conf_resolv_conf
    vda_already_set
}

# this is called in non-domain joined mode
function non_domain_joined_setup()
{
    log "Debug: Enter non_domain_joined_setup"

    conf_hostname_non_domain_joined
    setup_vda

    # configure websocket settings
    write_non_domain_info_to_registry
    # restart hdx and vda service to reload websocket settings
    restart_service ctxhdx $1
    restart_service ctxvda $1

    log "Debug: Exit non_domain_joined_setup"
}

#
#check character set if it supports en_US.utf8
#
function check_locale()
{
    log "Debug: Enter function check_locale"
    local char_set="$(locale -a | grep "en_US.utf8" 2>&1)"
    if [[ -z "${char_set}" ]]; then
        log_echo "will add en_US.utf8 to character set"
        sed -i "s/^# en_US.UTF-8 UTF-8/en_US.UTF-8 UTF-8/g"  /etc/locale.gen
        locale-gen
    else
        log_echo "character set has included en_US.utf8"
    fi
    log "Debug: Exit function check_locale"
}
