﻿# Get-WindowsOptionalFeature is quite slow. Instead of constantly running it, we are using cache to store the current packages in memory. Cache can be invalidated by changing the value of CTXOEFeatures_IsCacheValid to false, in which case it will be automatically reloaded. 
# Cache is NOT inialized when module is loaded. This prevents issues in environments where all modules are loaded at launch (PowerShell v2) and this module is not supported. In such environments, module is initialized, but unsupported cmdlet is never executed.
[Array]$CTXOEFeatures_Cache = @();
$CTXOEFeatures_IsCacheValid = $False;

Function Get-CTXOEFeatures {
    If ($CTXOEFeatures_IsCacheValid -eq $false) {
        $CTXOEFeatures_Cache = Get-WindowsOptionalFeature -Online
    }

    Return $CTXOEFeatures_Cache
}
Function Test-CTXOEFeaturesState ([String]$Name, [String]$State) {
    [Hashtable]$Return = @{}
    $Return.Result = $False; 

    # First test if feature even exists on this machine
    $m_RequestedFeature = $(Get-CTXOEFeatures | Where-Object {$_.FeatureName -eq $Name});

    # If not found, abort execution
    If ($m_RequestedFeature -isnot [Object]) {
        $Return.Details = "Feature $($Name) was not found on this machine";
        $Return.NotFound = $true;
        Return $Return;
    }

    # Check the state of feature
    $Return.LastState = $m_RequestedFeature.State;
    $Return.Result = $m_RequestedFeature.State -eq $State;
    
    # Supported values for requested state is Enable/Disable. However, State has different variations of this (e.g. DisablePending).
    # We are checking if current state begins with requested state.
    If ($m_RequestedFeature.State -like "$State*") {
        $Return.Result = $True;
        $Return.Details = "Feature $Name is set to $State";
    } Else {
        $Return.Details = "Feature $Name should be in state $State, but is $($m_RequestedFeature.State)";
    }

    Return $Return
}

Function Invoke-CTXOEFeaturesExecuteInternal ([Xml.XmlElement]$Params, [Boolean]$RollbackSupported = $False) {
    [Hashtable]$m_Result = Test-CTXOEFeaturesState -Name $Params.Name -State $Params.Value

    # If feature is already configured, no action need to be taken.
    If ($m_Result.Result -eq $true) {
        $Global:CTXOE_Result = $m_Result.Result;
        $Global:CTXOE_Details = $m_Result.Details;
        Return;
    }

    # If feature was not found, return
    If ($m_Result.NotFound) {
        # If feature should be disabled AND does not exists, that's OK. But if other value was configured than DISABLED, it's not OK. 
        $Global:CTXOE_Result = $Params.Value -eq "Disabled"
        $Global:CTXOE_Details = "Feature $($Params.Name) was not found on this machine"
        Return
    }

    # Invalidate cache
    $CTXOEFeatures_IsCacheValid = $False;

    # Set feature to requested state
    Try {
        If ($Params.Value -eq "Disable") {
            Disable-WindowsOptionalFeature -FeatureName $Params.Name -Online -NoRestart | Out-Null;
        } ElseIf ($Params.Value -eq "Enable") {
            Enable-WindowsOptionalFeature -FeatureName $Params.Name -Online -NoRestart | Out-Null;
        } Else {
            Throw "Unsupported value $($Params.Value) requested for feature $($Params.Name)";
        }
    } Catch {
        $Global:CTXOE_Result = $False; 
        $Global:CTXOE_Details = "Failed to change feature $($Params.Name) to state $($Params.State) with following error: $($_.Exception.Message)"; 
        Return
    }

    # Same the last known startup type for rollback instructions
    $Global:CTXOE_SystemChanged = $true;
    If ($RollbackSupported) {
        [Xml.XmlDocument]$m_RollbackElement = CTXOE\ConvertTo-CTXOERollbackElement -Element $Params

        # Rollback has to be hardcoded. This is required to prevent states like "EnablePending" from being saved. The only two supported states should be "Enable" and "Disable"
        If ($m_Result.LastState -like "Enable*") {
            $m_RollbackElement.rollbackparams.value = "Enable";
        } ElseIf ($m_Result.LastState -like "Disable*") {
            $m_RollbackElement.rollbackparams.value = "Disable";
        }
        $Global:CTXOE_ChangeRollbackParams = $m_RollbackElement
    }

    [Hashtable]$m_Result = Test-CTXOEFeaturesState -Name $Params.Name -State $Params.Value
    $Global:CTXOE_Result = $m_Result.Result
    $Global:CTXOE_Details = $m_Result.Details
    

    Return
}

Function Invoke-CTXOEFeaturesAnalyze ([Xml.XmlElement]$Params) {
    [Hashtable]$m_Result = Test-CTXOEFeaturesState -Name $Params.Name -State $Params.Value

    $Global:CTXOE_Result = $m_Result.Result
    $Global:CTXOE_Details = $m_Result.Details

    Return

}

Function Invoke-CTXOEFeaturesExecute ([Xml.XmlElement]$Params) {
    Invoke-CTXOEFeaturesExecuteInternal -Params $Params -RollbackSupported $True
}

Function Invoke-CTXOEFeaturesRollback ([Xml.XmlElement]$Params) {
    Invoke-CTXOEFeaturesExecuteInternal -Params $Params -RollbackSupported $False
}