# This function is used to check if all required XmlElement are defined. Using the .Item() method, because .name would return the name of the parent if element is missing. 
Function Test-CTXOERegistryInput ([Xml.XmlElement]$Params) {
    # 'Path' is the only property that is  required for all registry operations
    If ($Params.Item("path") -isnot [Xml.XmlElement]) {Throw "Required property 'Path' is missing"}; 
    
    # Test if 'Path' contains unsupported ":" character (PSdrive). ":" is supported as part of key name, but not as part of root drive (e.g. 'HKLM:\Software'), so we cannot use simple .Contains() and have to parse the first token instead
    If ($Params.Path.Split("\")[0][-1] -eq ":") {Throw "Registry path includes unsupported ':' character in root key path"};

    # DeleteKey requires only path, but doesn't care about value name. We don't need to validate any other parameters and can exit
    If ($Params.Value -eq "CTXOE_DeleteKey") {Return;}

    # "Name" is just an optional parameter. If it is not specified, Optimizer will just create empty registry key. However if 'Name' is provided, we need to make sure that also 'ValueType' and 'Value' are provided
    If ($Params.Item("name") -is [Xml.XmlElement]) {
        If ($Params.Item("value") -isnot [Xml.XmlElement]) {Throw "Required property Value is missing"};
        If ($Params.Item("valuetype") -isnot [Xml.XmlElement] -and $Params.value -ne "CTXOE_DeleteValue") {Throw "Required property ValueType is missing"};
    } Else {
        If ($Params.Value -eq "CTXOE_DeleteValue") {Throw "Required property Name is missing"};
    }
    
}

Function Invoke-CTXOERegistryAnalyze ([Xml.XmlElement]$Params) {
    Test-CtxOEREgistryInput -Params $Params

    # When modifications are made to default user, we need to load the registry hive file first
    # Since there are few different places where code can be returned in Test-CTXOERegistryValue function, we are handling hive loading here

    [Boolean]$m_IsDefaultUserProfile = $Params.Path -like "HKDU\*" -or $Params.Path -like "HKEY_USERS\DefaultUser\*";

    If ($m_IsDefaultUserProfile) {
            # Mount DefaultUser
            Reg Load "HKU\DefaultUser" "C:\Users\Default\NTUSER.DAT" | Out-Null;
            New-PSDrive -Name DefaultUser -PSProvider Registry -Root HKEY_USERS\DefaultUser | Out-Null;
    
            # Replace HKDU\ with actual registry path
            $Params.Path = $Params.Path -Replace "HKDU\\", "HKEY_USERS\DefaultUser\";
    }

    # Using psbase.InnerText because it's the only way how to handle (optional) XML elements. If we use $Params.Name and 'name' doesn't exist, parent element (params) is used by function instead.
    [Hashtable]$m_Result = CTXOE\Test-CTXOERegistryValue -Key $Params.Path -Name $Params.Item("name").psbase.InnerText -Value $Params.Item("value").psbase.InnerText
    $Global:CTXOE_Result = $m_Result.Result
    $Global:CTXOE_Details = $m_Result.Details

    If ($m_IsDefaultUserProfile) {
        # Unmount DefaultUser
        Remove-PSDrive -Name DefaultUser;
        [GC]::Collect();
        Reg Unload "HKU\DefaultUser" | Out-Null;
    }
    
}

Function Invoke-CTXOERegistryExecuteInternal ([Xml.XmlElement]$Params, [Boolean]$RollbackSupported = $False) {
    Test-CtxOEREgistryInput -Params $Params

    # When modifications are made to default user, we need to load the registry hive file first
    # Since there are few different places where code can be returned in Test-CTXOERegistryValue function, we are handling hive loading here

    [Boolean]$m_IsDefaultUserProfile = $Params.Path -Like "HKDU\*" -or $Params.Path -Like "HKEY_USERS\DefaultUser\*";

    If ($m_IsDefaultUserProfile) {
            # Mount DefaultUser
            Reg Load "HKU\DefaultUser" "C:\Users\Default\NTUSER.DAT" | Out-Null;
            New-PSDrive -Name DefaultUser -PSProvider Registry -Root HKEY_USERS\DefaultUser | Out-Null;
    
            # Replace HKDU\ with actual registry path
            $Params.Path = $Params.Path -Replace "HKDU\\","HKEY_USERS\DefaultUser\";
    }

    # Test if value is already configured or not
    [Hashtable]$m_Result = CTXOE\Test-CTXOERegistryValue -Key $Params.Path -Name $Params.Item("name").psbase.InnerText -Value $Params.Item("value").psbase.InnerText
    
    # If the value is not configured, change it
    If ($m_Result.Result -ne $true) {
        [Hashtable]$m_Result = CTXOE\Set-CTXOERegistryValue -Key $Params.Path -ValueType $Params.Item("valuetype").psbase.InnerText -Value $Params.Item("value").psbase.InnerText -Name $Params.Item("name").psbase.InnerText
        $Global:CTXOE_SystemChanged = $true;
        
        If ($RollbackSupported) {
            [Xml.XmlDocument]$m_RollbackElement = CTXOE\ConvertTo-CTXOERollbackElement -Element $Params
            $m_RollbackElement.rollbackparams.value = $m_Result.OriginalValue.ToString();
            $Global:CTXOE_ChangeRollbackParams = $m_RollbackElement
        }
    }

    $Global:CTXOE_Result = $m_Result.Result
    $Global:CTXOE_Details = $m_Result.Details

    If ($m_IsDefaultUserProfile) {
        # Unmount DefaultUser
        Remove-PSDrive -Name DefaultUser;
        [GC]::Collect();
        Reg Unload "HKU\DefaultUser" | Out-Null;
    }
}

Function Invoke-CTXOERegistryExecute ([Xml.XmlElement]$Params) {
    Invoke-CTXOERegistryExecuteInternal -Params $Params -RollbackSupported $True
}

Function Invoke-CTXOERegistryRollback ([Xml.XmlElement]$Params) {
    Invoke-CTXOERegistryExecuteInternal -Params $Params -RollbackSupported $False
}