﻿If (-not $(Get-Module -ListAvailable -Name Appx)) {
    Write-Host "Universal Windows Applications are not available on this system" 
    Return
}

# Get-AppxProvisionedPackage is quite slow. Instead of constantly running it, we are using cache to store the current packages in memory. Cache can be invalidated by changing the value of CTXOEUWP_IsCacheValid to false, in which case it will be automatically reloaded. 
$CTXOEUWP_Cache = Get-AppxProvisionedPackage -online
$CTXOEUWP_IsCacheValid = $true;

# In Windows 10 build 1709, new argument has been added (-AllUsers). This check determines if this argument is supported by current operating system or not. 
$CTXOEUWP_AllUsersSupported = $(Get-Command Remove-AppxProvisionedPackage -ParameterName AllUsers -ErrorAction SilentlyContinue) -is [System.Management.Automation.CmdletInfo];
Write-Host "Removal of already provisioned applications supported: $CTXOEUWP_AllUsersSupported"

Function Get-CTXOEUWPApp {
    If ($CTXOEUWP_IsCacheValid -eq $false) {
        $CTXOEUWP_Cache = Get-AppxProvisionedPackage -online
    }

    Return $CTXOEUWP_Cache
}

Function Test-CTXOEUWPAppExist ([String]$Name) {
    Return $(Get-CTXOEUWPApp | Where-Object {$_.DisplayName -eq $Name}) -is [Microsoft.Dism.Commands.AppxPackageObject]
}

Function Invoke-CTXOEUWPExecute ([Xml.XmlElement]$Params) {
    [Boolean]$m_Exists = Test-CTXOEUWPAppExist -Name $Params.Name

    If (-not $m_Exists) {
        $Global:CTXOE_Result = $True
        $Global:CTXOE_Details = "UWP app is not installed"
        Return
    }

    $m_PackageFullName = Get-AppxProvisionedPackage -online | Where-Object {$_.DisplayName -eq $Params.Name} | Select-Object -ExpandProperty PackageName

    Try {
        If ($CTXOEUWP_AllUsersSupported) {
            Remove-AppxProvisionedPackage -PackageName $m_PackageFullName -Online -AllUsers | Out-Null
        } Else {
            Remove-AppxProvisionedPackage -PackageName $m_PackageFullName -Online | Out-Null
        }
    } Catch {
        $Global:CTXOE_Details = "Failed to remove UWP app with following error: $($_.Exception.Message)"; 
    }

    $CTXOEUWP_IsCacheValid = $false; 
    $Global:CTXOE_SystemChanged = $true;

    [Boolean]$m_Exists = Test-CTXOEUWPAppExist -Name $Params.Name

    $Global:CTXOE_Result = $m_Exists - 1

    If (-not $m_Exists) {
        $Global:CTXOE_Details = "UWP app has been removed"; 
    }

    Return
}

Function Invoke-CTXOEUWPAnalyze ([Xml.XmlElement]$Params) {
    [Boolean]$m_Exists = Test-CTXOEUWPAppExist -Name $Params.Name

    [Boolean]$Global:CTXOE_Result = $m_Exists - 1
    If ($m_Exists) {
        $Global:CTXOE_Details = "UWP app is installed"
    } Else {
        $Global:CTXOE_Details = "UWP app is not installed"
    }

    Return
}

Function Invoke-CTXOEUWPRollback ([Xml.XmlElement]$Params) {
    $Global:CTXOE_Result = $False
    $Global:CTXOE_Details = "UWP module does NOT support rollback. If you need to rollback the changes, re-install the application manually"
}