﻿<# 
.SYNOPSIS
    Citrix Profile Management App Access Control Config Tool Implementation

.DESCRIPTION 
    Citrix Profile Management App Access Control Config Tool is complex, so we encapsulated some functions in this script to make it easier to maintain and understand.
 
.NOTES 
    This script is part of the Citrix Profile Management Tools. It provides functions for cpm_app_access_control_config.ps1 to call, also support common function for WEM to call.

.COMPONENT 
    This script is part of the Citrix Profile Management Tools. It provides functions for cpm_app_access_control_config.ps1 to call, also support common function for WEM to call.

.LINK 
    It is released with UPM
 
.Parameter ParameterName 
    NA 
#>
# CPM_App_Access_Control_Impl.psm1
#region define vars
$Global:importModulesCnt = 0
$Global:targetGPO = ''
$Global:targetApp = ''
$Global:targetExecutable = ''
$Global:rulesReadyForRegGPO = ''
#existing raw rule data that exists
$Global:existingRuleList = New-Object -TypeName 'System.Collections.ArrayList'
#rurrent app's raw rule data
$Global:rulesList = New-Object -TypeName 'System.Collections.ArrayList'

#app info got by uninstall key: Name, InstallDir
$Global:installedAppsFoundInUnInstallKey = @{}

$Global:existingHiddenAppNameList = New-Object -TypeName 'System.Collections.ArrayList'

#whole apps records
$Global:exeInfoList = New-Object -TypeName 'System.Collections.ArrayList'
$Global:exeSidList = New-Object -TypeName 'System.Collections.ArrayList'
$Global:appList = New-Object -TypeName 'System.Collections.ArrayList'
$Global:manuallyAppList = New-Object -TypeName 'System.Collections.ArrayList'
$Global:RulesInMemoryAppList = New-Object -TypeName 'System.Collections.ArrayList'
 
$Global:assList = New-Object -TypeName 'System.Collections.ArrayList'
$Global:fileRegList = New-Object -TypeName 'System.Collections.ArrayList'
$Global:userList = New-Object -TypeName 'System.Collections.ArrayList'
$Global:ouList = New-Object -TypeName 'System.Collections.ArrayList'
$Global:computerList = New-Object -TypeName 'System.Collections.ArrayList'
$Global:processList = New-Object -TypeName 'System.Collections.ArrayList'
$Global:redirectRuleList = New-Object -TypeName 'System.Collections.ArrayList'
$Global:ShotcutList = New-Object -TypeName 'System.Collections.ArrayList'

$Global:seperatorReplacingSlash = '@CTXBARSEP@'    
# a list of known apps of winows which is not our targets
$Global:knownIgnoredAppsList = @('AddressBook', 'Connection Manager', 'DirectDrawEx', 'DXM_Runtime', 'Fontcore', 'IE40', 'IE4Data', 'IE5BAKEX', 'IEData', 'MPlayer2', 'SchedulingAgent', 'WIC'
    , 'MobileOptionPack', 'Citrix Profile Management', 'Citrix Workspace Environment Management Agent', 'Redirect Rules')

$Global:unisntallKeySearchScope = @('HKLM:\SOFTWARE\Microsoft\Windows\CurrentVersion\Uninstall',
    'HKLM:\SOFTWARE\WOW6432Node\Microsoft\Windows\CurrentVersion\Uninstall',
    'HKCU:\SOFTWARE\Microsoft\Windows\CurrentVersion\Uninstall'
)
$Global:ServiceORDriverInfoSearchScope = @('HKLM:\system\currentcontrolset\Services')


#the paths below, and their parent paths all protected, ignore the corner case that c is not system drive, anyway, in UPM side, we protect [driver]:\windows\system32 and their parent paths [driver]:,[driver]:\windows
$Global:importantFileFolderPath = @(
    'c:\windows\system32\',
    '%windir%\system32\'
)
#just do the match, ignore corner case that user could hide the higher level of path below
$Global:importantAppPath = @(
    '*\Citrix\User Profile Manager\*',
    '*\Citrix\Workspace Environment Management Agent\*',
    '*\Citrix\XenDesktopVdaSetup\*'
)

$Global:redirectTypeString = @(
    'Unknown',
    'File',
    'Folder',
    'RegKey',
    'RegValue'
)
enum TargetObjectType {
    File = "0"
    RegistryKey = "1"
    RegistryValue = "2"
    Folder = "3"
}

#endregion
function Check-CTXImportantRegPath {
    param(
        [Parameter(Mandatory = $true)][string]$testPath

    )
    $importantRegPath = @(
        #upm
        'HKEY_LOCAL_MACHINE\SOFTWARE\Policies\Citrix\UserProfileManager\',
        'HKLM:\SOFTWARE\Policies\Citrix\UserProfileManager\',
        'HKEY_LOCAL_MACHINE\SOFTWARE\Citrix\UserProfileManager\',
        'HKLM:\SOFTWARE\Citrix\UserProfileManager\',
        #wem
        'HKEY_LOCAL_MACHINE\SOFTWARE\Policies\Norskale\',
        'HKEY_LOCAL_MACHINE\SOFTWARE\Citrix\WEM\',
        'HKEY_LOCAL_MACHINE\SYSTEM\CurrentControlSet\Control\Norskale\',
        'HKLM:\SOFTWARE\Policies\Norskale\',
        'HKLM:\SOFTWARE\Citrix\WEM\',
        'HKLM:\SYSTEM\CurrentControlSet\Control\Norskale\',
        #VDA
        'HKEY_LOCAL_MACHINE\SOFTWARE\Citrix\VirtualDesktopAgent\',
        'HKLM:\SOFTWARE\Citrix\VirtualDesktopAgent\',
        'HKEY_LOCAL_MACHINE\SOFTWARE\Microsoft\Windows\CurrentVersion\Uninstall\Citrix Virtual Desktop Agent\'
        'HKLM:\SOFTWARE\Microsoft\Windows\CurrentVersion\Uninstall\Citrix Virtual Desktop Agent\'
        #others
        'HKCU:\',
        'HKEY_CURRENT_USER\',
        'HKU:\',
        'HKEY_USERS\'

    )
    $upperTestPath = $testPath.TrimEnd('\') + '\'
    $upperTestPath = $upperTestPath.ToUpper()
    foreach ($var in $importantRegPath) {
        if ($var.ToUpper().Contains($upperTestPath)) {
            write-host 'Protected path:' $testPath' skipped' -ForegroundColor Yellow
            return $true
        }
    }
    return $false
}
function Import-CTXModules {
    $Global:importModulesCnt++
    
    $moduleGPMAvailable = Get-Module *grouppolicy*
    $moduleADAvailable = Get-Module *activedirectory*
	# ThreadJob contains command Start-ThreadJob
	# However, some newer version of powershell contains commandlet Start-ThreadJob without ThreadJob
	# So whether to proceed installing threadjob depends on only the result of whether we have already get Start_ThreadJob
	$cmdletStartThreadAvailable = Get-Command "Start-ThreadJob" -ErrorAction SilentlyContinue
    $moduleThreadJobAvailable = Get-Module *threadjob*
	$moduleMSThreadJobAvailable = Get-Module *Microsoft.PowerShell.ThreadJob*
	
	#already imported
    if (($moduleGPMAvailable -ne $null) -and ($moduleADAvailable -ne $null) -and ($cmdletStartThreadAvailable -ne $null)) {
        return
    }
	
    if ($Global:importModulesCnt -eq 1) {
        write-host "`r`nPlease wait while PowerShell is importing the necessary Windows modules. " -ForegroundColor Yellow
    }
	
    #not imported but installed
    try {
		if($moduleGPMAvailable -eq $null) {
			Import-Module grouppolicy -ErrorAction Stop -WarningAction SilentlyContinue | Out-Null
			$moduleGPMAvailable = Get-Module *grouppolicy*
		}
    }
    catch {
        
    }
	
	try {
		if($moduleADAvailable -eq $null) {
			Import-Module ActiveDirectory -ErrorAction Stop -WarningAction SilentlyContinue | Out-Null
			$moduleADAvailable = Get-Module *activedirectory*
		}
    }
    catch {
        
    }
	
	try {
        if($cmdletStartThreadAvailable -eq $null) {
			Import-Module -Name Microsoft.PowerShell.ThreadJob -ErrorAction Stop -WarningAction SilentlyContinue -Force | Out-Null
			$cmdletStartThreadAvailable = Get-Command "Start-ThreadJob" -ErrorAction SilentlyContinue
		}
    }
    catch {
        
    }
	
	try {
        if($cmdletStartThreadAvailable -eq $null) {
			Import-Module -Name ThreadJob -ErrorAction Stop -WarningAction SilentlyContinue -Force | Out-Null
			$cmdletStartThreadAvailable = Get-Command "Start-ThreadJob" -ErrorAction SilentlyContinue
		}
    }
    catch {
        
    }
	
    #uninstalled
    #check OS version
    $curOS = ''
	$curOS = (Get-CimInstance -ClassName Win32_OperatingSystem).Caption
	if ($Global:importModulesCnt -eq 1) {
        write-host "`r`nPlease wait while PowerShell is installing the necessary Windows additional features. " -ForegroundColor Yellow
    }
	
    try {
        Install-PackageProvider -Name NuGet -MinimumVersion 2.8.5.201 -Force -Confirm:$false | Out-Null
		
	    try {
            if($cmdletStartThreadAvailable -eq $null) {
	    		Install-Module -Name Microsoft.PowerShell.ThreadJob -ErrorAction Stop -WarningAction SilentlyContinue -Force | Out-Null
				Import-Module Microsoft.PowerShell.ThreadJob -ErrorAction Stop -WarningAction SilentlyContinue  | Out-Null
	    		$cmdletStartThreadAvailable = Get-Command "Start-ThreadJob" -ErrorAction SilentlyContinue
	    	}
        }
        catch {
            
        }
		
	    try {
            if($cmdletStartThreadAvailable -eq $null) {
	    		Install-Module -Name ThreadJob -ErrorAction Stop -WarningAction SilentlyContinue -Force | Out-Null
				Import-Module ThreadJob -ErrorAction Stop -WarningAction SilentlyContinue  | Out-Null
	    		$cmdletStartThreadAvailable = Get-Command "Start-ThreadJob" -ErrorAction SilentlyContinue
	    	}
        }
        catch {
            
        }

		if($cmdletStartThreadAvailable -eq $null) {
			write-host 'Unable to use this tool because installing for the ThreadJob module failed.' -ForegroundColor Yellow
			write-host 'Please manually ensure this module is installed and try again.' -ForegroundColor Yellow
            Start-Sleep -Seconds 3
            Exit
		}

        if ($curOS.Contains('Server')) {
            Import-Module ServerManager
            if ($moduleGPMAvailable -eq $null) {
                Install-WindowsFeature GPMC -IncludeManagementTools -ErrorAction Stop -WarningAction SilentlyContinue | Out-Null            
            }
            
			if ($moduleADAvailable -eq $null) {
                Add-WindowsFeature RSAT-AD-PowerShell -ErrorAction Stop -WarningAction SilentlyContinue | Out-Null            
            }		      
 			
        }			
        elseif ($curOS.Contains('Windows 11')) {
            if ($moduleGPMAvailable -eq $null) {
                DISM.exe /Online /add-capability /CapabilityName:Rsat.GroupPolicy.Management.Tools~~~~0.0.1.0 | Out-Null	
            }
			
            if ($moduleADAvailable -eq $null) {
                Add-WindowsCapability -Online -Name Rsat.ActiveDirectory.DS-LDS.Tools~~~~0.0.1.0  -ErrorAction Stop -WarningAction SilentlyContinue | Out-Null	
            }
		
        }
        #win10
        else {		
            if ($moduleGPMAvailable -eq $null) {
                Add-WindowsCapability -Online -Name Rsat.GroupPolicy.Management.Tools~~~~0.0.1.0 -ErrorAction Stop -WarningAction SilentlyContinue  | Out-Null             
            }
            
			if ($moduleADAvailable -eq $null) {
                try {
                    Add-WindowsCapability -Online -Name Rsat.ActiveDirectory.DS-LDS.Tools~~~~0.0.1.0 -ErrorAction Stop -WarningAction SilentlyContinue  | Out-Null
                }
                catch {
                    #old versions up to 1803
                    Enable-WindowsOptionalFeature -Online -FeatureName RSATClient-Roles-AD-Powershell  -ErrorAction Stop -WarningAction SilentlyContinue | Out-Null
                } 
            }
      
        }

        Import-Module grouppolicy -ErrorAction Stop -WarningAction SilentlyContinue  | Out-Null
        Import-Module ActiveDirectory -ErrorAction Stop -WarningAction SilentlyContinue  | Out-Null
    }
    catch {
		$exMsg = $_.Exception.Message
		write-host "Unable to complete the module preparation due to the following error:" -ForegroundColor Yellow
		write-host "$exMsg" -ForegroundColor Red
		
        if ($Global:importModulesCnt -eq 1) {
            #we try re-run for admin
            return $false

        }
        else {
            if ($curOS.Contains('Server')) {
                write-host 'Unable to use this tool because imports for the following modules failed: ServerManager, Group Policy Managemnt Component, ThreadJob, or Active Directory.' -ForegroundColor Yellow		
            }
            else {
                write-host 'Unable to use this tool because imports for the following modules failed: Group Policy Managemnt Component, ThreadJob, or Active Directory.' -ForegroundColor Yellow		
            }

            write-host 'If it does not work, restart the machine, make sure the Windows update service is running, and then run this tool again.' -ForegroundColor Yellow
            Start-Sleep -Seconds 5
            Exit
        }

    }
	
    return $true

}

#clear some vars when turns back to the begining of the script
function Set-CTXGlobalVars {
    $Global:targetGPO = ''
    $Global:targetApp = ''
    $Global:targetExecutable = ''
    $Global:exeSidList.Clear()
    $Global:exeInfoList.Clear()
    $Global:installedAppsFoundInUnInstallKey.Clear()

    #whole app list
    $Global:existingHiddenAppNameList.Clear()
    $Global:appList.Clear()

    #current app's records if an app is chosen
    $Global:assList.Clear()
    $Global:fileRegList.Clear()
    $Global:userList.Clear()
    $Global:ouList.Clear()
    $Global:computerList.Clear()
    $Global:processList.Clear()
    $Global:rulesList.Clear()	
}
#region find app list
function Get-CTXInstallApps {
    param (
        [Parameter(Mandatory = $true)]
        [ref]$UninstalledKey	
			
    )
    Common-GetInstalledApps -unInstalledKeyList ([ref]$UninstalledKey.Value) 

    if ($Global:manuallyAppList.Count -ne 0) {

        $uniqueManList = $Global:manuallyAppList | select -Unique 
        $Global:manuallyAppList.Clear()
        if ($uniqueManList.Count -eq 1) {
            [void]$Global:manuallyAppList.Add($uniqueManList)
        }
        else {
            [void]$Global:manuallyAppList.AddRange($uniqueManList)
        }

        $curNames = $UninstalledKey.Value.Keys

        foreach ($var in $Global:manuallyAppList) {
            if (!($curNames -Contains ($var))) {
                [void]$UninstalledKey.Value.Add($var, '') 
            }
        }           
    }

    #HKLM\Software\Classes\CLSID\{sid} HKLM\Software\Wow6432Node\Class\CLSID\{sid},HKCU\Software\Classes\CLSID\{sid} 
    #HKLM\Software\Classes\CLSID\{sid}\InprocServer32\, Default value('(Default)') contains installation folder, eg, C:\Program Files\VideoLAN\VLC\xx
    #HKLM\Software\Microsoft\Windows\CurrentVersion\Explorer\Browser Helper Objects\{sid}
    #HKLM\Software\Wow6432Node\Microsoft\Windows\CurrentVersion\Explorer\Browser Helper Objects\{sid}
    #for performance, run this for only one time, store the sid together with the name for frther usage
    #time consuming to loop and check reg values on above paths to find out possible sids/guids on every suspicious reg values, so only query as below to balance performance, and for those missing ones, add specifically, in a word, we might miss some, but we need to balance performance
    
    Common-GetExeSidList


}

#endregion
function Test-CTXInteger {
    param(
        [string]$curInput
    )
    return $curInput -match '^[0-9]+$'

}
function Show-CTXCurrentFileRegs {
    write-host "`r`nFile and registry list:" -ForegroundColor Green
    if ($Global:fileRegList.Count -eq 0) {
        write-host 'No files and registry entries exist. With no files and registry entries configured, this app is visible to all users, machines, and processes.' -ForegroundColor yellow
    }
    else {
        $Global:fileRegList | Format-Table -wrap -autosize -property Index, Type, Path, Value
    }              
}
function Show-CTXCurrentAssignments {
    write-host "`r`nAssignment list:" -ForegroundColor Green
    if ($Global:assList.Count -eq 0) {
        write-host 'No assignments configured. This app is not visible to any users, computers, or processes.' -ForegroundColor Yellow
    }
    else {
        $Global:assList | Select-object Index, Type, SID, Name | Format-Table
    }
}

function Show-CTXRedirectRules {
    write-host "`r`nCurrent redirect rules:" -ForegroundColor Yellow

    if ($Global:redirectRuleList.Count -eq 0) {
        write-host "`r`n        (No rules at the moment)`r`n" -ForegroundColor Green
    }
    else {
        $Global:redirectRuleList | Format-Table -Wrap -Autosize -Property Index, @{Label = 'Type'; Expression = { $Global:redirectTypeString[$_.Type] } }, @{Label = 'Target'; Expression = { $_.Target -replace ($Global:seperatorReplacingSlash, '|') } }, TargetV, @{Label = 'Destine'; Expression = { $_.Destine -replace [regex]::Escape($Global:seperatorReplacingSlash), '|' } }, DestV, Users, Groups, Comps, OU, Procs | Out-String -Width 4096
    }
}

function Get-CTXRedirectRules {
    # Frist, print all rules
    Show-CTXRedirectRules

    write-host "[1]: Add new redirect rule`r`n[2]: Delete existing redirect rule`r`n[3]: Back to previous Menu`r`n" -ForegroundColor Green
    # then we give options for operation
    $_opt = ''
    while (($_opt -ne '1') -and ($_opt -ne '2') -and ($_opt -ne '3')) {
        $_opt = (Read-Host  "Enter your option to proceed").Trim()
    }

    if ($_opt -eq '1') {
        write-host "Please enter the redirect target type:`r`n[1]: File`r`n[2]: Folder`r`n[3]: Registry Key`r`n[4]: Registry Value`r`n" -ForegroundColor Green
        $_type = ''
        while (($_type -ne '1') -and ($_type -ne '2') -and ($_type -ne '3') -and ($_type -ne '4')) {
            $_type = (Read-Host  "Redirect type").Trim()
        }

        $_targetPath = (Read-Host  "Please enter the redirect source (file/folder/key) path").Trim()
        Replace-CTXUserInfoForOneRule -targetpath  ([ref]$_targetPath)
        if ($_type -eq '4') {
            $_targetVal = (Read-Host  "Please enter the redirect source key value name").Trim()
        }

        if ($_type -eq '4') {
            $_destPath = $_targetPath
        }
        else {
            $_destPath = (Read-Host  "Please enter the redirect destine (file/folder/key) path").Trim()
            Replace-CTXUserInfoForOneRule -targetpath  ([ref]$_destPath)
        }
        
        if ($_type -eq '4') {
            $_destVal = (Read-Host  "Please enter the redirect destine key value name").Trim()
        }

        write-host "`r`nNow consider the assignment for this redirect rule..." -ForegroundColor Green
        write-host "=======================================================`r`n" -ForegroundColor Green

        write-host "Assignment for processes" -ForegroundColor Green
        write-host "------------------------" -ForegroundColor Green
        write-host "Enter comma separated process list for this rule to be effective:`r`n`    - To add exempted process, prefix it with '?'(a single question mark)." -ForegroundColor Green
        $_procs = (Read-Host  "Process list (* for all, default for all)").Trim()
        if ([string]::IsNullOrWhiteSpace($_procs)) {
            $_procs = "*"
        }
		
        write-host "`r`nAssignment for Users" -ForegroundColor Green
        write-host "---------------------------" -ForegroundColor Green
        write-host "Enter comma separated user list (SID for AD, OID for NDJ) for this rule to be effective:`r`n`    - To add exempted user, prefix it with 'EX>'." -ForegroundColor Green
        $_users = (Read-Host  "User list (* for all, default for all)").Trim()
        if ([string]::IsNullOrWhiteSpace($_users)) {
            $_users = "*"
        }
		
        $_users = $_users -replace [regex]::Escape('EX>'), "!";
        write-host "`r`nAssignment for Groups" -ForegroundColor Green
        write-host "---------------------------" -ForegroundColor Green
        write-host "Enter comma separated group list (SID for AD, OID for NDJ) for this rule to be effective:`r`n`    - To add exempted group, prefix it with 'EX>'." -ForegroundColor Green
        $_groups = (Read-Host  "Group list (* for all, default for all)").Trim()
        if ([string]::IsNullOrWhiteSpace($_groups)) {
            $_groups = "*"
        }
		
        $_groups = $_groups -replace [regex]::Escape('EX>'), "!";		
        write-host "`r`nAssignment for Computers" -ForegroundColor Green
        write-host "------------------------" -ForegroundColor Green
        write-host "Enter comma separated Computer list (regular expression can be used for NDJ machine name) for this rule to be effective."  -ForegroundColor Green
		write-host "    - To add exempted computer (AD only), prefix it with 'EX>'." -ForegroundColor Green
		write-host "    - Enter None to designate no computers (when only using computer groups)'." -ForegroundColor Green
        $_comps = (Read-Host  "Computer list (* for all, default for all)").Trim()
        if ([string]::IsNullOrWhiteSpace($_comps)) {
            $_comps = "*"
        }
		
        write-host "`r`nAssignment for Computer Groups" -ForegroundColor Green
        write-host "------------------------------" -ForegroundColor Green
        write-host "Enter comma separated OU (for AD) or Machine Catelog (Ndj) for this rule to be effective:" -ForegroundColor Green
        write-host "    - To add exempted computer group, prefix it with 'EX>'."  -ForegroundColor Green
		write-host "    - Enter 'None' to designate no computer groups (when only using computer)'." -ForegroundColor Green
        $_ou = (Read-Host  "OU/Machine Catelog list (* for all, AD or NDJ; default for all)").Trim()
        if ([string]::IsNullOrWhiteSpace($_ou)) {
            $_ou = "*"
        }
		
        # format the rule somehow, and add it to the list
        $_tmpRule = [PsCustomObject]@{'Index' = $Global:redirectRuleList.Count + 1; 'Type' = $_type; 'Target' = $_targetPath; 'TargetV' = $_targetVal; 'DestV' = $_destVal; 'Destine' = $_destPath; 'Users' = $_users; 'Groups' = $_groups; 'Comps' = $_comps; 'OU' = $_ou; 'Procs' = $_procs }
        $Global:redirectRuleList.Add($_tmpRule) | Out-Null

        $_tmpOutStr = $Global:redirectTypeString[$_type]
        write-host "`r`nNew $_tmpOutStr redirect rule has been added!" -ForegroundColor Green
    }
    elseif ($_opt -eq '2') {
        write-host "`r`nPlease enter the index of the rule to be deleted." -ForegroundColor Green
        $_idx = ''
        while ($true) {
            while (-not ($_idx -match '^\d+$')) {
                $_idx = (Read-Host  "Enter rule index to delete").Trim()
            }
        
            if (-not (($_idx -gt 0) -and ($_idx -lt $Global:redirectRuleList.Count + 1))) {
                write-host "Invalid input index number!`r`n" -ForegroundColor Red
                $_idx = ''
            }
            else {
                break
            }
        }

        $Global:redirectRuleList.RemoveAt($_idx - 1)
        write-host "Redirect rule at $_idx removed!" -ForegroundColor Green
        $index = 1
        foreach ($rule in $Global:redirectRuleList) {
            $rule.Index = $index
            $index++
        }
    }
    else {
        # ($_opt -eq '3')
        Get-CTXTargetApp
        return
    }

    $curCommand = "`r`nHow do you want to continue:`r`n[1] Continue processing other rules`r`n[2] Generate the rules for deployment to machines"
    $choiceList = @('1', '2')

    write-host $curCommand -ForegroundColor Green

    $tmpchoice = ''
    while (($tmpchoice -ne '1') -and ($tmpchoice -ne '2')) {
        $tmpchoice = (Read-Host  'Enter your choice').Trim()
    }

    if ($tmpchoice -eq '1') {
        Get-CTXRedirectRules
    }
    else {
        # must be '2'
        Format-CTXAppAccessControlRules -saveChangesAndReturnToBegining $false
    }
}

function Remove-CTXConfiguredAppRules {

    $_idx = ''
    while ($true) {
        while (-not ($_idx -match '^\d+$')) {
            $_idx = (Read-Host  "`r`nEnter app index to remove rules").Trim()
        }
        
        if (-not (($_idx -ge -0) -and ($_idx -lt $Global:appList.Count + 1))) {
            $tmpVal = $Global:appList.Count;
            write-host "Invalid input index number!`r`nPlease input a number between 0 and $tmpVal !" -ForegroundColor Red
            $_idx = ''
        }
        else {
            break
        }
    }


    if ($_idx -eq 0) {
        if ($Global:redirectRuleList.Count -eq 0) {
            return
        }

        Show-CTXRedirectRules

        write-host "All the above redirect rules will be removed. Please confirm to proceed!" -ForegroundColor Yellow
        $cfm = (Read-Host  "Enter 'Y' to proceed to delete all redirect rules").Trim()
        if ($cfm -eq 'Y') {
            $Global:redirectRuleList.Clear()

            Write-Host "`r`nRules for selected app have been deleted." -ForegroundColor green
        }
    }
    else {
        $_app = $Global:appList | Where-Object Index -EQ $_idx
        if ($Global:RulesInMemoryAppList -contains $_app.Name) {
            $Global:RulesInMemoryAppList.Remove($_app.Name)
        }

        for ($i = 0; $i -lt $Global:existingRuleList.Count; $i++) {
            if ($Global:existingRuleList[$i].Split('|')[-1] -eq $_app.Name) {
                $Global:existingRuleList.RemoveAt($i)
                $i--
            }
        }

        Write-Host "`r`nRules for selected app have been deleted." -ForegroundColor green
    }
}

function Get-CTXTargetApp {
    
    Write-Host "`r`n`r`n"
    $configuredRules = $Global:appList | Where-Object Status -ne 'Not configured  '
    $choiceList = @('D', 'V', 'R', 'A')
    if (($configuredRules -eq $null) -and ($Global:redirectRuleList.Count -eq 0)) {
        $choiceList = @('D', 'R', 'A')
        Write-Host "`r`n[D] Delete all configured rules`r`n[R] Remove one configured apps" -ForegroundColor green          
    }
    else {
        Write-Host "`r`n[D] Delete all configured rules`r`n[R] Remove one configured apps`r`n[V] View existing rules for all apps" -ForegroundColor green 
    }

    Write-Host "[A] Add an app to the list`r`n------------------------------------------------------------`r`nTo add rules for an app from the list above, enter its index.`r`n" -ForegroundColor green 
    $appChosenID = (Read-Host  'Enter input').Trim() 
    
    while (($appChosenID -eq $null) -or ($appChosenID -eq '') -or (((Test-CTXInteger -curInput $appChosenID) -eq $false) -and (!($choiceList -Contains $appChosenID))) -or (((Test-CTXInteger -curInput $appChosenID) -eq $true) -and (([int]$appChosenID -gt $Global:appList.Count) -or ([int]$appChosenID -lt 0)))) {
        Write-Host 'Invalid input. Enter the input correctly.' -ForegroundColor yellow
        $appChosenID = (Read-Host  'Enter input').Trim()
    }

    if ($appChosenID -eq 'D') {
        #generate rules
        Format-CTXAppAccessControlRules -saveChangesAndReturnToBegining $false -deleteAll $true
        return
    }
    elseif ($appChosenID -eq 'R') {
        Remove-CTXConfiguredAppRules
        Get-CTXTargetApp
    }
    elseif ($appChosenID -eq 'A') {
        #allow admin to manaully enter an app name
        $tmp = (Read-Host  'Enter the app name. If you do not want to add it now, press <Enter>').Trim()

        while (($tmp -eq $null) -or (($Global:appList | Select-Object Name).Name -contains $tmp) -or (($tmp -ne '') -and ((Test-CTXIsGuid -StringGuid $tmp) -or !(Test-CTXAppName -path $tmp) -or (($Global:knownIgnoredAppsList -Contains $tmp) -or ($tmp -like 'Citrix Virtual Apps and Desktops*'))))) {
            if (($tmp -eq $null) -or !(Test-CTXAppName -path $tmp)) {
                Write-Host 'Invalid input.' -ForegroundColor yellow
                $tmp = (Read-Host  'Enter the app name. If you do not want to add it now, press <Enter>').Trim()
            }
            elseif (Test-CTXIsGuid -StringGuid $tmp) {
                Write-Host 'Invalid input. Please enter an app name instead of a GUID.' -ForegroundColor yellow
                $tmp = (Read-Host  'Enter the app name. If you do not want to add it now, press <Enter>').Trim()
            }
            elseif (($Global:knownIgnoredAppsList -Contains $tmp) -or ($tmp -like 'Citrix Virtual Apps and Desktops*')) {
                Write-Host 'Protected app.' -ForegroundColor yellow
                $tmp = (Read-Host  'Enter the app name. If you do not want to add it now, press <Enter>').Trim()
            }
            else {
                Write-Host 'This app name is already on the app list.' -ForegroundColor yellow
                Get-CTXTargetApp
            }

            
            
        }
        if ($tmp -ne '') {
            $anotherApp = [PsCustomObject]@{'Index' = $Global:appList.Count + 1; 'Status' = 'Not configured  '; 'Name' = $tmp; 'InstallDir' = '' }
            $Global:appList.Add($anotherApp) | Out-Null
            $Global:manuallyAppList.Add($tmp) | Out-Null
            Show-CTXWholeAppList
        }

        Get-CTXTargetApp
    }
    elseif ($appChosenID -eq 'V') {
        write-host "`r`n`r`n************************************************************" -ForegroundColor Yellow
        

        #display redirect rules if any
        if ($Global:redirectRuleList.Count -gt 0) {
            Show-CTXRedirectRules

            write-host "************************************************************" -ForegroundColor Yellow
        }

        if ($configuredRules -ne $null) {
            write-host "`r`nAll Apps details:" -ForegroundColor Yellow

            #display ruls for all apps
            $Global:targetApp = ''
            $Global:targetExecutable = ''
            $allApps = ($Global:appList | Select-Object Name).Name
            foreach ($var in $allApps) {           
                Get-CTXAppRelatedRules -app $var
                if (($Global:fileRegList.Count -eq 0) -and ($Global:assList.Count -eq 0)) {
                    #skip
                }
                else {
                    write-host "`r`n`r`n************************************************************" -ForegroundColor Yellow
                    write-host "`r`nApp:"$var -ForegroundColor Green
                    Show-CTXCurrentFileRegs
                    Show-CTXCurrentAssignments                
                } 
                    
            }
            write-host "`r`n************************************************************`r`n`r`n" -ForegroundColor Yellow
        }

        #generate rules
        $configuredRules = $Global:appList | Where-Object { $_.Status -eq 'Configured  ' -or $_.Status -eq 'Configured & applied  ' }
        if (($configuredRules -ne $null) -or ($Global:redirectRuleList.Count -gt 0)) {
            Format-CTXAppAccessControlRules -saveChangesAndReturnToBegining $false         
        }
        else {
            write-host "`r`n[Y] Add more rules`r`n[N] Exit`r`n" -ForegroundColor Green
            $continueGeneratingRules = (Read-Host  'Enter input').Trim() 
            while (($continueGeneratingRules -ne 'Y') -and ($continueGeneratingRules -ne 'N')) {
                write-host 'Invalid input:'$continueGeneratingRules -ForegroundColor yellow
                $continueGeneratingRules = (Read-Host  'Enter input').Trim() 
            }
            # here we need another option to generate rules

            if ($continueGeneratingRules -eq 'Y') {
                Get-CTXAppAccessControlRules
            }
            else {
                Exit
            }
        }
       
        return
    }
    else {

        if ($appChosenID -eq '0') {
            # here we are to deal with all redirect rules
            # need to return to base selection afterwards
            Get-CTXRedirectRules
            return
        }
        else {
            $appChosenRecord = $Global:appList | Where-Object Index -EQ $appChosenID
            $Global:targetApp = $appChosenRecord.Name
            $Global:targetExecutable = ''
        }
    }

    Get-CTXAppRelatedRules -app $Global:targetApp

    #if app is a hidden app
    if ( ($appChosenRecord.Status -eq 'Configured & applied  ') -or ($appChosenRecord.Status -eq 'Configured  ')) {        
        #nothing to do
    }
    #app is not hidden, but there are remaining rules for it in $Global:existingRuleList which is set into $Global:fileRegList in above Get-CTXAppRelatedRules
    elseif ($Global:fileRegList.Count -gt 0) {
        #nothing to do
    }

    #if app is not hide now
    else {
        # collect the install dir
        $appChosenInstallPath = $appChosenRecord.InstallDir              

        if ($appChosenInstallPath -ne '') {  
            $appChosenInstallPath = $appChosenInstallPath.Trim().Trim('"').Trim().TrimEnd('\')      
        }
        else {
            Write-Host 'Installation folder not found. Enter the full installation path. If you do not want to add it now, press <Enter>' -ForegroundColor Yellow
            $appChosenInstallPath = (Read-Host  'Enter path').Trim().Trim('"').Trim().TrimEnd('\')
            while (($appChosenInstallPath -ne '') -and (((Test-CTXAppPath -path $appChosenInstallPath) -ne $true) -or (Check-CTXImportantFileFolderPath -testPath $appChosenInstallPath))) {
                if (($appChosenInstallPath -ne '') -and ((Test-CTXAppPath -path $appChosenInstallPath) -ne $true)) {
                    Write-Host 'Invalid path:'$appChosenInstallPath'. Make sure that you enter the path correctly, or press the Enter key directly if you do not want to add path here.' -ForegroundColor Yellow 
                }
                else {
                    Write-Host 'Make sure that you enter the path correctly, or press the Enter key directly if you do not want to add path here.' -ForegroundColor Yellow 

                }
                $appChosenInstallPath = (Read-Host  'Enter path').Trim().Trim('"').Trim().TrimEnd('\')
            }
        }
        <# we cannot correct this beasue in many places, workspace use such incorrect install dir, we have to follow it to collect other app related paths
        if ($Global:targetApp -like 'Citrix Workspace *') {
            $subFolders = Get-ChildItem -Path $appChosenInstallPath -Directory
            foreach ($subFolder in $subFolders) {
                if ($subFolder.Name -like 'Citrix Workspace *') {
                $appChosenInstallPath = $appChosenInstallPath + '\' + $subFolder.Name
                }
            }
        }
        #>
        #for some app like google drive, the install dir is incorrect as C:\Program Files\Google\Drive File Stream\89.0.2.0\GoogleDriveFS.exe, need to get its parent folder
        $res = Common-GetAppRelatedPath -appName $Global:targetApp -appChosenInstallPath ([ref]$appChosenInstallPath) -targetExecutable  ([ref]$Global:targetExecutable) -exeInfoList $Global:exeInfoList -exeSidList $Global:exeSidList
		
	
        #job status are also here, filter only our targets list
        $res = @($res | Where-Object { $_.PSObject.Properties.Name -contains 'Index' })
		
        for ($i = 0; $i -lt $res.Count; $i++) {
            $tmp = Replace-CTXUserInfoForOneRule -targetpath  ([ref](($res[$i]).Path))
			($res[$i]).Path = $tmp
        }		

        $Global:fileRegList.AddRange($res)
    }
    Get-CTXUserInteractions
}




#retrieve from both reg and the previous config after this tool started
function Get-CTXAppRelatedRules {
    param(
        [string]$app
    )
    $assignProcessed = $false
    $Global:fileRegList.Clear()
    $Global:assList.Clear()
    foreach ($curRule in $Global:existingRuleList) {
        $bSkip = $false       
        $suspiciousRule = $curRule.Split('|')

        if (($suspiciousRule.Count -eq 9) -and ($suspiciousRule[0] -eq '0') -and ($suspiciousRule[8] -eq $app)) {

            
            switch ($suspiciousRule[1]) {

                '0' {
                    $tmpTrimFilePath = $suspiciousRule[2].Trim().Trim('"').Trim().TrimEnd('\')
                    if (Check-CTXImportantFileFolderPath -testPath $tmpTrimFilePath) {
                        $bSkip = $true
                        break                       
                    }
                    $Global:fileRegList.Add([PsCustomObject]@{'Index' = $Global:fileRegList.Count + 1; 'Type' = 'File'; 'Path' = $tmpTrimFilePath; 'Value' = '' }) | Out-Null
                       
                }
                '1' {
                    if (Check-CTXImportantRegPath -testPath $suspiciousRule[2].Trim().TrimEnd('\')) {
                        $bSkip = $true
                        break 
                    }
                    $Global:fileRegList.Add([PsCustomObject]@{'Index' = $Global:fileRegList.Count + 1; 'Type' = 'Registry key'; 'Path' = $suspiciousRule[2].Trim().TrimEnd('\'); 'Value' = '' }) | Out-Null


                }
                '2' {
                    if (Check-CTXImportantRegPath -testPath ($suspiciousRule[2].Trim().TrimEnd('\') + '\' + $suspiciousRule[3].Trim().TrimEnd('\'))) {
                        $bSkip = $true
                        break 
                    }
                    $Global:fileRegList.Add([PsCustomObject]@{'Index' = $Global:fileRegList.Count + 1; 'Type' = 'Registry Value'; 'Path' = $suspiciousRule[2].Trim().TrimEnd('\') ; 'Value' = $suspiciousRule[3].Trim().TrimEnd('\') }) | Out-Null
                       
                }
                '3' {
                    $tmpTrimFolderPath = $suspiciousRule[2].Trim().Trim('"').Trim().TrimEnd('\')
                    if (Check-CTXImportantFileFolderPath -testPath $tmpTrimFolderPath) {
                        $bSkip = $true
                        break 
                    }
                    $Global:fileRegList.Add([PsCustomObject]@{'Index' = $Global:fileRegList.Count + 1; 'Type' = 'Folder'; 'Path' = $tmpTrimFolderPath; 'Value' = '' }) | Out-Null


                }
            }
            if ($bSkip) {
                continue
            }
            if ($assignProcessed -eq $false) {
                $bIsADAllComputers = $false
                $bIsNDJAllComputers = $false

                for ($i = 4; $i -lt 8; $i++) {
                    if ($suspiciousRule[$i] -eq '*') {
                   
                    }
                    else {

                        $tmpList = $suspiciousRule[$i].split(':')
                        $tmpCount = $tmpList.Count
                        foreach ($var in  $tmpList) { 
                            if (($var -eq '*') -and ($tmpCount -gt 1)) {
                                #we could see * in two situations, 1) one is only * in a field, means all should be hidden. 2) The other one is, not only * in an assignment field, this * means it is appended automatically for pure whitelist(pure whitelist means only has whitelist settings for this type of assignments), do not display it to users
                                continue
                            }					
                            $eachAss = $var -split '@CTXASSSEP@'
                            #AD*, NDJ* 
                            if (($eachAss[1] -eq 'AD*') -and ($eachAss[0] -eq 'Computer')) {
                                $bIsADAllComputers = $true
                                continue
                            }
                            elseif (($eachAss[1] -eq 'NDJ*') -and ($eachAss[0] -eq 'Computer')) {
                                $bIsNDJAllComputers = $true
                                continue
                            }

                            #user /group for NDJ                                                                                           
                            if ((($eachAss[0] -EQ 'User') -or ($eachAss[0] -EQ 'Group')) -and ($eachAss[1] -like '/*' )) {
                                $val = [PsCustomObject]@{'Index' = $Global:assList.Count + 1; 'Type' = $eachAss[0]; 'SID' = $eachAss[1]; 'Name' = $eachAss[2]; 'Scope' = 'NDJ' }
                                $Global:assList.Add($val) | Out-NUll
                                continue
                            } 
                            #others
                            $type = $eachAss[0]                    
                            if ($eachAss[1] -EQ 'NDJ') {
                                #machine or machine catalog for NDJ
                                #<action type>|<target type>|<target path>| <target value>| <user and group list>|<process list>|<OU list>|<computer list>|<app name>
                                $_tmpName = $eachAss[2]
                                if ($i -eq 6 -and $eachAss[0] -eq "OUX") {
                                    $_tmpName = [regex]::Escape('EX>') + $eachAss[2]
                                }
                                if ($i -eq 6) {
                                    $type = 'Machine catalog'
                                }
                                elseif ($i -eq 7) {
                                    $type = 'AAD/NDJ machine'
                                }
                                $val = [PsCustomObject]@{'Index' = $Global:assList.Count + 1; 'Type' = $type; 'SID' = ''; 'Name' = $_tmpName; 'Scope' = 'NDJ' }
                                $Global:assList.Add($val) | Out-NUll                            
                            }

                            else {

                                if ($i -eq 6) {
                                    $type = 'OU'
                                }
                                elseif ($i -eq 7) {
                                    $type = 'AD machine'
                                }

                                $_tmpName = $eachAss[2]
                                if ($i -eq 7 -and $eachAss[0] -eq "computerEx") {
                                    $_tmpName = [regex]::Escape('EX>') + $eachAss[2]
                                }
                                if ($i -eq 6 -and $eachAss[0] -eq "OUX") {
                                    $_tmpName = [regex]::Escape('EX>') + $eachAss[2]
                                }
                                $val = [PsCustomObject]@{'Index' = $Global:assList.Count + 1; 'Type' = $type; 'SID' = $eachAss[1]; 'Name' = $_tmpName }
                                $Global:assList.Add($val) | Out-NUll                         
                            }                                                

                        }  
                        
                                              
                    }
                }
                if ($bIsADAllComputers -and $bIsNDJAllComputers) {
                    #it equals to *
                    for ($i = $Global:assList.Count - 1; $i -ge 0; $i--) {
                        if ((($Global:assList[$i]).Type -eq 'AD machine') -and !(($Global:assList[$i]).Name.StartsWith([regex]::Escape('EX>')))) {
                            $Global:assList.RemoveAt($i)
                        }
                    }
                    for ($i = $Global:assList.Count - 1; $i -ge 0; $i--) {
                        if (($Global:assList[$i]).Type -eq 'AAD/NDJ machine') {
                            $Global:assList.RemoveAt($i)
                        }
                    }
                }
                elseif ($bIsADAllComputers) {
                    # delete assignments to other AD computers
                    
                    for ($i = $Global:assList.Count - 1; $i -ge 0; $i--) {
                        if ((($Global:assList[$i]).Type -eq 'AD machine') -and !(($Global:assList[$i]).Name.StartsWith([regex]::Escape('EX>')))) {
                            $Global:assList.RemoveAt($i)
                        }
                    }
                    $val = [PsCustomObject]@{'Index' = $Global:assList.Count + 1; 'Type' = 'AD machine'; 'SID' = ''; 'Name' = 'AD*' }
                    $Global:assList.Add($val) | Out-NUll

                }
                elseif ($bIsNDJAllComputers) {
                    for ($i = $Global:assList.Count - 1; $i -ge 0; $i--) {
                        if (($Global:assList[$i]).Type -eq 'AAD/NDJ machine') {
                            $Global:assList.RemoveAt($i)
                        }
                    }
                    $val = [PsCustomObject]@{'Index' = $Global:assList.Count + 1; 'Type' = 'AAD/NDJ machine'; 'SID' = ''; 'Name' = 'NDJ*'; 'Scope' = 'NDJ' }
                    $Global:assList.Add($val) | Out-NUll

                }


                $assignProcessed = $true  
            }
            
        }        
    }
}




         



function Remove-CTXMultpleItems {
    param(
        [Parameter(Mandatory = $true)][string]$deleteData,
        [Parameter(Mandatory = $true)][System.Collections.ArrayList]$sourceList
    )

    $invalidInputIndex = New-Object -TypeName 'System.Collections.ArrayList'
    $dupInputIndex = New-Object -TypeName 'System.Collections.ArrayList'

    try { 
        $tmp = $deleteData.Split(',')
        $IDs = New-Object -TypeName 'System.Collections.ArrayList'
        [void]$IDs.AddRange($tmp)
        for ($i = 0; $i -lt $IDs.Count; $i++) {
            if (((Test-CTXInteger -curInput $IDs[$i]) -eq $false) -or (([int]$IDs[$i]) -lt 1) -or (([int]$IDs[$i]) -gt $sourceList.Count)) { 
                [void]$invalidInputIndex.Add($IDs[$i])                 
                $IDs.RemoveAt($i--)                                                    
            }
        }
        if ($IDs.Count -gt 0) {	
            $uniqueIDs = $IDs | select -Unique

            $cmp = Compare-object -referenceobject $uniqueIDs -differenceobject $IDs
	
            if ($cmp.InputObject.Count -gt 0) {	
                [void]$dupInputIndex.AddRange($cmp.InputObject)                          
            } 
          						
                        
            for ($i = 0; $i -lt $sourceList.Count; $i++) {
                if ($uniqueIDs -Contains [INT](($sourceList[$i]).Index)) {
                    $sourceList.RemoveAt($i--)		                            
                }
            }
        
            for ($i = 0; $i -lt $sourceList.Count; $i++) {
                $sourceList[$i].Index = $i + 1
            }
        }

	
        if ($invalidInputIndex.Count -gt 0) {
            $res = ''
            foreach ($invalidID in $invalidInputIndex) {
                $res = $res + $invalidID + ','
            }
            write-host 'Invalid items skipped:'$res.ToString().TrimEnd(',') -ForegroundColor yellow	
                            
        }

        if ($dupInputIndex.Count -gt 0) {
            $res = ''
            foreach ($invalidID in $dupInputIndex) {
                $res = $res + $invalidID + ','
            }
            write-host 'Duplicated items skipped:'$res.ToString().TrimEnd(',') -ForegroundColor yellow	
                            
        }  					
                        			
    }
    catch {
        write-host 'Invalid value: '$deleteData	-ForegroundColor yellow			
    }

}
function Get-CTXAssignments {
    Show-CTXCurrentAssignments


    $curCommand = ''
    $choiceList = @('1', '2', '3', '4', '5', '6', '7', '8', '9', '10', '11', '12', '13')
    if ($Global:assList.Count -eq 0) {
        $curCommand = "`r`nDo you want to add an assignment for this app?`r`n`r`n[1] Discard the changes you made to the app and continue adding rules for other apps`r`n[2] Save your changes and continue adding rules for other apps`r`n[3] Edit files and registries`r`n[4] Generate the rules for deployment to machines`r`n    If no assignments are configured, this app is not visible`r`n------------------------------------------------------------`r`n[5] Add users`r`n[6] Add user groups`r`n[7] Add OUs `r`n[8] Add AAD/NDJ machine catalogs`r`n    AAD: Azure AD; NDJ: Non-Domain-Joined`r`n[9] Add AD machines`r`n[10] Add AAD/NDJ machines`r`n[11] Add processes`r`n" 
        $choiceList = @('1', '2', '3', '4', '5', '6', '7', '8', '9', '10', '11')
    }
    else {
        $curCommand = "`r`nDo you want to add an assignment for this app? Or want to delete one? `r`n`r`n[1] Discard the changes you made to the app and continue adding rules for other apps`r`n[2] Save your changes and continue adding rules for other apps`r`n[3] Edit files and resgistries`r`n[4] Generate the rules for deployment to machines`r`n    If no assignments are configured, this app is not visible`r`n------------------------------------------------------------`r`n[5] Add users`r`n[6] Add user groups`r`n[7] Add OUs `r`n[8] Add AAD/NDJ machine catalogs`r`n    AAD: Azure AD; NDJ: Non-Domain-Joined`r`n[9] Add AD machines`r`n[10] Add AAD/NDJ machines`r`n[11] Add processes`r`n[12] Delete specific assignments`r`n[13] Delete all assignments`r`n"      
    }
    write-host $curCommand -ForegroundColor Green

    $tmpchoice = (Read-Host  'Enter value').Trim()


    
    while ($true) {     
        if (($tmpchoice -eq $null) -or ($tmpchoice -eq '') -or ((Test-CTXInteger -curInput $tmpchoice) -ne $true) -or (($choiceList -Contains $tmpchoice) -ne $true)) {      
            $max = $choiceList.Count    
            write-host 'Invalid value. Enter a value of 1 -'$max "`r`n" -ForegroundColor yellow
            $tmpchoice = (Read-Host  'Enter value').Trim()
            continue
        }
        else {                       
            switch ([int]$tmpchoice) {
                '1' {
                    #return to the begining
                    Get-CTXAppAccessControlRules
                    return
                }
                '2' {
                    #generate rules and add to $Global:existingRuleList: raw data
                    if (!($Global:RulesInMemoryAppList -contains $Global:targetApp)) {
                        $Global:RulesInMemoryAppList.Add($Global:targetApp) | Out-Null
                    } 
                    Format-CTXAppAccessControlRules -saveChangesAndReturnToBegining $true
                    #return to the begining
                    Get-CTXAppAccessControlRules
                    return
                }
                '3' {
                    if (!($Global:RulesInMemoryAppList -contains $Global:targetApp)) {
                        $Global:RulesInMemoryAppList.Add($Global:targetApp) | Out-Null
                    } 
                    Get-CTXUserInteractions
                    return			 
                }
                '4' {
                    #go to generate rules
                    if (!($Global:RulesInMemoryAppList -contains $Global:targetApp)) {
                        $Global:RulesInMemoryAppList.Add($Global:targetApp) | Out-Null
                    } 
                    Format-CTXAppAccessControlRules -saveChangesAndReturnToBegining $false
                    return
                }
                '5' {  
                    $command = "Enter users you want to add, separated by pipe.`r`n`    - For AD users, enter <Domain name>\<User name1>|<Domain name>\<User name2\>.`r`n`    - For AAD/NDJ users, use the AAD/NDJ object selector in the WEM web console to collect their names and OIDs, and then enter <OID>\<User name1>|<OID>\<User name2>. `r`n`    - To add exempted user, prefix it with 'EX>'.`r`n`    Example: /azuread/989c2938-6527-4133-bab1-f3860dd15098\TestUser1|/azuread/82bdde32-d5d9-4d64-b0ff-9876d4488d05\TestUser2. `r`n`    For more information, see this WEM article. https://docs.citrix.com/en-us/workspace-environment-management/service/manage/configuration-sets/citrix-profile-management.html#app-access-control."
                    write-host $command -ForegroundColor Green
                    [string]$newData = (Read-Host  'Enter users').Trim()
               
                    $tmpDataList = $newData.split('|')
                    foreach ($t in $tmpDataList) {
                        $tmpData = $t.ToString().Trim()
                        if ($tmpData -eq '') {
                            continue
                        }
                        if ($tmpData -eq '*') {
                            $Global:assList.Add([PsCustomObject]@{'Index' = $Global:assList.Count + 1; 'Type' = 'User'; 'SID' = '*'; 'Name' = '*' }) | Out-Null
                            continue
                        }
                        try {
                            if ($tmpData.StartsWith([regex]::Escape('EX>'))) {
                                $_NameToTrans = $tmpData.Substring(3)
                                $tmpInput = $_NameToTrans.split('\')
                                if ($_NameToTrans -like '/*') {
										
                                    $Global:assList.Add([PsCustomObject]@{'Index' = $Global:assList.Count + 1; 'Type' = 'User'; 'SID' = "!" + $tmpInput[0]; 'Name' = $tmpInput[1]; 'Scope' = 'NDJ' }) | Out-Null
                                }
                                else {
										
                                    $curusersid = (New-Object System.Security.Principal.NTAccount($_NameToTrans)).Translate([System.Security.Principal.SecurityIdentifier]).value
                                    $Global:assList.Add([PsCustomObject]@{'Index' = $Global:assList.Count + 1; 'Type' = 'User'; 'SID' = "!" + $curusersid; 'Name' = $tmpData }) | Out-Null									
                                }
																	
                            }
                            else {
                                $_NameToTrans = $tmpData
                                $tmpInput = $_NameToTrans.split('\')
                                if ($_NameToTrans -like '/*') {
                                    $Global:assList.Add([PsCustomObject]@{'Index' = $Global:assList.Count + 1; 'Type' = 'User'; 'SID' = $tmpInput[0]; 'Name' = $tmpInput[1]; 'Scope' = 'NDJ' }) | Out-Null
                                }
                                else {
										
                                    $curusersid = (New-Object System.Security.Principal.NTAccount($tmpData)).Translate([System.Security.Principal.SecurityIdentifier]).value
                                    $Global:assList.Add([PsCustomObject]@{'Index' = $Global:assList.Count + 1; 'Type' = 'User'; 'SID' = $curusersid; 'Name' = $tmpData }) | Out-Null								
                                }
                            }
															

                        }
                        catch {
                            write-host 'Invalid user item skipped:' $tmpData -ForegroundColor Yellow
                        }                    
                    }
                    
                    break
                      
                }
                '6' {  
                    $command = "Enter user groups you want to add, separated by pipe.`r`n`    - For AD groups, enter `<Domain name`>`\`<Group name 1`>`|`<Domain name`>`\`<Group name 2`>.`r`n`    - For AAD/NDJ groups, use the AAD/NDJ object selector in the WEM web console to collect their names and OIDs, and then enter `<OID`>`\`<Group name1`>`|`<OID`>`\`<Group name2`>. `r`n`    - To add exempted group, prefix it with 'EX>'.`r`n`    Example: /azuread/989c2938-6527-4133-bab1-f3860dd15098\TestGroup1|/azuread/82bdde32-d5d9-4d64-b0ff-9876d4488d05\TestGroup2. `r`n`    For more information, see this WEM article, https://docs.citrix.com/en-us/workspace-environment-management/service/manage/configuration-sets/citrix-profile-management.html#app-access-control."
                    write-host $command -ForegroundColor Green
                    [string]$newData = (Read-Host  'Enter groups').Trim()
                    #we only support normal path/assignments that themselves does not contains '|'
                    $tmpDataList = $newData.split('|')
                    foreach ($t in $tmpDataList) {
                        $tmpData = $t.ToString().Trim()
                        if ($tmpData -eq '') {
                            continue
                        }
                        if ($tmpData -eq '*') {
                            $Global:assList.Add([PsCustomObject]@{'Index' = $Global:assList.Count + 1; 'Type' = 'Group'; 'SID' = '*'; 'Name' = '*' }) | Out-Null
                            continue
                        }
                        try {
                            if ($tmpData.StartsWith([regex]::Escape('EX>'))) {
                                $_NameToTrans = $tmpData.Substring(3)
                                $tmpInput = $_NameToTrans.split('\')
                                if ($_NameToTrans -like '/*') {
										
                                    $Global:assList.Add([PsCustomObject]@{'Index' = $Global:assList.Count + 1; 'Type' = 'Group'; 'SID' = "!" + $tmpInput[0]; 'Name' = $tmpInput[1]; 'Scope' = 'NDJ' }) | Out-Null
                                }
                                else {
										
                                    $curusersid = (New-Object System.Security.Principal.NTAccount($_NameToTrans)).Translate([System.Security.Principal.SecurityIdentifier]).value
                                    $Global:assList.Add([PsCustomObject]@{'Index' = $Global:assList.Count + 1; 'Type' = 'Group'; 'SID' = "!" + $curusersid; 'Name' = $tmpData }) | Out-Null									
                                }
																	
                            }
                            else {
                                $_NameToTrans = $tmpData
                                $tmpInput = $_NameToTrans.split('\')
                                if ($_NameToTrans -like '/*') {
                                    $Global:assList.Add([PsCustomObject]@{'Index' = $Global:assList.Count + 1; 'Type' = 'Group'; 'SID' = $tmpInput[0]; 'Name' = $tmpInput[1]; 'Scope' = 'NDJ' }) | Out-Null
                                }
                                else {
										
                                    $curusersid = (New-Object System.Security.Principal.NTAccount($tmpData)).Translate([System.Security.Principal.SecurityIdentifier]).value
                                    $Global:assList.Add([PsCustomObject]@{'Index' = $Global:assList.Count + 1; 'Type' = 'Group'; 'SID' = $curusersid; 'Name' = $tmpData }) | Out-Null								
                                }
                            }
															

                        }
                        catch {
                            write-host 'Invalid group item skipped:' $tmpData -ForegroundColor Yellow
                        }                      
                    }
                    
                    break
                      
                }				
                '7' {
                    $command = "Enter OUs you want to add, separated by pipe.`r`n    - To add exempted OU, prefix it with 'EX>'.`r`n    - To designate no OU, enter None.`r`n`    Example: <OU name1>|<OU name2>`r`n"
                    write-host $command -ForegroundColor Green
                    [string]$newData = (Read-Host  'Enter OUs').Trim()
                    $tmpDataList = $newData.split('|')
                    foreach ($t in $tmpDataList) {
                        $tmpData = $t.ToString().Trim()
                        if ($tmpData -eq '') {
                            continue
                        }
                  

                        $Global:assList.Add([PsCustomObject]@{'Index' = $Global:assList.Count + 1; 'Type' = 'OU'; 'SID' = ''; 'Name' = $tmpData }) | Out-Null
                    }
                    
                    break
                }
                '8' {
                    $command = "Enter AAD/NDJ machine catalogs you want to add, separated by pipe.`r`n    - To add exempted machine catalog, prefix it with 'EX>'.`r`n`    - To designate no machine catelogs, enter None.`r`n`    Example: <Machine catalog name 1>|<Machine catalog name 2>`r`n"
                    write-host $command -ForegroundColor Green
                    [string]$newData = (Read-Host  'Enter machine catalogs').Trim()
                    $tmpDataList = $newData.split('|')
                    foreach ($t in $tmpDataList) {
                        $tmpData = $t.ToString().Trim()
                        if ($tmpData -eq '') {
                            continue
                        }
                  

                        $Global:assList.Add([PsCustomObject]@{'Index' = $Global:assList.Count + 1; 'Type' = 'Machine catalog'; 'SID' = ''; 'Name' = $tmpData; 'Scope' = 'NDJ' }) | Out-Null
                    }
                    
                    break
                }
                '9' {
                    $command = "Enter AD machines you want to add, separated by pipe.`r`n`    - Example: <FQDN of machine 1>|<FQDN of machine 2>`r`n`    - To add all AD machines, enter AD*.`r`n    - To add exempted machine, prefix it with 'EX>'.`r`n`    - To add no machines (assign with OU), enter None."
                    write-host $command -ForegroundColor Green
                    [string]$newData = (Read-Host  'Enter computers').Trim()
                
                    $tmpDataList = $newData.split('|')

                    foreach ($t in $tmpDataList) { 
                        $tmpData = $t.ToString().Trim()
                        if ($tmpData -eq '') {
                            continue
                        }
						
						if ($tmpData -eq 'None') {
							$Global:assList.Add([PsCustomObject]@{'Index' = $Global:assList.Count + 1; 'Type' = 'AD machine'; 'SID' = ''; 'Name' = 'None' }) | Out-Null
                            continue
                        }
						
                        if ($tmpData -eq '*') {
                            $Global:assList.Add([PsCustomObject]@{'Index' = $Global:assList.Count + 1; 'Type' = 'AD machine'; 'SID' = '*'; 'Name' = '*' }) | Out-Null
                            continue
                        }
                        try {  
                            if ($tmpData -EQ 'AD*') {
                                $Global:assList.Add([PsCustomObject]@{'Index' = $Global:assList.Count + 1; 'Type' = 'AD machine'; 'SID' = ''; 'Name' = $tmpData }) | Out-Null
                            }
                            else {
                                if ($tmpData.StartsWith([regex]::Escape('EX>'))) {
                                    $tmpData = $tmpData.Substring(3)
                                    $queryString = '(DNSHostName=' + $tmpData + ')'
                                    $pcObj = Get-ADComputer -LDAPFilter $queryString
                                    if ($pcObj -eq $null) {
                                        write-host 'Invalid computer name skipped:' $tmpData  -ForegroundColor Yellow
                                    }
                                    else {
                                        $Global:assList.Add([PsCustomObject]@{'Index' = $Global:assList.Count + 1; 'Type' = 'AD machine'; 'SID' = $pcObj.SID; 'Name' = "EX>$tmpData" }) | Out-Null
                                    }
                                }
                                else {
                                    $queryString = '(DNSHostName=' + $tmpData + ')'
                                    $pcObj = Get-ADComputer -LDAPFilter $queryString
                                    if ($pcObj -eq $null) {
                                        write-host 'Invalid computer name skipped:' $tmpData  -ForegroundColor Yellow
                                    }
                                    else {
                                        $Global:assList.Add([PsCustomObject]@{'Index' = $Global:assList.Count + 1; 'Type' = 'AD machine'; 'SID' = $pcObj.SID; 'Name' = $tmpData }) | Out-Null
                                    }
                                }

                            }

                                                 
                        }
                        catch {
                            write-host 'Invalid computer name skipped:' $tmpData  -ForegroundColor Yellow
                        }  

                    }
                    
                    break          
                }
                '10' {
                    $command = "Enter AAD/NDJ machines you want to add, separated by pipe.`r`n`    - Use the AAD/NDJ object selector in the WEM web console to collect AAD/NDJ machine names. For more information, see this WEM article, https://docs.citrix.com/en-us/workspace-environment-management/service/manage/configuration-sets/citrix-profile-management.html#app-access-control.`r`n`    - Regular expression is supported for NDJ machine names.`r`n`    - To add all AAD/NDJ machines, enter NDJ*.    - To add No AAD/NDJ machines, enter None."
                    write-host $command -ForegroundColor Green
                    [string]$newData = (Read-Host  'Enter computers').Trim()
                
                    $tmpDataList = $newData.split('|')

                    foreach ($t in $tmpDataList) { 
                        $tmpData = $t.ToString().Trim()
                        if ($tmpData -eq '') {
                            continue
                        }

                        $Global:assList.Add([PsCustomObject]@{'Index' = $Global:assList.Count + 1; 'Type' = 'AAD/NDJ machine'; 'SID' = ''; 'Name' = $tmpData; 'Scope' = 'NDJ' }) | Out-Null

                    }
                    
                    break          
                }
                '11' {
                    write-host "Enter the names of the processes you want to add. Separate them with pipe. `r`n`    - To add exempted process, prefix it with '?'(a single question mark).`r`n`    Example: powershell.exe|cmd.exe." -ForegroundColor Green
                    [string]$newData = (Read-Host  'Enter processes').Trim()
                
                    $tmpDataList = $newData.split('|')

                    foreach ($t in $tmpDataList) {
                        $tmpData = $t.ToString().Trim()
                        if ($tmpData -eq '') {
                            continue
                        }
                   
                        $Global:assList.Add([PsCustomObject]@{'Index' = $Global:assList.Count + 1; 'Type' = 'Process'; 'SID' = ''; 'Name' = $tmpData }) | Out-Null
                    }
                    
                    break              
                }
                '12' {

                    write-host 'To delete items, enter their indexes. Separate them with commas.' -ForegroundColor Green
                    [string]$deleteData = (Read-Host  'Enter index').Trim()
                    if ($deleteData -eq '') {
                        write-host 'Invalid item:' $deleteData -ForegroundColor Yellow
                    }
                    else {
                        Remove-CTXMultpleItems -deleteData $deleteData -sourceList $Global:assList
                    }
                    break
                        
                }			
                '13' {
                    $Global:assList.Clear()
                    break
                }

            }
            Show-CTXCurrentAssignments
            if ($Global:assList.Count -eq 0) {
                $curCommand = "`r`nDo you want to add an assignment for this app?`r`n`r`n[1] Discard the changes you made to the app and continue adding rules for other apps`r`n[2] Save your changes and continue adding rules for other apps`r`n[3] Edit files and resgistries`r`n[4] Generate the rules for deployment to machines`r`n    If no assignments are configured, this app is not visible`r`n------------------------------------------------------------`r`n[5] Add users`r`n[6] Add user groups`r`n[7] Add OUs `r`n[8] Add AAD/NDJ machine catalogs`r`n    AAD: Azure AD; NDJ: Non-Domain-Joined`r`n[9] Add AD machines`r`n[10] Add AAD/NDJ machines`r`n[11] Add processes`r`n"                
                $choiceList = @('1', '2', '3', '4', '5', '6', '7', '8', '9', '10', '11')
            }
            else {
                $curCommand = "`r`nDo you want to add an assignment for this app? Or want to delete one? `r`n`r`n[1] Discard the changes you made to the app and continue adding rules for other apps`r`n[2] Save your changes and continue adding rules for other apps`r`n[3] Edit files and resgistries`r`n[4] Generate the rules for deployment to machines`r`n    If no assignments are configured, this app is not visible`r`n------------------------------------------------------------`r`n[5] Add users`r`n[6] Add user groups`r`n[7] Add OUs `r`n[8] Add AAD/NDJ machine catalogs`r`n    AAD: Azure AD; NDJ: Non-Domain-Joined`r`n[9] Add AD machines`r`n[10] Add AAD/NDJ machines`r`n[11] Add processes`r`n[12] Delete specific assignments`r`n[13] Delete all assignments`r`n"     
                $choiceList = @('1', '2', '3', '4', '5', '6', '7', '8', '9', '10', '11', '12', '13')  
            }
            write-host $curCommand -ForegroundColor Green
            $tmpchoice = (Read-Host  'Enter value').Trim()
        }
    }


}
function Test-CTXIsGuid {
    [OutputType([bool])]
    param
    (
        [Parameter(Mandatory = $true)]
        [string]$StringGuid
    )
 
    $curObjectGuid = [System.Guid]::empty
    return [System.Guid]::TryParse($StringGuid, [System.Management.Automation.PSReference]$curObjectGuid) 
}

function Show-CTXWholeAppList {
    Write-Host "`r`n`r`nList of apps to manage:" -ForegroundColor green 
    Common-ShowAppList -appList ([ref]$Global:appList) 
    $Global:tmpList = $Global:appList | Sort-Object -Property Status, Name
    $redirectApp = New-Object PSObject
    $redirectApp | Add-Member -MemberType NoteProperty -Name "Index" -Value null
    $redirectApp | Add-Member -MemberType NoteProperty -Name "Status" -Value null
    $redirectApp | Add-Member -MemberType NoteProperty -Name "Name" -Value null

    if ($Global:redirectRuleList.Count -gt 0) {
        if ($Global:readRedirectRules -gt 0) {
            $redirectApp.Index = 0;
            $redirectApp.Status = 'Configured & applied  ';
            $redirectApp.Name = "Redirect Rules";
            $tmpPrint = $redirectApp | Format-Table -AutoSize | Out-String
            Write-Host $tmpPrint -ForegroundColor Green
        }
        else {
            $redirectApp.Index = 0;
            $redirectApp.Status = "Configured      ";
            $redirectApp.Name = "Redirect Rules";
            $tmpPrint = $redirectApp | Format-Table -AutoSize | Out-String
            Write-Host $tmpPrint -ForegroundColor yellow
        }
    }
    else {
        $redirectApp.Index = 0;
        $redirectApp.Status = "Not Configured  ";
        $redirectApp.Name = "Redirect Rules";
        $tmpPrint = $redirectApp | Format-Table -AutoSize | Out-String
        Write-Host $tmpPrint
    }

    $appListTable = $Global:tmpList | Select-object Index, Status, Name | Format-Table -AutoSize | Out-String
    if (![string]::IsNullOrWhiteSpace($appListTable)) {
        #split the string on newlines and loop through each line
        $appListTable -split '\r?\n' | ForEach-Object {
            # do not process empty or whitespace-only strings
            if (!([string]::IsNullOrWhiteSpace($_))) {
                if (($_.ToString().Contains('Configured & applied  ') -eq $true) -and ($_.ToString().Contains('Not configured  ') -eq $false)) {
                    Write-Host $_ -ForegroundColor Green
                }

                elseif ($_.ToString().Contains('Configured  ') -eq $true) {
                    Write-Host $_ -ForegroundColor yellow
                }
                else {
                    Write-Host $_
                }
            }
        }
    }
}

function Get-CTXAppAccessControlRules {

    Set-CTXGlobalVars

    Get-CTXWindowsAppx  -UninstalledKey ([ref]$Global:installedAppsFoundInUnInstallKey)
    Get-CTXInstallApps  -UninstalledKey ([ref]$Global:installedAppsFoundInUnInstallKey)



    # retrieve exsting rules from above reg, filter all hidden apps there. Also add rules from reg to $Global:existingRuleList	
    Get-CTXHiddenApps

    Get-CTXMergedApps 

    Show-CTXWholeAppList

    Write-Host "`r`nConfigured & applied: rules are configured for the app and applied to a GPO or the local registry.`r`n`r`nConfigured: rules are configured for the app but not applied to a GPO or the local registry. Warning: Rules will be lost if you exit the tool.`r`n`r`nNot configured: no rules are configured for the app. " -ForegroundColor green
   
    # get target app from admin, and collect info for it
    Get-CTXTargetApp
}

#show admin app paths
function Get-CTXAppRelatedInfo {
    for ($i = 0; $i -lt $Global:fileRegList.Count; $i++) {
        if (($Global:fileRegList[$i].Path -ne '') -and ($Global:fileRegList[$i].Path -ne $null)) {
            $Global:fileRegList[$i].Path = ( $Global:fileRegList[$i].Path) -replace [regex]::Escape($Global:seperatorReplacingSlash), '|'
            $Global:fileRegList[$i].Path = ( $Global:fileRegList[$i].Path) -replace [regex]::Escape('\REGISTRY\MACHINE'), 'HKEY_LOCAL_MACHINE'
            if (( $Global:fileRegList[$i].Path).ToUpper().Contains('\REGISTRY\USER\CU\')) {
                $Global:fileRegList[$i].Path = ( $Global:fileRegList[$i].Path) -replace [regex]::Escape('\REGISTRY\USER\CU'), 'HKEY_CURRENT_USER'
            }
            elseif (( $Global:fileRegList[$i].Path).ToUpper().Contains('\REGISTRY\USER\')) {
                $Global:fileRegList[$i].Path = ( $Global:fileRegList[$i].Path) -replace [regex]::Escape('\REGISTRY\USER'), 'HKEY_USERS'
            }

            			
        }
        
        if (($Global:fileRegList[$i].Value -ne '') -and ($Global:fileRegList[$i].Value -ne $null)) {
            $Global:fileRegList[$i].Value = ( $Global:fileRegList[$i].Value.ToString()) -replace [regex]::Escape($Global:seperatorReplacingSlash), '|'
        }
        
    }
    
    for ($i = 0; $i -lt $Global:assList.Count; $i++) {
        if (($Global:assList[$i].Name -ne '') -and ($Global:assList[$i].Name -ne $null)) {
            $Global:assList[$i].Name = ($Global:assList[$i].Name) -replace [regex]::Escape($Global:seperatorReplacingSlash), '|'
        }

                
    }
    write-host "`r`n`r`n************************************************************" -ForegroundColor Yellow
    write-host "`r`nApp details:" -ForegroundColor Yellow
    Show-CTXCurrentFileRegs
    Show-CTXCurrentAssignments
    write-host "`r`n************************************************************`r`n`r`n" -ForegroundColor Yellow
}
function Test-CTXAppPath {
    param(
        [string]$path
    )
    if (($path -eq $null) -or ($path -eq '')) {
        return $false
    }
    $charCount = ($path. ToCharArray() | Where-Object { $_ -eq ':' } | Measure-Object). Count
    $charCount2 = ($path. ToCharArray() | Where-Object { $_ -eq '%' } | Measure-Object). Count
    # check for invalid charcters,eg, /:'<>| 
    # allow wildcard * and ? 
    $invalidList = @('/', '"', '<', '>', '|')
    foreach ($invalidinput in $invalidList) {
        if ($path.Contains($invalidinput) -or ($charCount -gt 1) -or ($path.Length -lt 4)) {
            return $false
        }
        if (($path.Contains('%') -eq $false) -and ($path.Contains(':') -eq $false)) {
            return $false
        }
        if ( $path.Contains('\\') -eq $true) {
            return $false
        }
        #path with drive
        if ($charCount -eq 1) {
            if (($path[1] -eq ':') -and ((($path[0] -ige 'a') -and ($path[0] -le 'z')) -or (($path[0] -ige 'A') -and ($path[0] -le 'Z')))) {

            }
            else {
                return $false
            }
        }



    }

    return $true
}
function Test-CTXAppName {
    param(
        [string]$path
    )
    if (($path -eq $null) -or ($path -eq '')) {
        return $false
    }

    # check for invalid charcters,eg, /\:*?"<>|
    $invalidList = @('/', '\', ':', '*', '?', '"', '<', '>', '|')
    foreach ($invalidinput in $invalidList) {
        if ($path.Contains($invalidinput)) {
            return $false
        }

    }

    return $true
}
function Replace-CTXUserInfoForOneRule {
    param([ref]$targetpath)  

    $pattern = "c:\\users\\([^\\]+)\\?"
    $sidpattern = "^S-1-\d{1,}-\d{1,}(-\d{1,}){1,}$"

    #for file/folder path
    if ($targetpath.Value -match $pattern) {
        # Get the username from the matched pattern
        $username = $matches[1]   
        #check if it is a profile
        $filename = "ntuser.dat"
        $profilePath = "c:\users\$username"
        $ntuserPath = Join-Path $profilePath $filename
        if (Test-Path $ntuserPath) {
            $targetpath.Value = $targetpath.Value -replace $pattern, '%USERPROFILE%\'
            $targetpath.Value = $targetpath.Value.TrimEnd('\')
        }

    } 

    #for reg path
    if ($targetpath.Value -like '*S-*') {
        $pathparts = $targetpath.Value -split "\\"
        $newpath = ""
        # Iterate through each part of the path
        foreach ($part in $pathParts) {
            # Check if the part matches the pattern
            if ($part -match $sidpattern) {
                $newPart = '!ctx_usersid!'
            }
            else {
                # Keep the original part
                $newPart = $part
            }
    
            $newPath += "$newPart\"
        }
        $targetpath.Value = $newPath.TrimEnd('\')
    }
    return $targetpath.Value
}
function Get-CTXUserInteractions {

    Get-CTXAppRelatedInfo
    $curQuestion = ''
    $curCommand = ''
    $choiceList = @('0', '1', '2', '3', '4', '5', '6', '7', '8', '9')
    if ($Global:fileRegList.Count -eq 0) {
        
        $curQuestion = "`r`nDo you want to add a file or registry entry for this app?`r`n"
        $curCommand = "`r`n[1] Discard the changes you made to the app and continue adding rules for other apps`r`n[2] Save your changes and continue adding rules for other apps`r`n[3] Generate the rules for deployment to machines`r`n    If no assignments are configured, this app is not visible`r`n------------------------------------------------------------`r`n[4] Add files `r`n[5] Add folders`r`n[6] Add registry keys`r`n[7] Add registry values`r`n" 
        $choiceList = @('0', '1', '2', '3', '4', '5', '6', '7')
    }
    else {
        $curQuestion = "`r`nDo you want to add a file or registry entry for this app? Or want to delete one?`r`n"
        $curCommand = "`r`n[1] Discard the changes you made to the app and continue adding rules for other apps`r`n[2] Save your changes and continue adding rules for other apps`r`n[3] Generate the rules for deployment to machines`r`n    If no assignments are configured, this app is not visible`r`n------------------------------------------------------------`r`n[4] Add files `r`n[5] Add folders`r`n[6] Add registry keys`r`n[7] Add registry values`r`n[8] Delete specific entries`r`n[9] Delete all entries`r`n------------------------------------------------------------`r`n[0] Go to the next step to manage assignments`r`n"       
    }
    write-host $curQuestion -ForegroundColor Green
    write-host $curCommand -ForegroundColor Green

    $tmpchoice = (Read-Host  'Enter value').Trim()
    
    while ( ((Test-CTXInteger -curInput $tmpchoice) -ne $true) -or ([int]$tmpchoice -ne 0)) {
        if (($tmpchoice -eq $null) -or ($tmpchoice -eq '') -or ((Test-CTXInteger -curInput $tmpchoice) -ne $true) -or (($choiceList -Contains $tmpchoice) -ne $true)) {     
            $max = $choiceList.Count - 1      
            write-host 'Invalid value. Supported values: 0 -'$max "`r`n" -ForegroundColor yellow
            $tmpchoice = (Read-Host  'Enter value').Trim()
            continue
        }
        else {
 
            switch ([int]$tmpchoice) {
                '1' {
                    Get-CTXAppAccessControlRules
                    return
                }
                '2' {
                    #generate raw rules, add to existin $Global:existingRuleList
                    if (!($Global:RulesInMemoryAppList -contains $Global:targetApp)) {
                        $Global:RulesInMemoryAppList.Add($Global:targetApp) | Out-Null
                    }            
                    Format-CTXAppAccessControlRules -saveChangesAndReturnToBegining $true
                    Get-CTXAppAccessControlRules
                    return
                }
                '3' {
                    #go to generate rules
                    if (!($Global:RulesInMemoryAppList -contains $Global:targetApp)) {
                        $Global:RulesInMemoryAppList.Add($Global:targetApp) | Out-Null
                    } 
                    Format-CTXAppAccessControlRules -saveChangesAndReturnToBegining $false
                    return
                }
                '4' {
    
                    write-host 'Enter the paths of the files you want to add. Separate them with pipe. Example: c:\users\public\a.txt|c:\users\public\b.log. Wildcard * and ? are supported.'  -ForegroundColor Green    
                    [string]$newData = (Read-Host  'Enter paths').Trim()
                    $tmpDataList = $newData.split('|')
                    foreach ($t in $tmpDataList) {
                        $tmpData = $t.ToString().Trim().Trim('"').Trim().TrimEnd('\')
                        if ($tmpData -eq '') {
                            continue
                        }
                        if (Check-CTXImportantFileFolderPath -testPath $tmpData) {
                        }
                        elseif (Test-CTXAppPath -path $tmpData) {
                            Replace-CTXUserInfoForOneRule -targetpath  ([ref]$tmpData)
                            $dup = $Global:fileRegList  | Where-Object Type -EQ 'File' | Select-Object Path
                            if (($dup -ne $null) -and ($dup.Path -contains $tmpData)) {
                                write-host 'Duplicated items skipped:' $tmpData -ForegroundColor yellow
                                continue
                            } 							
                            $Global:fileRegList.Add([PsCustomObject]@{'Index' = $Global:fileRegList.Count + 1; 'Type' = 'File'; 'Path' = $tmpData; 'Value' = '' }) | Out-NUll
                        }

                        else {
                            write-host 'Invalid path:' $tmpData -ForegroundColor Yellow
                        }                       
                    }

                    break
                      
                }
                '5' {
    
                    write-host 'Enter the paths of the folders you want to add. Separate them with pipe. Example: c:\users\public\folder1|c:\users\public\folder2. Wildcard * and ? are supported.'  -ForegroundColor Green    
                    [string]$newData = (Read-Host  'Enter paths').Trim()
                    $tmpDataList = $newData.split('|')
                    foreach ($t in $tmpDataList) {
                        $tmpData = $t.ToString().Trim().Trim('"').Trim().TrimEnd('\')
                        if ($tmpData -eq '') {
                            continue
                        }
                        if (Check-CTXImportantFileFolderPath -testPath $tmpData) {
                        }
                        elseif (Test-CTXAppPath -path $tmpData) {
                            Replace-CTXUserInfoForOneRule -targetpath  ([ref]$tmpData)
                            $dup = $Global:fileRegList  | Where-Object Type -EQ 'Folder' | Select-Object Path
                            if (($dup -ne $null) -and ($dup.Path -contains $tmpData)) {
                                write-host 'Duplicated items skipped:'$tmpData -ForegroundColor yellow
                                continue
                            } 							
                            $Global:fileRegList.Add([PsCustomObject]@{'Index' = $Global:fileRegList.Count + 1; 'Type' = 'Folder'; 'Path' = $tmpData; 'Value' = '' }) | Out-NUll
                        }
                        else {
                            write-host 'Invalid path:' $tmpData -ForegroundColor Yellow
                        }                       
                    }

                    break
                      
                }				
                '6' {
                    write-host 'Enter the paths of the registry keys you want to add. Separate them with pipe. Example: HKLM:\software\key1|HKCU:\software\key2.' -ForegroundColor Green      
                    [string]$newData = (Read-Host  'Enter paths').Trim()
                    $tmpDataList = $newData.split('|')
                    foreach ($t in $tmpDataList) {
                        $tmpData = $t.ToString().Trim().TrimEnd('\')
                        if ($tmpData -eq '') {
                            continue
                        }
                        Replace-CTXUserInfoForOneRule -targetpath  ([ref]$tmpData)
                        $dup = $Global:fileRegList  | Where-Object Type -EQ 'Registry key' | Select-Object Path
                        if (Check-CTXImportantRegPath -testPath $tmpData) {
                            continue
                        }
                        elseif (($dup -ne $null) -and ($dup.Path -contains $tmpData)) {
                            write-host 'Duplicated items skipped:'$tmpData -ForegroundColor yellow
                            continue
                        } 
                        if (($tmpData.StartsWith('HKLM:\') -eq $false) -and ($tmpData.StartsWith('HKCU:\') -eq $false) -and ($tmpData.StartsWith('HKEY_LOCAL_MACHINE\') -eq $false) -and ($tmpData.StartsWith('HKEY_CURRENT_USER\') -eq $false) -and ($tmpData.StartsWith('HKEY_USERS\') -eq $false) -and ($tmpData.StartsWith('HKU:\') -eq $false)) {
                            write-host 'Invalid path:' $tmpData -ForegroundColor Yellow
                            continue
                        }					
                        $Global:fileRegList.Add([PsCustomObject]@{'Index' = $Global:fileRegList.Count + 1; 'Type' = 'Registry key'; 'Path' = $tmpData; 'Value' = '' }) | Out-NUll
                    }

                    break
                }
                '7' {
                    write-host 'Enter the paths of the registry values you want to add. Separate them with pipe.Example: HKLM:\software\key1\val|HKCU:\software\key2\val.'  -ForegroundColor Green     
                    [string]$newData = (Read-Host  'Enter paths').Trim()
                    $tmpDataList = $newData.split('|')
                    foreach ($t in $tmpDataList) {   
                        $tmpData = $t.ToString().Trim().TrimEnd('\')   
                        if ($tmpData -eq '') {
                            continue
                        }
                        
                        $parts = $tmpData -split '\\'
                        $keyPath = ""
                        if ($parts.Length -gt 1) {
                            $keyPath = ([string]::Join('\', $parts[0..($parts.Length - 2)])).Trim().TrimEnd('\')
                            Replace-CTXUserInfoForOneRule -targetpath  ([ref]$keyPath)

                        }
                        else {
                            write-host 'Invalid path:' $tmpData -ForegroundColor Yellow
                            continue
                        }						
                        $val = ($tmpData.Split('\')[-1]).Trim().TrimEnd('\')
                        $dup = $Global:fileRegList  | Where-Object Type -EQ 'Registry value'
                        if (Check-CTXImportantRegPath -testPath $keyPath) {
                            continue
                        }
                        elseif ($dup -ne $null) {
                            $match = $dup | Where-Object { $_.Path -eq $keyPath -and $_.Value -eq $val }
                            if ($match) {
                                write-host 'Duplicated items skipped:'$tmpData -ForegroundColor yellow
                                continue								
                            }

                        } 
                        if (($tmpData.StartsWith('HKLM:\') -eq $false) -and ($tmpData.StartsWith('HKCU:\') -eq $false) -and ($tmpData.StartsWith('HKEY_LOCAL_MACHINE\') -eq $false) -and ($tmpData.StartsWith('HKEY_CURRENT_USER\') -eq $false) -and ($tmpData.StartsWith('HKEY_USERS\') -eq $false) -and ($tmpData.StartsWith('HKU:\') -eq $false)) {
                            write-host 'Invalid path:' $tmpData -ForegroundColor Yellow
                            continue
                        }
                        $Global:fileRegList.Add([PsCustomObject]@{'Index' = $Global:fileRegList.Count + 1; 'Type' = 'Registry value'; 'Path' = $keyPath; 'Value' = $val }) | Out-NUll
                    }  
                    break            
                }
                '8' {
                    write-host 'To delete entries, enter their indexes. Separate them with commas.'  -ForegroundColor Green
                    [string]$deleteData = (Read-Host  'Enter index').Trim()
                    if ($deleteData -eq '') {
                        write-host 'Invalid item:' $deleteData -ForegroundColor Yellow
                        
                    }
                    else {
                        Remove-CTXMultpleItems -deleteData $deleteData -sourceList $Global:fileRegList
                    }
                    
                    break             
                }			
                '9' {
                    $Global:fileRegList.Clear()
                    break
                }

            }
            Show-CTXCurrentFileRegs       
            if ($Global:fileRegList.Count -eq 0) {
                $curQuestion = "`r`nDo you want to add a file or registry entry for this app?`r`n"
                $curCommand = "`r`n[1] Discard the changes you made to the app and continue adding rules for other apps`r`n[2] Save your changes and continue adding rules for other apps`r`n[3] Generate the rules for deployment to machines`r`n    If no assignments are configured, this app is not visible`r`n------------------------------------------------------------`r`n[4] Add files `r`n[5] Add folders`r`n[6] Add registry keys`r`n[7] Add registry values`r`n" 
                $choiceList = @('0', '1', '2', '3', '4', '5', '6', '7')
            }
            else {
                $curQuestion = "`r`nDo you want to add a file or registry entry for this app? Or want to delete one?`r`n"
                $curCommand = "`r`n[1] Discard the changes you made to the app and continue adding rules for other apps`r`n[2] Save your changes and continue adding rules for other apps`r`n[3] Generate the rules for deployment to machines`r`n    If no assignments are configured, this app is not visible`r`n------------------------------------------------------------`r`n[4] Add files `r`n[5] Add folders`r`n[6] Add registry keys`r`n[7] Add registry values`r`n[8] Delete specific entries`r`n[9] Delete all entries`r`n------------------------------------------------------------`r`n[0] Go to the next step to manage assignments`r`n"       
                $choiceList = @('0', '1', '2', '3', '4', '5', '6', '7', '8', '9')
    
            }
            write-host $curQuestion -ForegroundColor Green
            write-host $curCommand -ForegroundColor Green
            $tmpchoice = (Read-Host  'Enter value').Trim()
        }
    }


    Get-CTXAssignments

}

#region finally generated formatted rules

function Add-CTXRedirectRules {
    if ($Global:redirectRuleList.Count -gt 0) {
        foreach ($redirRule in $Global:redirectRuleList) {
			
            [INT]$OUCnt = 0
            [INT]$OUWhitelistCnt = 0
            [INT]$procCnt = 0
            [INT]$procWhitelistCnt = 0
            [INT]$userGroupCnt = 0
            [INT]$userGroupWhitelistCnt = 0
            [INT]$machineCnt = 0
            [INT]$machineWhitelistCnt = 0
            # first, process user/group sids
            $sidStr = ''
            if ($redirRule.Users -ne "*") {
                $tmpSids = $redirRule.Users.Split(',')
                foreach ($_sid in $tmpSids) {
                    $tmpStr = 'User@CTXASSSEP@' + $_sid + '@CTXASSSEP@' + "NA"
                    $sidStr = $sidStr + $tmpStr + ":"
                    if ($_sid.StartsWith('!')) {						
                        $userGroupWhitelistCnt++
                    }
                    else {
                        $userGroupCnt++
                    }
                }
            }
            if ($redirRule.Groups -ne "*") {
                $tmpSids = $redirRule.Groups.Split(',')
                foreach ($_sid in $tmpSids) {
                    $tmpStr = 'Group@CTXASSSEP@' + $_sid + '@CTXASSSEP@' + "NA"
                    $sidStr = $sidStr + $tmpStr + ":"
                    if ($_sid.StartsWith('!')) {						
                        $userGroupWhitelistCnt++
                    }
                    else {
                        $userGroupCnt++
                    }					
					
                }
            }			
            if (($redirRule.Users -eq "*") -and ($redirRule.Groups -eq "*")) {
                $sidStr = "*"
            }
            elseif (($userGroupCnt -eq 0) -and ($userGroupWhitelistCnt -gt 0)) {
                #only has whitelist settings
                $sidStr += '*'
            }
            # process list here
            $procStr = ''
            if ($redirRule.Procs -ne "*") {
                $tmpProcs = $redirRule.Procs.Split(',')
                foreach ($_proc in $tmpProcs) {					
                    $tmpStr = 'Process@CTXASSSEP@@CTXASSSEP@' + $_proc
                    $procStr = $procStr + $tmpStr + ":"
                    if ($_proc.StartsWith('?')) {						
                        $procWhitelistCnt++
                    }
                    else {
                        $procCnt++
                    }	
                }
            }
            else {
                $procStr = $redirRule.Procs
            }
            if (($procCnt -eq 0) -and ($procWhitelistCnt -gt 0)) {
                #only has whitelist settings
                $procStr += '*'
            }
            # process machine
            $compStr = ''
            if ($redirRule.Comps -ne "*") {
                $tmpComps = $redirRule.Comps.Split(',')
                foreach ($_comp in $tmpComps) {
                    $_tmpComp = $_comp
                    $_tmpPrefix = 'Computer@CTXASSSEP@'
					
					if ($_comp -eq "None") {
						$compSid = ''
						$tmpStr = $_tmpPrefix + $compSid + '@CTXASSSEP@' + $_tmpComp
                        $compStr = $compStr + $tmpStr + ":"
						continue
					}

                    if ($_tmpComp.StartsWith([regex]::Escape('EX>'))) {
                        $_tmpPrefix = 'ComputerEx@CTXASSSEP@'
                        $_tmpComp = $_tmpComp.Substring(3)
                        $machineWhitelistCnt++
                    }
                    else {
                        $machineCnt++	
                    }

                    $queryString = '(DNSHostName=' + $_tmpComp + ')'
                    $pcObj = Get-ADComputer -LDAPFilter $queryString
                    $compSid = ''
                    if ($pcObj -eq $null) {
                        write-host "Invalid computer name $_tmpComp skipped!"  -ForegroundColor Yellow
                    }
                    else {
                        $compSid = $pcObj.SID
                    }

                    $tmpStr = $_tmpPrefix + $compSid + '@CTXASSSEP@' + $_tmpComp
                    $compStr = $compStr + $tmpStr + ":"
                }
            }
            else {
                $compStr = $redirRule.Comps
            }
            if (($machineCnt -eq 0) -and ($machineWhitelistCnt -gt 0)) {
                #only has whitelist settings
                $compStr += '*'
            }
            # process OU or categlog
            $ouStr = ''
            if ($redirRule.OU -ne "*") {
                $tmpOUs = $redirRule.OU.Split(',')
                foreach ($_OU in $tmpOUs) {
					
                    if ($_OU.StartsWith([regex]::Escape('EX>'))) {
                        $tmpStr = 'OUX@CTXASSSEP@@CTXASSSEP@' + $_OU.Substring(3)
                        $OUWhitelistCnt++
                    }
                    else {
                        $tmpStr = 'OU@CTXASSSEP@@CTXASSSEP@' + $_OU
                        $OUCnt++
                    }
                    $ouStr = $ouStr + $tmpStr + ":"
                    $ouStr = $ouStr -replace [regex]::Escape('|'), $Global:seperatorReplacingSlash
                }
            }
            if ($ouStr -eq '') {
                $ouStr = '*'
            }
            elseif (($OUCnt -eq 0 ) -and ($OUWhitelistCnt -gt 0)) {
                $ouStr += '*'				
            }
            # Need to process the data fields here
            if (($redirRule.Type -eq '3') -or ($redirRule.Type -eq '4')) {
                $redirRule.Target = $redirRule.Target.TrimEnd('\')
                $redirRule.Target = $redirRule.Target -replace [regex]::Escape('|'), $Global:seperatorReplacingSlash
                $redirRule.Target = $redirRule.Target -replace [regex]::Escape('HKEY_LOCAL_MACHINE'), '\REGISTRY\MACHINE'
                $redirRule.Target = $redirRule.Target -replace [regex]::Escape('HKEY_CURRENT_USER'), '\REGISTRY\USER\CU'
                $redirRule.Target = $redirRule.Target -replace [regex]::Escape('HKEY_USERS'), '\REGISTRY\USER'			
                $redirRule.Target = $redirRule.Target -replace [regex]::Escape('HKLM:'), '\REGISTRY\MACHINE'
                $redirRule.Target = $redirRule.Target -replace [regex]::Escape('HKCU:'), '\REGISTRY\USER\CU'			
                $redirRule.Target = $redirRule.Target -replace [regex]::Escape('HKU:'), '\REGISTRY\USER'						

                $redirRule.Destine = $redirRule.Destine.TrimEnd('\')
                $redirRule.Destine = $redirRule.Destine -replace [regex]::Escape('|'), $Global:seperatorReplacingSlash
                $redirRule.Destine = $redirRule.Destine -replace [regex]::Escape('HKEY_LOCAL_MACHINE'), '\REGISTRY\MACHINE'
                $redirRule.Destine = $redirRule.Destine -replace [regex]::Escape('HKEY_CURRENT_USER'), '\REGISTRY\USER\CU'
                $redirRule.Destine = $redirRule.Destine -replace [regex]::Escape('HKEY_USERS'), '\REGISTRY\USER'			
                $redirRule.Destine = $redirRule.Destine -replace [regex]::Escape('HKLM:'), '\REGISTRY\MACHINE'
                $redirRule.Destine = $redirRule.Destine -replace [regex]::Escape('HKCU:'), '\REGISTRY\USER\CU'			
                $redirRule.Destine = $redirRule.Destine -replace [regex]::Escape('HKU:'), '\REGISTRY\USER'
                
                if ($redirRule.Type -eq '4') {
                    $redirRule.TargetV = $redirRule.TargetV -replace [regex]::Escape('|'), $Global:seperatorReplacingSlash
                    $redirRule.DestV = $redirRule.DestV -replace [regex]::Escape('|'), $Global:seperatorReplacingSlash
                }
            }

            # update value to * for better compatibility
            $_tmpTV = Invoke-Expression '$redirRule.TargetV'
            if ("$_tmpTV" -eq "") {
                $redirRule.TargetV = "*"
            }

            $_tmpDV = Invoke-Expression '$redirRule.DestV'
            if ("$_tmpDV" -eq "") {
                $redirRule.DestV = "*"
            }

            # align local value to service value for target type
            $_type = switch ($redirRule.Type) {
                "1" { [TargetObjectType]::File }
                "3" { [TargetObjectType]::RegistryKey }
                "4" { [TargetObjectType]::RegistryValue }
                "2" { [TargetObjectType]::Folder }
                default { [TargetObjectType]::File } 
            }
            # finalize the rule string
            $tmpRuleStr = "1`|" + [int]$_type + "`|" + $redirRule.Target + "`|" + $redirRule.TargetV + "`|" + $sidStr + "`|" + $procStr + "`|" + $ouStr + "`|" + $compStr + "`|" + $redirRule.Destine + "`|" + $redirRule.DestV
            
            # add to the final data string
            $Global:rulesReadyForRegGPO += $tmpRuleStr
            $Global:rulesReadyForRegGPO += "`n"

        }
    }
}

function Format-CTXAppAccessControlRules {
    param(
        [Parameter(Mandatory = $true)][bool]$saveChangesAndReturnToBegining,
        [Parameter(Mandatory = $false)][bool]$deleteAll = $false
    )
    if (!$deleteAll) {
        # generate rules for current app
        $Global:rulesList.Clear()		
        $res = Common-GenerateAppAccessControlRules -appAssList $Global:assList -curAppName $Global:targetApp -appFileRegList ([ref]$Global:fileRegList) -ruleType '0'
		$res = @($res)
        if ($res) {
            [void]$Global:rulesList.AddRange($res) 
        }
        

    
        #delete app related rules from $Global:existingRuleList
        for ($i = 0; $i -lt $Global:existingRuleList.Count; $i++) {
            if ($Global:existingRuleList[$i].Split('|')[-1] -eq $Global:targetApp) {
                $Global:existingRuleList.RemoveAt($i)
                $i--
            }
        }
    }


    #in normal workflow,the last step to generate raw rules,
    if ($saveChangesAndReturnToBegining -eq $false) {
        if ($deleteAll) {
            $Global:existingRuleList.Clear()
            $Global:RulesInMemoryAppList.Clear()
            $Global:redirectRuleList.Clear()
        }

        $tmpList = New-Object -TypeName 'System.Collections.ArrayList' 
        [void]$tmpList.AddRange($Global:existingRuleList)
        if ($Global:targetApp -ne '') {
        
            [void]$tmpList.AddRange($Global:rulesList)
        }
        else {
            #when admin just want to see existing rules for all apps, no current app specified, so ignore $Global:rulesList
        }
        $tmp = $tmpList | select -Unique 
        $tmpList.Clear()
        if ($tmp.Count -eq 1) {
            [void]$tmpList.Add($tmp)
        }
        elseif ($tmp.Count -gt 1) {
            [void]$tmpList.AddRange($tmp)
        }

        $Global:rulesReadyForRegGPO = ''
        foreach ($i in $tmpList) {
            $Global:rulesReadyForRegGPO += $i
            $Global:rulesReadyForRegGPO += "`n"
        } 

        # format and add redirect rules
        Add-CTXRedirectRules

        $Global:existingRuleList.Clear()
        [void]$Global:existingRuleList.AddRange($tmpList) 
        Get-CTXAdminGPO
        write-host "`r`n[Y] Add more rules`r`n[N] Exit`r`n" -ForegroundColor Green
        $continueGeneratingRules = (Read-Host  'Enter input').Trim() 
        while (($continueGeneratingRules -ne 'Y') -and ($continueGeneratingRules -ne 'N')) {
            write-host 'Invalid input:'$continueGeneratingRules -ForegroundColor yellow
            $continueGeneratingRules = (Read-Host  'Enter input').Trim() 
        }                                         
        if ($continueGeneratingRules -eq 'Y') {
            Get-CTXAppAccessControlRules
        }
        else {
            Exit
        }

    }
    else {
        #return when modifying file/reg list and assignments and admin choose to save
        #add newly app related rules into $Global:existingRuleList
        for ($i = 0; $i -lt $Global:rulesList.Count; $i++) {
            if (($Global:existingRuleList -Contains $Global:rulesList[$i]) -eq $false) {
                [void]$Global:existingRuleList.Add($Global:rulesList[$i])
            }
        } 
        if (!($Global:RulesInMemoryAppList -contains $Global:targetApp)) {
            [void]$Global:RulesInMemoryAppList.Add($Global:targetApp)
        }
                
    }
}

function Check-CTXImportantFileFolderPath {
    param(
        [Parameter(Mandatory = $true)][string]$testPath

    )
    $lowerTestPath = $testPath.TrimEnd('\') + '\'
    $lowerTestPath = $lowerTestPath.ToLower()
    foreach ($var in $Global:importantFileFolderPath) {
        if ($var.ToLower().Contains($lowerTestPath)) {
            write-host 'Protected path:' $testPath' skipped' -ForegroundColor Yellow
            return $true
        }
    }


    foreach ($var in $Global:importantAppPath) {
        if ($lowerTestPath -like $var) {
            write-host 'Protected path:' $testPath' skipped' -ForegroundColor Yellow
            return $true
        }
    }

    return $false
}
function Get-CTXAdminGPO {
    $choiceList = @('1', '2', '3', '4')
    $forceApplySuc = $false
    write-host "`r`n[1] Apply rules to a GPO`r`n[2] Apply rules to a local registry`r`n[3] Save rules for future use`r`n[4] Return to previous Menu" -ForegroundColor Green
    $apply = (Read-Host  'Enter value').Trim()

    while (((Test-CTXInteger -curInput $apply) -ne $true) -or (($choiceList -Contains $apply) -ne $true)) {
        write-host 'Invalid value. Enter a value of 1 -'$choiceList.Count "`r`n" -ForegroundColor yellow
        write-host "`r`nTo apply rules to GPO, enter 1.`r`nTo apply rules to local registry, enter 2.`r`nSave rules for future use, enter 3." -ForegroundColor Green
        $apply = (Read-Host  'Enter value').Trim()
    }





    if ($apply -eq '1') {

        #list exisiting GPOs
        write-host "`r`nIt will take a moment to query current domain. Please wait..." -ForegroundColor Yellow
        $tmp = (systeminfo | findstr /B /C:'Domain').split(':')
        #eg, ctxxa.local or workgroup
        $curDomainName = $tmp[1].TrimStart()
        
        try {
            write-host "`r`nIt will take a moment to query Group Policies in current domain. Please wait..." -ForegroundColor Yellow
            $allGposInDomain = (Get-GPO -All -Domain $curDomainName).DisplayName 

            $allGPOCnt = 0
            [System.Collections.ArrayList]$allGposInDomainList = @()
            foreach ($curGPO in $allGposInDomain) {
                $val = [PsCustomObject]@{'Index' = ++$allGPOCnt; 'GPO' = $curGPO }
                $allGposInDomainList.Add($val) | Out-NUll
            }
            
            $allGposInDomainList | Format-Table
            
            while ($True) {
                write-host "`r`nEnter the index of your GPO. To skip this step and continue, enter 0." -ForegroundColor Green
                $curID = (Read-Host  'Enter index').Trim()
                try {
                    if (Test-CTXInteger -curInput $curID) {
                        if ([INT]$curID -eq 0) {
                            Get-CTXAdminGPO
                            write-host "`r`n[Y] Add more rules`r`n[N] Exit`r`n" -ForegroundColor Green
                            $continueGeneratingRules = (Read-Host  'Enter input').Trim() 
                            while (($continueGeneratingRules -ne 'Y') -and ($continueGeneratingRules -ne 'N')) {
                                write-host 'Invalid input:'$continueGeneratingRules -ForegroundColor yellow
                                $continueGeneratingRules = (Read-Host  'Enter input').Trim() 
                            }                                         
                            if ($continueGeneratingRules -eq 'Y') {
                                Get-CTXAppAccessControlRules
                            }
                            else {
                                Exit
                            }
                        }
                        elseif (($allGposInDomainList | Select-Object Index).Index -Contains [INT]$curID) {              
                            $Global:targetGPO = Get-GPO -name $allGposInDomainList[[INT]$curID - 1].GPO
                            break
                        }
                        else {           
                            Write-Host 'Invalid index.' -ForegroundColor Yellow
                        }
                    }
                    else {
                        Write-Host 'Invalid index.' -ForegroundColor Yellow
                    }				
                }
                catch {
                    Write-Host 'Invalid index.'  -ForegroundColor Yellow
                }


            }

               

            Set-GPRegistryValue  -Key 'HKLM\SOFTWARE\Policies\Citrix\UserProfileManager' -ValueName 'AppAccessControlRules' -Type String -Value $Global:rulesReadyForRegGPO -Guid $Global:targetGPO.Id  | Out-Null                       

            write-host  "`r`nIt will take a moment for Windows Group Policy Management to update the policy to the local machine. Please wait..." -ForegroundColor Yellow
            gpupdate /force

            $forceApplySuc = $true
            write-host "`r`nRules applied successfully."
            $Global:RulesInMemoryAppList.Clear()
        }
        catch {
            write-host 'Unable to find or update the GPO. Veirfy that the domain where the current machine resides is reachable and then try again.' -ForegroundColor Yellow
        
        } 
        if ($forceApplySuc -ne $true) {
            if ($emptyRule) {
                write-host 'An error occurred while applying the rules. Your changes have made the rules empty. To apply your changes, set the Profile Management group policy 'App access control' as empty. Or, you can locate the registry key HKLM\SOFTWARE\Policies\Citrix\UserProfileManager\UserConfigDriverActionRules, and set the registry value AppAccessControlRules as empty.' -ForegroundColor Yellow
            }
            else {
                write-host 'An error occurred while applying the rules. To apply your changes, set the Profile Management group policy 'App access control' with the string below. Or, you can locate the registry key HKLM\SOFTWARE\Policies\Citrix\UserProfileManager\UserConfigDriverActionRules, and set the registry value AppAccessControlRules to the string below.' -ForegroundColor Yellow
                $Global:rulesReadyForRegGPO
            }

        }

    }
    elseif ($apply -eq '2') {
        $serviceRestartStatus = $false
        try {
            $tmpString = ''
            $upmRegKeyPath = 'HKLM:\SOFTWARE\Policies\Citrix\UserProfileManager'

            if ((Test-Path  $upmRegKeyPath) -eq $false) {
                New-Item -Path $upmRegKeyPath -Force | Out-Null
            }    
            if ((Get-Item -Path $upmRegKeyPath).GetValue('AppAccessControlRules') -ne $null) {
                Set-ItemProperty -Path $upmRegKeyPath -Name 'AppAccessControlRules' -Value $Global:rulesReadyForRegGPO
            }
            else {
                New-ItemProperty -Path $upmRegKeyPath -Name 'AppAccessControlRules' -Value $Global:rulesReadyForRegGPO -Force
            }

            write-host 'Rules applied successfully.'
            $Global:RulesInMemoryAppList.Clear()

        }
        catch {
            {
                write-host 'An error occurred while restarting user profile manager service, restart manually.' -ForegroundColor Yellow
            }
        }
    }
    elseif ($apply -eq '4') {
        Get-CTXAppAccessControlRules
    }
    else {
        if ($emptyRule) {
            $tmpString = "Your changes have made the rules empty. To apply your changes, set the Profile Management group policy 'App access control' as empty. Or, you can locate the registry key HKLM\SOFTWARE\Policies\Citrix\UserProfileManager\UserConfigDriverActionRules, and set the registry value AppAccessControlRules as empty."
            write-host $tmpString -ForegroundColor Yellow
        }
        else {
            $documentsPath = [Environment]::GetFolderPath('MyDocuments')
            $dateTime = Get-Date -Format "yyyyMMdd-HHmmss"
            $filePath = $documentsPath + '\' + "appaccess-rules-$dateTime.txt"
			
            $Global:rulesReadyForRegGPO

            # Create a new file with the generated name
            New-Item -ItemType File -Path $filePath | Out-NUll
            Set-Content -Path $filePath -Value $Global:rulesReadyForRegGPO -NoNewline | Out-NUll
			
            Write-Host "Rules you have configured are saved at $filePath. To apply them using GPOs, copy the file's content and paste it to the Profile Management policy 'App access control'." -ForegroundColor Yellow
        }

    }

}


#endregion

#region for wem
function Common-GenerateAppAccessControlRules {
    param (
        [Parameter(Mandatory = $true)]
        $appAssList,
        [Parameter(Mandatory = $true)]
        $curAppName,
        [Parameter(Mandatory = $true)]
        [ref]$appFileRegList,	
        [Parameter(Mandatory = $true)]
        $ruleType				
    )	
    #eg, file rule 0|0|c:\users\public\aa.exe|*|User@CTXASSSEP@S-1-5-21-674278408-26188528-2146851469-1334@CTXASSSEP@fuser11|*|*|*
    #assignement contains : ID Type SID Name
    #for group, user,computer,process, their name could not contains '|'
    #for reg key/value, and for ou name in assignments, they could contain '|', replace with @CTXBARSEP@
    $tmpRulesList = New-Object -TypeName 'System.Collections.ArrayList'
    if ($appFileRegList.Value.Count -eq 0) {
        return $tmpRulesList
    }

    $assFormated = '' 
    $userList = ''
    $processList = ''
    $ouList = ''
    $computerList = ''
    #in assignements, scope field stores 'NDJ' to show this is for NDJ assignment
    #in raw rule data structure, above info are stored in its sid field for machine catalog and computer, for user/groups, sid field stores OID which itself shows it is a NDJ assignment
    if ($appAssList -ne $null) {
        $bIsADAllComputers = $false
        $bIsNDJAllComputers = $false

        $isADAll = $appAssList | Where-Object { (($_.Type -eq 'AD machine') -and ($_.Name -eq 'AD*')) }
        $isNDJAll = $appAssList | Where-Object { (($_.Type -eq 'AAD/NDJ machine') -and ($_.Name -eq 'NDJ*')) }
        if ($isADAll -and $isNDJAll) {
            $bIsADAllComputers = $true
            $bIsNDJAllComputers = $true
        }
        elseif ($isADAll) {
            $bIsADAllComputers = $true
            $t = 'Computer@CTXASSSEP@' + 'AD*' + '@CTXASSSEP@'             
            $computerList += $t
            $computerList += ':'  
        }
    
        elseif ($isNDJAll) {
            $bIsNDJAllComputers = $true
            $t = 'Computer@CTXASSSEP@' + 'NDJ*' + '@CTXASSSEP@'           
            $computerList += $t
            $computerList += ':' 
        }

        [INT]$OUCnt = 0
        [INT]$OUWhitelistCnt = 0
        [INT]$procCnt = 0
        [INT]$procWhitelistCnt = 0
        [INT]$userGroupCnt = 0
        [INT]$userGroupWhitelistCnt = 0
        [INT]$machineCnt = 0
        [INT]$machineWhitelistCnt = 0
        foreach ($tmp in $appAssList) {
        
            switch ($tmp.Type) {
                'User' {
                    $userList += 'User@CTXASSSEP@' + $tmp.SID + '@CTXASSSEP@' + $tmp.Name         
                    $userList += ':' 
                    if ($tmp.Name.StartsWith([regex]::Escape('EX>'))) {						
                        $userGroupWhitelistCnt++
                    }
                    else {
                        $userGroupCnt++
                    }					
                }
                'Group' {
                    $userList += 'Group@CTXASSSEP@' + $tmp.SID + '@CTXASSSEP@' + $tmp.Name      
                    $userList += ':'
                    if ($tmp.Name.StartsWith([regex]::Escape('EX>'))) {						
                        $userGroupWhitelistCnt++
                    }
                    else {
                        $userGroupCnt++
                    }					
									
            
                }           
                'Process' {
                    $t = 'Process@CTXASSSEP@' + $tmp.SID + '@CTXASSSEP@' + $tmp.Name
            
                    $processList += $t
            
                    $processList += ':'       
                    if ($tmp.Name.StartsWith('?')) {						
                        $procWhitelistCnt++
                    }
                    else {
                        $procCnt++
                    }
                }
                'OU' {

                    $rep = ($tmp.Name) -replace [regex]::Escape('|'), $Global:seperatorReplacingSlash
                    if ($rep.StartsWith([regex]::Escape('EX>'))) {
                        $t = 'OUX@CTXASSSEP@' + $tmp.SID + '@CTXASSSEP@' + $rep.Substring(3)
                        $OUWhitelistCnt++
                    }
                    else {
                        $t = 'OU@CTXASSSEP@' + $tmp.SID + '@CTXASSSEP@' + $rep
                        $OUCnt++						
                    }    
                    $ouList += $t
                    $ouList += ':'          
            
                }
                'Machine catalog' {
                    $rep = ($tmp.Name) -replace [regex]::Escape('|'), $Global:seperatorReplacingSlash
                    if ($rep.StartsWith([regex]::Escape('EX>'))) {
                        $t = 'OUX@CTXASSSEP@' + $tmp.Scope + '@CTXASSSEP@' + $rep.Substring(3)
                        $OUWhitelistCnt++
                    }
                    else {
                        $t = 'OU@CTXASSSEP@' + $tmp.Scope + '@CTXASSSEP@' + $rep
                        $OUCnt++
                    }        
                    $ouList += $t
                    $ouList += ':'         
                }

                'AD machine' {


                    if ($tmp.Name.StartsWith([regex]::Escape('EX>'))) {
                        $t = 'ComputerEx@CTXASSSEP@' + $tmp.SID + '@CTXASSSEP@' + $tmp.Name.Substring(3)
                        $computerList += $t
                        $computerList += ':'
                        $machineWhitelistCnt++						
                    }
                    elseif ($bIsADAllComputers -ne $true) {
                        $t = 'Computer@CTXASSSEP@' + $tmp.SID + '@CTXASSSEP@' + $tmp.Name
                        $computerList += $t
                        $computerList += ':' 
                        $machineCnt++						
                    }       
            
                }
                'AAD/NDJ machine' {

                    if ($bIsNDJAllComputers -ne $true) {
                        $t = 'Computer@CTXASSSEP@' + $tmp.Scope + '@CTXASSSEP@' + $tmp.Name           
                        $computerList += $t
                        $computerList += ':'
                        $machineCnt++						
                    } 
          
                        
                }           
            }

        }
    }
    #only has whitelist settings
    if (($OUCnt -eq 0) -and ($OUWhitelistCnt -gt 0)) {
        $ouList += '*'
    }
    if (($userGroupCnt -eq 0) -and ($userGroupWhitelistCnt -gt 0)) {
        $userList += '*'
    }
    if (($procCnt -eq 0) -and ($procWhitelistCnt -gt 0)) {
        $processList += '*'
    }
    if (($machineCnt -eq 0) -and ($machineWhitelistCnt -gt 0)) {
        $computerList += '*'
    }		
    $userList = $userList.Trim(':')
    $processList = $processList.Trim(':')
    $ouList = $ouList.Trim(':')
    $computerList = $computerList.Trim(':')


    if ($userList -eq '') {
        $assFormated += '*'
    }
    else {
        $assFormated += $userList
    }


    $assFormated += '|'

    if ($processList -eq '') {
        $assFormated += '*'

    }
    else {
        $assFormated += $processList
    }

    $assFormated += '|'

    if ($ouList -eq '') {
        $assFormated += '*'

    }
    else {
        $assFormated += $ouList
    }
    $assFormated += '|'

    if ($computerList -eq '') {
        $assFormated += '*'

    }
    else {
        $assFormated += $computerList
    }


    #for targets , it contains : ID Type Path Value
    #replace '|' with $Global:seperatorReplacingSlash
    foreach ($cur in $appFileRegList.Value) {
        if ($cur.Destine -eq "") {
            $cur.Destine = "*"
        }
        if ($cur.DestV -eq "") {
            $cur.DestV = "*"
        }

        if ($cur.Type -eq 'File') {
            if ($ruleType -eq '0') {
                $currule = '0|0|' + $cur.Path + '|*|' + $assFormated + '|' + $curAppName
            }
            elseif ($ruleType -eq '1') {
            
                $currule = '1|0|' + $cur.Path + '|*|' + $assFormated + '|' + $cur.Destine + '|' + $cur.DestV + '|' + $curAppName
            }
            
            [void]$tmpRulesList.Add($currule)           
        }
        elseif ($cur.Type -eq 'Folder') {
            if ($ruleType -eq '0') {
                $currule = '0|3|' + $cur.Path.TrimEnd('\') + '|*|' + $assFormated + '|' + $curAppName
            }
            elseif ($ruleType -eq '1') {
                $currule = '1|3|' + $cur.Path + '|*|' + $assFormated + '|' + $cur.Destine + '|' + $cur.DestV + '|' + $curAppName
            }           
            
            [void]$tmpRulesList.Add($currule)           
        }		
        elseif (($cur.Type -eq 'Registry key') -or ($cur.Type -eq 'Registry value')) {
            $cur.Path = $cur.Path.TrimEnd('\')
            $cur.Path = $cur.Path -replace [regex]::Escape('|'), $Global:seperatorReplacingSlash
            $cur.Path = $cur.Path -replace [regex]::Escape('HKEY_LOCAL_MACHINE'), '\REGISTRY\MACHINE'
            $cur.Path = $cur.Path -replace [regex]::Escape('HKEY_CURRENT_USER'), '\REGISTRY\USER\CU'
            $cur.Path = $cur.Path -replace [regex]::Escape('HKEY_USERS'), '\REGISTRY\USER'			
            $cur.Path = $cur.Path -replace [regex]::Escape('HKLM:'), '\REGISTRY\MACHINE'
            $cur.Path = $cur.Path -replace [regex]::Escape('HKCU:'), '\REGISTRY\USER\CU'			
            $cur.Path = $cur.Path -replace [regex]::Escape('HKU:'), '\REGISTRY\USER'	
            if ($ruleType -eq '1') {
                $cur.Destine = $cur.Destine.TrimEnd('\')
                $cur.Destine = $cur.Destine -replace [regex]::Escape('|'), $Global:seperatorReplacingSlash
                $cur.Destine = $cur.Destine -replace [regex]::Escape('HKEY_LOCAL_MACHINE'), '\REGISTRY\MACHINE'
                $cur.Destine = $cur.Destine -replace [regex]::Escape('HKEY_CURRENT_USER'), '\REGISTRY\USER\CU'
                $cur.Destine = $cur.Destine -replace [regex]::Escape('HKEY_USERS'), '\REGISTRY\USER'			
                $cur.Destine = $cur.Destine -replace [regex]::Escape('HKLM:'), '\REGISTRY\MACHINE'
                $cur.Destine = $cur.Destine -replace [regex]::Escape('HKCU:'), '\REGISTRY\USER\CU'		
                $cur.Destine = $cur.Destine -replace [regex]::Escape('HKU:'), '\REGISTRY\USER'			
            }


            if ($cur.Type -eq 'Registry key') {
                if ($ruleType -eq '0') {
                    $currule = '0|1|' + $cur.Path + '|*|' + $assFormated + '|' + $curAppName
                }
                elseif ($ruleType -eq '1') {
                    $currule = '1|1|' + $cur.Path + '|*|' + $assFormated + '|' + $cur.Destine + '|' + $cur.DestV + '|' + $curAppName
                }

            }
            else {
                $cur.Value = $cur.Value.TrimEnd('\')		
                $cur.Value = $cur.Value -replace [regex]::Escape('|'), $Global:seperatorReplacingSlash
                if ($ruleType -eq '0') {
                    $currule = '0|2|' + $cur.Path.Trim().TrimEnd('\') + '|' + $cur.Value + '|' + $assFormated + '|' + $curAppName
                }
                elseif ($ruleType -eq '1') {
                    $cur.DestV = $cur.DestV.TrimEnd('\')		
                    $cur.DestV = $cur.DestV -replace [regex]::Escape('|'), $Global:seperatorReplacingSlash
                    $currule = '1|2|' + $cur.Path + '|' + $cur.Value + '|' + $assFormated + '|' + $cur.Destine + '|' + $cur.DestV + '|' + $curAppName
                }

            }
					            
            [void]$tmpRulesList.Add($currule)
        }

    }
    return $tmpRulesList
}
function Get-CTXWindowsAppx {
    param (
        [Parameter(Mandatory = $true)]
        [ref]$UninstalledKey	
			
    )
    $res = Get-AppxPackage -AllUsers | Select-Object Name, InstallLocation
    if ($res -eq $null) {
        return
    }
    foreach ($curApp in $res) {
        if ($UninstalledKey.Value.ContainsKey($curApp.Name) -eq $false) {
            [void]$UninstalledKey.Value.Add($curApp.Name, $curApp.InstallLocation) 
        }
    }
}
function Get-CTXMergedApps {

    #loop $Global:installedAppsFoundInUnInstallKey, add to $Global:appList
    foreach ($tmp in $Global:installedAppsFoundInUnInstallKey.Keys) {
        if ((($Global:existingHiddenAppNameList -Contains $tmp) -eq $false) -and !(Test-CTXIsGuid -StringGuid $tmp)) {
            #Not a hidden app
            if (!($Global:RulesInMemoryAppList -contains $tmp)) {
                $val = [PsCustomObject]@{'Index' = $Global:appList.Count + 1; 'Status' = 'Not configured  '; 'Name' = $tmp; 'InstallDir' = $Global:installedAppsFoundInUnInstallKey[$tmp] }
            }
            else {
                $val = [PsCustomObject]@{'Index' = $Global:appList.Count + 1; 'Status' = 'Configured  '; 'Name' = $tmp; 'InstallDir' = $Global:installedAppsFoundInUnInstallKey[$tmp] }
            }
            
            $Global:appList.Add($val) | Out-NUll 
        }
    }    

}
function Commont-GetRulesFromString {
    param (
        [Parameter(Mandatory = $true)]
        [string]$rawRuleData,		
        [Parameter(Mandatory = $true)]
        [ref]$appList
       		
    )
    $curRuleListFromReg = New-Object -TypeName 'System.Collections.ArrayList'

    if ($rawRuleData -eq '') {
    }
    else {       
        $rawRuleData = $rawRuleData -replace [regex]::Escape("`r`n"), "`n"
        $splitList = $rawRuleData -split "`n"
        $listHashtable = @{}
        foreach ($curValue in $splitList) {        
            if (($curValue -eq $null) -or ($curValue -eq '')) {
                continue
            }

						
            $fields = $curValue -split '\|'
            #app name
            $name = $fields[-1] + '_' + $fields[0]
            # different types of rules could have same name, so use name_type as key
            if (-not $listHashtable.ContainsKey($name)) {
                $listHashtable[$name] = @()
            }

            # Add the line to the appropriate list based on the last item
            $listHashtable[$name] += $curValue
        }
        foreach ($key in $listHashtable.Keys) {
            #raw for each app
            $rawdatas = $listHashtable[$key]
            $tmp = @{
                "name"                       = ($key.Split('_'))[0]
                "type"                       = ""
                "targetObjects"              = @()
                "assignments"                = @()
                "exclusions"                 = @()
                "enableConfigureADMachines"  = $true
                "enableConfigureNDJMachines" = $true
                "expanded"                   = $false
            }
            $bAssProcessed = $false
            foreach ($rawdata in $rawdatas) {
                $fields = $rawdata -split '\|'
                $tarType = ''
                $tmp["type"] = $fields[0]
                $convertedPath = $fields[2] -replace [regex]::Escape('\REGISTRY\MACHINE'), 'HKEY_LOCAL_MACHINE'
                $convertedPath = $convertedPath -replace [regex]::Escape('\REGISTRY\USER\CU'), 'HKEY_CURRENT_USER'
                $convertedPath = $convertedPath -replace [regex]::Escape('\REGISTRY\USER'), 'HKEY_USERS'	
                $convertedValue = $fields[3] -replace [regex]::Escape('\REGISTRY\MACHINE'), 'HKEY_LOCAL_MACHINE'
                $convertedValue = $convertedValue -replace [regex]::Escape('\REGISTRY\USER\CU'), 'HKEY_CURRENT_USER'
                $convertedValue = $convertedValue -replace [regex]::Escape('\REGISTRY\USER'), 'HKEY_USERS'
                if ($convertedValue -eq '*') {
                    $convertedValue = ''
                }
                if ($tmp["type"] -eq '0') {
					
                    $target = [PsCustomObject]@{'type' = $fields[1]; 'path' = $convertedPath; 'value' = $convertedValue }
                    $tmp["targetObjects"] += $target 

                }
                elseif ($tmp["type"] -eq '1') {
					
                    $convertedTarPath = $fields[8] -replace [regex]::Escape('\REGISTRY\MACHINE'), 'HKEY_LOCAL_MACHINE'
                    $convertedTarPath = $convertedTarPath -replace [regex]::Escape('\REGISTRY\USER\CU'), 'HKEY_CURRENT_USER'
                    $convertedTarPath = $convertedTarPath -replace [regex]::Escape('\REGISTRY\USER'), 'HKEY_USERS'				
					
                    $convertedTarValue = $fields[9] -replace [regex]::Escape('\REGISTRY\MACHINE'), 'HKEY_LOCAL_MACHINE'
                    $convertedTarValue = $convertedTarValue -replace [regex]::Escape('\REGISTRY\USER\CU'), 'HKEY_CURRENT_USER'
                    $convertedTarValue = $convertedTarValue -replace [regex]::Escape('\REGISTRY\USER'), 'HKEY_USERS'
                    if ($convertedTarValue -eq '*') {
                        $convertedTarValue = ''
                    }			
                    $target = [PsCustomObject]@{'type' = $fields[1]; 'path' = $convertedPath; 'value' = $convertedValue; 'targetPath' = $convertedTarPath; 'targetValue' = $convertedTarValue }
                    #$target = @{'type' = $fields[1]; 'path' = $fields[2]; 'value' = $fields[3]; 'targetPath' = $fields[8]; 'targetValue' = $fields[9]}
                    $tmp["targetObjects"] += $target
                }
				
                
                if ($bAssProcessed -eq $false) {
                    #process assignments: 
                    $bIsADAllComputers = $false
                    $bIsNDJAllComputers = $false
                    for ($i = 4; $i -lt 8; $i++) {
                        if ($fields[$i] -eq '*') {

                        }
                        else {
                            $tmpList = $fields[$i].split(':')
                            $tmpCount = $tmpList.Count
                            foreach ($var in  $tmpList) { 
                                if (($var -eq '*') -and ($tmpCount -gt 1)) {
                                    #we could see * in two situations, 1) one is only * in a field, means all should be hidden. 2) The other one is, not only * in an assignment field, this * means it is appended automatically for pure whitelist(pure whitelist means only has whitelist settings for this type of assignments), do not display it to users
                                    continue
                                }								
							
                                $eachAss = $var -split '@CTXASSSEP@'
                                #AD*, NDJ* 
                                if (($eachAss[1] -eq 'AD*') -and ($eachAss[0] -eq 'Computer')) {
                                    $bIsADAllComputers = $true
                                    continue
                                }
                                elseif (($eachAss[1] -eq 'NDJ*') -and ($eachAss[0] -eq 'Computer')) {
                                    $bIsNDJAllComputers = $true
                                    continue
                                }
                                $bIsExclusion = $false
                                $assType = $eachAss[0]
                                $assSid = $eachAss[1]
                                $assName = $eachAss[2]
                                #user /group for NDJ                                                                                           
                                if ((($eachAss[0] -EQ 'User') -or ($eachAss[0] -EQ 'Group')) -and (($eachAss[1] -like '/*') -or ($eachAss[1] -like '!/*'))) {
                                    if ($eachAss[1] -like '!/*') {
                                        $bIsExclusion = $true
                                        $assSid = $assSid.Substring(1)
                                        if ($assName.StartsWith([regex]::Escape('EX>')) -and ($assName -ne [regex]::Escape('EX>'))) {
                                            $assName = $assName.Substring(3)
                                        }										
                                    }	
                                    if ($eachAss[0] -EQ 'User') {
                                        $assType = '7'
                                    }
                                    else {
                                        $assType = '8'
                                    }
									


                                } 
                                #others                   
                                elseif ($eachAss[1] -EQ 'NDJ') {
                                    #machine or machine catalog for NDJ
                                    #<action type>|<target type>|<target path>| <target value>| <user and group list>|<process list>|<OU list>|<computer list>|<app name>

                                    if ($i -eq 6) {
                                        #machine catalog
                                        $assType = '4' 
                                        if ($eachAss[0] -eq 'OUX') {
                                            $bIsExclusion = $true
                                        }
                                    }
                                    elseif ($i -eq 7) {
                                        #NDJ machine, does not support white list
                                        $assType = '6'
                                    }                          
                                }

                                else {
                                    if ($i -eq 4) {
                                        if ($eachAss[1].StartsWith('!')) {
                                            $bIsExclusion = $TRUE
                                            $assSid = $assSid.Substring(1)
                                            if ($assName.StartsWith([regex]::Escape('EX>')) -and ($assName -ne [regex]::Escape('EX>'))) {
                                                $assName = $assName.Substring(3)
                                            }
                                            
                                        }										
                                        if ($eachAss[0] -EQ 'User') {
                                            $assType = '0'
                                        }
                                        else {
                                            $assType = '1'
                                        }
                                    }
                                    elseif ($i -eq 5) {
                                        #process
                                        $assType = '2'
                                        if ($eachAss[2].StartsWith('?')) {
                                            $bIsExclusion = $true
                                            $assName = $assName.Substring(1)
                                        }										
                                    }
                                    elseif ($i -eq 6) {
                                        #OU
                                        $assType = '3'
                                        if ($eachAss[0] -eq 'OUX') {
                                            $bIsExclusion = $true
                                        }										
                                    }
                                    elseif ($i -eq 7) {
                                        #AD machine
                                        $assType = '5'
                                        if ($eachAss[0] -eq 'ComputerEx') {
                                            $bIsExclusion = $true
                                        }										
                                    }                       
                                } 
                                if ($bIsExclusion) {
                                    $ass = [PsCustomObject]@{'sid' = $assSid; 'name' = $assName; 'type' = $assType }
                                    $tmp["exclusions"] += $ass									
                                }
                                else {
                                    $ass = [PsCustomObject]@{'sid' = $assSid; 'name' = $assName; 'type' = $assType }
                                    $tmp["assignments"] += $ass
                                }	

                            }  

                        }

                    }

               
                    $bAssProcessed = $true
                }
       
            }
		
            $appList.Value += $tmp

        }

        #parse: <action type>|<target type>|<target path>| <target value>| <user and group list>|<process list>|<OU list>|<computer list>|<app name>          		
    }

}

function Get-CTXHiddenApps {
    $curRuleListFromReg = New-Object -TypeName 'System.Collections.ArrayList'
    #read policy HKEY_LOCAL_MACHINE\SOFTWARE\Policies\Citrix\UserProfileManager\AppAccessControlRules
    $regPath = 'HKLM:\SOFTWARE\Policies\Citrix\UserProfileManager'
    if (Test-Path $regPath) {
        $regKey = Get-Item -Path  $regPath

        $regValue = $regKey.GetValue('AppAccessControlRules', '')
        if ($regValue -eq '') {
        }
        else {
            $regValue = $regValue -replace [regex]::Escape("`r`n"), "`n"
            $splitList = $regValue -split "`n"
            $hideAppwithchanges = New-Object -TypeName 'System.Collections.ArrayList'
    
            foreach ($curValue in $splitList) {        
                if (($curValue -eq $null) -or ($curValue -eq '')) {
                    continue
                }
                $curApp = $curValue.Split('|')[-1]

                if ($curValue.Split('|')[0] -eq '1') {
                    #special handling for redirect rules
                    if ($Global:RegProcessedFlag -eq '1') {
                        continue
                    }
	
                    $curValues = $curValue.Split('|')
                    $_tmpType = 1
                    if ($curValues[1] -ne 0) {
                        $_tmpType = $curValues[1] + 2
                    }

                    $_tmpUsers = ''
                    $_tmpGroups = ''
                    $_tmpUGList = $curValues[4].Split(':')
                    $tmpCount = $_tmpUGList.Count
                    foreach ($_tmpUGItem in $_tmpUGList) {
                        if ("$_tmpUGItem" -eq "") {
                            continue
                        }
		
                        if ($_tmpUGItem -eq '*') {
                            if ($tmpCount -gt 1) {
                                continue
                            }
                            else {
                                $_tmpUsers = '*'
                                $_tmpGroups = '*'
                            }
                            
                        }
                        else {
                            if ($_tmpUGItem.StartsWith('Group')) {
                                $_tmpGroups = ($_tmpUGItem -replace [regex]::Escape("@CTXASSSEP@"), "|").Split("|")[1] + "," + $_tmpGroups
                            
                            }
                            else {
                                $_tmpUsers = ($_tmpUGItem -replace [regex]::Escape("@CTXASSSEP@"), "|").Split("|")[1] + "," + $_tmpUsers
                            }
                        }
                    }
                    if ($_tmpUsers -eq '') {
                        $_tmpUsers = '*'
                    }
                    if ($_tmpGroups -eq '') {
                        $_tmpGroups = '*'
                    }
                    $_tmpUsers = $_tmpUsers.TrimEnd(',')
                    $_tmpGroups = $_tmpGroups.TrimEnd(',')
                    $_tmpComps = ''
                    $_tmpCompList = $curValues[7].Split(':')
                    $tmpCount = $_tmpCompList.Count
                    foreach ($_tmpUGItem in $_tmpCompList) {
                        if ("$_tmpUGItem" -eq "") {
                            continue
                        }
		
                        if ($_tmpUGItem -eq '*') {
                            if ($tmpCount -gt 1) {
                                continue
                            }
                            else {
                                $_tmpComps = $_tmpUGItem + "," + $_tmpComps
                            }
                        }
                        else {
                            $_tmpComps = ($_tmpUGItem -replace [regex]::Escape("@CTXASSSEP@"), "|").Split("|")[2] + "," + $_tmpComps
                        }
                    }
                    $_tmpComps = $_tmpComps.TrimEnd(',')
	
                    $_tmpOUs = ''
                    $_tmpOUList = $curValues[6].Split(':')
                    $tmpCount = $_tmpOUList.Count
                    foreach ($_tmpUGItem in $_tmpOUList) {
                        if ("$_tmpUGItem" -eq "") {
                            continue
                        }
		
                        if ($_tmpUGItem -eq '*') {
                            if ($tmpCount -gt 1) {
                                continue
                            }
                            else {
                                $_tmpOUs = $_tmpUGItem + "," + $_tmpOUs
                            }
                        }
                        else {
                            $_tmpOUs = ($_tmpUGItem -replace [regex]::Escape("@CTXASSSEP@"), "|").Split("|")[2] + "," + $_tmpOUs
                        }
                    }
                    $_tmpOUs = $_tmpOUs.TrimEnd(',')
	
                    $_tmpProcs = ''
                    $_tmpProcList = $curValues[5].Split(':')
                    $tmpCount = $_tmpProcList.Count
                    foreach ($_tmpUGItem in $_tmpProcList) {
                        if ("$_tmpUGItem" -eq "") {
                            continue
                        }
		
                        if ($_tmpUGItem -eq '*') {
                            if ($tmpCount -gt 1) {
                                continue
                            }
                            else {
                                $_tmpProcs = $_tmpUGItem + "," + $_tmpProcs
                            }
                        }
                        else {
                            $_tmpProcs = ($_tmpUGItem -replace [regex]::Escape("@CTXASSSEP@"), "|").Split("|")[2] + "," + $_tmpProcs
                        }
                    }
                    $_tmpProcs = $_tmpProcs.TrimEnd(',')
	
                    $_tmpRule1 = [PsCustomObject]@{'Index' = $Global:redirectRuleList.Count + 1; 'Type' = $curValues[1] + 1; 'Target' = $curValues[2]; 'TargetV' = $curValues[3]; 'DestV' = $curValues[9]; 'Destine' = $curValues[8]; 'Users' = $_tmpUsers; 'Groups' = $_tmpGroups; 'Comps' = $_tmpComps; 'OU' = $_tmpOUs; 'Procs' = $_tmpProcs }
                    $Global:redirectRuleList.Add($_tmpRule1) | Out-Null
                }
                else {
                    #if remaining changes of apps exists in memory, ignore those from reg

                    if (!($Global:RulesInMemoryAppList -contains $curApp)) {
                        [void]$Global:existingRuleList.Add($curValue)
                        [void]$curRuleListFromReg.Add($curValue)
                    }
                    else {
                        [void]$hideAppwithchanges.Add($curValue)
                    }
                }
            }
            
            $Global:RegProcessedFlag = 1
			


            if ($curRuleListFromReg.Count -gt 0) {
                #$uniqueList is in fixed size         
                $uniqueList = $curRuleListFromReg | select -Unique 
                $curRuleListFromReg.Clear()
                if ($uniqueList.Count -eq 1) {
                    [void]$curRuleListFromReg.Add($uniqueList)
                }
                else {
                    [void]$curRuleListFromReg.AddRange($uniqueList)
                }
               
                $uniqueList2 = $Global:existingRuleList | select -Unique 
                $Global:existingRuleList.Clear()

                if ($uniqueList2.Count -eq 1) {
                    [void]$Global:existingRuleList.Add($uniqueList2)
                }
                else {
                    [void]$Global:existingRuleList.AddRange($uniqueList2)
                }
            }


            #parse: <action type>|<target type>|<target path>| <target value>| <user and group list>|<process list>|<OU list>|<computer list>|<app name>
    
            foreach ($curRule in $curRuleListFromReg) {       
                $suspiciousRule = $curRule.Split('|')

                if (($suspiciousRule.Count -eq 9) -and ($suspiciousRule[0] -eq '0')) {
                    [void]$Global:existingHiddenAppNameList.Add($suspiciousRule[8])
                }
        
            } 
            foreach ($curRule in $hideAppwithchanges) {       
                $suspiciousRule = $curRule.Split('|')

                if (($suspiciousRule.Count -eq 9) -and ($suspiciousRule[0] -eq '0')) {
                    [void]$Global:existingHiddenAppNameList.Add($suspiciousRule[8])
                }       
            }

            if ($Global:existingHiddenAppNameList.Count -gt 0) {
                $uniqueListofHiddenApp = $Global:existingHiddenAppNameList | select -Unique
                $Global:existingHiddenAppNameList.Clear()
                if ($uniqueListofHiddenApp.Count -eq 1) {
                    [void]$Global:existingHiddenAppNameList.Add($uniqueListofHiddenApp)
                }
                else {
                    [void]$Global:existingHiddenAppNameList.AddRange($uniqueListofHiddenApp)
                }        
            }

 
            $found = $false 
            foreach ($tmpAppName in $Global:existingHiddenAppNameList) {

                for ($i = 0; $i -lt $Global:appList.Count; $i++) {
                    if ($Global:appList[$i].Name -eq $tmpAppName) {
                        $found = $true
                        $Global:appList.RemoveAt($i)
                        $i--
                        break
                    }
                }


                if ($found) {

                    $found = $false
                }
                if (!(Test-CTXIsGuid -StringGuid $tmpAppName)) {
                    if (!($Global:RulesInMemoryAppList -contains $tmpAppName)) {
                        $newval = [PsCustomObject]@{'Index' = $Global:appList.Count + 1; 'Status' = 'Configured & applied  '; 'Name' = $tmpAppName; 'InstallDir' = '' }
                    }
                    else {
                        $newval = [PsCustomObject]@{'Index' = $Global:appList.Count + 1; 'Status' = 'Configured  '; 'Name' = $tmpAppName; 'InstallDir' = '' }
                    }
                    $Global:appList.Add($newval) | Out-NUll 
                }
            } 		
        }
    }  
} 
function Common-GetInstalledApps {		
    param (
        [Parameter(Mandatory = $true)]
        [ref]$unInstalledKeyList					
    )
    foreach ($curScope in $Global:unisntallKeySearchScope) {   
        if (Test-Path $curScope) {
            $res = foreach ($key in (Get-ChildItem $curScope)) {              
                $tmp = $key.Name.split('\')[-1]


                if ($tmp.Contains('{')) {
                    if (($key.GetValue('DisplayName', '') -ne '') -and ($key.GetValue('SystemComponent', 0) -ne 1)) {

                        if (($Global:knownIgnoredAppsList -Contains $key.GetValue('DisplayName', '')) -or ($key.GetValue('DisplayName', '') -like 'Citrix Virtual Apps and Desktops*')) {
                            continue
                        }

                        #if already found 
                        if ($unInstalledKeyList.Value.ContainsKey($key.GetValue('DisplayName', ''))) {
                            continue
                        }

                        #if not found
                        else {                           
                            $curInstallPath = $key.GetValue('InstallLocation', '')
                            if ($curInstallPath -eq '') {                              
                                if ($key.GetValue('DisplayIcon', '') -eq '') {

                                }
                                else {
                                    #if the path is 'c:\users\public\a.exe',c:\users\public\a.exe, 'c:\users\public\a.exe 0' all works
                                    $curInstallPath = Split-Path -parent $key.GetValue('DisplayIcon', '')
                                }
                            
                            }

                            [void]$unInstalledKeyList.Value.Add($key.GetValue('DisplayName', ''), $curInstallPath)                         
                        }
                    }
                }
                else {
                    if (($Global:knownIgnoredAppsList -Contains $tmp) -or ($tmp -like 'Citrix Virtual Apps and Desktops*')) {
                        continue
                    }
                    if (($unInstalledKeyList.Value.ContainsKey($tmp)) -or ($key.GetValue('SystemComponent', 0) -eq 1) -or ($key.GetValue('DisplayName', '') -eq '')) {
                        continue
                    }
                    #if not exists, add it
                    else {
                        $curInstallPath = $key.GetValue('InstallLocation', '')
                        if ($curInstallPath -eq '') {                               
                            if ($key.GetValue('DisplayIcon', '') -eq '') {
                               
                            }
                            else {
                                #if the path is 'c:\users\public\a.exe',c:\users\public\a.exe, 'c:\users\public\a.exe 0' all works
                                $curInstallPath = Split-Path -parent $key.GetValue('DisplayIcon', '')
                            }                           
                        }

                        $curDisplayName = $key.GetValue('DisplayName', '')
                        if ($curDisplayName -ne '') {
                            $tmp = $curDisplayName
                        }
                        if ($unInstalledKeyList.Value.ContainsKey($tmp) -eq $false) {
                            [void]$unInstalledKeyList.Value.Add($tmp, $curInstallPath)
                        }
                         
                    }
                }
            }       
        } 

    } 


}
function Common-ShowAppList {
    param (

        [Parameter(Mandatory = $true)]
        [ref]$appList
	
			
    )
    if (-not $appList.Value) {

    }
    elseif ($appList.Value.Count -gt 1) {
        #Sort-Object returns list in fixed size 
        $sortedTmpList = $appList.Value | Sort-Object -Property Status, Name
        $appList.Value.Clear()
        for ($i = 0; $i -lt $sortedTmpList.Count; $i++) {
            $sortedTmpList[$i].Index = $i + 1
            if (!($sortedTmpList[$i].Status.contains(' '))) {
                $sortedTmpList[$i].Status = $sortedTmpList[$i].Status + '  '
            }
            
        }
        $appList.Value.AddRange($sortedTmpList) | Out-Null
    }
    elseif ($appList.Value.Count -eq 1) {
        $appList.Value[0].Index = 1
        $appList.Value[0].Status = $sortedTmpList[0].Status
    }
}
function Common-GetExeSidList {
    $Global:exeInfoList =
    Get-ItemProperty -Path "HKLM:\SOFTWARE\Microsoft\Windows\CurrentVersion\App Paths\*"  |
    Where-Object { $_."(default)" -ne $null } |
    Select-Object @{ expression = { $_.PSChildName }; label = 'Program' } , @{ expression = { $q + $_."(default)" + $q }; label = 'CommandLine' }
    
    $UninstallKeys = "HKLM:\Software\Microsoft\Windows\CurrentVersion\Uninstall", "HKLM:\SOFTWARE\Wow6432Node\Microsoft\Windows\CurrentVersion\Uninstall"
    $null = New-PSDrive -Name HKU -PSProvider Registry -Root Registry::HKEY_USERS -ErrorAction SilentlyContinue
    $UninstallKeys += Get-ChildItem HKU: -ErrorAction SilentlyContinue | Where-Object { $_.Name -match 'S-\d-\d+-(\d+-){1,14}\d+$' } | ForEach-Object { "HKU:\$($_.PSChildName)\Software\Microsoft\Windows\CurrentVersion\Uninstall" }
    $tmpList = foreach ($UninstallKey in $UninstallKeys) {
        Get-ChildItem -Path $UninstallKey -ErrorAction SilentlyContinue | Where { $_.PSChildName -match '^{[A-Z0-9]{8}-([A-Z0-9]{4}-){3}[A-Z0-9]{12}}$' } | Select-Object  @{n = 'Name'; e = { $_.GetValue('DisplayName') } }, @{n = 'GUID'; e = { $_.PSChildName } }
    }
    foreach ($t in $tmpList) {
        $Global:exeSidList.Add($t) | Out-NUll
    }
    $Global:exeSidList.Add([PsCustomObject]@{'Name' = '7-Zip 23.01 (x64)'; 'GUID' = '{23170F69-40C1-278A-1000-000100020000}'; }) | Out-NUll
    $Global:exeSidList.Add([PsCustomObject]@{'Name' = 'Google Drive'; 'GUID' = '{03E6C474-8D95-4C1B-9268-4AA3FA16DE4F}'; }) | Out-NUll
    $Global:exeSidList.Add([PsCustomObject]@{'Name' = 'Google Drive'; 'GUID' = '{51EF1569-67EE-4AD6-9646-E726C3FFC8A2}'; }) | Out-NUll
    $Global:exeSidList.Add([PsCustomObject]@{'Name' = 'Google Drive'; 'GUID' = '{8AD5CECD-DF0D-41C3-BA21-1E22114CC73C}'; }) | Out-NUll
    $Global:exeSidList.Add([PsCustomObject]@{'Name' = 'Google Drive'; 'GUID' = '{A8E52322-8734-481D-A7E2-27B309EF8D56}'; }) | Out-NUll  
    $Global:exeSidList.Add([PsCustomObject]@{'Name' = 'Google Drive'; 'GUID' = '{C973DA94-CBDF-4E77-81D1-E5B794FBD146}'; }) | Out-NUll 
    $Global:exeSidList.Add([PsCustomObject]@{'Name' = 'Google Drive'; 'GUID' = '{A8E52322-8734-481D-A7E2-27B309EF8D56}'; }) | Out-NUll 
    $Global:exeSidList.Add([PsCustomObject]@{'Name' = 'Google Drive'; 'GUID' = '{C973DA94-CBDF-4E77-81D1-E5B794FBD146}'; }) | Out-NUll
    $Global:exeSidList.Add([PsCustomObject]@{'Name' = 'Google Drive'; 'GUID' = '{CFE8B367-77A7-41D7-9C90-75D16D7DC6B6}'; }) | Out-NUll
    $Global:exeSidList.Add([PsCustomObject]@{'Name' = 'Google Drive'; 'GUID' = '{EE15C2BD-CECB-49F8-A113-CA1BFC528F5B}'; }) | Out-NUll
    $Global:exeSidList.Add([PsCustomObject]@{'Name' = 'Google Drive'; 'GUID' = '{F1196F08-BAFE-4C9C-AEE7-71C69DA5B818}'; }) | Out-NUll		
}
function WEM-GetInstalledApps {


    try {
        $outputAppList = New-Object -TypeName 'System.Collections.ArrayList'
        $tmpInstalledAppsFoundInUnInstallKey = @{}
        $tmpRulesInMemoryAppList = @{}
        $tmpExistingRuleList = New-Object -TypeName 'System.Collections.ArrayList'
        $tmpExistingHiddenAppNameList = New-Object -TypeName 'System.Collections.ArrayList'
        #1 Get-CTXWindowsAppx
        Get-CTXWindowsAppx -UninstalledKey ([ref]$tmpInstalledAppsFoundInUnInstallKey)
        #2 Get-CTXInstallApps
        Common-GetInstalledApps -unInstalledKeyList ([ref]$tmpInstalledAppsFoundInUnInstallKey)      
        foreach ($tmp in $tmpInstalledAppsFoundInUnInstallKey.Keys) {
            if (!(Test-CTXIsGuid -StringGuid $tmp)) {
                $val = [PsCustomObject]@{'Index' = $OutputAppList.Count + 1; 'Status' = 'Not configured  '; 'Name' = $tmp; 'InstallDir' = [string]($tmpInstalledAppsFoundInUnInstallKey[$tmp]) }
                $OutputAppList.Add($val) | Out-NUll 			
            }

        }    
        Common-ShowAppList -appList ([ref]$OutputAppList) 
        Common-GetExeSidList
        $extraInfo = [PSCustomObject]@{
            exeSidList  = $Global:exeSidList
            exeInfoList = $Global:exeInfoList	 
        }
        $retJson = [PSCustomObject]@{
            ErrorCode    = 0
            ErrorMessage = 'Success'
            AppList      = $OutputAppList
            ExtraInfo    = $extraInfo
        }
        $retJson = $retJson | ConvertTo-Json -Depth 4
        return $retJson
	
    }
    catch {
        $retJson = [PSCustomObject]@{
            ErrorCode    = $_.Exception.HResult
            ErrorMessage = $_.Exception.Message
            AppList      = $null  
            ExtraInfo    = $null
        }
        $retJson = $retJson | ConvertTo-Json
        return $retJson		
		
    }

}
function WEM-GenerateRawData {
    param (
        [Parameter(Mandatory = $true)]
        [string]$JsonData
    )

    $rawRule = ''
    $tmpRuleType = ''
    try {
        $objects = ConvertFrom-Json $JsonData

        $tmpAssList = New-Object -TypeName 'System.Collections.ArrayList'
        $tmpFileRegList = New-Object -TypeName 'System.Collections.ArrayList'
        $tmpAppName = ''
        $tmpRuleType = ''
		
        $destPath = ''
        $destVal = ''  
        foreach ($obj in $objects) {
		
            #for each app: if it is app access control 
            $tmpAssList.Clear()
            $tmpFileRegList.Clear()
            $tmpAppName = $obj.name
            $tmpRuleType = $obj.type
		
            $destPath = ''
            $destVal = ''

            # Process 'targetObjects'
            
            foreach ($target in $obj.targetObjects) {
                if ($tmpRuleType -eq '0') {
                    #app access control rules

                }
                elseif ($tmpRuleType -eq '1') {
                    #redirect
                    $destPath = $target.targetPath
                    $destVal = $target.targetValue
                }
                $tarType = '0'
                $tarValue = ''
                switch ($target.type) {					 
                    '0' { $tarType = 'File' }					
                    '3' { $tarType = 'Folder' }
                    '1' { $tarType = 'Registry key' }
                    '2' {
                        $tarType = 'Registry value'
                        $tarValue = $target.value
                    }
                }                
                $tmpFileRegList.Add([PsCustomObject]@{'Index' = 0; 'Type' = $tarType; 'Path' = $target.path; 'Value' = $tarValue; 'DestV' = $destVal; 'Destine' = $destPath }) | Out-NUll
            }
        
            # Process 'assignments'
            $hasADMachines = $false
            $hasNDJMachines = $false
            foreach ($exclusions in $obj.exclusions) {

                $assType = '0'
                $adSid = ''
                $scope = ''
                $exName = $exclusions.name
                switch ($exclusions.type) {					 
                    '0' {
                        $assType = 'User'
                        $adSid = "!" + $exclusions.SID
                        $exName = [regex]::Escape('EX>') + $exclusions.name
                    }					
                    '1' {
                        $assType = 'Group'
                        $adSid = "!" + $exclusions.SID
                        $exName = [regex]::Escape('EX>') + $exclusions.name
                    }
                    '2' {
                        $assType = 'Process'
                        $exName = "?" + $exclusions.name
                    }
                    '3' {
                        $assType = 'OU'
                        $adSid = $exclusions.SID 
                        $exName = [regex]::Escape('EX>') + $exclusions.name
                    }
                    '4' {
                        $assType = 'Machine catalog'
                        $scope = 'NDJ'
                        $exName = [regex]::Escape('EX>') + $exclusions.name
                    }					
                    '5' {
                        $assType = 'AD machine'
                        $adSid = $exclusions.SID
                        $hasADMachines = $true
                        $exName = [regex]::Escape('EX>') + $exclusions.name
                    }
                    #AAD/NDJ machine does not support whitelist
                    '6' {
                        continue
					
                    }				
                    '7' {
                        $assType = 'User'
                        $adSid = "!" + $exclusions.SID
                        $scope = 'NDJ'
                        $exName = [regex]::Escape('EX>') + $exclusions.name
                    }
                    '8' {
                        $assType = 'Group'
                        $adSid = "!" + $exclusions.SID
                        $scope = 'NDJ'
                        $exName = [regex]::Escape('EX>') + $exclusions.name
					
                    }						
                }
                $val = [PsCustomObject]@{'Index' = 0; 'Type' = $assType; 'SID' = $adSid; 'Name' = $exName; 'Scope' = $scope }
                $tmpAssList.Add($val) | Out-NUll

			
			
            }			
            foreach ($assignment in $obj.assignments) {

                $assType = '0'
                $adSid = ''
                $scope = ''
                switch ($assignment.type) {					 
                    '0' {
                        $assType = 'User'
                        $adSid = $assignment.SID
                    
                    }					
                    '1' {
                        $assType = 'Group'
                        $adSid = $assignment.SID
                    }
                    '2' {
                        $assType = 'Process'
					
                    }
                    '3' {
                        $assType = 'OU'
                        $adSid = $assignment.SID
                    }
                    '4' {
                        $assType = 'Machine catalog'
                        $scope = 'NDJ'
                    }					
                    '5' {
                        $assType = 'AD machine'
                        $adSid = $assignment.SID
                        $hasADMachines = $true
                    }
                    '6' {
                        $assType = 'AAD/NDJ machine'
                        $scope = 'NDJ'
                        $hasNDJMachines = $true
					
                    }				
                    '7' {
                        $assType = 'User'
                        $adSid = $assignment.SID
                        $scope = 'NDJ'
                    }
                    '8' {
                        $assType = 'Group'
                        $adSid = $assignment.SID
                        $scope = 'NDJ'
					
                    }						
                }
                $val = [PsCustomObject]@{'Index' = 0; 'Type' = $assType; 'SID' = $adSid; 'Name' = $assignment.name; 'Scope' = $scope }
                $tmpAssList.Add($val) | Out-NUll

			
			
            }
            #we processedall assignments, now heck AD* and NDJ* for machines
            if ($obj.enableConfigureADMachines -and ($hasADMachines -eq $false) ) {
                #AD*
                $val = [PsCustomObject]@{'Index' = 0; 'Type' = 'AD machine'; 'SID' = ''; 'Name' = 'AD*'; 'Scope' = '' }
                $tmpAssList.Add($val) | Out-Null
            }
            if ($obj.enableConfigureNDJMachines -and ($hasNDJMachines -eq $false)) {
                $val = [PsCustomObject]@{'Index' = 0; 'Type' = 'AAD/NDJ machine'; 'SID' = ''; 'Name' = 'NDJ*'; 'Scope' = 'NDJ' }	
                $tmpAssList.Add($val) | Out-Null
            }
            #app access control
            $rulesList = Common-GenerateAppAccessControlRules -appAssList $tmpAssList -curAppName $tmpAppName -appFileRegList ([ref]$tmpFileRegList) -ruleType $tmpRuleType
		
            foreach ($i in $rulesList) {
                $rawRule += $i
                $rawRule += "`n"
            } 		
        }
        $retJson = [PSCustomObject]@{
            ErrorCode    = 0
            ErrorMessage = 'Success'
            RawRule      = $rawRule 
        }
        $retJson = $retJson | ConvertTo-Json
        return $retJson	
    }
    catch {		
        $retJson = [PSCustomObject]@{
            ErrorCode    = $_.Exception.HResult
            ErrorMessage = $_.Exception.Message
            RawRule      = '' 
        }
        $retJson = $retJson | ConvertTo-Json
        return $retJson											
    }
    
}
function WEM-ReadRulesFromFile {
    param (
        [Parameter(Mandatory = $true)]
        [string]$RawRuledata
    )
    try {
        $tmpInstalledAppsFoundInUnInstallKey = @{}
        $tmpRulesInMemoryAppList = @{}
        $tmpExistingRuleList = New-Object -TypeName 'System.Collections.ArrayList'
        $tmpExistingHiddenAppNameList = New-Object -TypeName 'System.Collections.ArrayList'
        $OutputAppList = New-Object -TypeName 'System.Collections.ArrayList'
        $errorCode = '0'
        $errorMessage = 'success'

        Commont-GetRulesFromString -rawRuleData $RawRuledata -appList (([ref]$OutputAppList))

        $retJson = [PSCustomObject]@{
            ErrorCode    = 0
            ErrorMessage = 'Success'
            Rules        = $OutputAppList
        }
        $retJson = $retJson | ConvertTo-Json -Depth 5
        return $retJson		
    }
    catch {

        $retJson = [PSCustomObject]@{
            ErrorCode    = $_.Exception.HResult
            ErrorMessage = $_.Exception.Message
            Rules        = $null
        }
        $retJson = $retJson | ConvertTo-Json -Depth 5
        return $retJson					
    }
}
$init = {
    function Check-CTXImportantFileFolderPath {
        param(
            [Parameter(Mandatory = $true)][string]$testPath

        )
        $lowerTestPath = $testPath.TrimEnd('\') + '\'
        $lowerTestPath = $lowerTestPath.ToLower()
        foreach ($var in $Global:importantFileFolderPath) {
            if ($var.ToLower().Contains($lowerTestPath)) {
                write-host 'Protected path:' $testPath' skipped' -ForegroundColor Yellow
                return $true
            }
        }


        foreach ($var in $Global:importantAppPath) {
            if ($lowerTestPath -like $var) {
                write-host 'Protected path:' $testPath' skipped' -ForegroundColor Yellow
                return $true
            }
        }

        return $false
    }

    function  Get-CTXInstallKeyUninstallKey {
        param($appName)
        $installKeySearchScope = 'HKLM:\software\classes\installer\products'
        $fileRegList = New-Object -TypeName 'System.Collections.ArrayList'
        if (Test-Path $installKeySearchScope) {
            $res = foreach ($key in (Get-ChildItem $installKeySearchScope)) {
                if (($key.GetValue('ProductName') -like $appName) -and !(Check-CTXImportantRegPath -testPath $key.Name.Trim().TrimEnd('\'))) {        
                    #eg,KEY_LOCAL_MACHINE\software\classes\installer\products\023B5DCD9D98C7C4E9B84894568847E4
                    $fileRegList.Add([PsCustomObject]@{'Index' = 1; 'Type' = 'Registry key'; 'Path' = $key.Name.Trim().TrimEnd('\'); 'Value' = '' }) | Out-Null
                }   
            }
        }
	
        $unisntallKeySearchScope = @('HKLM:\SOFTWARE\Microsoft\Windows\CurrentVersion\Uninstall',
            'HKLM:\SOFTWARE\WOW6432Node\Microsoft\Windows\CurrentVersion\Uninstall',
            'HKCU:\SOFTWARE\Microsoft\Windows\CurrentVersion\Uninstall')
        foreach ($tmpKey in $unisntallKeySearchScope) {
            if (Test-Path $tmpKey) {
                $res = foreach ($key in (Get-ChildItem $tmpKey)) {

                    if (($key.GetValue('DisplayName') -like $appName) -and !(Check-CTXImportantRegPath -testPath $key.Name.Trim().TrimEnd('\'))) {
                        $fileRegList.Add([PsCustomObject]@{'Index' = 1; 'Type' = 'Registry key'; 'Path' = $key.Name.Trim().TrimEnd('\'); 'Value' = '' }) | Out-Null
                    }
                }
            }

        }

        return $fileRegList
    }
    function Get-CTXRegKeyUnderUserData {
        param($appName, $exeSidList)	
        $fileRegList = New-Object -TypeName 'System.Collections.ArrayList'   
        $installKeySearchScope2 = 'HKLM:\software\Microsoft\windows\currentversion\installer\userdata'
        $toMatch = $appName
        if ($appName -like 'Citrix Workspace *') {
            $toMatch = 'Citrix Workspace'

        }
        $sidList = $exeSidList |  Where-Object { ($_.Name -like ($toMatch + '*')) }
        if (Test-Path $installKeySearchScope2) {
            $res = foreach ($key in (Get-ChildItem $installKeySearchScope2)) {
                #HKEY_LOCAL_MACHINE\SOFTWARE\Microsoft\Windows\CurrentVersion\Installer\UserData\S-1-5-18
                $subKeyName = $installKeySearchScope2 + '\' + $key.Name.split('\')[-1] + '\Products\'  
                foreach ($tmp in $sidList) {
                    #eg, HKEY_LOCAL_MACHINE\SOFTWARE\Microsoft\Windows\CurrentVersion\Installer\UserData\S-1-5-18\Products\1A66E47F49A35C54A94E209149516CB2\InstallProperties
                    $subkeyName2 = $subKeyName + $tmp + '\InstallProperties'
                    if (Test-Path $subkeyName2) {
                        $subKey2 = Get-Item -Path $subkeyName2
                        $t = $subKeyName + $tmp
                        if (($subKey2.GetValue('DisplayName', '') -like $appName) -and !(Check-CTXImportantRegPath -testPath $t.Name.Trim().TrimEnd('\'))) {                            

                            $fileRegList.Add([PsCustomObject]@{'Index' = 1; 'Type' = 'Registry key'; 'Path' = $t3.Trim().TrimEnd('\'); 'Value' = '' }) | Out-Null

                        }  
                    }

                }

            }
        }	
        return $fileRegList	
			
    }

    function Get-CTXServiceDriverInfo {
        param([string]$appName, [string]$appChosenInstallPath)  
        $fileRegList = New-Object -TypeName 'System.Collections.ArrayList'
        $tmp = 'HKLM:\system\currentcontrolset\Services'
        if (($appChosenInstallPath -ne '') -and (Test-Path $tmp)) {
       
            $pattern = $appChosenInstallPath
            if ($appName -like 'Citrix Workspace *') {
                $pattern = $pattern.TrimEnd('\') + '\Citrix Workspace *'
            }		
            $res = foreach ($key in (Get-ChildItem $tmp)) {
                $matchapp = $key.GetValue('DisplayName') -like $appName
                $matchdir = $key.GetValue('ImagePath') -like ('*' + $pattern + '*')
                $conditions = $matchapp -or $matchdir
                
                
                if ($conditions -and !(Check-CTXImportantRegPath -testPath $key.Name.Trim().TrimEnd('\'))) {

                    $fileRegList.Add([PsCustomObject]@{'Index' = 1; 'Type' = 'Registry key'; 'Path' = $key.Name.Trim().TrimEnd('\'); 'Value' = '' }) | Out-Null
                }
                
            }
        }

        return $fileRegList
    }

    function Check-CTXImportantRegPath {
        param(
            [Parameter(Mandatory = $true)][string]$testPath

        )
        $importantRegPath = @(
            #upm
            'HKEY_LOCAL_MACHINE\SOFTWARE\Policies\Citrix\UserProfileManager\',
            'HKLM:\SOFTWARE\Policies\Citrix\UserProfileManager\',
            'HKEY_LOCAL_MACHINE\SOFTWARE\Citrix\UserProfileManager\',
            'HKLM:\SOFTWARE\Citrix\UserProfileManager\',
            #wem
            'HKEY_LOCAL_MACHINE\SOFTWARE\Policies\Norskale\',
            'HKEY_LOCAL_MACHINE\SOFTWARE\Citrix\WEM\',
            'HKEY_LOCAL_MACHINE\SYSTEM\CurrentControlSet\Control\Norskale\',
            'HKLM:\SOFTWARE\Policies\Norskale\',
            'HKLM:\SOFTWARE\Citrix\WEM\',
            'HKLM:\SYSTEM\CurrentControlSet\Control\Norskale\',
            #VDA
            'HKEY_LOCAL_MACHINE\SOFTWARE\Citrix\VirtualDesktopAgent\',
            'HKLM:\SOFTWARE\Citrix\VirtualDesktopAgent\',
            'HKEY_LOCAL_MACHINE\SOFTWARE\Microsoft\Windows\CurrentVersion\Uninstall\Citrix Virtual Desktop Agent\'
            'HKLM:\SOFTWARE\Microsoft\Windows\CurrentVersion\Uninstall\Citrix Virtual Desktop Agent\'
            #others
            'HKCU:\',
            'HKEY_CURRENT_USER\',
            'HKU:\',
            'HKEY_USERS\'

        )
        $upperTestPath = $testPath.TrimEnd('\') + '\'
        $upperTestPath = $upperTestPath.ToUpper()
        foreach ($var in $importantRegPath) {
            if ($var.ToUpper().Contains($upperTestPath)) {
                write-host 'Protected path:' $testPath' skipped' -ForegroundColor Yellow
                return $true
            }
        }
        return $false
    }
    function Get-CTXOthers {
        param([string]$appName, [string]$appChosenInstallPath, $exeSidList)  

        $fileRegList = New-Object -TypeName 'System.Collections.ArrayList'	
	
        #HKLM\SOFTWARE\Classes\<ChromeHTML>\Application, it has value 'ApplicationIcon' contains install dir
        if ($appChosenInstallPath -ne '') {

            $tmpFolderList = New-Object -TypeName 'System.Collections.ArrayList'
            if ($appName -like '*Google Chrome*') {
                $tmpFolderList.Add('HKLM:\SOFTWARE\Classes\ChromeHTML\Application')
            }

            foreach ($regKeyPath in $tmpFolderList) {   
                if (Test-Path $regKeyPath) {

                    $registryKey = Get-Item -Path $regKeyPath
                    $valueNames = $registryKey.GetValueNames()

                    foreach ($valueName in $valueNames) {
                        $valueData = $registryKey.GetValue($valueName)
                        $pattern = '*' + $appChosenInstallPath + '*'
                        if ($valueData -and ($valueData -like $pattern)) {
                            $t = $regKeyPath
                            if (!(Check-CTXImportantRegPath -testPath $t.Trim().TrimEnd('\'))) {
                                $fileRegList.Add([PsCustomObject]@{'Index' = 1; 'Type' = 'Registry key'; 'Path' = $t.Trim().TrimEnd('\'); 'Value' = '' }) | Out-Null
                            }                    
                        }
                    }
                } 
            }
        }


        #HKLM\Software\Classes\CLSID\{sid} HKLM\Software\Wow6432Node\Class\CLSID\{sid},HKCU\Software\Classes\CLSID\{sid} 
        #HKLM\Software\Classes\CLSID\{sid}\InprocServer32\, Default value('(Default)') contains installation folder, eg, C:\Program Files\VideoLAN\VLC\xx
        #HKLM\Software\Microsoft\Windows\CurrentVersion\Explorer\Browser Helper Objects\{sid}
        #HKLM\Software\Wow6432Node\Microsoft\Windows\CurrentVersion\Explorer\Browser Helper Objects\{sid}
        #HKLM\Software\Microsoft\Windows\CurrentVersion\Installer\UserData\S-1-5-18\Products\940D8C463B6731445AF18F5599F53BD8
        $toMatch = $appName
        if ($appName -like 'Citrix Workspace *') {
            $toMatch = 'Citrix Workspace'

        }
        $sidList = $exeSidList |  Where-Object { ($_.Name -like ($toMatch + '*')) }
        foreach ($curSid in $sidList) {
            $t = 'HKLM:\Software\Classes\CLSID\' + $curSid.GUID
            if (Test-Path $t) {
                if (!(Check-CTXImportantRegPath -testPath $t.Trim().TrimEnd('\'))) {
                    $fileRegList.Add([PsCustomObject]@{'Index' = 1; 'Type' = 'Registry key'; 'Path' = $t.Trim().TrimEnd('\'); 'Value' = '' }) | Out-Null				
                } 
            }
            $t = 'HKLM:\Software\Wow6432Node\Classes\CLSID\' + $curSid.GUID
            if (Test-Path $t) {
                if (!(Check-CTXImportantRegPath -testPath $t.Trim().TrimEnd('\'))) {
                    $fileRegList.Add([PsCustomObject]@{'Index' = 1; 'Type' = 'Registry key'; 'Path' = $t.Trim().TrimEnd('\'); 'Value' = '' }) | Out-Null			
                } 
            }
            $t = 'HKCU:\Software\Classes\CLSID\' + $curSid.GUID
            if (Test-Path $t) {
                if (!(Check-CTXImportantRegPath -testPath $t.Trim().TrimEnd('\'))) {
                    $fileRegList.Add([PsCustomObject]@{'Index' = 1; 'Type' = 'Registry key'; 'Path' = $t.Trim().TrimEnd('\'); 'Value' = '' }) | Out-Null			
                } 
            }

            $t = 'HKLM:\Software\Microsoft\Windows\CurrentVersion\Explorer\Browser Helper Objects\' + $curSid.GUID
            if (Test-Path $t) {
                if (!(Check-CTXImportantRegPath -testPath $t.Trim().TrimEnd('\'))) {
                    $fileRegList.Add([PsCustomObject]@{'Index' = 1; 'Type' = 'Registry key'; 'Path' = $t.Trim().TrimEnd('\'); 'Value' = '' }) | Out-Null			
                } 
            }     
            $t = 'HKLM:\Software\Wow6432Node\Microsoft\Windows\CurrentVersion\Explorer\Browser Helper Objects\' + $curSid.GUID
            if (Test-Path $t) {
                if (!(Check-CTXImportantRegPath -testPath $t.Trim().TrimEnd('\'))) {
                    $fileRegList.Add([PsCustomObject]@{'Index' = 1; 'Type' = 'Registry key'; 'Path' = $t.Trim().TrimEnd('\'); 'Value' = '' }) | Out-Null				
                } 
            }   

        }

        # findout HKLM\software\
        $keyUnderSoftware = 'HKLM:\software\' + $appName 
        $res = Test-Path $keyUnderSoftware
        if ($res -eq $True) { 
		    if(!(Check-CTXImportantRegPath -testPath $keyUnderSoftware.Trim().TrimEnd('\'))) {
                $fileRegList.Add([PsCustomObject]@{'Index' = 1; 'Type' = 'Registry key'; 'Path' = $keyUnderSoftware.Trim().TrimEnd('\'); 'Value' = '' }) | Out-Null		
			}
        }
		else {
			# here we do not have a complete match, so we have to further treat the appName string to see if somehow matches
			# First, remove the string in bracket, which usual specify platform, which is not needed
			$strippedName = $appName -replace "\(.*\)", ""
			$strippedName = $strippedName.TrimEnd(' ')

			# and check word by word
			$tokens = $strippedName -split " "
			
			# Second, strip case for like 1.1.1.1, which is for version
			$tokens = @($tokens | Where-Object { -not ($_ -match '^[0-9.]*$') })

			# combine the stripped parts
			$strippedName = $tokens -join " "
			
			# Do another check with the stripped name
			$keyUnderSoftware = 'HKLM:\software\' + $strippedName 
            $res = Test-Path $keyUnderSoftware
			if ($res -eq $True) { 
		        if(!(Check-CTXImportantRegPath -testPath $keyUnderSoftware.Trim().TrimEnd('\'))) {
                    $fileRegList.Add([PsCustomObject]@{'Index' = 1; 'Type' = 'Registry key'; 'Path' = $keyUnderSoftware.Trim().TrimEnd('\'); 'Value' = '' }) | Out-Null		
			    }
            }
			else {
			    # First, check the first part if matches
			    $keySoftwareLvl1 = 'HKLM:\software\' + $tokens[0]
			    $res = Test-Path $keySoftwareLvl1
			    
			    if ($res -eq $True) {
			    	# The first part matches, however, that may not be the one to add.
			    	# A common practice is company name in first level, and then software.
			    	# So proceed to check the next one
			    	$tmpAdded = $false
			    	if($tokens.Count -gt 1) {
			    	    foreach ($AppSubkey in (Get-ChildItem $keySoftwareLvl1)) {
							$AppSubkeyName = $AppSubkey.PSChildName
			    			$option1 = $tokens[1]
			    			$option2 = $tokens[0] + " " + $tokens[1]
                            if ($AppSubkeyName -eq $option1 -or $AppSubkeyName -eq $option2 -or $AppSubkeyName.StartsWith($tokens[1])) {
                                $t = 'HKLM:\SOFTWARE\' + $tokens[0] + '\' + $AppSubkeyName
                                if (!(Check-CTXImportantRegPath -testPath $t.Trim().TrimEnd('\'))) {
                                    $fileRegList.Add([PsCustomObject]@{'Index' = 1; 'Type' = 'Registry key'; 'Path' = $t.Trim().TrimEnd('\'); 'Value' = '' }) | Out-Null	
			    				    $tmpAdded = $true
                                    break									
                                }                                         
                            }
                        }
			    	}
			    	
					# if not added, we have to assume the first part represent this whole app.
					# Anyway, it is to be verified
					if(-not $tmpAdded) {
						$t = $keySoftwareLvl1
                        if (!(Check-CTXImportantRegPath -testPath $t.Trim().TrimEnd('\'))) {
                            $fileRegList.Add([PsCustomObject]@{'Index' = 1; 'Type' = 'Registry key'; 'Path' = $t.Trim().TrimEnd('\'); 'Value' = '' }) | Out-Null								
                        }  
					}
			    }
		    }
		}

        # findout HKLM\software\
        $keyUnderSoftware = 'HKLM:\software\WOW6432Node\' + $appName 
        $res = Test-Path $keyUnderSoftware
        if ($res -eq $True) { 
		    if(!(Check-CTXImportantRegPath -testPath $keyUnderSoftware.Trim().TrimEnd('\'))) {
                $fileRegList.Add([PsCustomObject]@{'Index' = 1; 'Type' = 'Registry key'; 'Path' = $keyUnderSoftware.Trim().TrimEnd('\'); 'Value' = '' }) | Out-Null		
			}
        }
		else {
			# here we do not have a complete match, so we have to further treat the appName string to see if somehow matches
			# First, remove the string in bracket, which usual specify platform, which is not needed
			$strippedName = $appName -replace "\(.*\)", ""
			$strippedName = $strippedName.TrimEnd(' ')

			# and check word by word
			$tokens = $strippedName -split " "
			
			# Second, strip case for like 1.1.1.1, which is for version
			$tokens = @($tokens | Where-Object { -not ($_ -match '^[0-9.]*$') })

			# combine the stripped parts
			$strippedName = $tokens -join " "
			
			# Do another check with the stripped name
			$keyUnderSoftware = 'HKLM:\software\WOW6432Node\' + $strippedName 
            $res = Test-Path $keyUnderSoftware
			if ($res -eq $True) { 
		        if(!(Check-CTXImportantRegPath -testPath $keyUnderSoftware.Trim().TrimEnd('\'))) {
                    $fileRegList.Add([PsCustomObject]@{'Index' = 1; 'Type' = 'Registry key'; 'Path' = $keyUnderSoftware.Trim().TrimEnd('\'); 'Value' = '' }) | Out-Null		
			    }
            }
			else {
			    # First, check the first part if matches
			    $keySoftwareLvl1 = 'HKLM:\software\WOW6432Node\' + $tokens[0]
			    $res = Test-Path $keySoftwareLvl1
			    
			    if ($res -eq $True) {
			    	# The first part matches, however, that may not be the one to add.
			    	# A common practice is company name in first level, and then software.
			    	# So proceed to check the next one
			    	$tmpAdded = $false
			    	if($tokens.Count -gt 1) {
			    	    foreach ($AppSubkey in (Get-ChildItem $keySoftwareLvl1)) {
							$AppSubkeyName = $AppSubkey.PSChildName
			    			$option1 = $tokens[1]
			    			$option2 = $tokens[0] + " " + $tokens[1]
                            if ($AppSubkeyName -eq $option1 -or $AppSubkeyName -eq $option2 -or $AppSubkeyName.StartsWith($tokens[1])) {
                                $t = 'HKLM:\software\WOW6432Node\' + $tokens[0] + '\' + $AppSubkeyName
                                if (!(Check-CTXImportantRegPath -testPath $t.Trim().TrimEnd('\'))) {
                                    $fileRegList.Add([PsCustomObject]@{'Index' = 1; 'Type' = 'Registry key'; 'Path' = $t.Trim().TrimEnd('\'); 'Value' = '' }) | Out-Null	
			    				    $tmpAdded = $true
                                    break									
                                }                                         
                            }
                        }
			    	}
			    	
					# if not added, we have to assume the first part represent this whole app.
					# Anyway, it is to be verified
					if(-not $tmpAdded) {
						$t = $keySoftwareLvl1
                        if (!(Check-CTXImportantRegPath -testPath $t.Trim().TrimEnd('\'))) {
                            $fileRegList.Add([PsCustomObject]@{'Index' = 1; 'Type' = 'Registry key'; 'Path' = $t.Trim().TrimEnd('\'); 'Value' = '' }) | Out-Null								
                        }  
					}
			    }
		    }
		}

        ##HKLM:\software\*\<app>
        #$indirect = @('HKLM:\SOFTWARE\')
        #if (Test-Path $indirect) {
        #    $res = foreach ($key in (Get-ChildItem $indirect)) {
        #        $res2 = foreach ($key2Name in $key.GetSubKeyNames()) {
        #            if ($key2Name -eq $appName) {
        #                $t = 'HKLM:\SOFTWARE\' + $key.Name.Split('\')[-1] + '\' + $key2Name 
        #                if (!(Check-CTXImportantRegPath -testPath $t.Trim().TrimEnd('\'))) {
        #                    $fileRegList.Add([PsCustomObject]@{'Index' = 1; 'Type' = 'Registry key'; 'Path' = $t.Trim().TrimEnd('\'); 'Value' = '' }) | Out-Null						
        #                }                                         
        #            }
        #        }
        #    }
        #}


        return 	$fileRegList
    }

    function Get-CTXRegKeyUnderSoftware {
        param($appName)

        $fileRegList = New-Object -TypeName 'System.Collections.ArrayList'	
	
        $regPath = 'HKLM:\SOFTWARE\'
        if (Test-Path $regPath) {
            $res = foreach ($key in (Get-ChildItem $regPath)) {
                $res2 = foreach ($key2Name in $key.GetSubKeyNames()) {
                    if ($key2Name -eq $appName) {
                        $t = $regPath + $key.Name.Split('\')[-1] + '\' + $key2Name 
                        if (!(Check-CTXImportantRegPath -testPath $t.Trim().TrimEnd('\'))) {
                            $fileRegList.Add([PsCustomObject]@{'Index' = 1; 'Type' = 'Registry key'; 'Path' = $t.Trim().TrimEnd('\'); 'Value' = '' }) | Out-Null						
                        }                                         
                    }
                }
            }
        } 


        return 	$fileRegList
    }

    function Get-CTXShotcuts {   
        param([string]$appName, [string]$appChosenInstallPath, $fileRegList)  
        $pattern = '*' + $appName + '*'


        $tmpShotcutList = New-Object -TypeName 'System.Collections.ArrayList'
        if (Test-Path 'c:\users') {
            $curList = Get-ChildItem 'c:\users'
            foreach ($curPath in $curList) {
                #C:\Users\*\AppData\Roaming\Microsoft\Windows\Start Menu\Programs\app.ink 
                #C:\Users\*\AppData\Roaming\Microsoft\Windows\Start Menu\Programs\app
                $shortCutInStartMenu = 'c:\users\' + $curPath + '\AppData\Roaming\Microsoft\Windows\Start Menu\Programs\'
                Get-CTXShotcutImpl -testPath $shortCutInStartMenu -addFolder $false  -appName $appName -fileRegList $fileRegList -ShotcutList ([ref]$tmpShotcutList)
                #C:\Users\*\AppData\Roaming\Microsoft\Windows\Start Menu\Programs\<company>\
                $shortCutInStartMenu = 'c:\users\' + $curPath + '\AppData\Roaming\Microsoft\Windows\Start Menu\Programs\'       
                $res = Test-Path $shortCutInStartMenu
                if ($res -eq $True) {
                    $tmpList = Get-ChildItem $shortCutInStartMenu
                    foreach ($potentialCompany in $tmpList) {
                        if (($potentialCompany.Name -like $pattern) -or ($appName.ToLower().Contains($potentialCompany.Name.ToLower()))) {
                            $curFullpath = $shortCutInStartMenu + $potentialCompany.Name + '\'
                            Get-CTXShotcutImpl -testPath $curFullpath -addFolder $true -appName $appName -fileRegList $fileRegList -ShotcutList ([ref]$tmpShotcutList)
                        }

                    }
                }
                #C:\Users\*\AppData\Roaming\Microsoft\Windows\Start Menu\Programs\CloudSGApps
                if ($Global:targetApp -like 'Citrix Workspace *') {
                    $shortCutInStartMenu = 'c:\users\' + $curPath + '\AppData\Roaming\Microsoft\Windows\Start Menu\Programs\CloudSGApps\'
                    Get-CTXShotcutImpl -testPath $shortCutInStartMenu -addFolder $true  -appName $appName -fileRegList $fileRegList -ShotcutList ([ref]$tmpShotcutList)
                }
             
                  
                #Desktop\app.lnk
                $shortCutInDesktop = 'c:\users\' + $curPath + '\Desktop\'
                Get-CTXShotcutImpl -testPath $shortCutInDesktop -addFolder $false  -appName $appName -fileRegList $fileRegList -ShotcutList ([ref]$tmpShotcutList)


                #AppData\Roaming\Microsoft\Internet Explorer\Quick Launch\app.lnk
                $shortCutInExplorer = 'c:\users\' + $curPath + '\AppData\Roaming\Microsoft\Internet Explorer\Quick Launch\'
                Get-CTXShotcutImpl -testPath $shortCutInExplorer -addFolder $false  -appName $appName -fileRegList $fileRegList -ShotcutList ([ref]$tmpShotcutList)

                #AppData\Roaming\Microsoft\Internet Explorer\Quick Launch\User Pinned\app.lnk
                $shortCutInExplorer = 'c:\users\' + $curPath + '\AppData\Roaming\Microsoft\Internet Explorer\Quick Launch\User Pinned\'
                Get-CTXShotcutImpl -testPath $shortCutInExplorer -addFolder $false  -appName $appName -fileRegList $fileRegList -ShotcutList ([ref]$tmpShotcutList)

                #AppData\Roaming\Microsoft\Internet Explorer\Quick Launch\User Pinned\TaskBar\app.lnk
                $shortCutInExplorer = 'c:\users\' + $curPath + '\AppData\Roaming\Microsoft\Internet Explorer\Quick Launch\User Pinned\TaskBar\'
                Get-CTXShotcutImpl -testPath $shortCutInExplorer -addFolder $false  -appName $appName -fileRegList $fileRegList -ShotcutList ([ref]$tmpShotcutList)

                #Appdata\roaming\Microsoft\Windows\Start Menu\Programs\Startup
                $shortCutInExplorer = 'c:\users\' + $curPath + '\AppData\Roaming\Microsoft\Windows\Start Menu\Programs\Startup\'
                Get-CTXShotcutImpl -testPath $shortCutInExplorer -addFolder $false  -appName $appName -fileRegList $fileRegList -ShotcutList ([ref]$tmpShotcutList)

            }
        }
        #C:\ProgramData\Microsoft\Windows\Start Menu\Programs\app.lnk
        #C:\ProgramData\Microsoft\Windows\Start Menu\Programs\app
        $shortCutInStartMenu = 'C:\ProgramData\Microsoft\Windows\Start Menu\Programs\'
        Get-CTXShotcutImpl -testPath $shortCutInStartMenu -addFolder $false  -appName $appName -fileRegList $fileRegList -ShotcutList ([ref]$tmpShotcutList)
    

        #C:\ProgramData\Microsoft\Windows\Start Menu\Programs\<company>\
        $shortCutInStartMenu = 'C:\ProgramData\Microsoft\Windows\Start Menu\Programs\'      
        $res = Test-Path $shortCutInStartMenu
        $pattern = $appName
        if ($appName -like 'putty *') {
            $pattern = 'putty *'
        }
        if ($res -eq $True) {
            $tmpList = Get-ChildItem $shortCutInStartMenu
            foreach ($potentialCompany in $tmpList) {
                if (($potentialCompany.Name -like $pattern) -or ($appName.ToLower().Contains($potentialCompany.Name.ToLower()))) {
                    $curFullpath = $shortCutInStartMenu + $potentialCompany.Name + '\'
                    Get-CTXShotcutImpl -testPath $curFullpath -addFolder $true  -appName $appName -fileRegList $fileRegList -ShotcutList ([ref]$tmpShotcutList)
                }

            }
        }
	
        $finalShotcutList = New-Object -TypeName 'System.Collections.ArrayList'
        foreach ($curShotcut in $tmpShotcutList) {
            if (($finalShotcutList.Count -eq 0) -or (!($finalShotcutList.Path -contains $curShotcut.path))) {
                $finalShotcutList.Add([PsCustomObject]@{'Index' = 1; 'Type' = $curShotcut.Type; 'Path' = $curShotcut.Path; 'Value' = '' }) | Out-Null
            }
        }
        return $finalShotcutList
    
    }
    function Get-CTXRegValues {
        param([string]$appName, [string]$appChosenInstallPath)  
	
        $fileRegList = New-Object -TypeName 'System.Collections.ArrayList'
	
        #HKLM\Software\Microsoft\Windows\CurrentVersion\Run, it has a reg value and its data contains installation path
        #similarly, HKCU\SOFTWARE\Microsoft\Windows\CurrentVersion\Run
        $tmpList = @('HKLM:\SOFTWARE\Microsoft\Windows\CurrentVersion\Run', 'HKCU:\SOFTWARE\Microsoft\Windows\CurrentVersion\Run')
        if ($appChosenInstallPath -ne '') {
            foreach ($regKeyPath in $tmpList) {   
                if (Test-Path $regKeyPath) {

                    $registryKey = Get-Item -Path $regKeyPath
                    $valueNames = $registryKey.GetValueNames()

                    foreach ($valueName in $valueNames) {
                        $valueData = $registryKey.GetValue($valueName)
                        $matched = $false
                        if ($valueData) {
                            $matched = $valueData -like ('*' + $appChosenInstallPath + '*')
                        }
                        if ((!$matched) -and ($appName -ne '')) {
                            $matched = $valueData -like ('*' + $appName + '*')
                        }
                        # Check if the value data exists and contains the specified string
                        if ($matched) {
                            $t = $regKeyPath + '\' + $valueName
                            if (!(Check-CTXImportantRegPath -testPath $t.Trim().TrimEnd('\'))) {
                                $fileRegList.Add([PsCustomObject]@{'Index' = 1; 'Type' = 'Registry value'; 'Path' = $regKeyPath.Trim().TrimEnd('\'); 'Value' = $valueName }) | Out-Null
                            }

                        }
                    } 
                }


            }
        }
        return $fileRegList
    }
    function Get-CTXInfoUnderShellPath {
        param([string]$targetExe, [string]$regPath)  

        $fileRegList = New-Object -TypeName 'System.Collections.ArrayList'
	
        # HKLM\Software\Classes\<xx>\shell\<yy>\command: the default value 'Default' contains '\<Executable>.'  
        if ($targetExe -ne "") {
            if (Test-Path $regPath) {
                $res = foreach ($key in (Get-ChildItem $regPath)) {
                    $t = $regPath + $key.Name.Split('\')[-1] + '\shell'
                    if (Test-Path $t) {
                        $res2 = foreach ($key2 in (Get-ChildItem $t)) {
                            $t2 = $regPath + $key.Name.Split('\')[-1] + '\shell\' + $key2.Name.Split('\')[-1] + '\command'
                            $pattern = '*\' + $targetExe + '.*'
                            if (Test-Path $t2) {
                                $value = Get-ItemProperty -Path $t2 | Select-Object -ExpandProperty 'Default' -ErrorAction SilentlyContinue
                                if ($value -and $value -like $pattern) {
                                    if (!(Check-CTXImportantRegPath -testPath $t2.Trim().TrimEnd('\'))) {

                                        $fileRegList.Add([PsCustomObject]@{'Index' = 1; 'Type' = 'Registry key'; 'Path' = $t2.Trim().TrimEnd('\'); 'Value' = '' }) | Out-Null
                                    } 
                                }

                            }

                        }
                    }

                }
            }
        }

        return $fileRegList
    } 
    #find out  $testPath\app or $testPath\app.lnk, $testPath ends with '\'
    function Get-CTXShotcutImpl {
        param(
            [Parameter(Mandatory = $true)][string]$testPath,
            [Parameter(Mandatory = $true)][bool]$addFolder,
            [Parameter(Mandatory = $true)][string]$appName,
            [Parameter(Mandatory = $true)]$fileRegList,
            [Parameter(Mandatory = $true)][ref]$ShotcutList
        )
        if (($testPath -eq $null) -or ($testPath -eq '')) {
            return 
        }
        $blnkFound = $false
        $existCnt = $ShotcutList.Value.count
        $sh = New-Object -ComObject WScript.Shell
    

        $res = Test-Path $testPath
        if ($res -eq $True) {
            #1 add $testPath\*.lnk that has target involved
            $potentialShotcutList = Get-ChildItem $testPath | Where-Object Name -Like *.lnk
            if ($potentialShotcutList.Count -gt 0) {
                foreach ($t in $potentialShotcutList) {
                    $blnkFound = $false

                    $tmpPath = $testPath + $t.Name.Trim().TrimEnd('\')
                    $target = $sh.CreateShortcut($tmpPath).TargetPath

                    #target is not empty, and target is already found or link name is app name, then this link is what we want
                    $tmpMatch = $target -ne ''
                    if ($appName -like 'Citrix Workspace *') {
                        $tmpMatch = $tmpMatch -and ($t.Name -like 'Citrix Workspace*')
                    }
                    if ($tmpMatch) {   

                        if ($fileRegList.Count -gt 0) {
                            foreach ($curPath in $fileRegList) {
                                if ($curPath.Type -ne 'Folder') {
                                    continue
                                }
                                $lowercurPath = $curPath.Path.TrimEnd('\').ToLower()
                                $lowerTarget = $target.TrimEnd('\').ToLower()

                                if ($lowercurPath -eq $lowerTarget) {
                                    $blnkFound = $true
                                    break
                                }
                                elseif ($lowerTarget.Contains($lowercurPath) -and $lowerTarget.Contains($lowercurPath + '\')) {
                                    $blnkFound = $true
                                    break
                                }

                            }
                        }

                        else {
                        
                            continue
                        }
                        if (!$blnkFound) {
                            continue
                        }
                        #lnk found now                                                     
                        if (!(Check-CTXImportantFileFolderPath -testPath $tmpPath)) {
                            $ShotcutList.Value.Add([PsCustomObject]@{'Index' = 1; 'Type' = 'File'; 'Path' = $tmpPath; 'Value' = '' }) | Out-Null
                        }

                    
                    }
                }           
            }

            #2 add sub folder with name $Global:targetApp
            $cur = (Get-ChildItem $testPath | Where-Object Name -eq $Global:targetApp)
            if ($cur -ne $null) {
                $ShotcutList.Value.Add([PsCustomObject]@{'Index' = 1; 'Type' = 'Folder'; 'Path' = $testPath + $appName; 'Value' = '' }) | Out-Null
            }

        }
        #add the whole folder if anything found in it
        if ($addFolder -and ($ShotcutList.count -gt $existCnt) ) {
            $ShotcutList.Value.Add([PsCustomObject]@{'Index' = 1; 'Type' = 'Folder'; 'Path' = $testPath.TrimEnd('\'); 'Value' = '' }) | Out-Null
        }



    }
    function Get-CTXExePaths {
        param([string]$appName, [string]$targetExe)  
        $fileRegList = New-Object -TypeName 'System.Collections.ArrayList'
        # HKLM\Software\Classes\<exe>.*  HKCU\Software\Classes\<exe>.*   HKLM\Software\Classes\<xx>\shell\<yy>\command: the default value 'Default' contains '\<Executable>.'  HKLM\Software\Microsoft\Windows\CurrentVersion\Run
        # HKCU\Software\Classes\Application\<exe>.* 

        $tmpExe = $targetExe
        if ($tmpExe -ne '') {

            $tmpScope = @('HKLM:\SOFTWARE\Classes\', 'HKCU:\SOFTWARE\Classes\')
            foreach ($tmpPath in $tmpScope) {
                if (Test-Path $tmpPath) {
                    $res = foreach ($key in (Get-ChildItem $tmpPath)) {
                        if ($key.Name.Split('\')[-1] -like ($tmpExe + '*') ) {
                            $t = $tmpPath + $key.Name.Split('\')[-1]
						
                            if (!(Check-CTXImportantRegPath -testPath $t.Trim().TrimEnd('\'))) {
                                $fileRegList.Add([PsCustomObject]@{'Index' = 1; 'Type' = 'Registry key'; 'Path' = $t.Trim().TrimEnd('\'); 'Value' = '' }) | Out-Null
                            } 
                        }

                    }
                }
            }






            $tmpPath = 'HKCU:\SOFTWARE\Classes\Application\'
            if (Test-Path $tmpPath) {
                $res = foreach ($key in (Get-ChildItem $tmpPath)) {
                    if ($key.Name.Split('\')[-1] -like ($tmpExe + '*') ) {
                        $t = $tmpPath + $key.Name.Split('\')[-1]
                        if (!(Check-CTXImportantRegPath -testPath $t.Trim().TrimEnd('\'))) {
                            $fileRegList.Add([PsCustomObject]@{'Index' = 1; 'Type' = 'Registry key'; 'Path' = $t.Trim().TrimEnd('\'); 'Value' = '' }) | Out-Null
                        } 
                    }
                }
            }

        }
 
        if ($appName -like 'WPS Office*') {
            $tmpScope = @('HKLM:\SOFTWARE\Classes\', 'HKCU:\SOFTWARE\Classes\')
            #eg, HKCU\SOFTWARE\Classes\Equation.*
            $tmpList = @('Equation.', 'ET.', 'KET.', 'ksoqing', 'KWPP.', 'KWPS.', 'WPP.')
            foreach ($tmpPath in $tmpScope) {
                if (Test-Path $tmpPath) {
                    $res = foreach ($key in (Get-ChildItem $tmpPath)) {
                        foreach ($tmp in $tmpList) {
                            if ($key.Name.Split('\')[-1] -like ($tmp + '*') ) {
                                $t = $tmpPath + $key.Name.Split('\')[-1]
                                if (!(Check-CTXImportantRegPath -testPath $t.Trim().TrimEnd('\'))) {
                                    $fileRegList.Add([PsCustomObject]@{'Index' = 1; 'Type' = 'Registry key'; 'Path' = $t.Trim().TrimEnd('\'); 'Value' = '' }) | Out-Null
                                } 
                            }
                        }


                    }
                }              
            }



        }
        $t = ''
        $t2 = ''
        switch -Wildcard ($appName) {
            "*Adobe Acrobat*" {
                $t = 'HKLM:\SOFTWARE\Classes\launchreader'
            }
            "*Azure Data Studio*" {
                $t = 'HKLM:\SOFTWARE\Classes\azuredatastudioSourceFile'
            }
            "*Microsoft Edge*" {
                $t = 'HKLM:\SOFTWARE\Classes\MSEdgeHTM'
                $t2 = 'HKLM:\SOFTWARE\Classes\MSEdgeMHT'

            }
            "*Mozilla Firefox*" {
                $t = 'HKLM:\SOFTWARE\Classes\azuredatastudioSourceFile'
            }
            "*Azure Data Studio*" {
                $t = 'HKLM:\SOFTWARE\Classes\azuredatastudioSourceFile'
            }


            Default {
        
            }
        }
        if (($t -ne '') -and (Test-Path $t)) {
            $fileRegList.Add([PsCustomObject]@{'Index' = 1; 'Type' = 'Registry key'; 'Path' = $t.Trim().TrimEnd('\'); 'Value' = '' }) | Out-Null
        }
        if (($t2 -ne '') -and (Test-Path $t2)) {
            $fileRegList.Add([PsCustomObject]@{'Index' = 1; 'Type' = 'Registry key'; 'Path' = $t2.Trim().TrimEnd('\'); 'Value' = '' }) | Out-Null
        }
        <# this is time-consuming, ignore this region here currently
        #HKCU\SOFTWARE\Classes\<xx> ,<xx>  has 'DefaultIcon' key which has 'Default' value that contains installation path 
        $tmpPath = 'HKCU:\SOFTWARE\Classes\'
        if (($appChosenInstallPath -ne '') -and (Test-Path $tmpPath)) {
            $res = foreach ($key in (Get-ChildItem $tmpPath)) {
                $subkeyPath = "HKCU:\SOFTWARE\Classes\$($key.PSChildName)\DefaultIcon"

                if (Test-Path -Path $subkeyPath) {
                    # Get the default value of the DefaultIcon key
                    $defaultValue = (Get-ItemProperty -Path $subkeyPath -Name "(Default)")."(Default)"
        
                    # Check if the default value matches the desired path
                    if ($defaultValue.'(Default)' -like ('*' + $appChosenInstallPath + '*')) {
                        $t = 'HKCU:\SOFTWARE\Classes\' + $key.Name.Split('\')[-1]
                        if (!(Check-CTXImportantRegPath -testPath $t.Trim().TrimEnd('\'))) {
                        $paths += $t
                        } 
                    }
                }

            }
        } #>
        return $fileRegList
		
    }

}
function Common-GetAppRelatedPath {
    param(
        [Parameter(Mandatory = $true)]
        [string]$appName,
        [Parameter(Mandatory = $true)]
        [ref]$appChosenInstallPath,
        [Parameter(Mandatory = $true)]
        [ref]$targetExecutable,
        [Parameter(Mandatory = $true)]
        $exeInfoList,
        [Parameter(Mandatory = $true)]
        $exeSidList
    )       		
		

	
    $tmpFileRegList = New-Object -TypeName 'System.Collections.ArrayList'
	
    #get other paths
    if ($appChosenInstallPath.Value -ne '') {
        if (!(Test-Path -Path $appChosenInstallPath.Value -PathType Container)) {
            $appChosenInstallPath.Value = Split-Path -Path $appChosenInstallPath.Value -Parent
        }
        if ( !(Check-CTXImportantFileFolderPath -testPath $appChosenInstallPath.Value)) {
            $tmpFileRegList.Add([PsCustomObject]@{'Index' = $tmpFileRegList.Count + 1; 'Type' = 'Folder'; 'Path' = $appChosenInstallPath.Value; 'Value' = '' }) | Out-NUll
        }	

        # if UWP, stop here
        if ($appChosenInstallPath.Value -like 'C:\Program Files\WindowsApps\*') {
            #C:\Program Files\WindowsApps\%PackageName%_%SID%  
            #C:\ProgramData\Packages\%PackageName%*_%SID%  
            #C:\ProgramData\Microsoft\Windows\AppRepository\Packages\%PackageName%*_%SID%.pckgdep#
            #package name or more than package name
            $packageName = ($appChosenInstallPath.Value.Split('\')[-1]).Split('__')[0]
            $sid = ($appChosenInstallPath.Value.Split('\')[-1]).Split('__')[-1]
            $tmpList = @('C:\ProgramData\Packages\', 'C:\ProgramData\Microsoft\Windows\AppRepository\Packages' )
            foreach ($folder in  $tmpList) {
                $subFolders = Get-ChildItem -Path $folder -Directory
                # Loop through each subfolder and check if its name matches the pattern
                foreach ($subFolder in $subFolders) {
                    $packageName2 = $subFolder.Name.Split('_')[0]
                    $sid2 = $subFolder.Name.Split('_')[-1]
                    $packageMatched = ($packageName2 -like ($packageName + '*')) -or ( $packageName -like ($packageName2 + '*'))
                    if ($sid -ne '') {
                        $sidMatched = $sid -eq $sid2
                        $packageMatched = $packageMatched -and $sidMatched

                    }
                    if ($packageMatched) {
                        $t = $folder + $subFolder.Name
                        if (!(Check-CTXImportantFileFolderPath -testPath $t)) {
                            $tmpFileRegList.Add([PsCustomObject]@{'Index' = $tmpFileRegList.Count + 1; 'Type' = 'Folder'; 'Path' = $t; 'Value' = '' }) | Out-NUll
                        }
                    }
                }


            }
			
            return $tmpFileRegList
        }


		
    }


 
    $installDir = $appChosenInstallPath.Value

    # get executable name if found,eg, everything.exe, then executable name is everything
    switch -Wildcard ($appName) {
        "Citrix Workspace*" {
            $targetExecutable.Value = 'Citrix Workspace'
        }
        "*Microsoft Visual Studio Code*" {
            $targetExecutable.Value = 'VSCode'
        }
        "Everything *" {
            $targetExecutable.Value = 'Everything'
        }
        "*Microsoft Access Runtime*" {
            $targetExecutable.Value = 'Access'
        }
        "*XenCenter*" {
            $targetExecutable.Value = 'XenCenter'
        }
        "Zoom*" {
            $targetExecutable.Value = 'Zoom'
        }
        "Slack" {
            $targetExecutable.Value = 'Slack'
        }
        "WPS Office*" {
            $targetExecutable.Value = 'WPS'
        }
        "Google Drive*" {
            $targetExecutable.Value = 'GoogleDrive'
        }
        Default {
            if (($targetExecutable.Value -eq '') -and ($appChosenInstallPath.Value -ne '')) {

                $tmp = $exeInfoList | Where-Object { $_.CommandLine -like "*$installDir*" }
                if ($tmp.Count -ne 0) {
                    foreach ($t in $tmp) {
                        if (($tmp[0].Program.Split('.')[0] -like ($appName + '*')) -or ($appName -like ('*' + $tmp[0].Program.Split('.')[0] + '*'))) {
                            $targetExecutable.Value = $tmp[0].Program.Split('.')[0]
                            break
                        }
                    }
                
                }
            }
        }
    }
		
    $jobs = @()	

    $targetExe = $targetExecutable.Value
    $tmpFileRegListForFileFolders = New-Object -TypeName 'System.Collections.ArrayList'
    $tmpFileRegListForFileFolders.AddRange($tmpFileRegList)
    $jobs += start-threadjob -init $init -ScriptBlock {
        param($appName)
        Get-CTXInstallKeyUninstallKey -appName $appName 
    } -ArgumentList $appName
  
  
    $jobs += start-threadjob -init $init -ScriptBlock {
        param($appName, $exeSidList)
        Get-CTXRegKeyUnderUserData -appName $appName -exeSidList $exeSidList
    } -ArgumentList $appName, $exeSidList

    $jobs += start-threadjob -init $init -ScriptBlock {
        param($appName, $installDir)
        Get-CTXServiceDriverInfo -appName $appName -appChosenInstallPath $installDir
    } -ArgumentList $appName, $installDir
  
    $jobs += start-threadjob -init $init -ScriptBlock {
        param($appName, $targetExe)
        Get-CTXExePaths -appName $appName -targetExe $targetExe
    } -ArgumentList $appName, $targetExe 
    <# time-consuming, skip now
$regPath = 'HKLM:\SOFTWARE\Classes\'
    $jobs += start-threadjob -init $init -ScriptBlock {
param($targetExe, $regPath)
 Get-CTXInfoUnderShellPath -targetExe $targetExe -regPath $regPath
  } -ArgumentList $targetExe, $regPath 
$regPath = 'HKCU:\SOFTWARE\Classes\'
    $jobs += start-threadjob -init $init -ScriptBlock {
param($targetExe, $regPath)
 Get-CTXInfoUnderShellPath -targetExe $targetExe -regPath $regPath
  } -ArgumentList $targetExe, $regPath 
#>
    $jobs += start-threadjob -init $init -ScriptBlock {
        param($appName, $installDir, $exeSidList )
        Get-CTXOthers -appName $appName -appChosenInstallPath $installDir -exeSidList $exeSidList
    } -ArgumentList $appName, $installDir, $exeSidList

    $jobs += start-threadjob -init $init -ScriptBlock {
        param($appName)
        Get-CTXRegKeyUnderSoftware -appName $appName 
    } -ArgumentList $appName 

    $jobs += start-threadjob -init $init -ScriptBlock {
        param($appName, $installDir)
        Get-CTXRegValues -appName $appName -appChosenInstallPath $installDir
    } -ArgumentList $appName, $installDir
  
    $jobs += start-threadjob -init $init -ScriptBlock {
        param($appName, $installDir, $tmpFileRegListForFileFolders)
        Get-CTXShotcuts -appName $appName -appChosenInstallPath $installDir -fileRegList $tmpFileRegListForFileFolders
    } -ArgumentList $appName, $installDir, $tmpFileRegListForFileFolders 

  

    $jobs | Wait-Job
    $results = $jobs | ForEach-Object { Receive-Job $_ }



    foreach ($t in $results) {
        if (($t -ne $null ) -and ($t -ne '')) {
            $tmpFileRegList.Add([PsCustomObject]@{'Index' = $tmpFileRegList.Count + 1; 'Type' = $t.Type; 'Path' = $t.Path; 'Value' = $t.Value }) | Out-Null
        }
		
    }
	
    return $tmpFileRegList
}

function WEM-GetDetailedAppInfo {
    param (
        [Parameter(Mandatory = $true)]
        [string]$AppName,
        [Parameter(Mandatory = $false)]
        [string]$AppInstallPath = "",		
        [Parameter(Mandatory = $true)]
        $ExtraInfo	
    )
    # Both ThreadJob and Microsoft.PowerShell.ThreadJob have the command Start-ThreadJob. But ThreadJob has been deprecated.
    # The latest version of ThreadJob (2.1.0) requires Microsoft.PowerShell.ThreadJob as dependency.
    if (-not (Get-Command "Start-ThreadJob" -ErrorAction SilentlyContinue)) {
        try {
            Import-Module -Name Microsoft.PowerShell.ThreadJob -ErrorAction Stop -WarningAction SilentlyContinue -Force | Out-Null
        }
        catch {
            try {
                Import-Module -Name ThreadJob -ErrorAction Stop -WarningAction SilentlyContinue -Force | Out-Null
            }
            catch {
                if ((Get-PackageProvider -Name NuGet -Force).Version -lt '2.8.5.201') {
                    Install-PackageProvider -Name NuGet -MinimumVersion 2.8.5.201 -Scope CurrentUser -Force | Out-Null
                }
                Install-Module Microsoft.PowerShell.ThreadJob -Scope CurrentUser -ErrorAction Stop -WarningAction SilentlyContinue -Force | Out-Null	
                Import-Module Microsoft.PowerShell.ThreadJob -ErrorAction Stop -WarningAction SilentlyContinue -Force | Out-Null
            }
        }
    }
	
    try {
        $objects = ConvertFrom-Json $ExtraInfo 
        $targetExecutable = ''
        $res = Common-GetAppRelatedPath -appName $AppName -appChosenInstallPath ([ref]$AppInstallPath) -targetExecutable  ([ref]$targetExecutable) -exeInfoList $objects.exeInfoList -exeSidList $objects.exeSidList
        $res = $res | Where-Object { $_.PSObject.Properties.Name -contains 'Index' }
        for ($i = 0; $i -lt $res.Count; $i++) {
            $item = $res[$i]
            $tmp = Replace-CTXUserInfoForOneRule -targetpath  ([ref]($item.Path))
            $item.Path = $tmp
            $item.Type = switch ($item.Type) {
                "File" { "0" }
                "Registry Key" { "1" }
                "Registry Value" { "2" }
                "Folder" { "3" }
                default { "0" } 
            }	
            $res[$i] = $item
        }
	
	
        $retJson = [PSCustomObject]@{
            ErrorCode    = '0'
            ErrorMessage = 'success'
            AppInfo      = $res
        }
        $retJson = $retJson | ConvertTo-Json	
        return $retJson 	
    }
    catch {
        $retJson = [PSCustomObject]@{
            ErrorCode    = $_.Exception.HResult
            ErrorMessage = $_.Exception.Message
            AppInfo      = $null  
        }
        $retJson = $retJson | ConvertTo-Json
        return $retJson	
    }
 
}

#endregion      

# SIG # Begin signature block
# MIIoogYJKoZIhvcNAQcCoIIokzCCKI8CAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCC93YjDdVectSyN
# RWnRsO99eCExzZK4er7+mnuLQ7ax5qCCDcAwggawMIIEmKADAgECAhAIrUCyYNKc
# TJ9ezam9k67ZMA0GCSqGSIb3DQEBDAUAMGIxCzAJBgNVBAYTAlVTMRUwEwYDVQQK
# EwxEaWdpQ2VydCBJbmMxGTAXBgNVBAsTEHd3dy5kaWdpY2VydC5jb20xITAfBgNV
# BAMTGERpZ2lDZXJ0IFRydXN0ZWQgUm9vdCBHNDAeFw0yMTA0MjkwMDAwMDBaFw0z
# NjA0MjgyMzU5NTlaMGkxCzAJBgNVBAYTAlVTMRcwFQYDVQQKEw5EaWdpQ2VydCwg
# SW5jLjFBMD8GA1UEAxM4RGlnaUNlcnQgVHJ1c3RlZCBHNCBDb2RlIFNpZ25pbmcg
# UlNBNDA5NiBTSEEzODQgMjAyMSBDQTEwggIiMA0GCSqGSIb3DQEBAQUAA4ICDwAw
# ggIKAoICAQDVtC9C0CiteLdd1TlZG7GIQvUzjOs9gZdwxbvEhSYwn6SOaNhc9es0
# JAfhS0/TeEP0F9ce2vnS1WcaUk8OoVf8iJnBkcyBAz5NcCRks43iCH00fUyAVxJr
# Q5qZ8sU7H/Lvy0daE6ZMswEgJfMQ04uy+wjwiuCdCcBlp/qYgEk1hz1RGeiQIXhF
# LqGfLOEYwhrMxe6TSXBCMo/7xuoc82VokaJNTIIRSFJo3hC9FFdd6BgTZcV/sk+F
# LEikVoQ11vkunKoAFdE3/hoGlMJ8yOobMubKwvSnowMOdKWvObarYBLj6Na59zHh
# 3K3kGKDYwSNHR7OhD26jq22YBoMbt2pnLdK9RBqSEIGPsDsJ18ebMlrC/2pgVItJ
# wZPt4bRc4G/rJvmM1bL5OBDm6s6R9b7T+2+TYTRcvJNFKIM2KmYoX7BzzosmJQay
# g9Rc9hUZTO1i4F4z8ujo7AqnsAMrkbI2eb73rQgedaZlzLvjSFDzd5Ea/ttQokbI
# YViY9XwCFjyDKK05huzUtw1T0PhH5nUwjewwk3YUpltLXXRhTT8SkXbev1jLchAp
# QfDVxW0mdmgRQRNYmtwmKwH0iU1Z23jPgUo+QEdfyYFQc4UQIyFZYIpkVMHMIRro
# OBl8ZhzNeDhFMJlP/2NPTLuqDQhTQXxYPUez+rbsjDIJAsxsPAxWEQIDAQABo4IB
# WTCCAVUwEgYDVR0TAQH/BAgwBgEB/wIBADAdBgNVHQ4EFgQUaDfg67Y7+F8Rhvv+
# YXsIiGX0TkIwHwYDVR0jBBgwFoAU7NfjgtJxXWRM3y5nP+e6mK4cD08wDgYDVR0P
# AQH/BAQDAgGGMBMGA1UdJQQMMAoGCCsGAQUFBwMDMHcGCCsGAQUFBwEBBGswaTAk
# BggrBgEFBQcwAYYYaHR0cDovL29jc3AuZGlnaWNlcnQuY29tMEEGCCsGAQUFBzAC
# hjVodHRwOi8vY2FjZXJ0cy5kaWdpY2VydC5jb20vRGlnaUNlcnRUcnVzdGVkUm9v
# dEc0LmNydDBDBgNVHR8EPDA6MDigNqA0hjJodHRwOi8vY3JsMy5kaWdpY2VydC5j
# b20vRGlnaUNlcnRUcnVzdGVkUm9vdEc0LmNybDAcBgNVHSAEFTATMAcGBWeBDAED
# MAgGBmeBDAEEATANBgkqhkiG9w0BAQwFAAOCAgEAOiNEPY0Idu6PvDqZ01bgAhql
# +Eg08yy25nRm95RysQDKr2wwJxMSnpBEn0v9nqN8JtU3vDpdSG2V1T9J9Ce7FoFF
# UP2cvbaF4HZ+N3HLIvdaqpDP9ZNq4+sg0dVQeYiaiorBtr2hSBh+3NiAGhEZGM1h
# mYFW9snjdufE5BtfQ/g+lP92OT2e1JnPSt0o618moZVYSNUa/tcnP/2Q0XaG3Ryw
# YFzzDaju4ImhvTnhOE7abrs2nfvlIVNaw8rpavGiPttDuDPITzgUkpn13c5Ubdld
# AhQfQDN8A+KVssIhdXNSy0bYxDQcoqVLjc1vdjcshT8azibpGL6QB7BDf5WIIIJw
# 8MzK7/0pNVwfiThV9zeKiwmhywvpMRr/LhlcOXHhvpynCgbWJme3kuZOX956rEnP
# LqR0kq3bPKSchh/jwVYbKyP/j7XqiHtwa+aguv06P0WmxOgWkVKLQcBIhEuWTatE
# QOON8BUozu3xGFYHKi8QxAwIZDwzj64ojDzLj4gLDb879M4ee47vtevLt/B3E+bn
# KD+sEq6lLyJsQfmCXBVmzGwOysWGw/YmMwwHS6DTBwJqakAwSEs0qFEgu60bhQji
# WQ1tygVQK+pKHJ6l/aCnHwZ05/LWUpD9r4VIIflXO7ScA+2GRfS0YW6/aOImYIbq
# yK+p/pQd52MbOoZWeE4wggcIMIIE8KADAgECAhADWiAGLDpbT4qELERZjaX6MA0G
# CSqGSIb3DQEBCwUAMGkxCzAJBgNVBAYTAlVTMRcwFQYDVQQKEw5EaWdpQ2VydCwg
# SW5jLjFBMD8GA1UEAxM4RGlnaUNlcnQgVHJ1c3RlZCBHNCBDb2RlIFNpZ25pbmcg
# UlNBNDA5NiBTSEEzODQgMjAyMSBDQTEwHhcNMjUwMjA3MDAwMDAwWhcNMjYwMjA2
# MjM1OTU5WjCBjzELMAkGA1UEBhMCVVMxEDAOBgNVBAgTB0Zsb3JpZGExGDAWBgNV
# BAcTD0ZvcnQgTGF1ZGVyZGFsZTEdMBsGA1UEChMUQ2l0cml4IFN5c3RlbXMsIElu
# Yy4xFjAUBgNVBAsTDUNpdHJpeCBYZW5BcHAxHTAbBgNVBAMTFENpdHJpeCBTeXN0
# ZW1zLCBJbmMuMIIBojANBgkqhkiG9w0BAQEFAAOCAY8AMIIBigKCAYEAnBm4JMW+
# W2tRIJltyaJM/q5G6WRAN293DlUc71U/dg6HzXRtahZP2kvV6ii32orbgdNkCwig
# +bo0W7AKjBBHKJqP96nuTNTsHoz+aeEE6PfgaFzGTRq77+dGDCk5DuPbFaXJCn9s
# UjVdbJTG1YXAcll8ZU6Iarg3eOokp2CcejMcjeD2dO8y89o6y2W5sWj2oIA+QRE9
# iU5qNfDLUtXo5i017JTN+qs9RtuwD77svXoV29wmkPmGuUq625ZZYFtRa0/t/C7w
# /k00UOFjykbvNBPj/cT67i/J/Um8tOanuC3cYU18VToMsfpS2t4irTdtrBzHupr/
# MB2DzZTwr5x2m3UzgEsnrr9bYnCIpOuW+K/oExuTpHtZsk6fnpoteOfyP059dNMg
# i1Gj074k6JfaJG+6fwKW0i2Unf7NDBArHdoHA6eIYB/OivPt4cmusgzRr2PziAl4
# LpA/9VRcnR68CWjnoTTr7qhdDeGqMgk24cNtmg+6BDt65GDVDX2ycthpAgMBAAGj
# ggIDMIIB/zAfBgNVHSMEGDAWgBRoN+Drtjv4XxGG+/5hewiIZfROQjAdBgNVHQ4E
# FgQUEpWeJpU+v9Nt97sQvlQgm4DN5wMwPgYDVR0gBDcwNTAzBgZngQwBBAEwKTAn
# BggrBgEFBQcCARYbaHR0cDovL3d3dy5kaWdpY2VydC5jb20vQ1BTMA4GA1UdDwEB
# /wQEAwIHgDATBgNVHSUEDDAKBggrBgEFBQcDAzCBtQYDVR0fBIGtMIGqMFOgUaBP
# hk1odHRwOi8vY3JsMy5kaWdpY2VydC5jb20vRGlnaUNlcnRUcnVzdGVkRzRDb2Rl
# U2lnbmluZ1JTQTQwOTZTSEEzODQyMDIxQ0ExLmNybDBToFGgT4ZNaHR0cDovL2Ny
# bDQuZGlnaWNlcnQuY29tL0RpZ2lDZXJ0VHJ1c3RlZEc0Q29kZVNpZ25pbmdSU0E0
# MDk2U0hBMzg0MjAyMUNBMS5jcmwwgZQGCCsGAQUFBwEBBIGHMIGEMCQGCCsGAQUF
# BzABhhhodHRwOi8vb2NzcC5kaWdpY2VydC5jb20wXAYIKwYBBQUHMAKGUGh0dHA6
# Ly9jYWNlcnRzLmRpZ2ljZXJ0LmNvbS9EaWdpQ2VydFRydXN0ZWRHNENvZGVTaWdu
# aW5nUlNBNDA5NlNIQTM4NDIwMjFDQTEuY3J0MAkGA1UdEwQCMAAwDQYJKoZIhvcN
# AQELBQADggIBAMGu7yYWlIS96uBwpNvMWgRgVFRwDx0HBdxGRi2+XvrtKkooOglx
# eUl/IYzZb8WhEBs/vknAmZkxBnkbbdgKKqxDeIkT2JX1sNGLKIzBBLPPjGMd2wxH
# CAVssWrA9xMESJ5Zx+5jHyoeVPt4XdPAXNB6Ca1JCa68Ip8Bn+atqop5wOmhXK85
# 2V2PdCWXjQr3Yl0k0PY/OtDDxnQc0dClj9TYlfGsOwyVNNQh/eExH2wkVlYYxgcj
# Zn89SmO4L+9AHkfpj/wd+/pJ+BYSflAotC6d9l4dft6f6PZ6KkbAz56dVkBX20xz
# YKKIK9tCf5sa1wnjkBEm2yn1IDn6K1cwUGO0whhtzGgcv4vPfv5bkumCSD7AquQv
# 37LP/gOZT9v3CiZAlTbZAoOK105qFmawt7AKPFzNFq9EjbUHYVBeYFnETDoa9zLz
# KCCcF/xJsjfn7YqFM+b1zq3C1YJK1B6+f3b5acpXNHbyMxPSOeNDZ13wMAYKah17
# D/4GZe74bsVqFo0oHlB82Bocr0AnnrLnLvfEirc0KIon5CQ5cci3LolBE0aNhD+n
# UtInPAJN1uRFpvhSK1ucVO/afmTQVtUgp5a8a8Dthc0bqh2ere4hoY+9U08Bda5g
# mFOO9YU8xEU3Do0AbuJPUVVALAAs0I+38RjYtIfPPt/T1zrMFHv+CMsKMYIaODCC
# GjQCAQEwfTBpMQswCQYDVQQGEwJVUzEXMBUGA1UEChMORGlnaUNlcnQsIEluYy4x
# QTA/BgNVBAMTOERpZ2lDZXJ0IFRydXN0ZWQgRzQgQ29kZSBTaWduaW5nIFJTQTQw
# OTYgU0hBMzg0IDIwMjEgQ0ExAhADWiAGLDpbT4qELERZjaX6MA0GCWCGSAFlAwQC
# AQUAoIHQMBkGCSqGSIb3DQEJAzEMBgorBgEEAYI3AgEEMBwGCisGAQQBgjcCAQsx
# DjAMBgorBgEEAYI3AgEVMC8GCSqGSIb3DQEJBDEiBCAT46OXguQCdvTMfvaR4Ssh
# Xq9ER7zQI3rKDjxrAsH8QDBkBgorBgEEAYI3AgEMMVYwVKA4gDYAQwBQAE0AXwBB
# AHAAcABfAEEAYwBjAGUAcwBzAF8AQwBvAG4AdAByAG8AbABfAEkAbQBwAGyhGIAW
# aHR0cDovL3d3dy5jaXRyaXguY29tIDANBgkqhkiG9w0BAQEFAASCAYA8aK0yp8Tq
# 8CZS/pGimjpsw0KaaAK3F8Grq+8O2LxjeIPGSCYERHx8Iq8KO103kU28VlOn8P7L
# WK+iPCkUtR4iYW3fgjCRHWxv+VZ/KBgy3a2k7+V7qI0d2nzN5yQd1eBHtgMyvRY7
# lF2flgVjFZ7CXqaVe7ZXAFiCD2gyyHm+lt/dPU0M5W6Eg5tft6zQ1qpmBCeSuoWK
# k1UwU030yIRYnmo/QBL2ljYc+kb5SPpp5iuvNndaRZ2VFXR9xgTp69gxRBKhbEoz
# BSBSVL1aTT4Vz/CDXDDcfvjpJodqdv+i9msx9Rjiz8eErJUObLAQmpQRy5aYFXWY
# Pp0tV+OV7wmJkyezz68kpZROigxA96D7dh33Hv04IAGQnjI8EkaMeWQldeufuZkk
# QUQIjZo3WtQT0fDIaVcCRc2Foa8HUWz9Oh1WoEaeCGl1jJ47CWx8Mcas84yEh9Uh
# gsmkkQ34jI0lU8AYlHwAODOq/IGVep1Tb3wmc2dPzDAwibzXGNOPWsqhghc5MIIX
# NQYKKwYBBAGCNwMDATGCFyUwghchBgkqhkiG9w0BBwKgghcSMIIXDgIBAzEPMA0G
# CWCGSAFlAwQCAQUAMHcGCyqGSIb3DQEJEAEEoGgEZjBkAgEBBglghkgBhv1sBwEw
# MTANBglghkgBZQMEAgEFAAQgk+vW3qgT88CyvE23eA7K+/H+tSKCNplkATeOjvLg
# Y40CEBNDYXYC4EArDGAMVEUVsW0YDzIwMjUwNTA4MDgzNDIzWqCCEwMwgga8MIIE
# pKADAgECAhALrma8Wrp/lYfG+ekE4zMEMA0GCSqGSIb3DQEBCwUAMGMxCzAJBgNV
# BAYTAlVTMRcwFQYDVQQKEw5EaWdpQ2VydCwgSW5jLjE7MDkGA1UEAxMyRGlnaUNl
# cnQgVHJ1c3RlZCBHNCBSU0E0MDk2IFNIQTI1NiBUaW1lU3RhbXBpbmcgQ0EwHhcN
# MjQwOTI2MDAwMDAwWhcNMzUxMTI1MjM1OTU5WjBCMQswCQYDVQQGEwJVUzERMA8G
# A1UEChMIRGlnaUNlcnQxIDAeBgNVBAMTF0RpZ2lDZXJ0IFRpbWVzdGFtcCAyMDI0
# MIICIjANBgkqhkiG9w0BAQEFAAOCAg8AMIICCgKCAgEAvmpzn/aVIauWMLpbbeZZ
# o7Xo/ZEfGMSIO2qZ46XB/QowIEMSvgjEdEZ3v4vrrTHleW1JWGErrjOL0J4L0HqV
# R1czSzvUQ5xF7z4IQmn7dHY7yijvoQ7ujm0u6yXF2v1CrzZopykD07/9fpAT4Bxp
# T9vJoJqAsP8YuhRvflJ9YeHjes4fduksTHulntq9WelRWY++TFPxzZrbILRYynyE
# y7rS1lHQKFpXvo2GePfsMRhNf1F41nyEg5h7iOXv+vjX0K8RhUisfqw3TTLHj1uh
# S66YX2LZPxS4oaf33rp9HlfqSBePejlYeEdU740GKQM7SaVSH3TbBL8R6HwX9QVp
# GnXPlKdE4fBIn5BBFnV+KwPxRNUNK6lYk2y1WSKour4hJN0SMkoaNV8hyyADiX1x
# uTxKaXN12HgR+8WulU2d6zhzXomJ2PleI9V2yfmfXSPGYanGgxzqI+ShoOGLomMd
# 3mJt92nm7Mheng/TBeSA2z4I78JpwGpTRHiT7yHqBiV2ngUIyCtd0pZ8zg3S7bk4
# QC4RrcnKJ3FbjyPAGogmoiZ33c1HG93Vp6lJ415ERcC7bFQMRbxqrMVANiav1k42
# 5zYyFMyLNyE1QulQSgDpW9rtvVcIH7WvG9sqYup9j8z9J1XqbBZPJ5XLln8mS8wW
# mdDLnBHXgYly/p1DhoQo5fkCAwEAAaOCAYswggGHMA4GA1UdDwEB/wQEAwIHgDAM
# BgNVHRMBAf8EAjAAMBYGA1UdJQEB/wQMMAoGCCsGAQUFBwMIMCAGA1UdIAQZMBcw
# CAYGZ4EMAQQCMAsGCWCGSAGG/WwHATAfBgNVHSMEGDAWgBS6FtltTYUvcyl2mi91
# jGogj57IbzAdBgNVHQ4EFgQUn1csA3cOKBWQZqVjXu5Pkh92oFswWgYDVR0fBFMw
# UTBPoE2gS4ZJaHR0cDovL2NybDMuZGlnaWNlcnQuY29tL0RpZ2lDZXJ0VHJ1c3Rl
# ZEc0UlNBNDA5NlNIQTI1NlRpbWVTdGFtcGluZ0NBLmNybDCBkAYIKwYBBQUHAQEE
# gYMwgYAwJAYIKwYBBQUHMAGGGGh0dHA6Ly9vY3NwLmRpZ2ljZXJ0LmNvbTBYBggr
# BgEFBQcwAoZMaHR0cDovL2NhY2VydHMuZGlnaWNlcnQuY29tL0RpZ2lDZXJ0VHJ1
# c3RlZEc0UlNBNDA5NlNIQTI1NlRpbWVTdGFtcGluZ0NBLmNydDANBgkqhkiG9w0B
# AQsFAAOCAgEAPa0eH3aZW+M4hBJH2UOR9hHbm04IHdEoT8/T3HuBSyZeq3jSi5GX
# eWP7xCKhVireKCnCs+8GZl2uVYFvQe+pPTScVJeCZSsMo1JCoZN2mMew/L4tpqVN
# bSpWO9QGFwfMEy60HofN6V51sMLMXNTLfhVqs+e8haupWiArSozyAmGH/6oMQAh0
# 78qRh6wvJNU6gnh5OruCP1QUAvVSu4kqVOcJVozZR5RRb/zPd++PGE3qF1P3xWvY
# ViUJLsxtvge/mzA75oBfFZSbdakHJe2BVDGIGVNVjOp8sNt70+kEoMF+T6tptMUN
# lehSR7vM+C13v9+9ZOUKzfRUAYSyyEmYtsnpltD/GWX8eM70ls1V6QG/ZOB6b6Yu
# m1HvIiulqJ1Elesj5TMHq8CWT/xrW7twipXTJ5/i5pkU5E16RSBAdOp12aw8IQhh
# A/vEbFkEiF2abhuFixUDobZaA0VhqAsMHOmaT3XThZDNi5U2zHKhUs5uHHdG6BoQ
# au75KiNbh0c+hatSF+02kULkftARjsyEpHKsF7u5zKRbt5oK5YGwFvgc4pEVUNyt
# mB3BpIiowOIIuDgP5M9WArHYSAR16gc0dP2XdkMEP5eBsX7bf/MGN4K3HP50v/01
# ZHo/Z5lGLvNwQ7XHBx1yomzLP8lx4Q1zZKDyHcp4VQJLu2kWTsKsOqQwggauMIIE
# lqADAgECAhAHNje3JFR82Ees/ShmKl5bMA0GCSqGSIb3DQEBCwUAMGIxCzAJBgNV
# BAYTAlVTMRUwEwYDVQQKEwxEaWdpQ2VydCBJbmMxGTAXBgNVBAsTEHd3dy5kaWdp
# Y2VydC5jb20xITAfBgNVBAMTGERpZ2lDZXJ0IFRydXN0ZWQgUm9vdCBHNDAeFw0y
# MjAzMjMwMDAwMDBaFw0zNzAzMjIyMzU5NTlaMGMxCzAJBgNVBAYTAlVTMRcwFQYD
# VQQKEw5EaWdpQ2VydCwgSW5jLjE7MDkGA1UEAxMyRGlnaUNlcnQgVHJ1c3RlZCBH
# NCBSU0E0MDk2IFNIQTI1NiBUaW1lU3RhbXBpbmcgQ0EwggIiMA0GCSqGSIb3DQEB
# AQUAA4ICDwAwggIKAoICAQDGhjUGSbPBPXJJUVXHJQPE8pE3qZdRodbSg9GeTKJt
# oLDMg/la9hGhRBVCX6SI82j6ffOciQt/nR+eDzMfUBMLJnOWbfhXqAJ9/UO0hNoR
# 8XOxs+4rgISKIhjf69o9xBd/qxkrPkLcZ47qUT3w1lbU5ygt69OxtXXnHwZljZQp
# 09nsad/ZkIdGAHvbREGJ3HxqV3rwN3mfXazL6IRktFLydkf3YYMZ3V+0VAshaG43
# IbtArF+y3kp9zvU5EmfvDqVjbOSmxR3NNg1c1eYbqMFkdECnwHLFuk4fsbVYTXn+
# 149zk6wsOeKlSNbwsDETqVcplicu9Yemj052FVUmcJgmf6AaRyBD40NjgHt1bicl
# kJg6OBGz9vae5jtb7IHeIhTZgirHkr+g3uM+onP65x9abJTyUpURK1h0QCirc0PO
# 30qhHGs4xSnzyqqWc0Jon7ZGs506o9UD4L/wojzKQtwYSH8UNM/STKvvmz3+Drhk
# Kvp1KCRB7UK/BZxmSVJQ9FHzNklNiyDSLFc1eSuo80VgvCONWPfcYd6T/jnA+bIw
# pUzX6ZhKWD7TA4j+s4/TXkt2ElGTyYwMO1uKIqjBJgj5FBASA31fI7tk42PgpuE+
# 9sJ0sj8eCXbsq11GdeJgo1gJASgADoRU7s7pXcheMBK9Rp6103a50g5rmQzSM7TN
# sQIDAQABo4IBXTCCAVkwEgYDVR0TAQH/BAgwBgEB/wIBADAdBgNVHQ4EFgQUuhbZ
# bU2FL3MpdpovdYxqII+eyG8wHwYDVR0jBBgwFoAU7NfjgtJxXWRM3y5nP+e6mK4c
# D08wDgYDVR0PAQH/BAQDAgGGMBMGA1UdJQQMMAoGCCsGAQUFBwMIMHcGCCsGAQUF
# BwEBBGswaTAkBggrBgEFBQcwAYYYaHR0cDovL29jc3AuZGlnaWNlcnQuY29tMEEG
# CCsGAQUFBzAChjVodHRwOi8vY2FjZXJ0cy5kaWdpY2VydC5jb20vRGlnaUNlcnRU
# cnVzdGVkUm9vdEc0LmNydDBDBgNVHR8EPDA6MDigNqA0hjJodHRwOi8vY3JsMy5k
# aWdpY2VydC5jb20vRGlnaUNlcnRUcnVzdGVkUm9vdEc0LmNybDAgBgNVHSAEGTAX
# MAgGBmeBDAEEAjALBglghkgBhv1sBwEwDQYJKoZIhvcNAQELBQADggIBAH1ZjsCT
# tm+YqUQiAX5m1tghQuGwGC4QTRPPMFPOvxj7x1Bd4ksp+3CKDaopafxpwc8dB+k+
# YMjYC+VcW9dth/qEICU0MWfNthKWb8RQTGIdDAiCqBa9qVbPFXONASIlzpVpP0d3
# +3J0FNf/q0+KLHqrhc1DX+1gtqpPkWaeLJ7giqzl/Yy8ZCaHbJK9nXzQcAp876i8
# dU+6WvepELJd6f8oVInw1YpxdmXazPByoyP6wCeCRK6ZJxurJB4mwbfeKuv2nrF5
# mYGjVoarCkXJ38SNoOeY+/umnXKvxMfBwWpx2cYTgAnEtp/Nh4cku0+jSbl3ZpHx
# cpzpSwJSpzd+k1OsOx0ISQ+UzTl63f8lY5knLD0/a6fxZsNBzU+2QJshIUDQtxMk
# zdwdeDrknq3lNHGS1yZr5Dhzq6YBT70/O3itTK37xJV77QpfMzmHQXh6OOmc4d0j
# /R0o08f56PGYX/sr2H7yRp11LB4nLCbbbxV7HhmLNriT1ObyF5lZynDwN7+YAN8g
# Fk8n+2BnFqFmut1VwDophrCYoCvtlUG3OtUVmDG0YgkPCr2B2RP+v6TR81fZvAT6
# gt4y3wSJ8ADNXcL50CN/AAvkdgIm2fBldkKmKYcJRyvmfxqkhQ/8mJb2VVQrH4D6
# wPIOK+XW+6kvRBVK5xMOHds3OBqhK/bt1nz8MIIFjTCCBHWgAwIBAgIQDpsYjvnQ
# Lefv21DiCEAYWjANBgkqhkiG9w0BAQwFADBlMQswCQYDVQQGEwJVUzEVMBMGA1UE
# ChMMRGlnaUNlcnQgSW5jMRkwFwYDVQQLExB3d3cuZGlnaWNlcnQuY29tMSQwIgYD
# VQQDExtEaWdpQ2VydCBBc3N1cmVkIElEIFJvb3QgQ0EwHhcNMjIwODAxMDAwMDAw
# WhcNMzExMTA5MjM1OTU5WjBiMQswCQYDVQQGEwJVUzEVMBMGA1UEChMMRGlnaUNl
# cnQgSW5jMRkwFwYDVQQLExB3d3cuZGlnaWNlcnQuY29tMSEwHwYDVQQDExhEaWdp
# Q2VydCBUcnVzdGVkIFJvb3QgRzQwggIiMA0GCSqGSIb3DQEBAQUAA4ICDwAwggIK
# AoICAQC/5pBzaN675F1KPDAiMGkz7MKnJS7JIT3yithZwuEppz1Yq3aaza57G4QN
# xDAf8xukOBbrVsaXbR2rsnnyyhHS5F/WBTxSD1Ifxp4VpX6+n6lXFllVcq9ok3DC
# srp1mWpzMpTREEQQLt+C8weE5nQ7bXHiLQwb7iDVySAdYyktzuxeTsiT+CFhmzTr
# BcZe7FsavOvJz82sNEBfsXpm7nfISKhmV1efVFiODCu3T6cw2Vbuyntd463JT17l
# Necxy9qTXtyOj4DatpGYQJB5w3jHtrHEtWoYOAMQjdjUN6QuBX2I9YI+EJFwq1WC
# QTLX2wRzKm6RAXwhTNS8rhsDdV14Ztk6MUSaM0C/CNdaSaTC5qmgZ92kJ7yhTzm1
# EVgX9yRcRo9k98FpiHaYdj1ZXUJ2h4mXaXpI8OCiEhtmmnTK3kse5w5jrubU75KS
# Op493ADkRSWJtppEGSt+wJS00mFt6zPZxd9LBADMfRyVw4/3IbKyEbe7f/LVjHAs
# QWCqsWMYRJUadmJ+9oCw++hkpjPRiQfhvbfmQ6QYuKZ3AeEPlAwhHbJUKSWJbOUO
# UlFHdL4mrLZBdd56rF+NP8m800ERElvlEFDrMcXKchYiCd98THU/Y+whX8QgUWtv
# sauGi0/C1kVfnSD8oR7FwI+isX4KJpn15GkvmB0t9dmpsh3lGwIDAQABo4IBOjCC
# ATYwDwYDVR0TAQH/BAUwAwEB/zAdBgNVHQ4EFgQU7NfjgtJxXWRM3y5nP+e6mK4c
# D08wHwYDVR0jBBgwFoAUReuir/SSy4IxLVGLp6chnfNtyA8wDgYDVR0PAQH/BAQD
# AgGGMHkGCCsGAQUFBwEBBG0wazAkBggrBgEFBQcwAYYYaHR0cDovL29jc3AuZGln
# aWNlcnQuY29tMEMGCCsGAQUFBzAChjdodHRwOi8vY2FjZXJ0cy5kaWdpY2VydC5j
# b20vRGlnaUNlcnRBc3N1cmVkSURSb290Q0EuY3J0MEUGA1UdHwQ+MDwwOqA4oDaG
# NGh0dHA6Ly9jcmwzLmRpZ2ljZXJ0LmNvbS9EaWdpQ2VydEFzc3VyZWRJRFJvb3RD
# QS5jcmwwEQYDVR0gBAowCDAGBgRVHSAAMA0GCSqGSIb3DQEBDAUAA4IBAQBwoL9D
# XFXnOF+go3QbPbYW1/e/Vwe9mqyhhyzshV6pGrsi+IcaaVQi7aSId229GhT0E0p6
# Ly23OO/0/4C5+KH38nLeJLxSA8hO0Cre+i1Wz/n096wwepqLsl7Uz9FDRJtDIeuW
# cqFItJnLnU+nBgMTdydE1Od/6Fmo8L8vC6bp8jQ87PcDx4eo0kxAGTVGamlUsLih
# Vo7spNU96LHc/RzY9HdaXFSMb++hUD38dglohJ9vytsgjTVgHAIDyyCwrFigDkBj
# xZgiwbJZ9VVrzyerbHbObyMt9H5xaiNrIv8SuFQtJ37YOtnwtoeW/VvRXKwYw02f
# c7cBqZ9Xql4o4rmUMYIDdjCCA3ICAQEwdzBjMQswCQYDVQQGEwJVUzEXMBUGA1UE
# ChMORGlnaUNlcnQsIEluYy4xOzA5BgNVBAMTMkRpZ2lDZXJ0IFRydXN0ZWQgRzQg
# UlNBNDA5NiBTSEEyNTYgVGltZVN0YW1waW5nIENBAhALrma8Wrp/lYfG+ekE4zME
# MA0GCWCGSAFlAwQCAQUAoIHRMBoGCSqGSIb3DQEJAzENBgsqhkiG9w0BCRABBDAc
# BgkqhkiG9w0BCQUxDxcNMjUwNTA4MDgzNDIzWjArBgsqhkiG9w0BCRACDDEcMBow
# GDAWBBTb04XuYtvSPnvk9nFIUIck1YZbRTAvBgkqhkiG9w0BCQQxIgQgQgwzEmyv
# IRmGwFIujsqjMe4yaCjk/fg2kYsPOfFpQK8wNwYLKoZIhvcNAQkQAi8xKDAmMCQw
# IgQgdnafqPJjLx9DCzojMK7WVnX+13PbBdZluQWTmEOPmtswDQYJKoZIhvcNAQEB
# BQAEggIAicvT4E3yY39MxUcrwQ/Dbf/QO608ol0QIzO2xnUDBlGj1w3hNdesNmue
# J1px+a/i8nEbn9VB3FFGLDsI4rwgUtwB+mtkC2tGqLeflIGKxXsHKKMVY/371dEk
# 0jxonrUGzEuujk0kDT7xcbV3dcqr2se7PnIzdXapC/4eiU1SknNhmAGW4PLvy7Oy
# QLS9hhyyevnSLt28nTlrg9bMVj1UVkrOIedWgt82/MFABTUhtLAu3jEeT5w78n1k
# /IWuGxkg7BG3x4KOjxhcOKrKCroS0GgUldoaucGqoZs5xpK9ZZ2WMzP/fguz3ui3
# LtHA6bosO8kBjVDz9U6FDobEUSVZVKgyxUpOSkLk5ElzGOlfC7xmu6KA4g5XNnMt
# 0viM2akF50oaoJsVvvq3SrGgZ7k9zNfOrziFrFxwcaIlJVri2PGPWUqArNWDXI5j
# muiZHbeI2He+acHpgQf7alOUpQDy50t21yaIqPe+CINk752BmgqmQZjlA2HrE7gg
# lWm6TQp+EoBo7j7OuaPLq3HKRNBdarjfbMatRPFkk4giLzmc1L6w7IcAfJby39Kh
# 6cBB9mHKDgFagSFoRqkMd5LWZPPQ/Bhe9N/0BWbUm3uK9LsnMUWm/R8ZQ0Ujyo7i
# gTt/mi3xoQQVMkS/MpdtZIVpS7UA3DggzhjPpBhb5m5OwhgEkjE=
# SIG # End signature block
