<# 
.SYNOPSIS
    Citrix Profile Management Migration Tool

.DESCRIPTION 
    This script should be run as an administrator.
 
.NOTES 
    This PowerShell script was developed to help users migration from other profile scenaro to Citrix Profile Container.

.COMPONENT 
    Required Module AD

.LINK 
    It is released with Citrix Profile Management
 
.Parameter ParameterName 
    NA 
#>


param (
    [Parameter(Mandatory=$false)]
    [int]$migrationType,

    [Parameter(Mandatory=$false)]
    [string]$usersAndGroups,

    [Parameter(Mandatory=$false)]
    [string]$sourcePath,

    [Parameter(Mandatory=$false)]
    [string]$targetPath,

    [Parameter(Mandatory=$false)]
    [string]$osVersion,

    [Parameter(Mandatory = $false)]
    [PSCredential] $remoteCredential = $null
)

$Global:importModulesCnt = 0

function WriteLogWithLevel {
    param (
        [Parameter(ValueFromPipeline = $true, Mandatory = $true)]
        [string] $Message,
        [Parameter(Mandatory = $true)]
        [string] $Level
    )

    $systemDrive = $env:SystemDrive
    $logfilePath = "$systemDrive\ProgramData\Citrix\ProfileManagement\Logs\ScriptTool\MigrationScriptTool.log"

    # Set the maximum size of the log file in MB
    $maxSizeMB = 10

    try {

        if (-not (Test-Path -Path $logfilePath)) {
            $logfileDirectory = Split-Path $logfilePath -Parent

            if (-not (Test-Path $logfileDirectory)) {
                New-Item -ItemType Directory -Path $logfileDirectory -Force | Out-Null
            }

            New-Item -ItemType File -Path $logfilePath | Out-Null
        }

        # check the size of the log file
        $logFileSizeMB = (Get-Item $logfilePath).length / 1MB

        if ($logFileSizeMB -ge $maxSizeMB) {

            # Create a new log file with a timestamp
            $timestamp = Get-Date -Format "yyyyMMdd_HHmmss"
            $archivePath = "$logfilePath.$timestamp.bak"
            Rename-Item -Path $logfilePath -NewName $archivePath -Force
            Write-Host "Log file size exceeded $maxSizeMB MB. Archived to $archivePath"

            New-Item -ItemType File -Path $logfilePath | Out-Null
        }

    } catch {

        Write-Host "Create new log file: $logfilePath failed, the reason is: $_"
        return

    }

    $timestamp = Get-Date -Format "yyyy-MM-dd HH:mm:ss"
    $logEntry = "$timestamp [$Level] - $Message"

    Add-Content -Path $logfilePath -Value $logEntry
}

function WriteLogInfo {
    param (
        [Parameter(ValueFromPipeline = $true, Mandatory = $true)]
        [string] $Message
    )

    WriteLogWithLevel -Message $Message -Level "INFO"
}


function WriteLogWarning {
    param (
        [Parameter(ValueFromPipeline = $true, Mandatory = $true)]
        [string] $Message
    )

    WriteLogWithLevel -Message $Message -Level "WARNING"
}


function WriteLogError {
    param (
        [Parameter(ValueFromPipeline = $true, Mandatory = $true)]
        [string] $Message
    )

    WriteLogWithLevel -Message $Message -Level "ERROR"
}

function WriteLogDebug {
    param (
        [Parameter(ValueFromPipeline = $true, Mandatory = $true)]
        [string] $Message
    )

    WriteLogWithLevel -Message $Message -Level "DEBUG"
}


function WriteJsonFile {
	param (
        [Parameter(ValueFromPipeline = $true, Mandatory = $true)]
        [string] $UserName,
        [Parameter(Mandatory = $true)]
        [string] $Status,
        [Parameter(Mandatory = $true)]
        [string] $ErrorMessage,
        [Parameter(Mandatory = $false)]
        [bool] $isClear = $false
    )

    $systemDrive = $env:SystemDrive
    $jsonfilePath = "$systemDrive\ProgramData\Citrix\ProfileManagement\Logs\ScriptTool\StatusJson.log"

    try {

        if (-not (Test-Path -Path $jsonfilePath)) {
            $jsonfileDirectory = Split-Path $jsonfilePath -Parent
            if (-not (Test-Path $jsonfileDirectory)) {
                New-Item -ItemType Directory -Path $jsonfileDirectory -Force | Out-Null
            }

            New-Item -ItemType File -Path $jsonfilePath | Out-Null
        }

    } catch {
        Write-Host "Create log file: $jsonfilePath failed, the reason is: $_"
        return
    }

	if ($isClear) {
		Clear-Content -Path $jsonfilePath
		return
	}

    $jsonLine = @"
{"UserName" : "$UserName","Status" : "$Status","ErrorMessage" : "$ErrorMessage"},
"@

	$jsonLine | Add-Content -Path $jsonfilePath
}

function StopMigrationProcess {

    $systemDrive = $env:SystemDrive
    $jsonfilePath = "$systemDrive\ProgramData\Citrix\ProfileManagement\Logs\ScriptTool\StatusJson.log"

	if (Test-Path $jsonfilePath) {

		$content = Get-Content -Path $jsonfilePath -Raw

		if ($content -match "Stop migration") {
			return $true
		} else {
			return $false
		}

	} else {

		WriteLogError("Status file does not exist")
		return $false
  
	}

}


function CreateJunctionPoints {
	param (
        [Parameter(ValueFromPipeline = $true, Mandatory = $true)]
        [string] $BaseDirectory
    )

    WriteLogInfo("Entering the CreateJunctionPoints")

    $BaseDirectory = Join-Path -Path $BaseDirectory -ChildPath "Profiles"

    $sourceDocumentsPath = Join-Path -Path $BaseDirectory -ChildPath "My Documents"
    $destDocumentsPath = Join-Path -Path $BaseDirectory -ChildPath "Documents"

    $sourcePicturesPath = Join-Path -Path $BaseDirectory -ChildPath "Documents\My Pictures"
    $destPicturesPath = Join-Path -Path $BaseDirectory -ChildPath "Pictures"

    $sourceMusicPath = Join-Path -Path $BaseDirectory -ChildPath "Documents\My Music"
    $destMusicPath = Join-Path -Path $BaseDirectory -ChildPath "Music"

    $sourceVideosPath = Join-Path -Path $BaseDirectory -ChildPath "Documents\My Videos"
    $destVideosPath = Join-Path -Path $BaseDirectory -ChildPath "Videos"

    $sourceAppdataPath = Join-Path -Path $BaseDirectory -ChildPath "Application Data"
    $destAppdataPath = Join-Path -Path $BaseDirectory -ChildPath "AppData\Roaming"

    $sourceCookiesPath = Join-Path -Path $BaseDirectory -ChildPath "Cookies"
    $destCookiesPath = Join-Path -Path $BaseDirectory -ChildPath "AppData\Roaming\Microsoft\Windows\Cookies"

    $sourcePrintHoodPath = Join-Path -Path $BaseDirectory -ChildPath "PrintHood"
    $destPrintHoodPath = Join-Path -Path $BaseDirectory -ChildPath "AppData\Roaming\Microsoft\Windows\Printer Shortcuts"

    $sourceNetHoodPath = Join-Path -Path $BaseDirectory -ChildPath "NetHood"
    $destNetHoodPath = Join-Path -Path $BaseDirectory -ChildPath "AppData\Roaming\Microsoft\Windows\Network Shortcuts"

    $sourceRecentPath = Join-Path -Path $BaseDirectory -ChildPath "Recent"
    $destRecentPath = Join-Path -Path $BaseDirectory -ChildPath "AppData\Roaming\Microsoft\Windows\Recent"

    $sourceSendToPath = Join-Path -Path $BaseDirectory -ChildPath "SendTo"
    $destSendToPath = Join-Path -Path $BaseDirectory -ChildPath "AppData\Roaming\Microsoft\Windows\SendTo"

    $sourceTemplatesPath = Join-Path -Path $BaseDirectory -ChildPath "Templates"
    $destTemplatesPath = Join-Path -Path $BaseDirectory -ChildPath "AppData\Roaming\Microsoft\Windows\Templates"

    $sourceStartMenuPath = Join-Path -Path $BaseDirectory -ChildPath "Start Menu"
    $destStartMenuPath = Join-Path -Path $BaseDirectory -ChildPath "AppData\Roaming\Microsoft\Windows\Start Menu"

    $sourceStartMenuProgramsPath = Join-Path -Path $BaseDirectory -ChildPath "AppData\Roaming\Microsoft\Windows\Start Menu\Program Files"
    $destStartMenuProgramsPath = Join-Path -Path $BaseDirectory -ChildPath "AppData\Roaming\Microsoft\Windows\Start Menu\Programs"

    $sourceLocalSettingsPath = Join-Path -Path $BaseDirectory -ChildPath "Local Settings"
    $destLocalSettingsPath = Join-Path -Path $BaseDirectory -ChildPath "AppData\Local"

    $sourceAppDataLocalPath = Join-Path -Path $BaseDirectory -ChildPath "AppData\Local\Application Data"
    $destAppDataLocalPath = Join-Path -Path $BaseDirectory -ChildPath "AppData\Local"

    $sourceTempIEPath = Join-Path -Path $BaseDirectory -ChildPath "AppData\Local\Temporary Internet Files"
    $destTempIEPath = Join-Path -Path $BaseDirectory -ChildPath "AppData\Local\Microsoft\Windows\Temporary Internet Files"

    $sourceHistoryPath = Join-Path -Path $BaseDirectory -ChildPath "AppData\Local\History"
    $destHistoryPath = Join-Path -Path $BaseDirectory -ChildPath "AppData\Local\Microsoft\Windows\History"

    if (Test-Path $destDocumentsPath) {
        Start-Process -FilePath "cmd.exe" -ArgumentList "/c mklink /J `"$sourceDocumentsPath`" `"$destDocumentsPath`"" -NoNewWindow -Wait -RedirectStandardOutput ".\NUL"
        icacls $sourceDocumentsPath /deny "Everyone:(RD)" | Out-Null
    }

    if (Test-Path $destPicturesPath) {
        Start-Process -FilePath "cmd.exe" -ArgumentList "/c mklink /J `"$sourcePicturesPath`" `"$destPicturesPath`"" -NoNewWindow -Wait -RedirectStandardOutput ".\NUL"
        icacls $sourcePicturesPath /deny "Everyone:(RD)" | Out-Null
    }

    if (Test-Path $destMusicPath) {
        Start-Process -FilePath "cmd.exe" -ArgumentList "/c mklink /J `"$sourceMusicPath`" `"$destMusicPath`"" -NoNewWindow -Wait -RedirectStandardOutput ".\NUL"
        icacls $sourceMusicPath /deny "Everyone:(RD)" | Out-Null
    }

    if (Test-Path $destVideosPath) {
        Start-Process -FilePath "cmd.exe" -ArgumentList "/c mklink /J `"$sourceVideosPath`" `"$destVideosPath`"" -NoNewWindow -Wait -RedirectStandardOutput ".\NUL"
        icacls $sourceVideosPath /deny "Everyone:(RD)" | Out-Null
    }

    if (Test-Path $destAppdataPath) {
        Start-Process -FilePath "cmd.exe" -ArgumentList "/c mklink /J `"$sourceAppdataPath`" `"$destAppdataPath`"" -NoNewWindow -Wait -RedirectStandardOutput ".\NUL"
        icacls $sourceAppdataPath /deny "Everyone:(RD)" | Out-Null
    }

    if (Test-Path $destCookiesPath) {
        Start-Process -FilePath "cmd.exe" -ArgumentList "/c mklink /J `"$sourceCookiesPath`" `"$destCookiesPath`"" -NoNewWindow -Wait -RedirectStandardOutput ".\NUL"
        icacls $sourceCookiesPath /deny "Everyone:(RD)" | Out-Null
    }

    if (Test-Path $destPrintHoodPath) {
        Start-Process -FilePath "cmd.exe" -ArgumentList "/c mklink /J `"$sourcePrintHoodPath`" `"$destPrintHoodPath`"" -NoNewWindow -Wait -RedirectStandardOutput ".\NUL"
        icacls $sourcePrintHoodPath /deny "Everyone:(RD)" | Out-Null
    }

    if (Test-Path $destNetHoodPath) {
        Start-Process -FilePath "cmd.exe" -ArgumentList "/c mklink /J `"$sourceNetHoodPath`" `"$destNetHoodPath`"" -NoNewWindow -Wait -RedirectStandardOutput ".\NUL"
        icacls $sourceNetHoodPath /deny "Everyone:(RD)" | Out-Null
    }

    if (Test-Path $destRecentPath) {
        Start-Process -FilePath "cmd.exe" -ArgumentList "/c mklink /J `"$sourceRecentPath`" `"$destRecentPath`"" -NoNewWindow -Wait -RedirectStandardOutput ".\NUL"
        icacls $sourceRecentPath /deny "Everyone:(RD)" | Out-Null
    }

    if (Test-Path $destSendToPath) {
        Start-Process -FilePath "cmd.exe" -ArgumentList "/c mklink /J `"$sourceSendToPath`" `"$destSendToPath`"" -NoNewWindow -Wait -RedirectStandardOutput ".\NUL"
        icacls $sourceSendToPath /deny "Everyone:(RD)" | Out-Null
    }

    if (Test-Path $destTemplatesPath) {
        Start-Process -FilePath "cmd.exe" -ArgumentList "/c mklink /J `"$sourceTemplatesPath`" `"$destTemplatesPath`"" -NoNewWindow -Wait -RedirectStandardOutput ".\NUL"
        icacls $sourceTemplatesPath /deny "Everyone:(RD)" | Out-Null
    }

    if (Test-Path $destStartMenuPath) {
        Start-Process -FilePath "cmd.exe" -ArgumentList "/c mklink /J `"$sourceStartMenuPath`" `"$destStartMenuPath`"" -NoNewWindow -Wait -RedirectStandardOutput ".\NUL"
        icacls $sourceStartMenuPath /deny "Everyone:(RD)" | Out-Null
    }

    if (Test-Path $destStartMenuProgramsPath) {
        Start-Process -FilePath "cmd.exe" -ArgumentList "/c mklink /J `"$sourceStartMenuProgramsPath`" `"$destStartMenuProgramsPath`"" -NoNewWindow -RedirectStandardOutput ".\NUL"
        icacls $sourceStartMenuProgramsPath /deny "Everyone:(RD)" | Out-Null
    }

    if (Test-Path $destLocalSettingsPath) {
        Start-Process -FilePath "cmd.exe" -ArgumentList "/c mklink /J `"$sourceLocalSettingsPath`" `"$destLocalSettingsPath`"" -NoNewWindow -Wait -RedirectStandardOutput ".\NUL"
        icacls $sourceLocalSettingsPath /deny "Everyone:(RD)" | Out-Null
    }

    if (Test-Path $destAppDataLocalPath) {
        Start-Process -FilePath "cmd.exe" -ArgumentList "/c mklink /J `"$sourceAppDataLocalPath`" `"$destAppDataLocalPath`"" -NoNewWindow -Wait -RedirectStandardOutput ".\NUL"
        icacls $sourceAppDataLocalPath /deny "Everyone:(RD)" | Out-Null
    }

    if (Test-Path $destTempIEPath) {
        Start-Process -FilePath "cmd.exe" -ArgumentList "/c mklink /J `"$sourceTempIEPath`" `"$destTempIEPath`"" -NoNewWindow -Wait -RedirectStandardOutput ".\NUL"
        icacls $sourceTempIEPath /deny "Everyone:(RD)" | Out-Null
    }

    if (Test-Path $destHistoryPath) {
        Start-Process -FilePath "cmd.exe" -ArgumentList "/c mklink /J `"$sourceHistoryPath`" `"$destHistoryPath`"" -NoNewWindow -Wait -RedirectStandardOutput ".\NUL"
        icacls $sourceHistoryPath /deny "Everyone:(RD)" | Out-Null
    }

    WriteLogInfo("Exiting the CreateJunctionPoints")
}

function ImportModules {
    WriteLogInfo("Entering the ImportModules")
    $Global:importModulesCnt++

    # Already installed and imported ad module, no need to do anything
    $moduleADAvailable = Get-Module *activedirectory*
    if ($null -ne $moduleADAvailable) {
        return
    }

    # Not imported but already installed
    try {
        if ($Global:importModulesCnt -eq 1) {
            Write-Host "`r`nPlease wait while PowerShell is importing the necessary Windows modules. " -ForegroundColor Yellow
        }
 
        Import-Module ActiveDirectory -ErrorAction Stop -WarningAction SilentlyContinue | Out-Null

        return
    } catch {

        WriteLogError("Import the Active Directory module failed, the reason is: $_")

    }

    # Uninstalled
    # Check OS Version
    $curOS = ''
    try {
        if ($Global:importModulesCnt -eq 1) {
            Write-Host "`r`nPlease wait while PowerShell is installing the necessary Windows additional features. " -ForegroundColor Yellow
        }
 
        $curOS = wmic os get Caption

        if ($curOS[2].Contains('Server')) {
            Import-Module ServerManager

            if ($null -eq $moduleADAvailable) {
                Install-WindowsFeature -Name RSAT-AD-PowerShell -ErrorAction Stop -WarningAction SilentlyContinue | Out-Null
            }    			
        } elseif ($curOS[2].Contains('Windows 11')) {

            if ($null -eq $moduleADAvailable) {
                Add-WindowsCapability -online -Name Rsat.ActiveDirectory.DS-LDS.Tools~~~~0.0.1.0  -ErrorAction Stop -WarningAction SilentlyContinue | Out-Null
            }

        } else {
            # win10	
            if ($null -eq $moduleADAvailable) {
                try {
                    Add-WindowsCapability -online -Name Rsat.ActiveDirectory.DS-LDS.Tools~~~~0.0.1.0 -ErrorAction Stop -WarningAction SilentlyContinue  | Out-Null
                } catch {
                    # Old versions up to 1803
                    Enable-WindowsOptionalFeature -Online -FeatureName RSATClient-Roles-AD-Powershell  -ErrorAction Stop -WarningAction SilentlyContinue | Out-Null
                } 
            }     
        }

        Import-Module ActiveDirectory -ErrorAction Stop -WarningAction SilentlyContinue  | Out-Null

    } catch {
        WriteLogError("Install or import Active Directory module failed, the reason is: $_")

        if ($Global:importModulesCnt -eq 1) {
            return $false
        } else {
            if ($curOS[2].Contains('Server')) {
                Write-Host 'Unable to use this tool because imports for the following modules failed: Active Directory.' -ForegroundColor Yellow		
            } else {
                Write-Host 'Unable to use this tool because imports for the following modules failed: Active Directory.' -ForegroundColor Yellow		
            }

            Write-Host 'If it does not work, restart the machine, make sure the Windows update service is running, and then run this tool again.' -ForegroundColor Yellow
            Start-Sleep -Seconds 30
            Exit
        }
    }

    return $true
}

function ExpandStorePath {
    param (
	    [Parameter(Mandatory=$true)]
		[string] $storePath,
		[Parameter(Mandatory=$true)]
		[string] $username,
		[Parameter(Mandatory=$false)]
		[string] $osName=""
	)

	$expandedStorePath = $storePath

	WriteLogInfo("Entering the ExpandStorePath, storePath is $storePath, username is $username, osName is $osName")
	$computerName = $env:COMPUTERNAME
	$wmi = Get-WmiObject -Class Win32_ComputerSystem -ComputerName $computerName
    $fullDomain = $wmi.Domain
	$domain = $fullDomain.Split('.')[0]

	if ($storePath.ToLower().Contains("%username%")) {
		$expandedStorePath = $expandedStorePath -replace "%username%", $username
	}

	if ($storePath.ToLower().Contains("%userdomain%")) {
		$expandedStorePath = $expandedStorePath -replace "%userdomain%", $domain
	}

	if ($storePath.ToLower().Contains("!ctx_osbitness!")) {
		$expandedStorePath = $expandedStorePath -replace "!ctx_osbitness!", "x64"
	}

	if ($storePath.ToLower().Contains("!ctx_osname!")) {
		$expandedStorePath = $expandedStorePath -replace "!ctx_osname!", $osName
	}

    $adAttributePattern = '#(\w+)#'

    # Find all matches of #adattribute# in the user store path string
    $adAttributes = [regex]::Matches($storePath, $adAttributePattern)

    foreach ($adAttribute in $adAttributes) {
        # Get the attribute name from the match
        $attributeName = $adAttribute.Groups[1].Value

        # Get the Active Directory user object
        $userAndAttribute = Get-ADUser -Identity "$username" -Properties $attributeName

        # Check if the user object is not null
        if ($null -ne $userAndAttribute) {
            # Retrieve the value of the attribute
            $adAttributeValue = $userAndAttribute.$attributeName

            # Replace #adattribute# with the actual attribute value
            $expandedStorePath = $expandedStorePath -replace [regex]::Escape($adAttribute.Value), $adAttributeValue
        } else {
            Write-Output "User not found or attribute '$attributeName' does not exist."
        }
    }

    WriteLogInfo("The expanded store path is $expandedStorePath")
	return $expandedStorePath
}

function MigrateLocalProfile {
    param (
        [Parameter(Mandatory=$true)]
        [array] $userMembers,

        [Parameter(Mandatory=$true)]
        [string] $upmVhdxStorePath,

        [Parameter(Mandatory=$true)]
        [string] $osShortName,
		
		[Parameter(Mandatory=$false)]
		[PSCredential] $remoteCredential = $null
    )

    WriteLogInfo("Entering the MigrateLocalProfile")

    $computerName = $env:COMPUTERNAME
    $wmi = Get-WmiObject -Class Win32_ComputerSystem -ComputerName $computerName
    $fullDomain = $wmi.Domain
	$domain = $fullDomain.Split('.')[0]


    $upmStorePathArray = $upmVhdxStorePath -Split "\\%"
	$upmActualStorePath = $upmStorePathArray[0]

    if (-not (Test-Path $upmActualStorePath)) {
        Write-Host "$upmActualStorePath is not accessible, exit the migration work"
        WriteLogError("$upmActualStorePath is not accessible, exit the migration work")
		WriteJsonFile -UserName ("N/A") -Status "Failed" -ErrorMessage "Unable to access target file share $upmActualStorePath" -isClear $false
        return
    }

	if ($remoteCredential -ne $null) {
        New-PSDrive -Name "SharedDriveCPM" -PSProvider FileSystem -Root $upmActualStorePath -Credential $remoteCredential -Persist
		$upmVhdxStorePath = $upmVhdxStorePath.Replace($upmActualStorePath, "SharedDriveCPM:")
		WriteLogInfo("The remote credential path is $upmVhdxStorePath")
	}

    $successCount = 0
    $totalCount = $userMembers.Length
    $destinationVhdxFolder = $null
    $destinationVhdxPath = $null
    $driveLetterPath = $null
    $driveLetter = $null
    $fullDriveLetter = $null

    ForEach ($user in $userMembers) {
        if (StopMigrationProcess) {
            WriteLogInfo("Get the stop migration message, exit the migration process")
            exit			
        }

        $samAccountName = $user.SamAccountName
		$upmVhdxStoreExpandedPath = ExpandStorePath -storePath $upmVhdxStorePath -username $samAccountName -osName $osShortName

        $sid = (New-Object System.Security.Principal.NTAccount($samAccountName)).translate([System.Security.Principal.SecurityIdentifier]).Value
        $localProfilePath = Get-ItemProperty -Path "HKLM:\SOFTWARE\Microsoft\Windows NT\CurrentVersion\ProfileList\$sid" | Select-Object -ExpandProperty "ProfileImagePath"

        WriteLogInfo("LocalProfilePath is $localProfilePath")
        WriteLogInfo("User SID is $sid")
        WriteJsonFile -UserName ("$domain" + "\$samAccountName") -Status "In Progress" -ErrorMessage "N/A" -isClear $false

        if (Test-Path $localProfilePath) {

            $destinationVhdxFolder = Join-Path -Path (Join-Path -Path "$upmVhdxStoreExpandedPath" -ChildPath "ProfileContainer") -ChildPath "$osShortName"
            WriteLogInfo("DestinationVhdxFolder is $destinationVhdxFolder")
            if (-not (Test-Path $destinationVhdxFolder)) {
                New-Item -Path $destinationVhdxFolder -ItemType Directory | Out-Null
            }

            # Set permissions from destination folder
            try {
                #Recursively set owner, so only icacls the outermost directory
                icacls $upmVhdxStoreExpandedPath /setowner $domain\$samAccountName /T /C | Out-Null
                #After changing owner, need reset the acl
                icacls $upmVhdxStoreExpandedPath /reset /T | Out-Null
            } catch {
                WriteLogError("Set the $upmVhdxStoreExpandedPath folder permission failed")
                WriteLogError("Error: $_")
            }

            # Define destination file path
            $destinationVhdxPath = Join-Path -Path $destinationVhdxFolder -ChildPath "ProfileContainer.VHDX"
            WriteLogInfo("DestinationVhdxPath is $destinationVhdxPath")
            Write-Progress -Activity "Processing $samAccountName" -Status "In progress" -PercentComplete 10

            # Get available drive letter
            $usedDriveLetters = Get-WmiObject Win32_LogicalDisk | Select-Object -ExpandProperty DeviceID
            $availableDriveLetters = [char[]](68..90) | ForEach-Object { [string]([char]$_) } | Where-Object { $_ + ':' -notin $usedDriveLetters }
            $driveLetter = $availableDriveLetters[0]

            $scriptBlockCreate = {
                param($vhdxPath)
                "create vdisk file=`"$vhdxPath`" maximum 50000 type=expandable" | diskpart
            }

            $scriptBlockAttach = {
                param($vhdxPath)
                "select vdisk file=`"$vhdxPath`"`r`nattach vdisk" | diskpart
            }

            $scriptBlockFormatAndAssign = {
                param($vhdxPath, $assignedLetter)
                "select vdisk file=`"$vhdxPath`"`r`ncreate partition primary`r`nformat quick`r`nassign letter=`"$assignedLetter`"" | diskpart
            }

            $scriptBlockDetach = {
                param($vhdxPath)
                "select vdisk file`"$vhdxPath`"`r`ndetach vdisk" | diskpart
            }

            try {

                if (-not (test-path $destinationVhdxPath)) {
                    $jobCreate = Start-Job -ScriptBlock $scriptBlockCreate -ArgumentList $destinationVhdxPath
                    Wait-Job $jobCreate | Out-Null
                    $jobAttach = Start-Job -ScriptBlock $scriptBlockAttach -ArgumentList $destinationVhdxPath
                    Wait-Job $jobAttach | Out-Null
                    $jobFormatAndAssign = Start-Job -ScriptBlock $scriptBlockFormatAndAssign -ArgumentList $destinationVhdxPath, $driveLetter
                    Wait-Job $jobFormatAndAssign | Out-Null
                  
                    $fullDriveLetter = $driveLetter + ":"
                    & label $fullDriveLetter $samAccountName-Profile
                    $driveLetterPath = $fullDriveLetter + "\Profiles"
                    WriteLogInfo("DriveLetterPath is $driveLetterPath")

                    New-Item -Path $driveLetterPath -ItemType directory | Out-Null
                    icacls $driveLetterPath /inheritance:r | Out-Null
                    icacls $driveLetterPath /grant "Administrators:(OI)(CI)F" /T | Out-Null
                    icacls $driveLetterPath /grant "$domain\$samAccountName`:(OI)(CI)F" /T | Out-Null
                    icacls $driveLetterPath /grant "SYSTEM:(OI)(CI)F" /T | Out-Null
                    icacls $driveLetterPath /setowner $domain\$samAccountName | Out-Null
                } else {
                    WriteLogInfo("$destinationVhdxPath already exists, we don't need to do the migration")
					WriteJsonFile -UserName ("$domain" + "\$samAccountName") -Status "Failed" -ErrorMessage "Profile already exists at the target location $destinationVhdxPath." -isClear $false
                    Continue
                }

                Write-Progress -Activity "Processing $samAccountName" -Status "In progress" -PercentComplete 50

                $startTime = get-date
                & robocopy $localProfilePath $driveLetterPath /MIR /R:2 /W:1 /MT:8 /COPY:DATSOU /DCOPY:DAT /XJD | Out-Null
                $endTime = get-date
                WriteLogInfo("Finish copying from local profile to vhdx profile")
                $baseDirectory = (Get-WmiObject Win32_Volume | Where-Object { ($_.Label -eq "$samAccountName-Profile") -and ($_.DriveLetter -eq $fullDriveLetter) } | Select-Object DeviceID).DeviceID
                WriteLogInfo("baseDirectory is $baseDirectory")
				CreateJunctionPoints("$baseDirectory")
                $timeConsumed = ($endTime - $startTime).TotalMilliseconds/1000
                WriteLogInfo("Consumed total $timeConsumed seconds")
                $jobDetach = Start-Job -ScriptBlock $scriptBlockDetach -ArgumentList $destinationVhdxPath
                Wait-Job $jobDetach | Out-Null
                Write-Progress -Activity "Processing $samAccountName" -Status "In progress" -PercentComplete 90

                icacls $destinationVhdxPath /setowner $domain\$samAccountName /C | Out-Null
                icacls $destinationVhdxPath /reset | Out-Null
                Write-Progress -Activity "Processing $samAccountName" -Status "Completed" -PercentComplete 100
                $successCount++
                Write-Host "The migration for user $samAccountName was successfully completed." -ForegroundColor Green
				WriteJsonFile -UserName ("$domain" + "\$samAccountName") -Status "Success" -ErrorMessage "N/A" -isClear $false
                WriteLogInfo("The migration for user $samAccountName was successfully completed.")

            } catch {

                WriteLogError("$samAccountName migration failed, try the next user")
				WriteJsonFile -UserName ("$domain" + "\$samAccountName") -Status "Failed" -ErrorMessage "An error occurred. Check log for details." -isClear $false
                WriteLogError("Error: $_")

            }

        } else {

            WriteLogError("$localProfilePath is not accessible, skip this user")
			WriteJsonFile -UserName ("$domain" + "\$samAccountName") -Status "Failed" -ErrorMessage "Profile does not exist at the source location $localProfilePath." -isClear $false
            continue
   
        }
    }

    $failedCount = $totalCount - $successCount
    Write-Host ("`r`nMigration results:") -ForegroundColor Green
    Write-Host ("`r`n$successCount users were migrated successfully.") -ForegroundColor Green
    Write-Host ("`r`n$failedCount users were migrated failed.") -ForegroundColor Red
    Write-Host ("`r`nFor more information, see the log file at SystemDrive:\ProgramData\Citrix\ProfileManagement\Logs\ScriptTool\MigrationScriptTool.log.") -ForegroundColor Green

	if ($remoteCredential -ne $null) {
	    net use SharedDriveCPM: /delete /y
	}

    WriteLogInfo("There are a total of $totalCount users. Migration is successful for $successCount users and failed for $failedCount users.")
    WriteLogInfo("Exiting the MigrateLocalProfile")
}

function MigrateWindowsRoamingProfile {
    param (
        [Parameter()]
        [array] $userMembers,

        [Parameter(Mandatory=$true)]
        [string] $windowsRoamingStorePath,

        [Parameter(Mandatory=$true)]
        [string] $upmVhdxStorePath,

        [Parameter(Mandatory=$true)]
        [string] $osShortName,

        [Parameter(Mandatory=$false)]
        [PSCredential] $remoteCredential
    )

    WriteLogInfo("Entering the MigrateWindowsRoamingProfile")

    $computerName = $env:COMPUTERNAME
    $wmi = Get-WmiObject -Class Win32_ComputerSystem -ComputerName $computerName
    $fullDomain = $wmi.Domain
    $domain = $fullDomain.Split('.')[0]

    
    $windowsRoamingStorePathArray = $windowsRoamingStorePath -Split "\\%"
    $windowsRoamingActualStorePath = $windowsRoamingStorePathArray[0]
    
    $upmVhdxStorePathArray = $upmVhdxStorePath -Split "\\%"
    $upmActualVhdxStorePath = $upmVhdxStorePathArray[0]

    if (-not (Test-Path $upmActualVhdxStorePath)) {
        Write-Host "$upmActualVhdxStorePath is not accessible, exit the migration work"
        WriteLogError("$upmActualVhdxStorePath is not accessible, exit the migration work")
        WriteJsonFile -UserName ("N/A") -Status "Failed" -ErrorMessage "Unable to access target file share $upmActualVhdxStorePath." -isClear $false
        return
    }

    if ($remoteCredential -ne $null) {
        New-PSDrive -Name "SharedDriveCPM1" -PSProvider FileSystem -Root $upmActualVhdxStorePath -Credential $remoteCredential -Persist
        $upmVhdxStorePath = $upmVhdxStorePath.Replace($upmActualVhdxStorePath, "SharedDriveCPM1:")
        WriteLogInfo("The remote credential of UPM vhdx store path is $upmVhdxStorePath")

        New-PSDrive -Name "SharedDriveCPM2" -PSProvider FileSystem -Root $windowsRoamingActualStorePath -Credential $remoteCredential -Persist
        $windowsRoamingStorePath = $windowsRoamingStorePath.Replace($windowsRoamingActualStorePath, "SharedDriveCPM2:")
        WriteLogInfo("The remote credential of windows roaming store path is $windowsRoamingStorePath")
    }

    $successCount = 0
    $totalCount = $userMembers.Length
    $destinationVhdxFolder = $null
    $destinationVhdxPath = $null
    $destinationVhdxContainerFolder = $null
    $driveLetterPath = $null
    $driveLetter = $null
    $fullDriveLetter = $null

    foreach ($user in $userMembers) {
        if (StopMigrationProcess) {
            WriteLogInfo("Get the stop migration message, exit the migration process")
            exit			
        }

        $samAccountName = $user.SamAccountName
        $sid = (New-Object System.Security.Principal.NTAccount($samAccountName)).translate([System.Security.Principal.SecurityIdentifier]).Value
        $windowsRoamingStoreExpandedPath = ExpandStorePath -storePath $windowsRoamingStorePath -username $samAccountName -osName $osShortName
        $upmVhdxStoreExpandedPath = ExpandStorePath -storePath $upmVhdxStorePath -username $samAccountName -osName $osShortName

        # Check if the profile path includes the profile version number when the profile path is not found
        # please refer to the windows roaming profile document, the profile path may include the profile version number after the username
        if (-not (Test-Path $windowsRoamingStoreExpandedPath)) {

            if ($windowsRoamingStorePath.ToLower().EndsWith("%username%")) {
                $windowsRoamingStoreExpandedPath = $windowsRoamingStoreExpandedPath + ".V6"
            }

            WriteLogInfo("$windowsRoamingStoreExpandedPath have included the profile version number")
        }

        $sourceProfileFolder = $windowsRoamingStoreExpandedPath
        WriteLogInfo("SourceProfileFolder is $sourceProfileFolder")
        WriteLogInfo("User SID is $sid")
        WriteJsonFile -UserName ("$domain" + "\$samAccountName") -Status "In Progress" -ErrorMessage "N/A" -isClear $false

        if (Test-Path $sourceProfileFolder) {
            $destinationVhdxFolder = Join-Path -Path (Join-Path -Path "$upmVhdxStoreExpandedPath" -ChildPath "ProfileContainer") -ChildPath "$osShortName"
            WriteLogInfo("DestinationVhdxFolder is $destinationVhdxFolder")

            if (-not (Test-Path $destinationVhdxFolder)) {
                New-Item -Path $destinationVhdxFolder -ItemType Directory | Out-Null
            }

            # Set permissions from destination folder
            try {

                $destinationVhdxContainerFolder = Join-Path -Path "$upmVhdxStoreExpandedPath" -ChildPath "ProfileContainer"
                #Recursively set owner, so only icacls the outermost directory
                icacls $destinationVhdxContainerFolder /setowner $domain\$samAccountName /T /C | Out-Null
                #After changing owner, need reset the acl
                icacls $destinationVhdxContainerFolder /reset /T | Out-Null

            } catch {

                WriteLogError("Set the $destinationVhdxFolder folder permission failed")
                WriteLogError("Error: $_")

            }

            # Define destination file path
            $destinationVhdxPath = Join-Path -Path $destinationVhdxFolder -ChildPath "ProfileContainer.VHDX"
            WriteLogInfo("DestinationVhdxPath is $destinationVhdxPath")
            Write-Progress -Activity "Processing $samAccountName" -Status "In progress" -PercentComplete 10

            # Get available drive letter
            $usedDriveLetters = Get-WmiObject Win32_LogicalDisk | Select-Object -ExpandProperty DeviceID
            $availableDriveLetters = [char[]](68..90) | ForEach-Object { [string]([char]$_) } | Where-Object { $_ + ':' -notin $usedDriveLetters }
            $driveLetter = $availableDriveLetters[0]

            $scriptBlockCreate = {
                param($vhdxPath)
                "create vdisk file=`"$vhdxPath`" maximum 50000 type=expandable" | diskpart
            }

            $scriptBlockAttach = {
                param($vhdxPath)
                "select vdisk file=`"$vhdxPath`"`r`nattach vdisk" | diskpart
            }

            $scriptBlockFormatAndAssign = {
                param($vhdxPath, $assignedLetter)
                "select vdisk file=`"$vhdxPath`"`r`ncreate partition primary`r`nformat quick`r`nassign letter=`"$assignedLetter`"" | diskpart
            }

            $scriptBlockDetach = {
                param($vhdxPath)
                "select vdisk file`"$vhdxPath`"`r`ndetach vdisk" | diskpart
            }

            try {

                if (-not (test-path $destinationVhdxPath)) {
                    $jobCreate = Start-Job -ScriptBlock $scriptBlockCreate -ArgumentList $destinationVhdxPath
                    Wait-Job $jobCreate | Out-Null
                    $jobAttach = Start-Job -ScriptBlock $scriptBlockAttach -ArgumentList $destinationVhdxPath
                    Wait-Job $jobAttach | Out-Null
                    $jobFormatAndAssign = Start-Job -ScriptBlock $scriptBlockFormatAndAssign -ArgumentList $destinationVhdxPath, $driveLetter
                    Wait-Job $jobFormatAndAssign | Out-Null

                    $fullDriveLetter = $driveLetter + ":"
                    & label $fullDriveLetter $samAccountName-Profile
                    $driveLetterPath = $fullDriveLetter + "\Profiles"
                    WriteLogInfo("DriveLetterPath is $driveLetterPath")

                    New-Item -Path $driveLetterPath -ItemType directory | Out-Null
                    icacls $driveLetterPath /inheritance:r | Out-Null
                    icacls $driveLetterPath /grant "Administrators:(OI)(CI)F" /T | Out-Null
                    icacls $driveLetterPath /grant "$domain\$samAccountName`:(OI)(CI)F" /T | Out-Null
                    icacls $driveLetterPath /grant "SYSTEM:(OI)(CI)F" /T | Out-Null
                    icacls $driveLetterPath /setowner $domain\$samAccountName | Out-Null
                } else {
                    WriteLogInfo("$destinationVhdxPath already exists, we don't need to do the migration")
					WriteJsonFile -UserName ("$domain" + "\$samAccountName") -Status "Failed" -ErrorMessage "Profile already exists at the target location $destinationVhdxPath." -isClear $false
                    Continue
                }

                Write-Progress -Activity "Processing $samAccountName" -Status "In progress" -PercentComplete 50

                $startTime = get-date
                & robocopy $sourceProfileFolder $driveLetterPath /MIR /R:2 /W:1 /MT:8 /COPY:DATSOU /DCOPY:DAT /XJD | Out-Null
                $endTime = get-date
                WriteLogInfo("Finish copying from windows roaming profile to cpm vhdx profile")

                $baseDirectory = (Get-WmiObject Win32_Volume | Where-Object { ($_.Label -eq "$samAccountName-Profile") -and ($_.DriveLetter -eq $fullDriveLetter) } | Select-Object DeviceID).DeviceID
                WriteLogInfo("baseDirectory is $baseDirectory")

				CreateJunctionPoints("$baseDirectory")

                $timeConsumed = ($endTime - $startTime).TotalMilliseconds/1000
                WriteLogInfo("Consumed total $timeConsumed seconds")

                $jobDetach = Start-Job -ScriptBlock $scriptBlockDetach -ArgumentList $destinationVhdxPath
                Wait-Job $jobDetach | Out-Null
                Write-Progress -Activity "Processing $samAccountName" -Status "In progress" -PercentComplete 90

                icacls $destinationVhdxPath /setowner $domain\$samAccountName /C | Out-Null
                icacls $destinationVhdxPath /reset | Out-Null
                Write-Progress -Activity "Processing $samAccountName" -Status "Completed" -PercentComplete 100

                $successCount++
                Write-Host "The migration for user $samAccountName was successfully completed." -ForegroundColor Green
                WriteLogInfo("The migration for user $samAccountName was successfully completed.")
				WriteJsonFile -UserName ("$domain" + "\$samAccountName") -Status "Success" -ErrorMessage "N/A" -isClear $false

            } catch {

                WriteLogError("$samAccountName migration failed, try the next user")
				WriteJsonFile -UserName ("$domain" + "\$samAccountName") -Status "Failed" -ErrorMessage "An error occurred. Check log for details." -isClear $false
                WriteLogError("Error: $_")
 
            }                      
        } else {
            WriteLogError("$sourceProfileFolder is not accessible, return")
			WriteJsonFile -UserName ("$domain" + "\$samAccountName") -Status "Failed" -ErrorMessage "Profile does not exist at the source location $sourceProfileFolder." -isClear $false
            Continue
        }
    }

	if (($remoteCredential -ne $null)) {
		net use SharedDriveCPM1: /delete /y
		net use SharedDriveCPM2: /delete /y
	}

    $failedCount = $totalCount - $successCount
    Write-Host ("`r`nMigration results:") -ForegroundColor Green
    Write-Host ("`r`n$successCount users were migrated successfully.") -ForegroundColor Green
    Write-Host ("`r`n$failedCount users were migrated failed.") -ForegroundColor Red
    Write-Host ("`r`nFor more information, see the log file at SystemDrive:\ProgramData\Citrix\ProfileManagement\Logs\ScriptTool\MigrationScriptTool.log.") -ForegroundColor Green
    WriteLogInfo("There are a total of $totalCount users. Migration is successful for $successCount users and failed for $failedCount users.")
    WriteLogInfo("Exiting the migrate windows roaming profile.")
}

function MigrateUPMProfile {
    param (
        [Parameter()]
        [array] $userMembers,

        [Parameter(Mandatory=$true)]
        [string] $upmProfileStorePath,

        [Parameter(Mandatory=$true)]
        [string] $upmVhdxStorePath,

        [Parameter(Mandatory=$true)]
        [string] $osShortName,

        [Parameter(Mandatory=$false)]
        [PSCredential] $remoteCredential
    )

    WriteLogInfo("Entering the MigrateUPMProfile")
    $computerName = $env:COMPUTERNAME
    $wmi = Get-WmiObject -Class Win32_ComputerSystem -ComputerName $computerName
    $fullDomain = $wmi.Domain
	$domain = $fullDomain.Split('.')[0]

	$upmVhdxStorePathArray = $upmVhdxStorePath -Split "\\%"
	$upmActualVhdxStorePath = $upmVhdxStorePathArray[0]

	$upmProfileStorePathArray = $upmProfileStorePath -Split "\\%"
	$upmActualProfileStorePath = $upmProfileStorePathArray[0]

    if (-not (Test-Path $upmActualVhdxStorePath)) {
        Write-Host "$upmActualVhdxStorePath is not accessible, exit the migration work"
        WriteLogError("$upmActualVhdxStorePath is not accessible, exit the migration work")
		WriteJsonFile -UserName ("N/A") -Status "Failed" -ErrorMessage "Unable to access target file share $upmActualVhdxStorePath." -isClear $false
        return
    }

	if (($remoteCredential -ne $null) -and ($upmProfileStorePath -eq $upmVhdxStorePath)) {
        New-PSDrive -Name "SharedDriveCPM" -PSProvider FileSystem -Root $upmActualVhdxStorePath -Credential $remoteCredential -Persist
		$upmVhdxStorePath = $upmVhdxStorePath.Replace($upmActualVhdxStorePath, "SharedDriveCPM:")
		$upmProfileStorePath = $upmProfileStorePath.Replace($upmActualProfileStorePath, "SharedDriveCPM:")
		WriteLogInfo("The remote credential path is $upmVhdxStorePath")
	} elseif (($remoteCredential -ne $null)) {
		New-PSDrive -Name "SharedDriveCPM1" -PSProvider FileSystem -Root $upmActualVhdxStorePath -Credential $remoteCredential -Persist
		$upmVhdxStorePath = $upmVhdxStorePath.Replace($upmActualVhdxStorePath, "SharedDriveCPM1:")

		New-PSDrive -Name "SharedDriveCPM2" -PSProvider FileSystem -Root $upmActualProfileStorePath -Credential $remoteCredential -Persist
		$upmProfileStorePath = $upmProfileStorePath.Replace($upmActualProfileStorePath, "SharedDriveCPM2:")
	}

    $successCount = 0
    $totalCount = $userMembers.Length
    $destinationVhdxFolder = $null
    $destinationVhdxPath = $null
    $destinationVhdxContainerFolder = $null
    $driveLetterPath = $null
    $driveLetter = $null
    $fullDriveLetter = $null

    foreach ($user in $userMembers) {
		if (StopMigrationProcess) {
		    WriteLogInfo("Get the stop migration message, exit the migration process")
            exit			
		}

        $samAccountName = $user.SamAccountName
        $sid = (New-Object System.Security.Principal.NTAccount($samAccountName)).translate([System.Security.Principal.SecurityIdentifier]).Value
		$upmVhdxStoreExpandedPath = ExpandStorePath -storePath $upmVhdxStorePath -username $samAccountName -osName $osShortName
		$upmProfileStoreExpandedPath = ExpandStorePath -storePath $upmProfileStorePath -username $samAccountName -osName $osShortName

        $sourceProfileFolder = Join-Path -Path "$upmProfileStoreExpandedPath" -ChildPath "UPM_Profile"
        WriteLogInfo("SourceProfileFolder is $sourceProfileFolder")
        WriteLogInfo("User SID is $sid")
        WriteJsonFile -UserName ("$domain" + "\$samAccountName") -Status "In Progress" -ErrorMessage "N/A" -isClear $false

        if (Test-Path $sourceProfileFolder) {
            $destinationVhdxFolder = Join-Path -Path (Join-Path -Path "$upmVhdxStoreExpandedPath" -ChildPath "ProfileContainer") -ChildPath "$osShortName"
            WriteLogInfo("DestinationVhdxFolder is $destinationVhdxFolder")
            if (-not (Test-Path $destinationVhdxFolder)) {
                New-Item -Path $destinationVhdxFolder -ItemType Directory | Out-Null
            }

            # Set permissions from destination folder
            try {

                $destinationVhdxContainerFolder = Join-Path -Path "$upmVhdxStoreExpandedPath" -ChildPath "ProfileContainer"
                #Recursively set owner, so only icacls the outermost directory
                icacls $destinationVhdxContainerFolder /setowner $domain\$samAccountName /T /C | Out-Null
                #After changing owner, need reset the acl
                icacls $destinationVhdxContainerFolder /reset /T | Out-Null

            } catch {

                WriteLogError("Set the $destinationVhdxFolder folder permission failed")
                WriteLogError("Error: $_")

            }

            # Define destination file path
            $destinationVhdxPath = Join-Path -Path $destinationVhdxFolder -ChildPath "ProfileContainer.VHDX"
            WriteLogInfo("DestinationVhdxPath is $destinationVhdxPath")
            Write-Progress -Activity "Processing $samAccountName" -Status "In progress" -PercentComplete 10

            # Get available drive letter
            $usedDriveLetters = Get-WmiObject Win32_LogicalDisk | Select-Object -ExpandProperty DeviceID
            $availableDriveLetters = [char[]](68..90) | ForEach-Object { [string]([char]$_) } | Where-Object { $_ + ':' -notin $usedDriveLetters }
            $driveLetter = $availableDriveLetters[0]

            $scriptBlockCreate = {
                param($vhdxPath)
                "create vdisk file=`"$vhdxPath`" maximum 50000 type=expandable" | diskpart
            }

            $scriptBlockAttach = {
                param($vhdxPath)
                "select vdisk file=`"$vhdxPath`"`r`nattach vdisk" | diskpart
            }

            $scriptBlockFormatAndAssign = {
                param($vhdxPath, $assignedLetter)
                "select vdisk file=`"$vhdxPath`"`r`ncreate partition primary`r`nformat quick`r`nassign letter=`"$assignedLetter`"" | diskpart
            }

            $scriptBlockDetach = {
                param($vhdxPath)
                "select vdisk file`"$vhdxPath`"`r`ndetach vdisk" | diskpart
            }

            try {
                if (-not (test-path $destinationVhdxPath)) {
                    $jobCreate = Start-Job -ScriptBlock $scriptBlockCreate -ArgumentList $destinationVhdxPath
                    Wait-Job $jobCreate | Out-Null
                    $jobAttach = Start-Job -ScriptBlock $scriptBlockAttach -ArgumentList $destinationVhdxPath
                    Wait-Job $jobAttach | Out-Null
                    $jobFormatAndAssign = Start-Job -ScriptBlock $scriptBlockFormatAndAssign -ArgumentList $destinationVhdxPath, $driveLetter
                    Wait-Job $jobFormatAndAssign | Out-Null

                    $fullDriveLetter = $driveLetter + ":"
                    & label $fullDriveLetter $samAccountName-Profile
                    $driveLetterPath = $fullDriveLetter + "\Profiles"
                    WriteLogInfo("DriveLetterPath is $driveLetterPath")

                    New-Item -Path $driveLetterPath -ItemType directory | Out-Null
                    icacls $driveLetterPath /inheritance:r | Out-Null
                    icacls $driveLetterPath /grant "Administrators:(OI)(CI)F" /T | Out-Null
                    icacls $driveLetterPath /grant "$domain\$samAccountName`:(OI)(CI)F" /T | Out-Null
                    icacls $driveLetterPath /grant "SYSTEM:(OI)(CI)F" /T | Out-Null
                    icacls $driveLetterPath /setowner $domain\$samAccountName | Out-Null
                } else {
                    WriteLogError("$destinationVhdxPath already exists, we don't need to do the migration")
					WriteJsonFile -UserName ("$domain" + "\$samAccountName") -Status "Failed" -ErrorMessage "Profile already exists at the target location $destinationVhdxPath." -isClear $false
                    Continue
                }

                Write-Progress -Activity "Processing $samAccountName" -Status "In progress" -PercentComplete 50

                $startTime = get-date
                & robocopy $sourceProfileFolder $driveLetterPath /MIR /R:2 /W:1 /MT:8 /COPY:DATSOU /DCOPY:DAT /XJD | Out-Null
                $endTime = get-date
                WriteLogInfo("Finish copying from UPM file-based profile to vhdx profile")
 
                $baseDirectory = (Get-WmiObject Win32_Volume | Where-Object { ($_.Label -eq "$samAccountName-Profile") -and ($_.DriveLetter -eq $fullDriveLetter) } | Select-Object DeviceID).DeviceID
                WriteLogInfo("baseDirectory is $baseDirectory")

				CreateJunctionPoints("$baseDirectory")

                $timeConsumed = ($endTime - $startTime).TotalMilliseconds/1000
                WriteLogInfo("Consumed total $timeConsumed seconds")

                $jobDetach = Start-Job -ScriptBlock $scriptBlockDetach -ArgumentList $destinationVhdxPath
                Wait-Job $jobDetach | Out-Null
                Write-Progress -Activity "Processing $samAccountName" -Status "In progress" -PercentComplete 90

                icacls $destinationVhdxPath /setowner $domain\$samAccountName /C | Out-Null
                icacls $destinationVhdxPath /reset | Out-Null
                Write-Progress -Activity "Processing $samAccountName" -Status "Completed" -PercentComplete 100

                $successCount++
                Write-Host "The migration for user $samAccountName was successfully completed." -ForegroundColor Green
                WriteLogInfo("The migration for user $samAccountName was successfully completed.")
				WriteJsonFile -UserName ("$domain" + "\$samAccountName") -Status "Success" -ErrorMessage "N/A" -isClear $false
            } catch {
                WriteLogError("$samAccountName migration failed, try the next user")
				WriteJsonFile -UserName ("$domain" + "\$samAccountName") -Status "Failed" -ErrorMessage "An error occurred. Check log for details." -isClear $false
                WriteLogError("Error: $_")
            }                      
        } else {
            WriteLogError("$sourceProfileFolder is not accessible, return")
			WriteJsonFile -UserName ("$domain" + "\$samAccountName") -Status "Failed" -ErrorMessage "Profile does not exist at the source location $sourceProfileFolder." -isClear $false
            Continue
        }
    }

	if (($remoteCredential -ne $null) -and ($upmProfileStorePath -eq $upmVhdxStorePath)) {
	    net use SharedDriveCPM: /delete /y
	} elseif (($remoteCredential -ne $null)) {
		net use SharedDriveCPM1: /delete /y
		net use SharedDriveCPM2: /delete /y
	}

    $failedCount = $totalCount - $successCount
    Write-Host ("`r`nMigration results:") -ForegroundColor Green
    Write-Host ("`r`n$successCount users were migrated successfully.") -ForegroundColor Green
    Write-Host ("`r`n$failedCount users were migrated failed.") -ForegroundColor Red
    Write-Host ("`r`nFor more information, see the log file at SystemDrive:\ProgramData\Citrix\ProfileManagement\Logs\ScriptTool\MigrationScriptTool.log.") -ForegroundColor Green
    WriteLogInfo("There are a total of $totalCount users.Migration is successful for $successCount users and failed for $failedCount users.")
    WriteLogInfo("Exiting the MigrateUPMProfile.")
}

function MigrateFSLogixProfile {
    param (
        [Parameter()]
        [array] $userMembers,

        [Parameter(Mandatory=$true)]
        [string] $fslogixStorePath,

        [Parameter(Mandatory=$true)]
        [string] $upmVhdxStorePath,

        [Parameter(Mandatory=$true)]
        [string] $osShortName,
			
		[Parameter(Mandatory=$false)]
		[PSCredential] $remoteCredential = $null
    )

	WriteLogInfo("Entering the MigrateFSLogixProfile function")
	$computerName = $env:COMPUTERNAME
    $wmi = Get-WmiObject -Class Win32_ComputerSystem -ComputerName $computerName
    $fullDomain = $wmi.Domain
	$domain = $fullDomain.Split('.')[0]

	$userToProcess = @()

    foreach ($user in $userMembers) {
		if (StopMigrationProcess) {
		    WriteLogInfo("Get the stop migration message, exit the migration process")
            exit			
		}

        # User from the file corresponds to SAM
        $samAccountName = $user.SamAccountName
        # Read SID based on SamAccountName
        $sid = (New-Object System.Security.Principal.NTAccount($samAccountName)).translate([System.Security.Principal.SecurityIdentifier]).Value
        # Defining the path to the FSLogix vhdx path
        $sourceVhdFolderPath = $null
        $sidAndAccount = "$sid" + "_$samAccountName"
        $expandedFslogixStorePath = ExpandStorePath -storePath $fslogixStorePath -username $samAccountName
        WriteLogInfo("The expanded FSLogix store path is $expandedFslogixStorePath in the MigrateFSLogixProfile function")

        $sourceVhdFolderPath = Join-Path -Path "$expandedFslogixStorePath" -ChildPath "$sidAndAccount"
        WriteLogInfo("The source vhd/vhdx path is $sourceVhdFolderPath in the MigrateFSLogixProfile function")

        # Check if the source vhd/vhdx path exists, if not, revert the SID and SamAccountName
        # Maybe the customer configure the flipflopprofiledirectoryname policy, so we need to revert the sid and SamAccountName
        if (-not (Test-Path $sourceVhdFolderPath)) {
            WriteLogWarning("$sourceVhdFolderPath is not accessible, we revert the SID and SamAccountName")

            $sidAndAccount = "$samAccountName" + "_$sid"
            $sourceVhdFolderPath = Join-Path -Path "$expandedFslogixStorePath" -ChildPath "$sidAndAccount"

            WriteLogInfo("Revert the SID and SamAccountName, the sourceVhdFolderPath is $sourceVhdFolderPath, the sidAndAccount is $sidAndAccount")
        }

		$vhdxFlag = $true
        # Get a list of files in the directory
        $files = Get-ChildItem -Path $sourceVhdFolderPath

        # Iterate through each file and check the extension
        foreach ($file in $files) {
            if ($file.Extension -eq ".vhd") {
                $vhdxFlag = $false
            } elseif ($file.Extension -eq ".vhdx"){
                $vhdxFlag = $true
            }
        }

		$userToProcess += $user

        WriteJsonFile -UserName ("$domain" + "\$samAccountName") -Status "In Progress" -ErrorMessage "N/A" -isClear $false

        if ($vhdxFlag) {
			MigrateFSLogixVHDX -userMembers $userToProcess -fslogixVhdxStorePath $fslogixStorePath -upmVhdxStorePath $upmVhdxStorePath -osShortName $osShortName -remoteCredential $remoteCredential
		} else {
			MigrateFSLogixVHD -userMembers $userToProcess -fslogixVhdStorePath $fslogixStorePath -upmVhdxStorePath $upmVhdxStorePath -osShortName $osShortName -remoteCredential $remoteCredential
		}

		$userToProcess.Clear()

        WriteLogInfo("The user $samAccountName was successfully completed.")
	}

    WriteLogInfo("Exiting the MigrateFSLogixProfile function")
}


function MigrateFSLogixVHD {
    param (
        [Parameter()]
        [array] $userMembers,

        [Parameter(Mandatory=$true)]
        [string] $fslogixVhdStorePath,

        [Parameter(Mandatory=$true)]
        [string] $upmVhdxStorePath,

        [Parameter(Mandatory=$true)]
        [string] $osShortName,

        [Parameter(Mandatory=$false)]
        [PSCredential] $remoteCredential = $null
    )

    WriteLogInfo("Entering the MigrateFSLogixVHD")
    $computerName = $env:COMPUTERNAME
    $wmi = Get-WmiObject -Class Win32_ComputerSystem -ComputerName $computerName
    $fullDomain = $wmi.Domain
	$domain = $fullDomain.Split('.')[0]

    $successCount = 0
    $totalCount = $userMembers.Length
    $fullDestDriveLetter = $null
    $destDriveLetter = $null
    $destDriveLetterPath = $null


    $fslogixStorePathArray = $fslogixVhdStorePath -Split "\\%"
    $fslogixActualStorePath = $fslogixStorePathArray[0]

	$upmStorePathArray = $upmVhdxStorePath -Split "\\%"
	$upmActualStorePath = $upmStorePathArray[0]

    if (-not (Test-Path $upmActualStorePath)) {
        Write-Host "$upmActualStorePath is not accessible, exit the migration work"
        WriteLogError("$upmActualStorePath is not accessible, exit the migration work")
        return
    }

	if (($remoteCredential -ne $null) -and ($fslogixActualStorePath -eq $upmActualStorePath)) {
		New-PSDrive -Name "SharedDriveCPM" -PSProvider FileSystem -Root $fslogixActualStorePath -Credential $remoteCredential -Persist
		$upmVhdxStorePath = $upmVhdxStorePath.Replace($upmActualStorePath, "SharedDriveCPM:")
        $fslogixVhdStorePath = $fslogixVhdStorePath.Replace($fslogixActualStorePath, "SharedDriveCPM:")
    
		WriteLogInfo("fslogixActualStorePath is $fslogixActualStorePath, fslogixVhdStorePath is $fslogixVhdStorePath, upmActualStorePath is $upmActualStorePath, upmVhdxStorePath is $upmVhdxStorePath")
	} elseif (($remoteCredential -ne $null)) {
		New-PSDrive -Name "SharedDriveCPM1" -PSProvider FileSystem -Root $fslogixActualStorePath -Credential $remoteCredential -Persist
		$fslogixVhdStorePath = $fslogixVhdStorePath.Replace($fslogixActualStorePath, "SharedDriveCPM1:")

		New-PSDrive -Name "SharedDriveCPM2" -PSProvider FileSystem -Root $upmActualStorePath -Credential $remoteCredential -Persist
		$upmVhdxStorePath = $upmVhdxStorePath.Replace($upmActualStorePath, "SharedDriveCPM2:")

        WriteLogInfo("fslogixActualStorePath is $fslogixActualStorePath, fslogixVhdStorePath is $fslogixVhdStorePath, upmActualStorePath is $upmActualStorePath, upmVhdxStorePath is $upmVhdxStorePath")
	}

    foreach ($user in $userMembers) {
        # User from the file corresponds to SAM
        $samAccountName = $user.SamAccountName
        # Read SID based on SamAccountName
        $sid = (New-Object System.Security.Principal.NTAccount($samAccountName)).translate([System.Security.Principal.SecurityIdentifier]).Value

        # Defining the path to the FSLogix vhdx path		
		$upmVhdxStoreExpandedPath = ExpandStorePath -storePath $upmVhdxStorePath -username $samAccountName -osName $osShortName

        $sourceVhdPath = $null
        $destinationVhdxFolder = $null
        $destinationVhdxAccountFolder = $null
        $sidAndAccount = "$sid" + "_$samAccountName"

        $expandedFslogixStorePath = ExpandStorePath -storePath $fslogixVhdStorePath -username $samAccountName
        $sourceVhdPath = Join-Path -Path (Join-Path -Path "$expandedFslogixStorePath" -ChildPath "$sidAndAccount") -ChildPath "Profile_$samAccountName.VHD"

        # Check if the source vhd/vhdx path exists, if not, revert the SID and SamAccountName
        # Maybe the customer configure the flipflopprofiledirectoryname policy, so we need to revert the SID and SamAccountName
        if (-not (Test-Path $sourceVhdPath)) {
            WriteLogWarning("$sourceVhdPath is not accessible, we revert the SID and SamAccountName")

            $sidAndAccount = "$samAccountName" + "_$sid"
            $sourceVhdPath = Join-Path -Path (Join-Path -Path "$expandedFslogixStorePath" -ChildPath "$sidAndAccount") -ChildPath "Profile_$samAccountName.VHD"

            WriteLogInfo("Revert the SID and SamAccountName, the sourceVhdPath is $sourceVhdPath, the sidAndAccount is $sidAndAccount")
        }

        WriteLogInfo("The source vhd path is $sourceVhdPath")
		WriteLogInfo("The dest vhd path is $upmVhdxStoreExpandedPath")

        if (Test-Path $sourceVhdPath) { 
            $destinationVhdxFolder = Join-Path -Path (Join-Path -Path "$upmVhdxStoreExpandedPath" -ChildPath "ProfileContainer") -ChildPath "$osShortName"
            if (-not (Test-Path $destinationVhdxFolder)) {
                New-Item -Path $destinationVhdxFolder -ItemType Directory | Out-Null
            }
			WriteLogInfo("The destinationVhdxFolder is $destinationVhdxFolder")

            # Set permissions from destination folder
            try {
                $destinationVhdxAccountFolder = $upmVhdxStoreExpandedPath
                #Recursively set owner, so only icacls the outermost directory
                icacls $destinationVhdxAccountFolder /setowner $domain\$samAccountName /T /C | Out-Null
                #After changing owner, need reset the acl
                icacls $destinationVhdxAccountFolder /reset /T | Out-Null
            } catch {
                WriteLogError("Set the $destinationVhdxFolder folder permission failed")
                WriteLogError("Error: $_")
            }

            # Define migration's destination vhdx file path in the user store
            $destinationVhdxPath = Join-Path -Path $destinationVhdxFolder -ChildPath "ProfileContainer.VHDX"
            WriteLogInfo("DestinationVhdxPath is $destinationVhdxPath")
            Write-Progress -Activity "Processing $samAccountName" -Status "In progress" -PercentComplete 10

            # Get available drive letter for the destination vhdx
            $usedDriveLetters = Get-WmiObject Win32_LogicalDisk | Select-Object -ExpandProperty DeviceID
            $availableDriveLetters = [char[]](68..90) | ForEach-Object { [string]([char]$_) } | Where-Object { $_ + ':' -notin $usedDriveLetters }
            $destDriveLetter = $availableDriveLetters[0]

            $scriptBlockCreate = {
                param($vhdxPath)
                "create vdisk file=`"$vhdxPath`" maximum 50000 type=expandable" | diskpart
            }

            $scriptBlockAttach = {
                param($vhdxPath)
                "select vdisk file=`"$vhdxPath`"`r`nattach vdisk" | diskpart
            }

            $scriptBlockFormatAndAssign = {
                param($vhdxPath, $assignedLetter)
                "select vdisk file=`"$vhdxPath`"`r`ncreate partition primary`r`nformat quick`r`nassign letter=`"$assignedLetter`"" | diskpart
            }

            $scriptBlockDetach = {
                param($vhdxPath)
                "select vdisk file`"$vhdxPath`"`r`ndetach vdisk" | diskpart
            }

            try {
                if (-not (test-path $destinationVhdxPath)) {
                    $jobCreate = Start-Job -ScriptBlock $scriptBlockCreate -ArgumentList $destinationVhdxPath
                    Wait-Job $jobCreate | Out-Null
                    $jobAttach = Start-Job -ScriptBlock $scriptBlockAttach -ArgumentList $destinationVhdxPath
                    Wait-Job $jobAttach | Out-Null
                    $jobFormatAndAssign = Start-Job -ScriptBlock $scriptBlockFormatAndAssign -ArgumentList $destinationVhdxPath, $destDriveLetter
                    Wait-Job $jobFormatAndAssign | Out-Null

                    $fullDestDriveLetter = $destDriveLetter + ":"
                    & label $fullDestDriveLetter $samAccountName-Profile
                    $destDriveLetterPath = $fullDestDriveLetter + "\Profiles"
                    WriteLogInfo("destDriveLetterPath is $destDriveLetterPath")

                    New-Item -Path $destDriveLetterPath -ItemType directory | Out-Null

                    icacls $destDriveLetterPath /inheritance:r | Out-Null
                    icacls $destDriveLetterPath /grant "Administrators:(OI)(CI)F" /T | Out-Null
                    icacls $destDriveLetterPath /grant "$domain\$samAccountName`:(OI)(CI)F" /T | Out-Null
                    icacls $destDriveLetterPath /grant "SYSTEM:(OI)(CI)F" /T | Out-Null
                    icacls $destDriveLetterPath /setowner $domain\$samAccountName | Out-Null
                } else {
                    WriteLogError("$destinationVhdxPath already exists, we don't need to do the migration")
					WriteJsonFile -UserName ("$domain" + "\$samAccountName") -Status "Failed" -ErrorMessage "Profile already exists at the target location $destinationVhdxPath." -isClear $false
                    Continue
                }
                Write-Progress -Activity "Processing $samAccountName" -Status "In progress" -PercentComplete 50

                # Mound Disk Image
                Mount-DiskImage -ImagePath $SourceVhdPath -NoDriveLetter | Out-Null
                Start-Sleep -Seconds 2
                # Get available drive letter
                $partition = Get-DiskImage -ImagePath $SourceVhdPath | Get-Disk | Get-Partition
                $usedDriveLetters = Get-WmiObject Win32_LogicalDisk | Select-Object -ExpandProperty DeviceID
                $availableDriveLetters = [char[]](68..90) | ForEach-Object { [string]([char]$_) } | Where-Object { $_ + ':' -notin $usedDriveLetters }
                $partition | Set-Partition -NewDriveLetter $availableDriveLetters[0]
                $driveLetter = (Get-DiskImage -ImagePath $SourceVhdPath | Get-Disk | Get-Partition).DriveLetter
                $mountPoint = ($driveLetter + ':\')
                WriteLogInfo("Mountpoint is: $mountPoint")

                # Define path in the profile disk
                $fslogixDiskProfileFolder = "Profile"
                $fslogixDiskProfilePath = Join-Path -Path $MountPoint -ChildPath $fslogixDiskProfileFolder
                Write-Progress -Activity "Processing $samAccountName" -Status "In progress" -PercentComplete 60

                $startTime = get-date
                & robocopy $fslogixDiskProfilePath $destDriveLetterPath /MIR /R:2 /W:1 /MT:8 /COPY:DATSOU /DCOPY:DAT /XJD | Out-Null
                $endTime = get-date

                $baseDirectory = (Get-WmiObject Win32_Volume | Where-Object { ($_.Label -eq "$samAccountName-Profile") -and ($_.DriveLetter -eq $fullDestDriveLetter) } | Select-Object DeviceID).DeviceID
                WriteLogInfo("baseDirectory is $baseDirectory")

				CreateJunctionPoints("$baseDirectory")

                WriteLogInfo("Finish copying from FSLogix vhd profile to vhdx profile")

                $timeConsumed = ($endTime - $startTime).TotalMilliseconds/1000
                WriteLogInfo("Consumed total $timeConsumed seconds")

                $jobDetach = Start-Job -ScriptBlock $scriptBlockDetach -ArgumentList $destinationVhdxPath
                Wait-Job $jobDetach | Out-Null
                Write-Progress -Activity "Processing $samAccountName" -Status "In progress" -PercentComplete 90

                # Short delay and unmount the disk image
                Start-Sleep -Seconds 2
                Dismount-DiskImage -ImagePath $SourceVhdPath | Out-Null

                icacls $destinationVhdxPath /setowner $domain\$samAccountName /C | Out-Null
                icacls $destinationVhdxPath /reset | Out-Null
                Write-Progress -Activity "Processing $samAccountName" -Status "Completed" -PercentComplete 100
				WriteJsonFile -UserName ("$domain" + "\$samAccountName") -Status "Success" -ErrorMessage "N/A" -isClear $false
                $successCount++

                Write-Host "The migration for user $samAccountName was successfully completed." -ForegroundColor Green
                WriteLogInfo("The migration for user $samAccountName was successfully completed.")
            } catch {
                WriteLogError("$samAccountName migration failed, try the next user")
				WriteJsonFile -UserName ("$domain" + "\$samAccountName") -Status "Failed" -ErrorMessage "An error occurred. Check log for details." -isClear $false
                WriteLogError("Error: $_")
            }            
        }
    }

	if (($remoteCredential -ne $null) -and ($fslogixActualStorePath -eq $upmActualStorePath)) {
		net use SharedDriveCPM: /delete /y
	} elseif (($remoteCredential -ne $null)) {
		net use SharedDriveCPM1: /delete /y
		net use SharedDriveCPM2: /delete /y
	}

    $failedCount = $totalCount - $successCount
    Write-Host ("`r`nMigration results:") -ForegroundColor Green
    Write-Host ("`r`n$successCount users were migrated successfully.") -ForegroundColor Green
    Write-Host ("`r`n$failedCount users were migrated failed.") -ForegroundColor Red
    Write-Host ("`r`nFor more information, see the log file at SystemDrive:\ProgramData\Citrix\ProfileManagement\Logs\ScriptTool\MigrationScriptTool.log.") -ForegroundColor Green

    WriteLogInfo("There are a total of $totalCount users.Migration is successful for $successCount users and failed for $failedCount users.")
    WriteLogInfo("Exiting the MigrateFSLogixVHD.")
}

function MigrateFSLogixVHDX {
    param (
        [Parameter()]
        [array] $userMembers,

        [Parameter(Mandatory=$true)]
        [string] $fslogixVhdxStorePath,

        [Parameter(Mandatory=$true)]
        [string] $upmVhdxStorePath,

        [Parameter(Mandatory=$true)]
        [string] $osShortName,

        [Parameter(Mandatory=$false)]
        [PSCredential] $remoteCredential = $null
    )

    WriteLogInfo("Entering the MigrateFSLogixVHDX")
    $computerName = $env:COMPUTERNAME
    $wmi = Get-WmiObject -Class Win32_ComputerSystem -ComputerName $computerName
    $fullDomain = $wmi.Domain
	$domain = $fullDomain.Split('.')[0]

    $successCount = 0
    $totalCount = $userMembers.Length

    $fslogixStorePathArray = $fslogixVhdxStorePath -Split "\\%"
    $fslogixActualStorePath = $fslogixStorePathArray[0]

    $upmStorePathArray = $upmVhdxStorePath -Split "\\%"
	$upmActualStorePath = $upmStorePathArray[0]

    if (-not (Test-Path $upmActualStorePath)) {
        Write-Host "$upmActualStorePath is not accessible, exit the migration work"
        WriteLogInfo("$upmActualStorePath is not accessible, exit the migration work")
		WriteJsonFile -UserName ("N/A") -Status "Failed" -ErrorMessage "Unable to access target location $upmActualStorePath" -isClear $false
        return
    }

	WriteLogInfo("fslogixVhdxStorePath is $fslogixVhdxStorePath, upmActualStorePath is $upmActualStorePath, upmVhdxStorePath is $upmVhdxStorePath")
	if (($remoteCredential -ne $null) -and ($fslogixActualStorePath -eq $upmActualStorePath)) {
		New-PSDrive -Name "SharedDriveCPM" -PSProvider FileSystem -Root $fslogixActualStorePath -Credential $remoteCredential
        $fslogixVhdxStorePath = $fslogixVhdxStorePath.Replace($fslogixActualStorePath, "SharedDriveCPM:")
		$upmVhdxStorePath = $upmVhdxStorePath.Replace($upmActualStorePath, "SharedDriveCPM:")

		WriteLogInfo("fslogixVhdxStorePath is $fslogixVhdxStorePath, upmActualStorePath is $upmActualStorePath, upmVhdxStorePath is $upmVhdxStorePath")
	}
	elseif (($remoteCredential -ne $null)) {
		New-PSDrive -Name "SharedDriveCPM1" -PSProvider FileSystem -Root $fslogixActualStorePath -Credential $remoteCredential
        $fslogixVhdxStorePath = $fslogixVhdxStorePath.Replace($fslogixActualStorePath, "SharedDriveCPM1:")

		New-PSDrive -Name "SharedDriveCPM2" -PSProvider FileSystem -Root $upmActualStorePath -Credential $remoteCredential
		$upmVhdxStorePath = $upmVhdxStorePath.Replace($upmActualStorePath, "SharedDriveCPM2:")
 
		WriteLogInfo("fslogixVhdxStorePath is $fslogixVhdxStorePath, fslogixactualstorepath is $fslogixActualStorePath, upmActualStorePath is $upmActualStorePath, upmVhdxStorePath is $upmVhdxStorePath")
	}

    foreach ($user in $userMembers) {
        $samAccountName = $user.SamAccountName
        $sid = (New-Object System.Security.Principal.NTAccount($samAccountName)).translate([System.Security.Principal.SecurityIdentifier]).Value
		$upmVhdxStoreExpandedPath = ExpandStorePath -storePath $upmVhdxStorePath -username $samAccountName -osName $osShortName	

        $sourceVhdxPath = $null
        $destinationVhdxFolder = $null
        $destinationVhdxAccountFolder = $null
        $sidAndAccount = "$sid" + "_$samAccountName"

        $expandedFslogixStorePath = ExpandStorePath -storePath $fslogixVhdxStorePath -username $samAccountName
        $sourceVhdxPath = Join-Path -Path (Join-Path -Path "$expandedFslogixStorePath" -ChildPath "$sidAndAccount") -ChildPath "Profile_$samAccountName.VHDX"

        # Check if the source vhd/vhdx path exists, if not, revert the SID and SamAccountName
        # Maybe the customer configure the flipflopprofiledirectoryname policy, so we need to revert the SID and SamAccountName
        if (-not (Test-Path $sourceVhdxPath)) {
            WriteLogWarning("$sourceVhdPath is not accessible, we revert the SID and SamAccountName")
 
            $sidAndAccount = "$samAccountName" + "_$sid"
            $sourceVhdxPath = Join-Path -Path (Join-Path -Path "$expandedFslogixStorePath" -ChildPath "$sidAndAccount") -ChildPath "Profile_$samAccountName.VHDX"

            WriteLogInfo("Revert the SID and SamAccountName, the sourceVhdxPath is $sourceVhdxPath, the sidAndAccount is $sidAndAccount")
        }

        WriteLogInfo("The source vhdx path is $sourceVhdxPath")

        if (Test-Path $sourceVhdxPath) {
            $destinationVhdxFolder = Join-Path -Path (Join-Path -Path "$upmVhdxStoreExpandedPath" -ChildPath "ProfileContainer") -ChildPath "$osShortName"
            if (-not (Test-Path $destinationVhdxFolder)) {
                WriteLogInfo("Create Folder: $destinationVhdxFolder")
                New-Item -Path $destinationVhdxFolder -ItemType Directory | Out-Null
            }

            # Set permissions from destination folder
            try {
                $destinationVhdxAccountFolder = $upmVhdxStoreExpandedPath
                #recursive set owner, so only icacls the outermost directory
                icacls $destinationVhdxAccountFolder /setowner $domain\$samAccountName /T /C | Out-Null
                icacls $destinationVhdxAccountFolder /reset /T | Out-Null
            } catch {
                WriteLogError("Set the $destinationVhdxFolder folder permission failed")
                WriteLogError("Error: $_")
            }

            # Define destination file path
            $destinationVhdxPath = Join-Path -Path $destinationVhdxFolder -ChildPath "ProfileContainer.VHDX"
			if (Test-Path -Path $destinationVhdxPath) {
				WriteLogInfo("$destinationVhdxPath already exists, we don't need to do the migration")
				WriteJsonFile -UserName ("$domain" + "\$samAccountName") -Status "Failed" -ErrorMessage "Profile already exists at the target location $destinationVhdxPath." -isClear $false
                Continue
			}

            # Copy profile disk to new destination
            try {
                WriteLogInfo("Copying $sourceVhdxPath to $destinationVhdxPath")

                Write-Progress -Activity "Processing $samAccountName" -Status "In progress" -PercentComplete 10
                Copy-Item -Path $sourceVhdxPath -Destination $destinationVhdxPath | Out-Null
                Write-Progress -Activity "Processing $samAccountName" -Status "In progress" -PercentComplete 30

                icacls $destinationVhdxPath /setowner $domain\$samAccountName /C | Out-Null
                icacls $destinationVhdxPath /reset | Out-Null

                # Mound Disk Image
                Mount-DiskImage -ImagePath $destinationVhdxPath -NoDriveLetter | Out-Null
                Start-Sleep -Seconds 5
                WriteLogInfo("Mount-DiskImage $destinationVhdxPath success")
                Write-Progress -Activity "Processing $samAccountName" -Status "In progress" -PercentComplete 70

                # Get drive letter
                $partition = Get-DiskImage -ImagePath $destinationVhdxPath | Get-Disk | Get-Partition
                $usedDriveLetters = Get-WmiObject Win32_LogicalDisk | Select-Object -ExpandProperty DeviceID
                $availableDriveLetters = [char[]](68..90) | ForEach-Object { [string]([char]$_) } | Where-Object { $_ + ':' -notin $usedDriveLetters }
                $partition | Set-Partition -NewDriveLetter $availableDriveLetters[0]
                $driveLetter = (Get-DiskImage -ImagePath $destinationVhdxPath | Get-Disk | Get-Partition).DriveLetter
                $mountPoint = ($driveLetter + ':\')
                WriteLogInfo("Mountpoint is: $mountPoint")

                # Define path in the profile disk
                $fslogixDiskProfileFolder = "Profile"
                $upmDiskProfileFolder = "Profiles"
                $fslogixDiskProfilePath = Join-Path -Path $MountPoint -ChildPath $fslogixDiskProfileFolder 
                $upmDiskProfilePath = Join-Path -Path $MountPoint -ChildPath $upmDiskProfileFolder
                Rename-Item -Path $fslogixDiskProfilePath -NewName $upmDiskProfilePath
                Write-Progress -Activity "Processing $samAccountName" -Status "In progress" -PercentComplete 90
                WriteLogInfo("Rename from $fslogixDiskProfilePath to $upmDiskProfilePath")

                $fslogixRecycleBinPath = Join-Path -Path $MountPoint -ChildPath "PROFILE_RECYCLE.BIN"
                $upmRecycleBinPath = Join-Path -Path $MountPoint -ChildPath "`$RECYCLE.BIN"
                WriteLogInfo("FSLogix recyclebin path is $fslogixRecycleBinPath")
                WriteLogInfo("upmRecycleBinPath is $upmRecycleBinPath")

                if ((Test-Path $fslogixRecycleBinPath) -and (HasValidateFileInRecyclebin($fslogixRecycleBinPath))) {

                    $currentUserName = "$env:USERNAME"
                    icacls $fslogixRecycleBinPath /remove:g users | Out-Null
                    icacls $fslogixRecycleBinPath /inheritance:d | Out-Null
                    icacls $fslogixRecycleBinPath /setowner "$domain\$currentUserName" | Out-Null
                    icacls $fslogixRecycleBinPath /remove:g users | Out-Null
                    icacls $fslogixRecycleBinPath /remove:g EVERYONE | Out-Null
                    icacls $fslogixRecycleBinPath /remove:g SYSTEM | Out-Null
                    icacls $fslogixRecycleBinPath /remove:g Administrators | Out-Null
                    icacls $fslogixRecycleBinPath /remove:g "restricted" | Out-Null
                    icacls $fslogixRecycleBinPath /remove:g "all application packages" | Out-Null
                    icacls $fslogixRecycleBinPath /remove:g "CREATOR OWNER" | Out-Null
                    icacls $fslogixRecycleBinPath /remove:g "$domain\$currentUserName" | Out-Null
                    icacls $fslogixRecycleBinPath /remove:g "$domain\$samAccountName" | Out-Null
                    icacls $fslogixRecycleBinPath /grant "SYSTEM:(OI)(CI)F" | Out-Null
                    icacls $fslogixRecycleBinPath /grant "Administrators:(OI)(CI)F" | Out-Null
                    icacls $fslogixRecycleBinPath /grant "$domain\$samAccountName`:(OI)(CI)F" | Out-Null
                    icacls $fslogixRecycleBinPath /setowner $domain\$samAccountName | Out-Null
                    Rename-Item -Path $fslogixRecycleBinPath -NewName $upmRecycleBinPath | Out-Null

                    WriteLogInfo("We have set permission for the recycle bin")
                } else {
                    WriteLogError("FSLogix recycle bin not exist or have not available file")
                }

                # Short delay and unmount the disk image
                Start-Sleep -Seconds 2
                Dismount-DiskImage -ImagePath $destinationVhdxPath | Out-Null
                WriteLogInfo("Dismount the $destinationVhdxPath")
                Write-Progress -Activity "Processing $samAccountName" -Status "Completed" -PercentComplete 100
				WriteJsonFile -UserName ("$domain" + "\$samAccountName") -Status "Success" -ErrorMessage "N/A" -isClear $false

                $successCount++
                Write-Host "The migration for user $samAccountName was successfully completed." -ForegroundColor Green
                WriteLogInfo("The migration for user $samAccountName was successfully completed.")
            } catch {
                WriteLogError("$samAccountName migration failed, try the next user")
                WriteLogError("Error: $_")
				WriteJsonFile -UserName ("$domain" + "\$samAccountName") -Status "Failed" -ErrorMessage "An error occurred. Check log for details." -isClear $false
            }
        } else {
            WriteLogError("$SourceVhdxPath is not exists")
        }
    }

	if (($remoteCredential -ne $null) -and ($fslogixVhdxStorePath -eq $upmActualStorePath)) {
		Remove-PSDrive -Name "SharedDriveCPM"
	}
	elseif (($remoteCredential -ne $null)) {
		Remove-PSDrive -Name "SharedDriveCPM1"
		Remove-PSDrive -Name "SharedDriveCPM2"
	}

    $failedCount = $totalCount - $successCount
    Write-Host ("`r`nMigration results:") -ForegroundColor Green
    Write-Host ("`r`n$successCount users were migrated successfully.") -ForegroundColor Green
    Write-Host ("`r`n$failedCount users were migrated failed.") -ForegroundColor Red
    Write-Host ("`r`nFor more information, see the log file at SystemDrive:\ProgramData\Citrix\ProfileManagement\Logs\ScriptTool\MigrationScriptTool.log.") -ForegroundColor Green

    WriteLogInfo("Migration results:$successCount users were migrated successfully.")
    WriteLogInfo("$failedCount users could not be migrated.")
}

function HasValidateFileInRecyclebin {
    param (
        [Parameter(Mandatory=$true)]
        [string]$fslogixRecycleBinPath
    )

    $files = Get-ChildItem -Path $fslogixRecycleBinPath -File -Force

    if ($files.Count -eq 1 -and $files.Name -eq "desktop.ini") {
        WriteLogInfo("The directory contains only one file named desktop.ini.")

        return $False
    } else {
        WriteLogInfo("The directory does not meet the required conditions.")

        return $True
    }
}

function GetPendingMigrationUsers {
    param (
        [Parameter(Mandatory=$true)]
        [string]$userAndGroup
    )

    WriteLogInfo("Entering the GetPendingMigrationUsers, the userAndGroup is $userAndGroup")
    $usersAndGroups = $userAndGroup -Split ","
    $usersAndGroups = $usersAndGroups | ForEach-Object { $_.Trim() }

    $userMembers = @()

    foreach ($item in $usersAndGroups) {
        if ($item -notmatch '^[^\\]+\\[^\\]+$') {
            WriteLogError("$item is not the valid format")
            return $false
        }

        $domainObject = $item -split "\\"
        $domain = $domainObject[0]
        $objectName = $domainObject[1]

        if (Get-ADUser -Filter {SamAccountName -eq $objectName} -Server $domain) {
            $userMembers += Get-ADUser -Identity $objectName -Server $domain
        } elseif (Get-ADGroup -Filter {SamAccountName -eq $objectName} -Server $domain) {
            $groupMembers = Get-ADGroupMember -Identity $objectName -Server $domain
            $userMembers += $groupMembers | Where-Object {$_.objectClass -eq 'user'}
        } else {
            Write-Host "We can not find any domain user named $objectName" -ForegroundColor Red
            WriteLogError("We can not find any domain user named $objectName")
			WriteJsonFile -UserName ("$domain" + "\$objectName") -Status "Failed" -ErrorMessage "The user cannot be found." -isClear $false
            Continue
        }
    }

    $usersCount = $userMembers.Length

    if ($usersCount -gt 0) {
        Write-Host "A total of $usersCount users will be migrated. Please refer to the log file(SystemDrive:\ProgramData\ProfileManagement\Logs\ScriptTool\MigrationScriptTool.log) for the specific list of users." -ForegroundColor Green
        foreach ($userMember in $userMembers) {
            $samAccountName = $userMember.SamAccountName
            WriteLogInfo("Will processing account: $samAccountName")
        }
    } else {
        Write-Host "0 user in the input users and groups" -ForegroundColor Yellow
        WriteLogWarning("0 user in the input users and groups")
    }

    WriteLogInfo("Exiting the GetPendingMigrationUsers")
    return $userMembers
}

function DisplayWelcomeInfo {
    WriteLogInfo("Entering the DisplayWelcomeInfo")

    Write-Host "===============================================================================================================================================" -ForegroundColor Green
    Write-Host "Welcome to the Citrix Profile Container Migration Tool!`r`n" -ForegroundColor Green
    Write-Host "This tool lets you migrate user profiles from your current profile solution to Citrix container-based profile solution. It supports the following migration types:`r`n" -ForegroundColor Green
    Write-Host "- Local Profile Migration: Migrate Windows local profiles to Citrix container-based profile solution`r`n" -ForegroundColor Green
    Write-Host "- Citrix File-Based Profile Solution Migration: Migrate user profiles from Citrix file-based profile solution to Citrix container-based profile solution`r`n" -ForegroundColor Green
    Write-Host "- FSLogix Profile Container Migration: Migrate user profiles from FSLogix Profile Container to Citrix container-based profile solution`r`n" -ForegroundColor Green
    Write-Host "- Windows Roaming Profile Migration: Migrate user profiles from Windows Roaming Profile to Citrix container-based profile solution`r`n" -ForegroundColor Green
    Write-Host "================================================================================================================================================`r`n" -ForegroundColor Green
    Write-Host "Before you begin, make sure you meet the following prerequisites:`r`n" -ForegroundColor Green
    Write-Host "    - Set up the Citrix user store and configure its Windows Access Control Lists (ACL). For more information, see https://docs.citrix.com/en-us/profile-management/current-release/install-and-set-up/create-user-store.html." -ForegroundColor Green
    Write-Host "    - Run this tool using a domain admin account.`r`n" -ForegroundColor Green
    Write-Host "Choose a migration type by entering the corresponding number (1-4):`r`n" -ForegroundColor Green
    Write-Host "[1] Local Profile Migration`r`n" -ForegroundColor Green
    Write-Host "[2] Citrix File-Based Profile Solution Migration`r`n" -ForegroundColor Green
    Write-Host "[3] FSLogix Profile Container (VHDX) Migration`r`n" -ForegroundColor Green
    Write-Host "[4] FSLogix Profile Container (VHD) Migration`r`n" -ForegroundColor Green
    Write-Host "[5] Windows Roaming Profile Migration`r`n" -ForegroundColor Green

    WriteLogInfo("Exiting the DisplayWelcomeInfo")
}

function DisplayUserAndGroupPrompts {
    WriteLogInfo("Entering the DisplayUserAndGroupPrompts")

    Write-Host "`r`nSpecify users and groups you want to migrate, separated by commas. Use the following form: <DOMAIN NAME>\<USER NAME>,<DOMAIN NAME>\<GROUP NAME>" -ForegroundColor Green

    WriteLogInfo("Exiting the DisplayUserAndGroupPrompts")
}

function DisplayFslogixStorePrompts {
    WriteLogInfo("Entering the DisplayFslogixStorePrompts")
    Write-Host "`r`nSpecify the FSLogix VHDX location. %USERNAME%, %USERDOMAIN% and AD attributes are supported." -ForegroundColor Green
}

function DisplayUPMStorePrompts {
    WriteLogInfo("Entering the DisplayUPMStorePrompts")

    Write-Host "`r`nSpecify the path to Citrix VHDX store.`r`n*Notes: You can use the Citrix user store as the Citrix VHDX store, or specify a different network path as the Citrix VHDX store. %USERNAME% must be included, %USERDOMAIN% and AD attributes are supported." -ForegroundColor Green

    WriteLogInfo("Exiting the DisplayUPMStorePrompts")
}

function DisplayWindowsRoamingStorePrompts {
    WriteLogInfo("Entering the DisplayWindowsRoamingStorePrompts")

    Write-Host "`r`nSpecify the path to the Windows Roaming Profile store. %USERNAME% must be included, %USERDOMAIN% and AD attributes are supported." -ForegroundColor Green

    WriteLogInfo("Exiting the DisplayWindowsRoamingStorePrompts")
}

function DisplayOsnamePrompts {
    WriteLogInfo("Entering the DisplayOsnamePrompts")

    Write-Host "`r`nSpecify the Windows OS version of your machines by entering its short name." -ForegroundColor Green
    Write-Host "*Note: Only user profiles with the specified OS type will be migrated." -ForegroundColor Green
    Write-Host "Short names of commonly-used Windows OS versions:" -ForegroundColor Green
    Write-Host "|Long name              |Short name|" -ForegroundColor Green
    Write-Host "|Windows server 2022    |Win2022   |" -ForegroundColor Green
    Write-Host "|Windows server 2019    |Win2019   |" -ForegroundColor Green
    Write-Host "|Windows server 2016    |Win2016   |" -ForegroundColor Green
    Write-Host "|Windows 11             |Win11     |" -ForegroundColor Green
    Write-Host "|Windows 10 22H2        |Win10_22H2|" -ForegroundColor Green
    Write-Host "|Windows 10 21H2        |Win10_21H2|" -ForegroundColor Green
    Write-Host "|Windows 10 Redstone 6  |Win10RS6  |" -ForegroundColor Green
    Write-Host "|Windows 10 Redstone 5  |Win10RS5  |" -ForegroundColor Green
    Write-Host "To get short names of other Windows OS versions, see https://docs.citrix.com/en-us/profile-management/current-release/policies/settings.html." -ForegroundColor Green

    WriteLogInfo("Exiting the DisplayOsnamePrompts")
}

function DisplayFutureWork {
    WriteLogInfo("Entering the DisplayFutureWork")

    Write-Host "`r`nNext steps:" -ForegroundColor Green
    Write-Host "`r`nTo get started with Citrix container-based profile solution, configure the following settings:" -ForegroundColor Green
    Write-Host "`r`n- Enable Profile Management" -ForegroundColor Green
    Write-Host "`r`n- Path to user store" -ForegroundColor Green
    Write-Host "`r`n- Enable profile container" -ForegroundColor Green
    Write-Host "`r`n- (Optional) Customize storage path for VHD files" -ForegroundColor Green
    Write-Host "`r`nFor more information, see https://docs.citrix.com/en-us/profile-management/current-release.html." -ForegroundColor Green

    WriteLogInfo("Exiting the DisplayFutureWork")
}

function CheckOsShortNameFormat {
    param (
        [Parameter(Mandatory=$true)]
        [string]$osShortName
    )

    WriteLogInfo("Entering the CheckOsShortNameFormat")
    $osShortNameArray = @("Win10RS6","Win10RS5","Win10RS4","Win10RS3","Win10RS2","Win10RS1","Win11","Win10", "Win10_20H1", "Win10_20H2", "Win10_21H1", "Win10_21H2", "Win10_22H2", "Win2025", "Win2022", "Win2019", "Win2016")

    foreach ($shortName in $osShortNameArray) {
        if ($osShortName -eq $shortName) {
            WriteLogInfo("$osShortName is ok")

            return $true
        }
    }

    if (-not $osShortName.StartsWith("Win")) {
        WriteLogError("$osShortName is not valid")

        return $false
    }

    WriteLogInfo("Exiting the CheckOsShortNameFormat")
    return $false
}

function CheckFslogixStorePath {
    param (
        [Parameter(Mandatory=$true)]
        [string]$fslogixStorePath
    )

    WriteLogInfo("Entering the CheckFslogixStorePath, the fslogixStorePath is $fslogixStorePath")

    if (Test-Path -Path $fslogixStorePath) {
        WriteLogInfo("$fslogixStorePath can be accessed without variable")
        return $true
    }

    $removeUserStorePath = $fslogixStorePath -replace "%username%", ""
	WriteLogInfo("$removeUserStorePath is after remove username variable")

    if (Test-Path -Path $removeUserStorePath) {
        WriteLogInfo("$fslogixStorePath can be accessed after removing variable")
        return $true
    }

    WriteLogInfo("Exiting the CheckFslogixStorePath")
    return $false
}

function CheckUpmStorePath {
    param (
        [Parameter(Mandatory=$true)]
        [string]$upmStorePath
    )

    WriteLogInfo("Entering the CheckUpmStorePath, the upmStorePath is $upmStorePath")

    if (-not $upmStorePath.ToLower().Contains("%username%")) {
        WriteLogError("$upmStorePath does not contain USERNAME")
        return $false
    }
	
	$upmStorePathArray = $upmStorePath -Split "\\%"
	$upmActualStorePath = $upmStorePathArray[0]

    if (Test-Path -Path $upmActualStorePath) {
        WriteLogInfo("$upmActualStorePath can be accessed")
        return $true
    }

    WriteLogError("$upmActualStorePath can not be accessed")
    WriteLogError("$upmStorePath can not be accessed, exiting the CheckUpmStorePath")
    return $false
}

function MigrationProfilesGuideMode {
    WriteLogInfo("Entering the MigrationProfilesGuideMode")

    $choice = (Read-Host  "Enter number (1-5)").Trim()
    while (($choice -ne 1) -and ($choice -ne 2) -and ($choice -ne 3) -and ($choice -ne 4) -and ($choice -ne 5)) {
        Write-Host "`r`nInvalid value, make sure that you enter the right number 1-4" -ForegroundColor Yellow 
        Write-Host "`r`nIf you want to exit this tool, please input exit" -ForegroundColor Yellow

        $choice = (Read-Host  "`r`nEnter number (1-5)").Trim()
        if ($choice -eq "exit") {
            WriteLogInfo("User entered exit")
            exit
        }
    }

    WriteLogInfo("User selected migration type: $choice")

    switch ($choice) {
        1 {
            DisplayUserAndGroupPrompts
            $userAndGroup = Read-Host "Enter users and groups"
            $users = GetPendingMigrationUsers -userAndGroup $userAndGroup
            while ($users -eq $false) {
                Write-Host "`r`nThe user or group format you enter is invalid, please enter it again!" -ForegroundColor Red
                Write-Host "`r`nIf you want to exit this tool, please enter exit" -ForegroundColor Yellow

                $userAndGroup = Read-Host "`r`nEnter users and groups"
                if ($userAndGroup -eq "exit") {
                    WriteLogInfo("User enter exit")
                    exit
                }

                $users = GetPendingMigrationUsers -userAndGroup $userAndGroup
            }


            DisplayUPMStorePrompts
            $upmVhdxStorePath = Read-Host "Enter path"
            while ((CheckUpmStorePath -upmStorePath $upmVhdxStorePath) -eq $false) {
                Write-Host "`r`nThe Citrix VHDX store path you enter is invalid, please enter it again!" -ForegroundColor Red
                Write-Host "If you want to exit this tool, please enter exit" -ForegroundColor Green

                $upmVhdxStorePath = Read-Host "Please enter the Citrix VHDX store path"
                if ($upmVhdxStorePath -eq "exit") {
                    WriteLogInfo("User enter exit")
                    exit
                }
            }

            DisplayOsnamePrompts
            $osShortName = Read-Host "Enter short OS name"      
            while ((CheckOsShortNameFormat -osShortName $osShortName) -eq $false) {
                Write-Host "`r`nThe vda machine's os short name you enter is invalid, please enter it again!" -ForegroundColor Red
                Write-Host "`r`nIf you want to exit this tool, please enter exit" -ForegroundColor Green

                $osShortName = Read-Host "Please enter the vda machine's os short name"
                if ($osShortName -eq "exit") {
                    WriteLogInfo("User enter exit")
                    exit
                }
            }

            MigrateLocalProfile -userMembers $users -upmVhdxStorePath $upmVhdxStorePath -osShortName $osShortName
            DisplayFutureWork
        }

        2 {
            DisplayUserAndGroupPrompts
            $userAndGroup = Read-Host "Enter users and groups"
            $users = GetPendingMigrationUsers -userAndGroup $userAndGroup
            while ($users -eq $false) {
                Write-Host "`r`nThe user or group format you enter is invalid, please enter it again!" -ForegroundColor Red
                Write-Host "`r`nIf you want to exit this tool, please enter exit" -ForegroundColor Yellow

                $userAndGroup = Read-Host "`r`nEnter users and groups"
                if ($userAndGroup -eq "exit") {
                    WriteLogInfo("User entered exit")
                    exit
                }
                $users = GetPendingMigrationUsers -userAndGroup $userAndGroup
            }

            Write-Host "`r`nThe Citrix store path must be include %USERNAME%, %USERDOMAIN% and AD attributes are supported.`r`n" -ForegroundColor Green
            $upmProfileStorePath = Read-Host "Enter path"
            while ((CheckUpmStorePath -upmStorePath $upmProfileStorePath) -eq $false) {
                Write-Host "`r`nThe Citrix profile store path you enter is invalid, please enter it again!" -ForegroundColor Red
                Write-Host "If you want to exit this tool, please enter exit" -ForegroundColor Green

                $upmProfileStorePath = Read-Host "Enter path"
                if ($upmProfileStorePath -eq "exit") {
                    WriteLogInfo("User entered exit")
                    exit
                }
            }

            DisplayUPMStorePrompts
            $upmVhdxStorePath = Read-Host "Enter path"
            while ((CheckUpmStorePath -upmStorePath $upmVhdxStorePath) -eq $false) {
                Write-Host "`r`nThe Citrix VHDX store path you enter is invalid, please enter it again!" -ForegroundColor Red
                Write-Host "If you want to exit this tool, please enter exit" -ForegroundColor Green

                $upmVhdxStorePath = Read-Host "Enter path"
                if ($upmVhdxStorePath -eq "exit") {
                    WriteLogInfo("User entered exit")
                    exit
                }
            }

            DisplayOsnamePrompts
            $osShortName = Read-Host "Enter short OS name"
            while ((CheckOsShortNameFormat -osShortName $osShortName) -eq $false) {
                Write-Host "`r`nThe vda machine's os short name you enter is invalid, please enter it again!" -ForegroundColor Red
                Write-Host "`r`nIf you want to exit this tool, please enter exit" -ForegroundColor Green

                $osShortName = Read-Host "Enter short OS name"
                if ($osShortName -eq "exit") {
                    WriteLogInfo("User entered exit")
                    exit
                }
            }
            
            MigrateUPMProfile -userMembers $users -upmProfileStorePath $upmProfileStorePath -upmVhdxStorePath $upmVhdxStorePath -osShortName $osShortName
            DisplayFutureWork
        }

        3 {
            DisplayUserAndGroupPrompts
            $userAndGroup = Read-Host "Enter users and groups"
            $users = GetPendingMigrationUsers -userAndGroup $userAndGroup
            while ($users -eq $false) {
                Write-Host "`r`nThe user or group format you enter is invalid, please enter it again!" -ForegroundColor Red
                Write-Host "`r`nIf you want to exit this tool, please enter exit" -ForegroundColor Yellow

                $userAndGroup = Read-Host "`r`nEnter users and groups"
                if ($userAndGroup -eq "exit") {
                    WriteLogInfo("User entered exit")
                    exit
                }
                $users = GetPendingMigrationUsers -userAndGroup $userAndGroup
            }

            DisplayFslogixStorePrompts
            $fslogixVhdxStorePath = Read-Host "Enter path"
            while ((CheckFslogixStorePath -fslogixStorePath $fslogixVhdxStorePath) -eq $false) {
                Write-Host "`r`nThe FSLogix VHDX location you enter is invalid, please enter it again!" -ForegroundColor Red
                Write-Host "If you want to exit this tool, please enter exit" -ForegroundColor Green

                $fslogixVhdxStorePath = Read-Host "Enter path"
                if ($fslogixVhdxStorePath -eq "exit") {
                    WriteLogInfo("User entered exit")
                    exit
                }
            }

            DisplayUPMStorePrompts
            $upmVhdxStorePath = Read-Host "Enter path"
            while ((CheckUpmStorePath -upmStorePath $upmVhdxStorePath) -eq $false) {
                Write-Host "`r`nThe Citrix VHDX store path you enter is invalid, please enter it again!" -ForegroundColor Red
                Write-Host "If you want to exit this tool, please enter exit" -ForegroundColor Green

                $upmVhdxStorePath = Read-Host "Enter path"
                if ($upmVhdxStorePath -eq "exit") {
                    WriteLogInfo("User entered exit")
                    exit
                }
            }

            DisplayOsnamePrompts
            $osShortName = Read-Host "Enter short OS name"     
            while ((CheckOsShortNameFormat -osShortName $osShortName) -eq $false) {
                Write-Host "`r`nThe vda machine's os short name you enter is invalid, please enter it again!" -ForegroundColor Red
                Write-Host "`r`nIf you want to exit this tool, please enter exit" -ForegroundColor Green

                $osShortName = Read-Host "Enter short OS name"
                if ($osShortName -eq "exit") {
                    WriteLogInfo("User entered exit")
                    exit
                }
            }
            
            MigrateFSLogixVHDX -userMembers $users -fslogixVhdxStorePath $fslogixVhdxStorePath -upmVhdxStorePath $upmVhdxStorePath -osShortName $osShortName
            DisplayFutureWork
        }

        4 {
            DisplayUserAndGroupPrompts
            $userAndGroup = Read-Host "`r`nEnter users and groups"
            $usersTobeProcessed = GetPendingMigrationUsers -userAndGroup $userAndGroup
            while ($usersTobeProcessed -eq $false) {
                Write-Host "`r`nThe user or group format you enter is invalid, please enter it again!" -ForegroundColor Red
                Write-Host "`r`nIf you want to exit this tool, please enter exit" -ForegroundColor Yellow

                $userAndGroup = Read-Host "`r`nEnter users and groups"
                if ($userAndGroup -eq "exit") {
                    WriteLogInfo("User entered exit")
                    exit
                }
                $usersTobeProcessed = GetPendingMigrationUsers -userAndGroup $userAndGroup
            }

            DisplayFslogixStorePrompts
            $fslogixVhdStorePath = Read-Host "Enter path"
            while ((CheckFslogixStorePath -fslogixStorePath $fslogixVhdStorePath) -eq $false) {
                Write-Host "`r`nThe FSLogix VHD location you enter is invalid, please enter it again!" -ForegroundColor Red
                Write-Host "If you want to exit this tool, please enter exit" -ForegroundColor Green

                $fslogixVhdStorePath = Read-Host "Enter path"
                if ($fslogixVhdStorePath -eq "exit") {
                    WriteLogInfo("User entered exit")
                    exit
                }
            }

            DisplayUPMStorePrompts
            $upmVhdxStorePath = Read-Host "Enter path"
            while ((CheckUpmStorePath -upmStorePath $upmVhdxStorePath) -eq $false) {
                Write-Host "`r`nThe Citrix VHDX store path you enter is invalid, please enter it again!" -ForegroundColor Red
                Write-Host "If you want to exit this tool, please enter exit" -ForegroundColor Green

                $upmVhdxStorePath = Read-Host "Please enter the Citrix VHDX store path"
                if ($upmVhdxStorePath -eq "exit") {
                    WriteLogInfo("User entered exit")
                    exit
                }
            }

            DisplayOsnamePrompts
            $osShortName = Read-Host "Enter short OS name" 
            while ((CheckOsShortNameFormat -osShortName $osShortName) -eq $false) {
                Write-Host "`r`nThe vda machine's os short name you enter is invalid, please enter it again!" -ForegroundColor Red
                Write-Host "`r`nIf you want to exit this tool, please enter exit" -ForegroundColor Green

                $osShortName = Read-Host "Enter short OS name"
                if ($osShortName -eq "exit") {
                    WriteLogInfo("User entered exit")
                    exit
                }
            }

            
            MigrateFSLogixVHD -userMembers $usersTobeProcessed -fslogixVhdStorePath $fslogixVhdStorePath -upmVhdxStorePath $upmVhdxStorePath -osShortName $osShortName
            DisplayFutureWork
        }

        5 {
            DisplayUserAndGroupPrompts
            $userAndGroup = Read-Host "`r`nEnter users and groups"
            $usersTobeProcessed = GetPendingMigrationUsers -userAndGroup $userAndGroup
            while ($usersTobeProcessed -eq $false) {
                Write-Host "`r`nThe user or group format you enter is invalid, please enter it again!" -ForegroundColor Red
                Write-Host "`r`nIf you want to exit this tool, please enter exit" -ForegroundColor Yellow

                $userAndGroup = Read-Host "`r`nEnter users and groups"
                if ($userAndGroup -eq "exit") {
                    WriteLogInfo("User entered exit")
                    exit
                }
                $usersTobeProcessed = GetPendingMigrationUsers -userAndGroup $userAndGroup
            }

            DisplayWindowsRoamingStorePrompts
            $windowsRoamingStorePath = Read-Host "Enter path" 
            while ((CheckUpmStorePath -upmStorePath $windowsRoamingStorePath) -eq $false) {
                Write-Host "`r`nThe Windows Roaming Profile store path you enter is invalid, please enter it again!" -ForegroundColor Red
                Write-Host "If you want to exit this tool, please enter exit" -ForegroundColor Green

                $windowsRoamingStorePath = Read-Host "Enter path"
                if ($windowsRoamingStorePath -eq "exit") {
                    WriteLogInfo("User entered exit")
                    exit
                }
            }

            DisplayUPMStorePrompts
            $upmVhdxStorePath = Read-Host "Enter path"
            while ((CheckUpmStorePath -upmStorePath $upmVhdxStorePath) -eq $false) {
                Write-Host "`r`nThe Citrix VHDX store path you enter is invalid, please enter it again!" -ForegroundColor Red
                Write-Host "If you want to exit this tool, please enter exit" -ForegroundColor Green

                $upmVhdxStorePath = Read-Host "Please enter the Citrix VHDX store path"
                if ($upmVhdxStorePath -eq "exit") {
                    WriteLogInfo("User entered exit")
                    exit
                }
            }

            DisplayOsnamePrompts
            $osShortName = Read-Host "Enter short OS name" 
            while ((CheckOsShortNameFormat -osShortName $osShortName) -eq $false) {
                Write-Host "`r`nThe vda machine's os short name you enter is invalid, please enter it again!" -ForegroundColor Red
                Write-Host "`r`nIf you want to exit this tool, please enter exit" -ForegroundColor Green

                $osShortName = Read-Host "Enter short OS name"
                if ($osShortName -eq "exit") {
                    WriteLogInfo("User entered exit")
                    exit
                }
            }

            
            MigrateWindowsRoamingProfile -userMembers $usersTobeProcessed -windowsRoamingStorePath $windowsRoamingStorePath -upmVhdxStorePath $upmVhdxStorePath -osShortName $osShortName
            DisplayFutureWork
        }

        default {
            Write-Host "Please input 1-5" -ForegroundColor Yellow
        }
    }

    WriteLogInfo("Exiting the MigrationProfilesGuideMode")
}

function MigrationProfilesCommandMode {
    WriteLogInfo("Entering the MigrateProfilesCommandMode")

	if ($migrationType -eq 1) {
		WriteLogInfo("Enter the local profile migration command mode")
		
		if ($null -eq $usersAndGroups) {
			WriteLogError("Missing usersAndGroups parameter")
			return "Error: Missing usersAndGroups parameter"
		}
		
		if ($null -eq $osVersion) {
			WriteLogError("Missing osVersion parameter")
			return "Error: Missing osVersion parameter"
		}
		
		if ($null -eq $targetPath) {
			WriteLogError("Missing targetPath parameter")
			return "Error: Missing targetPath parameter"
		}
		
		$users = GetPendingMigrationUsers -userAndGroup $usersAndGroups
		if ($users -eq $false) {
			WriteLogError("The specified users are illegal")
            return "Error: The specified users are illegal"
        }
		
        $upmVhdxStorePath = $targetPath
        if ((CheckUpmStorePath -upmStorePath $upmVhdxStorePath) -eq $false) {
			WriteLogError("The upmStorePath is illegal!")
            return "Error: The upmStorePath is illegal!"
        }

        $osShortName = $osVersion
        if ((CheckOsShortNameFormat -osShortName $osShortName) -eq $false) {
			WriteLogError("The osshortname is invalid")
            return "Error: The osshortname is invalid"
        }

        MigrateLocalProfile -userMembers $users -upmVhdxStorePath $upmVhdxStorePath -osShortName $osShortName -remoteCredential $remoteCredential
		WriteLogInfo("Exiting the local profile migration command mode")
	}
	
	if ($migrationType -eq 2) {
		WriteLogInfo("Enter the UPM file-based profile migration command mode")
		
		if ($null -eq $usersAndGroups) {
			WriteLogError("Missing usersAndGroups parameter")
			return "Error: Missing usersAndGroups parameter"
		}
		
		if ($null -eq $osVersion) {
			WriteLogError("Missing osVersion parameter")
			return "Error: Missing osVersion parameter"
		}
		
		if ($null -eq $sourcePath) {
			WriteLogError("Missing sourcePath parameter")
			return "Error: Missing sourcePath parameter"
		}
		
		if ($null -eq $targetPath) {
			WriteLogError("Missing targetPath parameter")
			return "Error: Missing targetPath parameter"
		}

        $users = GetPendingMigrationUsers -userAndGroup $usersAndGroups
        if ($users -eq $false) {
			WriteLogError("The specified users are illegal")
            return "Error: The specified users are illegal"
        }

        $upmProfileStorePath = $sourcePath
        if ((CheckUpmStorePath -upmStorePath $upmProfileStorePath) -eq $false) {
			WriteLogError("The sourcePath is illegal")
            return "Error: The sourcePath is illegal!"
        }

        $upmVhdxStorePath = $targetPath
		if ((CheckUpmStorePath -upmStorePath $upmVhdxStorePath) -eq $false) {
			WriteLogError("The targetPath is illegal")
            return "Error: The targetPath is illegal!"
        }

        $osShortName = $osVersion
        if ((CheckOsShortNameFormat -osShortName $osShortName) -eq $false) {
			WriteLogError("The osshortname is invalid")
            return "Error: The osshortname is invalid"
        }
        
        MigrateUPMProfile -userMembers $users -upmProfileStorePath $upmProfileStorePath -upmVhdxStorePath $upmVhdxStorePath -osShortName $osShortName -remoteCredential $remoteCredential
	}
	
	if ($migrationType -eq 3) {
		WriteLogInfo("Enter the FSLogix profile migration command mode")
		if ($null -eq $usersAndGroups) {
			WriteLogError("Missing usersAndGroups parameter")
			return "Error: Missing usersAndGroups parameter"
		}
		
		if ($null -eq $osVersion) {
			WriteLogError("Missing osVersion parameter")
			return "Error: Missing osVersion parameter"
		}
		
		if ($null -eq $sourcePath) {
			WriteLogError("Missing sourcePath parameter")
			return "Error: Missing sourcePath parameter"
		}
		
		if ($null -eq $targetPath) {
			WriteLogError("Missing targetPath parameter")
			return "Error: Missing targetPath parameter"
		}

        $users = GetPendingMigrationUsers -userAndGroup $usersAndGroups
        if ($users -eq $false) {
			WriteLogError("The specified users are illegal")
            return "Error: The specified users are illegal"
        }

        $fslogixStorePath = $sourcePath
        if ((CheckFslogixStorePath -fslogixStorePath $fslogixStorePath) -eq $false) {
			WriteLogError("The fslogixStorePath is illegal")
            return "Error: The fslogixStorePath is illegal!"
        }

        $upmVhdxStorePath = $targetPath
		if ((CheckUpmStorePath -upmStorePath $upmVhdxStorePath) -eq $false) {
			WriteLogError("The upmProfileStorePath is illegal")
            return "Error: The upmProfileStorePath is illegal!"
        }

        $osShortName = $osVersion
        if ((CheckOsShortNameFormat -osShortName $osShortName) -eq $false) {
			WriteLogError("The osshortname is invalid")
            return "Error: The osshortname is invalid"
        }
        
        MigrateFSLogixProfile -userMembers $users -fslogixStorePath $fslogixStorePath -upmVhdxStorePath $upmVhdxStorePath -osShortName $osShortName -remoteCredential $remoteCredential
	}

    if ($migrationType -eq 4) {
        WriteLogInfo("Enter the windows roaming profile migration command mode")
        if ($null -eq $usersAndGroups) {
            WriteLogError("Missing usersAndGroups parameter")
            return "Error: Missing usersAndGroups parameter"
        }

        if ($null -eq $osVersion) {
            WriteLogError("Missing osVersion parameter")
            return "Error: Missing osVersion parameter"
        }

        if ($null -eq $sourcePath) {
            WriteLogError("Missing sourcePath parameter")
            return "Error: Missing sourcePath parameter"
        }

        if ($null -eq $targetPath) {
            WriteLogError("Missing targetPath parameter")
            return "Error: Missing targetPath parameter"
        }

        $users = GetPendingMigrationUsers -userAndGroup $usersAndGroups
        if ($users -eq $false) {
            WriteLogError("The specified users are illegal")
            return "Error: The specified users are illegal"
        }

        $windowsRoamingStorePath = $sourcePath
        if ((CheckUpmStorePath -upmStorePath $windowsRoamingStorePath) -eq $false) {
            WriteLogError("The windowsRoamingStorePath is illegal")
            return "Error: The windowsRoamingStorePath is illegal!"
        }

        $upmVhdxStorePath = $targetPath
        if ((CheckUpmStorePath -upmStorePath $upmVhdxStorePath) -eq $false) {
            WriteLogError("The upmProfileStorePath is illegal")
            return "Error: The upmProfileStorePath is illegal!"
        }

        $osShortName = $osVersion
        if ((CheckOsShortNameFormat -osShortName $osShortName) -eq $false) {
            WriteLogError("The osshortname is invalid")
            return "Error: The osshortname is invalid"
        }

        MigrateWindowsRoamingProfile -userMembers $users -windowsRoamingStorePath $windowsRoamingStorePath -upmVhdxStorePath $upmVhdxStorePath -osShortName $osShortName -remoteCredential $remoteCredential
    }
}

if (-not ([Security.Principal.WindowsPrincipal][Security.Principal.WindowsIdentity]::GetCurrent()).IsInRole([Security.Principal.WindowsBuiltInRole]::Administrator)) {
    write-host ''Run as administrator' is required for this tool. Exiting ...' -ForegroundColor Yellow
    Start-Sleep -Seconds 5
    Exit    
}

WriteJsonFile -UserName "N/A" -Status "N/A" -ErrorMessage "N/A" -isClear $true
WriteLogInfo("--------------------Migration task begin--------------------")

ImportModules

if (-not $migrationType -and -not $usersAndGroups -and -not $sourcePath -and -not $targetPath -and -not $osVersion -and -not $remoteCredential) {
    Write-Host "The migration tool is running in the guide mode."
	WriteLogInfo("The migration tool is running in the guide mode.")
    
    DisplayWelcomeInfo
	MigrationProfilesGuideMode
} else {
    WriteLogInfo("The migration type is $migrationType, the usersAndGroups is $usersAndGroups, the sourcePath is $sourcePath, the targetPath is $targetPath, the osVersion is $osVersion")

    MigrationProfilesCommandMode
}

WriteLogInfo("--------------------Migration task end----------------------")
# SIG # Begin signature block
# MIIo8AYJKoZIhvcNAQcCoIIo4TCCKN0CAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCCn1QpjwZqgejLz
# NSvXr/YYVKkUSsNo3f44qZzf5A4MFKCCDcAwggawMIIEmKADAgECAhAIrUCyYNKc
# TJ9ezam9k67ZMA0GCSqGSIb3DQEBDAUAMGIxCzAJBgNVBAYTAlVTMRUwEwYDVQQK
# EwxEaWdpQ2VydCBJbmMxGTAXBgNVBAsTEHd3dy5kaWdpY2VydC5jb20xITAfBgNV
# BAMTGERpZ2lDZXJ0IFRydXN0ZWQgUm9vdCBHNDAeFw0yMTA0MjkwMDAwMDBaFw0z
# NjA0MjgyMzU5NTlaMGkxCzAJBgNVBAYTAlVTMRcwFQYDVQQKEw5EaWdpQ2VydCwg
# SW5jLjFBMD8GA1UEAxM4RGlnaUNlcnQgVHJ1c3RlZCBHNCBDb2RlIFNpZ25pbmcg
# UlNBNDA5NiBTSEEzODQgMjAyMSBDQTEwggIiMA0GCSqGSIb3DQEBAQUAA4ICDwAw
# ggIKAoICAQDVtC9C0CiteLdd1TlZG7GIQvUzjOs9gZdwxbvEhSYwn6SOaNhc9es0
# JAfhS0/TeEP0F9ce2vnS1WcaUk8OoVf8iJnBkcyBAz5NcCRks43iCH00fUyAVxJr
# Q5qZ8sU7H/Lvy0daE6ZMswEgJfMQ04uy+wjwiuCdCcBlp/qYgEk1hz1RGeiQIXhF
# LqGfLOEYwhrMxe6TSXBCMo/7xuoc82VokaJNTIIRSFJo3hC9FFdd6BgTZcV/sk+F
# LEikVoQ11vkunKoAFdE3/hoGlMJ8yOobMubKwvSnowMOdKWvObarYBLj6Na59zHh
# 3K3kGKDYwSNHR7OhD26jq22YBoMbt2pnLdK9RBqSEIGPsDsJ18ebMlrC/2pgVItJ
# wZPt4bRc4G/rJvmM1bL5OBDm6s6R9b7T+2+TYTRcvJNFKIM2KmYoX7BzzosmJQay
# g9Rc9hUZTO1i4F4z8ujo7AqnsAMrkbI2eb73rQgedaZlzLvjSFDzd5Ea/ttQokbI
# YViY9XwCFjyDKK05huzUtw1T0PhH5nUwjewwk3YUpltLXXRhTT8SkXbev1jLchAp
# QfDVxW0mdmgRQRNYmtwmKwH0iU1Z23jPgUo+QEdfyYFQc4UQIyFZYIpkVMHMIRro
# OBl8ZhzNeDhFMJlP/2NPTLuqDQhTQXxYPUez+rbsjDIJAsxsPAxWEQIDAQABo4IB
# WTCCAVUwEgYDVR0TAQH/BAgwBgEB/wIBADAdBgNVHQ4EFgQUaDfg67Y7+F8Rhvv+
# YXsIiGX0TkIwHwYDVR0jBBgwFoAU7NfjgtJxXWRM3y5nP+e6mK4cD08wDgYDVR0P
# AQH/BAQDAgGGMBMGA1UdJQQMMAoGCCsGAQUFBwMDMHcGCCsGAQUFBwEBBGswaTAk
# BggrBgEFBQcwAYYYaHR0cDovL29jc3AuZGlnaWNlcnQuY29tMEEGCCsGAQUFBzAC
# hjVodHRwOi8vY2FjZXJ0cy5kaWdpY2VydC5jb20vRGlnaUNlcnRUcnVzdGVkUm9v
# dEc0LmNydDBDBgNVHR8EPDA6MDigNqA0hjJodHRwOi8vY3JsMy5kaWdpY2VydC5j
# b20vRGlnaUNlcnRUcnVzdGVkUm9vdEc0LmNybDAcBgNVHSAEFTATMAcGBWeBDAED
# MAgGBmeBDAEEATANBgkqhkiG9w0BAQwFAAOCAgEAOiNEPY0Idu6PvDqZ01bgAhql
# +Eg08yy25nRm95RysQDKr2wwJxMSnpBEn0v9nqN8JtU3vDpdSG2V1T9J9Ce7FoFF
# UP2cvbaF4HZ+N3HLIvdaqpDP9ZNq4+sg0dVQeYiaiorBtr2hSBh+3NiAGhEZGM1h
# mYFW9snjdufE5BtfQ/g+lP92OT2e1JnPSt0o618moZVYSNUa/tcnP/2Q0XaG3Ryw
# YFzzDaju4ImhvTnhOE7abrs2nfvlIVNaw8rpavGiPttDuDPITzgUkpn13c5Ubdld
# AhQfQDN8A+KVssIhdXNSy0bYxDQcoqVLjc1vdjcshT8azibpGL6QB7BDf5WIIIJw
# 8MzK7/0pNVwfiThV9zeKiwmhywvpMRr/LhlcOXHhvpynCgbWJme3kuZOX956rEnP
# LqR0kq3bPKSchh/jwVYbKyP/j7XqiHtwa+aguv06P0WmxOgWkVKLQcBIhEuWTatE
# QOON8BUozu3xGFYHKi8QxAwIZDwzj64ojDzLj4gLDb879M4ee47vtevLt/B3E+bn
# KD+sEq6lLyJsQfmCXBVmzGwOysWGw/YmMwwHS6DTBwJqakAwSEs0qFEgu60bhQji
# WQ1tygVQK+pKHJ6l/aCnHwZ05/LWUpD9r4VIIflXO7ScA+2GRfS0YW6/aOImYIbq
# yK+p/pQd52MbOoZWeE4wggcIMIIE8KADAgECAhADWiAGLDpbT4qELERZjaX6MA0G
# CSqGSIb3DQEBCwUAMGkxCzAJBgNVBAYTAlVTMRcwFQYDVQQKEw5EaWdpQ2VydCwg
# SW5jLjFBMD8GA1UEAxM4RGlnaUNlcnQgVHJ1c3RlZCBHNCBDb2RlIFNpZ25pbmcg
# UlNBNDA5NiBTSEEzODQgMjAyMSBDQTEwHhcNMjUwMjA3MDAwMDAwWhcNMjYwMjA2
# MjM1OTU5WjCBjzELMAkGA1UEBhMCVVMxEDAOBgNVBAgTB0Zsb3JpZGExGDAWBgNV
# BAcTD0ZvcnQgTGF1ZGVyZGFsZTEdMBsGA1UEChMUQ2l0cml4IFN5c3RlbXMsIElu
# Yy4xFjAUBgNVBAsTDUNpdHJpeCBYZW5BcHAxHTAbBgNVBAMTFENpdHJpeCBTeXN0
# ZW1zLCBJbmMuMIIBojANBgkqhkiG9w0BAQEFAAOCAY8AMIIBigKCAYEAnBm4JMW+
# W2tRIJltyaJM/q5G6WRAN293DlUc71U/dg6HzXRtahZP2kvV6ii32orbgdNkCwig
# +bo0W7AKjBBHKJqP96nuTNTsHoz+aeEE6PfgaFzGTRq77+dGDCk5DuPbFaXJCn9s
# UjVdbJTG1YXAcll8ZU6Iarg3eOokp2CcejMcjeD2dO8y89o6y2W5sWj2oIA+QRE9
# iU5qNfDLUtXo5i017JTN+qs9RtuwD77svXoV29wmkPmGuUq625ZZYFtRa0/t/C7w
# /k00UOFjykbvNBPj/cT67i/J/Um8tOanuC3cYU18VToMsfpS2t4irTdtrBzHupr/
# MB2DzZTwr5x2m3UzgEsnrr9bYnCIpOuW+K/oExuTpHtZsk6fnpoteOfyP059dNMg
# i1Gj074k6JfaJG+6fwKW0i2Unf7NDBArHdoHA6eIYB/OivPt4cmusgzRr2PziAl4
# LpA/9VRcnR68CWjnoTTr7qhdDeGqMgk24cNtmg+6BDt65GDVDX2ycthpAgMBAAGj
# ggIDMIIB/zAfBgNVHSMEGDAWgBRoN+Drtjv4XxGG+/5hewiIZfROQjAdBgNVHQ4E
# FgQUEpWeJpU+v9Nt97sQvlQgm4DN5wMwPgYDVR0gBDcwNTAzBgZngQwBBAEwKTAn
# BggrBgEFBQcCARYbaHR0cDovL3d3dy5kaWdpY2VydC5jb20vQ1BTMA4GA1UdDwEB
# /wQEAwIHgDATBgNVHSUEDDAKBggrBgEFBQcDAzCBtQYDVR0fBIGtMIGqMFOgUaBP
# hk1odHRwOi8vY3JsMy5kaWdpY2VydC5jb20vRGlnaUNlcnRUcnVzdGVkRzRDb2Rl
# U2lnbmluZ1JTQTQwOTZTSEEzODQyMDIxQ0ExLmNybDBToFGgT4ZNaHR0cDovL2Ny
# bDQuZGlnaWNlcnQuY29tL0RpZ2lDZXJ0VHJ1c3RlZEc0Q29kZVNpZ25pbmdSU0E0
# MDk2U0hBMzg0MjAyMUNBMS5jcmwwgZQGCCsGAQUFBwEBBIGHMIGEMCQGCCsGAQUF
# BzABhhhodHRwOi8vb2NzcC5kaWdpY2VydC5jb20wXAYIKwYBBQUHMAKGUGh0dHA6
# Ly9jYWNlcnRzLmRpZ2ljZXJ0LmNvbS9EaWdpQ2VydFRydXN0ZWRHNENvZGVTaWdu
# aW5nUlNBNDA5NlNIQTM4NDIwMjFDQTEuY3J0MAkGA1UdEwQCMAAwDQYJKoZIhvcN
# AQELBQADggIBAMGu7yYWlIS96uBwpNvMWgRgVFRwDx0HBdxGRi2+XvrtKkooOglx
# eUl/IYzZb8WhEBs/vknAmZkxBnkbbdgKKqxDeIkT2JX1sNGLKIzBBLPPjGMd2wxH
# CAVssWrA9xMESJ5Zx+5jHyoeVPt4XdPAXNB6Ca1JCa68Ip8Bn+atqop5wOmhXK85
# 2V2PdCWXjQr3Yl0k0PY/OtDDxnQc0dClj9TYlfGsOwyVNNQh/eExH2wkVlYYxgcj
# Zn89SmO4L+9AHkfpj/wd+/pJ+BYSflAotC6d9l4dft6f6PZ6KkbAz56dVkBX20xz
# YKKIK9tCf5sa1wnjkBEm2yn1IDn6K1cwUGO0whhtzGgcv4vPfv5bkumCSD7AquQv
# 37LP/gOZT9v3CiZAlTbZAoOK105qFmawt7AKPFzNFq9EjbUHYVBeYFnETDoa9zLz
# KCCcF/xJsjfn7YqFM+b1zq3C1YJK1B6+f3b5acpXNHbyMxPSOeNDZ13wMAYKah17
# D/4GZe74bsVqFo0oHlB82Bocr0AnnrLnLvfEirc0KIon5CQ5cci3LolBE0aNhD+n
# UtInPAJN1uRFpvhSK1ucVO/afmTQVtUgp5a8a8Dthc0bqh2ere4hoY+9U08Bda5g
# mFOO9YU8xEU3Do0AbuJPUVVALAAs0I+38RjYtIfPPt/T1zrMFHv+CMsKMYIahjCC
# GoICAQEwfTBpMQswCQYDVQQGEwJVUzEXMBUGA1UEChMORGlnaUNlcnQsIEluYy4x
# QTA/BgNVBAMTOERpZ2lDZXJ0IFRydXN0ZWQgRzQgQ29kZSBTaWduaW5nIFJTQTQw
# OTYgU0hBMzg0IDIwMjEgQ0ExAhADWiAGLDpbT4qELERZjaX6MA0GCWCGSAFlAwQC
# AQUAoIHgMBkGCSqGSIb3DQEJAzEMBgorBgEEAYI3AgEEMBwGCisGAQQBgjcCAQsx
# DjAMBgorBgEEAYI3AgEVMC8GCSqGSIb3DQEJBDEiBCBA6hkq/HonBwdUxV9JOUS+
# 6jW+VApPXlwecTQcN+sTdzB0BgorBgEEAYI3AgEMMWYwZKBIgEYAQwBQAE0AXwBQ
# AHIAbwBmAGkAbABlAEMAbwBuAHQAYQBpAG4AZQByAF8ATQBpAGcAcgBhAHQAaQBv
# AG4AXwBUAG8AbwBsoRiAFmh0dHA6Ly93d3cuY2l0cml4LmNvbSAwDQYJKoZIhvcN
# AQEBBQAEggGAXH5MgbQahGcwJ/wTyZprkmRHNu6axArdIq9mZ1ozNg1ciGEBy55+
# 3eJ3IPxoN4e6cezgGuAcfHEn1SCSi955wV2yJV4l0y+C0Wv4+amyziZHU0xp2JMQ
# gOsLxyBDvBzWBn3aESkAxRgYHyJq9Du+O1KhQFw1S1JbiGshrD7PhlXKVf/HLUK1
# vcC4O+QO9Se5JteTOlA7XiptF3bg0NsnBHMsG2d8rnhUMCMSNvzAM87IJm4xzF7J
# 4VboMWm8JTAK0KR1g27FuLB6piH3Zff/OmQS6M7XEjgq2Qh6AvMgSJpeMaFUqotZ
# ApXctao3GF5KW0+zcyXUiPQ8Emg/wX/CvhQWJdwUPEzlEbtiVkzB/l4El7prBlIs
# Yns/W4g14GGftlmC6S10cqK6ya3VVxd7i9kwE8AzPGkNn98C0LA3nU3DGR/xUGkl
# OBSOM7dR1GmmgQW+Ujadt0tafyGXLKDDzBzmH8WVdGRgODl/hg9fpGWgNbvc9r9t
# 6wjGWs4lr25JoYIXdzCCF3MGCisGAQQBgjcDAwExghdjMIIXXwYJKoZIhvcNAQcC
# oIIXUDCCF0wCAQMxDzANBglghkgBZQMEAgEFADB4BgsqhkiG9w0BCRABBKBpBGcw
# ZQIBAQYJYIZIAYb9bAcBMDEwDQYJYIZIAWUDBAIBBQAEINRCtCL8A95vOn76lkhC
# TYL09+LgsUi4YIw/+6eD6zPgAhEArnGWEUX5/hLU9o9v6kf0eRgPMjAyNTA2MTcx
# MDA3NTdaoIITOjCCBu0wggTVoAMCAQICEAqA7xhLjfEFgtHEdqeVdGgwDQYJKoZI
# hvcNAQELBQAwaTELMAkGA1UEBhMCVVMxFzAVBgNVBAoTDkRpZ2lDZXJ0LCBJbmMu
# MUEwPwYDVQQDEzhEaWdpQ2VydCBUcnVzdGVkIEc0IFRpbWVTdGFtcGluZyBSU0E0
# MDk2IFNIQTI1NiAyMDI1IENBMTAeFw0yNTA2MDQwMDAwMDBaFw0zNjA5MDMyMzU5
# NTlaMGMxCzAJBgNVBAYTAlVTMRcwFQYDVQQKEw5EaWdpQ2VydCwgSW5jLjE7MDkG
# A1UEAxMyRGlnaUNlcnQgU0hBMjU2IFJTQTQwOTYgVGltZXN0YW1wIFJlc3BvbmRl
# ciAyMDI1IDEwggIiMA0GCSqGSIb3DQEBAQUAA4ICDwAwggIKAoICAQDQRqwtEsae
# 0OquYFazK1e6b1H/hnAKAd/KN8wZQjBjMqiZ3xTWcfsLwOvRxUwXcGx8AUjni6bz
# 52fGTfr6PHRNv6T7zsf1Y/E3IU8kgNkeECqVQ+3bzWYesFtkepErvUSbf+EIYLkr
# LKd6qJnuzK8Vcn0DvbDMemQFoxQ2Dsw4vEjoT1FpS54dNApZfKY61HAldytxNM89
# PZXUP/5wWWURK+IfxiOg8W9lKMqzdIo7VA1R0V3Zp3DjjANwqAf4lEkTlCDQ0/fK
# JLKLkzGBTpx6EYevvOi7XOc4zyh1uSqgr6UnbksIcFJqLbkIXIPbcNmA98Oskkkr
# vt6lPAw/p4oDSRZreiwB7x9ykrjS6GS3NR39iTTFS+ENTqW8m6THuOmHHjQNC3zb
# J6nJ6SXiLSvw4Smz8U07hqF+8CTXaETkVWz0dVVZw7knh1WZXOLHgDvundrAtuvz
# 0D3T+dYaNcwafsVCGZKUhQPL1naFKBy1p6llN3QgshRta6Eq4B40h5avMcpi54wm
# 0i2ePZD5pPIssoszQyF4//3DoK2O65Uck5Wggn8O2klETsJ7u8xEehGifgJYi+6I
# 03UuT1j7FnrqVrOzaQoVJOeeStPeldYRNMmSF3voIgMFtNGh86w3ISHNm0IaadCK
# CkUe2LnwJKa8TIlwCUNVwppwn4D3/Pt5pwIDAQABo4IBlTCCAZEwDAYDVR0TAQH/
# BAIwADAdBgNVHQ4EFgQU5Dv88jHt/f3X85FxYxlQQ89hjOgwHwYDVR0jBBgwFoAU
# 729TSunkBnx6yuKQVvYv1Ensy04wDgYDVR0PAQH/BAQDAgeAMBYGA1UdJQEB/wQM
# MAoGCCsGAQUFBwMIMIGVBggrBgEFBQcBAQSBiDCBhTAkBggrBgEFBQcwAYYYaHR0
# cDovL29jc3AuZGlnaWNlcnQuY29tMF0GCCsGAQUFBzAChlFodHRwOi8vY2FjZXJ0
# cy5kaWdpY2VydC5jb20vRGlnaUNlcnRUcnVzdGVkRzRUaW1lU3RhbXBpbmdSU0E0
# MDk2U0hBMjU2MjAyNUNBMS5jcnQwXwYDVR0fBFgwVjBUoFKgUIZOaHR0cDovL2Ny
# bDMuZGlnaWNlcnQuY29tL0RpZ2lDZXJ0VHJ1c3RlZEc0VGltZVN0YW1waW5nUlNB
# NDA5NlNIQTI1NjIwMjVDQTEuY3JsMCAGA1UdIAQZMBcwCAYGZ4EMAQQCMAsGCWCG
# SAGG/WwHATANBgkqhkiG9w0BAQsFAAOCAgEAZSqt8RwnBLmuYEHs0QhEnmNAciH4
# 5PYiT9s1i6UKtW+FERp8FgXRGQ/YAavXzWjZhY+hIfP2JkQ38U+wtJPBVBajYfrb
# IYG+Dui4I4PCvHpQuPqFgqp1PzC/ZRX4pvP/ciZmUnthfAEP1HShTrY+2DE5qjzv
# Zs7JIIgt0GCFD9ktx0LxxtRQ7vllKluHWiKk6FxRPyUPxAAYH2Vy1lNM4kzekd8o
# EARzFAWgeW3az2xejEWLNN4eKGxDJ8WDl/FQUSntbjZ80FU3i54tpx5F/0Kr15zW
# /mJAxZMVBrTE2oi0fcI8VMbtoRAmaaslNXdCG1+lqvP4FbrQ6IwSBXkZagHLhFU9
# HCrG/syTRLLhAezu/3Lr00GrJzPQFnCEH1Y58678IgmfORBPC1JKkYaEt2OdDh4G
# mO0/5cHelAK2/gTlQJINqDr6JfwyYHXSd+V08X1JUPvB4ILfJdmL+66Gp3CSBXG6
# IwXMZUXBhtCyIaehr0XkBoDIGMUG1dUtwq1qmcwbdUfcSYCn+OwncVUXf53VJUNO
# aMWMts0VlRYxe5nK+At+DI96HAlXHAL5SlfYxJ7La54i71McVWRP66bW+yERNpbJ
# CjyCYG2j+bdpxo/1Cy4uPcU3AWVPGrbn5PhDBf3Froguzzhk++ami+r3Qrx5bIbY
# 3TVzgiFI7Gq3zWcwgga0MIIEnKADAgECAhANx6xXBf8hmS5AQyIMOkmGMA0GCSqG
# SIb3DQEBCwUAMGIxCzAJBgNVBAYTAlVTMRUwEwYDVQQKEwxEaWdpQ2VydCBJbmMx
# GTAXBgNVBAsTEHd3dy5kaWdpY2VydC5jb20xITAfBgNVBAMTGERpZ2lDZXJ0IFRy
# dXN0ZWQgUm9vdCBHNDAeFw0yNTA1MDcwMDAwMDBaFw0zODAxMTQyMzU5NTlaMGkx
# CzAJBgNVBAYTAlVTMRcwFQYDVQQKEw5EaWdpQ2VydCwgSW5jLjFBMD8GA1UEAxM4
# RGlnaUNlcnQgVHJ1c3RlZCBHNCBUaW1lU3RhbXBpbmcgUlNBNDA5NiBTSEEyNTYg
# MjAyNSBDQTEwggIiMA0GCSqGSIb3DQEBAQUAA4ICDwAwggIKAoICAQC0eDHTCphB
# cr48RsAcrHXbo0ZodLRRF51NrY0NlLWZloMsVO1DahGPNRcybEKq+RuwOnPhof6p
# vF4uGjwjqNjfEvUi6wuim5bap+0lgloM2zX4kftn5B1IpYzTqpyFQ/4Bt0mAxAHe
# HYNnQxqXmRinvuNgxVBdJkf77S2uPoCj7GH8BLuxBG5AvftBdsOECS1UkxBvMgEd
# gkFiDNYiOTx4OtiFcMSkqTtF2hfQz3zQSku2Ws3IfDReb6e3mmdglTcaarps0wjU
# jsZvkgFkriK9tUKJm/s80FiocSk1VYLZlDwFt+cVFBURJg6zMUjZa/zbCclF83bR
# VFLeGkuAhHiGPMvSGmhgaTzVyhYn4p0+8y9oHRaQT/aofEnS5xLrfxnGpTXiUOeS
# LsJygoLPp66bkDX1ZlAeSpQl92QOMeRxykvq6gbylsXQskBBBnGy3tW/AMOMCZIV
# NSaz7BX8VtYGqLt9MmeOreGPRdtBx3yGOP+rx3rKWDEJlIqLXvJWnY0v5ydPpOjL
# 6s36czwzsucuoKs7Yk/ehb//Wx+5kMqIMRvUBDx6z1ev+7psNOdgJMoiwOrUG2Zd
# SoQbU2rMkpLiQ6bGRinZbI4OLu9BMIFm1UUl9VnePs6BaaeEWvjJSjNm2qA+sdFU
# eEY0qVjPKOWug/G6X5uAiynM7Bu2ayBjUwIDAQABo4IBXTCCAVkwEgYDVR0TAQH/
# BAgwBgEB/wIBADAdBgNVHQ4EFgQU729TSunkBnx6yuKQVvYv1Ensy04wHwYDVR0j
# BBgwFoAU7NfjgtJxXWRM3y5nP+e6mK4cD08wDgYDVR0PAQH/BAQDAgGGMBMGA1Ud
# JQQMMAoGCCsGAQUFBwMIMHcGCCsGAQUFBwEBBGswaTAkBggrBgEFBQcwAYYYaHR0
# cDovL29jc3AuZGlnaWNlcnQuY29tMEEGCCsGAQUFBzAChjVodHRwOi8vY2FjZXJ0
# cy5kaWdpY2VydC5jb20vRGlnaUNlcnRUcnVzdGVkUm9vdEc0LmNydDBDBgNVHR8E
# PDA6MDigNqA0hjJodHRwOi8vY3JsMy5kaWdpY2VydC5jb20vRGlnaUNlcnRUcnVz
# dGVkUm9vdEc0LmNybDAgBgNVHSAEGTAXMAgGBmeBDAEEAjALBglghkgBhv1sBwEw
# DQYJKoZIhvcNAQELBQADggIBABfO+xaAHP4HPRF2cTC9vgvItTSmf83Qh8WIGjB/
# T8ObXAZz8OjuhUxjaaFdleMM0lBryPTQM2qEJPe36zwbSI/mS83afsl3YTj+IQhQ
# E7jU/kXjjytJgnn0hvrV6hqWGd3rLAUt6vJy9lMDPjTLxLgXf9r5nWMQwr8Myb9r
# EVKChHyfpzee5kH0F8HABBgr0UdqirZ7bowe9Vj2AIMD8liyrukZ2iA/wdG2th9y
# 1IsA0QF8dTXqvcnTmpfeQh35k5zOCPmSNq1UH410ANVko43+Cdmu4y81hjajV/gx
# dEkMx1NKU4uHQcKfZxAvBAKqMVuqte69M9J6A47OvgRaPs+2ykgcGV00TYr2Lr3t
# y9qIijanrUR3anzEwlvzZiiyfTPjLbnFRsjsYg39OlV8cipDoq7+qNNjqFzeGxcy
# tL5TTLL4ZaoBdqbhOhZ3ZRDUphPvSRmMThi0vw9vODRzW6AxnJll38F0cuJG7uEB
# YTptMSbhdhGQDpOXgpIUsWTjd6xpR6oaQf/DJbg3s6KCLPAlZ66RzIg9sC+NJpud
# /v4+7RWsWCiKi9EOLLHfMR2ZyJ/+xhCx9yHbxtl5TPau1j/1MIDpMPx0LckTetiS
# uEtQvLsNz3Qbp7wGWqbIiOWCnb5WqxL3/BAPvIXKUjPSxyZsq8WhbaM2tszWkPZP
# ubdcMIIFjTCCBHWgAwIBAgIQDpsYjvnQLefv21DiCEAYWjANBgkqhkiG9w0BAQwF
# ADBlMQswCQYDVQQGEwJVUzEVMBMGA1UEChMMRGlnaUNlcnQgSW5jMRkwFwYDVQQL
# ExB3d3cuZGlnaWNlcnQuY29tMSQwIgYDVQQDExtEaWdpQ2VydCBBc3N1cmVkIElE
# IFJvb3QgQ0EwHhcNMjIwODAxMDAwMDAwWhcNMzExMTA5MjM1OTU5WjBiMQswCQYD
# VQQGEwJVUzEVMBMGA1UEChMMRGlnaUNlcnQgSW5jMRkwFwYDVQQLExB3d3cuZGln
# aWNlcnQuY29tMSEwHwYDVQQDExhEaWdpQ2VydCBUcnVzdGVkIFJvb3QgRzQwggIi
# MA0GCSqGSIb3DQEBAQUAA4ICDwAwggIKAoICAQC/5pBzaN675F1KPDAiMGkz7MKn
# JS7JIT3yithZwuEppz1Yq3aaza57G4QNxDAf8xukOBbrVsaXbR2rsnnyyhHS5F/W
# BTxSD1Ifxp4VpX6+n6lXFllVcq9ok3DCsrp1mWpzMpTREEQQLt+C8weE5nQ7bXHi
# LQwb7iDVySAdYyktzuxeTsiT+CFhmzTrBcZe7FsavOvJz82sNEBfsXpm7nfISKhm
# V1efVFiODCu3T6cw2Vbuyntd463JT17lNecxy9qTXtyOj4DatpGYQJB5w3jHtrHE
# tWoYOAMQjdjUN6QuBX2I9YI+EJFwq1WCQTLX2wRzKm6RAXwhTNS8rhsDdV14Ztk6
# MUSaM0C/CNdaSaTC5qmgZ92kJ7yhTzm1EVgX9yRcRo9k98FpiHaYdj1ZXUJ2h4mX
# aXpI8OCiEhtmmnTK3kse5w5jrubU75KSOp493ADkRSWJtppEGSt+wJS00mFt6zPZ
# xd9LBADMfRyVw4/3IbKyEbe7f/LVjHAsQWCqsWMYRJUadmJ+9oCw++hkpjPRiQfh
# vbfmQ6QYuKZ3AeEPlAwhHbJUKSWJbOUOUlFHdL4mrLZBdd56rF+NP8m800ERElvl
# EFDrMcXKchYiCd98THU/Y+whX8QgUWtvsauGi0/C1kVfnSD8oR7FwI+isX4KJpn1
# 5GkvmB0t9dmpsh3lGwIDAQABo4IBOjCCATYwDwYDVR0TAQH/BAUwAwEB/zAdBgNV
# HQ4EFgQU7NfjgtJxXWRM3y5nP+e6mK4cD08wHwYDVR0jBBgwFoAUReuir/SSy4Ix
# LVGLp6chnfNtyA8wDgYDVR0PAQH/BAQDAgGGMHkGCCsGAQUFBwEBBG0wazAkBggr
# BgEFBQcwAYYYaHR0cDovL29jc3AuZGlnaWNlcnQuY29tMEMGCCsGAQUFBzAChjdo
# dHRwOi8vY2FjZXJ0cy5kaWdpY2VydC5jb20vRGlnaUNlcnRBc3N1cmVkSURSb290
# Q0EuY3J0MEUGA1UdHwQ+MDwwOqA4oDaGNGh0dHA6Ly9jcmwzLmRpZ2ljZXJ0LmNv
# bS9EaWdpQ2VydEFzc3VyZWRJRFJvb3RDQS5jcmwwEQYDVR0gBAowCDAGBgRVHSAA
# MA0GCSqGSIb3DQEBDAUAA4IBAQBwoL9DXFXnOF+go3QbPbYW1/e/Vwe9mqyhhyzs
# hV6pGrsi+IcaaVQi7aSId229GhT0E0p6Ly23OO/0/4C5+KH38nLeJLxSA8hO0Cre
# +i1Wz/n096wwepqLsl7Uz9FDRJtDIeuWcqFItJnLnU+nBgMTdydE1Od/6Fmo8L8v
# C6bp8jQ87PcDx4eo0kxAGTVGamlUsLihVo7spNU96LHc/RzY9HdaXFSMb++hUD38
# dglohJ9vytsgjTVgHAIDyyCwrFigDkBjxZgiwbJZ9VVrzyerbHbObyMt9H5xaiNr
# Iv8SuFQtJ37YOtnwtoeW/VvRXKwYw02fc7cBqZ9Xql4o4rmUMYIDfDCCA3gCAQEw
# fTBpMQswCQYDVQQGEwJVUzEXMBUGA1UEChMORGlnaUNlcnQsIEluYy4xQTA/BgNV
# BAMTOERpZ2lDZXJ0IFRydXN0ZWQgRzQgVGltZVN0YW1waW5nIFJTQTQwOTYgU0hB
# MjU2IDIwMjUgQ0ExAhAKgO8YS43xBYLRxHanlXRoMA0GCWCGSAFlAwQCAQUAoIHR
# MBoGCSqGSIb3DQEJAzENBgsqhkiG9w0BCRABBDAcBgkqhkiG9w0BCQUxDxcNMjUw
# NjE3MTAwNzU3WjArBgsqhkiG9w0BCRACDDEcMBowGDAWBBTdYjCshgotMGvaOLFo
# eVIwB/tBfjAvBgkqhkiG9w0BCQQxIgQg/QzP3Gb3LcMBG7pT0Lpz2HcSLCzL2eqp
# U22llaRJNy4wNwYLKoZIhvcNAQkQAi8xKDAmMCQwIgQgSqA/oizXXITFXJOPgo5n
# a5yuyrM/420mmqM08UYRCjMwDQYJKoZIhvcNAQEBBQAEggIALCkgJM+eDsmCmtf1
# 6MfeTDfar849NB8W8QczVpDqihIlKSccRsluz+kOOcmr1iKE3M4/siCpkggrPORH
# Qd/OpYwnMkdf+uMi7r7zs1FQ6NuzTVcJOh0/veytVJFisHo3ZRySyL37eISaJHwK
# vZGW8byX10c8vGw4Y0mmySFuMwylBZkqorsx+M1yHqRIwGDgUy7rEdMIh3Flai5s
# HR3FJVWFw3t2vS+74ym0P6ZApB3ZEU5zgJ6l0M3VQDdEXKQNmiCmVigfVoIDngMP
# xchMPR+1JeM2q4ZD/+uaRMHx21Zoe9kD0voIQuKDrnaGEhSum1qoigWNkQqABp3B
# JnU3mG4PhPBOyw88YE6UCEHsDay2vSh66AqR2xqgywJEUwnCha3kWKcEAGPuA0P2
# Pw00iw5i1TqPkltMg3nm0xnSTQRsFUXksQglSaIzqGDg0tbVg86egjs2EFFZAmSL
# o2gXEV/htFBWGUcNZMaq1M+yofg7IjFxUGltlUT7kPstwuM+pXl+HLcoi44kE4cT
# P3Nd4agUtZHfD3Fl+dDrICukNoAprUfOYpnNVkMCekAYZs8Qp67XFP8SvWi/svBC
# PpG4lAe+q+JZyQypqtImVsKOr1lx4pAgxi60NPSOZDJqqO7G1f9qfRU0fcWQHKbp
# KTNjE8mfKT75VeH2D7A2PAtimE4=
# SIG # End signature block
