<# 
.SYNOPSIS
    Citrix Profile Management User Store Tool

.DESCRIPTION
    This PowerShell script was developed to help admins creating and checking user store
 
.NOTES

.LINK
    It is released with UPM
 
.Parameter = 
    Run the script without any parameters and let the script guide you through setting up the user store.
    Or provide the following parameters:
    1.(Mandatory) [String] MachineName - The filer server machine hosting the user store.
    2.(Mandatory) [String] FolderPath - The full path of the user store on the machine. (The path must be fully qualified. Relative paths or paths that contain wildcard characters are not permitted.)
    3.(Mandatory) [String] SmbShareName - The SMB share name of the user store.
    4.(Mandatory) [String] FolderPermissions - The folder permission entries (the users\groups that using this user store), separated by pipe.
    5.(Optional)  [String] FolderAdministrators - The users\groups that will grant full control of the entire user store.
    6.(Optional)  [Bool]   ForceFolderPath - Ignore file existence checks, forcing use of FolderPath as the user store. (Warning: This will change the permission settings of the folder, subfolder and subfiles, and share folder to the network. But will not delete any files\folders under the path.)
    7.(Optional)  [Bool]   ForceSmbShareName - Ignore SMB existence checks, forcing use of SmbShareName as the user store SMB name. (Warning: This will stop the existing share and create a new share.)
    8.(Optional)  [PSCredential] RemoteCredential - The credential to connect to the file server for user store setting up. (Ignore this parameter to input in the pop up when the script is running)' 
#>

Param(
    [Parameter(Mandatory = $false)] [String] $MachineName = '',
    [Parameter(Mandatory = $false)] [String] $FolderPath = '',
    [Parameter(Mandatory = $false)] [Bool]   $ForceFolderPath = $false,
    [Parameter(Mandatory = $false)] [String] $SmbShareName = '',
    [Parameter(Mandatory = $false)] [Bool]   $ForceSmbShareName = $false,
    [Parameter(Mandatory = $false)] [String] $FolderPermissions = '',
    [Parameter(Mandatory = $false)] [String] $FolderAdministrators = '',
    [Parameter(Mandatory = $false)] [PSCredential] $RemoteCredential = $null
)

$CheckUserStorePath = {
    Param ($folderPath)

    $exitcode = 0
    if (Test-Path -Path "$folderPath\*") {
        $exitcode = 2
    }
    elseif (Test-Path -Path "$folderPath") {
        if (Test-Path -Path "$folderPath" -PathType 'Container') {
            $exitcode = 1
        }
        else {
            $exitcode = 3   # path is a file
        }
    }
    $exitcode
}

$CheckSmbShareName = {
    Param ($smbShareName)

    $smb_share_path = ''
    if (Get-SmbShare | Where-Object -Property Name -eq $smbShareName) {
        $smb_share_path = 'P' + (Get-SmbShare -Name $smbShareName).Path     # Append a P in case the path is empty
    }
    $smb_share_path
}

$CreateUserStoreProcess = {
    Param ($folderPath, $smbShareName, $sharePermissions, $folderPermissions, $folderAdministrators)

    $SmbShareEveryone = $false
    if (($sharePermissions.Count -eq 1) -and ($sharePermissions[0] -eq "Everyone")) {
        $SmbShareEveryone = $true
    }

    if (!(Test-Path -Path $folderPath)) {
        try {
            New-Item $folderPath -ItemType Directory -ErrorAction Stop
        }
        catch {
            return "Unable to create user store folder: $_"
        }
    }

    try {
        if (Get-SmbShare | Where-Object -Property Name -eq $smbShareName) {
            Remove-SmbShare -Name $smbShareName -Confirm:$false -ErrorAction Stop
        }
    }
    catch {
        return "Unable to check SMB file share: $_"
    }
    
    # Setup smb share
    if ($SmbShareEveryone -eq $true) {
        $SmbShareParams = @{
            Name       = "$smbShareName"
            Path       = "$folderPath"
            FullAccess = "Everyone"
        }
    }
    else {
        $SmbShareParams = @{
            Name         = "$smbShareName"
            Path         = "$folderPath"
            FullAccess   = "$DomainName\Domain Computers", "Administrators"
            ChangeAccess = $sharePermissions
        }
    }
    try {
        New-SmbShare @SmbShareParams -ErrorAction Stop | Out-Null
    }
    catch {
        return "Unable to create SMB file share: $_"
    }

    # Set ACLs
    try {
        $acl = Get-Acl $folderPath -ErrorAction Stop
        $acl.Access | % { $acl.RemoveAccessRule($_) | Out-Null }
        $acl.SetAccessRuleProtection($true, $false)

        $AccessRule = New-Object system.security.accesscontrol.filesystemaccessrule("System", "FullControl", "ContainerInherit, ObjectInherit", "None", "Allow")
        $acl.SetAccessRule($AccessRule)

        $AccessRule = New-Object system.security.accesscontrol.filesystemaccessrule("CREATOR OWNER", "FullControl", "ContainerInherit, ObjectInherit", "InheritOnly", "Allow")
        $acl.SetAccessRule($AccessRule)

        foreach ($FolderPermission in $folderPermissions) {
            $AccessRule = New-Object system.security.accesscontrol.filesystemaccessrule("$FolderPermission", "AppendData,ReadData", "None", "None", "Allow")
            $acl.SetAccessRule($AccessRule)
        }

        # Make sure this is after FolderPermission (if same user\group in both, the user\group will get FullControl)
        foreach ($folerAdmin in $folderAdministrators) {
            $AccessRule = New-Object system.security.accesscontrol.filesystemaccessrule("$folerAdmin", "FullControl", "ContainerInherit, ObjectInherit", "None", "Allow")
            $acl.SetAccessRule($AccessRule)
        }

        if ($SmbShareEveryone -eq $false) {
            $AccessRule = New-Object system.security.accesscontrol.filesystemaccessrule("$DomainName\Domain Computers", "FullControl", "ContainerInherit, ObjectInherit", "None", "Allow")
            $acl.SetAccessRule($AccessRule)
        }

        Set-Acl -Path $folderPath -AclObject $acl -ErrorAction Stop
    }
    catch {
        return "Unable to set folder permissions: $_"
    }

    return 'Success'
}

function Test-SmbSharePath {
    param(
        [String]$folderPath
    )

    if (($folderPath -eq $null) -or ($folderPath -eq '')) {
        return $false
    }

    $res = $true
    try {
        $res = [System.IO.Path]::IsPathRooted("$folderPath")
    }
    catch {
        $res = $false
    }
    if ($res -ne $true) {
        return $false
    }

    $charCount = ($folderPath.ToCharArray() | Where-Object { $_ -eq ':' } | Measure-Object).Count
    $invalidList = @('/', '"', '<', '>', '|', '*', '?')
    foreach ($invalidinput in $invalidList) {
        if ($folderPath.Contains($invalidinput) -or ($charCount -gt 1) -or ($folderPath.Length -lt 4)) {
            return $false
        }
        if (($folderPath.Contains('%') -eq $false) -and ($folderPath.Contains(':') -eq $false)) {
            return $false
        }
        if ( $folderPath.Contains('\\') -eq $true) {
            return $false
        }
        #path with drive
        if ($charCount -eq 1) {
            if (($folderPath[1] -eq ':') -and ((($folderPath[0] -ige 'a') -and ($folderPath[0] -le 'z')) -or (($folderPath[0] -ige 'A') -and ($folderPath[0] -le 'Z')))) {
            }
            else {
                return $false
            }
        }
    }

    return $true
}

function Convert-PSSessionError {
    param(
        [String]$errorMsg
    )

    if ($errorMsg.Contains('The user name or password is incorrect.')) {
        $Global:gLastErrorMsg = "{`"ErrorCode`": `"ERROR_CREDENTIAL_INVALID_PASSWORD`", `"Details`": `"$_`"}"
    }
    elseif ($errorMsg.Contains('Access is denied.')) {
        $Global:gLastErrorMsg = "{`"ErrorCode`": `"ERROR_CREDENTIAL_ACCESS_DENIED`", `"Details`": `"$_`"}"
    }
    else {
        $Global:gLastErrorMsg = "{`"ErrorCode`": `"ERROR_OTHERS`", `"Details`": `"$_`"}"
    }
}

function Confirm-MachineName {
    param(
        [String]$machineName
    )

    # Local
    if ($machineName -eq '' -or $machineName -eq $null) {
        $Global:gMachineName = $Global:gLocalMachineName
        return $true
    }

    # Local
    if ($machineName -eq $Global:gLocalMachineName) {
        $Global:gMachineName = $machineName
        return $true
    }

    if ($Global:gRemoteCredential -eq $null) {
        $Global:gLastErrorMsg = '{"ErrorCode": "ERROR_OTHERS", "Details": "Invalid script parameter: RemoteCredential."}'
        return $false
    }

    try {
        if ($Global:gRemoteSession -ne $null) {
            Remove-PSSession $Global:gRemoteSession
            $Global:gRemoteSession = $null
        }
        $Global:gRemoteSession = New-PSSession -ComputerName $machineName -Credential $Global:gRemoteCredential -ErrorAction Stop
    }
    catch {
        Convert-PSSessionError -errorMsg $_
        return $false
    }

    $Global:gMachineName = $machineName
    return $true
}

function Confirm-FolderPath {
    param(
        [String]$folderPath
    )

    if ($folderPath -eq '' -or $folderPath -eq $null) {
        $Global:gLastErrorMsg = '{"ErrorCode": "ERROR_OTHERS", "Details": "Invalid script parameter."}'
        return $false
    }

    if (Test-SmbSharePath -folderPath $folderPath -eq $true) {
        if ($Global:gMachineName -eq $Global:gLocalMachineName) {
            $remote_last_exit_value = Invoke-Command -ScriptBlock $CheckUserStorePath -ArgumentList $folderPath
        }
        else {
            $remote_last_exit_value = Invoke-Command -Session $Global:gRemoteSession -ScriptBlock $CheckUserStorePath -ArgumentList $folderPath
        }
    }
    else {
        $remote_last_exit_value = '4'
    }

    switch ($remote_last_exit_value) {
        '0' {
            $Global:gFolderPath = $folderPath
            return $true
        }
        '1' {
            $Global:gLastErrorMsg = '{"ErrorCode": "ERROR_FILE_EXISTS", "Details": "The folder already exists and empty."}'
            return $false
        }
        '2' {
            $Global:gLastErrorMsg = '{"ErrorCode": "ERROR_FILE_EXISTS", "Details": "The folder already exists and contains items."}'
            return $false
        }
        '3' {
            $Global:gLastErrorMsg = '{"ErrorCode": "ERROR_FILE_EXISTS", "Details": "The file already exists and not a dirtory."}'
            return $false
        }
        '4' {
            $Global:gLastErrorMsg = '{"ErrorCode": "ERROR_OTHERS", "Details": "Invalid folder path format."}'
            return $false
        }
    }
}

function Confirm-SmbShareName {
    param(
        [String]$smbShareName
    )

    if ($smbShareName -eq '' -or $smbShareName -eq $null) {
        $Global:gLastErrorMsg = '{"ErrorCode": "ERROR_OTHERS", "Details": "Invalid script parameter."}'
        return $false
    }

    # Return the path of the share if the share exists
    if ($Global:gMachineName -eq $Global:gLocalMachineName) {
        $smb_share_path = Invoke-Command -ScriptBlock $CheckSmbShareName -ArgumentList $smbShareName
    }
    else {
        $smb_share_path = Invoke-Command -Session $Global:gRemoteSession -ScriptBlock $CheckSmbShareName -ArgumentList $smbShareName
    }

    if ($smb_share_path -eq '') {
        $Global:gSmbShareName = $smbShareName
        return $true
    }
    else {
        $Global:gLastErrorMsg = '{"ErrorCode": "ERROR_SMB_NAME_EXISTS", "Details": "The SMB share name already exists."}'
        return $false
    }
}

function Start-UserStoreCreationProcess {
    if ($Global:gSharePermissions.Count -eq 0) {
        $Global:gSharePermissions.Add("Everyone") | Out-Null
    }

    if ($Global:gMachineName -eq $Global:gLocalMachineName) {
        $create_output = Invoke-Command -ScriptBlock $CreateUserStoreProcess -ArgumentList $Global:gFolderPath, $Global:gSmbShareName, $Global:gSharePermissions, $Global:gFolderPermissions, $Global:gFolderAdministrators
    }
    else {
        $create_output = Invoke-Command -Session $Global:gRemoteSession -ScriptBlock $CreateUserStoreProcess -ArgumentList $Global:gFolderPath, $Global:gSmbShareName, $Global:gSharePermissions, $Global:gFolderPermissions, $Global:gFolderAdministrators
    }

    if ($create_output -eq 'success') {
        return $true
    }
    else {
        $Global:gLastErrorMsg = "{`"ErrorCode`": `"ERROR_OTHERS`", `"Details`": `"$create_output`"}"
        return $false
    }
}

# Direct create the user store with all parameters
function Start-UserStoreCreationDirectionCreationMode {
    param(
        [String]$machineName,
        [String]$folderPath,
        [Bool]  $forceFolderPath,
        [String]$smbShareName,
        [Bool]  $forceSmbShareName,
        [String]$folderPermissions,
        [String]$folderAdministrators
    )

    $confirm_res = Confirm-MachineName -machineName $machineName
    if ($confirm_res -ne $true) {
        return $false
    }

    if ($forceFolderPath -ne $true) {
        $confirm_res = Confirm-FolderPath -folderPath $folderPath
        if ($confirm_res -ne $true) {
            return $false
        }
    }
    else {
        $Global:gFolderPath = $folderPath
    }

    if ($forceSmbShareName -ne $true) {
        $confirm_res = Confirm-SmbShareName -smbShareName $smbShareName
        if ($confirm_res -ne $true) {
            return $false
        }
    }
    else {
        $Global:gSmbShareName = $smbShareName
    }

    $Global:gSharePermissions.Clear()

    $Global:gFolderPermissions.Clear()
    $group_user_list = $folderPermissions.split('|')
    foreach ($tmp in $group_user_list) {
        $group_user = $tmp.ToString().Trim()
        if ($group_user -eq '') {
            continue
        }
        try {
            $obj_sid = New-Object System.Security.Principal.SecurityIdentifier($group_user)
            $obj_user = $obj_sid.Translate( [System.Security.Principal.NTAccount])
            $Global:gFolderPermissions.Add($obj_user.Value) | Out-Null
        }
        catch {
            $Global:gLastErrorMsg = "{`"ErrorCode`": `"ERROR_OTHERS`", `"Details`": `"Invalid script parameter: $group_user not found.`"}"
            return $false
        }
    }

    $Global:gFolderAdministrators.Clear()
    if ($folderAdministrators -ne '') {
        $group_user_list = $folderAdministrators.split('|')
        foreach ($tmp in $group_user_list) {
            $group_user = $tmp.ToString().Trim()
            if ($group_user -eq '') {
                continue
            }
            try {
                $obj_sid = New-Object System.Security.Principal.SecurityIdentifier($group_user)
                $obj_user = $obj_sid.Translate( [System.Security.Principal.NTAccount])
                $Global:gFolderAdministrators.Add($obj_user.Value) | Out-Null
            }
            catch {
                $Global:gLastErrorMsg = "{`"ErrorCode`": `"ERROR_OTHERS`", `"Details`": `"Invalid script parameter: $group_user not found.`"}"
                return $false
            }
        }
    }

    $res = Start-UserStoreCreationProcess
    if ($res -eq $true) {
        $Global:gPathToUserStoreSetting = "\\$Global:gMachineName\$Global:gSmbShareName\%USERNAME%.%USERDOMAIN%\!CTX_OSNAME!!CTX_OSBITNESS!"
        return $true
    }
    else {
        $Global:gPathToUserStoreSetting = ''
        return $false
    }
}

function Complete-Script {
    if ($Global:gRemoteSession -ne $null) {
        Remove-PSSession $Global:gRemoteSession
        Remove-Variable -Name gRemoteSession -Scope Global
    }
    if ($Global:gMachineName -ne $null) {
        Remove-Variable -Name gMachineName -Scope Global
    }
    if ($Global:gFolderPath -ne $null) {
        Remove-Variable -Name gFolderPath -Scope Global
    }
    if ($Global:gSmbShareName -ne $null) {
        Remove-Variable -Name gSmbShareName -Scope Global
    }
    if ($Global:gSharePermissions -ne $null) {
        Remove-Variable -Name gSharePermissions -Scope Global
    }
    if ($Global:gFolderPermissions -ne $null) {
        Remove-Variable -Name gFolderPermissions -Scope Global
    }
    if ($Global:gFolderAdministrators -ne $null) {
        Remove-Variable -Name gFolderAdministrators -Scope Global
    }
    if ($Global:gLocalMachineName -ne $null) {
        Remove-Variable -Name gLocalMachineName -Scope Global
    }
    if ($Global:gRemoteCredential -ne $null) {
        Remove-Variable -Name gRemoteCredential -Scope Global
    }
    if ($Global:gPathToUserStoreSetting -ne $null) {
        Remove-Variable -Name gPathToUserStoreSetting -Scope Global
    }
    if ($Global:gLastErrorMsg -ne $null) {
        Remove-Variable -Name gLastErrorMsg -Scope Global
    }
}

# The user store setting vars
$Global:gMachineName = ''
$Global:gFolderPath = ''
$Global:gSmbShareName = ''
$Global:gSharePermissions = New-Object -TypeName 'System.Collections.ArrayList'
$Global:gFolderPermissions = New-Object -TypeName 'System.Collections.ArrayList'
$Global:gFolderAdministrators = New-Object -TypeName 'System.Collections.ArrayList'

$Global:gLocalMachineName = ''
$Global:gRemoteCredential = $null
$Global:gRemoteSession = $null
$Global:gPathToUserStoreSetting = ''

$Global:gLastErrorMsg = ''

# Init
$Global:gLocalMachineName = $env:computername
if ($RemoteCredential -ne $null) {
    $Global:gRemoteCredential = $RemoteCredential
}

# Parameters check
if ($FolderPath -eq '') {
    $output = '{"ErrorCode": "ERROR_OTHERS", "Details": "Invalid script parameter: FolderPath."}'
    Write-Output $output
    return
}
if ($SmbShareName -eq '') {
    $output = '{"ErrorCode": "ERROR_OTHERS", "Details": "Invalid script parameter: SmbShareName."}'
    Write-Output $output
    return
}
if ($FolderPermissions -eq '') {
    $output = '{"ErrorCode": "ERROR_OTHERS", "Details": "Invalid script parameter: FolderPermissions."}'
    Write-Output $output
    return
}

# Create process
$res = Start-UserStoreCreationDirectionCreationMode -machineName $MachineName -folderPath $FolderPath -forceFolderPath $ForceFolderPath -smbShareName $SmbShareName -forceSmbShareName $ForceSmbShareName -folderPermissions $FolderPermissions -folderAdministrators $FolderAdministrators
if ($res -eq $true) {
    $output = '{"ErrorCode": "SUCCESS", "Details": "' + $Global:gPathToUserStoreSetting + '"}'
    Write-Output $output
}
else {
    Write-Output $Global:gLastErrorMsg
}

Complete-Script

# SIG # Begin signature block
# MIIolwYJKoZIhvcNAQcCoIIoiDCCKIQCAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCDoVaYpU/me9FuI
# GIOcU/DpIe/5De2LwYYlskf1mZ8ktaCCDcAwggawMIIEmKADAgECAhAIrUCyYNKc
# TJ9ezam9k67ZMA0GCSqGSIb3DQEBDAUAMGIxCzAJBgNVBAYTAlVTMRUwEwYDVQQK
# EwxEaWdpQ2VydCBJbmMxGTAXBgNVBAsTEHd3dy5kaWdpY2VydC5jb20xITAfBgNV
# BAMTGERpZ2lDZXJ0IFRydXN0ZWQgUm9vdCBHNDAeFw0yMTA0MjkwMDAwMDBaFw0z
# NjA0MjgyMzU5NTlaMGkxCzAJBgNVBAYTAlVTMRcwFQYDVQQKEw5EaWdpQ2VydCwg
# SW5jLjFBMD8GA1UEAxM4RGlnaUNlcnQgVHJ1c3RlZCBHNCBDb2RlIFNpZ25pbmcg
# UlNBNDA5NiBTSEEzODQgMjAyMSBDQTEwggIiMA0GCSqGSIb3DQEBAQUAA4ICDwAw
# ggIKAoICAQDVtC9C0CiteLdd1TlZG7GIQvUzjOs9gZdwxbvEhSYwn6SOaNhc9es0
# JAfhS0/TeEP0F9ce2vnS1WcaUk8OoVf8iJnBkcyBAz5NcCRks43iCH00fUyAVxJr
# Q5qZ8sU7H/Lvy0daE6ZMswEgJfMQ04uy+wjwiuCdCcBlp/qYgEk1hz1RGeiQIXhF
# LqGfLOEYwhrMxe6TSXBCMo/7xuoc82VokaJNTIIRSFJo3hC9FFdd6BgTZcV/sk+F
# LEikVoQ11vkunKoAFdE3/hoGlMJ8yOobMubKwvSnowMOdKWvObarYBLj6Na59zHh
# 3K3kGKDYwSNHR7OhD26jq22YBoMbt2pnLdK9RBqSEIGPsDsJ18ebMlrC/2pgVItJ
# wZPt4bRc4G/rJvmM1bL5OBDm6s6R9b7T+2+TYTRcvJNFKIM2KmYoX7BzzosmJQay
# g9Rc9hUZTO1i4F4z8ujo7AqnsAMrkbI2eb73rQgedaZlzLvjSFDzd5Ea/ttQokbI
# YViY9XwCFjyDKK05huzUtw1T0PhH5nUwjewwk3YUpltLXXRhTT8SkXbev1jLchAp
# QfDVxW0mdmgRQRNYmtwmKwH0iU1Z23jPgUo+QEdfyYFQc4UQIyFZYIpkVMHMIRro
# OBl8ZhzNeDhFMJlP/2NPTLuqDQhTQXxYPUez+rbsjDIJAsxsPAxWEQIDAQABo4IB
# WTCCAVUwEgYDVR0TAQH/BAgwBgEB/wIBADAdBgNVHQ4EFgQUaDfg67Y7+F8Rhvv+
# YXsIiGX0TkIwHwYDVR0jBBgwFoAU7NfjgtJxXWRM3y5nP+e6mK4cD08wDgYDVR0P
# AQH/BAQDAgGGMBMGA1UdJQQMMAoGCCsGAQUFBwMDMHcGCCsGAQUFBwEBBGswaTAk
# BggrBgEFBQcwAYYYaHR0cDovL29jc3AuZGlnaWNlcnQuY29tMEEGCCsGAQUFBzAC
# hjVodHRwOi8vY2FjZXJ0cy5kaWdpY2VydC5jb20vRGlnaUNlcnRUcnVzdGVkUm9v
# dEc0LmNydDBDBgNVHR8EPDA6MDigNqA0hjJodHRwOi8vY3JsMy5kaWdpY2VydC5j
# b20vRGlnaUNlcnRUcnVzdGVkUm9vdEc0LmNybDAcBgNVHSAEFTATMAcGBWeBDAED
# MAgGBmeBDAEEATANBgkqhkiG9w0BAQwFAAOCAgEAOiNEPY0Idu6PvDqZ01bgAhql
# +Eg08yy25nRm95RysQDKr2wwJxMSnpBEn0v9nqN8JtU3vDpdSG2V1T9J9Ce7FoFF
# UP2cvbaF4HZ+N3HLIvdaqpDP9ZNq4+sg0dVQeYiaiorBtr2hSBh+3NiAGhEZGM1h
# mYFW9snjdufE5BtfQ/g+lP92OT2e1JnPSt0o618moZVYSNUa/tcnP/2Q0XaG3Ryw
# YFzzDaju4ImhvTnhOE7abrs2nfvlIVNaw8rpavGiPttDuDPITzgUkpn13c5Ubdld
# AhQfQDN8A+KVssIhdXNSy0bYxDQcoqVLjc1vdjcshT8azibpGL6QB7BDf5WIIIJw
# 8MzK7/0pNVwfiThV9zeKiwmhywvpMRr/LhlcOXHhvpynCgbWJme3kuZOX956rEnP
# LqR0kq3bPKSchh/jwVYbKyP/j7XqiHtwa+aguv06P0WmxOgWkVKLQcBIhEuWTatE
# QOON8BUozu3xGFYHKi8QxAwIZDwzj64ojDzLj4gLDb879M4ee47vtevLt/B3E+bn
# KD+sEq6lLyJsQfmCXBVmzGwOysWGw/YmMwwHS6DTBwJqakAwSEs0qFEgu60bhQji
# WQ1tygVQK+pKHJ6l/aCnHwZ05/LWUpD9r4VIIflXO7ScA+2GRfS0YW6/aOImYIbq
# yK+p/pQd52MbOoZWeE4wggcIMIIE8KADAgECAhAExKVRXTiJBY6ZBfjtmlFwMA0G
# CSqGSIb3DQEBCwUAMGkxCzAJBgNVBAYTAlVTMRcwFQYDVQQKEw5EaWdpQ2VydCwg
# SW5jLjFBMD8GA1UEAxM4RGlnaUNlcnQgVHJ1c3RlZCBHNCBDb2RlIFNpZ25pbmcg
# UlNBNDA5NiBTSEEzODQgMjAyMSBDQTEwHhcNMjQwMzA4MDAwMDAwWhcNMjUwMzA3
# MjM1OTU5WjCBjzELMAkGA1UEBhMCVVMxEDAOBgNVBAgTB0Zsb3JpZGExGDAWBgNV
# BAcTD0ZvcnQgTGF1ZGVyZGFsZTEdMBsGA1UEChMUQ2l0cml4IFN5c3RlbXMsIElu
# Yy4xFjAUBgNVBAsTDUNpdHJpeCBYZW5BcHAxHTAbBgNVBAMTFENpdHJpeCBTeXN0
# ZW1zLCBJbmMuMIIBojANBgkqhkiG9w0BAQEFAAOCAY8AMIIBigKCAYEA+gJ84IK2
# t0aBgBuXB3yfqoC+riwIEohXTQHVqNvJ7v3xiiIYD8PvzYLhaQq2dG4euY1AOhXL
# 0Uc2rNLcQZKEB4PslYrrorg+x5iRJl2k9XltDEpDbqJJqkUELY6d7GEIpp4ChdGf
# NJuGU9GCAgiml88tzZMOjtAqcLIXSRSOugQgjUOR+EvfeLlMTLs/ADjyOi898+0m
# TtdE6Sy/a2bn/fgsJaU8O2Kn78YiHX1N6lklKKBzO5tYfRD2VZhOn2kx6PUI+n+R
# ckwE22o0hBhckBeBGNW2DUrABDLwYuNOEWZEGx1KC2UDO4p+eDmxANjC/+q9t53d
# Z4V1Y/qL9CRnin0JNowVcLT3zGqeebgIag2Pqsh6m3lvASiy7DpmDtyaTCwv/aLd
# 1x5TZvUPiKcfvMOSCRxTP+a7L4DdjuqcQxWENk1kNEZjZa/ZS0BDWalklp6Ji9ki
# JpUKxB5C6GD/Lns+hbdp/2XHnEsqGbeQBCiW7ftXO3ugffq+/Bz1CelPAgMBAAGj
# ggIDMIIB/zAfBgNVHSMEGDAWgBRoN+Drtjv4XxGG+/5hewiIZfROQjAdBgNVHQ4E
# FgQUh00+M0JEuzdBGq4PW6l3UIoBQsIwPgYDVR0gBDcwNTAzBgZngQwBBAEwKTAn
# BggrBgEFBQcCARYbaHR0cDovL3d3dy5kaWdpY2VydC5jb20vQ1BTMA4GA1UdDwEB
# /wQEAwIHgDATBgNVHSUEDDAKBggrBgEFBQcDAzCBtQYDVR0fBIGtMIGqMFOgUaBP
# hk1odHRwOi8vY3JsMy5kaWdpY2VydC5jb20vRGlnaUNlcnRUcnVzdGVkRzRDb2Rl
# U2lnbmluZ1JTQTQwOTZTSEEzODQyMDIxQ0ExLmNybDBToFGgT4ZNaHR0cDovL2Ny
# bDQuZGlnaWNlcnQuY29tL0RpZ2lDZXJ0VHJ1c3RlZEc0Q29kZVNpZ25pbmdSU0E0
# MDk2U0hBMzg0MjAyMUNBMS5jcmwwgZQGCCsGAQUFBwEBBIGHMIGEMCQGCCsGAQUF
# BzABhhhodHRwOi8vb2NzcC5kaWdpY2VydC5jb20wXAYIKwYBBQUHMAKGUGh0dHA6
# Ly9jYWNlcnRzLmRpZ2ljZXJ0LmNvbS9EaWdpQ2VydFRydXN0ZWRHNENvZGVTaWdu
# aW5nUlNBNDA5NlNIQTM4NDIwMjFDQTEuY3J0MAkGA1UdEwQCMAAwDQYJKoZIhvcN
# AQELBQADggIBAI1e2H/UCCob4N14i0h2Z1Yi7dInt7RICuwc9yiMcDeGvlvtgcJT
# zp/CyQwpDh58/WGxR0HX3TuLjLh0Foxzm6aSgTLnyvI65jugv24L7yu1Bl5n7WR1
# x4tqtvVI1DOCTWNN9BqokeBTuT7NVwDPyoLWxz8f3GqnERJ/xe4DJOv9RXre6sy0
# NF9a+d4oqvEJ5r03WOpI6jO3YG6hyxCF7611ZhVdvBuw0ZfVgXPH1n/l+kGb4Mdc
# s0kJ6ny/auQ3Cm0lpCxrjh+LxwrxYtk1s/iZMd9T+C5vjUzH1LWhqlPur0ur05yU
# cG61A8pPwKLE02FdW9b6miAPknpUI6o6EgdU1KRCCjtYoR1iBSagB75k0RZBcgor
# B6mQKcFF5EwRhw5rSJdn1r7rTjRgVTzgRjgHL7QBY27yshp9rTwqHzO9VUmfTxg1
# 4UT5dcZ3xGc8U4j+gMSjNXmJJEpRaNtuh1QDXoQCwC0fM/9r4BxIlS50nuWrhnBL
# zSbrdxbmDK+iW5UU2oCKs3zHbrPmqQgWd4CDi/g/kUEruRIL+HMS1eAdZlyZ8DCw
# H8Qwv6+T6we/iDDbHtxLy7TjGsYChqEgaVOT7gj++iIdoHq3+mb1bxtDgovELXFN
# EwOxXT5mt7qZlc4ApoUTdruNzS9aMofoIB2SIVo/P6FCryDE5YxVmv6zMYIaLTCC
# GikCAQEwfTBpMQswCQYDVQQGEwJVUzEXMBUGA1UEChMORGlnaUNlcnQsIEluYy4x
# QTA/BgNVBAMTOERpZ2lDZXJ0IFRydXN0ZWQgRzQgQ29kZSBTaWduaW5nIFJTQTQw
# OTYgU0hBMzg0IDIwMjEgQ0ExAhAExKVRXTiJBY6ZBfjtmlFwMA0GCWCGSAFlAwQC
# AQUAoIHEMBkGCSqGSIb3DQEJAzEMBgorBgEEAYI3AgEEMBwGCisGAQQBgjcCAQsx
# DjAMBgorBgEEAYI3AgEVMC8GCSqGSIb3DQEJBDEiBCAidOzmnXntDTTaHfCqAl3M
# 7u+QcXvOWDXHsh+zJNfumzBYBgorBgEEAYI3AgEMMUowSKAsgCoAQwBQAE0AXwBV
# AHMAZQByAF8AUwB0AG8AcgBlAF8AQwByAGUAYQB0AGWhGIAWaHR0cDovL3d3dy5j
# aXRyaXguY29tIDANBgkqhkiG9w0BAQEFAASCAYDcTzYK0WQmeB4q+GdIi8bk0s7b
# AUDjjZTWiIWPtG3OQyTlebfDYdp1QRurvEAD0gPt4SC3bhYZNxOwJ+cd/loMzi0p
# OVnVppdooALHIxEFM6Eq69ny3NTGaEFc6+H2b/dh1yBLBhzK3eTninGK/y1ZyceQ
# qxdy7utX3txOGtph1p66+iKTrkxsGvnPLvKE+0bXTJSlVJGhJBEoOg4fZOzYu+3g
# gYwoIoNl+nd4RtRWhGpukIfUegyGEHNVJqzNOfT6PhNocq6nW/xPM2oarbuGRolh
# 9D3AT2bpKJE9Hl+YJ+dhdFNAUvz5+JQn46NpT9WoZXV9BCUMkOTKl7u4SGU8DK7Y
# 1Tvvu4+bqGk90KKRn4pqqeLZju94ZW4CQ5g+kzOb9wEUYWaVxxFg1q3IbBIajvAG
# APhafw0EeMEOB+0BrWgy+noyHlVuBlRb+XPM4nmgTk8PbQZDMCVdNoe8hZj6N6Ff
# S3AiOP6yS/q5OWQYbjK44rGSXcDWuhPsEg4EU4Ghghc6MIIXNgYKKwYBBAGCNwMD
# ATGCFyYwghciBgkqhkiG9w0BBwKgghcTMIIXDwIBAzEPMA0GCWCGSAFlAwQCAQUA
# MHgGCyqGSIb3DQEJEAEEoGkEZzBlAgEBBglghkgBhv1sBwEwMTANBglghkgBZQME
# AgEFAAQgwK5pqKhLoC4CKI7r6oe13Ogf3XmhESuNnSyiFsFuNV4CEQD5kebTa5mr
# Ylq2RUsUfrNSGA8yMDI0MTIwNDA5MDYzMFqgghMDMIIGvDCCBKSgAwIBAgIQC65m
# vFq6f5WHxvnpBOMzBDANBgkqhkiG9w0BAQsFADBjMQswCQYDVQQGEwJVUzEXMBUG
# A1UEChMORGlnaUNlcnQsIEluYy4xOzA5BgNVBAMTMkRpZ2lDZXJ0IFRydXN0ZWQg
# RzQgUlNBNDA5NiBTSEEyNTYgVGltZVN0YW1waW5nIENBMB4XDTI0MDkyNjAwMDAw
# MFoXDTM1MTEyNTIzNTk1OVowQjELMAkGA1UEBhMCVVMxETAPBgNVBAoTCERpZ2lD
# ZXJ0MSAwHgYDVQQDExdEaWdpQ2VydCBUaW1lc3RhbXAgMjAyNDCCAiIwDQYJKoZI
# hvcNAQEBBQADggIPADCCAgoCggIBAL5qc5/2lSGrljC6W23mWaO16P2RHxjEiDtq
# meOlwf0KMCBDEr4IxHRGd7+L660x5XltSVhhK64zi9CeC9B6lUdXM0s71EOcRe8+
# CEJp+3R2O8oo76EO7o5tLuslxdr9Qq82aKcpA9O//X6QE+AcaU/byaCagLD/GLoU
# b35SfWHh43rOH3bpLEx7pZ7avVnpUVmPvkxT8c2a2yC0WMp8hMu60tZR0ChaV76N
# hnj37DEYTX9ReNZ8hIOYe4jl7/r419CvEYVIrH6sN00yx49boUuumF9i2T8UuKGn
# 9966fR5X6kgXj3o5WHhHVO+NBikDO0mlUh902wS/Eeh8F/UFaRp1z5SnROHwSJ+Q
# QRZ1fisD8UTVDSupWJNstVkiqLq+ISTdEjJKGjVfIcsgA4l9cbk8Smlzddh4EfvF
# rpVNnes4c16Jidj5XiPVdsn5n10jxmGpxoMc6iPkoaDhi6JjHd5ibfdp5uzIXp4P
# 0wXkgNs+CO/CacBqU0R4k+8h6gYldp4FCMgrXdKWfM4N0u25OEAuEa3JyidxW48j
# wBqIJqImd93NRxvd1aepSeNeREXAu2xUDEW8aqzFQDYmr9ZONuc2MhTMizchNULp
# UEoA6Vva7b1XCB+1rxvbKmLqfY/M/SdV6mwWTyeVy5Z/JkvMFpnQy5wR14GJcv6d
# Q4aEKOX5AgMBAAGjggGLMIIBhzAOBgNVHQ8BAf8EBAMCB4AwDAYDVR0TAQH/BAIw
# ADAWBgNVHSUBAf8EDDAKBggrBgEFBQcDCDAgBgNVHSAEGTAXMAgGBmeBDAEEAjAL
# BglghkgBhv1sBwEwHwYDVR0jBBgwFoAUuhbZbU2FL3MpdpovdYxqII+eyG8wHQYD
# VR0OBBYEFJ9XLAN3DigVkGalY17uT5IfdqBbMFoGA1UdHwRTMFEwT6BNoEuGSWh0
# dHA6Ly9jcmwzLmRpZ2ljZXJ0LmNvbS9EaWdpQ2VydFRydXN0ZWRHNFJTQTQwOTZT
# SEEyNTZUaW1lU3RhbXBpbmdDQS5jcmwwgZAGCCsGAQUFBwEBBIGDMIGAMCQGCCsG
# AQUFBzABhhhodHRwOi8vb2NzcC5kaWdpY2VydC5jb20wWAYIKwYBBQUHMAKGTGh0
# dHA6Ly9jYWNlcnRzLmRpZ2ljZXJ0LmNvbS9EaWdpQ2VydFRydXN0ZWRHNFJTQTQw
# OTZTSEEyNTZUaW1lU3RhbXBpbmdDQS5jcnQwDQYJKoZIhvcNAQELBQADggIBAD2t
# Hh92mVvjOIQSR9lDkfYR25tOCB3RKE/P09x7gUsmXqt40ouRl3lj+8QioVYq3igp
# wrPvBmZdrlWBb0HvqT00nFSXgmUrDKNSQqGTdpjHsPy+LaalTW0qVjvUBhcHzBMu
# tB6HzeledbDCzFzUy34VarPnvIWrqVogK0qM8gJhh/+qDEAIdO/KkYesLyTVOoJ4
# eTq7gj9UFAL1UruJKlTnCVaM2UeUUW/8z3fvjxhN6hdT98Vr2FYlCS7Mbb4Hv5sw
# O+aAXxWUm3WpByXtgVQxiBlTVYzqfLDbe9PpBKDBfk+rabTFDZXoUke7zPgtd7/f
# vWTlCs30VAGEsshJmLbJ6ZbQ/xll/HjO9JbNVekBv2Tgem+mLptR7yIrpaidRJXr
# I+UzB6vAlk/8a1u7cIqV0yef4uaZFORNekUgQHTqddmsPCEIYQP7xGxZBIhdmm4b
# hYsVA6G2WgNFYagLDBzpmk9104WQzYuVNsxyoVLObhx3RugaEGru+SojW4dHPoWr
# UhftNpFC5H7QEY7MhKRyrBe7ucykW7eaCuWBsBb4HOKRFVDcrZgdwaSIqMDiCLg4
# D+TPVgKx2EgEdeoHNHT9l3ZDBD+XgbF+23/zBjeCtxz+dL/9NWR6P2eZRi7zcEO1
# xwcdcqJsyz/JceENc2Sg8h3KeFUCS7tpFk7CrDqkMIIGrjCCBJagAwIBAgIQBzY3
# tyRUfNhHrP0oZipeWzANBgkqhkiG9w0BAQsFADBiMQswCQYDVQQGEwJVUzEVMBMG
# A1UEChMMRGlnaUNlcnQgSW5jMRkwFwYDVQQLExB3d3cuZGlnaWNlcnQuY29tMSEw
# HwYDVQQDExhEaWdpQ2VydCBUcnVzdGVkIFJvb3QgRzQwHhcNMjIwMzIzMDAwMDAw
# WhcNMzcwMzIyMjM1OTU5WjBjMQswCQYDVQQGEwJVUzEXMBUGA1UEChMORGlnaUNl
# cnQsIEluYy4xOzA5BgNVBAMTMkRpZ2lDZXJ0IFRydXN0ZWQgRzQgUlNBNDA5NiBT
# SEEyNTYgVGltZVN0YW1waW5nIENBMIICIjANBgkqhkiG9w0BAQEFAAOCAg8AMIIC
# CgKCAgEAxoY1BkmzwT1ySVFVxyUDxPKRN6mXUaHW0oPRnkyibaCwzIP5WvYRoUQV
# Ql+kiPNo+n3znIkLf50fng8zH1ATCyZzlm34V6gCff1DtITaEfFzsbPuK4CEiiIY
# 3+vaPcQXf6sZKz5C3GeO6lE98NZW1OcoLevTsbV15x8GZY2UKdPZ7Gnf2ZCHRgB7
# 20RBidx8ald68Dd5n12sy+iEZLRS8nZH92GDGd1ftFQLIWhuNyG7QKxfst5Kfc71
# ORJn7w6lY2zkpsUdzTYNXNXmG6jBZHRAp8ByxbpOH7G1WE15/tePc5OsLDnipUjW
# 8LAxE6lXKZYnLvWHpo9OdhVVJnCYJn+gGkcgQ+NDY4B7dW4nJZCYOjgRs/b2nuY7
# W+yB3iIU2YIqx5K/oN7jPqJz+ucfWmyU8lKVEStYdEAoq3NDzt9KoRxrOMUp88qq
# lnNCaJ+2RrOdOqPVA+C/8KI8ykLcGEh/FDTP0kyr75s9/g64ZCr6dSgkQe1CvwWc
# ZklSUPRR8zZJTYsg0ixXNXkrqPNFYLwjjVj33GHek/45wPmyMKVM1+mYSlg+0wOI
# /rOP015LdhJRk8mMDDtbiiKowSYI+RQQEgN9XyO7ZONj4KbhPvbCdLI/Hgl27Ktd
# RnXiYKNYCQEoAA6EVO7O6V3IXjASvUaetdN2udIOa5kM0jO0zbECAwEAAaOCAV0w
# ggFZMBIGA1UdEwEB/wQIMAYBAf8CAQAwHQYDVR0OBBYEFLoW2W1NhS9zKXaaL3WM
# aiCPnshvMB8GA1UdIwQYMBaAFOzX44LScV1kTN8uZz/nupiuHA9PMA4GA1UdDwEB
# /wQEAwIBhjATBgNVHSUEDDAKBggrBgEFBQcDCDB3BggrBgEFBQcBAQRrMGkwJAYI
# KwYBBQUHMAGGGGh0dHA6Ly9vY3NwLmRpZ2ljZXJ0LmNvbTBBBggrBgEFBQcwAoY1
# aHR0cDovL2NhY2VydHMuZGlnaWNlcnQuY29tL0RpZ2lDZXJ0VHJ1c3RlZFJvb3RH
# NC5jcnQwQwYDVR0fBDwwOjA4oDagNIYyaHR0cDovL2NybDMuZGlnaWNlcnQuY29t
# L0RpZ2lDZXJ0VHJ1c3RlZFJvb3RHNC5jcmwwIAYDVR0gBBkwFzAIBgZngQwBBAIw
# CwYJYIZIAYb9bAcBMA0GCSqGSIb3DQEBCwUAA4ICAQB9WY7Ak7ZvmKlEIgF+ZtbY
# IULhsBguEE0TzzBTzr8Y+8dQXeJLKftwig2qKWn8acHPHQfpPmDI2AvlXFvXbYf6
# hCAlNDFnzbYSlm/EUExiHQwIgqgWvalWzxVzjQEiJc6VaT9Hd/tydBTX/6tPiix6
# q4XNQ1/tYLaqT5Fmniye4Iqs5f2MvGQmh2ySvZ180HAKfO+ovHVPulr3qRCyXen/
# KFSJ8NWKcXZl2szwcqMj+sAngkSumScbqyQeJsG33irr9p6xeZmBo1aGqwpFyd/E
# jaDnmPv7pp1yr8THwcFqcdnGE4AJxLafzYeHJLtPo0m5d2aR8XKc6UsCUqc3fpNT
# rDsdCEkPlM05et3/JWOZJyw9P2un8WbDQc1PtkCbISFA0LcTJM3cHXg65J6t5TRx
# ktcma+Q4c6umAU+9Pzt4rUyt+8SVe+0KXzM5h0F4ejjpnOHdI/0dKNPH+ejxmF/7
# K9h+8kaddSweJywm228Vex4Ziza4k9Tm8heZWcpw8De/mADfIBZPJ/tgZxahZrrd
# VcA6KYawmKAr7ZVBtzrVFZgxtGIJDwq9gdkT/r+k0fNX2bwE+oLeMt8EifAAzV3C
# +dAjfwAL5HYCJtnwZXZCpimHCUcr5n8apIUP/JiW9lVUKx+A+sDyDivl1vupL0QV
# SucTDh3bNzgaoSv27dZ8/DCCBY0wggR1oAMCAQICEA6bGI750C3n79tQ4ghAGFow
# DQYJKoZIhvcNAQEMBQAwZTELMAkGA1UEBhMCVVMxFTATBgNVBAoTDERpZ2lDZXJ0
# IEluYzEZMBcGA1UECxMQd3d3LmRpZ2ljZXJ0LmNvbTEkMCIGA1UEAxMbRGlnaUNl
# cnQgQXNzdXJlZCBJRCBSb290IENBMB4XDTIyMDgwMTAwMDAwMFoXDTMxMTEwOTIz
# NTk1OVowYjELMAkGA1UEBhMCVVMxFTATBgNVBAoTDERpZ2lDZXJ0IEluYzEZMBcG
# A1UECxMQd3d3LmRpZ2ljZXJ0LmNvbTEhMB8GA1UEAxMYRGlnaUNlcnQgVHJ1c3Rl
# ZCBSb290IEc0MIICIjANBgkqhkiG9w0BAQEFAAOCAg8AMIICCgKCAgEAv+aQc2je
# u+RdSjwwIjBpM+zCpyUuySE98orYWcLhKac9WKt2ms2uexuEDcQwH/MbpDgW61bG
# l20dq7J58soR0uRf1gU8Ug9SH8aeFaV+vp+pVxZZVXKvaJNwwrK6dZlqczKU0RBE
# EC7fgvMHhOZ0O21x4i0MG+4g1ckgHWMpLc7sXk7Ik/ghYZs06wXGXuxbGrzryc/N
# rDRAX7F6Zu53yEioZldXn1RYjgwrt0+nMNlW7sp7XeOtyU9e5TXnMcvak17cjo+A
# 2raRmECQecN4x7axxLVqGDgDEI3Y1DekLgV9iPWCPhCRcKtVgkEy19sEcypukQF8
# IUzUvK4bA3VdeGbZOjFEmjNAvwjXWkmkwuapoGfdpCe8oU85tRFYF/ckXEaPZPfB
# aYh2mHY9WV1CdoeJl2l6SPDgohIbZpp0yt5LHucOY67m1O+SkjqePdwA5EUlibaa
# RBkrfsCUtNJhbesz2cXfSwQAzH0clcOP9yGyshG3u3/y1YxwLEFgqrFjGESVGnZi
# fvaAsPvoZKYz0YkH4b235kOkGLimdwHhD5QMIR2yVCkliWzlDlJRR3S+Jqy2QXXe
# eqxfjT/JvNNBERJb5RBQ6zHFynIWIgnffEx1P2PsIV/EIFFrb7GrhotPwtZFX50g
# /KEexcCPorF+CiaZ9eRpL5gdLfXZqbId5RsCAwEAAaOCATowggE2MA8GA1UdEwEB
# /wQFMAMBAf8wHQYDVR0OBBYEFOzX44LScV1kTN8uZz/nupiuHA9PMB8GA1UdIwQY
# MBaAFEXroq/0ksuCMS1Ri6enIZ3zbcgPMA4GA1UdDwEB/wQEAwIBhjB5BggrBgEF
# BQcBAQRtMGswJAYIKwYBBQUHMAGGGGh0dHA6Ly9vY3NwLmRpZ2ljZXJ0LmNvbTBD
# BggrBgEFBQcwAoY3aHR0cDovL2NhY2VydHMuZGlnaWNlcnQuY29tL0RpZ2lDZXJ0
# QXNzdXJlZElEUm9vdENBLmNydDBFBgNVHR8EPjA8MDqgOKA2hjRodHRwOi8vY3Js
# My5kaWdpY2VydC5jb20vRGlnaUNlcnRBc3N1cmVkSURSb290Q0EuY3JsMBEGA1Ud
# IAQKMAgwBgYEVR0gADANBgkqhkiG9w0BAQwFAAOCAQEAcKC/Q1xV5zhfoKN0Gz22
# Ftf3v1cHvZqsoYcs7IVeqRq7IviHGmlUIu2kiHdtvRoU9BNKei8ttzjv9P+Aufih
# 9/Jy3iS8UgPITtAq3votVs/59PesMHqai7Je1M/RQ0SbQyHrlnKhSLSZy51PpwYD
# E3cnRNTnf+hZqPC/Lwum6fI0POz3A8eHqNJMQBk1RmppVLC4oVaO7KTVPeix3P0c
# 2PR3WlxUjG/voVA9/HYJaISfb8rbII01YBwCA8sgsKxYoA5AY8WYIsGyWfVVa88n
# q2x2zm8jLfR+cWojayL/ErhULSd+2DrZ8LaHlv1b0VysGMNNn3O3AamfV6peKOK5
# lDGCA3YwggNyAgEBMHcwYzELMAkGA1UEBhMCVVMxFzAVBgNVBAoTDkRpZ2lDZXJ0
# LCBJbmMuMTswOQYDVQQDEzJEaWdpQ2VydCBUcnVzdGVkIEc0IFJTQTQwOTYgU0hB
# MjU2IFRpbWVTdGFtcGluZyBDQQIQC65mvFq6f5WHxvnpBOMzBDANBglghkgBZQME
# AgEFAKCB0TAaBgkqhkiG9w0BCQMxDQYLKoZIhvcNAQkQAQQwHAYJKoZIhvcNAQkF
# MQ8XDTI0MTIwNDA5MDYzMFowKwYLKoZIhvcNAQkQAgwxHDAaMBgwFgQU29OF7mLb
# 0j575PZxSFCHJNWGW0UwLwYJKoZIhvcNAQkEMSIEICEAaudcKrqMvn2E6X7KdVD4
# bjT6Wae7ezipPETJs6RZMDcGCyqGSIb3DQEJEAIvMSgwJjAkMCIEIHZ2n6jyYy8f
# Qws6IzCu1lZ1/tdz2wXWZbkFk5hDj5rbMA0GCSqGSIb3DQEBAQUABIICAEJxh27K
# byGHlfFI06nIurhbrBaIy8qNmPjRRtyJoY6U7hdyKqFfKdm7gONmFstyX6uKezpS
# KM1Y5yFvBCSZJje7vl8TEhPuupEvB0RyZ8iWtqlvSsKSo8upfq8LJRjZIThsXfj4
# Em4JMUJT6L3H3ZYzHyBoVFbh4BvLW+wQByeceAGoKbUfhjpEiDDxKB7m9dDwjD6u
# 5M60YRrwHpdZ6hJe4DWxl+J5BdUKxBB/mlPGJGQ+ZounCYABOA4bEU06QkGhtoWx
# FOaG03anbT3axx0VL6X+2sP0yrW2KxAAH9cGoN588H6Qh6EuDdk+7IEdhse5t78D
# 8KZDjtZpAOHKLCY6WkBTLKXF3tVPreezWtnEPZKzoQkOoQBJIxe0tefZ5KM+U/UJ
# NeqdLJ6TsrIt2e9fI8s9LRxFZwOOanWJ01JrV/u7qKI84t/UJZN7CKQlYhS19+Nk
# TljPcXAxrXChSxGgRsMmegIeeNCVytIqHtx/yd7FKuty2vFZO3N6DFyd1YXzUOgb
# fZ5nMq/ZtCtnkLhZ9HaG43G5CdFPaesgOBsj1ne82+uSkfd5pnkFqEaWurqS+3fs
# 6t3rxgiFleD/Y3ELvco7dk7kioiWvieBpLmn6Tibcq6u3ufc/EfVxjng8aU7qFHq
# 5L5KfgCxAI35zbNtsr4kOTrsslAHMkCekb9q
# SIG # End signature block
