﻿function Clean-Up {
    if($pmcPath -ne $null)
    {
        #Delete PMC file
        Remove-Item -Force -ErrorAction SilentlyContinue -Path $pmcPath
    }
    #Delete scheduler task
    Unregister-ScheduledTask -ErrorAction SilentlyContinue -Confirm:$false -TaskName "AutoStopBootLogProcessMonitor*"
}

#Pre-check
$pathItem = Get-ItemProperty -Path HKLM:\SOFTWARE\Citrix\WEM\AdminTool -Name "procMonPath" -ErrorAction SilentlyContinue
if($pathItem -eq $null)
{
    Write-Host "Process monitor path not configured, exit."
    Clean-Up
    exit -1
}
$path = $pathItem.procMonPath
if(-not [System.IO.File]::Exists($path))
{
    Write-Host "Process monitor path unexist, exit."
    Clean-Up
    exit -1
}

$publisher = (Get-AppLockerFileInformation $path).Publisher

if($publisher -eq $null)
{
    Write-Host "Invalid process monitor publisher."
    Clean-Up
    exit -1
}

if(-not $publisher.PublisherName -eq "O=MICROSOFT CORPORATION, L=REDMOND, S=WASHINGTON, C=US")
{
    Write-Host "Invalid process monitor publisher."
    Clean-Up
    exit -1
}

if(-not $publisher.ProductName -eq "SYSINTERNALS PROCMON")
{
    Write-Host "Invalid process monitor publisher."
    Clean-Up
    exit -1
}

if(-not $publisher.BinaryName -eq "PROCESS MONITOR")
{
    Write-Host "Invalid process monitor publisher."
    Clean-Up
    exit -1
}

#Get log folder
$logPath = "C:\\ProgramData"
$logPathItem = Get-ItemProperty -Path HKLM:\SOFTWARE\Citrix\WEM\AdminTool -Name "procMonLogPath" -ErrorAction SilentlyContinue
if($pathItem -eq $null -or $logPathItem.length -eq 0)
{
    Write-Host "Process monitor log path not configured, use the default one: C:\\ProgramData."
}
else
{
    $logPath = $logPathItem.procMonLogPath 
}

#Generate pml file
$DatetimeStr=[System.DateTime]::UtcNow.ToString("yyyyMMddHHmmssfff")
$fileName="log_$DatetimeStr.pml"

$logFullName=[System.IO.Path]::Combine($logPath, $fileName)
Start-Process $path -ArgumentList "/AcceptEula /quiet /ConvertBootLog `"$logFullName`"" -Wait -WindowStyle Hidden
if(-not [System.IO.File]::Exists($logFullName))
{
    Write-Host "No boot logging pml file, exit."
    Clean-Up
    exit -1
}
#Get PMC path
$pmcPathItem = Get-ItemProperty -Path HKLM:\SOFTWARE\Citrix\WEM\AdminTool -Name "bootLoggingPmcPath" -ErrorAction SilentlyContinue
if($pmcPathItem -eq $null -or $pmcPathItem.length -eq 0)
{
    Write-Host "Process monitor configure file not configured, keep the full generated pml file."
}
else
{
    if(-not [System.IO.File]::Exists($pmcPathItem))
    {
        Write-Host "No boot logging pmc file, exit."
        Clean-Up
        exit -1
    }
    #PMC file exists, filter the boot log pml file
    $DatetimeStr=[System.DateTime]::UtcNow.ToString("yyyyMMddHHmmssfff")
    $newFileName="log_$DatetimeStr.pml"
    $pmcPath=$pmcPathItem.bootLoggingPmcPath
    $newLogName=[System.IO.Path]::Combine($logPath, $newFileName)
    Start-Process $path -ArgumentList "/AcceptEula /quiet /OpenLog `"$logFullName`" /SaveApplyFilter /LoadConfig `"$pmcPath`" /SaveAs `"$newLogName`"" -Wait -WindowStyle Hidden

    #Delete the Legacy PML files
    $logNameWithoutExtension=[System.IO.Path]::GetFileNameWithoutExtension($fileName)
    Remove-Item -Force -ErrorAction SilentlyContinue -Path $logFullName
    $index=1
    while([System.IO.File]::Exists([System.IO.Path]::Combine($logPath, "$logNameWithoutExtension-$index.pml")))
    {
        $toBeRemove=[System.IO.Path]::Combine($logPath, "$logNameWithoutExtension-$index.pml")
        Remove-Item -Force -ErrorAction SilentlyContinue -Path $toBeRemove
        $index++
    }

}
Clean-Up

exit 0


# SIG # Begin signature block
# MIInEwYJKoZIhvcNAQcCoIInBDCCJwACAQExCzAJBgUrDgMCGgUAMGkGCisGAQQB
# gjcCAQSgWzBZMDQGCisGAQQBgjcCAR4wJgIDAQAABBAfzDtgWUsITrck0sYpfvNR
# AgEAAgEAAgEAAgEAAgEAMCEwCQYFKw4DAhoFAAQUbaePTBLOiki2J3+sS1smMauf
# 1lSggiDDMIIFjTCCBHWgAwIBAgIQDpsYjvnQLefv21DiCEAYWjANBgkqhkiG9w0B
# AQwFADBlMQswCQYDVQQGEwJVUzEVMBMGA1UEChMMRGlnaUNlcnQgSW5jMRkwFwYD
# VQQLExB3d3cuZGlnaWNlcnQuY29tMSQwIgYDVQQDExtEaWdpQ2VydCBBc3N1cmVk
# IElEIFJvb3QgQ0EwHhcNMjIwODAxMDAwMDAwWhcNMzExMTA5MjM1OTU5WjBiMQsw
# CQYDVQQGEwJVUzEVMBMGA1UEChMMRGlnaUNlcnQgSW5jMRkwFwYDVQQLExB3d3cu
# ZGlnaWNlcnQuY29tMSEwHwYDVQQDExhEaWdpQ2VydCBUcnVzdGVkIFJvb3QgRzQw
# ggIiMA0GCSqGSIb3DQEBAQUAA4ICDwAwggIKAoICAQC/5pBzaN675F1KPDAiMGkz
# 7MKnJS7JIT3yithZwuEppz1Yq3aaza57G4QNxDAf8xukOBbrVsaXbR2rsnnyyhHS
# 5F/WBTxSD1Ifxp4VpX6+n6lXFllVcq9ok3DCsrp1mWpzMpTREEQQLt+C8weE5nQ7
# bXHiLQwb7iDVySAdYyktzuxeTsiT+CFhmzTrBcZe7FsavOvJz82sNEBfsXpm7nfI
# SKhmV1efVFiODCu3T6cw2Vbuyntd463JT17lNecxy9qTXtyOj4DatpGYQJB5w3jH
# trHEtWoYOAMQjdjUN6QuBX2I9YI+EJFwq1WCQTLX2wRzKm6RAXwhTNS8rhsDdV14
# Ztk6MUSaM0C/CNdaSaTC5qmgZ92kJ7yhTzm1EVgX9yRcRo9k98FpiHaYdj1ZXUJ2
# h4mXaXpI8OCiEhtmmnTK3kse5w5jrubU75KSOp493ADkRSWJtppEGSt+wJS00mFt
# 6zPZxd9LBADMfRyVw4/3IbKyEbe7f/LVjHAsQWCqsWMYRJUadmJ+9oCw++hkpjPR
# iQfhvbfmQ6QYuKZ3AeEPlAwhHbJUKSWJbOUOUlFHdL4mrLZBdd56rF+NP8m800ER
# ElvlEFDrMcXKchYiCd98THU/Y+whX8QgUWtvsauGi0/C1kVfnSD8oR7FwI+isX4K
# Jpn15GkvmB0t9dmpsh3lGwIDAQABo4IBOjCCATYwDwYDVR0TAQH/BAUwAwEB/zAd
# BgNVHQ4EFgQU7NfjgtJxXWRM3y5nP+e6mK4cD08wHwYDVR0jBBgwFoAUReuir/SS
# y4IxLVGLp6chnfNtyA8wDgYDVR0PAQH/BAQDAgGGMHkGCCsGAQUFBwEBBG0wazAk
# BggrBgEFBQcwAYYYaHR0cDovL29jc3AuZGlnaWNlcnQuY29tMEMGCCsGAQUFBzAC
# hjdodHRwOi8vY2FjZXJ0cy5kaWdpY2VydC5jb20vRGlnaUNlcnRBc3N1cmVkSURS
# b290Q0EuY3J0MEUGA1UdHwQ+MDwwOqA4oDaGNGh0dHA6Ly9jcmwzLmRpZ2ljZXJ0
# LmNvbS9EaWdpQ2VydEFzc3VyZWRJRFJvb3RDQS5jcmwwEQYDVR0gBAowCDAGBgRV
# HSAAMA0GCSqGSIb3DQEBDAUAA4IBAQBwoL9DXFXnOF+go3QbPbYW1/e/Vwe9mqyh
# hyzshV6pGrsi+IcaaVQi7aSId229GhT0E0p6Ly23OO/0/4C5+KH38nLeJLxSA8hO
# 0Cre+i1Wz/n096wwepqLsl7Uz9FDRJtDIeuWcqFItJnLnU+nBgMTdydE1Od/6Fmo
# 8L8vC6bp8jQ87PcDx4eo0kxAGTVGamlUsLihVo7spNU96LHc/RzY9HdaXFSMb++h
# UD38dglohJ9vytsgjTVgHAIDyyCwrFigDkBjxZgiwbJZ9VVrzyerbHbObyMt9H5x
# aiNrIv8SuFQtJ37YOtnwtoeW/VvRXKwYw02fc7cBqZ9Xql4o4rmUMIIGrjCCBJag
# AwIBAgIQBzY3tyRUfNhHrP0oZipeWzANBgkqhkiG9w0BAQsFADBiMQswCQYDVQQG
# EwJVUzEVMBMGA1UEChMMRGlnaUNlcnQgSW5jMRkwFwYDVQQLExB3d3cuZGlnaWNl
# cnQuY29tMSEwHwYDVQQDExhEaWdpQ2VydCBUcnVzdGVkIFJvb3QgRzQwHhcNMjIw
# MzIzMDAwMDAwWhcNMzcwMzIyMjM1OTU5WjBjMQswCQYDVQQGEwJVUzEXMBUGA1UE
# ChMORGlnaUNlcnQsIEluYy4xOzA5BgNVBAMTMkRpZ2lDZXJ0IFRydXN0ZWQgRzQg
# UlNBNDA5NiBTSEEyNTYgVGltZVN0YW1waW5nIENBMIICIjANBgkqhkiG9w0BAQEF
# AAOCAg8AMIICCgKCAgEAxoY1BkmzwT1ySVFVxyUDxPKRN6mXUaHW0oPRnkyibaCw
# zIP5WvYRoUQVQl+kiPNo+n3znIkLf50fng8zH1ATCyZzlm34V6gCff1DtITaEfFz
# sbPuK4CEiiIY3+vaPcQXf6sZKz5C3GeO6lE98NZW1OcoLevTsbV15x8GZY2UKdPZ
# 7Gnf2ZCHRgB720RBidx8ald68Dd5n12sy+iEZLRS8nZH92GDGd1ftFQLIWhuNyG7
# QKxfst5Kfc71ORJn7w6lY2zkpsUdzTYNXNXmG6jBZHRAp8ByxbpOH7G1WE15/teP
# c5OsLDnipUjW8LAxE6lXKZYnLvWHpo9OdhVVJnCYJn+gGkcgQ+NDY4B7dW4nJZCY
# OjgRs/b2nuY7W+yB3iIU2YIqx5K/oN7jPqJz+ucfWmyU8lKVEStYdEAoq3NDzt9K
# oRxrOMUp88qqlnNCaJ+2RrOdOqPVA+C/8KI8ykLcGEh/FDTP0kyr75s9/g64ZCr6
# dSgkQe1CvwWcZklSUPRR8zZJTYsg0ixXNXkrqPNFYLwjjVj33GHek/45wPmyMKVM
# 1+mYSlg+0wOI/rOP015LdhJRk8mMDDtbiiKowSYI+RQQEgN9XyO7ZONj4KbhPvbC
# dLI/Hgl27KtdRnXiYKNYCQEoAA6EVO7O6V3IXjASvUaetdN2udIOa5kM0jO0zbEC
# AwEAAaOCAV0wggFZMBIGA1UdEwEB/wQIMAYBAf8CAQAwHQYDVR0OBBYEFLoW2W1N
# hS9zKXaaL3WMaiCPnshvMB8GA1UdIwQYMBaAFOzX44LScV1kTN8uZz/nupiuHA9P
# MA4GA1UdDwEB/wQEAwIBhjATBgNVHSUEDDAKBggrBgEFBQcDCDB3BggrBgEFBQcB
# AQRrMGkwJAYIKwYBBQUHMAGGGGh0dHA6Ly9vY3NwLmRpZ2ljZXJ0LmNvbTBBBggr
# BgEFBQcwAoY1aHR0cDovL2NhY2VydHMuZGlnaWNlcnQuY29tL0RpZ2lDZXJ0VHJ1
# c3RlZFJvb3RHNC5jcnQwQwYDVR0fBDwwOjA4oDagNIYyaHR0cDovL2NybDMuZGln
# aWNlcnQuY29tL0RpZ2lDZXJ0VHJ1c3RlZFJvb3RHNC5jcmwwIAYDVR0gBBkwFzAI
# BgZngQwBBAIwCwYJYIZIAYb9bAcBMA0GCSqGSIb3DQEBCwUAA4ICAQB9WY7Ak7Zv
# mKlEIgF+ZtbYIULhsBguEE0TzzBTzr8Y+8dQXeJLKftwig2qKWn8acHPHQfpPmDI
# 2AvlXFvXbYf6hCAlNDFnzbYSlm/EUExiHQwIgqgWvalWzxVzjQEiJc6VaT9Hd/ty
# dBTX/6tPiix6q4XNQ1/tYLaqT5Fmniye4Iqs5f2MvGQmh2ySvZ180HAKfO+ovHVP
# ulr3qRCyXen/KFSJ8NWKcXZl2szwcqMj+sAngkSumScbqyQeJsG33irr9p6xeZmB
# o1aGqwpFyd/EjaDnmPv7pp1yr8THwcFqcdnGE4AJxLafzYeHJLtPo0m5d2aR8XKc
# 6UsCUqc3fpNTrDsdCEkPlM05et3/JWOZJyw9P2un8WbDQc1PtkCbISFA0LcTJM3c
# HXg65J6t5TRxktcma+Q4c6umAU+9Pzt4rUyt+8SVe+0KXzM5h0F4ejjpnOHdI/0d
# KNPH+ejxmF/7K9h+8kaddSweJywm228Vex4Ziza4k9Tm8heZWcpw8De/mADfIBZP
# J/tgZxahZrrdVcA6KYawmKAr7ZVBtzrVFZgxtGIJDwq9gdkT/r+k0fNX2bwE+oLe
# Mt8EifAAzV3C+dAjfwAL5HYCJtnwZXZCpimHCUcr5n8apIUP/JiW9lVUKx+A+sDy
# Divl1vupL0QVSucTDh3bNzgaoSv27dZ8/DCCBrAwggSYoAMCAQICEAitQLJg0pxM
# n17Nqb2TrtkwDQYJKoZIhvcNAQEMBQAwYjELMAkGA1UEBhMCVVMxFTATBgNVBAoT
# DERpZ2lDZXJ0IEluYzEZMBcGA1UECxMQd3d3LmRpZ2ljZXJ0LmNvbTEhMB8GA1UE
# AxMYRGlnaUNlcnQgVHJ1c3RlZCBSb290IEc0MB4XDTIxMDQyOTAwMDAwMFoXDTM2
# MDQyODIzNTk1OVowaTELMAkGA1UEBhMCVVMxFzAVBgNVBAoTDkRpZ2lDZXJ0LCBJ
# bmMuMUEwPwYDVQQDEzhEaWdpQ2VydCBUcnVzdGVkIEc0IENvZGUgU2lnbmluZyBS
# U0E0MDk2IFNIQTM4NCAyMDIxIENBMTCCAiIwDQYJKoZIhvcNAQEBBQADggIPADCC
# AgoCggIBANW0L0LQKK14t13VOVkbsYhC9TOM6z2Bl3DFu8SFJjCfpI5o2Fz16zQk
# B+FLT9N4Q/QX1x7a+dLVZxpSTw6hV/yImcGRzIEDPk1wJGSzjeIIfTR9TIBXEmtD
# mpnyxTsf8u/LR1oTpkyzASAl8xDTi7L7CPCK4J0JwGWn+piASTWHPVEZ6JAheEUu
# oZ8s4RjCGszF7pNJcEIyj/vG6hzzZWiRok1MghFIUmjeEL0UV13oGBNlxX+yT4Us
# SKRWhDXW+S6cqgAV0Tf+GgaUwnzI6hsy5srC9KejAw50pa85tqtgEuPo1rn3MeHc
# reQYoNjBI0dHs6EPbqOrbZgGgxu3amct0r1EGpIQgY+wOwnXx5syWsL/amBUi0nB
# k+3htFzgb+sm+YzVsvk4EObqzpH1vtP7b5NhNFy8k0UogzYqZihfsHPOiyYlBrKD
# 1Fz2FRlM7WLgXjPy6OjsCqewAyuRsjZ5vvetCB51pmXMu+NIUPN3kRr+21CiRshh
# WJj1fAIWPIMorTmG7NS3DVPQ+EfmdTCN7DCTdhSmW0tddGFNPxKRdt6/WMtyEClB
# 8NXFbSZ2aBFBE1ia3CYrAfSJTVnbeM+BSj5AR1/JgVBzhRAjIVlgimRUwcwhGug4
# GXxmHM14OEUwmU//Y09Mu6oNCFNBfFg9R7P6tuyMMgkCzGw8DFYRAgMBAAGjggFZ
# MIIBVTASBgNVHRMBAf8ECDAGAQH/AgEAMB0GA1UdDgQWBBRoN+Drtjv4XxGG+/5h
# ewiIZfROQjAfBgNVHSMEGDAWgBTs1+OC0nFdZEzfLmc/57qYrhwPTzAOBgNVHQ8B
# Af8EBAMCAYYwEwYDVR0lBAwwCgYIKwYBBQUHAwMwdwYIKwYBBQUHAQEEazBpMCQG
# CCsGAQUFBzABhhhodHRwOi8vb2NzcC5kaWdpY2VydC5jb20wQQYIKwYBBQUHMAKG
# NWh0dHA6Ly9jYWNlcnRzLmRpZ2ljZXJ0LmNvbS9EaWdpQ2VydFRydXN0ZWRSb290
# RzQuY3J0MEMGA1UdHwQ8MDowOKA2oDSGMmh0dHA6Ly9jcmwzLmRpZ2ljZXJ0LmNv
# bS9EaWdpQ2VydFRydXN0ZWRSb290RzQuY3JsMBwGA1UdIAQVMBMwBwYFZ4EMAQMw
# CAYGZ4EMAQQBMA0GCSqGSIb3DQEBDAUAA4ICAQA6I0Q9jQh27o+8OpnTVuACGqX4
# SDTzLLbmdGb3lHKxAMqvbDAnExKekESfS/2eo3wm1Te8Ol1IbZXVP0n0J7sWgUVQ
# /Zy9toXgdn43ccsi91qqkM/1k2rj6yDR1VB5iJqKisG2vaFIGH7c2IAaERkYzWGZ
# gVb2yeN258TkG19D+D6U/3Y5PZ7Umc9K3SjrXyahlVhI1Rr+1yc//ZDRdobdHLBg
# XPMNqO7giaG9OeE4Ttpuuzad++UhU1rDyulq8aI+20O4M8hPOBSSmfXdzlRt2V0C
# FB9AM3wD4pWywiF1c1LLRtjENByipUuNzW92NyyFPxrOJukYvpAHsEN/lYgggnDw
# zMrv/Sk1XB+JOFX3N4qLCaHLC+kxGv8uGVw5ceG+nKcKBtYmZ7eS5k5f3nqsSc8u
# pHSSrds8pJyGH+PBVhsrI/+PteqIe3Br5qC6/To/RabE6BaRUotBwEiES5ZNq0RA
# 443wFSjO7fEYVgcqLxDEDAhkPDOPriiMPMuPiAsNvzv0zh57ju+168u38HcT5uco
# P6wSrqUvImxB+YJcFWbMbA7KxYbD9iYzDAdLoNMHAmpqQDBISzSoUSC7rRuFCOJZ
# DW3KBVAr6kocnqX9oKcfBnTn8tZSkP2vhUgh+Vc7tJwD7YZF9LRhbr9o4iZghurI
# r6n+lB3nYxs6hlZ4TjCCBrwwggSkoAMCAQICEAuuZrxaun+Vh8b56QTjMwQwDQYJ
# KoZIhvcNAQELBQAwYzELMAkGA1UEBhMCVVMxFzAVBgNVBAoTDkRpZ2lDZXJ0LCBJ
# bmMuMTswOQYDVQQDEzJEaWdpQ2VydCBUcnVzdGVkIEc0IFJTQTQwOTYgU0hBMjU2
# IFRpbWVTdGFtcGluZyBDQTAeFw0yNDA5MjYwMDAwMDBaFw0zNTExMjUyMzU5NTla
# MEIxCzAJBgNVBAYTAlVTMREwDwYDVQQKEwhEaWdpQ2VydDEgMB4GA1UEAxMXRGln
# aUNlcnQgVGltZXN0YW1wIDIwMjQwggIiMA0GCSqGSIb3DQEBAQUAA4ICDwAwggIK
# AoICAQC+anOf9pUhq5Ywultt5lmjtej9kR8YxIg7apnjpcH9CjAgQxK+CMR0Rne/
# i+utMeV5bUlYYSuuM4vQngvQepVHVzNLO9RDnEXvPghCaft0djvKKO+hDu6ObS7r
# JcXa/UKvNminKQPTv/1+kBPgHGlP28mgmoCw/xi6FG9+Un1h4eN6zh926SxMe6We
# 2r1Z6VFZj75MU/HNmtsgtFjKfITLutLWUdAoWle+jYZ49+wxGE1/UXjWfISDmHuI
# 5e/6+NfQrxGFSKx+rDdNMsePW6FLrphfYtk/FLihp/feun0eV+pIF496OVh4R1Tv
# jQYpAztJpVIfdNsEvxHofBf1BWkadc+Up0Th8EifkEEWdX4rA/FE1Q0rqViTbLVZ
# Iqi6viEk3RIySho1XyHLIAOJfXG5PEppc3XYeBH7xa6VTZ3rOHNeiYnY+V4j1XbJ
# +Z9dI8ZhqcaDHOoj5KGg4YuiYx3eYm33aebsyF6eD9MF5IDbPgjvwmnAalNEeJPv
# IeoGJXaeBQjIK13SlnzODdLtuThALhGtyconcVuPI8AaiCaiJnfdzUcb3dWnqUnj
# XkRFwLtsVAxFvGqsxUA2Jq/WTjbnNjIUzIs3ITVC6VBKAOlb2u29Vwgfta8b2ypi
# 6n2PzP0nVepsFk8nlcuWfyZLzBaZ0MucEdeBiXL+nUOGhCjl+QIDAQABo4IBizCC
# AYcwDgYDVR0PAQH/BAQDAgeAMAwGA1UdEwEB/wQCMAAwFgYDVR0lAQH/BAwwCgYI
# KwYBBQUHAwgwIAYDVR0gBBkwFzAIBgZngQwBBAIwCwYJYIZIAYb9bAcBMB8GA1Ud
# IwQYMBaAFLoW2W1NhS9zKXaaL3WMaiCPnshvMB0GA1UdDgQWBBSfVywDdw4oFZBm
# pWNe7k+SH3agWzBaBgNVHR8EUzBRME+gTaBLhklodHRwOi8vY3JsMy5kaWdpY2Vy
# dC5jb20vRGlnaUNlcnRUcnVzdGVkRzRSU0E0MDk2U0hBMjU2VGltZVN0YW1waW5n
# Q0EuY3JsMIGQBggrBgEFBQcBAQSBgzCBgDAkBggrBgEFBQcwAYYYaHR0cDovL29j
# c3AuZGlnaWNlcnQuY29tMFgGCCsGAQUFBzAChkxodHRwOi8vY2FjZXJ0cy5kaWdp
# Y2VydC5jb20vRGlnaUNlcnRUcnVzdGVkRzRSU0E0MDk2U0hBMjU2VGltZVN0YW1w
# aW5nQ0EuY3J0MA0GCSqGSIb3DQEBCwUAA4ICAQA9rR4fdplb4ziEEkfZQ5H2Edub
# Tggd0ShPz9Pce4FLJl6reNKLkZd5Y/vEIqFWKt4oKcKz7wZmXa5VgW9B76k9NJxU
# l4JlKwyjUkKhk3aYx7D8vi2mpU1tKlY71AYXB8wTLrQeh83pXnWwwsxc1Mt+FWqz
# 57yFq6laICtKjPICYYf/qgxACHTvypGHrC8k1TqCeHk6u4I/VBQC9VK7iSpU5wlW
# jNlHlFFv/M93748YTeoXU/fFa9hWJQkuzG2+B7+bMDvmgF8VlJt1qQcl7YFUMYgZ
# U1WM6nyw23vT6QSgwX5Pq2m0xQ2V6FJHu8z4LXe/371k5QrN9FQBhLLISZi2yemW
# 0P8ZZfx4zvSWzVXpAb9k4Hpvpi6bUe8iK6WonUSV6yPlMwerwJZP/Gtbu3CKldMn
# n+LmmRTkTXpFIEB06nXZrDwhCGED+8RsWQSIXZpuG4WLFQOhtloDRWGoCwwc6ZpP
# ddOFkM2LlTbMcqFSzm4cd0boGhBq7vkqI1uHRz6Fq1IX7TaRQuR+0BGOzISkcqwX
# u7nMpFu3mgrlgbAW+BzikRVQ3K2YHcGkiKjA4gi4OA/kz1YCsdhIBHXqBzR0/Zd2
# QwQ/l4Gxftt/8wY3grcc/nS//TVkej9nmUYu83BDtccHHXKibMs/yXHhDXNkoPId
# ynhVAku7aRZOwqw6pDCCBwgwggTwoAMCAQICEANaIAYsOltPioQsRFmNpfowDQYJ
# KoZIhvcNAQELBQAwaTELMAkGA1UEBhMCVVMxFzAVBgNVBAoTDkRpZ2lDZXJ0LCBJ
# bmMuMUEwPwYDVQQDEzhEaWdpQ2VydCBUcnVzdGVkIEc0IENvZGUgU2lnbmluZyBS
# U0E0MDk2IFNIQTM4NCAyMDIxIENBMTAeFw0yNTAyMDcwMDAwMDBaFw0yNjAyMDYy
# MzU5NTlaMIGPMQswCQYDVQQGEwJVUzEQMA4GA1UECBMHRmxvcmlkYTEYMBYGA1UE
# BxMPRm9ydCBMYXVkZXJkYWxlMR0wGwYDVQQKExRDaXRyaXggU3lzdGVtcywgSW5j
# LjEWMBQGA1UECxMNQ2l0cml4IFhlbkFwcDEdMBsGA1UEAxMUQ2l0cml4IFN5c3Rl
# bXMsIEluYy4wggGiMA0GCSqGSIb3DQEBAQUAA4IBjwAwggGKAoIBgQCcGbgkxb5b
# a1EgmW3Jokz+rkbpZEA3b3cOVRzvVT92DofNdG1qFk/aS9XqKLfaituB02QLCKD5
# ujRbsAqMEEcomo/3qe5M1OwejP5p4QTo9+BoXMZNGrvv50YMKTkO49sVpckKf2xS
# NV1slMbVhcByWXxlTohquDd46iSnYJx6MxyN4PZ07zLz2jrLZbmxaPaggD5BET2J
# Tmo18MtS1ejmLTXslM36qz1G27APvuy9ehXb3CaQ+Ya5SrrblllgW1FrT+38LvD+
# TTRQ4WPKRu80E+P9xPruL8n9Sby05qe4LdxhTXxVOgyx+lLa3iKtN22sHMe6mv8w
# HYPNlPCvnHabdTOASyeuv1ticIik65b4r+gTG5Oke1myTp+emi145/I/Tn100yCL
# UaPTviTol9okb7p/ApbSLZSd/s0MECsd2gcDp4hgH86K8+3hya6yDNGvY/OICXgu
# kD/1VFydHrwJaOehNOvuqF0N4aoyCTbhw22aD7oEO3rkYNUNfbJy2GkCAwEAAaOC
# AgMwggH/MB8GA1UdIwQYMBaAFGg34Ou2O/hfEYb7/mF7CIhl9E5CMB0GA1UdDgQW
# BBQSlZ4mlT6/0233uxC+VCCbgM3nAzA+BgNVHSAENzA1MDMGBmeBDAEEATApMCcG
# CCsGAQUFBwIBFhtodHRwOi8vd3d3LmRpZ2ljZXJ0LmNvbS9DUFMwDgYDVR0PAQH/
# BAQDAgeAMBMGA1UdJQQMMAoGCCsGAQUFBwMDMIG1BgNVHR8Ega0wgaowU6BRoE+G
# TWh0dHA6Ly9jcmwzLmRpZ2ljZXJ0LmNvbS9EaWdpQ2VydFRydXN0ZWRHNENvZGVT
# aWduaW5nUlNBNDA5NlNIQTM4NDIwMjFDQTEuY3JsMFOgUaBPhk1odHRwOi8vY3Js
# NC5kaWdpY2VydC5jb20vRGlnaUNlcnRUcnVzdGVkRzRDb2RlU2lnbmluZ1JTQTQw
# OTZTSEEzODQyMDIxQ0ExLmNybDCBlAYIKwYBBQUHAQEEgYcwgYQwJAYIKwYBBQUH
# MAGGGGh0dHA6Ly9vY3NwLmRpZ2ljZXJ0LmNvbTBcBggrBgEFBQcwAoZQaHR0cDov
# L2NhY2VydHMuZGlnaWNlcnQuY29tL0RpZ2lDZXJ0VHJ1c3RlZEc0Q29kZVNpZ25p
# bmdSU0E0MDk2U0hBMzg0MjAyMUNBMS5jcnQwCQYDVR0TBAIwADANBgkqhkiG9w0B
# AQsFAAOCAgEAwa7vJhaUhL3q4HCk28xaBGBUVHAPHQcF3EZGLb5e+u0qSig6CXF5
# SX8hjNlvxaEQGz++ScCZmTEGeRtt2AoqrEN4iRPYlfWw0YsojMEEs8+MYx3bDEcI
# BWyxasD3EwRInlnH7mMfKh5U+3hd08Bc0HoJrUkJrrwinwGf5q2qinnA6aFcrznZ
# XY90JZeNCvdiXSTQ9j860MPGdBzR0KWP1NiV8aw7DJU01CH94TEfbCRWVhjGByNm
# fz1KY7gv70AeR+mP/B37+kn4FhJ+UCi0Lp32Xh1+3p/o9noqRsDPnp1WQFfbTHNg
# oogr20J/mxrXCeOQESbbKfUgOforVzBQY7TCGG3MaBy/i89+/luS6YJIPsCq5C/f
# ss/+A5lP2/cKJkCVNtkCg4rXTmoWZrC3sAo8XM0Wr0SNtQdhUF5gWcRMOhr3MvMo
# IJwX/EmyN+ftioUz5vXOrcLVgkrUHr5/dvlpylc0dvIzE9I540NnXfAwBgpqHXsP
# /gZl7vhuxWoWjSgeUHzYGhyvQCeesucu98SKtzQoiifkJDlxyLcuiUETRo2EP6dS
# 0ic8Ak3W5EWm+FIrW5xU79p+ZNBW1SCnlrxrwO2FzRuqHZ6t7iGhj71TTwF1rmCY
# U471hTzERTcOjQBu4k9RVUAsACzQj7fxGNi0h88+39PXOswUe/4IywoxggW6MIIF
# tgIBATB9MGkxCzAJBgNVBAYTAlVTMRcwFQYDVQQKEw5EaWdpQ2VydCwgSW5jLjFB
# MD8GA1UEAxM4RGlnaUNlcnQgVHJ1c3RlZCBHNCBDb2RlIFNpZ25pbmcgUlNBNDA5
# NiBTSEEzODQgMjAyMSBDQTECEANaIAYsOltPioQsRFmNpfowCQYFKw4DAhoFAKBw
# MBAGCisGAQQBgjcCAQwxAjAAMBkGCSqGSIb3DQEJAzEMBgorBgEEAYI3AgEEMBwG
# CisGAQQBgjcCAQsxDjAMBgorBgEEAYI3AgEVMCMGCSqGSIb3DQEJBDEWBBSMB1rS
# VlmLjt0JXH2m+Zp7xNyJFzANBgkqhkiG9w0BAQEFAASCAYBUcLOe0PZ23+wU80aI
# zlHJjiDopkUT8Eplmto90aRFY1p2ln+sggdZ6cNshk8x2qEsslmlyuX49CYHVy9o
# vFMP9VC7NdN4QDY2VZ75NikfzRKirOxu7EsRGviahfiIOqtYuYvYUrm3Vrj3EHT/
# yjRl0X7AffT69KKMOKLODfJAydMhgPk1LyRHyN06xdJo0vOTpQd/mKSJVks3GJPP
# T59Wla0L2crrHsr+XXCOaTJb76R/WErl6VoGCX4O2aAxyVrmnzznkNmyUYly/Qid
# qfMsZqL542yeOm9DhQhI68plY6sVTL+sY4jiCbmMAqdPX4+nBe4M3LaqagNKRXQ6
# Sv/GgeBzKNLG7zEmyaA9W5BK78SA+2DXUyXggVeX1Z+/yIr/xn+UlUctWTvcXx98
# FhjOIr7m+IhXJ0UWPs14LvkEYyQBjfrOw2nCh4lcP0Y+o4nTMQPSag/j1W4VEW21
# ViMjf3rL9wYJ2nwayQgQ3qWpYfS/At2I+C4skia/pXgUCLWhggMgMIIDHAYJKoZI
# hvcNAQkGMYIDDTCCAwkCAQEwdzBjMQswCQYDVQQGEwJVUzEXMBUGA1UEChMORGln
# aUNlcnQsIEluYy4xOzA5BgNVBAMTMkRpZ2lDZXJ0IFRydXN0ZWQgRzQgUlNBNDA5
# NiBTSEEyNTYgVGltZVN0YW1waW5nIENBAhALrma8Wrp/lYfG+ekE4zMEMA0GCWCG
# SAFlAwQCAQUAoGkwGAYJKoZIhvcNAQkDMQsGCSqGSIb3DQEHATAcBgkqhkiG9w0B
# CQUxDxcNMjUwNzA0MDIzNTAwWjAvBgkqhkiG9w0BCQQxIgQgDVlzjHB5ZeCkGzk2
# XsRbWu844pUv05baJqqhh58NYmUwDQYJKoZIhvcNAQEBBQAEggIAc39MtUNcE0tR
# nc5Wx9UDdm1YeOR4HVd9bK41pf9+VBvmbxVETFj1AQq2UxR7wJvdVVaJ1Y7gFvw1
# ZRVfzaFD6xL6wJQCUoedwNbMVgarKYPmtAjdp+aTMqIELpWJ7aYeDQaNkXYd2Sfn
# xpDk7oXJLBvz2QuyXXpZjPRy3Px6Xko6/iyKW5Z59jXBpF+t6OLUrsEwVI2o4NSl
# MoM3fDfGfCAtSPgB2wfJc19eibfl+9w/WDwQt8IHZRHRqgKlMOndb9Hblnt+tDUy
# ia06enyxPpSlikgxl11/BkuseGNKCFMkk87twSAmeAtcW/cKvDHrfR4lOJ6k25ca
# zLpiEnz3mMw9eelXSJIIS8TP5gH8NPPhbbi7urJq9ZqZWHSS+pSLqDylCKJuc/MJ
# gS7vRAJlic8C211KGE5qqIf5y49yJU4hlMXx+3vg+BM6xOpvgO2vSX1xKa4ZnInO
# QzFD62gN6B1jJBQyOvEJkFHRlEEOzB5QNrzRtUiOSvNJBB9JW0/wUOi4bMHZYtnM
# 6tPjBa2P5QLtaF8imgDuYSlLdKRnsl9OgK66/aOJ9knhMHqQadRGVOygI2bgBfNt
# 1YMrsQNo3UwR3LjcokB7/hYc1nc4Po+IITPcvlq8fiRmiinQBn8eti/vl9/d8KD6
# sRYr/RcrlOObAu5GCXzMQ4BqOoeUBds=
# SIG # End signature block
