#
# Copyright Citrix Systems 2011-2025
#
# Script to check Profile Management configuration
# $Revision: #2504 $
#

<#
.SYNOPSIS
This script analyses a machine running Profile Management and inspects the environment to generate configuration recommendations.
The environment data can be exported to a CSV file.
.PARAMETER ProfileDriveThresholdPercent
 ProfileDriveThresholdPercent sets a disk threshold for the Profile Volume.  The script warns if the free disk on the Profile Volume is below the stated percentage.
.PARAMETER WriteCsvFiles
 WriteCsvFiles causes the script to write fixed-named data files in CSV format, to assist with offline analysis.
.PARAMETER iniFilePath
 iniFilePath configures an alternate file path for the INI file.  There are two use-cases - debugging the script is one, but the parameter could also be used for testing an INI file, prior to deploying it in a live environment
.DESCRIPTION
The script inspects the Profile Management Service status and configuration, the machine environment and information on other installed Citrix products, such as PVD, VDI-in-a-Box, XenApp, XenDesktop, XenServer, as well as third-party products, including hypervisors.  The script predominatly obtains its information through WMI queries, and by examining the registry.

The script recommends changes to the Profile Management configuration, where the environment corresponds to one that has been studied by the Profile Management team.  

However, this is at best "good faith" advice, and your next step should always to review the advice alongside other Citrix and trusted third-party documentation.  Bear in mind that usage patterns, as well as other factors that the script cannot detect, may invalidate the recommendations of the script.

When WriteCsvFiles is specified, three CSV files will be generated:
-  UPMPolicySummary.csv contains information on which Profile Management policies have been tested, whether the policies were detected in an INI file, Group Policy (FullArmor) or defaulted.
-  UPMEnvironmentSummary.csv contains summary information an the environment, including Profile Management itself, Windows, Hypervisors, XenApp, XenDesktop, PVD.
-  UPMListPolicySummary.csv contains the information about policies which are configured using lists of strings.

The script is designed to be run from a powershell command line logged-on as a domain user.  Some operations may not be available without elevated privilege, and the script will advise in such cases.

Supported Platforms: Windows XP, Windows Server 2003, Windows Vista, Windows 7, Windows Server 2008 R2, Windows 8/8.1, Windows Server 2012 R2.  Support for this script on a platform does not imply that Citrix supports any specific features, products or combination of products on that platform, including but not limited to Profile Management (Citrix Profile Management), XenApp, XenDesktop, Xen Server, Personal vDisk and/or VDI-in-a-Box.

Errors detected: None.
.LINK
The latest version of the script can be downloaded from citrix.com:
Download: http://support.citrix.com/article/CTX132805
Blog:     http://blogs.citrix.com/2012/09/18/did-i-get-it-right-the-profile-management-configuration-check-tool-upmconfigcheck/
#>


param (
  [int]$ProfileDriveThresholdPercent = 15,

  [Parameter(Mandatory=$false)]
  [string]$OutputXmlPath,

  [Parameter(Mandatory=$false)]
  [string]$OutputPath = $env:Temp,

  [Parameter(Mandatory=$false)]  
  [switch]$WriteCsvFiles,

  [Parameter(Mandatory=$false)]
  [string]$iniFilePath="",

  [Parameter(Mandatory=$false)]
  [string]$ProgressFilePath,
  
  [bool]$CancelWriteHost=$true,

  [Parameter(Mandatory=$false)]
  [string]$CEIPFilePath,

  [Parameter(Mandatory=$false)]
  [bool]$CheckNewTeams=$true,

  [Parameter(Mandatory=$false)]
  [bool]$ProfileSumCheck=$true
)

$ChaOutputXmlPath = ""
if ($CancelWriteHost){
  $DynWriteHost =
    "
      function global:Write-Host(){    
      }
    "
  Invoke-Expression $DynWriteHost
}

# PowerShell v2.0 compatible version of [string]::IsNullOrWhitespace.
function StringIsNullOrWhitespace([string] $string)
{
    if ($string -ne $null) { $string = $string.Trim() }
    return [string]::IsNullOrEmpty($string)
}

if (StringIsNullOrWhitespace($ProgressFilePath))
{
  $IsChaCall = $true
}

$StartTime = Get-Date

if (StringIsNullOrWhitespace($OutputXmlPath)){
  $ChaOutputXmlPath = [System.IO.Path]::GetTempPath()+ "\UpmConfigCheckToolOutput.xml"
}


if (StringIsNullOrWhitespace($OutputPath)){
  $OutputXmlPath = (Get-Item -Path ".\").FullName + '\' + 'UPMConfigCheckOutput.xml'
  $ChaOutputXmlPath = (Get-Item -Path ".\").FullName + '\' + 'UpmConfigCheckToolOutput.xml'
  $OutputJsonPath = (Get-Item -Path ".\").FullName + '\' + 'UpmConfigCheckOutput.json'
}
else{
  $OutputXmlPath = $OutputPath + '\' + 'UPMConfigCheckOutput.xml'
  $ChaOutputXmlPath = $OutputPath + "\UpmConfigCheckToolOutput.xml"
  $OutputJsonPath =  $OutputPath + '\UpmConfigCheckOutput.json'
}

if (-not ([System.Management.Automation.PSTypeName]'UpmConfigCheckOutputInfo').Type)
{

  Add-Type @'
  public struct UpmConfigCheckOutputInfo{
    public string CheckCategory;
    public string CheckTitle;
    public string Info;    
    public string Type;
    public string Reason;
    public string PolicyName;    
  }
'@
}

if (-not ([System.Management.Automation.PSTypeName]'UpmConfigCheckCHAOutputInfo').Type)
{
  Add-Type @'
  public struct UpmConfigCheckCHAOutputInfo{
      public int CheckId;
      public string CheckTitle;        
      public string CheckResult;
      public string CheckOutput;    
      public string PolicyName;
      public string KBlinks;    
  }
'@
}

if (-not ([System.Management.Automation.PSTypeName]'LogonTimings').Type)
{
  Add-Type @'
  public struct LogonTimings{
    public string SessionId;
    public string DesktopReady;
    public string GroupPolicyStart;
    public string GroupPolicyComplete;
    public string LogonScriptsStart;
    public string LogonScriptsComplete;
    public string ProfileLoadStart;
    public string ProfileLoaded;
    public string UPMStart;
    public string UserInitStart;
    public string UserInitComplete;
  }
'@
}

class EventLog {
  [string]$eventLogName
  [string]$eventLogDisplayName
  [string]$evtxPath
  [string]$regPath
}

$errorInfoList = New-Object System.Collections.ArrayList
$warningInfoList = New-Object System.Collections.ArrayList
$invalidInfoList = New-Object System.Collections.ArrayList
$noticeInfoList = New-Object System.Collections.ArrayList

$CHACheckInfoList = New-Object System.Collections.ArrayList

$copyright = "Citrix Systems 2011-2025"
$upmCheckVersion = 'Profile Management Configuration checking tool version $Revision: #2504 $'
$scriptRunDate = date
"Run at " + $scriptRunDate.DateTime


############################################################################
#
# ... start of the function definitions + associated data structures
#
############################################################################

$appVDefaultExcludedFolder      = @('!ctx_localappdata!\Microsoft\AppV','!ctx_roamingappdata!\Microsoft\AppV\Client\Catalog')
$groupPolicyExclusionFolder = '!ctx_localappdata!\GroupPolicy'
$shareFileDefaultExcludedFolder = 'sharefile'
$win8DefaultExcludedFolderList = @('!ctx_localappdata!\Packages','!ctx_localappdata!\Microsoft\Windows\Application Shortcuts')

$recommendedSyncExclusionListDir = @(
'$Recycle.Bin',
'AppData\LocalLow',
$groupPolicyExclusionFolder,
'!ctx_internetcache!',
$appVDefaultExcludedFolder,
'!ctx_localappdata!\Microsoft\Windows\Burn',
'!ctx_localappdata!\Microsoft\Windows\CD Burning',
'!ctx_localappdata!\Microsoft\Windows Live',
'!ctx_localappdata!\Microsoft\Windows Live Contacts',
'!ctx_localappdata!\Microsoft\Terminal Server Client',
'!ctx_localappdata!\Microsoft\Messenger',
'!ctx_localappdata!\Microsoft\OneNote',
'!ctx_localappdata!\Microsoft\Outlook',
'!ctx_localappdata!\Windows Live',
'!ctx_localappdata!\Sun',
'!ctx_localsettings!\Temp',
'!ctx_roamingappdata!\Sun\Java\Deployment\cache',
'!ctx_roamingappdata!\Sun\Java\Deployment\log',
'!ctx_roamingappdata!\Sun\Java\Deployment\tmp',
'!ctx_localappdata!\Google\Chrome\User Data\Default\Cache',
'!ctx_localappdata!\Google\Chrome\User Data\Default\Cached Theme Images',
'!ctx_startmenu!'
)
$recommendedStreamExclusionList =@(
'!ctx_localappdata!\Microsoft\Credentials',
'!ctx_roamingappdata!\Microsoft\Credentials',
'!ctx_roamingappdata!\Microsoft\Crypto',
'!ctx_roamingappdata!\Microsoft\Protect',
'!ctx_roamingappdata!\Microsoft\SystemCertificates'
)
$recommendedSyncExclusionListReg =@(
'Software\Microsoft\AppV\Client\Integration',
'Software\Microsoft\AppV\Client\Publishing'
)

$reg2wemConsoleNameTable = @{
  ServiceActive = "Enable Profile Management";
  ProcessAdmins = "Process logons of local administrators";
  PSMidSessionWriteBack = "Enable active write back";
  OfflineSupport = "Enable offline profile support";
  PSEnabled = "Enable profile streaming";
  PSForFoldersEnabled = "Enable profile streaming for folders";
  ProcessCookieFiles = "Process Internet cookie files on logoff";
  CEIPEnabled = "Join the Citrix Customer Experience Improvement Program";
  OutlookSearchRoamingEnabled = "Enable search index roaming for Microsoft Outlook users";
  DeleteCachedProfilesOnLogoff = "Delete locally cached profiles on logoff";
  ProfileDeleteDelay = "Delay before deleting cached profiles";
  LastKnownGoodRegistry = "NTUSER.DAT backup";
  SyncGpoStateEnabled = "Enable asynchronous processing for user Group Policy on logon";
  OutlookEdbBackupEnabled = "Outlook search index database - backup and restore";
  OutlookSearchRoamingConcurrentSessionEnabled = "Enable concurrent session support";
  DisableConcurrentAccessToOneDriveContainer = "Enable exclusive access to OneDrive container";
  AccelerateFolderMirroring = "Accelerate folder mirroring";
}

$newTeamsAppDataFolderList = @(
    "$env:APPDATA", 
    "$env:APPDATA\Microsoft", 
    "$env:APPDATA\Microsoft\Crypto", 
    "$env:APPDATA\Microsoft\Internet Explorer", 
    "$env:APPDATA\Microsoft\Internet Explorer\UserData", 
    "$env:APPDATA\Microsoft\Internet Explorer\UserData\Low",
    "$env:APPDATA\Microsoft\Spelling", 
    "$env:APPDATA\Microsoft\SystemCertificates",
    "$env:APPDATA\Microsoft\Windows", 
    "$env:APPDATA\Microsoft\Windows\Libraries",
    "$env:APPDATA\Microsoft\Windows\Recent",
    "$env:LOCALAPPDATA",
    "$env:LOCALAPPDATA\Microsoft",
    "$env:LOCALAPPDATA\Microsoft\Windows",
    "$env:LOCALAPPDATA\Microsoft\Windows\Explorer",
    "$env:LOCALAPPDATA\Microsoft\Windows\History",
    "$env:LOCALAPPDATA\Microsoft\Windows\History\Low",
    "$env:LOCALAPPDATA\Microsoft\Windows\History\Low\History.IE5",
    "$env:LOCALAPPDATA\Microsoft\Windows\IECompatCache",
    "$env:LOCALAPPDATA\Microsoft\Windows\IECompatCache\Low",
    "$env:LOCALAPPDATA\Microsoft\Windows\IECompatUaCache",
    "$env:LOCALAPPDATA\Microsoft\Windows\IECompatUaCache\Low",
    "$env:LOCALAPPDATA\Microsoft\Windows\INetCache",
    "$env:LOCALAPPDATA\Microsoft\Windows\INetCookies",
    "$env:LOCALAPPDATA\Microsoft\Windows\INetCookies\DNTException",
    "$env:LOCALAPPDATA\Microsoft\Windows\INetCookies\DNTException\Low",
    "$env:LOCALAPPDATA\Microsoft\Windows\INetCookies\Low",
    "$env:LOCALAPPDATA\Microsoft\Windows\INetCookies\PrivacIE",
    "$env:LOCALAPPDATA\Microsoft\Windows\INetCookies\PrivacIE\Low",
    "$env:LOCALAPPDATA\Microsoft\Windows\PPBCompatCache",
    "$env:LOCALAPPDATA\Microsoft\Windows\PPBCompatCache\Low",
    "$env:LOCALAPPDATA\Microsoft\Windows\PPBCompatUaCache",
    "$env:LOCALAPPDATA\Microsoft\Windows\PPBCompatUaCache\Low",
    "$env:LOCALAPPDATA\Microsoft\WindowsApps",
    "$env:LOCALAPPDATA\Packages",
    "$env:LOCALAPPDATA\Publishers",
    "$env:LOCALAPPDATA\Publishers\8wekyb3d8bbwe",
    "$env:LOCALAPPDATA\Temp",
    "$env:USERPROFILE\AppData\LocalLow",
    "$env:USERPROFILE\AppData\LocalLow\Microsoft",
    "$env:USERPROFILE\AppData\LocalLow\Microsoft\Internet Explorer",
    "$env:USERPROFILE\AppData\LocalLow\Microsoft\Internet Explorer\DOMStore",
    "$env:USERPROFILE\AppData\LocalLow\Microsoft\Internet Explorer\EdpDomStore",
    "$env:USERPROFILE\AppData\LocalLow\Microsoft\Internet Explorer\EmieSiteList",
    "$env:USERPROFILE\AppData\LocalLow\Microsoft\Internet Explorer\EmieUserList",
    "$env:USERPROFILE\AppData\LocalLow\Microsoft\Internet Explorer\IEFlipAheadCache"
)

# Get the current Windows version from the registry
$windowsVersion = (Get-ItemProperty -Path "HKLM:\SOFTWARE\Microsoft\Windows NT\CurrentVersion" -Name ReleaseId).ReleaseId

# Convert the version to an integer for comparison
$windowsVersionInt = [int]$windowsVersion

function addRecommendedFolderExclusions ($Folder) {
  $FolderList = @($Folder)
  for ($ix = 0; $ix -lt $FolderList.Length; $ix++) {
    $f = $FolderList[$ix]
    $script:recommendedSyncExclusionListDir = $script:recommendedSyncExclusionListDir + $f
    $script:recommendedSyncExclusionListDir = @($script:recommendedSyncExclusionListDir)  #force it to be an array
  }
}

filter v1SubstituteNames {
  begin {
    }
  process {
      $_ `
           -replace "!ctx_internetcache!",  "Local Settings\Temporary Internet Files" `
           -replace "!ctx_localappdata!",   "Local Settings\Application Data" `
           -replace "!ctx_localsettings!",  "Local Settings" `
           -replace "!ctx_roamingappdata!", "Application Data" `
           -replace "!ctx_startmenu!",      "Start Menu" 
    }
  end {
    }
}

filter v1RemoveNames {
  begin {
    }
  process {
      $line = $_
      switch -regex ($line) {
        "^Local Settings" {}
        "^Application Data" {}
        "^Start Menu" {}
        default { $line }
      }
    }
  end {
    }
}

function v1CompatibleNames ($stringArray) {
  $hasNames = $false
  switch -regex ($stringArray) {
    "^Local Settings" { $hasNames = $true }
    "^Application Data" { $hasNames = $true }
    "^Start Menu" { $hasNames = $true }
    default {}
  }
  $hasNames
}

function neutralNames ($stringArray) {
  $hasNames = $false
  switch -regex ($stringArray) {
    "^\!CTX_[A-Z]+\!" { $hasNames = $true }
    default {}
  }
  $hasNames
}

filter v2SubstituteNames {
  begin {
    }
  process {
      $_ `
           -replace "!ctx_internetcache!",  "AppData\Local\Microsoft\Windows\Temporary Internet Files" `
           -replace "!ctx_localappdata!",   "AppData\Local" `
           -replace "!ctx_localsettings!",  "AppData\Local" `
           -replace "!ctx_roamingappdata!", "AppData\Roaming" `
           -replace "!ctx_startmenu!",      "Appdata\Roaming\Microsoft\Windows\Start Menu" 
    }
  end {
    }
}

filter v2RemoveNames {
  begin {
    }
  process {
      $line = $_
      switch -regex ($line) {
        "^AppData" {}
        default { $line }
      }
    }
  end {
    }
}

function v2CompatibleNames ($stringArray) {
  $hasNames = $false
  switch -regex ($stringArray) {
    "^AppData" { $hasNames = $true }
    default {}
  }
  $hasNames
}

filter ReportDifferences ($type) {
  begin {
    }
  process {
      $item = $_.InputObject
      $indicator = $_.SideIndicator
      $t = $null
      switch ($indicator) {
        "=>" { $t = "Added" }
        "<=" { $t = "Missing" }
        "==" { $t = "Same" }
      }
      if ($t -ne $null) {
        New-Object Object |
          Add-Member NoteProperty ComparisonType $type -PassThru |
          Add-Member NoteProperty Difference     $t    -PassThru |
          Add-Member NoteProperty LineItem       $item -PassThru 
      }
    }
  end {
    }
}

function CompareLists ($preferredList, $specimenList) {
  #
  # force both lists to be arrays - this helps cookie / mirrored folder processing
  $preferredList = @($preferredList)
  $specimenList = @($specimenList)
  #
  # the specimenList just needs to be sorted once, into alphabetical order
  # note that duplicates are removed here, because duplicate testing is
  # performed elsewhere
  #
  $processedSpecimen = $specimenList | sort-object -Unique
  if (neutralNames($processedSpecimen)) {
    #
    # if the speciment array contains neutral names, then we sort both 
    # the preferred and specimen arrays alphabetically, removing duplicates
    # and compare them without processing to either v1 or v2 names
    #
    $processedPreferred = $preferredList | sort-object -Unique
    write-host "Compare as Neutral List"
    Compare-Object -ReferenceObject $processedPreferred -DifferenceObject $processedSpecimen -IncludeEqual | ReportDifferences -type "Neutral"
  } else {
    #
    # the specimen contains no neutral names, so we process the list once 
    # for v1 names and once for v2 names
    # It is possible for both v1 and v2 names to be present in
    # an exclusion list - some customers share the list for both
    # XP and Win7 machines in a single OU
    #
    # Remove any V2 profile names from the list and if not empty,
    # test against the V1 version of the list
    #
    $processedV1List = $processedSpecimen | v2RemoveNames
    if (v1CompatibleNames($processedV1List)) {
      $processedPreferred = $preferredList | v1SubstituteNames | sort-object      
      Compare-Object -ReferenceObject $processedPreferred -DifferenceObject $processedV1List -IncludeEqual | ReportDifferences -type "V1"
    }
    #
    # Remove any V1 profile names from the list and if not empty,
    # test against the V2 version of the list
    #
    $processedV2List = $processedSpecimen | v1RemoveNames
    if (v2CompatibleNames($processedV2List)) {
      $processedPreferred = $preferredList | v2SubstituteNames | sort-object
      Compare-Object -ReferenceObject $processedPreferred -DifferenceObject $processedV2List -IncludeEqual | ReportDifferences -type "V2"
    }

    $processedPreferred = $preferredList | v2SubstituteNames | sort-object
    Compare-Object -ReferenceObject $processedPreferred -DifferenceObject $processedV2List -IncludeEqual | ReportDifferences -type "V2"
  }
}

function Get-ScriptDirectory
{
  $Invocation = (Get-Variable MyInvocation -Scope 1).Value
  Split-Path $Invocation.MyCommand.Path
}

$sdir = Get-ScriptDirectory

#
# standard files are created relative to the folder where we place the script
#
$csvSinglePolicySummaryFile    = $sdir + "\UPMPolicySummary.csv"
$csvListPolicySummaryFile      = $sdir + "\UPMListPolicySummary.csv"
$csvEnvironmentSummaryFile     = $sdir + "\UPMEnvironmentSummary.csv"

#
# customise colours used by Write-Host
#
$errorColours = @{foreground="red";background="black"}
$warnColours = @{foreground="magenta";background="blue"}
$infoColours = @{foreground="cyan";background="blue"}
$hilight1 = @{foreground="red";background="blue"}
$hilight2 = @{foreground="yellow";background="blue"}

$physical = "Physical"   # constant denoting physical (non-hypervisor) and non-provisioned environments

#
# used to build up list of recommendations
#
$errStrings = @()
$warningStrings = @()
$noticeStrings = @()
#
# Functions for maintaining a "database" of policies
#

$policyList = @(
  #  PolicyName                               Default Value
  @("DeleteCachedProfilesOnLogoff",           0),
  @("DisableDynamicConfig",                   ""),
  @("ExcludedGroups",                         ""),
  @("ExclusionListRegistry",                  ""),
  @("InclusionListRegistry",                  ""),
  @("LoadRetries",                            5),
  @("LocalProfileConflictHandling",           1),
  @("LogLevelActiveDirectoryActions",         0),
  @("LogLevelFileSystemActions",              0),
  @("LogLevelFileSystemNotification",         0),
  @("LogLevelInformation",                    1),
  @("LogLevelLogoff",                         1),
  @("LogLevelLogon",                          1),
  @("LogLevelPolicyUserLogon",                0),
  @("LogLevelRegistryActions",                0),
  @("LogLevelRegistryDifference",             0),
  @("LogLevelUserName",                       1),
  @("LogLevelWarnings",                       1),
  @("LoggingEnabled",                         0),
  @("LogoffRatherThanTempProfile",            ""),
  @("MaxLogSize",                             1048576),
  @("MigrateWindowsProfilesToUserStore",      1),
  @("MirrorFoldersList",                      ""),
  @("OfflineSupport",                         0),
  @("PSAlwaysCache",                          0),
  @("PSAlwaysCacheSize",                      0),
  @("PSEnabled",                              0),
  @("PSMidSessionWriteBack",                  0),
  @("PSPendingLockTimeout",                   1),
  @("PSUserGroupsList",                       ""),
  @("PathToLogFile",                          ""),
  @("PathToUserStore",                        "Windows"),
  @("ProcessAdmins",                          0),
  @("ProcessCookieFiles",                     0),
  @("ProcessedGroups",                        ""),
  @("ProfileDeleteDelay",                     ""),
  @("ServiceActive",                          0),
  @("SyncDirList",                            ""),
  @("SyncExclusionListDir",                   ""),
  @("SyncExclusionListFiles",                 ""),
  @("SyncFileList",                           ""),
  @("TemplateProfileIsMandatory",             ""),
  @("TemplateProfileOverridesLocalProfile",   ""),
  @("TemplateProfileOverridesRoamingProfile", ""),
  @("TemplateProfilePath",                    ""),  
  @("PSMidSessionWriteBackReg",               0),
  @("LastKnownGoodRegistry",                  0),
  @("DefaultExclusionListRegistry",           ""),
  @("DefaultSyncExclusionListDir",            ""),
  @("StreamingExclusionList",                 ""),
  @("CEIPEnabled",                            1),
  @("LogonExclusionCheck",                    0),
  @("XenAppOptimization",                     0),
  @("LargeFileHandling",                      0),
  @("OutlookSearchRoamingEnabled",            0)
)

#############################################################
# functions and datastructures for reporting on policy
# lists, such as inclusion and exclusion
#
$policyListDb = @()

$realPathForDefaultExclusionReg = @{      
    "ExclusionDefaultRegistry01" = "Software\Microsoft\AppV\Client\Integration";
    "ExclusionDefaultRegistry02" = "Software\Microsoft\AppV\Client\Publishing";
}
$realPathForDefaultExclusionDir = @{
    "ExclusionDefaultDir01" = "!ctx_internetcache!";
    "ExclusionDefaultDir02" = "!ctx_localappdata!\\Google\\Chrome\\User Data\\Default\\Cache";
    "ExclusionDefaultDir03" = "!ctx_localappdata!\Google\Chrome\User Data\Default\Cached Theme Images";
    "ExclusionDefaultDir04" = "!ctx_localappdata!\Google\Chrome\User Data\Default\JumpListIcons";
    "ExclusionDefaultDir05" = "!ctx_localappdata!\Google\Chrome\User Data\Default\JumpListIconsOld";
    "ExclusionDefaultDir06" = "!ctx_localappdata!\GroupPolicy";
    "ExclusionDefaultDir07" = "!ctx_localappdata!\Microsoft\AppV";
    "ExclusionDefaultDir08" = "!ctx_localappdata!\Microsoft\Messenger";
    "ExclusionDefaultDir09" = "!ctx_localappdata!\Microsoft\Office\15.0\Lync\Tracing";
    "ExclusionDefaultDir10" = "!ctx_localappdata!\Microsoft\OneNote";
    "ExclusionDefaultDir11" = "!ctx_localappdata!\Microsoft\Outlook";
    "ExclusionDefaultDir12" = "!ctx_localappdata!\Microsoft\Terminal Server Client";
    "ExclusionDefaultDir13" = "!ctx_localappdata!\Microsoft\UEV";
    "ExclusionDefaultDir14" = "!ctx_localappdata!\Microsoft\Windows Live";
    "ExclusionDefaultDir15" = "!ctx_localappdata!\Microsoft\Windows Live Contacts";
    "ExclusionDefaultDir16" = "!ctx_localappdata!\Microsoft\Windows\Application Shortcuts";
    "ExclusionDefaultDir17" = "!ctx_localappdata!\Microsoft\Windows\Burn";
    "ExclusionDefaultDir18" = "!ctx_localappdata!\Microsoft\Windows\CD Burning";
    "ExclusionDefaultDir19" = "!ctx_localappdata!\Microsoft\Windows\Notifications";
    "ExclusionDefaultDir20" = "!ctx_localappdata!\Packages";
    "ExclusionDefaultDir21" = "!ctx_localappdata!\Sun";
    "ExclusionDefaultDir22" = "!ctx_localappdata!\Windows Live";
    "ExclusionDefaultDir23" = "!ctx_localsettings!\Temp";
    "ExclusionDefaultDir24" = "!ctx_roamingappdata!\Microsoft\AppV\Client\Catalog";
    "ExclusionDefaultDir25" = "!ctx_roamingappdata!\Sun\Java\Deployment\cache";
    "ExclusionDefaultDir26" = "!ctx_roamingappdata!\Sun\Java\Deployment\log";
    "ExclusionDefaultDir27" = "!ctx_roamingappdata!\Sun\Java\Deployment\tmp";
    "ExclusionDefaultDir28" = "$Recycle.Bin";
    "ExclusionDefaultDir29" = "AppData\LocalLow";
    "ExclusionDefaultDir30" = "Tracing"
}

function Add-PolicyListRecord($PolicyName,$ProfileType,$DifferenceType,$Value,$Advice="",$Notes="",$Origin="") {
  $offset = $script:policyListDb.Length
  $policyObj = New-Object Object |
      Add-Member NoteProperty Name                  $PolicyName              -PassThru |
      Add-Member NoteProperty ProfileType           $ProfileType             -PassThru |
      Add-Member NoteProperty MatchesBestPractice   $DifferenceType          -PassThru |
      Add-Member NoteProperty Value                 $Value                   -PassThru |
      Add-Member NoteProperty Advice                $Advice                  -PassThru |
      Add-Member NoteProperty Notes                 $Notes                   -PassThru |
      Add-Member NoteProperty Origin                $Origin                  -PassThru 
  $script:policyListDb = $script:policyListDb + $policyObj
}


#############################################################
# functions and datastructures for reporting on policy
#
$policyDbIndex = @{}
$policyDb = @()

function Add-PolicyRecord($PolicyName,$Default) {
  $offset = $script:policyDb.Length
  $policyObj = New-Object Object |
      Add-Member NoteProperty Name             $PolicyName              -PassThru |
      Add-Member NoteProperty DefaultValue     $Default                 -PassThru |
      Add-Member NoteProperty PreferredValue   ""                       -PassThru |
      Add-Member NoteProperty EffectiveValue   ""                       -PassThru |
      Add-Member NoteProperty Advice           ""                       -PassThru |
      Add-Member NoteProperty Notes            ""                       -PassThru |
      Add-Member NoteProperty Origin           "Not Checked"            -PassThru 
  $script:policyDb = $script:policyDb + $policyObj
  $policyDbIndex[$PolicyName] = $offset
}

for ($ix = 0; $ix -lt $policyList.Length; $ix++) {
  $pn = $policyList[$ix][0]
  $dv = $policyList[$ix][1]
  Add-PolicyRecord -PolicyName $pn -Default $dv
}

function Replace-PolicyRecordProperty ([string]$PolicyName, [string]$PropertyName, $NewValue) {
  $offset = $policyDbIndex[$PolicyName]
  if ($offset -ne $null) {
    $obj = $script:policyDb[$offset]
    if (($null -ne $NewValue) -and ($NewValue.GetType().BaseType.Name -eq "Array")) {
      $sub = $NewValue -Join "`n"
    } else {
      $sub = $NewValue
    }
    $newobj = $obj | Add-Member NoteProperty $PropertyName -Value $sub -Force -PassThru
    # use Select-Object to force the same order for fields
    $newobj2 = $newobj | Select-Object -Property Name,DefaultValue,PreferredValue,EffectiveValue,Advice,Notes,Origin 
    $script:policyDb[$offset] = $newobj2
  }
}

function Get-PolicyRecordProperty ([string]$PolicyName, [string]$PropertyName) {
  $offset = $policyDbIndex[$PolicyName]
  if ($offset -ne $null) {
    $obj = $script:policyDb[$offset]
    $obj.$PropertyName
  }
}

#
# filter which captures recommendations in the "errStrings" array, but also passes them on
#
filter CaptureRecommendations ([string]$CheckTitle, [string]$PolicyName, $Reason = $null, $Category = $null, $InfoType = "Error", $KBLinks = $null) {
  #
  # print the recommendation, but also capture it to the $errStrings array
  #
  begin {
  }
  process {
    $errInfo = $_
    write-host @warnColours $errInfo
    Replace-PolicyRecordProperty -PolicyName $PolicyName -PropertyName Advice -NewValue $errInfo
    <#$infoObj = New-Object Object |
      Add-Member NoteProperty PolicyName    $PolicyName         -PassThru |
      Add-Member NoteProperty Info          $errInfo            -PassThru |
      Add-Member NoteProperty InfoType      $InfoType           -PassThru |
      Add-Member NoteProperty Reason        $Reason             -PassThru #>
    $infoObj = New-Object UpmConfigCheckOutputInfo
    $infoObj.CheckTitle = $CheckTitle
    $infoObj.PolicyName = $PolicyName
    $infoObj.Info = $errInfo
    $infoObj.Reason = $Reason
    $infoObj.Type = $InfoType
    $infoObj.CheckCategory = $Category
    if ($InfoType -eq "Error") {
      $script:errorInfoList.Add($infoObj) > $null
      $script:errStrings += $errInfo
      $script:errStrings = @($script:errStrings)  # force it to be an array        
    }
    else {
      if ($InfoType -eq "Warning") {  
        $script:warningInfoList.Add($infoObj) > $null
        $script:warningStrings += $errInfo
        $script:warningStrings = @($script:warningStrings)  # force it to be an array        
      }
      elseif ($InfoType -eq "Notice") {
        $script:noticeInfoList.Add($infoObj) > $null
        #Still treat it as error in console output
        $script:noticeStrings += $errInfo
        $script:noticeStrings = @($script:noticeStrings)  # force it to be an array
      }
      elseif ($InfoType -eq "Invalid") {
        $script:invalidInfoList.Add($infoObj) > $null
        #Still treat it as error in console output
        $script:errStrings += $errInfo
        $script:errStrings = @($script:errStrings)  # force it to be an array
      }
    }
    
    if ($null -ne $Reason) {
      if ($InfoType -eq "Error") {
        $script:errStrings += "    Notes: $Reason"
      }
      elseif ($InfoType -eq "Warning") {
        $script:warningStrings += "    Notes: $Reason"
      }
      else {
        $script:noticeStrings += "    Notes: $Reason"
      }
    }

    $errInfo | CaptureCheckResult $CheckTitle $PolicyName $Reason $InfoType $KBLinks    
  }
  end {
  }
}

filter CaptureCheckResult ([string]$CheckTitle, [string]$PolicyName, $Reason=$null, $InfoType="Error", $KBLinks = $null) {
  #
  # print the recommendation, but also capture it to the $errStrings array
  #
  begin{
  }
  process {
    $errInfo = $_    
    Replace-PolicyRecordProperty -PolicyName $PolicyName -PropertyName Advice -NewValue $errInfo   
    $infoObj = New-Object UpmConfigCheckCHAOutputInfo    
    <#
     public string CheckTitle;        
    public string CheckResult;
    public string CheckOutput;    
    public string PolicyName;
    public string KBLinks;
    #>
    $infoObj.CheckTitle  = $CheckTitle
    $infoObj.PolicyName  = $PolicyName
    $infoObj.CheckOutput = $errInfo
    $infoObj.KBLinks = $KBLinks
    if ($InfoType -eq "Error"){
        $infoObj.CheckResult = "Failed"
    }else {
        if ($InfoType -eq "Warning"){
            $infoObj.CheckResult = "Passed with warning"
        }else
        {
            Write-Host -ForegroundColor Green $errInfo
            $infoObj.CheckResult = "Passed"
        }
    }   
    
    if ($null -ne $Reason) {
      $infoObj.CheckOutput += " Reason: $Reason"
    }
    
    $script:CHACheckInfoList.Add($infoObj) > $null
        
    if (!(StringIsNullOrWhitespace($ProgressFilePath))){
        $fs = [IO.File]::Open($ProgressFilePath, [IO.FileMode]::OpenOrCreate, [IO.FileAccess]::Write, [IO.FileShare]::ReadWrite)
        $enc = [system.Text.Encoding]::UTF8
        $data = $enc.GetBytes($script:CHACheckInfoList.Count)
        $fs.Write($data, 0, $data.length)
        $fs.close()
    }
  }
  end{
  }
}


function Convert-InfoToXml($xmlDoc,$infoList, $infoType, $infoListNode)
{
    #Convert info to XML like:
    # <$infoType id="1">
    #   <Title></Title>
    #   <Info></Info>
    #   <Reason></Reason>
    #   <PolicyName></PolicyName>
    # </$infoType>
    #
    #
    #
    $id = 0
    foreach ($info in $infoList) {
        #InfoType node
        $id += 1        
        $infoTypeNode = $infoListNode.AppendChild($xmlDoc.CreateElement($infoType))
        $infoTypeNode.SetAttribute("Id", $id)
        #Title node
        $infoTitleNode = $infoTypeNode.AppendChild($xmlDoc.CreateElement("Title"))
        $infoTitleNode.AppendChild($xmlDoc.CreateTextNode($info.CheckTitle)) | out-null
        #Info node
        $infoContentNode = $infoTypeNode.AppendChild($xmlDoc.CreateElement("Info"))
        $infoContentNode.AppendChild($xmlDoc.CreateTextNode($info.Info)) | out-null
        #Reason node
        $reasonNode = $infoTypeNode.AppendChild($xmlDoc.CreateElement("Reason"))
        $reasonNode.AppendChild($xmlDoc.CreateTextNode($info.Reason)) | out-null
        #PolicyName node
        $policyNameNode = $infoTypeNode.AppendChild($xmlDoc.CreateElement("PolicyName"))
        $policyNameNode.AppendChild($xmlDoc.CreateTextNode($info.PolicyName)) | out-null
    }    
}

function Convert-CheckInfoToXml($xmlDoc,$infoList, $infoListNode){
  #public string CheckTitle;        
  #public string CheckResult;
  #public string CheckOutput;    
  #public string PolicyName;
  #Convert info to XML like:
  # <CheckItem id="1">
  #   <Title></Title>
  #   <Result></Result>
  #   <Output></Output>
  #   <PolicyName></PolicyName>
  # </CheckItem>
  #
  #
  #
  $id = 0
  foreach ($info in $infoList) {
      #InfoType node
      $id += 1        
      $infoTypeNode = $infoListNode.AppendChild($xmlDoc.CreateElement("CheckItem"))
      $infoTypeNode.SetAttribute("Id", $id)
      #Title node
      $infoTitleNode = $infoTypeNode.AppendChild($xmlDoc.CreateElement("Title"))
      $infoTitleNode.AppendChild($xmlDoc.CreateTextNode($info.CheckTitle)) | out-null
      #Result node
      $infoContentNode = $infoTypeNode.AppendChild($xmlDoc.CreateElement("Result"))
      $infoContentNode.AppendChild($xmlDoc.CreateTextNode($info.CheckResult)) | out-null
      #Output node
      $reasonNode = $infoTypeNode.AppendChild($xmlDoc.CreateElement("Output"))

      
      if ($info.CheckTitle -eq "LogonTimings") {
        $logonTimingsList = $info.CheckOutput | ConvertFrom-Json

        $logonTimingsList | ForEach-Object {
          $logonTimingsNode = $reasonNode.AppendChild($xmlDoc.CreateElement("logonTimings"))
          $logonTimingsNode.SetAttribute("SessionId", $_.SessionId)

          $logonTimingsObject = $_
          $logonTimingsObject.PSObject.Properties | ForEach-Object {
            $node = $logonTimingsNode.AppendChild($xmlDoc.CreateElement($_.Name))
            $node.AppendChild($xmlDoc.CreateTextNode($_.Value)) | out-null
          }
        }
      }else {
        $reasonNode.AppendChild($xmlDoc.CreateTextNode($info.CheckOutput)) | out-null
      }
      #PolicyName node
      $policyNameNode = $infoTypeNode.AppendChild($xmlDoc.CreateElement("PolicyName"))
      $policyNameNode.AppendChild($xmlDoc.CreateTextNode($info.PolicyName)) | out-null
      
      if ($null -ne $info.KBLinks) {
          $linksNode = $infoTypeNode.AppendChild($xmlDoc.CreateElement("KBLinks"))
          $links = $info.KBLinks.Split(";")
          foreach ($link in $links) {
              if ($link -ne ""){
                  $linkNode = $linksNode.AppendChild($xmlDoc.CreateElement("Link"))
                  $linkNode.AppendChild($xmlDoc.CreateTextNode($link)) | out-null
              }
          }
      }
  }
}



function Export-ToXml([string]$xmlPath, [bool]$isCategorized){
    $XML_Path = $xmlPath
    $xmlWriter = New-Object System.XMl.XmlTextWriter($XML_Path,$Null)
    $xmlWriter.Formatting = 'Indented'
    $xmlWriter.Indentation = 1
    $XmlWriter.IndentChar = "`t"

    $xmlWriter.WriteStartDocument()
    $xmlWriter.WriteComment('Result of UpmConfigCheck tool')
    $xmlWriter.WriteStartElement('Result')
    $xmlWriter.WriteEndElement()
    $xmlWriter.WriteEndDocument()
    $xmlWriter.Flush()
    $xmlWriter.Close()

    $xmlCreated = Test-Path $XML_Path
    if(-not $xmlCreated){
      Write-Host "$XML_Path does not exist."
      return
    }

    # Create the Initial  Node
    $xmlDoc = [System.Xml.XmlDocument](Get-Content $XML_Path)
    if (!$isCategorized)
    {
      $xmlDoc.SelectSingleNode("//Result").AppendChild($xmlDoc.CreateElement("CheckItems")) | out-null    
      $xmlDoc.Save($XML_Path)
      $xmlDoc = [System.Xml.XmlDocument](Get-Content $XML_Path); 
      Convert-CheckInfoToXml $xmlDoc $CHACheckInfoList $xmlDoc.SelectSingleNode("//Result/CheckItems")
  
    }else
    {
      $xmlDoc.SelectSingleNode("//Result").AppendChild($xmlDoc.CreateElement("Errors")) | out-null
      $xmlDoc.SelectSingleNode("//Result").AppendChild($xmlDoc.CreateElement("Warnings")) | out-null
      $xmlDoc.SelectSingleNode("//Result").AppendChild($xmlDoc.CreateElement("Notices")) | out-null
      $xmlDoc.SelectSingleNode("//Result").AppendChild($xmlDoc.CreateElement("Invalid")) | out-null
      $xmlDoc.Save($XML_Path)

      $xmlDoc = [System.Xml.XmlDocument](Get-Content $XML_Path);    
   
      Convert-InfoToXml $xmlDoc $errorInfoList "Error" $xmlDoc.SelectSingleNode("//Result/Errors")
      Convert-InfoToXml $xmlDoc $warningInfoList "Warning" $xmlDoc.SelectSingleNode("//Result/Warnings")
      Convert-InfoToXml $xmlDoc $noticeInfoList "Notice" $xmlDoc.SelectSingleNode("//Result/Notices")
      Convert-InfoToXml $xmlDoc $invalidInfoList "Invalid" $xmlDoc.SelectSingleNode("//Result/Invalid")
      
    }
   
    $xmlDoc.Save($XML_Path)
}

function Export-ToJson([string]$jsonPath) {
  # (Get-UICulture).Calendar | ConvertTo-Json -depth 100 | Out-File $jsonPath
  $json = @{
    Result = @{
      Warnings = $warningInfoList
      Errors   = $errorInfoList
      Notices = $noticeInfoList
      Invalid  = $invalidInfoList
    }
  }
  $json.Result.Warnings + $json.Result.Errors + $json.Result.Notices + $json.Result.Invalid | ForEach-Object {
    if ($_.CheckTitle -eq "SpecialFileExclusionSetting") {
      $_.CheckTitle = "Special File Exclusion Setting"
    }
    if ($_.CheckTitle -eq "SpecialExclusionRegKeyCheck") {
      $_.CheckTitle = "Special Exclusion Registry Key Check"
    }
    if ($_.CheckTitle -eq "Profile Management Driver Status") {
      $_.CheckTitle = "Profile Management Driver Installation Status"
    }
    if ($_.CheckTitle -eq "StreamingExclusionList") {
      $_.CheckTitle = "Streaming Exclusion list"
    }
  }

  $json | ConvertTo-Json -depth 32 | Remove-Prefix | Format-Json | Out-File $jsonPath

}

# Formats JSON in a nicer format than the built-in ConvertTo-Json does.
function Format-Json([Parameter(Mandatory, ValueFromPipeline)][String] $json) {
  $indent = 0;
  ($json -Split '\n' |
    % {
      if ($_ -match '[\}\]]') {
        # This line contains  ] or }, decrement the indentation level
        $indent--
      }
      $line = (' ' * $indent * 2) + $_.TrimStart().Replace(':  ', ': ')
      if ($_ -match '[\{\[]') {
        # This line contains [ or {, increment the indentation level
        $indent++
      }
      $line
  }) -Join "`n"
}

function Remove-Prefix([Parameter(Mandatory, ValueFromPipeline)][String] $json) {
  $json.Replace("*** Error: ", "").Replace("*** Warning: ", "").Replace("*** Notice: ", "")
}

function CanPolicyBeUploaded([string]$policyName){
  $forbidenList = @("DefaultExclusionListRegistry", "SyncDirList", "StreamingExclusionList", "SyncExclusionListDir", "ProcessedGroups", "PathToUserStore", "PathToLogFile", "MirrorFoldersList", "ExclusionListRegistry", "ExcludedGroups", "DefaultSyncExclusionListDir")
  return !($forbidenList -contains $policyName)    
}

#CEIP data points includes UPM version and all settings
function Export-CEIPData([string]$xmlPath){
  
  
  $xmlWriter = New-Object System.XMl.XmlTextWriter($xmlPath,$Null)
  $xmlWriter.Formatting = 'Indented'
  $xmlWriter.Indentation = 1
  $XmlWriter.IndentChar = "`t"

  $xmlWriter.WriteStartDocument()
  $xmlWriter.WriteComment('CEIP data points')
  $xmlWriter.WriteStartElement('Data')
  $xmlWriter.WriteEndElement()
  $xmlWriter.WriteEndDocument()
  $xmlWriter.Flush()
  $xmlWriter.Close()
  
  $xmlDoc = [System.Xml.XmlDocument](Get-Content $xmlPath)
  $versionNode = $xmlDoc.SelectSingleNode("//Data").AppendChild($xmlDoc.CreateElement("UpmVersion"))    
  $versionNode.AppendChild($xmlDoc.CreateTextNode($script:upmversion)) | out-null
  $settingsNode = $xmlDoc.SelectSingleNode("//Data").AppendChild($xmlDoc.CreateElement("UpmSettings"))
  
  for ($pix = 0; $pix -lt $policyDb.Length; $pix++) {
      if (($policyDb[$pix].Origin -ne "Not Checked") -and (CanPolicyBeUploaded -PolicyName $policyDb[$pix].Name)) {            
          $settingNode         = $settingsNode.AppendChild($xmlDoc.CreateElement("Setting"))
          $settingNameNode     = $settingNode.AppendChild($xmlDoc.CreateElement("Name"))
          $settingNameNode.AppendChild($xmlDoc.CreateTextNode($policyDb[$pix].Name)) | out-null
          $settingValueNode    = $settingNode.AppendChild($xmlDoc.CreateElement("Value"))
          $settingValueNode.AppendChild($xmlDoc.CreateTextNode($policyDb[$pix].EffectiveValue)) | out-null
      }
  }
  
  $xmlDoc.Save($xmlPath)
}
  
#
# get the user object for an AD user
# 
#
function Get-ADUser( [string]$samid=$env:username){
   $searcher=New-Object DirectoryServices.DirectorySearcher
   $searcher.Filter="(&(objectcategory=person)(objectclass=user)(sAMAccountname=$samid))"
   if ($searcher -ne $null){
      try{
       $user=$searcher.FindOne()
       }catch{
          Write-Host "user info cannot be retrieved."
       }
        if ($user -ne $null ){
            $user.getdirectoryentry()
       }
   }else{
      $user = $null
   }
}

function Get-ADUserVariable ([string]$variableName, [string]$defaultString) {
$user = Get-ADUser
if ($user -ne $null)
{
    $answer = $user.psbase.properties[$variableName]
    if ($null -ne $answer) {
      if ($answer -ne "") { return $answer }
    }      
}
$defaultString
}
function Get-EnvVariable    ([string]$variableName, [string]$defaultString) {
  #
  # only supports
  dir env: | foreach { if ($_.Name -eq $variableName) { $answer = $_.Value } }
  if ($null -ne $answer) {
    return $answer
  }
  $defaultString
}

function Get-OsShortName () {
  if ($osMajorVer -eq "5") {
    if ($IsWorkstation) {
      return "WinXP"
    } else {
      return "Win2003"
    }
  } elseif($osMajorVer -eq "6"){
    if ($IsWorkstation) {
      switch ($osMinorVer) {
      0 { return "WinVista" }
      1 { return "Win7" }
      2 { return "Win8" }
      default{return "Win8.1" }
      }
    } else {
      switch ($osMinorVer) {
      0 { return "Win2008" }
      1 { return "Win2008" }    # R2 - but the short string is the same
      2 { return "Win2012" }
      default { return "Win2012" } # R2 - but the short string is the same
      }
    }
  }elseif($osMajorVer -eq "10")
  {
      if($IsWorkstation) {
        if ($osBuildVer -gt 22000){
          return "Win11"
        }elseif ($osBuildVer -gt 19000){
          return "Win10 2003"
        }elseif ($osBuildVer -gt 18200){
            return "Win10 1903"
          }elseif ($osBuildVer -gt 17700){
            return "Win10RS5"
          }else{
            if ($osBuildVer -gt 17000){
                return "Win10RS4"
            }else{
                if ($osBuildVer -gt 16000){
                    return "Win10RS3"
                }else{
                    if ($osBuildVer -gt 15000){
                        return "Win10RS2"
                    }else{
                        if ($osBuildVer -gt 14000){
                            return "Win10RS1"
                        }else{
                            return "Win10"
                        }
                    }
                }
            }
          }
        } else {
          if ($osBuildVer -gt 20300){
            return "Win2022"
          }elseif ($osBuildVer -gt 18000){
            return "Win2019"
          }else{
            return "Win2016"
          }
        }
  }
}

function Get-IECookieFolder () {
  if ($osMajorVer -eq "5") {
    return 'Cookies'
  } else {
    return @("AppData\Local\Microsoft\Windows\INetCookies",
             "AppData\Local\Microsoft\Windows\WebCache",
             "AppData\Roaming\Microsoft\Windows\Cookies")
  }
}

function Get-CitrixVariable ([string]$variableName, [string]$defaultString) {
  switch ($variableName) {
  "CTX_PROFILEVER" { $osMajorVer = ([string]$script:osinfo.Version)[0] ; if ($osMajorVer -eq "5") { return "v1" } ; return "v2" }
  "CTX_OSBITNESS" { if ($script:osinfo.OSArchitecture -eq "64-bit") { return "x64" } ; return "x86" }
  "CTX_OSNAME" { return Get-OsShortName }
  }
  $defaultString
}

function Get-SubstituteString ([string]$delimitedString) {
  $delimiter = $delimitedString[0]
  $strippedString = $delimitedString.Substring(1, $delimitedString.Length - 2)
  $answer = $delimitedString       # default is to not change the string, if we can't find a match
  switch ($delimiter) {
    '#' { $answer = Get-ADUserVariable -variableName $strippedString -defaultString $delimitedString }
    '%' { $answer = Get-EnvVariable    -variableName $strippedString -defaultString $delimitedString }
    '!' { $answer = Get-CitrixVariable -variableName $strippedString -defaultString $delimitedString }
  }
  $answer
}

function Get-ProcessedPath ([string]$path) {
    try {
      $specials = '#!%'
      $startOfSearch = 0
      while ($true) {
        $firstDelimiterOffset = $path.IndexOfAny($specials, $startOfSearch)
        $delimiterChar = $path[$firstDelimiterOffset]
        
        if ($firstDelimiterOffset -lt 0) { break }
        $secondDelimiterOffset = $path.IndexOf($delimiterChar, $firstDelimiterOffset + 1)
        
        if ($secondDelimiterOffset -lt 0) { break }
        $parameterStringLength = $secondDelimiterOffset - $firstDelimiterOffset
        $parameterStringLength += 1
        $seg1 = $path.substring(0, $firstDelimiterOffset)
        $seg2 = $path.substring($firstDelimiterOffset, $parameterStringLength)
        $seg3 = $path.substring($firstDelimiterOffset + $parameterStringLength, $path.Length - ($firstDelimiterOffset + $parameterStringLength))
        # write-host "Keep start " $seg1
        # write-host "Substitute " $seg2
        # write-host "Keep end   " $seg3
        $seg2 = Get-SubstituteString -delimitedString $seg2
        $path = $seg1 + $seg2
        $startOfSearch = $path.Length
        $path = $path + $seg3
      }
      $path
    }
    Catch {
      $ErrorMessage = $_.Exception.Message
      $FailedItem = $_.Exception.ItemName
      Write-Host "Failed $FailedItem. The error message was $ErrorMessage"
    }
}

#
# returns an array of lines from the section identified
#
function Get-IniList ( $textCollection, $sectionName ) {
  $copying = $false
  for ($ix = 0; $ix -lt $textCollection.Length; $ix++ ) {
    $line = $textCollection[$ix]
    # write-host @hilight1 "inistring   " $line
    if ($line -match "\[") {
      $copying = $false
      $matchString = "[" + $sectionName + "]"
      if ( [string]$line -eq [string]$matchString ) {
        $copying = $true
        # write-host $hilight2 "start copy   " $line
      }
    } else {
      if ($copying) {
        if ( ([string]$line).Length -gt 0 ) {
          # write-host $hilight2 "inistring   " $line
          ([string]$line) -replace "=\s*$",""
        }
      }
    }
  }
}

#
# Harmonise reporting of policies
#

$strPoliciesDetected = @{}
$strHDXPoliciesDetected = @{}
$strIniLinesDetected = @{}
$strDefaultsDetected = @{}

function ReportPolicyDetected($policyName, $policyValue, $policySource="Policy") {
  $reportLine = "$policySource ""$policyName"" detected in registry - INI file was not checked"
  switch ($policySource) {
  "Policy" { $strPoliciesDetected[$policyName] = $policyValue }
  "HDXPolicy" { $strHDXPoliciesDetected[$policyName] = $policyValue }
  }
}

function ReportIniLineDetected($policyName, $policyValue) {
  $reportLine = "Policy '" + $policyName + "' detected in INI file after failing to find in registry"
  $strIniLinesDetected[$policyName] = $policyValue
}

function ReportDefaultDetected($policyName, $policyValue) {
  $reportLine = "Policy '" + $policyName + "' using default as no registry or INI file values detected"
  $strDefaultsDetected[$policyName] = $policyValue
}

#
# Functions for reading settings from the registry and from the INI file
#
function Get-IniSetting ( $textCollection, $sectionName, $settingName ) {
  $list = Get-IniList -textCollection $textCollection -sectionName $sectionName
  $list = @($list)
  for ($ix = 0; $ix -lt $list.Length; $ix++ ) {
    $keyPlusVal = $list[$ix] -split '=',2
    $key = $keyPlusVal[0]
    if ($keyPlusVal.Length -eq 2) {
      $val = $keyPlusVal[1]
    } else {
      $val = ""
    }
    if ($key -like $settingName ) {
      if ($val -match '^".*"$' ) {
        $val = ([string]$val).Substring(1,([string]$val).Length - 2)
      }
      return $val
    }
  }
}

$policyOriginTable = @(
  @( "Policy",    "HKLM:\\SOFTWARE\Policies\Citrix\UserProfileManager\" ),
  @( "HDXPolicy", "HKLM:\\SOFTWARE\Policies\Citrix\UserProfileManagerHDX\" )
)

function GetPolicyGeneralSetting($regName="" , $policyName ) {
  $retVal = "unset"
  #
  # look in registry first
  #
  for ($policySourceIndex = 0; $policySourceIndex -lt $policyOriginTable.Length; $policySourceIndex++) {
    $policyOrigin = ($policyOriginTable[$policySourceIndex])[0]
    $registryBase = ($policyOriginTable[$policySourceIndex])[1]
    $regPath = $registryBase + $regName
    #Write-Host @infoColours "GetPolicyGeneralSetting: Check " $regPath ":" $policyName
    Get-ItemProperty $regPath -name $policyName -ErrorAction SilentlyContinue | foreach {
      $retVal = $_.$policyName
    }
    if ($retval -ne "unset" ) {
      ReportPolicyDetected -policyName $policyName -policyValue @($retval) -policySource $policyOrigin
      Replace-PolicyRecordProperty -PolicyName $policyName -PropertyName EffectiveValue -NewValue $retval
      Replace-PolicyRecordProperty -PolicyName $policyName -PropertyName Origin         -NewValue $policyOrigin
      return $retVal
    }
  }

  #Write-Host @infoColours "GetPolicyGeneralSetting: Check INI file section General Settings, setting name:" $policyName
  $retval = Get-IniSetting -textCollection $iniContent -sectionName "General Settings" -settingName $policyName
  if ($retval -ne $null) {
    ReportIniLineDetected -policyName $policyName -policyValue @($retval)
    Replace-PolicyRecordProperty -PolicyName $policyName -PropertyName EffectiveValue -NewValue $retval
    Replace-PolicyRecordProperty -PolicyName $policyName -PropertyName Origin         -NewValue "IniFile"
  } else {
    ReportDefaultDetected -policyName $policyName -policyValue @($retval)
    Replace-PolicyRecordProperty -PolicyName $policyName -PropertyName EffectiveValue -NewValue $retval
    Replace-PolicyRecordProperty -PolicyName $policyName -PropertyName Origin         -NewValue "Default"
  }
  $retval
}

function GetPolicySingleSetting($regName , $valName, $defaultSetting, $autoSetting ) {
  $retVal = "unset"
  #
  # look in registry first
  #
  for ($policySourceIndex = 0; $policySourceIndex -lt $policyOriginTable.Length; $policySourceIndex++) {
    $policyOrigin = ($policyOriginTable[$policySourceIndex])[0]
    $registryBase = ($policyOriginTable[$policySourceIndex])[1]
    $regPath = $registryBase + $regName
    #Write-Host @infoColours "GetPolicySingleSetting: Check " $regPath ":" $valName
    Get-ItemProperty $regPath -name $valName -ErrorAction SilentlyContinue | foreach {
      $retVal = $_.$valName
    }
    if ($retval -ne "unset" ) {
      ReportPolicyDetected -policyName $valName -policyValue @($retval) -policySource $policyOrigin
      $policyObj = New-Object Object |
        Add-Member NoteProperty Value            $retVal                  -PassThru |
        Add-Member NoteProperty Origin           $policyOrigin            -PassThru 
      Replace-PolicyRecordProperty -PolicyName $valName -PropertyName EffectiveValue -NewValue $policyObj.Value
      Replace-PolicyRecordProperty -PolicyName $valName -PropertyName Origin         -NewValue $policyObj.Origin
      return $policyObj
    }
  }
  #
  # failed to find in either registry location
  #
  #Write-Host @infoColours "GetPolicySingleSetting: Check INI file section General Settings, setting name:" $valName
  $retVal = Get-IniSetting -textCollection $iniContent -sectionName "General Settings" -settingName $valName
  if ($retVal -ne $null) {
    ReportIniLineDetected -policyName $valName -policyValue @($retval)
    $policyObj = New-Object Object |
      Add-Member NoteProperty Value            $retVal                  -PassThru |
      Add-Member NoteProperty Origin           "IniFile"                -PassThru 
    Replace-PolicyRecordProperty -PolicyName $valName -PropertyName EffectiveValue -NewValue $policyObj.Value
    Replace-PolicyRecordProperty -PolicyName $valName -PropertyName Origin         -NewValue $policyObj.Origin
    return $policyObj
  } else {
    ReportDefaultDetected -policyName $valName -policyValue @($retval)
    if ($autoSetting -eq $null) {
      $policyObj = New-Object Object |
        Add-Member NoteProperty Value            $defaultSetting          -PassThru |
        Add-Member NoteProperty Origin           "Default"                -PassThru 
    } else {
      # optionally report this as an autosetting, rather than a default
      $policyObj = New-Object Object |
        Add-Member NoteProperty Value            $autoSetting             -PassThru |
        Add-Member NoteProperty Origin           "Default"                -PassThru 
    }
    Replace-PolicyRecordProperty -PolicyName $valName -PropertyName EffectiveValue -NewValue $policyObj.Value
    Replace-PolicyRecordProperty -PolicyName $valName -PropertyName Origin         -NewValue $policyObj.Origin
    return $policyObj
  }
}

function Get-ListItemCountRaw ($list) {
  if ($list -is [array]) {
    [string]($list.Length)
  } elseif ($list -is [string]) {
    1
  } else {
    0
  }
}

function Get-ListItemCount ($list) {
  $count = Get-ListItemCountRaw -list $list
  "List contains " + $count + " item(s)"
}

function GetPolicyDefaultListSetting($regName){
  $retVal = "unset"
  $realPath={};
  switch($regName)
  {
    "DefaultExclusionListRegistry"{$realPath = $realPathForDefaultExclusionReg}
    "DefaultSyncExclusionListDir"{$realPath = $realPathForDefaultExclusionDir}
  }
  for ($policySourceIndex = 0; $policySourceIndex -lt $policyOriginTable.Length; $policySourceIndex++) {
    $policyOrigin = ($policyOriginTable[$policySourceIndex])[0]
    $registryBase = ($policyOriginTable[$policySourceIndex])[1]
    $regPath = $registryBase + $regName
    #Write-Host @infoColours "GetPolicyDefaultListSetting: Check " $regPath ":" $policyName
    $keyObj = Get-Item $regPath -ErrorAction SilentlyContinue 
    if ($keyObj.ValueCount -gt 0)
    {
       $retVal = for ($ix = 0; $ix -lt $keyObj.ValueCount; $ix++ )
       {
          $realPathItem = $realPath.Get_Item($keyObj.Property[$ix]);
          if($realPathItem -ne $null)
          {
            $realPathItem
          }
       }
       ReportPolicyDetected -policyName $regName  -policyValue @($retVal)
       Replace-PolicyRecordProperty -PolicyName $regName -PropertyName EffectiveValue -NewValue $retval
       Replace-PolicyRecordProperty -PolicyName $regName -PropertyName Origin         -NewValue $policyOrigin
       return $retVal
    }
  }
  # look in INI file
  # return $retVal
  #Write-Host @infoColours "GetPolicyDefaultListSetting: Check INI file section" $regName
  $retval = Get-IniList -textCollection $iniContent -sectionName $regName
  if ($retval -ne $null) {
    ReportIniLineDetected -policyName $regName  -policyValue @($retVal)
    Replace-PolicyRecordProperty -PolicyName $regName -PropertyName EffectiveValue -NewValue $retval
    Replace-PolicyRecordProperty -PolicyName $regName -PropertyName Origin         -NewValue "IniFile"
  } else {
    ReportDefaultDetected -policyName $regName  -policyValue @($retVal)
    Replace-PolicyRecordProperty -PolicyName $regName -PropertyName EffectiveValue -NewValue $retval
    Replace-PolicyRecordProperty -PolicyName $regName -PropertyName Origin         -NewValue "Default"
  }
  $retval
}
function GetPolicyListSettingRaw($regName ) {
  $retVal = "unset"
  #
  # look in registry first
  #
  for ($policySourceIndex = 0; $policySourceIndex -lt $policyOriginTable.Length; $policySourceIndex++) {
    $policyOrigin = ($policyOriginTable[$policySourceIndex])[0]
    $registryBase = ($policyOriginTable[$policySourceIndex])[1]
    $regPath = $registryBase + $regName
    #Write-Host @infoColours "GetPolicyListSettingRaw: Check" $regPath
    switch ($policyOrigin) {
      "Policy" {
        $keyObj = Get-ChildItem $regPath -ErrorAction SilentlyContinue 
        if ($keyObj.ValueCount -gt 0) {
          $retVal = for ($ix = 0; $ix -lt $keyObj.ValueCount; $ix++ ) {
            $keyObj.Property[$ix]
          }
          ReportPolicyDetected -policyName $regName  -policyValue @($retVal)
          Replace-PolicyRecordProperty -PolicyName $regName -PropertyName EffectiveValue -NewValue $retval
          Replace-PolicyRecordProperty -PolicyName $regName -PropertyName Origin         -NewValue $policyOrigin
          return $retVal
        }
      }
      "HDXPolicy" {
        #
        # This will be a registry multi-string, with no "List" subkey
        #
        $keyArr = Get-ChildItem $registryBase -ErrorAction SilentlyContinue
        $keyArr = @($keyArr)
        $keyObj = for ($kix = 0 ; $kix -lt $keyArr.Length; $kix++ ) {
          $kname = $keyArr[$kix].Name -replace ".*\\",""
          if ($kname -eq $regName) {
            $keyArr[$kix]
          }
        }
        if ($keyObj -ne $null) {
          $retVal = for ($ix = 0; $ix -lt $keyObj.ValueCount; $ix++ ) {
            $keyObj.GetValue($regName)
          }
          ReportPolicyDetected -policyName $regName  -policyValue @($retVal) -policySource $policyOrigin
          Replace-PolicyRecordProperty -PolicyName $regName -PropertyName EffectiveValue -NewValue $retval
          Replace-PolicyRecordProperty -PolicyName $regName -PropertyName Origin         -NewValue $policyOrigin
          return $retVal
        }
      }
    }
  }

  #
  # look in INI file
  # return $retVal
  #Write-Host @infoColours "GetPolicyListSettingRaw: Check INI file section" $regName
  $retval = Get-IniList -textCollection $iniContent -sectionName $regName
  if ($retval -ne $null) {
    ReportIniLineDetected -policyName $regName  -policyValue @($retVal)
    Replace-PolicyRecordProperty -PolicyName $regName -PropertyName EffectiveValue -NewValue $retval
    Replace-PolicyRecordProperty -PolicyName $regName -PropertyName Origin         -NewValue "IniFile"
  } else {
    ReportDefaultDetected -policyName $regName  -policyValue @($retVal)
    Replace-PolicyRecordProperty -PolicyName $regName -PropertyName EffectiveValue -NewValue $retval
    Replace-PolicyRecordProperty -PolicyName $regName -PropertyName Origin         -NewValue "Default"
  }
  $retval
}

function GetPolicyListSetting($regName ) {
  $resp = GetPolicyListSettingRaw -regName $regName
  $rv = @($resp)  # force the response to be an array
  $rv
}

function ValidateList($list, $policyName, $category=$null) {
  Get-ListItemCount -list $list

  $sortPol = $list | sort
  $uniqPol = $list | sort -Unique
  if ($sortPol.Length -ne $uniqPol.Length) {
    "*** Warning: Some lines are duplicated in the setting: " + $policyName + "." | CaptureRecommendations -CheckTitle "Policy Duplication" -PolicyName $policyName -InfoType "Warning" -Category "Policy Duplication"
    Compare-Object -ReferenceObject $uniqPol -DifferenceObject $sortPol | foreach { $_.InputObject }
  } else {
    "No duplicates found in Policy:" + $policyName
  }
  $list | foreach {
    $item = $_
    switch -wildcard ($item) {
      "%USERPROFILE%*" { "*** Warning: Do not add %USERPROFILE% in the policy ""$policyName"", items ""$item""." | CaptureRecommendations -CheckTitle "Policy Content Validation" -InfoType "Warning" -PolicyName $policyName -Reason "Profile Management automatically assumes that the paths are relative to %USERPROFILE%." -Category $category}
    }
    switch -regex ($item) {
      ".*[\s=]$" { "*** Warning: Trailing spaces or ""="" exist in setting ""$policyName"",item ""$item""." | CaptureRecommendations  -CheckTitle "Policy Content Validation" -InfoType "Warning" -PolicyName $policyName -Reason "Trailing spaces can lead to unpredictable and hard-to-diagnose behaviors in Profile Management." -Category $category}
    }
  }
}

#
# This routine returns an object indicating the effective value
# and whether the value was explicitly set or defaulted
#
function GetEffectivePolicyFlag ($policyName, $defaultSetting, $autoSetting, [switch]$AsNumber) {
  #
  # Let's check a policy - $policyName - and calculate the effective value if defaulted
  #
  $polName = $policyName
  $pol = GetPolicySingleSetting -valName $polName -defaultSetting $defaultSetting -autoSetting $autoSetting
  if ($AsNumber) {
    $val = $pol.Value
  } else {
    if ($pol.Value -eq 0) {
      $val = "Disabled"
    } else {
      $val = "Enabled"
    }
  }
  $origin = $pol.Origin
  write-host "$polName is $val from $origin"
  $pol
}

function AssertFlagNotSet ($policyName, $Reason=$null,  $Category=$null) {
  $temp = GetEffectivePolicyFlag -policyName $policyName -defaultSetting 0
  $warningInfo = ""
  switch ($policyName) {
    "TemplateProfileOverridesLocalProfile" { 
      $warningInfo = "Profile handling > Template profile overrides local profile, Profile handling > Use template profile as Citrix mandatory profile for all logons: conflicting settings. Only one of these two settings is allowed to be enabled." 
    }

    "TemplateProfileOverridesRoamingProfile" {
      $warningInfo = "Profile handling > Template profile overrides roaming profile, Profile handling > Use template profile as Citrix mandatory profile for all logons: conflicting settings. Only one of these two settings is allowed to be enabled."
    }
    Default {}
  }
  $origin = $temp.Origin
  if (($origin -ne "Default") -and ($temp.Value -eq 1)) {
    $warningInfo | CaptureRecommendations -CheckTitle $policyName -PolicyName $policyName -Reason $Reason -Category $Category -InfoType "Notice"
  }
}

function PreferPolicyFlag ($policyName, $defaultSetting, $preferredSetting, $autoSetting, $InfoType = "Notice", $Reason=$null, [switch]$ShowAsNumber, $Category=$null)  {
  #
  # Let's check a policy - $policyName - and calculate the effective value if defaulted
  #
  if (($upmmajor -eq 5) -and ($autoSetting -ne $null)) {
    Replace-PolicyRecordProperty -PolicyName $policyName -PropertyName DefaultValue -NewValue $autoSetting
    if ($autosetting -ne $defaultSetting) {
      #write-host @infoColours "*** overwriting default for policy $policyName: default $defaultSetting changed to autoSetting $autoSetting"
      $defaultSetting = $autoSetting
    }
  } else {
    Replace-PolicyRecordProperty -PolicyName $policyName -PropertyName DefaultValue -NewValue $defaultSetting
  }
  Replace-PolicyRecordProperty -PolicyName $policyName -PropertyName PreferredValue -NewValue $preferredSetting
  $defaultString = "Enabled"
  if ($defaultSetting -eq 0) {
    $defaultString = "Disabled"
  }
  $polName = $policyName
  $pol = GetPolicyGeneralSetting -policyName $polName
  $polName + " = " + $pol
  if ($null -eq $pol) {
    if ($ShowAsNumber) {
      $polName + " is unset and will default to " + $defaultSetting ; $actualSetting = $defaultSetting
    } else {
      $polName + " is unset and will default to " + $defaultString ; $actualSetting = $defaultSetting
    }
  } else {
      $actualSetting = $pol
  }
  Replace-PolicyRecordProperty -PolicyName $policyName -PropertyName EffectiveValue -NewValue $actualSetting
  #
  # now report on the difference between what we've got and what we recommend
  #
  if ($ShowAsNumber) {
    $preferredString = [string]$preferredSetting
    $actualString = [string]$actualSetting
  } else {
    $preferredString = "Enabled"
    if ($preferredSetting -eq 0) {
      $preferredString = "Disabled"
    }
    $actualString = "Enabled"
    if ($actualSetting -eq 0) {
      $actualString = "Disabled"
    }
  }
  if ($actualSetting -ne $preferredSetting) {
    $wemConsoleName = $reg2wemConsoleNameTable.$policyName
    "$Category > $wemConsoleName" + ": the effective setting ""$actualString"" does not match the preferred setting ""$preferredString""." | CaptureRecommendations -CheckTitle "Recommended Value Check: $policyName" -InfoType $InfoType -PolicyName $policyName -Reason $Reason -Category $Category
  } else {
    $info = $polName + ": actual/effective setting (" + $actualString + ") matches preferred setting (" + $preferredString + ")." 
    $info | CaptureCheckResult -CheckTitle "Check $policyName" -PolicyName $policyName -InfoType "Info"
  }
}

#
# Get-VolumeMountPoints: returns the path names of all mounted volumes that 
#                         are mounted on a drive letter or a sub-folder
#
function Get-VolumeMountPoints() {
  $comp = Get-WmiObject Win32_Volume
  $comp | foreach {
    $mp = $_.Name
    switch -regex ($mp) {
    "^[a-z]:" { [string]$mp }
    }
  } | sort
}

#
# Get-MountPointCount: returns the number of matches of the given path 
#                       against the list of mount points.  We use this to detect
#                       detect when (say) c:\Users is a separate volume
#                       (This breaks the use of Change Journalling)
#
function Get-MountPointCount ($path) {
  $mountpoints = Get-VolumeMountPoints
  $mpc = 0
  # Added extra check to get the type of returned result.
  # If the type is array, calculate the length as usual;
  # Else if the type is string, we don't want to set the length as string.length, so set it as 1.
  # This is to be used in for loop below
  
  if ($mountpoints.GetType().BaseType.Name -eq "Array") 
    { $ml = $mountpoints.Length }
  elseif ($mountpoints.GetType().Name -eq "String") # the BaseType of String is Object, so we use GetType().Name directly here.
    { $ml = 1 }
  for ($ix = 0; $ix -lt $ml; $ix++) {
    $p = [string]($mountpoints[$ix])
    [string]$path | select-string -simplematch $p | foreach {
      $mpc++
    }
  }
  $mpc
}

function ConvenientBytesString($bytes) {
  if ($bytes -gt 1GB) {
    $gb = [math]::Round($bytes / 1GB,1)
    return [string]($gb) + " GB"
  }
  if ($bytes -gt 1MB) {
    $mb = [math]::Round($bytes / 1MB,1)
    return [string]($mb) + " MB"
  }
  if ($bytes -gt 1KB) {
    $kb = [math]::Round($bytes / 1KB,1)
    return [string]($kb) + " KB"
  }
  return [string]($bytes) + " bytes"
}

$envLogArray = @()

function AddEnvironmentLine ([string]$section, [string]$item, $value) {
  $policyObj = New-Object Object |
      Add-Member NoteProperty Section          $section          -PassThru |
      Add-Member NoteProperty Item             $item             -PassThru |
      Add-Member NoteProperty Value            $value            -PassThru 
  $script:envLogArray = $script:envLogArray + $policyObj
}

function ReportEnvironment ([string]$section, [string]$item, $value) {
  AddEnvironmentLine $section $item $value
  $padSection = $section.PadRight(10)
  $padItem = $item.PadRight(45)
  "$padSection" + ": $padItem" + ": $value"
}

$pilot = $true

function Test-UserStore ($Path, $ExpandedPath) {
  "Testing $Path"
  #
  # Test that the path is well-formed
  # It must contain a user name: one of #CN# , #sAMAccountName# or %USERNAME%
  # Any AD attributes should precede the user name - these are usually used for load-balancing across multiple servers / DFS namespaces
  # Any CTX-variables should follow the user name - these are usually used for keeping the user's different profiles together
  #
  # $components = $Path -split "\\"
  $adAttributes = $false
  $userNames = $false
  $ctxVars = $false
  $username = ""
  $usernameIndex = 0
  $backslashCount = 0
  $pathToTest = ""
  $parentPathToTest = ""
  $errors = @()
  $varMatches = $Path | Select-String -Pattern "(#[^#]+#|%[^%]+%|![^!]+!)" -AllMatches
  $varMatches.Matches | foreach {
    $matchItem = $_
    $comp = $matchItem.Value
    #write-host "Testing path component $comp"
    switch -regex ($comp) {
      "(#cn#|%username%|#sAMAccountName#)" {
          $username = $matches[1]
          $userNames = $true
          #
          # work out how many backslashes up to this point
          $usernameIndex = $matchItem.Index
          $prePath = $Path.Substring(0,$usernameIndex)
          $splitPath = $prePath.split("\\")
          $backslashCount = $splitPath.Length
          $iy = 0
          $ep = $ExpandedPath + "\"   # add a trailing slash to ensure we catch an unterminated path
          for ($ix = 0; $ix -lt $backslashCount; $ix++) {
            $iy = $ep.IndexOf("\",$iy)
            $iy++
            $parentPathToTest = $pathToTest
            $pathToTest = $ep.Substring(0,$iy)
          }
          if ($username -eq "#cn#") {
            "*** Warning: Basic settings > Path to user store: the path ""$Path"" contains #cn#." | CaptureRecommendations -CheckTitle "User Store Path" -PolicyName "PathToUserStore" -Reason "We recommend that you use %USERNAME% or #sAMAccountName# instead of #cn# in the path setting, because #cn# might not be unique. Also, #cn# often contains white spaces, punctuation marks, or national language characters. Those characters might cause errors when used in folder or file names." -InfoType "Warning"
          }
        }
      "(%[^%]+%)" {
          $m = $matches[1]
          switch -regex ($m) {
            '%username%|%userdomain%' {
                # these are fine
              }
            default {
                "*** Error: Basic settings > Path to user store: the variable ""$m"" in the path ""$Path"" is not supported. The user profile cannot be synced." | CaptureRecommendations -CheckTitle "User Store Path" -PolicyName "PathToUserStore" -Reason "By default, Profile Management supports only %USERNAME% and %USERDOMAIN% as environment variables. Support for more environment variables requires extra scripting."
              }
          }
        }
      "(!ctx.*!)" {
          $m = $matches[1]
          if (-not $userNames) {
            "Basic settings > Path to user store: the CTX variable ""$m"" precedes the user name in ""$Path""." | CaptureRecommendations -CheckTitle "User Store Path" -InfoType "Warning" -PolicyName "PathToUserStore" -Reason "We recommend that you place the CTX variable behind the user name. Doing so makes it easier to back up a full copy of user settings. It can also simplify the setup of ACLs."
          }
          $ctxVars = $true
        }
    }
  }
  if (-not $userNames) {
    "*** Error: Basic settings > Path to user store: the path ""$Path"" does not contain a user name. The user profile cannot be synced." | CaptureRecommendations -CheckTitle "User Store Path" -PolicyName "PathToUserStore" -Reason "The path must contain a user name. Specify the user name by using %USERNAME%, #sAMAccountName#, or #cn#. #cn# is least preferred because it might not be unique."
  }
}

function Test-UserStore-FileShare($Path, $ExpandedPath) {
  $result = $false
  $varMatches = $Path | Select-String -Pattern "(#cn#|%username%|#sAMAccountName#)" -AllMatches
  if ($varMatches.Matches.Count -gt 0) {
    $usernameVar = $varMatches.Matches[0].Groups[1]

    $username = Get-SubstituteString -delimitedString $usernameVar.Value
    $index = $ExpandedPath.IndexOf($username)
    $tempPath = $ExpandedPath.Substring(0, $index)
    $latestBackslash = $tempPath.LastIndexOf("\")
    $testPath = $tempPath.Substring(0, $latestBackslash + 1)

    if (Test-Path $testPath -ea SilentlyContinue) {
      #"File Share $FileSharePath exists"
      $result = $true
    }
    else {
      $result = $false
    }
  }
  else {
    "*** Error: Basic settings > Path to user store: the path ""$Path"" does not contain a user name. The user profile cannot be synced." | CaptureRecommendations -CheckTitle "User Store Path" -PolicyName "PathToUserStore" -Reason "The path must contain a user name. Specify the user name by using %USERNAME%, #sAMAccountName#, or #cn#. #cn# is least preferred because it might not be unique."
    $result = $false
  }
  return $result
}

#
# packages and urclass.dat should be included in the same time
function CheckPackageAndUsrclassDat {

  $packagesIsIncluded = $false
  $usrClassDatIsIncluded = $false
  
  $mirroFolders = GetPolicyListSetting -regName "MirrorFoldersList" | v2SubstituteNames
  $excludedDirs = (@(GetPolicyListSetting -regName "SyncExclusionListDir") + @(GetPolicyDefaultListSetting -regName "DefaultSyncExclusionListDir")) | v2SubstituteNames
  
  $excludedFiles = GetPolicyListSetting -regName "SyncExclusionListFiles" | v2SubstituteNames
  v2SubstituteNames
  $includedFiles = GetPolicyListSetting -regName "SyncFileList" | v2SubstituteNames

  if ($null -ne $mirroFolders -and $mirroFolders -contains "AppData\Local\Packages") {
    $packagesIsIncluded = $true
  }
  elseif ($null -ne $excludedDirs -and $excludedDirs -contains "AppData\Local\Packages") {
    $packagesIsIncluded = $false
  }
  else {
    $packagesIsIncluded = $true
  }

  if ($null -ne $includedFiles -and $includedFiles -contains "AppData\Local\Microsoft\Windows\UsrClass.dat*") {
    $usrClassDatIsIncluded = $true
  }
  elseif ($null -ne $excludedFiles -and $excludedFiles -contains "AppData\Local\Microsoft\Windows\UsrClass.dat*") {
    $usrClassDatIsIncluded = $false
  }
  else {
    $usrClassDatIsIncluded = $true
  }
    
  if (-not($packagesIsIncluded -and $usrClassDatIsIncluded)) {
    "*** Warning: File system: The Start menu might not work properly." | CaptureRecommendations -CheckTitle "Special File Exclusion Setting" -InfoType "Warning" -PolicyName "" -Reason "The file !ctx_localappdata!\Microsoft\Windows\UsrClass.dat* and the folder !ctx_localappdata!\Packages must both be included. Otherwise, the Start menu might not work properly." -Category "Profile Management File System Settings"
  }
  else {
    "UsrClass.dat* and Appdata\Local\Package are correctly configured." | CaptureCheckResult -CheckTitle "SpecialFileExclusionSetting" -InfoType "Info"
  }
}

function Test-StartMenu {
  if ($upmDisplayVersion -ge $upmCR2106 ) {
    PreferPolicyFlag -policyName "AccelerateFolderMirroring" -defaultSetting 0 -preferredSetting 1 -Reason "We recommend enabling this policy to eliminate the need to copy the folders between the user store and local profiles and accelerates folder mirroring." -Category "Profile Management File System Settings"
    $accelerateFolderMirroringRecommended = $true
  }

  $mirroFolders = GetPolicyListSetting -regName "MirrorFoldersList" | v2SubstituteNames  
  $includedFiles = GetPolicyListSetting -regName "SyncFileList" | v2SubstituteNames

  $packagesIsIncluded = $mirroFolders -contains "AppData\Local\Packages"
  $cachesIsIncluded = $mirroFolders -contains "Appdata\Local\Microsoft\Windows\Caches"
  $tileDataLayerIsIncluded = $mirroFolders -contains "AppData\Local\TileDataLayer"
  $usrClassDatIsIncluded = $includedFiles -contains "AppData\Local\Microsoft\Windows\UsrClass.dat*"

  $allIncluded = $packagesIsIncluded -and $cachesIsIncluded -and $usrClassDatIsIncluded

  if ($allIncluded) {
    if ($ostype -eq "Workstation" -and $windowsVersionInt -le 1607 -and -not $tileDataLayerIsIncluded) {
      "*** Warning: File system: The Start menu might not work properly." | CaptureRecommendations -CheckTitle "Special File Exclusion Setting" -InfoType "Warning" -PolicyName "" -Reason "The folder !ctx_localappdata!\TileDataLayer must be included. Otherwise, the Start menu might not work properly." -Category "Profile Management File System Settings"
    }
    else {
      "Start menu roaming settings are correctly configured." | CaptureCheckResult -CheckTitle "SpecialFileExclusionSetting" -InfoType "Info"
    }
  }
  else {
    "*** Warning: File system: The Start menu might not work properly." | CaptureRecommendations -CheckTitle "Special File Exclusion Setting" -InfoType "Warning" -PolicyName "" -Reason "Both the file !ctx_localappdata!\Microsoft\Windows\UsrClass.dat and the folders !ctx_localappdata!\Packages and !ctx_localappdata!\Microsoft\Windows\Caches must be included. Otherwise, the Start menu might not work properly." -Category "Profile Management File System Settings"
  }
}

function CheckSpeechOneCore
{
    $excludedRegistries = GetPolicyListSetting -regName "ExclusionListRegistry"
    $defaultExcludedRegistries=(GetPolicyDefaultListSetting -regName "DefaultExclusionListRegistry")
    $excludedRegistriesAll = $excludedRegistries + $defaultexcludedRegistries;
    if($excludedRegistriesAll-ne $null -and $excludedRegistriesAll -Contains "Software\Microsoft\Speech_OneCore")
    {
      "*** Error: Registry > Enable registry exclusions: the registry key ""\HKCU\Software\Microsoft\Speech_OneCore"" exists in the Registry Exclusions list. File type associations might not work properly."  | CaptureRecommendations -CheckTitle "Special Exclusion Registry Key Check" -PolicyName "" -Reason "Remove this registry key from the Registry Exclusion list." -Category "Profile Management Registry Settings"
    }else{
      "\\HKCU\Software\\Microsoft\\Speech_OneCore is correctly set." | CaptureCheckResult -CheckTitle "SpecialExclusionRegKeyCheck" -InfoType "Info"
    }
}

function pause ($message)
{
    # Check if running Powershell ISE
    if ($psISE)
    {
        Add-Type -AssemblyName System.Windows.Forms
        [System.Windows.Forms.MessageBox]::Show("$message")
    }
    else
    {
        try{
            Write-Host "$message" -ForegroundColor Yellow
            $x = $host.ui.RawUI.ReadKey("NoEcho,IncludeKeyDown")
        }catch{
        }
    }
}

########################################################################################
#
# The following logic duplicates the environment checking in UpmConfig\UpmConfig.cpp
# We use global variables already determined
# This is used to determine autoconfig properties for Profile Management v5 and defaults on older versions             
#
########################################################################################
function Get-AutoconfigSettingsFromEnv ($enabled) {
  #
  # Set the basics for "original" hardwired defaults
  $settingDeleteProfiles = 0
  $settingProfileDeleteDelay = 0
  $settingStreaming = 0
  $settingStreamingForFolders = 0
  $settingAlwaysCache = 0
  $settingActiveWriteBack = 0
  #
  # let's put the globals into local variables
  
  switch ($upmmajor) {
    7 {
    }
    5 {
      #
      # set up for XenApp, then adjust
      $settingDeleteProfiles = 1
      $settingProfileDeleteDelay = 0
      $settingStreaming = 1
      $settingAlwaysCache = 0
      $settingActiveWriteBack = 1
      if ($IsWorkstation) {
        if ($IsRunningXD) {
          if ($vdiInfo.IsAssigned) {
            $settingDeleteProfiles = 0  # we want profiles to persist
          }
          else {
            $settingDeleteProfiles = 1  # we delete profiles on pooled desktops
          }
          if (-not $vdiInfo.OSChangesPersist) {
            $settingProfileDeleteDelay = 60    # allow a minute before we start deleting, as the disk may be discarded before we waste IOPS
          }
          $settingStreaming = 1
          $settingAlwaysCache = 0
          $settingActiveWriteBack = 1
        }
        else {
          $settingDeleteProfiles = 0
          $settingStreaming = 1
          $settingAlwaysCache = 1
          $settingActiveWriteBack = 1
        }
      }
    }
    3 {
      $settingActiveWriteBack = 1    # this is the default on Profile Management v3 (which is effectively the same as autoconfig)
    }
    4 {
      $settingActiveWriteBack = 0    # this is the default on Profile Management v4 (which is effectively the same as autoconfig)
    }
    default {
      $settingActiveWriteBack = 1
      $settingAlwaysCache = 0
      $settingDeleteProfiles = 1
      $settingProfileDeleteDelay = 0
      $settingStreaming = 1
      $settingStreamingForFolders = 1
      if ($IsWorkstation) {
        $settingAlwaysCache = 1
        
        if (-not $IsRunningXD) {
          $settingDeleteProfiles = 0
        }else{
          if ($vdiInfo.IsAssigned) {
            $settingDeleteProfiles = 0  # we want profiles to persist
          }
          if (-not $vdiInfo.OSChangesPersist) {
            $settingProfileDeleteDelay = 60    # allow a minute before we start deleting, as the disk may be discarded before we waste IOPS
          }
        }
      }
    }
  }
  New-Object Object |
  Add-Member NoteProperty DeleteCachedProfilesOnLogoff   $settingDeleteProfiles               -PassThru |
  Add-Member NoteProperty ProfileDeleteDelay             $settingProfileDeleteDelay           -PassThru |
  Add-Member NoteProperty PSEnabled                      $settingStreaming                    -PassThru |
  Add-Member NoteProperty PSForFoldersEnabled            $settingStreamingForFolders          -PassThru |
  Add-Member NoteProperty PSAlwaysCache                  $settingAlwaysCache                  -PassThru |
  Add-Member NoteProperty PSMidSessionWriteBack          $settingActiveWriteBack              -PassThru 
}

$localSystemSid = "S-1-5-18" # Well-known SID of LocalSystem

function CheckEventLogPermission ([EventLog]$eventLog) {
  $evtxPermission = $false
  $regPermission = $false
  # check permission of .evtx file
  $evtxPath = Get-WinEvent -ListLog $eventLog.eventLogName | Select-Object -ExpandProperty LogFilePath
  $expandedEvtxPath = [System.Environment]::ExpandEnvironmentVariables($evtxPath) 
  try {
    $acl = Get-Acl -Path $expandedEvtxPath

    # Iterate through the access control entries
    foreach ($ace in $acl.Access) {
      # Convert the ACE IdentityReference to SID
      try {
        $aceSid = $ace.IdentityReference.Translate([System.Security.Principal.SecurityIdentifier]).Value
      }
      catch {
        continue
      }

      if ($aceSid -eq $localSystemSid) {
        if ($ace.FileSystemRights -band [System.Security.AccessControl.FileSystemRights]::Read) {
          $evtxPermission = $true
          break
        }
      }
    }

    if ($evtxPermission) {
      Write-Host "Local System has read permission on the file $filePath."
    }
    else {
      Write-Host "Local System does NOT have read permission on the file $filePath."
    }
  }
  catch {
    Write-Host "Error: $_"
  }

  # check permission of registry key
  $regPath = $eventLog.regPath
  try {
    $acl = Get-Acl -Path $regPath
    # Iterate through the access control entries
    foreach ($ace in $acl.Access) {
      # Convert the ACE IdentityReference to SID
      try {
        $aceSid = $ace.IdentityReference.Translate([System.Security.Principal.SecurityIdentifier]).Value
      }
      catch {
        continue
      }

      if ($aceSid -eq $localSystemSid) {
        if ($ace.RegistryRights -band [System.Security.AccessControl.RegistryRights]::ReadKey) {
          $regPermission = $true
          break
        }
      }
    }

    if ($regPermission) {
      Write-Host "Local System has read permission on the registry $regPath."
    }
    else {
      Write-Host "Local System does NOT have read permission on the registry $regPath."
    }
  }
  catch {
    Write-Host "Error: $_"
  }

  if ($evtxPermission -and $regPermission) {
    "Citrix Profile Management has Read permission on Windows event logs ""$($eventLog.eventLogDisplayName)""." | CaptureCheckResult -CheckTitle "Windows Event Logs: $($eventLog.eventLogDisplayName)" -PolicyName "" -InfoType "Info"
  }
  else {
    "Citrix Profile Management does not have Read permission on Windows event logs ""$($eventLog.eventLogDisplayName)""." | CaptureRecommendations -CheckTitle "Windows Event Logs: $($eventLog.eventLogDisplayName)" -PolicyName "" -InfoType "Warning"
  }
}

function Test-OutlookInstallation {
  # Check if Outlook is installed by looking for the executable in common paths
  $outlookPaths = @(
    "$env:ProgramFiles\Microsoft Office\root\Office16\OUTLOOK.EXE",
    "$env:ProgramFiles(x86)\Microsoft Office\root\Office16\OUTLOOK.EXE",
    "$env:ProgramFiles\Microsoft Office\Office15\OUTLOOK.EXE",
    "$env:ProgramFiles(x86)\Microsoft Office\Office15\OUTLOOK.EXE",
    "$env:ProgramFiles\Microsoft Office\Office14\OUTLOOK.EXE",
    "$env:ProgramFiles(x86)\Microsoft Office\Office14\OUTLOOK.EXE"
  )

  $outlookInstalled = $false

  foreach ($path in $outlookPaths) {
    if (Test-Path $path) {
      $outlookInstalled = $true
      break
    }
  }

  if ($outlookInstalled) {
    Write-Host "Outlook is installed."
    return $true
  }
  else {
    # Check for Outlook registry entries
    $outlookRegistryPaths = @(
      "HKLM:\Software\Microsoft\Office\16.0\Outlook",
      "HKLM:\Software\Microsoft\Office\15.0\Outlook",
      "HKLM:\Software\Microsoft\Office\14.0\Outlook",
      "HKLM:\Software\Wow6432Node\Microsoft\Office\16.0\Outlook",
      "HKLM:\Software\Wow6432Node\Microsoft\Office\15.0\Outlook",
      "HKLM:\Software\Wow6432Node\Microsoft\Office\14.0\Outlook"
    )

    $outlookInstalled = $false

    foreach ($regPath in $outlookRegistryPaths) {
      if (Test-Path $regPath) {
        $outlookInstalled = $true
        break
      }
    }

    if ($outlookInstalled) {
          
      Write-Host "Outlook is installed."
      return $true
    }
    else {
          
      Write-Host "Outlook is not installed."
      return $false
    }
  }
}


function Test-OneDriveInstallation {
  $oneDrivePaths = @(
    "$env:LOCALAPPDATA\Microsoft\OneDrive\OneDrive.exe",
    "$env:ProgramFiles\Microsoft OneDrive\OneDrive.exe",
    "$env:ProgramFiles(x86)\Microsoft OneDrive\OneDrive.exe"
  )

  foreach ($path in $oneDrivePaths) {
    if (Test-Path -Path $path) {
      Write-Host "OneDrive is installed." 
      return $true
    }
  }

  Write-Host "OneDrive is not installed." -ForegroundColor Red
  return $false
}

function Test-NewTeamsInstallation {

  try { $teamsApp = Get-AppxPackage -AllUsers -Name "MSTeams" }
  catch {
    Write-Host "An error occurred:"
    Write-Host $_
  }

  if ($teamsApp) {
    Write-Host "Microsoft Teams is installed."
    return $true
  }
  else {
    Write-Host "Microsoft Teams is not installed."
    return $false
  }

}

function Test-RecommendedFolderExclusionForNewTeams {
  $commonFolders = @(
    "LocalCache\Microsoft\MSTeams\Logs",
    "LocalCache\Microsoft\MSTeams\PerfLog",
    "LocalCache\Microsoft\MSTeams\EBWebView\WV2Profile_tfw\WebStorage",
    "LocalCache\Microsoft\MSTeams\EBWebView\WV2Profile_tfw\GPUCache",
    "AppData\Local\Packages\Microsoft.Windows.StartMenuExperienceHost_cw5n1h2txyewy\TempState"
  )

  $foldersForWinServer2019 = @(
    "AppData\Local\Packages\Microsoft.Windows.ShellExperienceHost_cw5n1h2txyewy\TempState"
  )

  $excludedFolders = GetPolicyListSetting -regName "SyncExclusionListDir"

  $commonFolders | ForEach-Object {
    if (-not ($excludedFolders -contains $_)) {
        "File system > Excluded folders: the folder ""$($_)"" is not in the exclusion list, but it must be excluded to ensure Microsoft Teams functions correctly." | CaptureRecommendations -CheckTitle "Exclusion list" -PolicyName "SyncExclusionListDir" -Reason "We recommend adding the folder to the exclusion list." -Category "Profile Management File System Settings" -InfoType "Notice"
    }
  }

  if ($winVer -eq "Win2019") {
    $foldersForWinServer2019 | ForEach-Object {
      if (-not ($excludedFolders -contains $_)) {
          "File system > Excluded folders: the folder ""$($_)"" is not in the exclusion list, but it must be excluded to ensure Microsoft Teams functions correctly." | CaptureRecommendations -CheckTitle "Exclusion list" -PolicyName "SyncExclusionListDir" -Reason "We recommend adding the folder to the exclusion list." -Category "Profile Management File System Settings" -InfoType "Notice"
      }
    }
  }

}

function Test-FoldersToMirrorForNewTeams {
  $folders = @(
    "AppData\Roaming\Microsoft\Windows\Cookies",
    "AppData\Local\Microsoft\Windows\INetCookies",
    "AppData\Local\Microsoft\Windows\WebCache",
    "AppData\Local\Microsoft\Vault",
    "AppData\Local\Microsoft\Windows\Caches",
    "AppData\Local\Packages"
  )

  $MirrorFoldersList = GetPolicyListSetting -regName "MirrorFoldersList"

  $folders | ForEach-Object {
    if (-not ($MirrorFoldersList -contains $_)) {
        "File system > Mirrored folders: the folder ""$($_)"" is not in the exclusion list, but it is required for Microsoft Teams to function correctly." | CaptureRecommendations -CheckTitle "Folder mirroring" -PolicyName "MirrorFoldersList" -Reason "We recommend adding the folder to the mirror folder list." -Category "Profile Management File System Settings" -InfoType "Notice"
    }
  }
}

function Test-PsExclusionFolderListForNewTeams {
  $folders = @(
    "AppData\Local\Packages\MSTeams_8wekyb3d8bbwe\AC",
    "AppData\Local\Packages\MSTeams_8wekyb3d8bbwe\AppData",
    "AppData\Local\Packages\MSTeams_8wekyb3d8bbwe\LocalCache",
    "AppData\Local\Packages\MSTeams_8wekyb3d8bbwe\LocalState",
    "AppData\Local\Packages\MSTeams_8wekyb3d8bbwe\RoamingState",
    "AppData\Local\Packages\MSTeams_8wekyb3d8bbwe\Settings",
    "AppData\Local\Packages\MSTeams_8wekyb3d8bbwe\SystemAppData",
    "AppData\Local\Packages\MSTeams_8wekyb3d8bbwe\TempState"
  )

  $PsExclusionFolderList = GetPolicyListSetting -regName "StreamingExclusionList"

  $folders | ForEach-Object {
    if (-not ($PsExclusionFolderList -contains $_)) {
        "Streamed user profiles > Excluded folders: the folder ""$($_)"" is not in the exclusion list, but it must be excluded to ensure Microsoft Teams functions correctly." | CaptureRecommendations -CheckTitle "Profile streaming exclusion list" -PolicyName "StreamingExclusionList" -Reason "We recommend adding the folder to the exclusion list." -Category "Streamed user profiles" -InfoType "Notice"
    }
  }
}

############################################################################
#
# ... end of the function definitions
#
############################################################################

#
# This is all set up at the top of the script
#
ReportEnvironment "UpmCheck" "Version"   $upmCheckVersion
ReportEnvironment "UpmCheck" "Copyright" $copyright
ReportEnvironment "UpmCheck" "RunDate"   $scriptRunDate.DateTime


"=========================================================="
"= Gathering Environment Information for further checks   ="
"=========================================================="

#
# Take an inventory of Citrix products
#
New-PSDrive -Name Uninstall -PSProvider Registry -Root HKLM:\SOFTWARE\Microsoft\Windows\CurrentVersion\Uninstall -ea SilentlyContinue >$null
New-PSDrive -Name Uninstall32 -PSProvider Registry -Root HKLM:\SOFTWARE\Wow6432Node\Microsoft\Windows\CurrentVersion\Uninstall -ea SilentlyContinue >$null
$citrixProds = Get-ChildItem -Path @("Uninstall:","Uninstall32:") -ea SilentlyContinue | ForEach-Object -Process {
  $q = $_
  $dn = $q.GetValue("DisplayName") 
  # write-host $dn
  $dp = $q.GetValue("Publisher") 
  $dv = $q.GetValue("DisplayVersion") 
  $cs = New-Object Object |
    Add-Member NoteProperty Product      $dn           -PassThru |
    Add-Member NoteProperty Publisher    $dp           -PassThru |
    Add-Member NoteProperty Version      $dv           -PassThru
  $cs
} | where-object { (($_.Product -match "Citrix") -or ($_.Publisher -match "Citrix") -or ($_.Product -match "ShareFile") -or ($_.Publisher -match "ShareFile")) }

$xenDesktopPresent = $false
$xenAppPresent = $false
$shareFilePresent = $false ; $wrongShareFilePresent = $false
foreach ($prod in $citrixProds) {
  #ReportEnvironment "CtxProduct" $prod.Product $prod.Version
  switch -wildcard ($prod.Product) {
  "*MetaFrame*" { $xenAppPresent = $true; $xenAppVersion = $prod.Version }
  "*Presentation Server*" { $xenAppPresent = $true; $xenAppVersion = $prod.Version }
  "*XenApp*" { $xenAppPresent = $true; $xenAppVersion = $prod.Version }
  "*Virtual Desktop Agent*" { $xenDesktopPresent = $true; $xenDesktopVersion = $prod.Version }
  "Citrix ShareFile Sync" { $shareFilePresent = $true; $shareFileVersion = $prod.Version }
  "ShareFile Desktop Sync" { $wrongShareFilePresent = $true; $wrongShareFileProduct = $prod.Product; $wrongShareFileVersion = $prod.Version }
  }
}
#
# Detect the environment
#
$currentUser = "unset"
$currentDomain = "unset"
$currentLocalProfile = "unset"
dir env: | foreach {
  $x = $_
  if ($x.Key -eq "USERNAME") { $currentUser = $x.Value }
  if ($x.Key -eq "USERDOMAIN") { $currentDomain = $x.Value }
  if ($x.Key -eq "USERPROFILE") { $currentLocalProfile = $x.Value }
}

#
# see http://blogs.technet.com/b/heyscriptingguy/archive/2010/10/11/use-wmi-and-powershell-to-get-a-user-s-sid.aspx
#
try{
    $userSID = ([wmi]"win32_userAccount.Domain='$env:userdomain',Name='$env:username'").SID
}catch{
    #Cannot get user sid, do nothing. Keep it null
    $userSID = $null
}

#
# work out if this is the user's first Profile Management logon - there won't be a UPMSettings.ini file at the path to user store
#
#
# See http://www.powershellmagazine.com/2012/10/31/pstip-new-powershell-drive-psdrive/
#
# Set up a PSDrive for HKEY_USERS (which doesn't have a drive by default)
#

$hasHku = $false
Get-PSDrive -PSProvider Registry | foreach {
  if ($_.Root -eq "HKEY_USERS") {
    $hasHku = $true
  }
}

#
# create the drive
#
if (-not $hasHku) {
  New-PSDrive -Name HKU  -PSProvider Registry -Root HKEY_USERS > $null
}



#
# This section looks at 4 related policies, and is now fairly complex
#
# Set up constants and preferred settings for all policies, then adjust as we go
#
$DELETE_ON_LOGOFF = 1                                         # DeleteCachedProfilesOnLogoff
$KEEP_ON_LOGOFF = 0                                           # DeleteCachedProfilesOnLogoff
$preferredProfileDisposition = $KEEP_ON_LOGOFF                # DeleteCachedProfilesOnLogoff
$PROFILE_DELETE_IMMEDIATE = 0                                 # ProfileDeleteDelay
$PROFILE_DELETE_DEFERRED = 60                                 # ProfileDeleteDelay
$preferredDeleteDelay = $PROFILE_DELETE_IMMEDIATE             # ProfileDeleteDelay
$PROFILES_STREAMED_ON_DEMAND = 1                              # PSEnabled
$PROFILES_COPIED_IN_FULL = 0                                  # PSEnabled
$ACTIVE_WRITE_BACK_DISABLED = 0                               # PSMidSessionWriteBack
$ACTIVE_WRITE_BACK_ENABLED = 1                                # PSMidSessionWriteBack
$activeWriteBackPreferred = $ACTIVE_WRITE_BACK_ENABLED        # PSMidSessionWriteBack
$activeWriteBackDefault = $ACTIVE_WRITE_BACK_ENABLED          # PSMidSessionWriteBack


#
# Start point is the OS version
#
$osinfo =   Get-WmiObject Win32_OperatingSystem          # this is useful info - hang on to it
$versionMajor   = $script:osinfo.Version -replace "\..*","" -as [int]
$lastBoot       = $script:osinfo.ConvertToDateTime($script:osinfo.LastBootUpTime)
$installTime    = $script:osinfo.ConvertToDateTime($script:osinfo.InstallDate)
$IsWorkstation  = $script:osinfo.ProductType -eq 1
$versionDetails = ([string]$script:osinfo.Version).Split('.')
$osMajorVer     = $versionDetails[0]
$osMinorVer     = $versionDetails[1]
$osBuildVer     = $versionDetails[2]
#######################################################################################################
#
#       Basic settings check: 
#           1. Profile Management installation(Including both service and driver)
#           2. Profile Management service enabled
#           3. User store path validation
#           4. Processed Group
#           5. Excluded Group
#           6. Process logons of local administrators
#           7. Active write back
#           8. Offline support
#
#######################################################################################################
#
# discover where Profile Management is installed correctly
#
"---------------------------------------------"
"- Checking Profile Management Installation  -"
"- Step 1: Checking installation status      -"
"---------------------------------------------"
$UPMBase = "C:\Program Files\Citrix\User Profile Manager"     # default location
$UPMPath = "HKLM:\\SYSTEM\CurrentControlSet\services\ctxProfile"
$c = Get-ItemProperty $UPMPath -name "ImagePath" -ea SilentlyContinue
$UPMExe = [string]$c.ImagePath
if ($UPMExe.Length -eq 0) {
  "*** Invalid: Profile Management is not installed." | CaptureRecommendations -CheckTitle "Profile Management Installation Status" -PolicyName "" -Reason "" -InfoType "Invalid" -Category "Profile Management Installation"
  Export-ToXml -xmlPath $OutputXmlPath -isCategorized $true
  Export-ToXml -xmlPath $ChaOutputXmlPath -isCategorized $false
  Export-ToJson -jsonPath $OutputJsonPath

  if (!(StringIsNullOrWhitespace($CEIPFilePath))){
    Export-CEIPData($CEIPFilePath)
  }
  exit;
} else {
  $u = $UPMExe -replace '"([^"]*)"', '$1' # Use -replace to remove double quotes
  $fo = dir $u
  $UPMBase = [string]($fo.DirectoryName)
  
  "Profile Management is installed in folder ""$UPMBase""." | CaptureCheckResult -CheckTitle "Profile Management Installation Status" -PolicyName "" -InfoType "Info"
  
  $UPMBaseDriverPath = "$UPMBase\Driver\upmjit.sys"

  if (-not (Test-Path $UPMBaseDriverPath) -and (Test-Path "$UPMBase\upmjit.sys")) {
      $UPMBaseDriverPath = "$UPMBase\upmjit.sys"
  }
  
  $UPMBaseDriverVersion = (Get-Command $UPMBaseDriverPath).FileVersionInfo.FileVersion
}

"--------------------------------------------"
"- Checking Profile Management Installation -"
"- Step 2: Checking winLogon hook            -"
"--------------------------------------------"
if ($versionMajor -ge 6) {
  $regPath = "HKLM:\\System\CurrentControlSet\Control\Winlogon\Notifications\Configurations\Default"
  "Checking registry: $regPath"
  $Logon = Get-ItemProperty $regPath -name "Logon"
  ReportEnvironment "WinLogon" "LogonHook" $Logon.Logon
  $Logoff = Get-ItemProperty $regPath -name "Logoff"
  ReportEnvironment "WinLogon" "LogoffHook" $Logoff.Logoff
  $logonSeq = $Logon.Logon -split ","
  $upmLogonFound = $false
  $logonCheck = $false
  for ($ix = 0; $ix -lt $logonSeq.Length; $ix++) {
    switch ($logonSeq[$ix]) {
      "UserProfileMan" { $upmLogonFound = $true }
      "Profiles" { if ($upmLogonFound) { $logonCheck = $true } }
    }
  }
  if ($logonCheck) {
    #Write-Host -ForegroundColor Green "Profile Management correctly hooked for logon processing"
    "Profile Management is correctly hooked to logon processing." | CaptureCheckResult -CheckTitle "Profile Management WinLogon Logon Hook" -PolicyName "" -InfoType "Info"
  } else {
    "*** Error: Profile Management is not correctly hooked to logon processing. Make sure that Profile Management is installed by an administrator account." | CaptureRecommendations -CheckTitle "Profile Management WinLogon Logon Hook" -PolicyName "" -Reason "Profile Management requires administrator privilege during installation to write to certain protected areas of the registry." -Category "Profile Management Installation"
  }
  $logoffSeq = $Logoff.Logoff -split ","
  $profileLogoffFound = $false
  $logoffCheck = $false
  for ($ix = 0; $ix -lt $logoffSeq.Length; $ix++) {
    switch ($logoffSeq[$ix]) {
      "Profiles" { $profileLogoffFound = $true }
      "UserProfileMan" { if ($profileLogoffFound) { $logoffCheck = $true } }
    }
  }
  if ($logoffCheck) {
     #Write-Host -ForegroundColor Green "Profile Management correctly hooked for logoff processing"
     "Profile Management is correctly hooked to logoff processing." | CaptureCheckResult -CheckTitle "Profile Management WinLogon Logoff Hook" -PolicyName "" -InfoType "Info"
  } else {
    "*** Error: Profile Management is not correctly hooked to logoff processing. Make sure that Profile Management is installed by an administrator account." | CaptureRecommendations -CheckTitle "Profile Management WinLogon Logoff Hook" -PolicyName "" -Reason "Profile Management requires administrator privilege during installation to write to certain protected areas of the registry." -Category "Profile Management Installation"
  }
}

"--------------------------------------------------"
"- Checking Profile Management Installation       -"
"- Step 3: Checking Profile Management Driver     -"
"--------------------------------------------------"
#
# Detect the Profile Management version
#
$upmCR2103 = [version]"21.3.0.29010"
$upmCR2106 = [version]"21.6.0.30011"
$upmLTSR1912CU2 = [version]"19.12.2000.3"
$upmname = "unset"
$upmpublisher = "unset"
$upmversion = "unset"
$upmmajor = 0
$upmminor = 0

$exePath = $UPMBase + "\UserProfileManager.exe"
$versionInfo = (Get-Command $exePath).FileVersionInfo

$upmname = $versionInfo.ProductName
$upmpublisher = $versionInfo.CompanyName
$upmversion = $versionInfo.ProductVersion
$upmmajor = $versionInfo.ProductMajorPart
$upmminor = $versionInfo.ProductMinorPart


# Query Profile Management Driver Version in windows
$sysRoot = $env:SystemRoot
$UPMDriver = "$sysRoot\System32\Drivers\UPMJIT.sys"
$upmjitversion = "unset"
if(Test-path $UPMDriver){
	$upmjitversion = ((Get-Command $UPMDriver).FileVersionInfo).FileVersion
	if($upmjitversion -lt $UPMBaseDriverVersion ) {
		"*** Warning: An earlier version of Profile Management driver is being used. Profile Management might not function properly." | CaptureRecommendations -CheckTitle "Profile Management Driver Installation Status" -PolicyName "" -Reason "Profile Management driver in use is an earlier version than expected." -InfoType "Warning"
   }
   "Profile Management driver is correctly installed." | CaptureCheckResult -CheckTitle "Profile Management Driver Status" -PolicyName "" -InfoType "Info"
}
else{
	"*** Error: Profile Management driver is not installed." | CaptureRecommendations -CheckTitle "Profile Management Driver Installation Status" -PolicyName "" -Reason "Profile Management could not work properly." -Category "Profile Management Installation"
}

ReportEnvironment "Profile Management" "ProductName"     $upmname
ReportEnvironment "Profile Management" "CompanyName"     $upmpublisher
ReportEnvironment "Profile Management" "DisplayVersion"  $upmversion
ReportEnvironment "Profile Management" "VersionMajor"    $upmmajor
ReportEnvironment "Profile Management" "VersionMinor"    $upmminor
ReportEnvironment "Profile Management Driver" "DisplayVersion" $upmjitversion

$upmDisplayVersion = [version]$upmversion

#
# look for INI files
#   For Profile Management version 5, there is only one INI file
#   For Profile Management versions 2-4:
#   On XP and 2k3, the path is language-dependent, so we look for the local language first, then fall back to English
#   On all other OS we are language-independent
#
if ($iniFilePath -eq "") {
  if ($upmmajor -lt 5) {
    if ($versionMajor -lt 6) {
      # look for language-specific version first
      $c = Get-Culture
      $lang = $c.TwoLetterISOLanguageName
      $iniFile = $UPMBase + "\UPMPolicyDefaults_V1Profile_" + "$lang" + ".ini"
      $targetExists = Test-Path -LiteralPath $iniFile -ea SilentlyContinue
      if ($targetExists -eq $false) {
        $iniFile = $UPMBase + "\UPMPolicyDefaults_V1Profile_en.ini"
      }
    } else {
      $iniFile = $UPMBase + "\UPMPolicyDefaults_V2Profile_all.ini"
    }
  } else {
    $iniFile = $UPMBase + "\UPMPolicyDefaults_all.ini"
  }
} else {
  $iniFile = $iniFilePath
}

#
# If we have an INI file we read it into a variable, else
# we set up an empty variable
#
if (Test-Path -LiteralPath $iniFile) {
  $iniContent = Get-Content $iniFile | Select-String -Pattern "^;" -NotMatch
  #"Found INI file at '" + $iniFile + "' - contents after stripping comments:"
  #$iniContent
} else {
  $iniContent = @("")
}
"-------------------------------------------------------"
"-                                                     -"
"- Collect other services and environment info         -"
"-------------------------------------------------------"
"-----------------------------------------"
"- Checking XenDesktop Environment       -"
"-----------------------------------------"
$prov = $physical         # assume physically provisioned
$IsRunningXD = $false     # this is specifically used to mimic logic in autoConfig, and doesn't care about older versions of XD
$mcsPersistentFolder = "C:\Program Files\Citrix\PvsVM\Service\PersistedData"
try {
  $vdiInfo = Get-WmiObject -Namespace ROOT\citrix\DesktopInformation -class Citrix_VirtualDesktopInfo -ErrorAction "SilentlyContinue" 
}
catch {
  "No XenDesktop WMI interface available"
}
if ($null -ne $vdiInfo) {
  $IsRunningXD = $true
  $isAssigned = $vdiInfo.IsAssigned
  $isVirtualMachine = $vdiInfo.IsVirtualMachine
  $osChangesPersist = $vdiInfo.OSChangesPersist
  if ($vdiInfo.OSChangesPersist -eq $false) {
    $prov = "PVS/MCS"
  }
  if ($vdiInfo.IsProvisioned) {
    $prov = $vdiInfo.ProvisioningType
  }
}
else {
  "checking C:\personality.ini"
  #check for provisioned

  if ( test-path -path "C:\personality.ini" ) {
    #
    $personalityObj = Get-ChildItem "C:\Personality.ini"
    $personalitytime = $personalityObj.LastWriteTime

    if ( $personalitytime -ge $lastBoot ) {
      # only if personality was created on this boot...
      if ( test-path -path $mcsPersistentFolder ) {
        $prov = "MCS"
      }
      else {
        $pvsmode = "Private"
        Get-Content "C:\personality.ini" | select-string -pattern "DiskMode=Shared" | foreach {
          $pvsmode = "Shared"
        }
        $prov = "PVS_" + $pvsmode
      }
    }
  }
}

ReportEnvironment "Machine" "Provisioning" $prov

"--------------------------------"
"- Checking Hypervisor          -"
"--------------------------------"

$hypervisor = $physical
$shutdownService = "No Shutdown Service"

$comp = Get-WmiObject Win32_ComputerSystem
switch ( $comp ) {
  { $_.Manufacturer -match "Xen" }                { $hypervisor = "XenServer" ; $shutdownService = "xensvc" }
  { $_.Manufacturer -match "Microsoft" }          { $hypervisor = "Hyper-V" ;   $shutdownService = "vmicshutdown" }
  { $_.Manufacturer -match "vmware" }             { $hypervisor = "VMWare" ;    $shutdownService = "VMTools" }
  { $_.Model -match "vmware" }                    { $hypervisor = "VMWare" ;    $shutdownService = "VMTools" }
}

$bios = Get-WmiObject Win32_Bios
switch ( $bios ) {
  { $_.Manufacturer -match "Xen" }                { $hypervisor = "XenServer" ; $shutdownService = "xensvc" }
  { $_.Version -match "Xen" }                     { $hypervisor = "XenServer" ; $shutdownService = "xensvc" }
  { $_.Version -match "VRTUAL" }                  { $hypervisor = "Hyper-V" ;   $shutdownService = "vmicshutdown" }
  { $_.SerialNumber -match "vmware" }             { $hypervisor = "VMWare" ;    $shutdownService = "VMTools" }
}

ReportEnvironment "Machine" "Hypervisor" $hypervisor

"---------------------------------"
"   Checking Laptop              -"
"---------------------------------"
$isLaptop = $false

$sysEnc = Get-WmiObject Win32_SystemEnclosure
switch ($sysEnc.ChassisTypes) {
8 { "SystemEnclosure: Portable" ; $isLaptop = $true }
9 { "SystemEnclosure: Laptop" ; $isLaptop = $true }
10 { "SystemEnclosure: Notebook" ; $isLaptop = $true }
11 { "SystemEnclosure: Handheld" ; $isLaptop = $true }
14 { "SystemEnclosure: Sub-Notebook" ; $isLaptop = $true }
default { "SystemEnclosure: Not Portable/Laptop/Notebook/Handheld/Sub-Notebook" }
}

$Battery = Get-WmiObject Win32_Battery
if ($Battery -eq $null) {
  "No Battery"
} else {
  "Has Battery"
  $isLaptop = $true
}

$PortableBattery = Get-WmiObject Win32_PortableBattery
if ($PortableBattery -eq $null) {
  "No PortableBattery"
} else {
  "Has PortableBattery"
  $isLaptop = $true
}

$PcmCiaController = Get-WmiObject Win32_PCMCIAController
if ($PcmCiaController -eq $null) {
  "No PCMCIA Controller"
} else {
  "Has PCMCIA Controller"
  $isLaptop = $true
}

ReportEnvironment "Machine" "Laptop" $isLaptop

#
# for now, just get the policies
#
# Get-ChildItem -path HKLM:\SOFTWARE\Policies\Citrix\UserProfileManager -recurse


"---------------------------------"
"-  Checking OS                  -"
"---------------------------------"
#
# check OS version, boot time, install date
#

ReportEnvironment "OS" "Version"          $script:osinfo.Version
ReportEnvironment "OS" "BuildNumber"      $script:osinfo.BuildNumber
ReportEnvironment "OS" "Caption"          $script:osinfo.Caption
ReportEnvironment "OS" "LastBootUpTime"   $lastBoot
ReportEnvironment "OS" "InstallDate"      $installTime
ReportEnvironment "OS" "OSArchitecture"   $script:osinfo.OSArchitecture
# Product type 1=workstation, 2=domain controller, 3=server
$ostype = "unset"
switch ( $script:osinfo.ProductType ) {
1 { $ostype = "Workstation" }
2 { $ostype = "Domain Controller" }
3 { $ostype = "Server" }
}
ReportEnvironment "OS" "ProductType"  $ostype
ReportEnvironment "OS" "Machine Name" $script:osinfo.__SERVER

#
# Save OS identifier for specific tests elsewhere
#
$winVer = Get-OsShortName

"-----------------------------------------"
"- Checking Terminal Service / Profiles  -"
"-----------------------------------------"
#
# check for Roaming profiles
#
#
# Query in the following order:
#
#  1. Group policy TS profile directory (if in TS session)
#  2. User account TS profile directory (if in TS session)
#  3. Group policy "normal" profile directory (if on Vista/2008)
#  4. User account "normal" profile directory
#
$roaming = $false
#
$regPath = "HKLM:SOFTWARE\Policies\Microsoft\Windows NT\Terminal Services"
$valname = "WFProfilePath"
Get-ItemProperty $regPath -name $valName -ErrorAction SilentlyContinue | foreach {
  #ReportEnvironment "TsProf" "Group policy TS profile directory"   $_.$valName
  $roaming = $true
}

#
# Adsi query goes here (User account TS profile directory, look at #TerminalServicesProfilePath# in user object)
#
$aduser = Get-ADUser
if ($aduser -ne $null){
    $aduser.psbase.properties.propertynames | foreach {
      if ($_ -eq "TerminalServicesProfilePath") {
        #ReportEnvironment "TsProf" "#TerminalServicesProfilePath#"   $aduser.psbase.properties["TerminalServicesProfilePath"]
        $roaming = $true
      }
    }
}

$regPath = "HKLM:SOFTWARE\Policies\Microsoft\Windows\System"
$valname = "MachineProfilePath"
Get-ItemProperty $regPath -name $valName -ErrorAction SilentlyContinue | foreach {
  #ReportEnvironment "TsProf" "Group policy normal profile directory" $_.$valName
  $roaming = $true
}

#
# Adsi query goes here (User account TS profile directory, look at #ProfilePath# in user object)
#
if ($aduser -ne $null){
    $aduser.psbase.properties.propertynames | foreach {
      if ($_ -eq "ProfilePath") {
        #ReportEnvironment "TsProf" "#ProfilePath#" $aduser.psbase.properties["ProfilePath"]
        $roaming = $true
      }
    }
}



if ($roaming) {
  "Roaming Profile path is detected"
} else {
  "Roaming Profile path is not detected"
}



"------------------------------------------------"
"- Checking Profile Management Basic Settings   -"
"- Step 1: Checking ServiceActive               -"
"------------------------------------------------"
###
#
#
# Let's check whether Profile Management is enabled. If not, exit directly.
#
$polName = "ServiceActive"
PreferPolicyFlag -policyName $polName -defaultSetting 0 -preferredSetting 1 -Reason "Enable Profile Management." -Category "Profile Management Base Settings"
$serviceActive = Get-PolicyRecordProperty -PolicyName $polName -PropertyName EffectiveValue

if ($serviceActive -eq 0){
    $info = "Unable to perform configuration check because Profile Management is disabled."
    Write-Host $info
    #This extra invalid object is for wem to mark this as invalid instead of warning.
    $infoObj = New-Object UpmConfigCheckOutputInfo
    $infoObj.CheckTitle = "ServiceActive"
    $infoObj.PolicyName = "ServiceActive"
    $infoObj.Info = $info
    $infoObj.Reason = "To enable configuration check, enable Profile Management. " 
    $infoObj.Type = "Invalid"
	  $script:invalidInfoList.Add($infoObj)
    
    Export-ToXml -xmlPath $OutputXmlPath -isCategorized $true
    Export-ToXml -xmlPath $ChaOutputXmlPath -isCategorized $false
    Export-ToJson -jsonPath $OutputJsonPath

    if (!(StringIsNullOrWhitespace($CEIPFilePath))){
        Export-CEIPData($CEIPFilePath)
    }
    exit;
 }
 
"--------------------------------------------------"
"- Checking Profile Management Basic Settings     -"
"- Step 2: Checking User Store Path               -"
"--------------------------------------------------"
$polName = "PathToUserStore"
$pol = GetPolicyGeneralSetting -policyName $polName
$polName + " = " + $pol
$expandedPathToUserStore = Get-ProcessedPath -path $pol
"Expand the user store path to $expandedPathToUserStore using current environment variables."
switch -wildcard ($pol) {
'\\*' { 'Path to user store appears to be a fileshare or DFS namespace, assuming Production.' ; $pilot = $false }
'* ' { '*** Notice: Basic settings > Path to user store: the path contains one or more trailing spaces.' | CaptureRecommendations -CheckTitle "User Store Path" -InfoType "Notice" -PolicyName $polName -Reason "We recommend removing trailing spaces. Trailing spaces can lead to unpredictable and hard-to-diagnose behaviors in Profile Management." }
}



if (($null -ne $expandedPathToUserStore) -and ($expandedPathToUserStore -ne "")) {
  if ($IsChaCall) {
    $targetExists = Test-Path -LiteralPath $expandedPathToUserStore -ea SilentlyContinue
    if ($targetExists) {
      "The path to the user store is valid. The path $expandedPathToUserStore exists already." | CaptureCheckResult -CheckTitle "User Store Path" -PolicyName $polName -InfoType "Info" -KBLinks "https://docs.citrix.com/en-us/profile-management/current-release/configure/specify-user-store-path.html"
    }
  } else {
    "The path to the user store is configured." | CaptureCheckResult -CheckTitle "User Store Path" -PolicyName $polName -InfoType "Info" -KBLinks "https://docs.citrix.com/en-us/profile-management/current-release/configure/specify-user-store-path.html"
  }
} else {
  "*** Error: Basic settings > Path to user store: the path is not configured. The user profile cannot be synced." | CaptureRecommendations -CheckTitle "User Store Path" -PolicyName $polName -Reason "Configure the setting with the user store path."
}

if (-not $pilot) {
  # only test path to user store for production
  Test-UserStore -Path $pol -ExpandedPath $expandedPathToUserStore
}

"==================================================="

$polName = "ProfileContainer"
$pol = GetPolicyListSetting -regName $polName
$polName + ":"
$pol

ValidateList -list $pol -policyName $polName -category "Profile container"

# Check if the policy is set to use full container
$polcount = Get-ListItemCountRaw -list $pol
$useFullContainer = ($null -ne $pol) -and ($polcount -eq 1) -and ($pol[0] -eq "*")

# Check if the policy is set to use mandatory profile
$isUsingMandatoryProfile = $false
$polName = "TemplateProfileIsMandatory"
$templateProfileIsMandatory = GetEffectivePolicyFlag -policyName $polName -defaultSetting 0
if ($templateProfileIsMandatory.Value -eq 1) {
  $isUsingMandatoryProfile = $true
}
    
if ($useFullContainer) {
  "*** Using full container."
}

if ($isUsingMandatoryProfile) {
  "*** Using mandatory profile."
}

"-------------------------------------------------"
"- Checking Profile Management Basic Settings    -"
"- Step 3: Checking ProcessAdmins                -"
"-------------------------------------------------"

"Skip checking ProcessAdmins"

"-------------------------------------------------"
"- Checking Profile Management Basic Settings    -"
"- Step 4: Checking ProcessedGroups              -"
"-------------------------------------------------"

$polName = "ProcessedGroups"
$pol = GetPolicyListSetting -regName $polName
$polName + ":"
$pol

ValidateList -list $pol -policyName $polName -category "Profile Management Basic Settings"

$polcount = Get-ListItemCountRaw -list $pol
if (($polcount -eq 0) -and ($pilot -eq $false)) { "*** Processed Groups is not configured. By default, all groups are processed." }

"-------------------------------------------------"
"- Checking Profile Management Basic Settings    -"
"- Step 5: Checking ExcludedGroups               -"
"-------------------------------------------------"

$polName = "ExcludedGroups"
$pol = GetPolicyListSetting -regName $polName
$polName + ":"
$pol

ValidateList -list $pol -policyName $polName -category "Profile Management Basic Settings"

if ((-not $useFullContainer) -and (-not $isUsingMandatoryProfile)) {
  "-------------------------------------------------"
  "- Checking Profile Management Basic Settings    -"
  "- Step 6: Checking ActiveWriteBack              -"
  "-------------------------------------------------"
  $activeWriteBackReason = "We recommend enabling this setting to prevent loss of profile changes in the event of power outages."
   
  if ($upmmajor -ge 4) {
    $activeWriteBackDefault = $ACTIVE_WRITE_BACK_DISABLED
  }
  
  #ignore below as MSS is recommended to be enabled by default
  if ($vmIsVolatile -eq $false) {
    $activeWriteBackPreferred = $ACTIVE_WRITE_BACK_ENABLED
  }
  
  $polName = "PSMidSessionWriteBack"
  $PSMidSessionWriteBack = GetPolicyGeneralSetting -policyName $polName
  $polName + " = " + $PSMidSessionWriteBack

  "-------------------------------------------------"
  "- Checking Profile Management Basic Settings    -"
  "- Step 7: Checking PSMidSessionWriteBackReg     -"
  "-------------------------------------------------"

  $polName = "PSMidSessionWriteBackReg"
  $PSMidSessionWriteBackReg = GetPolicyGeneralSetting -policyName $polName
  $polName + " = " + $PSMidSessionWriteBackReg

  $polName = "PSMidSessionWriteBackSessionLock"

  if ($PSMidSessionWriteBack -eq 1 -or $PSMidSessionWriteBackReg -eq 1) {
    PreferPolicyFlag -policyName $polName -defaultSetting 0 -preferredSetting 1 -Reason "We recommend enabling this setting when Active write back or Active write back registry is enabled." -Category "Profile Management Basic Settings"
  
  }
  
  "-------------------------------------------------"
  "- Checking Profile Management Basic Settings    -"
  "- Step 8: Checking OfflineSupport               -"
  "-------------------------------------------------"
  #
  # note that this is now largely covered elsewhere, but the laptop case is new
  $reported = $false
  
  if ($upmmajor -ge 4) {
    if ($isLaptop) {
      PreferPolicyFlag -policyName "OfflineSupport" -defaultSetting 0 -preferredSetting 1 -Reason "We recommend enabling this setting for laptop environments because laptops need to sync user profiles while disconnected from the domain." -Category "Profile Management Basic Settings"
      PreferPolicyFlag -policyName "PSEnabled" -defaultSetting $PROFILES_COPIED_IN_FULL -preferredSetting $PROFILES_COPIED_IN_FULL -autoSetting $autoConfigSettings.PSEnabled  -Reason "We recommend disabling this setting in laptop environments. Laptops can be removed from the domain after the first logon. As a result, user profiles must be fully synced to the laptop on the first logon." -Category "Streaming Settings"
      PreferPolicyFlag -policyName "DeleteCachedProfilesOnLogoff" -defaultSetting $KEEP_ON_LOGOFF -preferredSetting $KEEP_ON_LOGOFF -autoSetting $autoConfigSettings.DeleteCachedProfilesOnLogoff  -Reason "We recommend disabling this setting in laptop environments. Laptops can be removed from the domain after the first logon. As a result, locally cached profiles must be retained on logoff." -InfoType "Warning" -Category "Profile Management Profile Handling Settings"
      $reported = $true
    }
  }
  
  if (-not $reported) {
    PreferPolicyFlag -policyName "OfflineSupport" -defaultSetting 0 -preferredSetting 0 -Reason "We recommend disabling this setting because the device does not require or support the offline profile." -Category "Profile Management Basic Settings"
  }

  "--------------------------------------------------------"
  "- Checking Profile Container Settings                  -"
  "- Step 1: Checking ProfileContainerExclusionListDir   -"
  "--------------------------------------------------------"
  $polName = "ProfileContainerExclusionListDir"
  $ProfileContainerExclusionFolderList = GetPolicyListSetting -regName $polName
  $polName + ":"
  $ProfileContainerExclusionFolderList
  
  ValidateList -list $ProfileContainerExclusionFolderList -policyName $polName -category "Profile container"

  "--------------------------------------------------------"
  "- Checking Profile Container Settings                  -"
  "- Step 2: Checking ProfileContainerInclusionListDir   -"
  "--------------------------------------------------------"
  $polName = "ProfileContainerInclusionListDir"
  $pol = GetPolicyListSetting -regName $polName
  $polName + ":"
  $pol
  
  ValidateList -list $pol -policyName $polName -category "Profile container"

  "--------------------------------------------------------"
  "- Checking Profile Container Settings                  -"
  "- Step 3: Checking ProfileContainerExclusionListFile   -"
  "--------------------------------------------------------"
  $polName = "ProfileContainerExclusionListFile"
  $pol = GetPolicyListSetting -regName $polName
  $polName + ":"
  $pol
  
  ValidateList -list $pol -policyName $polName -category "Profile container"

  "--------------------------------------------------------"
  "- Checking Profile Container Settings                  -"
  "- Step 4: Checking ProfileContainerInclusionListFile   -"
  "--------------------------------------------------------"
  $polName = "ProfileContainerInclusionListFile"
  $pol = GetPolicyListSetting -regName $polName
  $polName + ":"
  $pol
  
  ValidateList -list $pol -policyName $polName -category "Profile container"
  
  "--------------------------------------------------------"
  "- Checking Profile Management Advanced Settings        -"
  "- Step 1: Checking ProcessCookieFiles                  -"
  "--------------------------------------------------------"

  "Skip checking ProcessCookieFiles"

  $polName = "ProcessCookieFiles"
  $pol = GetPolicyGeneralSetting -policyName $polName
  $isCookieEnabled = 1
  if (($null -eq $pol) -or ($pol -eq 0)) {
    $isCookieEnabled = 0
  } 

  "--------------------------------------------------------"
  "- Checking Profile Management Advanced Settings        -"
  "- Step 2: Check AutomaticConfiguration                 -"
  "--------------------------------------------------------"
  #
  # test DisableDynamicConfig
  #
  $dynamic = GetEffectivePolicyFlag -policyName "DisableDynamicConfig" -defaultSetting 0
  if ($upmmajor -lt 5) {
    $dynamicConfigEnabled = $false
  }
  else {
    if ($dynamic.Value -eq 0) {
      $dynamicConfigEnabled = $true
    }
    else {
      $dynamicConfigEnabled = $false
    }
  }

  if ($dynamicConfigEnabled) {
    $autoConfigSettings = Get-AutoconfigSettingsFromEnv -enabled $dynamicConfigEnabled
  }
  else {
    $autoConfigSettings = $null
  }

  "------------------------------------------------------"
  "- Checking Profile Management Advanced Settings      -"
  "- Step 3: Checking LogoffRatherThanTempProfile       -"
  "------------------------------------------------------"
  $preferLogoffToTempProfile = GetEffectivePolicyFlag -policyName "LogoffRatherThanTempProfile" -defaultSetting 0
  if (($preferLogoffToTempProfile.Value -eq 1) -and ($upmmajor -gt 4)) {
    #
    "If attempts to synchronize the user profile fail, Profile Management assigns a temporary profile to the current user. After the user logs on, session logoff is forced immediately."
  }
  else {
    #
    "If attempts to synchronize the user profile fail, Profile Management assigns a temporary profile to the current user."
  }
}

if (-not $isUsingMandatoryProfile) {
  "-----------------------------------------------------"
  "- Checking Profile Management Advanced Settings     -"
  "- Step 4: Checking OneDriveContainer                -"
  "-----------------------------------------------------"

  $polName = "OneDriveContainer"
  $pol = GetPolicyListSetting -regName $polName
  $polName + ":"
  $pol

  if (Test-OneDriveInstallation) {
    ValidateList -list $pol -policyName $polName -category "Profile Management Advanced Settings"

    if ($pol) {
      $polName = "DisableConcurrentAccessToOneDriveContainer"
      PreferPolicyFlag -policyName $polName -defaultSetting 0 -preferredSetting 0 -Reason "We recommend disabling this setting to reduce synchronization conflicts and ensure data integrity." -Category "Profile Management Advanced Settings"
    }
  }

  "-----------------------------------------------------"
  "- Checking Profile Management Advanced Settings     -"
  "- Step 5: Checking search index roaming for Outlook -"
  "-----------------------------------------------------"

  $polName = "OutlookSearchRoamingEnabled"

  if (Test-OutlookInstallation) {
    PreferPolicyFlag -policyName $polName -defaultSetting 0 -preferredSetting 1 -Reason "We recommend enabling this setting to improve the user experience when searching for emails in Microsoft Outlook. When enabled, the Offline Outlook Data File (*.ost) and Microsoft search database roam with the user profile." -Category "Profile Management Advanced Settings"  
  }
  else {
    "Outlook is not installed." | CaptureCheckResult -CheckTitle "Check $polName" -PolicyName $polName -InfoType "Info"
  }

  $OutlookSearchRoamingEnabled = Get-PolicyRecordProperty -PolicyName $polName -PropertyName EffectiveValue

  if ($OutlookSearchRoamingEnabled -eq 1) {
    $polName = "OutlookEdbBackupEnabled"
    PreferPolicyFlag -policyName $polName -defaultSetting 0 -preferredSetting 1 -Reason "We recommend enabling this setting to improve the user experience when Outlook search roaming is enabled." -Category "Profile Management Advanced Settings"  

    $polName = "OutlookSearchRoamingConcurrentSessionEnabled"
    PreferPolicyFlag -policyName $polName -defaultSetting 0 -preferredSetting 1 -Reason "We recommend enabling this setting to improve the user experience when Outlook search roaming is enabled." -Category "Profile Management Advanced Settings"  
  }

  "-----------------------------------------------------"
  "- Checking Profile Management Advanced Settings     -"
  "- Step 6: Checking SyncGpoStateEnabled              -"
  "-----------------------------------------------------"

  $polName = "SyncGpoStateEnabled"
  PreferPolicyFlag -policyName $polName -defaultSetting 0 -preferredSetting 1 -Reason "We recommend enabling this setting to ensure the appropriate processing mode for user Group Policy is applied at each logon." -Category "Profile Management Advanced Settings"
}

if ((-not $useFullContainer) -and (-not $isUsingMandatoryProfile)) {

  "-----------------------------------------------------"
  "- Checking Profile Management Advanced Settings     -"
  "- Step 7: Checking LoadRetries                      -"
  "-----------------------------------------------------"
  #
  # Let's check a policy - LoadRetries
  #
  $polName = "LoadRetries"
  $pol = GetPolicyGeneralSetting -policyName $polName
  $polName + " = " + $pol
  if ( $null -ne $pol ) {
    "*** Notice: Advanced settings > Number of retries when accessing locked files should not be set." | CaptureRecommendations -CheckTitle "Recommended Value Check: LoadRetries" -InfoType "Notice" -PolicyName $polName -Reason "We recommend that you not configure this setting unless instructed by Citrix authorized support personnel." -Category "Profile Management Advanced Settings"
  }
  else {
    "LoadRetries is correctly configured." | CaptureCheckResult -CheckTitle "LoadRetries" -PolicyName $polName -InfoType "Info"
  }


  "-------------------------------------------------------"
  "- Checking Profile Management File System Settings    -"
  "- Step 1: Checking SyncDirList                        -"
  "-------------------------------------------------------"
  #
  # Let's check a policy - SyncDirList
  #
  $polName = "SyncDirList"
  $pol = GetPolicyListSetting -regName $polName
  $polName + ":"
  $pol

  ValidateList -list $pol -policyName $polName -category "Profile Management File System Settings"

  $startMenuFound = $false
  $pol | foreach { 
    $item = $_
    switch ($item) {
      "AppData\Roaming\Microsoft\Windows\Start Menu" { $startMenuFound = $true }
    }
  }
  if ($startMenuFound -eq $false) { "recommend adding 'AppData\Roaming\Microsoft\Windows\Start Menu' to " + $polName }

  "-------------------------------------------------------"
  "- Checking Profile Management File System Settings    -"
  "- Step 2: Checking SyncFileList                       -"
  "-------------------------------------------------------"

  #
  # Let's check a policy - SyncFileList
  #
  $polName = "SyncFileList"
  $pol = GetPolicyListSetting -regName $polName
  $polName + ":"
  $pol

  ValidateList -list $pol -policyName $polName -category "Profile Management File System Settings"

  "-------------------------------------------------------"
  "- Checking Profile Management File System Settings    -"
  "- Step 3: Checking SyncExclusionListDir               -"
  "-------------------------------------------------------"
  #
  # Let's check a policy - SyncExclusionListDir
  #
  $polName = "SyncExclusionListDir"
  $polExclusionDir = GetPolicyListSetting -regName $polName
  $polName + ":"
  $pol

  ValidateList -list $polExclusionDir -policyName $polName -category "Profile Management File System Settings"

  #
  # while the above comparison is useful, we can also alert if we find specific apps or services
  # in the environment, and these need to be flagged with more emphasis
  #

  $accelerateFolderMirroringRecommended = $false

  if (($upmDisplayVersion -lt $upmLTSR1912CU2) -or ($dynamicConfigEnabled -eq $false)) {
    # CheckPackageAndUsrclassDat
    Test-StartMenu
  }
  else {
    "No special registry setting is needed." | CaptureCheckResult -InfoType "Info" -CheckTitle "SpecialFileExclusionSetting"
  }

  "-------------------------------------------------------"
  "- Checking Profile Management File System Settings    -"
  "- Step 4: Checking DefaultSyncExclusionListDir        -"
  "-------------------------------------------------------"
  #
  # Let's check a policy - DefaultSyncExclusionListDir
  #
  $polName = "DefaultSyncExclusionListDir"
  $polDefaultExclusionDir = GetPolicyDefaultListSetting -regName $polName
  $polName + ":"
  $pol

  ValidateList -list $polDefaultExclusionDir -policyName $polName -category "Profile Management File System Settings"


  "-------------------------------------------------------"
  "- Checking Profile Management File System Settings    -"
  "- Step 5: Checking SyncExclusionListFiles             -"
  "-------------------------------------------------------"
  #Excluded files: 2010-06-02;23:39:48.871;ERROR;CSCSOLUTIONS;jmacioci;0;3088;DeleteAnyFile: Deleting the file <C:\documents and settings\jmacioci\Local Settings\Application Data\VMware\hgfs.dat> failed with: The process cannot access the file because it is being used by another process.
  # See also: http://support.citrix.com/proddocs/topic/user-profile-manager-sou/upm-using-with-vmware.html
  #
  # Let's check a policy - SyncExclusionListFiles
  #

  $profileRoot = dir env: | foreach { if ($_.Name -eq "USERPROFILE") { $_.Value } }
  $charsToRemove = [string]$profileRoot.Length
  $charsToRemove = 1 + $charsToRemove
  $local = dir env: | foreach { if ($_.Name -eq "LOCALAPPDATA") { $_.Value } }
  $vmfile = Get-ChildItem $local -recurse -ea SilentlyContinue | foreach {
    if ($_.Name -eq "hgfs.dat") {
      $_.FullName
    }
  }

  if ($null -ne $vmfile) {
    $vmexclude = [string]$vmfile.Remove(0, $charsToRemove)
  }

  $polName = "SyncExclusionListFiles"
  $pol = GetPolicyListSetting -regName $polName
  $polName + ":"
  $pol

  ValidateList -list $pol -policyName $polName -category "Profile Management File System Settings"

  $vmwareToolsExclusionFound = $false
  if ($null -ne $vmexclude) {
    $pol | foreach { 
      $item = $_
      switch ($item) {
        $vmexclude { $vmwareToolsExclusionFound = $true }
      }
    }
  }

  if (($vmwareToolsExclusionFound -eq $false) -and ($hypervisor -eq "VMWare")) {
    if ($vmexclude -ne $null) {
      "File system > Excluded files: the list is missing ""$vmexclude"". Data loss might occur." | CaptureRecommendations -CheckTitle "ExclusionList" -InfoType "Notice" -PolicyName $polName -Reason "We recommend adding the file ""$vmexclude"" to this list. Otherwise, the profile will be locked during logoff, leading to data loss."
    }
  }
  else {
    "SyncExclusionListFile is correctly configured." | CaptureCheckResult -CheckTitle "SyncExclusionListFile" -PolicyName $polName -InfoType "Info"
  }

  "-------------------------------------------------------"
  "- Checking Profile Management File System Settings    -"
  "- Step 6: Checking LogonExclusionCheck                -"
  "-------------------------------------------------------"
  $LogonExclusionCheck = GetEffectivePolicyFlag -policyName "LogonExclusionCheck" -defaultSetting 0 -AsNumber
  switch ($LogonExclusionCheck.Value) {
    0 { "LogonExclusionCheck is configured to sync excluded files or folders from the user store to local profile" }
    1 { "LogonExclusionCheck is configured to ignore files and folders specified in exclusion list from the user store to local profile" }
    2 { "LogonExclusionCheck is configured to delete files and folders specified in exclusion list from the user store" }
  }

  "LogonExclusionCheck is correctly configured." | CaptureCheckResult -CheckTitle "LogonExclusionCheck" -PolicyName $polName -InfoType "Info"

  "-------------------------------------------------------"
  "- Checking Profile Management File System Settings    -"
  "- Step 7: Checking FoldersToMirror                    -"
  "-------------------------------------------------------"
  $polName = "MirrorFoldersList"
  $pol = GetPolicyListSetting -regName $polName
  $polName + ":"
  $pol

  ValidateList -list $pol -policyName $polName -category "Profile Management File System Settings"

  # If mirror folder is enabled, recommend to enable Accelerate folder mirroring
  if ($pol -and (-not $accelerateFolderMirroringRecommended)) {
    PreferPolicyFlag -policyName "AccelerateFolderMirroring" -defaultSetting 0 -preferredSetting 1 -Reason "We recommend enabling this policy to eliminate the need to copy the folders between the user store and local profiles and accelerate folder mirroring." -Category "Profile Management File System Settings"
  }

  if ($isCookieEnabled -eq 1) {
    # Calculate the preferred cookie folder and note it
    $cookieFolder = Get-IECookieFolder
    "Recommended Internet Explorer Cookie Folder = ("
    $cookieFolder
    ")"
    Replace-PolicyRecordProperty -PolicyName $polName -PropertyName PreferredValue -NewValue $cookieFolder

    $polcount = Get-ListItemCountRaw -list $pol
    if (($polcount -eq 0) -and ($pilot -eq $false)) { "File system > Enable folder mirroring: the setting ""disabled"" does not match the preferred setting ""enabled"". The Process Internet cookie file on logoff setting is enabled, but it does not take effect." | CaptureRecommendations -CheckTitle "Mirror Folder" -InfoType "Notice" -PolicyName $polName -Reason "For the Process Internet cookie file on logoff setting to take effect, we recommend enabling the Enable folder mirror setting and add at least the IE cookie folders "$cookieFolder" to the list." }

    #
    #As we need to add three folders in the mirror folder list for IE, Change the code logic
    if ($polcount -gt 0) {
      $polmatch = 0
      $cookieFolderCount = $cookieFolder.Length
      $cookieFolderHashTable = @{}
      $printForCookieFolder = ""
      for ($ix = 0; $ix -lt $cookieFolderCount; $ix++) {
        $Folder = ([array]$cookieFolder)[$ix]
        $cookieFolderHashTable.Add($ix, $Folder)
        $printForCookieFolder = $printForCookieFolder + "`r`n" + $Folder
      }
      for ($ix = 0; $ix -lt $polcount; $ix++) {
        $folder = ([array]$pol)[$ix]
        if ($cookieFolderHashTable -ne $null -and $cookieFolderHashTable.ContainsValue($folder)) {
          $polmatch += 1
        } 
      }
      if ($polmatch -ne $cookieFolderCount) {
        "File system > Enable folder mirroring: cookie folders are missing from the folder mirroring list. The Process Internet cookie file on logoff setting does not take effect." | CaptureRecommendations -CheckTitle "Mirror Folder" -InfoType "Notice" -PolicyName $polName -Reason "For the Process Internet cookie file on logoff setting to take effect, we recommend adding the following cookie folders to the folder mirroring list: $printForCookieFolder"
      }
      else {
        "Cookie folders ( $printForCookieFolder ) are added to FoldersToMirror." | CaptureCheckResult -CheckTitle "Mirror Folder" -PolicyName $polName -InfoType "Info"
      }
    }
  }
  #
  # Now compare the actual list with the recommended list
  #

  "--------------------------------------------------"
  "- Checking Profile Management Folder Redirection -"
  "--------------------------------------------------"

  $foldersRecommendedToBeRedirected = ""
  $foldersRecommendedNotToBeRedirected = ""

  function AdviseAgainstRedirecting ($isLocalFolder, $shortName, $longName, $location) {
    if (-not $isLocalFolder) { 
      $script:foldersRecommendedNotToBeRedirected += "$location, "
      #"*** consider not redirecting local folder $shortName ($location) to a network share."  | CaptureRecommendations -CheckTitle "Folder Redirection" -InfoType "Warning" -PolicyName "" -Reason "Citrix recommends that $longName should be kept inside the profile" 
    }
  }

  function ConsiderRedirecting ($isLocalFolder, $shortName, $longName, $location, $explanation) {
    if ($null -eq $explanation) { $explanation = "Citrix recommends that $longName should be redirected" }
    if ($isLocalFolder) { 
      #"*** consider redirecting local folder $shortName ($location) to a network share."  | CaptureRecommendations -CheckTitle "Folder Redirection" -InfoType "Warning" -PolicyName "" -Reason $explanation 
      $script:foldersRecommendedToBeRedirected += "$location, "
    }
  }

  $profileFolderHash = @{}   # this gives us a list of where all the profile folders are

  Get-ChildItem "HKCU:\Software\Microsoft\Windows\CurrentVersion\Explorer" | foreach {
    $k = $_
    if ($k.PSChildName -eq "User Shell Folders") {
      foreach ($folderName in $k.Property) {
        $folderTarget = $k.GetValue($folderName)
        if ($folderName -eq "{374DE290-123F-4565-9164-39C4925E467B}") { $folderName = "Downloads" }
        ReportEnvironment "Folders" $folderName $folderTarget
        $profileFolderHash[$folderName] = $folderTarget
        $isLocalFolder = $false
        if ($folderTarget.StartsWith($currentLocalProfile)) {
          $isLocalFolder = $true
        }
        switch ($folderName) {
          "AppData" {  }
          "Cache" {  }
          "Cookies" { AdviseAgainstRedirecting $isLocalFolder $folderName "Cookies" $folderTarget }
          "Desktop" {  }
          "Favorites" {  }
          "History" {  }
          "Local AppData" {  }
          "My Music" { ConsiderRedirecting $isLocalFolder $folderName "My Music" $folderTarget }
          "My Pictures" { ConsiderRedirecting $isLocalFolder $folderName "My Pictures" $folderTarget }
          "My Video" { ConsiderRedirecting $isLocalFolder $folderName "My Video" $folderTarget }
          "NetHood" {  }
          "Personal" { ConsiderRedirecting $isLocalFolder $folderName "My Documents" $folderTarget }
          "Programs" {  }
          "Recent" {  }
          "SendTo" {  }
          "Startup" {  }
          "Start Menu" {  }
          "Templates" {  }
          "Downloads" { if ($xenDesktopPresent -or $xenAppPresent) { ConsiderRedirecting $isLocalFolder $folderName "Downloads" $folderTarget "Citrix recommends that Downloads should be redirected when XenDesktop or XenApp Published Desktops are used" } }
          "PrintHood" {  }
        }
      }
      if ($foldersRecommendedToBeRedirected -ne "") {         
        "*** Consider redirecting local folder ($foldersRecommendedToBeRedirected) to a network share."  | CaptureRecommendations -CheckTitle "Folder Redirection" -InfoType "Info" -PolicyName "FolderRedirection" -Category "FolderRedirection"
      }
    
      if ($foldersRecommendedNotToBeRedirected -ne "") {         
        "*** Consider not redirecting local folder $shortName ($location) to a network share."  | CaptureRecommendations -CheckTitle "Folder Redirection" -InfoType "Info" -PolicyName "FolderRedirection" -Reason "Citrix recommends that $longName should be kept inside the profile"  -Category "FolderRedirection"
      }
    
      if (($foldersRecommendedToBeRedirected -eq "") -and ($foldersRecommendedNotToBeRedirected -eq "")) {
        "Folder redirection is correctly configured." | CaptureCheckResult -CheckTitle "FolderRedirection" -PolicyName "FolderRedirection" -InfoType "Info"
      }
    }
  }
}

"------------------------------------------------"
"- Checking Profile Management Log Settings     -"
"- Step 1: Checking Profile Management Logging  -"
"------------------------------------------------"
                    # FlagName                          Default
$logLevelFlags = @( @("LogLevelActiveDirectoryActions", 0),
                    @("LogLevelFileSystemActions",      0),
                    @("LogLevelFileSystemNotification", 0),
                    @("LogLevelInformation",            1),
                    @("LogLevelLogoff",                 1),
                    @("LogLevelLogon",                  1),
                    @("LogLevelPolicyUserLogon",        0),
                    @("LogLevelRegistryActions",        0),
                    @("LogLevelRegistryDifference",     0),
                    @("LogLevelUserName",               1),
                    @("LogLevelWarnings",               1)
)

$allFlagsSet = $true
for ($ix = 0; $ix -lt $logLevelFlags.Length; $ix++) {
  $logFlag = $logLevelFlags[$ix]
  $logFlagName = $logFlag[0]
  $logFlagDefault = $logFlag[1]
  $logSetting = GetEffectivePolicyFlag -policyName $logFlagName -defaultSetting $logFlagDefault
  if ($logSetting.Value -eq 0) {
    $allFlagsSet = $false
  }
}

"------------------------------------------------"
"- Checking Profile Management Log Settings     -"
"- Step 2: Checking LoggingEnabled              -"
"------------------------------------------------"
$polName = "LoggingEnabled"
$loggingEnabled = GetEffectivePolicyFlag -policyName $polName -defaultSetting 0

"------------------------------------------------"
"- Checking Profile Management Log Settings     -"
"- Step 3: Checking MaxLogSize                  -"
"------------------------------------------------"
$polName = "MaxLogSize"
$maxLogSizeDefault = 10485760
$maxLogSizeMcs = 7864320 # 7MB
if ($IsWorkstation) {
  $maxLogSizePreferred = 10485760
  $maxLogText = "workstation"
} else {
  $maxLogSizePreferred = 10485760
  $maxLogText = "server"
}
$logFileSize = GetEffectivePolicyFlag -policyName $polName -defaultSetting $maxLogSizeDefault -AsNumber
$lfs = $logFileSize.Value
if (($loggingEnabled.Value -eq 1) -and ($lfs -lt $maxLogSizePreferred)) {
  # convert to Megabyte to align with WEM web console.
  $lfs /= 1024 * 1024
  $maxLogSizeDefault /= 1024 * 1024
  #
  # warn that the logfile size might be on the small side
  "*** Notice: Log settings > Maximum size (MB): the setting ""$lfs (MB)"" is too small." | CaptureRecommendations -CheckTitle "Profile Management log size" -InfoType "Notice" -PolicyName $polName -Reason "In both server and desktop environments, a minimum size of $maxLogSizeDefault MB is recommended." -Category "Profile Management log size"
}else{
  "Profile Management log size is correctly configured." | CaptureCheckResult -CheckTitle "Profile Management log size" -InfoType "Info"
}

"------------------------------------------------"
"- Checking Profile Management Log Settings     -"
"- Step 4: Checking PathToLogFile               -"
"------------------------------------------------"

$sysRoot = $env:SystemRoot
$polName = "PathToLogFile"

$logPathPreferred = $logPathPreferred -replace "\\$",""

$logFilePath = GetEffectivePolicyFlag -policyName $polName -defaultSetting $logPathDefault
$expandedTemplatePath = Get-ProcessedPath -path $logFilePath.Value
"After expansion -> $expandedTemplatePath"
#$lfp = $logFilePath.Value -replace "\\$",""

if (($logFilePath -eq $null) -or ($logFilePath.Value -eq $null)) {
  "Path to log file is not configured. By default, log files are saved to the following path: %systemroot%\System32\LogFiles\UserProfileManager." | CaptureCheckResult -CheckTitle "PathToLogFile" -PolicyName $polName -InfoType "Info"
}
else {
  if (Test-Path $expandedTemplatePath) {
    "Path to log file is correctly configured." | CaptureCheckResult -CheckTitle "PathToLogFile" -PolicyName $polName -InfoType "Info"
  }
  else {
    "*** Error: Path to log file " + ($expandedTemplatePath) + " is not accessible." |  CaptureRecommendations -CheckTitle "PathToLogFile" -PolicyName $polName -Reason "Verify that the log file path is configured correctly and that there is no network connection problem." -Category "Profile Management Log Settings"
  }

  if (($loggingEnabled.Value -eq 1) -and ($logFileSize.Value -gt $maxLogSizeMcs) -and ($expandedTemplatePath -eq $mcsPersistentFolder)) {
    "*** Notice: We recommend not setting ""Path to log file"" to C:\Program Files\Citrix\PvsVM\Service\PersistedData." | CaptureRecommendations -CheckTitle "PathToLogFile" -InfoType "Notice" -PolicyName $polName -Reason "This path can hold up to 15 MB. It's unsuitable for a log file larger than 7 MB because the .bak log file doubles the space needed.
    " -Category "Profile Management Log Settings"
  }
}

if ((-not $useFullContainer) -and (-not $isUsingMandatoryProfile)) {

  "----------------------------------------------------------"
  "- Checking Profile Management Profile Handling Settings  -"
  "- Step 1: Check for locally-cached Profiles               -"
  "----------------------------------------------------------"

  $locallyCachedUpmProfiles = 0

  $ptusFromRegistry = "unset"

  $profileListKey = 'HKLM:\SOFTWARE\Microsoft\Windows NT\CurrentVersion\ProfileList'

  write-host "checking $profileListKey"
  if ($userSID -ne $null) {
    Get-ChildItem $profileListKey | foreach {
      $key = $_
      # $key.Name
      $pip = $key.GetValue("ProfileImagePath")
      # $pip
      $upmv = $key.GetValue("UPMProfileVersion")
      if ($key.PSChildName -eq $userSID) {
        # mark current user
        $markCurrentUser = '* '
      }
      else {
        $markCurrentUser = ''
      }
      if ($null -ne $upmv) {
        $upmusl = $key.GetValue("UPMUserStoreLocation")
        "$markCurrentUser$pip is a locally-cached Profile Management profile stored at $upmusl"
        $locallyCachedUpmProfiles++
        # and save the path if not already done
        if ($key.PSChildName -eq $userSID) {
          if ($userProfileRoot -eq "unset") {
            $userProfileRoot = $upmusl
          }
          $ptusFromRegistry = $upmusl
        }
      }
      else {
        #
        # current profile is not a Profile Management Profile - so see what it is
        # and whether it belongs to the current user
        # see what flags are set
        # 
        # Get the "State" flags
        $profileStateFlags = $key.GetValue("State")
        $profileStateFlagsHex = "0x" + $profileStateFlags.ToString("x4")
        switch ($profileStateFlags) {
          0 { "$markCurrentUser$pip is a Local Profile" }
          0x100 { "$markCurrentUser$pip is an Administrator Profile" }
          0x21c { $netPath = $key.GetValue("CentralProfile"); "$markCurrentUser$pip is a Roaming Profile stored at $netPath" }
          default {
            #
            # don't recognise this, so just decode the flags
            $flagString = ""
            $bitMask = 1
            for ($bitCount = 0; $bitCount -lt 32; $bitCount++) {
              if ($profileStateFlags -band $bitMask) {
                $bitMaskString = "0x" + $bitMask.ToString("x4")
                switch ($bitMask) {
                  0x0001 { $flagString = $flagString + "Mandatory ($bitMaskString), " }
                  0x0002 { $flagString = $flagString + "UseCache ($bitMaskString), " }
                  0x0004 { $flagString = $flagString + "NewLocal ($bitMaskString), " }
                  0x0008 { $flagString = $flagString + "NewCentral ($bitMaskString), " }
                  0x0010 { $flagString = $flagString + "UpdateCentral ($bitMaskString), " }
                  0x0020 { $flagString = $flagString + "DeleteCache ($bitMaskString), " }
                  0x0040 { $flagString = $flagString + "(notused) ($bitMaskString), " }
                  0x0080 { $flagString = $flagString + "GuestUser ($bitMaskString), " }
                  0x0100 { $flagString = $flagString + "AdminUser ($bitMaskString), " }
                  0x0200 { $flagString = $flagString + "NetReady ($bitMaskString), " }
                  0x0400 { $flagString = $flagString + "SlowLink ($bitMaskString), " }
                  0x0800 { $flagString = $flagString + "TempAssigned ($bitMaskString), " }
                  0x1000 { $flagString = $flagString + "(notused) ($bitMaskString), " }
                  0x2000 { $flagString = $flagString + "PartlyLoaded ($bitMaskString), " }
                  0x4000 { $flagString = $flagString + "BackupExists ($bitMaskString), " }
                  0x8000 { $flagString = $flagString + "ThisIsBak ($bitMaskString), " }
                }
              }
              $bitMask = 2 * $bitMask
            }
            $flagString = $flagString -replace ", $", ""  # remove trailing comma
            "$markCurrentUser$pip is an unknown Profile type, with State ($profileStateFlagsHex), flags ($flagString)"
          }
        }
      }
    }
  }

  #
  # Examine UserProfileOrigin.ini
  #
  $sourceProfileType = "unknown"
  if ($userProfileRoot -ne "unset") {
    $upoFilePath = $userProfileRoot + "\UserProfileOrigin.ini"
    Get-ChildItem $upoFilePath -ea SilentlyContinue | foreach {
      $upoFileObj = $_
      $creationTime = $upoFileObj.CreationTime
      cat $upoFilePath | foreach {
        $line = $_
        switch -regex ($line) {
          "OP([a-zA-Z]+)=(.*$)" {
            $sourceProfileType = $matches[1]
            $sourceProfile = $matches[2]
            if ($sourceProfile -eq "C:\Users\Default") {
              $sourceProfileType = "Default"
            }
            "Profile created from $sourceProfileType Profile using $sourceProfile at $creationTime"
          }
        }
      }
    }
  }

  #
  # The policy was originally designed for tidying up XenApp servers at session end.  But things aren't so simple...
  #
  # You should not delete locally cached profiles on logoff if...
  # * The (virtual) machine is volatile and will be destroyed on logoff, or
  # * Specifically, VDI-in-a-Box is in use, or
  # * The machine is "assigned" to one user (XenDesktop), or
  # * The profile is stored in a Personal vDisk, or
  # * The machine is dedicated for the use of a small number of users, and has a suitably-large persistent disk.
  #
  # You should delete locally cached profiles on logoff if...
  # * XenApp persistent.  Delete, to avoid the proliferation of stale profiles, or
  # * XenDesktop pooled.  Delete if the desktops can be recycled between users, rather than being created/destroyed on demand
  #

  #
  # $vmIsVolatile - set this if we are running on a hypervisor and are provisioned
  # But clear it if there is evidence that this machine was created well before logon
  # or has been re-used
  # Clues: big difference between session start and boot time
  #        other profiles present in file system
  #        registry entries for other profiles in HKLM
  # 
  $vmIsVolatile = $false

  if ($null -eq $vdiInfo) {
    #
    # need to make best guess at whether we have a volatile environment
    #
    "No XenDesktop WMI - assume volatile if provisioning in use"
    if ($prov -ne $physical) {
      $vmIsVolatile = $true
      #
      # get the start time for the session
      #
      $sess = Get-WmiObject -Class Win32_Session
      #  $logonTime = $sess.ConvertToDateTime($sess.StartTime)
      foreach ($s in $sess) {
        $logonTime = $s.ConvertToDateTime($s.StartTime)
      }
      $elapsed = $logonTime - $lastBoot
    }
  }
  else {
    #
    # we know from Santa Cruz WMI if we have a volatile environment
    #
    if ($vdiInfo.OSChangesPersist) {
      $vmIsVolatile = $false
    }
    else {
      $vmIsVolatile = $true
    }
  }

  $preferredProfileDisposition = $KEEP_ON_LOGOFF   # this is the default
  $profileDispositionReason = "We recommend disabling this setting because no special environment is detected."
  $deleteDelayReason = "We recommend that you have the profile deleted immediately on logoff to free disk space."

  # * The (virtual) machine is volatile and will be destroyed on logoff - don't delete
  if ($vmIsVolatile) {
    $profileDispositionReason = "We recommend disabling this setting because the machine is in a volatile environment and likely to be deleted."
    $profileDispositionReason
    $preferredProfileDisposition = $KEEP_ON_LOGOFF   # why waste time deleting - it'll be destroyed
    $preferredDeleteDelay = $PROFILE_DELETE_DEFERRED                # ProfileDeleteDelay
    $deleteDelayReason = "We recommend that you postpone profile deletion for a period of time in volatile environments where machines are likely to be deleted. Starting profile deletion immediately wastes IOPS for machines that will be deleted."
  }

  # * The machine is "assigned" to one user (XenDesktop), or
  if ($null -ne $vdiInfo) {
    if ($vdiInfo.IsAssigned) {
      $profileDispositionReason = "We recommend disabling this setting because this VDA machine is a dedicated desktop. "
      $profileDispositionReason
      $preferredProfileDisposition = $KEEP_ON_LOGOFF
      $deleteDelayReason = "We recommend that you retain profiles until the end of the session on dedicated desktops."
    }
  }
  else {
    # CAN'T DETECT THIS IF WMI NOT AVAILABLE
  }

  # * The machine is dedicated for the use of a small number of users, and has a suitably-large persistent disk.
  # CAN'T DETECT THIS

  # * XenApp persistent.  Delete, to avoid the proliferation of stale profiles, or
  if ($ostype -eq "Server") {
    if ($vmIsVolatile) {
      $profileDispositionReason = "We recommend enabling this setting because this VDA machine is an MCS or PVS provisioned machine."
      $profileDispositionReason
      $preferredProfileDisposition = $DELETE_ON_LOGOFF
    }
    else {
      $profileDispositionReason = "We recommend enabling this setting because this VDA machine is a physical machine."
      $profileDispositionReason
      $preferredProfileDisposition = $DELETE_ON_LOGOFF
    }
  }

  # * XenDesktop pooled.  Delete if the desktops can be recycled between users, rather than being created/destroyed on demand
  if (($ostype -eq "Workstation") -and ($locallyCachedUpmProfiles -gt 1)) {
    $profileDispositionReason = "We recommend enabling this setting because this VDA machine is a random desktop."
    $profileDispositionReason
    $preferredProfileDisposition = $DELETE_ON_LOGOFF
    if (-not $vmIsVolatile) {
      $deleteDelayReason = "For XenDesktop pooled desktops, we recommend that you have profiles deleted immediately on logoff to free disk space."
    }
  }

  # we've reached the end
  # do the test

  "Final recommendation: $profileDispositionReason"
  PreferPolicyFlag -policyName "DeleteCachedProfilesOnLogoff" -defaultSetting $KEEP_ON_LOGOFF -preferredSetting $preferredProfileDisposition -autoSetting $autoConfigSettings.DeleteCachedProfilesOnLogoff -Reason $profileDispositionReason -Category "Profile Management Profile Handling Settings"

  "----------------------------------------------------------"
  "- Checking Profile Management Profile Handling Settings  -"
  "- Step 2: Checking migration settings                    -"
  "----------------------------------------------------------"
  #
  # Let's check a policy - MigrateWindowsProfilesToUserStore
  #
  $polName = "MigrateWindowsProfilesToUserStore"
  $pol = GetPolicyGeneralSetting -policyName $polName
  $polName + " = " + $pol
  $migrateProfiles = $pol

  if ( $pol) {
    switch ( $pol ) {
      1 { "MigrateWindowsProfilesToUserStore is set to All" }
      2 { "MigrateWindowsProfilesToUserStore is set to Local" }
      3 { "MigrateWindowsProfilesToUserStore is set to Roaming" }
      4 { "MigrateWindowsProfilesToUserStore is set to None" }
      default { "*** MigrateWindowsProfilesToUserStore is not configured and the default value is `None`." }
    }
  }
  else {
    "*** MigrateWindowsProfilesToUserStore is not configured and the default value is `None`."
  }

  if ($roaming -and (($pol -eq 4) -or ($pol -eq 2))) {
    if ($pol -eq 4) {
      "*** Notice: Profile handling > Type of user profiles to migrate if the user store is empty: the effective setting ""None"" does not match the preferred setting ""Roaming"" or ""Local and Roaming"". The Windows profile is in use but cannot be migrated to the user store." | CaptureRecommendations -CheckTitle "Migrate Options" -InfoType "Notice" -PolicyName $polName -Reason "To enable the migration, we recommend that you change this setting to the preferred setting." -Category "Profile Management Profile Handling Settings"
    }
    else {
      "*** Notice: Profile handling > Type of user profiles to migrate if the user store is empty: the effective setting ""Local"" does not match the preferred setting ""Roaming"" or ""Local and Roaming"". The Windows profile is in use but cannot be migrated to the user store." | CaptureRecommendations -CheckTitle "Migrate Options" -InfoType "Notice" -PolicyName $polName -Reason "To enable the migration, we recommend that you change this setting to the preferred setting." -Category "Profile Management Profile Handling Settings"
    }
  }
  else {
    "`MigrateWindowsProfilesToUserStore` is correctly configured." | CaptureCheckResult -CheckTitle "Migrate Options" -InfoType "Info"
  }
  if ($sourceProfileType -eq "Local") {
    if ($firstLogon) {
      "Profile origin = $sourceProfileType; First Logon = $firstLogon" + ": Profile was created from a Local profile on this machine"
    }
    else {
      "Profile origin = $sourceProfileType; First Logon = $firstLogon" + ": Profile was created from a Local profile on an unknown machine."
    }
  }
  else {
    "Profile origin = $sourceProfileType; First Logon = $firstLogon" + ": Profile was not created from a Local profile."
  }
  
  "----------------------------------------------------------"
  "- Checking Profile Management Profile Handling Settings  -"
  "- Step 3: Checking LocalProfileConflictHandling          -"
  "----------------------------------------------------------"
  #
  # if the current profile is a temporary profile and the user store contains a Profile Management profile
  # then we have a profile locking problem,
  # if the current profile is a local profile  and the user store contains a Profile Management profile
  # then we have triggered a local profile conflict, resolved by keeping the local profile
  # if we have a Profile Management profile and there is also a profile in c:\Users\<user>.upm.backup
  # then we have triggered a local profile conflict, resolved by backing up the local profile
  # we cannot tell if a local profile has been silently obliterated
  #
  $localProfileConflicts = GetEffectivePolicyFlag -policyName "LocalProfileConflictHandling" -defaultSetting 1 -AsNumber
  switch ($localProfileConflicts.Value) {
    1 { "LocalProfileConflictHandling is set to Use the Local Profile" }
    2 { "LocalProfileConflictHandling is set to Delete the Local Profile" }
    3 { "LocalProfileConflictHandling is set to Rename the Local Profile" }
  }
}

if (-not $useFullContainer) {
  "----------------------------------------------------------"
  "- Checking Profile Management Profile Handling Settings  -"
  "- Step 4: Checking TemplateProfilePath                   -"
  "----------------------------------------------------------"
  $polName = "TemplateProfilePath"
  $pol = GetPolicyGeneralSetting -policyName $polName
  $polName + " = " + $pol
  if (($upmmajor -ge 5) -or (($upmmajor -eq 4) -and ($upmminor -gt 0))) {
    $expandedTemplatePath = Get-ProcessedPath -path $pol
    "After expansion -> $expandedTemplatePath"
  }
  else {
    $expandedTemplatePath = $pol
  }
  
  $templateIsValid = $true
  switch -wildcard ($expandedTemplatePath) {
    '*ntuser.dat' { 
      $templateIsValid = $false
      '*** Error: Profile handling > Template profile path: Incorrect path setting. The setting cannot contain a file name.' | CaptureRecommendations -CheckTitle "Template Profile Path" -PolicyName $polName -Reason "The setting must contain only the path to the template profile and not contain any file name." -Category "Profile Management Profile Handling Settings" 
    } 
    '\\*' { 'Template Profile path appears to be a fileshare or DFS namespace.' }
    '* ' { 
      $templateIsValid = $false
      '*** Notice: Profile handling > Template profile path: the setting contains trailing spaces.' | CaptureRecommendations -CheckTitle "Template Profile Path" -InfoType "Notice" -PolicyName $polName -Reason "We recommend removing trailing spaces. Trailing spaces can lead to unpredictable and hard-to-diagnose behaviors in Profile Management." -Category "Profile Management Profile Handling Settings"
    }
  }
  
  if ($templateIsValid) {
    # validate the path
    if (($null -eq $expandedTemplatePath) -or ($expandedTemplatePath -eq "")) {
      "Template path is not configured. New users will receive an initial profile copied from the Windows default profile." | CaptureCheckResult -CheckTitle "Template Profile Path" -InfoType "Info" #Treat it as a normal behavior. Do not show warning.
    }
  }
  
  #
  # if we have a template path, it can either be used as a template
  # or as a mandatory profile
  if ($templateIsValid) {
    "Template path has been correctly configured" | CaptureCheckResult -CheckTitle "Template Profile Path" -InfoType "Info"
    if ($isUsingMandatoryProfile) {
      #
      # mandatory, therefore neither TemplateProfileOverridesLocalProfile nor 
      # TemplateProfileOverridesRoamingProfile should be set
      AssertFlagNotSet -policyName "TemplateProfileOverridesLocalProfile" -Reason "If used as the Citrix mandatory profile, the template profile cannot be used to override the local profile. Keep only one of these two settings enabled." -Category "Profile Management Profile Handling Settings"
      AssertFlagNotSet -policyName "TemplateProfileOverridesRoamingProfile" -Reason "If used as the Citrix mandatory profile, the template profile cannot be used to override the roaming profile. Keep only one of these two settings enabled." -Category "Profile Management Profile Handling Settings"
    }
    else {
      #
      # template profile is being used as a template profile
      # either TemplateProfileOverridesLocalProfile or
      # TemplateProfileOverridesRoamingProfile (or both) can be set
      # all combinations are meaningful
      # (if we were picky, we could object to roaming profile option if there's no roaming profile)
      "TemplateProfileOverridesLocalProfile", "TemplateProfileOverridesRoamingProfile" | foreach {
        $polName = $_
        $pol = GetPolicyGeneralSetting -policyName $polName
        $polName + " = " + $pol
      }
    }
  }
  else {
    "Template profile has not been set up" | CaptureCheckResult -CheckTitle "TemplateProfileIsMandatory" -InfoType "Info"
    "Template profile has not been set up" | CaptureCheckResult -CheckTitle "TemplateProfileOverridesLocalProfile" -InfoType "Info"
    "Template profile has not been set up" | CaptureCheckResult -CheckTitle "TemplateProfileOverridesRoamingProfile" -InfoType "Info"
  }
}

if ((-not $useFullContainer) -and (-not $isUsingMandatoryProfile)) {
  "-------------------------------------------------"
  "- Checking Profile Management Registry Settings -"
  "- Step 1: Checking InclusionListRegistry        -"
  "-------------------------------------------------"
  #
  # Let's check a policy - InclusionListRegistry
  #
  $polName = "InclusionListRegistry"
  $pol = GetPolicyListSetting -regName $polName
  $polName + ":"
  $pol
  
  ValidateList -list $pol -policyName $polName -category "Profile Management Registry Settings"
  
  "-------------------------------------------------"
  "- Checking Profile Management Registry Settings -"
  "- Step 2: Checking ExclusionListRegistry         -"
  "-------------------------------------------------"
  #
  # Let's check a policy - ExclusionListRegistry
  #
  $polName = "ExclusionListRegistry"
  $polExclusionReg = GetPolicyListSetting -regName $polName
  $polName + ":"
  $pol
  
  ValidateList -list $polExclusionReg -policyName $polName -category "Profile Management Registry Settings"
  
  #Check special Registry settings for Win10/Win2016/Win2019
  if (($winver.StartsWith("Win10")) -or ($winver -eq "Win2016") -or ($winver -eq "Win2019")) {
    CheckSpeechOneCore
  }
  else {
    "No special registry setting is needed." | CaptureCheckResult -InfoType "Info" -CheckTitle "SpecialExclusionRegKeyCheck"
  }
  
  "-------------------------------------------------"
  "- Checking Profile Management Registry Settings -"
  "- Step 3: Checking DefaultExclusionListRegistry -"
  "-------------------------------------------------"
  #
  # Let's check a policy - DefaultExclusionListRegistry
  #
  $polName = "DefaultExclusionListRegistry"
  $polDefaultExclusionReg = GetPolicyDefaultListSetting -regName $polName
  $polName + ":"
  $pol
  
  ValidateList -list $polDefaultExclusionReg -policyName $polName -category "Profile Management Registry Settings"
  
  "-------------------------------------------------"
  "- Checking Profile Management Registry Settings -"
  "- Step 4.Checking NTUSER.DAT backup             -"
  "-------------------------------------------------"
  $polName = "LastKnownGoodRegistry"
  PreferPolicyFlag -policyName $polName -defaultSetting 1 -preferredSetting 1 -Reason "We recommend enabling this setting. With it enabled, Profile Management saves the last known good copy of the NTUSER.DAT file. This copy enables automatic rollback if the NTUSER.DAT file is corrupted." -Category "Profile Management Registry Settings"
  $LastKnownGoodRegistry = Get-PolicyRecordProperty -PolicyName $polName -PropertyName EffectiveValue
  
  "---------------------------------------"
  "- Checking Streaming Settings         -"
  "- Step 1: Checking Profile Streaming  -"
  "---------------------------------------"

  # We're recommending to enable profile streaming
  PreferPolicyFlag -policyName "PSEnabled" -defaultSetting $PROFILES_COPIED_IN_FULL -preferredSetting $PROFILES_STREAMED_ON_DEMAND -autoSetting $autoConfigSettings.PSEnabled -Reason "We recommend enabling this setting unless offline profile support is enabled." -InfoType "Warning" -Category "Streaming Settings"

  # We're recommending to enable profile streaming for folders
  PreferPolicyFlag -policyName "PSForFoldersEnabled" -defaultSetting $PROFILES_COPIED_IN_FULL -preferredSetting $PROFILES_STREAMED_ON_DEMAND -autoSetting $autoConfigSettings.PSForFoldersEnabled -Reason "We recommend enabling this setting unless offline profile support is enabled." -InfoType "Warning" -Category "Streaming Settings"

  "--------------------------------------"
  "- Checking Streaming Settings         -"
  "- Step 2: Checking PSAlwaysCache      -"
  "---------------------------------------"
  
  
  
  "-----------------------------------------------------------------------"
  "- Checking Streaming Settings                                         -"
  "- Step 3. Compare Streaming exclusion list with Citrix recommendation -"
  "-----------------------------------------------------------------------"
  $polName = "StreamingExclusionList"
  $pol = GetPolicyListSetting -regName $polName
  $exclusionAnalysis = CompareLists -preferredList $recommendedStreamExclusionList -specimenList $pol
  
  $missingItems = ""
  
  $exclusionAnalysis | foreach {
    $item = $_
    $diff = ($item.Difference).PadRight(10)
    $lineItem = $item.LineItem
    "$diff" + ": $lineItem"  
    switch ($item.Difference) {
      "Missing" {
        # we only care about missing items (but we care very much!)
        $missingItems += "$lineItem, "
      }
    }  
    Add-PolicyListRecord -PolicyName $polName -ProfileType $item.ComparisonType -DifferenceType $item.Difference -Value $item.LineItem
  }
  if ($missingItems -ne "") {
    "*** Error: Streamed user profiles > Excluded folders list: the list is missing ""$missingItems""." | CaptureRecommendations -CheckTitle "Streaming Exclusion list" -PolicyName $polName -Reason "We recommend adding the missing items to the list. Otherwise, users might be blocked on logon." -Category "Streaming Settings"
  }
  else {
    "StreamingExclusionList is correctly configured." | CaptureCheckResult -CheckTitle "StreamingExclusionList" -PolicyName $polName -InfoType "Info"
  }
  
  "-----------------------------------------"
  "- Checking Streaming Settings           -"
  "- Step 4: Checking PSPendingLockTimeout -"
  "-----------------------------------------"
  #
  # Let's check a policy - PSPendingLockTimeout
  #
  $polName = "PSPendingLockTimeout"
  $pol = GetPolicyGeneralSetting -policyName $polName
  $polName + " = " + $pol
  if ( $null -ne $pol ) {
    "*** Notice: Streamed user profiles > Set timeout for files in pending area when user store remains locked > Timeout (days): the setting should not be configured." | CaptureRecommendations -CheckTitle "Recommended Value Check: PSPendingLockTimeout" -InfoType "Notice" -PolicyName $polName -Reason "We recommend that you not configure this setting unless instructed by Citrix authorized support personnel." -Category "Streaming Settings"
  }
  else {
    "PSPendingLockTimeout is correctly set to `Not configured`." | CaptureCheckResult -CheckTitle "PSPendingLockTimeout" -PolicyName $polName -InfoType "Info"
  }
}


########################################################################################
#
# Miscellaneous checks
#
# Anything not covered - support list topics?
#
########################################################################################

"----------------------------------------"
"- Miscellaneous checks                 -"
"- Checking environment inconsistencies -"
"----------------------------------------"
filter sumProfile ($RootFolder) {
  begin {
    $totalFiles = 0
    $totalFolders = 0
    $totalSize = 0
  }
  process {
    if ($_.PSIsContainer) {
      $totalFolders++
    } else {
      $totalFiles++
      $totalSize += $_.Length
    }
  }
  end {
    $ts = ConvenientBytesString $totalSize
    ReportEnvironment "UpmProf" "Profile Location" $RootFolder
    ReportEnvironment "UpmProf" "Total Folders"    $totalFolders
    ReportEnvironment "UpmProf" "Total Files"      $totalFiles
    ReportEnvironment "UpmProf" "Total Size"       $ts
  }
}

if ($versionMajor -gt 5) {
  $userprofile = dir env: | foreach { if ($_.Name -eq "USERPROFILE") { $_.Value } }
  $mpc = Get-MountPointCount -path $userprofile
  if($ProfileSumCheck)
  {
  Get-ChildItem $userprofile -Recurse -Force -ea SilentlyContinue | sumProfile -RootFolder $userprofile
  }
  if ($mpc -ne 1) {
    "*** Error: Profile Management cannot work because the profile folder $userprofile is not located on the system volume." | CaptureRecommendations -CheckTitle "Profile volume" -PolicyName "" -Reason "Profile Management can work only when the profile folder is located on the system volume." -Category "Miscellaneous"
  } else {
    #
    # report on disk usage
    $drive = $userprofile.Substring(0,2)
    Get-WmiObject Win32_LogicalDisk | where-object {$_.DeviceID -eq $drive} | foreach {
      $driveName = $_.DeviceID
      $volName = $_.VolumeName
      $freespace = ConvenientBytesString -bytes $_.FreeSpace
      $size = ConvenientBytesString -bytes $_.Size
      $freePercent = [math]::Floor(($_.FreeSpace * 100) / $_.Size)
      $driveStatus = "Profile Drive status - $driveName (Volume Name ""$volName"") Size = $size, Free Space = $freespace ($freePercent%)."
      ReportEnvironment "UpmProf" "Profile Drive"       $driveName
      ReportEnvironment "UpmProf" "Volume Name"         $volName
      ReportEnvironment "UpmProf" "Size"                $size
      ReportEnvironment "UpmProf" "Free Space"          $freespace
      ReportEnvironment "UpmProf" "Free Space %"        $freePercent
      # 15% was a bit arbitrary - making tpc a command-line parameter and rename to ProfileDriveThresholdPercent
      if ($freePercent -lt $ProfileDriveThresholdPercent) {
        if ($ostype -eq "Server") {
          "*** Warning: " + $driveStatus | CaptureRecommendations -CheckTitle "Storage space of local disk" -InfoType "Warning" -PolicyName "" -Reason "The drive where the profile folder is located has less than $ProfileDriveThresholdPercent% free disk space. Consider increasing the size of the drive, removing old and unused profiles using a tool such as delprof2.exe, or enabling DeleteCachedProfilesOnLogoff." -Category "Miscellaneous"
        } 
      }else{
        "Profile Quota status is good." | CaptureCheckResult -CheckTitle "Storage space of local disk" -PolicyName "" -InfoType "Info"
      }
      }
    }
} else {
  "*** Warning: Unable to check for volume mount issues on Windows XP/2003." | CaptureRecommendations -CheckTitle "Storage space of local disk" -InfoType "Warning" -PolicyName "" -Reason "Windows XP and Windows Server 2003 do not support the necessary WMI classes." -Category "Miscellaneous"
}

#
# TODO
#
# Add functionality to take a path and check whether (net) it is included or excluded
# This makes it easy to add one-liners for specific support issues
#
# Check for RingCube and suggest disabling Profile Streaming and Deleted Locally Cached Profiles on Logoff - DONE
#
# Check for whitespace and = at the end of include/exclude folders - DONE
#
# Check for use of extended synchronization on XA - not supported, see
#   http://forums.citrix.com/thread.jspa?threadID=294190&tstart=0
#
# Check for use of %appdata% in paths - not supported, see
#   http://forums.citrix.com/thread.jspa?threadID=294190&tstart=0
#
# Check for 8.3 filename support disabled - DONE
#
$longPathCheck = $false        # only care if XP / 2k3 AND 'long' install path chosen AND 8.3 file support disabled
$x8dot3disabled = 0
$regPath = "HKLM:System\CurrentControlSet\Control\FileSystem"
$valname = "NtfsDisable8Dot3NameCreation"
Get-ItemProperty $regPath -name $valName -ErrorAction SilentlyContinue | foreach {
  $x8dot3disabled = $_.NtfsDisable8Dot3NameCreation
}
"NtfsDisable8Dot3NameCreation = " + $x8dot3disabled
if ($versionMajor -eq 5) {
  switch ($x8dot3disabled) {
  0 { "8Dot3 filename support enabled - supported case for UPM" }
  1 { 
      #"*** Error  8Dot3 file name support disabled - you must not install Profile Management to a 'long' path." | CaptureRecommendations  -CheckTitle "8Dot3 Filename Support" -PolicyName "" -Reason "On Windows XP and Windows Server 2003, Profile Management requires that each component in its installation path conforms to 8.3 filename limitations, otherwise Profile Management is unable to detect logons." -Category "Miscellaneous"
      $longPathCheck = $true
    }
  default { "*** Error: 8Dot3 file name support is not enabled on all volumes. For Profile Management to work, 8Dot3 filename support must be enabled on all volumes." | CaptureRecommendations -CheckTitle "8Dot3 Filename Support" -PolicyName "" -Reason "On Windows XP and Windows Server 2003, Profile Management requires that each component in its installation path conforms to 8.3 filename limitations. Otherwise, Profile Management is unable to detect logons. UpmConfigCheck is unable to verify this condition for all drives in your system." -Category "Miscellaneous"}
  }
} else {
  "On current Windows version, Profile Management does not require any specific setting for NtfsDisable8Dot3NameCreation." | CaptureCheckResult -CheckTitle "8Dot3 Filename Support" -InfoType "Info"
}

#
# Check for valid install path (normally c:\Program Files\Citrix\User Profile Manager )
#
$pathOk = $true
if ($longPathCheck) {
  $UpmFolderList = $UPMBase.Split('`\',[stringsplitoptions]::RemoveEmptyEntries)
  for ($ix = 0; $ix -lt $UpmFolderList.Length; $ix++) {
    if (($UpmFolderList[$ix]).Length -gt 8) {
      $pathOk = $false
    }
  }
  if ($pathOk -eq $false) {
    "*** Error: One or more folder names in the Profile Management installation folder ""$UPMBase"" exceed 8 chars." | CaptureRecommendations -CheckTitle "Installation Path Check" -PolicyName "" -Reason "Reinstall Profile Management to a path where all folder names are 8 chars or less." -Category "Miscellaneous"
  }else{
    "Profile Management installation path is correctly configured." | CaptureCheckResult -CheckTitle "Installation Path Check" -InfoType "Info"
  }
}else{
    "Profile Management installation path is correctly configured." | CaptureCheckResult -CheckTitle "Installation Path Check" -InfoType "Info"
}

#############################################################

########################################################################################
#
# MS New Teams checks#
#
########################################################################################
$teamsIsInstalled = Test-NewTeamsInstallation
if ($CheckNewTeams -and $teamsIsInstalled) {
  "----------------------------------------"
  "- New Teams checks                     -"
  "- Checking New Teams related settings  -"
  "----------------------------------------"

  
  $isContainerBased = GetPolicyListSetting -regName "ProfileContainer"
  if ($isContainerBased) {
    # Container-based solution
    $ProfileContainerLocalCacheEnabled = GetEffectivePolicyFlag -policyName "ProfileContainerLocalCache" -defaultSetting 0 -AsNumber
    if ($ProfileContainerLocalCacheEnabled.Value -eq 0) {
      # If the Enable local caching for profile containers policy is disabled

      # Don't follow Microsoft's recommendations to exclude folders.
      # Ensure that no folders under AppData\Local\Packages\MSTeams_8wekyb3d8bbwe are in the container exclusion list.
      $ProfileContainerExclusionFolderList = GetPolicyListSetting -regName "ProfileContainerExclusionListDir"
      $ProfileContainerExclusionFolderList | ForEach-Object {
        if ($_.Contains("AppData\Local\Packages\MSTeams_8wekyb3d8bbwe")) {
          "Profile Management > Profile Container > Exclusion list: the folder ""$($_)"" is in the exclusion list, but it is required for Microsoft Teams to function correctly." | CaptureRecommendations -CheckTitle "Profile container exclusion list" -PolicyName "ProfileContainerExclusionListDir" -Reason "We recommend removing the folder from the exclusion list." -Category "Profile container" -InfoType "Notice"
          
        }
      }

      # Ensure that all folders in the list aren't included in the container exclusion list.
      $newTeamsAppDataFolderList | ForEach-Object {
        if ($ProfileContainerExclusionFolderList -contains $_) {
          "Profile Management > Profile Container > Exclusion list: the folder ""$($_)"" is in the exclusion list, but it is required for Microsoft Teams to function correctly." | CaptureRecommendations -CheckTitle "Profile container exclusion list" -PolicyName "ProfileContainerExclusionListDir" -Reason "We recommend removing the folder from the exclusion list." -Category "Profile container" -InfoType "Notice"
        }
      }
    }elseif ($ProfileContainerLocalCacheEnabled.Value -eq 1) {
      # If the Enable local caching for profile containers policy is enabled
      # Follow Microsoft's recommendations to exclude folders.
      Test-RecommendedFolderExclusionForNewTeams

      # Test for the Folders to mirror policy
      Test-FoldersToMirrorForNewTeams
    }


  }else{
    # File-based solution
    # Test for the Folders to mirror policy
    Test-FoldersToMirrorForNewTeams

    $accelerateFolderMirroringPolicy = GetPolicyGeneralSetting -policyName "AccelerateFolderMirroring"
    if ($accelerateFolderMirroringPolicy -eq 0) {
      # If the Accelerate folder mirroring policy is disabled
      # Follow Microsoft's recommendations to exclude folders.
      Test-RecommendedFolderExclusionForNewTeams
    }else {
      # If the Accelerate folder mirroring policy is enabled

      # Don't follow Microsoft's recommendations to exclude folders.
      # Ensure that no folders under AppData\Local\Packages\MSTeams_8wekyb3d8bbwe are in the exclusion list if AppData\Local\Packages or AppData\Local\Packages\MSTeams_8wekyb3d8bbwe is included in the Folders to mirror policy.
      $polName = "MirrorFoldersList"
      $pol = GetPolicyListSetting -regName $polName
      if ($pol -contains "AppData\Local\Packages" -or $pol -contains "AppData\Local\Packages\MSTeams_8wekyb3d8bbwe") {
        $ProfileExclusionFolderList = GetPolicyListSetting -regName "SyncExclusionListDir"
        $ProfileExclusionFolderList | ForEach-Object {
          if ($_.Contains("AppData\Local\Packages\MSTeams_8wekyb3d8bbwe")) {
            "File system > Excluded folders: the folder ""$($_)"" is in the exclusion list, but it is required for Microsoft Teams to function correctly." | CaptureRecommendations -CheckTitle "Exclusion list" -PolicyName "SyncExclusionListDir" -Reason "We recommend removing the folder from the exclusion list." -Category "Profile Management File System Settings" -InfoType "Notice"
          }
        }
      }

      # Ensure that all folders in the list, except those folders under AppData\LocalLow, aren't included in the exclusion list.
      $newTeamsAppDataFolderList | ForEach-Object {
        if ($ProfileExclusionFolderList -contains $_) {
          if (-not $_.Contains("AppData\LocalLow")) {
            "File system > Excluded folders: the folder ""$($_)"" is in the exclusion list, but it is required for Microsoft Teams to function correctly." | CaptureRecommendations -CheckTitle "Exclusion list" -PolicyName "SyncExclusionListDir" -Reason "We recommend removing the folder from the exclusion list." -Category "Profile Management File System Settings" -InfoType "Notice"
          }
        }
      }
    }

    # Check Profile streaming exclusion list - directories policy
    Test-PsExclusionFolderListForNewTeams
  }

}


#############################################################
$regCount = 0
$gpoCount = 0
$hdxCount = 0
""
"The following policies are found in the Policy registry."
"========================================================"
foreach ($pair in $strPoliciesDetected.GetEnumerator()) {
  $pair.Key # + ": " + $pair.Value
  $regCount++
  $gpoCount++
}

$iniCount = 0
""
"The following policies are found in the HDX Policy registry."
"============================================================"
foreach ($pair in $strHDXPoliciesDetected.GetEnumerator()) {
  $pair.Key # + ": " + $pair.Value
  $regCount++
  $hdxCount++
}

$iniCount = 0
""
"The following policies are found in the INI file."
"================================================="
foreach ($pair in $strIniLinesDetected.GetEnumerator()) {
  $pair.Key # + ": " + $pair.Value
  $iniCount++
}

if (($gpoCount -gt 0) -and ($hdxCount -gt 0)) {
  "*** Warning: Configuration mixes GPO policies and HDX policies." | CaptureRecommendations -CheckTitle "Policy Source" -InfoType "Warning" -PolicyName "" -Reason "We recommend that you choose only one of the following locations to configure Profile Management: HDX policies in Citrix Studio, or GPO in Active Directory." -Category "Miscellaneous"
}else{
  "Policy source is correct." | CaptureCheckResult -CheckTitle "Policy Source" -InfoType "Info"
}

"--------------------------------------------------------"
"- Miscellaneous checks                                 -"
"- Checking access status of related Windows Event Logs -"
"--------------------------------------------------------"
$securityEventLog = [EventLog]::new()
$securityEventLog.eventLogName = "Security"
$securityEventLog.eventLogDisplayName = "Windows Logs\Security"
# $securityEventLog.evtxPath = "C:\Windows\System32\winevt\Logs\Security.evtx"
$securityEventLog.regPath = "HKLM:\SYSTEM\CurrentControlSet\Services\EventLog\Security"

$groupPolicyEventLog = [EventLog]::new()
$groupPolicyEventLog.eventLogName = "Microsoft-Windows-GroupPolicy/Operational"
$groupPolicyEventLog.eventLogDisplayName = "Microsoft-Windows-GroupPolicy"
# $groupPolicyEventLog.evtxPath = "C:\Windows\System32\winevt\Logs\Microsoft-Windows-GroupPolicy%4Operational.evtx"
$groupPolicyEventLog.regPath = "HKLM:\SOFTWARE\Microsoft\Windows\CurrentVersion\WINEVT\Channels\Microsoft-Windows-GroupPolicy/Operational"

CheckEventLogPermission -eventLog $securityEventLog
CheckEventLogPermission -eventLog $groupPolicyEventLog 

"------------------------------------"
"- Miscellaneous checks             -"
"- Checking Logon Timings -----------"
"------------------------------------"
try{
  $logonTimings = Get-WmiObject -Namespace ROOT\citrix\profiles\metrics -class LogonTimings -ErrorAction "SilentlyContinue"
}
catch{
  "No LogonTimings WMI interface available"
}
$logonTimingsList = New-Object System.Collections.ArrayList

if ($null -ne $logonTimings) {
  $logonTimings | ForEach-Object {
    ReportEnvironment "LogonTimings" "SessionId"   $_.SessionId
    ReportEnvironment "LogonTimings" "DesktopReady"   $_.DesktopReady
    ReportEnvironment "LogonTimings" "GroupPolicyStart"   $_.GroupPolicyStart
    ReportEnvironment "LogonTimings" "GroupPolicyComplete"   $_.GroupPolicyComplete
    ReportEnvironment "LogonTimings" "LogonScriptsStart"   $_.LogonScriptsStart
    ReportEnvironment "LogonTimings" "LogonScriptsComplete"   $_.LogonScriptsComplete
    ReportEnvironment "LogonTimings" "ProfileLoadStart"   $_.ProfileLoadStart
    ReportEnvironment "LogonTimings" "ProfileLoaded"   $_.ProfileLoaded
    ReportEnvironment "LogonTimings" "UPMStart"   $_.UPMStart
    ReportEnvironment "LogonTimings" "UserInitStart"   $_.UserInitStart
    ReportEnvironment "LogonTimings" "UserInitComplete"   $_.UserInitComplete

    $logonTimingsStruct = New-Object LogonTimings
    $logonTimingsStruct.SessionId = $_.SessionId
    $logonTimingsStruct.DesktopReady = $_.DesktopReady
    $logonTimingsStruct.GroupPolicyStart = $_.GroupPolicyStart
    $logonTimingsStruct.GroupPolicyComplete = $_.GroupPolicyComplete
    $logonTimingsStruct.LogonScriptsStart = $_.LogonScriptsStart
    $logonTimingsStruct.LogonScriptsComplete = $_.LogonScriptsComplete
    $logonTimingsStruct.ProfileLoadStart = $_.ProfileLoadStart
    $logonTimingsStruct.ProfileLoaded = $_.ProfileLoaded
    $logonTimingsStruct.UPMStart = $_.UPMStart
    $logonTimingsStruct.UserInitStart = $_.UserInitStart
    $logonTimingsStruct.UserInitComplete = $_.UserInitComplete

    $script:logonTimingsList.Add($logonTimingsStruct) > $null
  }
  $logonTimingsList | ConvertTo-Json | CaptureCheckResult -CheckTitle "LogonTimings" -PolicyName "" -InfoType "Info"
}

""
"The following policies are checked. Default settings are applied."
"================================================="
foreach ($pair in $strDefaultsDetected.GetEnumerator()) {
  $pair.Key # + ": " + $pair.Value
}

""
"The following policies are not checked."
"======================================="
for ($pix = 0; $pix -lt $policyDb.Length; $pix++) {
  if (!(StringIsNullOrWhitespace($CEIPFilePath))) {
      
  }
  if ($policyDb[$pix].Origin -eq "Not Checked") {
    $policyDb[$pix].Name
  }
}

if (!(StringIsNullOrWhitespace($CEIPFilePath))){
  Export-CEIPData($CEIPFilePath)
}

function Write-Wrapped ($str) {
  switch -regex ($str) {
  "^\*\*\* " { $str }
  "^(    notes: )(.*)$" {
      $prefix = $matches[1]
      $rest = $matches[2]
      $prefixlen = $prefix.Length
      $leadingSpace = ""
      for ($ix = 0; $ix -lt $prefixlen; $ix++) {
        $leadingSpace += " "
      }
      while ($rest.Length -gt 0) {
        $outStr = $prefix
        $prefix = $leadingSpace
        # now work out how many characters we can copy without blowing 80 chars
        $max = 80 - $prefixlen         # stop when we've got 80 chars
        if ($max -gt $rest.Length) {
          $outStr + $rest
          $rest = ""
        } else {
          $splitAt = $max
          $offset = $rest.LastIndexOf(" ",$max)
          if ($offset -eq -1) {
            # not found - truncate after $max characters
            $s1 = $rest.Substring(0,$max)
            $rest = $rest.Substring($max,$rest.Length - $max)
            $outStr + $s1
          } else {
            # found - truncate after $offset+1 characters
            $offset++
            $s1 = $rest.Substring(0,$offset)
            $rest = $rest.Substring($offset,$rest.Length - $offset)
            $outStr + $s1
          }
        }
      }
    }
  }
}

function Write-Info ($infoList){
  $LastCategory = $null
  foreach ($info in $infoList) { 
      $CurrentCategory = $info.CheckCategory
      if ($CurrentCategory -ne $LastCategory)
      {
          Write-Host -ForegroundColor Yellow $CurrentCategory
          $LastCategory = $CurrentCategory
      }
      if (($info.Reason -ne $null) -and ($info.Reason -ne "")){
          $str = $info.Info + " Reason: " + $info.Reason
      }else{
          $str = $info.Info
      }
      
      Write-Wrapped $str
  }
}

if ($WriteCsvFiles) {
  #
  # Export the Policy Summary array
  #
  $policyDb     | Export-Csv $csvSinglePolicySummaryFile -NoType
  $policyListDb | Export-Csv $csvListPolicySummaryFile   -NoType
  $envLogArray  | Export-Csv $csvEnvironmentSummaryFile  -NoType
}

if (($errStrings.Count -gt 0) -or ($warningStrings.Count -gt 0))
{
    "===================================================================="
    "The following items should be reviewed, as they might be suboptimal."
    "===================================================================="
    if ($errStrings.Count -gt 0)
    {
        Write-Host  @errorColours "************************************Errors******************************************************"
        #foreach ($s in $errStrings) { Write-Wrapped $s }
        Write-Info $errorInfoList
    }
    
    if ($warningStrings.Count -gt 0)
    {
        Write-Host @warnColours "************************************Warnings****************************************************"        
        Write-Info $warningInfoList
    }
}
Export-ToXml -xmlPath $OutputXmlPath -isCategorized $true
Export-ToXml -xmlPath $ChaOutputXmlPath -isCategorized $false
Export-ToJson -jsonPath $OutputJsonPath
$elapsedTime = new-timespan $script:StartTime $(get-date)
write-host "ElapsedTime: $elapsedTime"
# SIG # Begin signature block
# MIIoTgYJKoZIhvcNAQcCoIIoPzCCKDsCAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCAzmfQY6G6yDszT
# 0nl4z+mPt/+FEUTc7HLobfBpznT+SqCCDcAwggawMIIEmKADAgECAhAIrUCyYNKc
# TJ9ezam9k67ZMA0GCSqGSIb3DQEBDAUAMGIxCzAJBgNVBAYTAlVTMRUwEwYDVQQK
# EwxEaWdpQ2VydCBJbmMxGTAXBgNVBAsTEHd3dy5kaWdpY2VydC5jb20xITAfBgNV
# BAMTGERpZ2lDZXJ0IFRydXN0ZWQgUm9vdCBHNDAeFw0yMTA0MjkwMDAwMDBaFw0z
# NjA0MjgyMzU5NTlaMGkxCzAJBgNVBAYTAlVTMRcwFQYDVQQKEw5EaWdpQ2VydCwg
# SW5jLjFBMD8GA1UEAxM4RGlnaUNlcnQgVHJ1c3RlZCBHNCBDb2RlIFNpZ25pbmcg
# UlNBNDA5NiBTSEEzODQgMjAyMSBDQTEwggIiMA0GCSqGSIb3DQEBAQUAA4ICDwAw
# ggIKAoICAQDVtC9C0CiteLdd1TlZG7GIQvUzjOs9gZdwxbvEhSYwn6SOaNhc9es0
# JAfhS0/TeEP0F9ce2vnS1WcaUk8OoVf8iJnBkcyBAz5NcCRks43iCH00fUyAVxJr
# Q5qZ8sU7H/Lvy0daE6ZMswEgJfMQ04uy+wjwiuCdCcBlp/qYgEk1hz1RGeiQIXhF
# LqGfLOEYwhrMxe6TSXBCMo/7xuoc82VokaJNTIIRSFJo3hC9FFdd6BgTZcV/sk+F
# LEikVoQ11vkunKoAFdE3/hoGlMJ8yOobMubKwvSnowMOdKWvObarYBLj6Na59zHh
# 3K3kGKDYwSNHR7OhD26jq22YBoMbt2pnLdK9RBqSEIGPsDsJ18ebMlrC/2pgVItJ
# wZPt4bRc4G/rJvmM1bL5OBDm6s6R9b7T+2+TYTRcvJNFKIM2KmYoX7BzzosmJQay
# g9Rc9hUZTO1i4F4z8ujo7AqnsAMrkbI2eb73rQgedaZlzLvjSFDzd5Ea/ttQokbI
# YViY9XwCFjyDKK05huzUtw1T0PhH5nUwjewwk3YUpltLXXRhTT8SkXbev1jLchAp
# QfDVxW0mdmgRQRNYmtwmKwH0iU1Z23jPgUo+QEdfyYFQc4UQIyFZYIpkVMHMIRro
# OBl8ZhzNeDhFMJlP/2NPTLuqDQhTQXxYPUez+rbsjDIJAsxsPAxWEQIDAQABo4IB
# WTCCAVUwEgYDVR0TAQH/BAgwBgEB/wIBADAdBgNVHQ4EFgQUaDfg67Y7+F8Rhvv+
# YXsIiGX0TkIwHwYDVR0jBBgwFoAU7NfjgtJxXWRM3y5nP+e6mK4cD08wDgYDVR0P
# AQH/BAQDAgGGMBMGA1UdJQQMMAoGCCsGAQUFBwMDMHcGCCsGAQUFBwEBBGswaTAk
# BggrBgEFBQcwAYYYaHR0cDovL29jc3AuZGlnaWNlcnQuY29tMEEGCCsGAQUFBzAC
# hjVodHRwOi8vY2FjZXJ0cy5kaWdpY2VydC5jb20vRGlnaUNlcnRUcnVzdGVkUm9v
# dEc0LmNydDBDBgNVHR8EPDA6MDigNqA0hjJodHRwOi8vY3JsMy5kaWdpY2VydC5j
# b20vRGlnaUNlcnRUcnVzdGVkUm9vdEc0LmNybDAcBgNVHSAEFTATMAcGBWeBDAED
# MAgGBmeBDAEEATANBgkqhkiG9w0BAQwFAAOCAgEAOiNEPY0Idu6PvDqZ01bgAhql
# +Eg08yy25nRm95RysQDKr2wwJxMSnpBEn0v9nqN8JtU3vDpdSG2V1T9J9Ce7FoFF
# UP2cvbaF4HZ+N3HLIvdaqpDP9ZNq4+sg0dVQeYiaiorBtr2hSBh+3NiAGhEZGM1h
# mYFW9snjdufE5BtfQ/g+lP92OT2e1JnPSt0o618moZVYSNUa/tcnP/2Q0XaG3Ryw
# YFzzDaju4ImhvTnhOE7abrs2nfvlIVNaw8rpavGiPttDuDPITzgUkpn13c5Ubdld
# AhQfQDN8A+KVssIhdXNSy0bYxDQcoqVLjc1vdjcshT8azibpGL6QB7BDf5WIIIJw
# 8MzK7/0pNVwfiThV9zeKiwmhywvpMRr/LhlcOXHhvpynCgbWJme3kuZOX956rEnP
# LqR0kq3bPKSchh/jwVYbKyP/j7XqiHtwa+aguv06P0WmxOgWkVKLQcBIhEuWTatE
# QOON8BUozu3xGFYHKi8QxAwIZDwzj64ojDzLj4gLDb879M4ee47vtevLt/B3E+bn
# KD+sEq6lLyJsQfmCXBVmzGwOysWGw/YmMwwHS6DTBwJqakAwSEs0qFEgu60bhQji
# WQ1tygVQK+pKHJ6l/aCnHwZ05/LWUpD9r4VIIflXO7ScA+2GRfS0YW6/aOImYIbq
# yK+p/pQd52MbOoZWeE4wggcIMIIE8KADAgECAhADWiAGLDpbT4qELERZjaX6MA0G
# CSqGSIb3DQEBCwUAMGkxCzAJBgNVBAYTAlVTMRcwFQYDVQQKEw5EaWdpQ2VydCwg
# SW5jLjFBMD8GA1UEAxM4RGlnaUNlcnQgVHJ1c3RlZCBHNCBDb2RlIFNpZ25pbmcg
# UlNBNDA5NiBTSEEzODQgMjAyMSBDQTEwHhcNMjUwMjA3MDAwMDAwWhcNMjYwMjA2
# MjM1OTU5WjCBjzELMAkGA1UEBhMCVVMxEDAOBgNVBAgTB0Zsb3JpZGExGDAWBgNV
# BAcTD0ZvcnQgTGF1ZGVyZGFsZTEdMBsGA1UEChMUQ2l0cml4IFN5c3RlbXMsIElu
# Yy4xFjAUBgNVBAsTDUNpdHJpeCBYZW5BcHAxHTAbBgNVBAMTFENpdHJpeCBTeXN0
# ZW1zLCBJbmMuMIIBojANBgkqhkiG9w0BAQEFAAOCAY8AMIIBigKCAYEAnBm4JMW+
# W2tRIJltyaJM/q5G6WRAN293DlUc71U/dg6HzXRtahZP2kvV6ii32orbgdNkCwig
# +bo0W7AKjBBHKJqP96nuTNTsHoz+aeEE6PfgaFzGTRq77+dGDCk5DuPbFaXJCn9s
# UjVdbJTG1YXAcll8ZU6Iarg3eOokp2CcejMcjeD2dO8y89o6y2W5sWj2oIA+QRE9
# iU5qNfDLUtXo5i017JTN+qs9RtuwD77svXoV29wmkPmGuUq625ZZYFtRa0/t/C7w
# /k00UOFjykbvNBPj/cT67i/J/Um8tOanuC3cYU18VToMsfpS2t4irTdtrBzHupr/
# MB2DzZTwr5x2m3UzgEsnrr9bYnCIpOuW+K/oExuTpHtZsk6fnpoteOfyP059dNMg
# i1Gj074k6JfaJG+6fwKW0i2Unf7NDBArHdoHA6eIYB/OivPt4cmusgzRr2PziAl4
# LpA/9VRcnR68CWjnoTTr7qhdDeGqMgk24cNtmg+6BDt65GDVDX2ycthpAgMBAAGj
# ggIDMIIB/zAfBgNVHSMEGDAWgBRoN+Drtjv4XxGG+/5hewiIZfROQjAdBgNVHQ4E
# FgQUEpWeJpU+v9Nt97sQvlQgm4DN5wMwPgYDVR0gBDcwNTAzBgZngQwBBAEwKTAn
# BggrBgEFBQcCARYbaHR0cDovL3d3dy5kaWdpY2VydC5jb20vQ1BTMA4GA1UdDwEB
# /wQEAwIHgDATBgNVHSUEDDAKBggrBgEFBQcDAzCBtQYDVR0fBIGtMIGqMFOgUaBP
# hk1odHRwOi8vY3JsMy5kaWdpY2VydC5jb20vRGlnaUNlcnRUcnVzdGVkRzRDb2Rl
# U2lnbmluZ1JTQTQwOTZTSEEzODQyMDIxQ0ExLmNybDBToFGgT4ZNaHR0cDovL2Ny
# bDQuZGlnaWNlcnQuY29tL0RpZ2lDZXJ0VHJ1c3RlZEc0Q29kZVNpZ25pbmdSU0E0
# MDk2U0hBMzg0MjAyMUNBMS5jcmwwgZQGCCsGAQUFBwEBBIGHMIGEMCQGCCsGAQUF
# BzABhhhodHRwOi8vb2NzcC5kaWdpY2VydC5jb20wXAYIKwYBBQUHMAKGUGh0dHA6
# Ly9jYWNlcnRzLmRpZ2ljZXJ0LmNvbS9EaWdpQ2VydFRydXN0ZWRHNENvZGVTaWdu
# aW5nUlNBNDA5NlNIQTM4NDIwMjFDQTEuY3J0MAkGA1UdEwQCMAAwDQYJKoZIhvcN
# AQELBQADggIBAMGu7yYWlIS96uBwpNvMWgRgVFRwDx0HBdxGRi2+XvrtKkooOglx
# eUl/IYzZb8WhEBs/vknAmZkxBnkbbdgKKqxDeIkT2JX1sNGLKIzBBLPPjGMd2wxH
# CAVssWrA9xMESJ5Zx+5jHyoeVPt4XdPAXNB6Ca1JCa68Ip8Bn+atqop5wOmhXK85
# 2V2PdCWXjQr3Yl0k0PY/OtDDxnQc0dClj9TYlfGsOwyVNNQh/eExH2wkVlYYxgcj
# Zn89SmO4L+9AHkfpj/wd+/pJ+BYSflAotC6d9l4dft6f6PZ6KkbAz56dVkBX20xz
# YKKIK9tCf5sa1wnjkBEm2yn1IDn6K1cwUGO0whhtzGgcv4vPfv5bkumCSD7AquQv
# 37LP/gOZT9v3CiZAlTbZAoOK105qFmawt7AKPFzNFq9EjbUHYVBeYFnETDoa9zLz
# KCCcF/xJsjfn7YqFM+b1zq3C1YJK1B6+f3b5acpXNHbyMxPSOeNDZ13wMAYKah17
# D/4GZe74bsVqFo0oHlB82Bocr0AnnrLnLvfEirc0KIon5CQ5cci3LolBE0aNhD+n
# UtInPAJN1uRFpvhSK1ucVO/afmTQVtUgp5a8a8Dthc0bqh2ere4hoY+9U08Bda5g
# mFOO9YU8xEU3Do0AbuJPUVVALAAs0I+38RjYtIfPPt/T1zrMFHv+CMsKMYIZ5DCC
# GeACAQEwfTBpMQswCQYDVQQGEwJVUzEXMBUGA1UEChMORGlnaUNlcnQsIEluYy4x
# QTA/BgNVBAMTOERpZ2lDZXJ0IFRydXN0ZWQgRzQgQ29kZSBTaWduaW5nIFJTQTQw
# OTYgU0hBMzg0IDIwMjEgQ0ExAhADWiAGLDpbT4qELERZjaX6MA0GCWCGSAFlAwQC
# AQUAoHwwEAYKKwYBBAGCNwIBDDECMAAwGQYJKoZIhvcNAQkDMQwGCisGAQQBgjcC
# AQQwHAYKKwYBBAGCNwIBCzEOMAwGCisGAQQBgjcCARUwLwYJKoZIhvcNAQkEMSIE
# IL3wliPJSO5OhSmuYQ8DBYXkFRWbg6AYS2nfiwwciMWyMA0GCSqGSIb3DQEBAQUA
# BIIBgDQveyDc6SnlLmQ+rhQGaeMRAEoYU8j8mCNLYN8ciR+AaaY+FbRsJRoAVOk1
# VQ3OOBZqv4ou7bju1pYuclVQ0I2kmxrNEhdWBlo4MeTSuRhipMMLWBikGGW445Te
# HV3SGpfPQy5PoIoRLYYgD5iIvavicPTEUOsEVmTqkM+rDqMmvhbxNZd53+3B6PCB
# lkP1J5mvi7TE1GfYyv3vDbyEvmvCsTsT+iCkcncmcxHPfNL68BC8eicefwzvzYgl
# w3K5GdRAYLjO1WJ5SWZXTv/lEFBWUXOeK67GGSRQKFGen4bprJUs1oDZITKLw5vz
# LYoPyh7d1JV8ON8oW98QjS2qGPVUf659b4o4TcUjfM1t9ec3whvcIYaj5fhtAO/g
# 8TE75fwTKOxNSuPrkNJIj/1Hcl5C+JWJ+mLyJcWw+pFKzveT3+3UO21BILy7jPpY
# puF/+ewWRpsIOESXexySvhs3On2bLLTK09vCuusgynx7RnN5WXDAAP7cRxl8bLUW
# XxoPEaGCFzowghc2BgorBgEEAYI3AwMBMYIXJjCCFyIGCSqGSIb3DQEHAqCCFxMw
# ghcPAgEDMQ8wDQYJYIZIAWUDBAIBBQAweAYLKoZIhvcNAQkQAQSgaQRnMGUCAQEG
# CWCGSAGG/WwHATAxMA0GCWCGSAFlAwQCAQUABCC9cYrZ9vKPodPbszNdtHItCSRg
# FUObDhkMXvdGlDEWPgIRALoca25YsVxwdSfbgXtjv+4YDzIwMjUwNzAzMDgyMzA4
# WqCCEwMwgga8MIIEpKADAgECAhALrma8Wrp/lYfG+ekE4zMEMA0GCSqGSIb3DQEB
# CwUAMGMxCzAJBgNVBAYTAlVTMRcwFQYDVQQKEw5EaWdpQ2VydCwgSW5jLjE7MDkG
# A1UEAxMyRGlnaUNlcnQgVHJ1c3RlZCBHNCBSU0E0MDk2IFNIQTI1NiBUaW1lU3Rh
# bXBpbmcgQ0EwHhcNMjQwOTI2MDAwMDAwWhcNMzUxMTI1MjM1OTU5WjBCMQswCQYD
# VQQGEwJVUzERMA8GA1UEChMIRGlnaUNlcnQxIDAeBgNVBAMTF0RpZ2lDZXJ0IFRp
# bWVzdGFtcCAyMDI0MIICIjANBgkqhkiG9w0BAQEFAAOCAg8AMIICCgKCAgEAvmpz
# n/aVIauWMLpbbeZZo7Xo/ZEfGMSIO2qZ46XB/QowIEMSvgjEdEZ3v4vrrTHleW1J
# WGErrjOL0J4L0HqVR1czSzvUQ5xF7z4IQmn7dHY7yijvoQ7ujm0u6yXF2v1CrzZo
# pykD07/9fpAT4BxpT9vJoJqAsP8YuhRvflJ9YeHjes4fduksTHulntq9WelRWY++
# TFPxzZrbILRYynyEy7rS1lHQKFpXvo2GePfsMRhNf1F41nyEg5h7iOXv+vjX0K8R
# hUisfqw3TTLHj1uhS66YX2LZPxS4oaf33rp9HlfqSBePejlYeEdU740GKQM7SaVS
# H3TbBL8R6HwX9QVpGnXPlKdE4fBIn5BBFnV+KwPxRNUNK6lYk2y1WSKour4hJN0S
# MkoaNV8hyyADiX1xuTxKaXN12HgR+8WulU2d6zhzXomJ2PleI9V2yfmfXSPGYanG
# gxzqI+ShoOGLomMd3mJt92nm7Mheng/TBeSA2z4I78JpwGpTRHiT7yHqBiV2ngUI
# yCtd0pZ8zg3S7bk4QC4RrcnKJ3FbjyPAGogmoiZ33c1HG93Vp6lJ415ERcC7bFQM
# RbxqrMVANiav1k425zYyFMyLNyE1QulQSgDpW9rtvVcIH7WvG9sqYup9j8z9J1Xq
# bBZPJ5XLln8mS8wWmdDLnBHXgYly/p1DhoQo5fkCAwEAAaOCAYswggGHMA4GA1Ud
# DwEB/wQEAwIHgDAMBgNVHRMBAf8EAjAAMBYGA1UdJQEB/wQMMAoGCCsGAQUFBwMI
# MCAGA1UdIAQZMBcwCAYGZ4EMAQQCMAsGCWCGSAGG/WwHATAfBgNVHSMEGDAWgBS6
# FtltTYUvcyl2mi91jGogj57IbzAdBgNVHQ4EFgQUn1csA3cOKBWQZqVjXu5Pkh92
# oFswWgYDVR0fBFMwUTBPoE2gS4ZJaHR0cDovL2NybDMuZGlnaWNlcnQuY29tL0Rp
# Z2lDZXJ0VHJ1c3RlZEc0UlNBNDA5NlNIQTI1NlRpbWVTdGFtcGluZ0NBLmNybDCB
# kAYIKwYBBQUHAQEEgYMwgYAwJAYIKwYBBQUHMAGGGGh0dHA6Ly9vY3NwLmRpZ2lj
# ZXJ0LmNvbTBYBggrBgEFBQcwAoZMaHR0cDovL2NhY2VydHMuZGlnaWNlcnQuY29t
# L0RpZ2lDZXJ0VHJ1c3RlZEc0UlNBNDA5NlNIQTI1NlRpbWVTdGFtcGluZ0NBLmNy
# dDANBgkqhkiG9w0BAQsFAAOCAgEAPa0eH3aZW+M4hBJH2UOR9hHbm04IHdEoT8/T
# 3HuBSyZeq3jSi5GXeWP7xCKhVireKCnCs+8GZl2uVYFvQe+pPTScVJeCZSsMo1JC
# oZN2mMew/L4tpqVNbSpWO9QGFwfMEy60HofN6V51sMLMXNTLfhVqs+e8haupWiAr
# SozyAmGH/6oMQAh078qRh6wvJNU6gnh5OruCP1QUAvVSu4kqVOcJVozZR5RRb/zP
# d++PGE3qF1P3xWvYViUJLsxtvge/mzA75oBfFZSbdakHJe2BVDGIGVNVjOp8sNt7
# 0+kEoMF+T6tptMUNlehSR7vM+C13v9+9ZOUKzfRUAYSyyEmYtsnpltD/GWX8eM70
# ls1V6QG/ZOB6b6Yum1HvIiulqJ1Elesj5TMHq8CWT/xrW7twipXTJ5/i5pkU5E16
# RSBAdOp12aw8IQhhA/vEbFkEiF2abhuFixUDobZaA0VhqAsMHOmaT3XThZDNi5U2
# zHKhUs5uHHdG6BoQau75KiNbh0c+hatSF+02kULkftARjsyEpHKsF7u5zKRbt5oK
# 5YGwFvgc4pEVUNytmB3BpIiowOIIuDgP5M9WArHYSAR16gc0dP2XdkMEP5eBsX7b
# f/MGN4K3HP50v/01ZHo/Z5lGLvNwQ7XHBx1yomzLP8lx4Q1zZKDyHcp4VQJLu2kW
# TsKsOqQwggauMIIElqADAgECAhAHNje3JFR82Ees/ShmKl5bMA0GCSqGSIb3DQEB
# CwUAMGIxCzAJBgNVBAYTAlVTMRUwEwYDVQQKEwxEaWdpQ2VydCBJbmMxGTAXBgNV
# BAsTEHd3dy5kaWdpY2VydC5jb20xITAfBgNVBAMTGERpZ2lDZXJ0IFRydXN0ZWQg
# Um9vdCBHNDAeFw0yMjAzMjMwMDAwMDBaFw0zNzAzMjIyMzU5NTlaMGMxCzAJBgNV
# BAYTAlVTMRcwFQYDVQQKEw5EaWdpQ2VydCwgSW5jLjE7MDkGA1UEAxMyRGlnaUNl
# cnQgVHJ1c3RlZCBHNCBSU0E0MDk2IFNIQTI1NiBUaW1lU3RhbXBpbmcgQ0EwggIi
# MA0GCSqGSIb3DQEBAQUAA4ICDwAwggIKAoICAQDGhjUGSbPBPXJJUVXHJQPE8pE3
# qZdRodbSg9GeTKJtoLDMg/la9hGhRBVCX6SI82j6ffOciQt/nR+eDzMfUBMLJnOW
# bfhXqAJ9/UO0hNoR8XOxs+4rgISKIhjf69o9xBd/qxkrPkLcZ47qUT3w1lbU5ygt
# 69OxtXXnHwZljZQp09nsad/ZkIdGAHvbREGJ3HxqV3rwN3mfXazL6IRktFLydkf3
# YYMZ3V+0VAshaG43IbtArF+y3kp9zvU5EmfvDqVjbOSmxR3NNg1c1eYbqMFkdECn
# wHLFuk4fsbVYTXn+149zk6wsOeKlSNbwsDETqVcplicu9Yemj052FVUmcJgmf6Aa
# RyBD40NjgHt1biclkJg6OBGz9vae5jtb7IHeIhTZgirHkr+g3uM+onP65x9abJTy
# UpURK1h0QCirc0PO30qhHGs4xSnzyqqWc0Jon7ZGs506o9UD4L/wojzKQtwYSH8U
# NM/STKvvmz3+DrhkKvp1KCRB7UK/BZxmSVJQ9FHzNklNiyDSLFc1eSuo80VgvCON
# WPfcYd6T/jnA+bIwpUzX6ZhKWD7TA4j+s4/TXkt2ElGTyYwMO1uKIqjBJgj5FBAS
# A31fI7tk42PgpuE+9sJ0sj8eCXbsq11GdeJgo1gJASgADoRU7s7pXcheMBK9Rp61
# 03a50g5rmQzSM7TNsQIDAQABo4IBXTCCAVkwEgYDVR0TAQH/BAgwBgEB/wIBADAd
# BgNVHQ4EFgQUuhbZbU2FL3MpdpovdYxqII+eyG8wHwYDVR0jBBgwFoAU7NfjgtJx
# XWRM3y5nP+e6mK4cD08wDgYDVR0PAQH/BAQDAgGGMBMGA1UdJQQMMAoGCCsGAQUF
# BwMIMHcGCCsGAQUFBwEBBGswaTAkBggrBgEFBQcwAYYYaHR0cDovL29jc3AuZGln
# aWNlcnQuY29tMEEGCCsGAQUFBzAChjVodHRwOi8vY2FjZXJ0cy5kaWdpY2VydC5j
# b20vRGlnaUNlcnRUcnVzdGVkUm9vdEc0LmNydDBDBgNVHR8EPDA6MDigNqA0hjJo
# dHRwOi8vY3JsMy5kaWdpY2VydC5jb20vRGlnaUNlcnRUcnVzdGVkUm9vdEc0LmNy
# bDAgBgNVHSAEGTAXMAgGBmeBDAEEAjALBglghkgBhv1sBwEwDQYJKoZIhvcNAQEL
# BQADggIBAH1ZjsCTtm+YqUQiAX5m1tghQuGwGC4QTRPPMFPOvxj7x1Bd4ksp+3CK
# Daopafxpwc8dB+k+YMjYC+VcW9dth/qEICU0MWfNthKWb8RQTGIdDAiCqBa9qVbP
# FXONASIlzpVpP0d3+3J0FNf/q0+KLHqrhc1DX+1gtqpPkWaeLJ7giqzl/Yy8ZCaH
# bJK9nXzQcAp876i8dU+6WvepELJd6f8oVInw1YpxdmXazPByoyP6wCeCRK6ZJxur
# JB4mwbfeKuv2nrF5mYGjVoarCkXJ38SNoOeY+/umnXKvxMfBwWpx2cYTgAnEtp/N
# h4cku0+jSbl3ZpHxcpzpSwJSpzd+k1OsOx0ISQ+UzTl63f8lY5knLD0/a6fxZsNB
# zU+2QJshIUDQtxMkzdwdeDrknq3lNHGS1yZr5Dhzq6YBT70/O3itTK37xJV77Qpf
# MzmHQXh6OOmc4d0j/R0o08f56PGYX/sr2H7yRp11LB4nLCbbbxV7HhmLNriT1Oby
# F5lZynDwN7+YAN8gFk8n+2BnFqFmut1VwDophrCYoCvtlUG3OtUVmDG0YgkPCr2B
# 2RP+v6TR81fZvAT6gt4y3wSJ8ADNXcL50CN/AAvkdgIm2fBldkKmKYcJRyvmfxqk
# hQ/8mJb2VVQrH4D6wPIOK+XW+6kvRBVK5xMOHds3OBqhK/bt1nz8MIIFjTCCBHWg
# AwIBAgIQDpsYjvnQLefv21DiCEAYWjANBgkqhkiG9w0BAQwFADBlMQswCQYDVQQG
# EwJVUzEVMBMGA1UEChMMRGlnaUNlcnQgSW5jMRkwFwYDVQQLExB3d3cuZGlnaWNl
# cnQuY29tMSQwIgYDVQQDExtEaWdpQ2VydCBBc3N1cmVkIElEIFJvb3QgQ0EwHhcN
# MjIwODAxMDAwMDAwWhcNMzExMTA5MjM1OTU5WjBiMQswCQYDVQQGEwJVUzEVMBMG
# A1UEChMMRGlnaUNlcnQgSW5jMRkwFwYDVQQLExB3d3cuZGlnaWNlcnQuY29tMSEw
# HwYDVQQDExhEaWdpQ2VydCBUcnVzdGVkIFJvb3QgRzQwggIiMA0GCSqGSIb3DQEB
# AQUAA4ICDwAwggIKAoICAQC/5pBzaN675F1KPDAiMGkz7MKnJS7JIT3yithZwuEp
# pz1Yq3aaza57G4QNxDAf8xukOBbrVsaXbR2rsnnyyhHS5F/WBTxSD1Ifxp4VpX6+
# n6lXFllVcq9ok3DCsrp1mWpzMpTREEQQLt+C8weE5nQ7bXHiLQwb7iDVySAdYykt
# zuxeTsiT+CFhmzTrBcZe7FsavOvJz82sNEBfsXpm7nfISKhmV1efVFiODCu3T6cw
# 2Vbuyntd463JT17lNecxy9qTXtyOj4DatpGYQJB5w3jHtrHEtWoYOAMQjdjUN6Qu
# BX2I9YI+EJFwq1WCQTLX2wRzKm6RAXwhTNS8rhsDdV14Ztk6MUSaM0C/CNdaSaTC
# 5qmgZ92kJ7yhTzm1EVgX9yRcRo9k98FpiHaYdj1ZXUJ2h4mXaXpI8OCiEhtmmnTK
# 3kse5w5jrubU75KSOp493ADkRSWJtppEGSt+wJS00mFt6zPZxd9LBADMfRyVw4/3
# IbKyEbe7f/LVjHAsQWCqsWMYRJUadmJ+9oCw++hkpjPRiQfhvbfmQ6QYuKZ3AeEP
# lAwhHbJUKSWJbOUOUlFHdL4mrLZBdd56rF+NP8m800ERElvlEFDrMcXKchYiCd98
# THU/Y+whX8QgUWtvsauGi0/C1kVfnSD8oR7FwI+isX4KJpn15GkvmB0t9dmpsh3l
# GwIDAQABo4IBOjCCATYwDwYDVR0TAQH/BAUwAwEB/zAdBgNVHQ4EFgQU7NfjgtJx
# XWRM3y5nP+e6mK4cD08wHwYDVR0jBBgwFoAUReuir/SSy4IxLVGLp6chnfNtyA8w
# DgYDVR0PAQH/BAQDAgGGMHkGCCsGAQUFBwEBBG0wazAkBggrBgEFBQcwAYYYaHR0
# cDovL29jc3AuZGlnaWNlcnQuY29tMEMGCCsGAQUFBzAChjdodHRwOi8vY2FjZXJ0
# cy5kaWdpY2VydC5jb20vRGlnaUNlcnRBc3N1cmVkSURSb290Q0EuY3J0MEUGA1Ud
# HwQ+MDwwOqA4oDaGNGh0dHA6Ly9jcmwzLmRpZ2ljZXJ0LmNvbS9EaWdpQ2VydEFz
# c3VyZWRJRFJvb3RDQS5jcmwwEQYDVR0gBAowCDAGBgRVHSAAMA0GCSqGSIb3DQEB
# DAUAA4IBAQBwoL9DXFXnOF+go3QbPbYW1/e/Vwe9mqyhhyzshV6pGrsi+IcaaVQi
# 7aSId229GhT0E0p6Ly23OO/0/4C5+KH38nLeJLxSA8hO0Cre+i1Wz/n096wwepqL
# sl7Uz9FDRJtDIeuWcqFItJnLnU+nBgMTdydE1Od/6Fmo8L8vC6bp8jQ87PcDx4eo
# 0kxAGTVGamlUsLihVo7spNU96LHc/RzY9HdaXFSMb++hUD38dglohJ9vytsgjTVg
# HAIDyyCwrFigDkBjxZgiwbJZ9VVrzyerbHbObyMt9H5xaiNrIv8SuFQtJ37YOtnw
# toeW/VvRXKwYw02fc7cBqZ9Xql4o4rmUMYIDdjCCA3ICAQEwdzBjMQswCQYDVQQG
# EwJVUzEXMBUGA1UEChMORGlnaUNlcnQsIEluYy4xOzA5BgNVBAMTMkRpZ2lDZXJ0
# IFRydXN0ZWQgRzQgUlNBNDA5NiBTSEEyNTYgVGltZVN0YW1waW5nIENBAhALrma8
# Wrp/lYfG+ekE4zMEMA0GCWCGSAFlAwQCAQUAoIHRMBoGCSqGSIb3DQEJAzENBgsq
# hkiG9w0BCRABBDAcBgkqhkiG9w0BCQUxDxcNMjUwNzAzMDgyMzA4WjArBgsqhkiG
# 9w0BCRACDDEcMBowGDAWBBTb04XuYtvSPnvk9nFIUIck1YZbRTAvBgkqhkiG9w0B
# CQQxIgQgWr/krf5Ur3WgvwnGSqze6YDluf2Y5zvaqHFicJ3VZzEwNwYLKoZIhvcN
# AQkQAi8xKDAmMCQwIgQgdnafqPJjLx9DCzojMK7WVnX+13PbBdZluQWTmEOPmtsw
# DQYJKoZIhvcNAQEBBQAEggIAEJ4oZz8/t2hYG92Es8EuU4zKwVUZEqZvujdsUUX2
# 6MzREYBb7xdEHrEqF0ECbHCQ+d35T9EUel/MVp/K533F3cKOclOcqq+Qu2AnZGFZ
# lMCvlS8AWNytHXHbDNiEwqL36npmB/SpxJjDGkyzhIGvl2B1knNnTuQ6WZJYQneV
# SoZkNDUDpbRp9EE4Wq+Xf6gE2oq89RySx8bBI+z2QIRzUglQ9JlZU36xteXSvaYp
# gKGaF7KVl8IgNiqb0PtLgVFXEr9obUlfXBtAZj3Wzjnpo3X4fbJhQ/5xQXioXtVw
# S81qAu9l2zp1gZNWcO2Bn4nUGY3lYaqXZPTca3TMyRpic2CUvBj7N0pWiYCt7V3K
# X3QtV1C0CQzPjwNGVQ7/Q2cQ0zMmUPtIklNn4BxHyVMdm6DUiR0PVdmLK5XiAbxC
# OPxz8LKknTmm9tD8O/m+piaP4TL23BWNm9yeZi33DkYGiJ3if17IOI4TigETQzUG
# +6Vsza+UlyobLTNOjudkWckTh+sUFSKU3d2Fzcuy5E+E7FloldbvtPSvj2vn6Wu+
# Tj0V85B2dEHCXKfheUPSz9vosZz62so0Rm1XtG520OD8jq8w5r11xi/TjLoYj+Tp
# TkSyMOCquPYHVulERocX6OynbeILq3v0exb/XSZQnOs/FVMR2otCXpmLVuMOtrs9
# wus=
# SIG # End signature block
