#!/bin/sh

###############################################################################
#
#   $Id$
#
#   Copyright 2011-2016 Citrix Systems, Inc.  All Rights Reserved.
#
###############################################################################

ICAROOT=/opt/Citrix/ICAClient
# Setup our generic ECHO_CMD  - same syntax across all platforms
ECHO_CMD=${ICAROOT}/util/echo_cmd export ECHO_CMD

rm_last_dot(){
	<&0 sed 's/\/.\$//' >&1
}
TopDir=`$ECHO_CMD $ICAROOT | rm_last_dot`

# replacements for prompting functions, that just do the default action.
# set ICAInstDir without prompting
setICAInstDirNoPrompt()
{
	SETDIR_OK=false
	# Set up a suitable default location.
	if [ "$ICAInstDir" = "" ]
	then
		if [ "$RunningFrom" = "INSTALLED" ]
		then
			ICAInstDir=$TopDir
		else
			ICAInstDir=$DefaultInstallDir
		fi
	fi
	SETDIR_OK=true
}

#!/bin/sh
###############################################################################
#
#	Name:		Version.msg
#
#	$Id$
#
#	Purpose:	Sets shell variables for the version of Client
#			to be installed.
#
#	Copyright 1996-2000, 2006, 2009 Citrix Systems, Inc. All rights reserved.
#
###############################################################################

# note: the INSTALLER_... strings should be in lowercase. The installer will then
# accept lowercase or uppercase versions of them as input.
INSTALLER_YES=yes
INSTALLER_NO=no
INSTALLER_QUIT=quit
INSTALLER_Y=y
INSTALLER_N=n
INSTALLER_Q=q
PACKAGE_LANGUAGE=en
DISP_PRODUCT_NAME="Citrix Workspace app for Linux"

#!/bin/sh
###############################################################################
#
#	Name:		config.sh
#
#	Version:	$Id$
#
#	Purpose:	Sets shell variables for things that may be specific
#			to the host operating system.  This allows other shell
#			script parts to be generic.  To adapt the install
#			script for an additional host, you should consider
#			ALL of the lines below.  Note that for some items,
#			and incorrect value might not show up on a simple test,
#			as the scripts might not follow a path containing
#			that item.
#			 
#
#	Copyright 1998-2014 Citrix Systems, Inc.  All rights reserved.
#
###############################################################################

# Ensure we have some sort of echo command set up. setupwfc should have set
# it to use our own version that has the same syntax across all platforms.
if [ -z "$ECHO_CMD" ]
then
    ECHO_CMD=echo
fi

#  Install in here by default.
DefaultInstallDir=/opt/Citrix/ICAClient

#  Default CDROM mount point
DefaultCDSourceDir=/cdrom

#  The directory name on the CDROM of the platform package we are installing.
PORT=linuxx64

#  The command to undo the archive on this platform.
EXTRACT_ARCHIVE="cpio -imud -H newc --no-preserve-owner"

# The thing that goes between the owner and group to set both by chown.
CHOWNGRP_TOKEN=":"

# The command to use to test for a symbolic link
LNKTST="test -L"

# The commands to use to change user/group of symbolic link
# In Linux, this information is included in the cpio archive
SYMCHOWN_CMD=
SYMCHGRP_CMD=

# The command to use to force a reread of inittab
READ_INITTAB_CMD="/sbin/telinit q"

# Arguments for lmhostid to get correct hostid
LMHOSTID_ARGS=ether

# The hosts file name
CAT_CMD=cat
CAT_HOST_ARG=/etc/hosts

# use less in preference to more as it appears to be less
# buggy wrt displaying Japanese characters than more.
if [ "$PAGER_CMD" ]
then
    PAGER_CMD=`which $PAGER_CMD`
    if [ "$PAGER_CMD" -a "$PAGER_PFX" ]
    then
	PAGER_CMD="$PAGER_PFX $PAGER_CMD"
    fi
fi
MORE_CMD="${PAGER_CMD:=`which more`}"

# The command to read host information from NIS
YPCAT_CMD=/usr/bin/ypcat
YPCAT_HOST_ARG=hosts

# The awk command
AWK_CMD=awk
BYTES_PER_BLOCK=512
BLOCKS_PER_ALLOCATION=2
#!/bin/sh
###############################################################################
#
#	Name:		instmode.sh
#
#	Version:	$Id$
#
#	Purpose:	Provides a method of querying and setting the 
#                       installation mode.
#`
#	Copyright 2002 Citrix Systems, Inc. All rights reserved.
#
###############################################################################

SU_INSTALL=1

###############################################################################
# get_user_id
#
# echoes the current user ID to STDOUT
###############################################################################

get_user_id()
{
    # Return number after first "equals" sign.
    id | sed "s/[^=]*=\([0-9]*\).*/\1/g" 
}

###############################################################################
# set_install_mode
#
# This should be called before the SU_INSTALL environment variable is tested.
# After calling set_install_mode, if SU_INSTALL is a non-zero length, then
# the super-user installation mode will be used.
###############################################################################
set_install_mode()
{
    # If superuser, then do multi-user install, otherwise only allow
    # single-user install
    uid=`get_user_id`

    if [ $uid != 0 ]
    then
	unset SU_INSTALL;
    fi
}

#!/bin/sh
###############################################################################
#
#	Name:		util_ver.sh
#
#	$Id$
#
#	Purpose:	Utilities for creting and checking version information
#			for both CD image and installation.
#
#	Copyright 1996-2011 Citrix Systems, Inc. All rights reserved.
#
###############################################################################


###############################################################################
#
#  put_version_info() creates version information in the installed package,
#  for later checking of the version and possibly for upgrading the package.
#
#  Takes four parameters:
#	package information filename (.psf file)
#	package data file
#	sub-package type (core/man/cus)
#	sub-package section (base/local/port)
#
#  The two filename parameters are assumed to already be expanded to full
#  pathnames, with any required name mappings completed.
#
###############################################################################
put_version_info(){
	pvi_package_info_file=$1
	pvi_package_data_file=$2
	pvi_package_type=$3
	pvi_package_section=$4

	pvi_output_dir="$INST_DIR/pkginf"
	create_dir "$pvi_output_dir"

	pvi_output_file="${pvi_output_dir}/Ver.${pvi_package_type}.${pvi_package_section}"

	"$ECHO_CMD" "ID_VERSION=$CurrentIDVER"	          >  "$pvi_output_file"
	"$ECHO_CMD" "DISP_VERSION=$CurrentDISPVER"	  >> "$pvi_output_file"
	"$ECHO_CMD" "CHANGE_NO=$CurrentCHANGENO"	          >> "$pvi_output_file"
	"$ECHO_CMD" "ID_PRODUCT_NAME=$CurrentIDNAME"        >> "$pvi_output_file"
	"$ECHO_CMD" "DISP_PRODUCT_NAME=$CurrentDISPNAME"    >> "$pvi_output_file"
	"$ECHO_CMD" "uname=`uname -a`"			  >> "$pvi_output_file"
	"$ECHO_CMD" "PORT=$PORT"			          >> "$pvi_output_file"
        "$ECHO_CMD" "LANGUAGE=*"                          >> "$pvi_output_file"

	"$ECHO_CMD" "$CurrentCHANGENO"	          > "${pvi_output_dir}/changeno.dat"

	if [ "$HOST_DATE_COMMAND" != "" ]
	then
		"$ECHO_CMD" "date=`$HOST_DATE_COMMAND`"	  >> "$pvi_output_file"
	else
		"$ECHO_CMD" "date=`date '+%Y:%m:%d:%H:%M:%S %Z' | sed 's/://g'`"	>> "$pvi_output_file"
	fi
	"$ECHO_CMD" "INFO_FILE=$pvi_package_info_file"	  >> "$pvi_output_file"
	"$ECHO_CMD" "DATA_FILE=$pvi_package_data_file"	  >> "$pvi_output_file"
	
	pkg_files_rec="${pvi_output_dir}/F.${pvi_package_type}.${pvi_package_section}"
	touch "$pkg_files_rec"
	cat "$pvi_package_info_file" 2> /dev/null | (
		while read line
		do
			set $line
			psf_package_type=$3
			if [ "$psf_package_type" = "$pvi_package_type" ]
			then
				"$ECHO_CMD" "$line" >> "$pkg_files_rec"
			fi
		done
	)
	chmod 644 "$pvi_output_file" "$pkg_files_rec"
	if [ -z "$SU_INSTALL" -a -n "$UMASK_STR" ] ;  then
	        chmod $UMASK_STR "$pvi_output_file" "$pkg_files_rec"
	fi
}

###############################################################################
#
#  inst_version_ok() sees if an installation is suitable for overwriting
#  with this installer.  Returns 'true' if it is OK, 'false' otherwise.
#  Takes one parameter, which is the directory containing the installation.
#
###############################################################################
# The root of the version information file names.
inst_version_ok(){
	root_inst_dir=$1
	ver_file="${root_inst_dir}/pkginf/Ver.core.$PORT"
	
	if [ ! -r "$ver_file" ]
	then
		# Can't find the file - installed version doesn't match
		false
		return
	fi
	
	egrep '^ID_PRODUCT_NAME' "$ver_file" 2>&1 | fgrep '='"$ID_PRODUCT_NAME" > /dev/null 2>&1
	if [ "$?" = "0" ]
	then
                # - we've got a match
                OLD_DISP_PRODUCT_NAME=`egrep '^DISP_PRODUCT_NAME' "$ver_file" 2>&1 | \
                                sed -e 's/.*=\(.*\)/\1/'`
                OLD_DISP_VERSION=`egrep '^DISP_VERSION' "$ver_file" 2>&1 | \
                                sed -e 's/.*=\([0-9\.]*\)/\1/'`
                true
                return
	fi
	# file contains another name
	false
	return
}

#!/bin/sh
###############################################################################
#
#	Name:		crdir.sh
#
#	$Id$
#
#	Purpose:	Creates a directory and any necessary parents.
#
#	Copyright 1996-2011 Citrix Systems, Inc. All rights reserved.
#
###############################################################################


###############################################################################
#
#  create_dir() creates a directory and any required parent directories.
#
#  This shell function is necessary on HP because of a bug with mkdir -p
#  that means that you cannot do mkdir -p on an automounted file system
#  What we have to do is manually create each directory in turn.
#  It returns 0 in $RETVAL if successful, otherwise 1 in $RETVAL
#
###############################################################################
create_dir(){
	TARGET_DIR="$1"
	CURR_DIR=""
	(
        	IFS=/
		# For each directory in the path, starting from the top, if it
		# doesn't exist, create it.
		for dir in $TARGET_DIR
		do
			CURR_DIR="$CURR_DIR"/"$dir"
			test -d "$CURR_DIR" || mkdir "$CURR_DIR" 2>/dev/null
		done
        )

	# The target directory should now be made. If not return an error
	# in RETVAL
	RETVAL=0
	test -d "$TARGET_DIR" || RETVAL=1
}

#!/bin/sh
###############################################################################
#
#       Name:           non-interactive.sh
#
#       Version:        $Id: $
#
#       Purpose:        Sets up a non-interactive working environment.
#
#       Copyright 2005 Citrix Systems, Inc. All rights reserved.
#
###############################################################################

user_prompt()
{
    true
}

user_echo()
{
    true
}

getyesno()
{
    ANSWER=$INSTALLER_YES
}

NON_INTERACTIVE=true
#!/bin/sh
###############################################################################
#
#	Name:		echo_c.sh
#
#	$Id$
#
#	Purpose:	Provides an echo function with the trailing line-feed
#			suppressed for those platforms that need -n to do the
#			suppression.
#
#	Copyright 1996-1998, 2008 Citrix Systems, Inc. All rights reserved.
#
###############################################################################


###############################################################################
#
#  Echo without a new line - required as a function because some platforms
#  like a \c and some like a -n to indicate that a new-line should not
#  be output.
#
#  Pass strings with different encodings as separate arguments.
#
###############################################################################
echo_no_nl(){
	"$ECHO_CMD" -n "$@"
}

#!/bin/sh 
###############################################################################
#
# Name:           setup-netscape.sh
#
# $Id$
#
#  Purpose:	Installs ICA plugin and wfica as a helper application,
#		and adds new mime type.
#
#  Copyright 1996-2017 Citrix Systems, Inc. All rights reserved.
#
###############################################################################

INST_DIR="$ICAInstDir"
TMP_NAME=`mktemp /tmp/ICAnetscape.XXXXXX`
NETSCAPE_DIR="/usr/local/lib/netscape"
PLUGIN_PATH="/usr/local/netscape/plugins"
PLUGIN_NAME=npica.so
MAILCAP_FILE=mailcap
MIMETYPE_FILE=mime.types
WFICA_SHELL=wfica.sh
OTHER_BROWSER_DIRS="/usr/lib/mozilla /usr/lib64/mozilla /usr/lib/epiphany/*"

PLUGIN_TXT1="application/x-ica;"
PLUGIN_TXT2="/${WFICA_SHELL} %s; x-mozilla-flags=plugin:Citrix ICA Client"
FILE_ASSOC_TEXT="type=application/x-ica exts=ica desc=\"Citrix ICA\""

if [ $PORT = "linuxx64" ]; then
	COMMON_PLUGIN_DIR=/usr/lib64/browser-plugins
else
	COMMON_PLUGIN_DIR=/usr/lib/browser-plugins
fi

op_system=`uname -s`
EXTRA_BROWSERS=

case $op_system in
Linux)
	if test -d "/opt/netscape/communicator"
	then
		# Caldera may need this, but we don't need the alternate integrations
		# of mailcap and mime.types
		ALT_PLUGIN_PATH="/opt/netscape/communicator"
	fi
	ALT_INSTALL=0
	;;
*)
	ALT_INSTALL=0
	;;
esac

###############################################################################
#
#	netscape_integrate()
#
#	This function is called to add entries to netscape configuration files
#	to support charlotte. The following is added:
#
#	${NETSCAPE_DIR}/${MAILCAP_FILE}:
#	application/x-ica; ${INST_DIR}/${WFICA_SHELL} %s; x-mozilla-flags=plugin:Citrix ICA Client
#
#	${NETSCAPE_DIR}/${MIMETYPE_FILE}:
#	type=application/x-ica exts=ica desc="Citrix ICA"
#
###############################################################################

netscape_integrate()
{
  user_echo $integrate_netscape1
  if test -z "$SU_INSTALL"
  then
    set_npui_variables
    found_netscape 
  else
    search_ok=0
    EXTRA_BROWSERS=`find_browsers`
  
    if  test -d "/usr/lib/netscape"\
	|| test -d "/usr/local/netscape"\
	|| test -d "/opt/netscape"\
	|| test -d "/usr/local/lib/netscape"\
	|| test -d "/usr/dt/appconfig/netscape"\
	|| test "$MOZILLA_HOME" != "" -a -d "$MOZILLA_HOME"\
	|| test "$EXTRA_BROWSERS" != ""
    then
  	user_echo $integrate_netscape3
  	found_netscape 
    else
	user_prompt "$integrate_netscape2"
 	getyesno $INSTALLER_NO
	if [ "$ANSWER" = "$INSTALLER_YES" ]
	then
		found_netscape
	else
  		user_echo $integrate_netscape6
	fi
    fi
  fi

  # Create utility script files in install root.
  create_wfica_script
}


################################################################################
#
# link_plugin()
#
# Create a symbolic link named $PLUGIN_NAME, in the $PLUGIN_PATH directory, 
# that points to the plugin in the installation directory.
# 
################################################################################
link_plugin()
{
    # Ensure the plugin directory exists
    if [ -d "$PLUGIN_PATH" ]
    then
	rm -f "$PLUGIN_PATH/$PLUGIN_NAME"
    else 
	create_dir "$PLUGIN_PATH"
    fi
    ln -s "$INST_DIR/$PLUGIN_NAME" "$PLUGIN_PATH/$PLUGIN_NAME"
}

################################################################################
#
# link_extra_plugins $PATH
#
# Puts a link to the plugin into installations of newer versions of mozilla
# that may not look at the traditional place for plugins.
# 
################################################################################
link_extra_plugins()
{
    for dir in "$@"
    do
	if [ -d "$dir" ]
	then
            if [ -d "$dir/lib/plugins" ]
	    then
	        PLUGIN_PATH=$dir/lib/plugins
            else
	        PLUGIN_PATH=$dir/plugins
            fi
	    link_plugin
	fi
    done
}

################################################################################
# unlink_one_plugin $PATH
#
# Removes a file only if it is a link to our plug-in.
# 
################################################################################
unlink_one_plugin()
{
    [ -h "$1" ] && \
    ls -l "$1" | grep "> $INST_DIR/$PLUGIN_NAME" > /dev/null && \
    rm -f "$1"
}

################################################################################
#
# unlink_plugins $PATH
#
# Removes any file named $PLUGIN_NAME in the given directory.
# 
################################################################################
unlink_plugins()
{
    for dir in "$@"
    do
	if [ -d "$dir" ]
	then
	    unlink_one_plugin "$dir/lib/plugins/$PLUGIN_NAME"
	    unlink_one_plugin "$dir/plugins/$PLUGIN_NAME"
	fi
    done
}

################################################################################
#
#	found_netscape()
#
#	Sub-function, called by netscape_integrate to do its dirty work. Do
#	not call this function separately.
#
################################################################################
found_netscape()
{
  ALREADY_MIMETYPE=0
  ALREADY_MAILCAP=0
  LINK_PLUGIN=1

  # Ensure the directory exists
  if [ ! -d "${NETSCAPE_DIR}" ]
  then
	create_dir "${NETSCAPE_DIR}"
  fi
  # does mailcap exist ?
  if [ ! -f "${NETSCAPE_DIR}"/${MAILCAP_FILE} ]
  then
	"$ECHO_CMD" "# ${NETSCAPE_DIR}/${MAILCAP_FILE}" > "${NETSCAPE_DIR}"/${MAILCAP_FILE}
  fi
  # does mime.types exist ?
  if [ ! -f "${NETSCAPE_DIR}"/${MIMETYPE_FILE} ]
  then
	# mime.types must have a distinctive string on the first line...
	"$ECHO_CMD" "#--Netscape Communications Corporation MIME Information" > "${NETSCAPE_DIR}"/${MIMETYPE_FILE}
  fi
  # does mime.types already contain our line?
  if grep application/x-ica "${NETSCAPE_DIR}"/${MIMETYPE_FILE} >/dev/null
  then   
	ALREADY_MIMETYPE=1
  fi
  # does mailcap already contain our line?
  if grep application/x-ica "${NETSCAPE_DIR}"/${MAILCAP_FILE} >/dev/null
  then
        ALREADY_MAILCAP=1
  fi
  if test $ALREADY_MIMETYPE -eq 1 || test $ALREADY_MAILCAP -eq 1
  then
	user_prompt "$integrate_netscape4"
	getyesno $INSTALLER_YES
        if [ "$ANSWER" = "$INSTALLER_YES" ]
	then
		if test $ALT_INSTALL -eq 1
		then
			alt_integrate
		fi
		if test "$EXTRA_BROWSERS" != ""
		then
		    extra_integrate $EXTRA_BROWSERS
                    save_extra_browser_list $EXTRA_BROWSERS
		fi
		if [ $ALREADY_MAILCAP -eq 1 ]
		then
			sed '/x-ica/d' "${NETSCAPE_DIR}"/${MAILCAP_FILE} >${TMP_NAME}
			mv ${TMP_NAME} "${NETSCAPE_DIR}"/${MAILCAP_FILE}
		fi

		if [ $ALREADY_MIMETYPE -eq 1 ]
		then
			sed '/x-ica/d' "${NETSCAPE_DIR}"/${MIMETYPE_FILE} >${TMP_NAME}
			mv ${TMP_NAME} "${NETSCAPE_DIR}"/${MIMETYPE_FILE}
		fi

		echo_no_nl ${PLUGIN_TXT1} "-i" "${INST_DIR}" >> "${NETSCAPE_DIR}"/${MAILCAP_FILE}
		"$ECHO_CMD" ${PLUGIN_TXT2} >> "${NETSCAPE_DIR}"/${MAILCAP_FILE}
		"$ECHO_CMD" $FILE_ASSOC_TEXT >> "${NETSCAPE_DIR}"/${MIMETYPE_FILE}
	else
                LINK_PLUGIN=0
  		user_echo $integrate_netscape6
        fi
  else
	if test $ALT_INSTALL -eq 1
	then
		alt_integrate
	fi
	if test "$EXTRA_BROWSERS" != ""
	then
	    extra_integrate $EXTRA_BROWSERS
        save_extra_browser_list $EXTRA_BROWSERS
	fi
	echo_no_nl ${PLUGIN_TXT1} "-i" "${INST_DIR}" >> "${NETSCAPE_DIR}"/${MAILCAP_FILE}
	"$ECHO_CMD" ${PLUGIN_TXT2} >> "${NETSCAPE_DIR}"/${MAILCAP_FILE}
	"$ECHO_CMD" $FILE_ASSOC_TEXT >> "${NETSCAPE_DIR}"/${MIMETYPE_FILE}
  fi

  if [ $LINK_PLUGIN -eq 1 ]
  then
    if [ -z "$SU_INSTALL" ]
    then
        if [ -d "$HOME/.netscape" ]
        then
            PLUGIN_PATH="$HOME/.netscape/plugins"
            link_plugin
        fi
        # If user has never run firefox this directory will not exist, so create it
        # assuming the user will want to run firefox later, and will want the plugin
        # to be integrated. DON'T do this for Netscape, as this browser is hardly ever
        # used now.
        if [ ! -d "$HOME/.mozilla" ]
        then
            mkdir -p "$HOME/.mozilla/plugins"
        fi
        PLUGIN_PATH="$HOME/.mozilla/plugins"
        link_plugin

    else
        # Integrate the plugin: use one of the following variables as the path, in order
        # of preference left to right :
        # $NPX_PLUGIN_PATH $MOZILLA_HOME/plugins $ALT_PLUGIN_PATH/plugins ${NETSCAPE_DIR}/plugins
        if test x$NPX_PLUGIN_PATH = x
        then
                if test x$MOZILLA_HOME = x
                then
			if test x$ALT_PLUGIN_PATH = x
			then
                        	PLUGIN_PATH="${NETSCAPE_DIR}"/plugins
			else
                        	PLUGIN_PATH=$ALT_PLUGIN_PATH/plugins
			fi
                else
                        PLUGIN_PATH=$MOZILLA_HOME/plugins
                fi
        else
                PLUGIN_PATH=$NPX_PLUGIN_PATH
        fi
        
        link_plugin
        if test "$EXTRA_BROWSERS" != ""
	then
	    link_extra_plugins $EXTRA_BROWSERS
	fi

	if test -d "$COMMON_PLUGIN_DIR"
        then
	    PLUGIN_PATH="$COMMON_PLUGIN_DIR"
	    link_plugin
	fi
    fi
    user_echo $integrate_netscape5
  fi
}

###########################################################################
#
#	netscape_disintegrate()
#
#	This functions checks if we have updated netscapes configuration.
#	If so, we offer to undo these changes.
#
###########################################################################

netscape_disintegrate()
{
  MAILCAP_MODIFIED=0
  MIMETYPES_MODIFIED=0
  INST_DIR=$ICAInstDir
  UNLINK_PLUGIN=1

  if test -z "$SU_INSTALL"
  then
	set_npui_variables
    EXTRA_BROWSERS=""
  else
	EXTRA_BROWSERS=`find_browsers`
  fi

  # Does file exist?
  if [ -f "${NETSCAPE_DIR}"/${MIMETYPE_FILE} ]
  then
	# Has it been altered?
	if grep application/x-ica "${NETSCAPE_DIR}"/${MIMETYPE_FILE} >/dev/null
  	then
        	MIMETYPES_MODIFIED=1
	fi
  fi

  # Does file exist?
  if [ -f "${NETSCAPE_DIR}"/${MAILCAP_FILE} ]
  then
	# Has it been altered?
	if grep "${ICAInstDir}"/wfica "${NETSCAPE_DIR}"/${MAILCAP_FILE} >/dev/null
	then
		MAILCAP_MODIFIED=1
	fi
  fi

  if test $MAILCAP_MODIFIED -eq 1
  then
        user_prompt "$disintegrate_netscape1"
        getyesno $INSTALLER_YES
        if [ "$ANSWER" = "$INSTALLER_YES" ]
	then
		if test $ALT_INSTALL -eq 1
		then
			alt_disintegrate
		fi
		if test "$EXTRA_BROWSERS" != ""
		then
		    extra_disintegrate $EXTRA_BROWSERS
		fi
		if [ $MAILCAP_MODIFIED -eq 1 ]
		then
			sed '/x-ica/d' "${NETSCAPE_DIR}"/${MAILCAP_FILE} >${TMP_NAME}
			mv ${TMP_NAME} "${NETSCAPE_DIR}"/${MAILCAP_FILE}
		fi

		if [ $MIMETYPES_MODIFIED -eq 1 ]
                then
                        sed '/x-ica/d' "${NETSCAPE_DIR}"/${MIMETYPE_FILE} >${TMP_NAME}
                        mv ${TMP_NAME} "${NETSCAPE_DIR}"/${MIMETYPE_FILE}
                fi
        else
                UNLINK_PLUGIN=0
	fi
  else
        UNLINK_PLUGIN=0
  fi

  if [ $UNLINK_PLUGIN -eq 1 ]
  then
    if [  -z "$SU_INSTALL" ]
    then
        unlink_plugins "$HOME/.netscape" "$HOME/.mozilla"
    else
	    # Remove any plugin symlinks from all possible locations
	    if test x$NPX_PLUGIN_PATH != x
	    then
        	rm -f $NPX_PLUGIN_PATH/$PLUGIN_NAME
	    fi
            unlink_one_plugin "$COMMON_PLUGIN_DIR/$PLUGIN_NAME"
	    unlink_plugins $MOZILLA_HOME $ALT_PLUGIN_PATH "$NETSCAPE_DIR" $EXTRA_BROWSERS
	fi
    user_echo $disintegrate_netscape2
  fi
}

################################################################################
#
#       alt_integrate()
#
#       special cases: IRIX and DEC. We add prefs to an additional location
#
################################################################################
alt_integrate()
{
  ALREADY_MIMETYPE2=0
  ALREADY_MAILCAP2=0
  # Ensure the directory exists
  if [ ! -d ${ALT_NETSCAPE_DIR} ]
  then
        create_dir ${ALT_NETSCAPE_DIR}
  fi
  # does mailcap exist ?
  if [ ! -f ${ALT_NETSCAPE_DIR}/${MAILCAP_FILE} ]
  then
        "$ECHO_CMD" "# ${ALT_NETSCAPE_DIR}/${MAILCAP_FILE}" >${ALT_NETSCAPE_DIR}/${MAILCAP_FILE}
  fi
  # does mime.types exist ?
  if [ ! -f ${ALT_NETSCAPE_DIR}/${MIMETYPE_FILE} ]
  then
        # mime.types must have a distinctive string on the first line...
        "$ECHO_CMD" "#--Netscape Communications Corporation MIME Information" >${ALT_NETSCAPE_DIR}/${MIMETYPE_FILE}
  fi
  # does mime.types already contain our line?
  if grep application/x-ica ${ALT_NETSCAPE_DIR}/${MIMETYPE_FILE} >/dev/null
  then
        ALREADY_MIMETYPE2=1
  fi
  # does mailcap already contain our line?
  if grep application/x-ica ${ALT_NETSCAPE_DIR}/${MAILCAP_FILE} >/dev/null
  then
        ALREADY_MAILCAP2=1
  fi
  if test $ALREADY_MIMETYPE2 -eq 1 || test $ALREADY_MAILCAP2 -eq 1
  then
        if [ $ALREADY_MAILCAP2 -eq 1 ]
        then
        	sed '/x-ica/d' ${ALT_NETSCAPE_DIR}/${MAILCAP_FILE} >${TMP_NAME}
                mv ${TMP_NAME} ${ALT_NETSCAPE_DIR}/${MAILCAP_FILE}
        fi

        if [ $ALREADY_MIMETYPE2 -eq 1 ]
        then
                sed '/x-ica/d' ${ALT_NETSCAPE_DIR}/${MIMETYPE_FILE} >${TMP_NAME}
                mv ${TMP_NAME} ${ALT_NETSCAPE_DIR}/${MIMETYPE_FILE}
        fi
  fi
  echo_no_nl ${PLUGIN_TXT1} "-i" "${INST_DIR}" >>${ALT_NETSCAPE_DIR}/${MAILCAP_FILE}
  "$ECHO_CMD" ${PLUGIN_TXT2} >>${ALT_NETSCAPE_DIR}/${MAILCAP_FILE}
  "$ECHO_CMD" $FILE_ASSOC_TEXT >>${ALT_NETSCAPE_DIR}/${MIMETYPE_FILE}
}

################################################################################
#
#       extra_integrate()
#
#       special cases: IRIX and DEC. We add prefs to an additional location
#
################################################################################
extra_integrate()
{
    SAVE_ALT_NETSCAPE_DIR=$ALT_NETSCAPE_DIR
    for dir in "$@"
    do
        ALT_NETSCAPE_DIR=$dir
        alt_integrate
    done
    ALT_NETSCAPE_DIR=$SAVE_ALT_NETSCAPE_DIR
}

###########################################################################
#
#       alt_disintegrate()
#
#       Special cases: IRIX and DEC. We uninstall from a second location.
#
###########################################################################

alt_disintegrate()
{
  # Does file exist?
  if [ -f ${ALT_NETSCAPE_DIR}/${MIMETYPE_FILE} ]
  then
        # Has it been altered?
        if grep application/x-ica ${ALT_NETSCAPE_DIR}/${MIMETYPE_FILE} >/dev/null
        then
                sed '/x-ica/d' ${ALT_NETSCAPE_DIR}/${MIMETYPE_FILE} >${TMP_NAME}
                mv ${TMP_NAME} ${ALT_NETSCAPE_DIR}/${MIMETYPE_FILE}
        fi
  fi

  # Does file exist?
  if [ -f ${ALT_NETSCAPE_DIR}/${MAILCAP_FILE} ]
  then
        # Has it been altered?
        if grep application/x-ica ${ALT_NETSCAPE_DIR}/${MAILCAP_FILE} >/dev/null
        then
        	sed '/x-ica/d' ${ALT_NETSCAPE_DIR}/${MAILCAP_FILE} >${TMP_NAME}
                mv ${TMP_NAME} ${ALT_NETSCAPE_DIR}/${MAILCAP_FILE}
        fi
  fi
}

###########################################################################
#
#       extra_disintegrate()
#
#       Special cases: IRIX and DEC. We uninstall from a second location.
#
###########################################################################

extra_disintegrate()
{
    SAVE_ALT_NETSCAPE_DIR=$ALT_NETSCAPE_DIR
    for dir in "$@"
    do
        ALT_NETSCAPE_DIR=$dir
        alt_disintegrate
    done
    ALT_NETSCAPE_DIR=$SAVE_ALT_NETSCAPE_DIR
}

################################################################################
#
#	set_npui_variables
#
#       Sets the environment variables used in the netscape configuration
#       routines for a non-privileged user installation.
#
################################################################################
set_npui_variables()
{
    MAILCAP_FILE=.mailcap
    MIMETYPE_FILE=.mime.types
    NETSCAPE_DIR=${HOME}
    ALT_INSTALL=0
}

################################################################################
#
# create_wfica_script()
#
# Create a script that when called with a filename argument, will attempt to
# launch a client session, using the argument as an ICA connection file.
# 
################################################################################
create_wfica_script()
{
    cat > "${INST_DIR}"/${WFICA_SHELL} << EOF
#!/bin/sh
ICAROOT=${INST_DIR} 
export ICAROOT
LD_LIBRARY_PATH=${INST_DIR}/lib
export LD_LIBRARY_PATH
\$ICAROOT/wfica -file \$1
EOF
    chmod 755 "${INST_DIR}"/${WFICA_SHELL}
    if [ -z "$SU_INSTALL" -a -n "$UMASK_STR" ] ;  then
	chmod $UMASK_STR "${INST_DIR}"/${WFICA_SHELL}
    fi
}

###############################################################################
#
# IsBrowserDir $DIR
# 
# Is this the configuration directory for a Web browser?
#
###############################################################################
IsBrowserDir()
{
    [ -d "$1" ] &&
      [ -x "$1/run-mozilla.sh" -o -d "$1/plugins" -o -d "$1/chrome" ]
}

###############################################################################
# find_extra_bin_dirs $PROGRAM
# 
# Given a path and name of an executable, recurse down symbolic links
# looking for an indication as to whether the installation requires its own
# integration. If so, echo the installation directory required.
#
###############################################################################
find_extra_bin_dirs()
{    
    BINFILE="$1"
    BASENAMES=
    DONE=0
    BINDIR=""
    LINKDEPTH=""
    MAXDEPTH="........"

    while [ $DONE -eq 0 ]
    do
	# Get containing directory and save current file name.
	DIR=`dirname $BINFILE`
        BASENAMES="$BASENAMES `basename $BINFILE`"

	# Is the mozilla script in this directory?
	if IsBrowserDir "$DIR"
	then
	    # this is the directory we want.
	    BINDIR=$DIR
	    DONE=1
	else
	    # Is the file a symbolic link
	    if [ -h "$BINFILE" ]
	    then
		# Look at the file that is pointed to.
		LINK=`ls -l "$BINFILE" | sed "s/.* -> //g"`
		echo $LINK | grep "^/" >/dev/null
		if [ $? -eq 0 ]
		then
		    BINFILE="$LINK"
		else
		    BINFILE="$DIR/$LINK"
		fi
		
		# Check that this really is a file.
		# Increment the link depth.
		LINKDEPTH="${LINKDEPTH}."
		# Is the depth getting too far?
		if [ $LINKDEPTH = $MAXDEPTH ]
		then
		    # Saveguard against circular references.
		    DONE=1
		fi
	    else
                # Some mozilla installations have the launch
                # script in a bin directory, rather than a
                # symbolic link. Attempt to find the actual
                # installation directory.
                BINDIR=`grep "^MOZ_DIST_BIN=" $BINFILE  | sed "s/^MOZ_DIST_BIN=//g" | sed "s/.*\"\(.*\)\".*/\1/g"`
                if IsBrowserDir "$BINDIR"
                then
		    DONE=1
		else
                    # Look for parallel "lib" directory
		    BINDIR=""
		    DIR="`dirname $DIR`/lib"
		    if [ -d "$DIR" ]
		    then
			if IsBrowserDir "$DIR"
			then
			    BINDIR=$DIR
                        else
			    # Look in plausible subdirs of lib.
			    for d in $BASENAMES
			    do
	                        TD="$DIR/$d"
				if [ $DONE -eq 0 ] && IsBrowserDir "$TD"
				then
				    DONE=1
				    BINDIR="$TD"
				fi
			    done
			fi
		    fi
		    DONE=1
                fi
	    fi
	fi
    done
    echo "$BINDIR"
}

###############################################################################
#
# find_mozilla_browsers
#
# Returns a list of directories containing new-style mozilla-based browsers 
# that may be launched by files on the $PATH.
#
###############################################################################
find_mozilla_browsers()
{

    PROGS="firefox mozilla iceweasel netscape"
    DIST_BIN_DIR=""
    STANDARD_DIRS="/usr/local/netscape /usr/local/mozilla /usr/local/firefox /usr/lib/mozilla"
    PKGINFO=/usr/bin/pkginfo

    if [ "$BROWSER" ]
    then
	PROGS="$PROGS `basename $BROWSER`"
	DIR=`dirname "$BROWSER"`
	if [ "$DIR" ]
	then
	    STANDARD_DIRS="$STANDARD_DIRS $DIR"
	fi
    fi
	
    for d in `echo $PATH | sed "s/:/ /g"` $STANDARD_DIRS
    do
	echo $d
    done | sort -u | while read d
    do 
	for p in $PROGS
	do    
	    # Find the each instance of each named browser
	    PROG=$d/$p
	    if test -f $PROG
	    then
		# Where is the installation and what version is it?
		find_extra_bin_dirs $PROG
	    fi
	done
    done
}

###############################################################################
#
# find_browsers
#
# Returns on stdout a list of directories containing browser files
# such as a "plugins" directory.
#
###############################################################################
find_browsers()
{
    {
	for i in $OTHER_BROWSER_DIRS
	do
	    IsBrowserDir $i && echo $i
	done
	find_mozilla_browsers
    } | sort -u
}

BROWSER_LIST_KEY="BROWSER_DIR"
################################################################################
#
# save_extra_browser_list $EXTRA_BROWSERS
#
# Saves the list of directories that were used to put extra mimetype, mailcap
# and possibly plugin settings into.
#
################################################################################
save_extra_browser_list()
{
    ver_file="${ICAInstDir}"/pkginf/Ver.core.$PORT
    for dir in "$@"
    do
        if test -f "${ver_file}"
        then
            echo "${BROWSER_LIST_KEY}=$dir" >> "${ver_file}"
        fi
    done
}
#!/bin/sh
###############################################################################
#
#	Name:		inst_mtch.sh
#
#	$Id$
#
#	Purpose:	Installs the MultiTOuch module, if present.
#
#	Copyright 2009-2023 Citrix Systems, Inc.  All Rights Reserved.
#
###############################################################################

install_mtch()
{
    ica_dir=$1
	installMtchRule "${ica_dir}"
}

remove_mtch()
{
	removeMtchRule
}

#!/bin/sh

UDEV_RULES_DIR=/etc/udev/rules.d
MTCH_RULE_FILE="50-ica-mtch.rules"

installMtchRule()
{
    ICAROOT=$1
    # Install the udev rule
    install -m 644 "$ICAROOT/config/$MTCH_RULE_FILE" "$UDEV_RULES_DIR" > /dev/null 2>&1

    # Reload the updated udev rule
    udevadm control --reload > /dev/null 2>&1
    udevadm trigger > /dev/null 2>&1
}

removeMtchRule()
{
    # Remove the mtch udev rule
    rm -f "$UDEV_RULES_DIR/$MTCH_RULE_FILE" > /dev/null 2>&1

    # Reload the updated udev rule
    udevadm control --reload > /dev/null 2>&1
    udevadm trigger > /dev/null 2>&1
}

#!/bin/sh
###############################################################################
#
#	Name:		inst_hid.sh
#
#	$Id$
#
#	Purpose:	Installs the HID module, if present.
#
#	Copyright 2009-2023 Citrix Systems, Inc.  All Rights Reserved.
#
###############################################################################

install_hid()
{
    ica_dir=$1
	installHidRule "${ica_dir}"
}

remove_hid()
{
	removeHidRule
}

#!/bin/sh

UDEV_RULES_DIR=/etc/udev/rules.d
HID_RULE_FILE="69-ica-hid.rules"

installHidRule()
{
    ICAROOT=$1
    # Install the hid udev rule
    install -m 644 "$ICAROOT/config/$HID_RULE_FILE" "$UDEV_RULES_DIR" > /dev/null 2>&1

    # Reload the updated udev rule
    udevadm control --reload > /dev/null 2>&1
    udevadm trigger > /dev/null 2>&1
}

removeHidRule()
{
    # Remove the hid udev rule
    rm -f "$UDEV_RULES_DIR/$HID_RULE_FILE" > /dev/null 2>&1

    # Reload the updated udev rule
    udevadm control --reload > /dev/null 2>&1
    udevadm trigger > /dev/null 2>&1
}

###############################################################################
#
#       Name:           link_util.sh
#
#       $Id:$
#
#       Purpose:        Functions for processing links.
#
#       Copyright 2011 Citrix Systems, Inc. All rights reserved.
#
###############################################################################

LNKTST0=`echo $LNKTST | sed -e "y/ /:/"`

# Return true if a link points to window the $ICAROOT directory
SymlinkPointsToICAInst()
{
    link="$1"

    (
        IFS=' '
        # If this is not actually a symlink then return "false".
        $LNKTST "$link" || return 1

        # If we cannot read the symlink, then return "true" - we will delete it anyway.
        if [ ! -r "$link" ]
        then
            return 0
        fi

        deref=`ls -l $link | sed -e "s%^.*-> *%%"`

        echo $deref | grep "$ICAInstDir" > /dev/null 2>&1

        if [ $? -eq 0 ]
        then
            return 0
        else
            return 1
        fi
    )
}
#!/bin/sh
###############################################################################
#
#       Name:           setup-GST.sh
#
#       $Id:
#
#       Purpose:        Handles integration of the client with GStreamer,
#                       by adding a link to libgstflatstm.so and libctxbeffect.so.
#                       This link goes into the GStreamer library directory
#                       for privileged installs and into
#                       $HOME/.gstreamer-[1.0|0.10]/plugins for user installs.
#
#       Copyright 2003-2017 Citrix Systems, Inc. All rights reserved.
#
###############################################################################

PLUGIN_LIBRARY=libgstflatstm
BEFFECT_LIBRARY=libctxbeffect

###############################################################################
#
#       Directories to search for the GStreamer installation.
#       The paths must not have embedded spaces otherwise the
#       search algorithm doesn't work.
#
###############################################################################

DEFAULT_GST_TARGET_DIRS="/usr/lib /usr/local/lib /opt/gnome/lib /usr/lib64"
X86_GST_TARGET_DIRS="/usr/lib/i386-linux-gnu /usr/lib/x86_64-linux-gnu"
ARMHF_GST_TARGET_DIRS="/usr/lib/arm-linux-gnueabihf"
ARMEL_GST_TARGET_DIRS="/usr/lib/arm-linux-gnueabi"
ARM64_GST_TARGET_DIRS="/usr/lib/aarch64-linux-gnu"

###############################################################################
#
#	Record which version of GStreamer found
#	Current strategy is to favour 0.10 as we have issues with 1.0
#	1.0 will be selected though if it's the only version installed
#	on this machine. This is enforced by the order of searching in
#	the function GST_get_target_dirs
#
###############################################################################

SUFFIX1X=1.0
SUFFIXPOINT10=0.10
VERSION_FOUND=

###############################################################################
#
#	turnoff_multimedia()
#
#	This function is called to turn off multi media by editing the
#	module.ini file.
#
###############################################################################

turnoff_multimedia()
{
    MODULE_INI="$ICAInstDir/config/module.ini"
    sed -e 's/MultiMedia=On/MultiMedia=Off/' < "$MODULE_INI" > "${MODULE_INI}.tmp"
    chmod u+w "$MODULE_INI"
    cp "${MODULE_INI}.tmp" "$MODULE_INI"
    chmod u-w "$MODULE_INI"
    rm -f "${MODULE_INI}.tmp"

    return
}

###############################################################################
#
#	GST_integrate()
#
#	This function is called to integrate the client with GStreamer.
#
###############################################################################

GST_integrate()
{
    GST_get_target_dirs

    (
        # By setting IFS to be :, we avoid space-separation issues.
        IFS=:

        INTEGRATE=-1

        for i in $GST_TARGET_DIRS
        do
            if [ -d $i ]
            then
                INTEGRATE=1
            fi
        done
        if [ $INTEGRATE -eq -1 ]
        then
            user_echo $integrate_GST1
            return 2
        fi

        return 0
    )

    if [ $? -ne 0 ]
    then
        turnoff_multimedia
        return
    fi

    if [ -z "$SU_INSTALL" ]
    then
        # Non-privileged install. Just put in link for this user.
        GST_TARGET_DIRS="$HOME/.gstreamer-${VERSION_FOUND}/plugins"
    fi

    # Find the right question to ask on whether to proceed.

    (
        IFS=:
        PREVIOUS=-1
        for i in $GST_TARGET_DIRS
        do
            link=$i/${PLUGIN_LIBRARY}.so
            if $LNKTST0 $link
            then
                PREVIOUS=1
                break
            fi
        done

        if [ $PREVIOUS -eq -1 ]
        then
            user_prompt $integrate_GST2
        else
            # If we've already integrated with GST, ask if we want to re-do it.

            user_prompt $integrate_GST3
        fi
    )

    getyesno $INSTALLER_YES
    if [ "$ANSWER" = "$INSTALLER_NO" ]
    then
        turnoff_multimedia
        return
    fi

    # Get correct gst_read and gst_play in place
    if [ -f "$ICAInstDir/util/gst_play" ]; then
        #Remove the symbolic link before creating for upgrade scenario
        unlink "$ICAInstDir/util/gst_play"
        unlink "$ICAInstDir/util/gst_read"
    fi
    ln -s $ICAInstDir/util/gst_play${VERSION_FOUND} $ICAInstDir/util/gst_play
    ln -s $ICAInstDir/util/gst_read${VERSION_FOUND} $ICAInstDir/util/gst_read

    if [ -z "$SU_INSTALL" ]
    then
        # For a user level installation only one directory needs a link.
        # Make sure the directory exists, if we can.
        create_dir $GST_TARGET_DIRS
        if [ ! -d $GST_TARGET_DIRS ]
        then
            "$ECHO_CMD" "$integrate_GST4a $GST_TARGET_DIRS $integrate_GST4b"
            return
        fi
    fi

    # Spray links into directories.

    (
        IFS=:
        FAILED_LINK_DIRS=
        for i in $GST_TARGET_DIRS
        do
            if [ -d $i -a -w $i ]
            then
                FAILED=
                link=$i/${PLUGIN_LIBRARY}.so
                $LNKTST0 $link && rm -f $link
                ln -s "$GST_FILE_DIR/${PLUGIN_LIBRARY}${VERSION_FOUND}.so" $link || FAILED=1

                if [ -z "$FAILED" -a -f "$ICAInstDir/lib/${BEFFECT_LIBRARY}.so" ]
                then
                    link=$i/${BEFFECT_LIBRARY}.so
                    $LNKTST0 $link && rm -f $link
                    ln -s "$ICAInstDir/lib/${BEFFECT_LIBRARY}.so" $link || FAILED=1
                fi

                if [ -n "$FAILED" ]
                then
                    FAILED_LINK_DIRS="$FAILED_LINK_DIRS $i"
                fi
            else
                FAILED_LINK_DIRS="$FAILED_LINK_DIRS $i"
            fi
        done
        if [ -n "$FAILED_LINK_DIRS" ]
        then
            "$ECHO_CMD" "$integrate_GST5a $FAILED_LINK_DIRS $integrate_GST5b"
        else
            # Create symlinks for dependencies soname
            ldconfig -n "$ICAInstDir/lib/third_party"
        fi
    )
}

###########################################################################
#
#	GST_disintegrate()
#
#	This function is called to disintegrate the client with GStreamer.
#
###########################################################################

GST_disintegrate()
{
    if [ -n "$SU_INSTALL" ]
    then
        GST_get_target_dirs
    else
        # Non-privileged install. Just remove link for this user.
        GST_TARGET_DIRS="$HOME/.gstreamer-0.10/plugins $HOME/.gstreamer-1.0/plugins"
    fi

    # Remove dangling symbolic links to removed library files.

    (
        IFS=:
        for i in $GST_TARGET_DIRS
        do
            link=$i/${PLUGIN_LIBRARY}.so
            SymlinkPointsToICAInst $link && rm -f $link
            link=$i/${BEFFECT_LIBRARY}.so
            SymlinkPointsToICAInst $link && rm -f $link
        done
    )

    # Remove links to gst_read and gst_play

    rm -f $ICAInstDir/util/gst_play 2>/dev/null
    rm -f $ICAInstDir/util/gst_read 2>/dev/null

    # Remove third party dependencies links
    
    find $ICAInstDir/lib/third_party -type l -exec rm {} \; 2>/dev/null

    if [ -z "$SU_INSTALL" ]
    then
        # Non-privileged install. Remove plugin directory if it's empty.
        rmdir "$GST_TARGET_DIRS" > /dev/null 2>&1
    fi

    # The exit code at this point is quite likely to be non-zero (ie
    # false).  This can cause problems, particularly when uninstalling
    # the RPM.
    return 0
}

###########################################################################
#
#       GST_get_target_dirs()
#
#       This function is called to work out directories should hold the
#       link to the new GStreamer library and put these into the
#       GST_TARGET_DIRS environment variable.
#
###########################################################################
GST_get_target_dirs()
{
    if [ -z "$GST_TARGET_DIRS" ]
    then
        if [ -n "$GST_PLUGIN_PATH"  -o -n "$GST_PLUGIN_SYSTEM_PATH" ]
        then
            GST_TARGET_DIRS=`echo "$GST_PLUGIN_PATH $GST_PLUGIN_SYSTEM_PATH" | sed -e "y/ /:/"`
        else
            # Look for directories with GStreamer 1.0 or 0.10 library

            case $PORT in
                linuxx86|linuxx64)
                    ADDITIONAL_GST_TARGET_DIRS=$X86_GST_TARGET_DIRS;
                    ;;
                linuxarm64)
                    ADDITIONAL_GST_TARGET_DIRS=$ARM64_GST_TARGET_DIRS;
                    ;;
                linuxarmhf)
                    ADDITIONAL_GST_TARGET_DIRS=$ARMHF_GST_TARGET_DIRS;
                    ;;
                solaris|solx86)
                    ;;
            esac

            for i in $DEFAULT_GST_TARGET_DIRS $ADDITIONAL_GST_TARGET_DIRS
            do
                GSTREAMER_LIB=

                for k in ${SUFFIXPOINT10} ${SUFFIX1X}
                do
                    for f in $i/libgstreamer-${k}.so $i/libgstreamer-${k}.so.0
                    do
                        if [ -f $f ]
                        then
                            # Test the version of Gstreamer installed satisfies the minimum
                            # requirements of 0.10.15. This will also filter out other issues
                            # such as symbol lookup errors for gst_play0.10 seen on Raspian.
                            "$ICAInstDir/util/gst_play${k}" --gst-version-check >/dev/null 2>&1
                            if [ $? -eq 0 ]
                            then
                                # We've found a gstreamer library (or link to it)
                                # which passes some basic tests
                                GSTREAMER_LIB=$f
                                VERSION_FOUND=${k}
                            fi
                        fi
                    done
                    
                    if [ -n "$GSTREAMER_LIB" ]
                    then
                        break;
                    fi
                done

                # We need to add the directory to the list. However, on 64-bit
                # systems we can have both 32 and 64 bit libraries - if possible
                # we only want the 64-bit directory in that case.
                if [ \( -n "$GSTREAMER_LIB" \) -a -d $i/gstreamer-${VERSION_FOUND} ]
                then
                    # Are we on a 32 or 64 bit architecture system?
                    uname -m | grep 64 > /dev/null 2>&1 
                    arch_64_bit=$? # 0 if 64-bit, 1 if 32-bit

                    # Is the library 32 or 64 bit?
                    file --help > /dev/null 2>&1 # Check for existence of 'file' command
                    if [ $? -eq 0 ]
                    then
                        file -L $GSTREAMER_LIB | grep -e "64-bit" > /dev/null 2>&1
                        dir_64_bit=$? # 0 if 64-bit, 1 if 32-bit
                    else
                        # If 'file' not present, assume dir matches architecture. This
                        # assumption is messy because we can end up having links to
                        # both 32 and 64 bit dirs on 64-bit, though this seems safe.
                        dir_64_bit=$arch_64_bit
                    fi

                    # Add the dir to the list if file and system bit-ness match.
                    if [ $arch_64_bit -eq $dir_64_bit ]
                    then
                        list="$list $i/gstreamer-${VERSION_FOUND}"
                    fi
                fi
            done

            GST_TARGET_DIRS=`echo $list | sed -e "y/ /:/"`
        fi
    fi
}
#!/bin/sh
###############################################################################
#
#	Name:		inst_AppProtection.sh
#
#	$Id$
#
#	Purpose:	Installs the AppProtection module, if present.
#
#	Copyright 2020 Citrix Systems, Inc.  All Rights Reserved.
#
###############################################################################
AppProtectionDir=""
AppProtectionVer=""

# This to have compatibility while upgrading from older version where the name was sentryservice
is_sentrybay_installed()
{
    [ -f /usr/lib/systemd/system/sentryservice-install.service ]
}

is_AppProtection_installed()
{
    [ -f /usr/lib/systemd/system/AppProtectionService-install.service ]
}

pending_reboot()
{
    if [ -f /usr/lib/systemd/system/AppProtectionService-remove.service ]; then
        user_echo "$instsentry8"
        exit 1
    fi
}

check_argument()
{
    if [ -f /opt/AppProtection/.version ]; then
        diff /opt/AppProtection/.version /usr/local/bin/AppProtection/.version &> /dev/null
    fi
}

remove_file_util()
{
    fileTobeRemoved=$1
    if [ -f $fileTobeRemoved ]
    then
		rm -rf $fileTobeRemoved
        if [ $? -ne 0 ]
        then
			user_echo "$instsentry4"
        fi
    fi
}

remove_dir_util()
{
    dirTobeRemoved=$1
    if [ -d $dirTobeRemoved ]
    then
		rm -rf $dirTobeRemoved
        if [ $? -ne 0 ]
        then
			user_echo "$instsentry4"
        fi
    fi
}

install_AppProtectionFiles()
{
    tar -C / -zxf ${ica_dir}/util/AppProtection-service.tar.gz --strip-components=1 AppProtectionService-install
    ln -sf /usr/local/lib/AppProtection/libAppProtection.so.$AppProtectionVer /usr/local/lib/AppProtection/libAppProtection.so
    systemctl daemon-reload

    for service in AppProtectionService-install preload-library-install preload-library-remove;
    do
        systemctl enable ${service} &> /dev/null
    done

    systemctl start preload-library-install
    systemctl start AppProtectionService-install

    if [ -x "/usr/bin/dconf" ]
    then
        /usr/bin/dconf update
    fi
}

upgrade_AppProtection()
{
    if [ -f /usr/local/bin/AppProtection/.version ]; then
        InstalledVer=`cat /usr/local/bin/AppProtection/.version`
        if ! check_argument; then
            systemctl stop AppProtectionService-install
            install_AppProtectionFiles
            user_echo "$instsentry5"
        fi
    fi
}

install_AppProtection()
{
    ica_dir=$1
    # user_echo "ICA dir = $ica_dir"
    AppProtectionSystemdSupport=$(which systemctl)
    AppProtectionDir=${ica_dir}/AppProtectionService-install
    mkdir -p /opt/AppProtection
    tar -C /opt/AppProtection -zxf ${ica_dir}/util/AppProtection-service.tar.gz --strip-components=5 AppProtectionService-install/usr/local/bin/AppProtection/.version
    AppProtectionVer=`cat /opt/AppProtection/.version`
    if [ ! -z $AppProtectionSystemdSupport ]; then
        # pending_reboot
        if [ -f /usr/lib/systemd/system/AppProtectionService-remove.service ]; then
            user_echo "$instsentry8"
        elif is_AppProtection_installed; then
            upgrade_AppProtection
        else #Install 
            install_AppProtectionFiles
        fi
    else
        user_echo "$instsentry3"
    fi
    rm -rf /opt/AppProtection
}

remove_AppProtection()
{
    # pending_reboot
    ica_dir=$1
    if [ -f /usr/lib/systemd/system/AppProtectionService-remove.service ]; then
        user_echo "$instsentry8"
	elif is_AppProtection_installed; then
        systemctl start preload-library-remove
        for service in preload-library-install preload-library-remove;
        do
            systemctl disable ${service} &> /dev/null
        done
        systemctl disable AppProtectionService-install &> /dev/null
        systemctl stop AppProtectionService-install
        remove_file_util "/usr/lib/systemd/system/AppProtectionService-install.service"
        tar -C / -zxf ${ica_dir}/util/AppProtection-service.tar.gz --strip-components=1 AppProtectionService-remove
        systemctl daemon-reload
        systemctl enable AppProtectionService-remove &> /dev/null
        user_echo "$instsentry7"
    fi
}

remove_sentrybay()
{
	if is_sentrybay_installed; then
        rm -rf /etc/ld.so.preload
        systemctl disable sentryservice-install &> /dev/null
        systemctl stop sentryservice-install
        remove_file_util "/usr/lib/systemd/system/sentryservice-install.service"
        systemctl daemon-reload
        #systemctl enable sentryservice-remove &> /dev/null
    fi
    remove_dir_util "/usr/local/bin/sentrybay"
    remove_dir_util "/usr/local/lib/sentrybay"
    remove_file_util "/etc/dbus-1/session.d/org.gnome.Shell.Screencast.conf"
    remove_file_util "/etc/dbus-1/session.d/org.gnome.Shell.Screenshot.conf"
}
#!/bin/sh
#
# This script now only drops a manifest file for google and edge browsers so that extension picks up the 
# right location for nativemessagingHost
#
# EXT ID: dbdlmgpfijccjgnnpacnamgdfmljoeee
#
# 
# Verify 1: check if manifest file is added at this path : ~/.config/google-chrome/NativeMessagingHosts
# Verify 2: check if manifest file is added at this path : ~/.config/microsoft-edge-beta/NativeMessagingHosts

chrome_extension_id="dbdlmgpfijccjgnnpacnamgdfmljoeee"
# the id 'dbdlmgpfijccjgnnpacnamgdfmljoeee' belongs to following ext in chrome web store
# https://chrome.google.com/webstore/detail/citrix-workspace-continui/dbdlmgpfijccjgnnpacnamgdfmljoeee
edge_extension_id="pmdpflpcmcomdkocbehamllbfkdgnalf"
# the id 'pmdpflpcmcomdkocbehamllbfkdgnalf' belongs to following ext in MS edge web store
# https://microsoftedge.microsoft.com/addons/detail/citrix-workspace-web-exte/pmdpflpcmcomdkocbehamllbfkdgnalf

manifest_file_name="com.citrix.workspace.native"

# For FIDO2
ceb_native_messaging_host_file_name="com.citrix.chrome.ipcbridge"
ceb_native_messaging_host_file_dir="/etc/chromium/native-messaging-hosts"
ceb_extension_id="gmbmklehihfgponfmjbodllmcacgmaoi"

create_ceb_native_messaging_host_file(){
      
    ceb_native_messaging_host_file_path="$1"
    echo "{" > $ceb_native_messaging_host_file_path 
    echo "  \"name\": \"$ceb_native_messaging_host_file_name\"," >> "$ceb_native_messaging_host_file_path"
    echo "  \"description\": \"Native Messaging Host for IPC\"," >> "$ceb_native_messaging_host_file_path"
    echo "  \"path\": \"$2/ceb/CtxChromiumBrowser/native_bridge\"," >> "$ceb_native_messaging_host_file_path"
    echo "  \"type\": \"stdio\"," >> "$ceb_native_messaging_host_file_path" 
    echo "  \"allowed_origins\": [ \"chrome-extension://$ceb_extension_id/\" ]" >> "$ceb_native_messaging_host_file_path"
    echo "}" >> "$ceb_native_messaging_host_file_path" 
    return
}

create_manifest_file(){

    manifest_file_path="$1"
    echo "{" > $manifest_file_path 
    echo "  \"name\": \"$manifest_file_name\"," >> "$manifest_file_path"
    echo "  \"description\": \"Launch NMH\"," >> "$manifest_file_path"
    echo "  \"path\": \"$2/NativeMessagingHost\"," >> "$manifest_file_path"
    echo "  \"type\": \"stdio\"," >> "$manifest_file_path" 
    echo "  \"allowed_origins\": [ \"chrome-extension://$chrome_extension_id/\", \"chrome-extension://$edge_extension_id/\" ]" >> "$manifest_file_path"
    echo "}" >> "$manifest_file_path" 
    return
    
}

install_browser_extension() {
  	installer=`whoami`
    if [ "$installer" = root ]
    then
        for user_home in `awk -F: '{print $6}' /etc/passwd | grep '/home'`
        do
        	install_extension $user_home $1
        done
    fi
    return
}

install_extension(){
  #For FIDO2
  if [ ! -d $ceb_native_messaging_host_file_dir ]
  then
    mkdir -p "$ceb_native_messaging_host_file_dir"
  fi    
    ceb_native_messaging_host_file_path="$ceb_native_messaging_host_file_dir/$ceb_native_messaging_host_file_name.json"
  if [ -f $ceb_native_messaging_host_file_path ]
  then
    rm $ceb_native_messaging_host_file_path
  fi
  create_ceb_native_messaging_host_file "$ceb_native_messaging_host_file_path" $2

	if which "google-chrome" 
  	then
  		chrome_manifest_file_dir="$1/.config/google-chrome/NativeMessagingHosts"
        if [ ! -d $chrome_manifest_file_dir ]
        then
          mkdir -p "$chrome_manifest_file_dir"
        fi
    
        chrome_manifest_file_path="$chrome_manifest_file_dir/$manifest_file_name.json"
        if [ -f $chrome_manifest_file_path ]
        then
          rm $chrome_manifest_file_path
        fi
        create_manifest_file "$chrome_manifest_file_path" $2
  	fi
  if which "microsoft-edge-beta" 
  then
  	  edge_manifest_file_dir="$1/.config/microsoft-edge-beta/NativeMessagingHosts"
	  
      if [ ! -d $edge_manifest_file_dir ]
      then
        mkdir -p "$edge_manifest_file_dir"
      fi
      
      edge_manifest_file_path="$edge_manifest_file_dir/$manifest_file_name.json"
      if [ -f $edge_manifest_file_path ]
      then
        rm $edge_manifest_file_path
      fi
      create_manifest_file "$edge_manifest_file_path" $2
  fi
  return
}

uninstall_browser_extension(){
	installer=`whoami`
    if [ "$installer" = root ]
    then
        for user_home in `awk -F: '{print $6}' /etc/passwd | grep '/home'`
        do
        	uninstall_extension $user_home
        done
    fi
    return
}

uninstall_extension(){
  #For FIDO2
  ceb_native_messaging_host_file_path="$ceb_native_messaging_host_file_dir/$ceb_native_messaging_host_file_name.json"
  if [ -f $ceb_native_messaging_host_file_path ]
  then
    rm $ceb_native_messaging_host_file_path
  fi

  chrome_manifest_file_dir="$1/.config/google-chrome/NativeMessagingHosts"
  chrome_manifest_file_path="$chrome_manifest_file_dir/$manifest_file_name.json"
  if [ -f $chrome_manifest_file_path ]
  then
    rm $chrome_manifest_file_path
  fi
  
  edge_manifest_file_dir="$1/.config/microsoft-edge-beta/NativeMessagingHosts"
  edge_manifest_file_path="$edge_manifest_file_dir/$manifest_file_name.json"
  if [ -f $edge_manifest_file_path ]
  then
    rm $edge_manifest_file_path
  fi
  return
}
#!/bin/sh
###############################################################################
#
#       Name:           setup-libsecret.sh
#
#       $Id:
#
#       Purpose:        Handled deletion of aml tokens during unintallation
#
#       Copyright 2020 Citrix Systems, Inc. All rights reserved.
#
###############################################################################


###########################################################################
#
#	LibSecret_disintegrate()
#
#	This function is called to disintegrate the client with LibSecret.
#
###########################################################################

LibSecret_disintegrate()
{
    STOREBROWSE_AMLCLEAN_CMD="$ICAInstDir/util/storebrowse -K"
    if [ `whoami` = "root" ]
    then
    # Remove all tokens from libsecret using storebrowse -K command for logged in user
        runuser -l  `echo $USER` -c "$STOREBROWSE_AMLCLEAN_CMD 2>/dev/null"
    else
    # Remove all tokens from libsecret using storebrowse -K command        
        $STOREBROWSE_AMLCLEAN_CMD 2>/dev/null

    fi
    return 0
}
#!/bin/sh
###############################################################################
#
#       Name:           setup-mime.sh
#
#       $Id$
#
#       Purpose:        Registers MIME types and handlers using xdg-xxx
#                       utility programs.
#
#       Copyright 2014-2017 Citrix Systems, Inc. All rights reserved.
#
###############################################################################

# Ubuntu's xdg-mime allows mv to prompt when replacing
# the default application file, mimeapps.list.  Pipe in some affirmation!

Do_MIME_default()
{
    xdg-mime default $* 2>/dev/null <<EOF
y
y
y
EOF
}

DT_register_MIMEs()
{
    # We have a wfica.desktop, make it the default ICA file handler
    # for application/x-ica, and similarly for the new_store script.

    xdg-icon-resource install --size 64 "$ICAInstDir/icons/000_Receiver_64.png" Citrix-Receiver 2>/dev/null
    xdg-mime install "$DT_FILE_DIR/Citrix-mime_types.xml" 2>/dev/null
    Do_MIME_default wfica.desktop application/x-ica
    Do_MIME_default new_store.desktop application/vnd.citrix.receiver.configu\
re
	Do_MIME_default receiver.desktop x-scheme-handler/receiver
	Do_MIME_default receiver_fido2.desktop x-scheme-handler/citrixauthwebviewdone
	Do_MIME_default fido2_llt.desktop.desktop x-scheme-handler/ctxlinuxamloauth
#    xdg-mime default wfica.desktop application/x-ica 2>/dev/null
#    xdg-mime default new_store.desktop application/vnd.citrix.receiver.configure 2>/dev/null
    
    # For a root install "xdg-mime default" does not work on Debian
    #  and derivatives.  Belt and braces!

    if [ -n "$SU_INSTALL" ]
    then
	MDIR=/etc/xdg
	MFILE=/etc/xdg/mimeapps.list

	[ ! -d $MDIR ] && mkdir -p $MDIR && chmod 0644 $MDIR
	if [ ! -e $MFILE ]
        then
            echo [Default Applications] > $MFILE
	    chmod 0644 $MFILE
	fi
	if grep -q x-scheme-handler/receiver $MFILE
	then
	    # Already present, do nothing.
	    true
	else
	    echo 'application/x-ica=wfica.desktop;' >> $MFILE
	    echo 'application/vnd.citrix.receiver.configure=new_store.desktop;' >> $MFILE
		echo 'x-scheme-handler/receiver=receiver.desktop;' >> $MFILE
		echo 'x-scheme-handler/citrixauthwebviewdone=receiver_fido2.desktop;' >> $MFILE
		echo 'x-scheme-handler/ctxlinuxamloauth=fido2_llt.desktop;' >> $MFILE
	fi

	# More braces

	update-desktop-database -q 2>/dev/null
    fi
}
#!/bin/sh
###############################################################################
#
#	Name:		util_tr.sh
#
#
#	Created:	May 1995
#
#	Version:	$Id$
#
#	Purpose:	Provides functions to translate character strings
#			to upper and lower case.  This is done using functions
#			because 'tr' is not portable, and cannot always
#			be guaranteed to be available.
#
#			This file contains versions using 'tr' in its
#			simplest form.
#
#	Copyright 1996-1998 Citrix Systems, Inc. All rights reserved.
#
###############################################################################


###############################################################################
#
#  toupper() copies the standard input to the output, translating any lower
#  case letters to upper case.
#
###############################################################################
toupper(){
	<&0 tr $lowercase_letters $uppercase_letters >&1
}

###############################################################################
#
#  tolower() copies the standard input to the output, translating any upper
#  case letters to lower case.
#
###############################################################################
tolower(){
	<&0 tr $uppercase_letters $lowercase_letters >&1
}

#!/bin/sh
###############################################################################
#
#	Name:		map.sh
#
#	Version:	$Id$
#
#	Purpose:	To map a file name from its original name to whatever
#			will be present on the CD ROM.  A file name may start
#			out as something like "install", but when put on a 
#			CD ROM and mounted, it may be changed to upper case,
#			may have a dot at the end, and may have a semi colon 1
#			at the end, so ending up as INSTALL, install., 
#			install.;1, or INSTALL.;1
#
#			To use these functions, you should include this source
#			in your shell script by concatenating it with your
#			source at build time, i.e. 
#				cat map.sh myscript.sh > myscript
#
#			To use the functions, you should first call get_tr_key
#			to set up the mapping keys, and then call tr_file for
#			each file you want to access.
#			E.g., 
#				get_tr_key $0
#				tr_file $TR_FILE_KEY /dir1/dir2/filename
#
#			The file name that is passed to get_tr_key should not
#			have any extensions in it. It should also have mixed
#			case.
#
#	Copyright 1996-1998, 2008-2009  Citrix Systems, Inc. All rights reserved.
#
###############################################################################


###############################################################################
#
#	get_tr_key <path/file>
#
#	Examines the given filename, and generates a key suitable for use in
#	tr_file().  The key is returned in $TR_FILE_KEY and $TR_DIR_KEY.
#	TR_FILE_KEY is to be used when mapping a file name, TR_DIR_KEY when
#	mapping a directory name only. TR_DIR_KEY always has the dot and semi
#	colon parts set to NN.
#
#	The input filename is in 8.3 format, but the given filename must not
#	have any extension, otherwise we cannot tell whether a dot is needed
#	at the end in the no-extension case.  There may be a path, but this
#	is ignored in the generation of the key. 
#	The given filename must be in mixed case to allow distinction between
#	upper case, lower case and mixed case filesystems.
#
#	The key generated consists of three characters:
#       Character       File system has
#       ---------       ---------------
#       L|U|M           Lower, Upper or Mixed case
#       D|N             Dot at end of a file with no extension or No dot.
#       S|N             Semicolon-one at end of filename or No semicolon-one.
#
###############################################################################

get_tr_key(){
	trkey_filename=`basename "$1"`
	TR_FILE_KEY=""
	TR_DIR_KEY=""

	# See if it's in upper or lower case.
	trkey_lower_filename=`echo $trkey_filename | tolower`
	if [ "$trkey_lower_filename" = "$trkey_filename" ]
	then
		# It is lower case
		TR_FILE_KEY=L
	else
		trkey_upper_filename=`echo $trkey_filename | toupper`
		if [ "$trkey_upper_filename" = "$trkey_filename" ]
		then
			# It is upper case
			TR_FILE_KEY=U
		else
			# It must be mixed case.
			TR_FILE_KEY=M
		fi
	fi

	# Only the case mapping is relevant for a directory, so set up the
	# directory mapping key now using the file case mapping, and NN for
	# dots and semi colons.
	TR_DIR_KEY=${TR_FILE_KEY}NN

	# See if it has a dot.
	echo $trkey_filename | grep '\.' > /dev/null 2>&1
	if [ "$?" = "0" ]
	then
		# Input filename has a dot
		TR_FILE_KEY=${TR_FILE_KEY}D
	else
		# No dot
		TR_FILE_KEY=${TR_FILE_KEY}N
	fi

	# See if it has a semi-colon-one
	echo $trkey_filename | grep ';1' > /dev/null 2>&1
	if [ "$?" = "0" ]
	then
		# Input filename has a semi-colon-one
		TR_FILE_KEY=${TR_FILE_KEY}S
	else
		# No semi-colon-one
		TR_FILE_KEY=${TR_FILE_KEY}N
	fi
}

###############################################################################
#
#	tr_file key <path/file>
#
#	Translates a filename to correspond to the format indicated.
#
#	The first parameter is the format to be used, and is a sequence of
#	three letters as follows:
#	U, L or M - Upper case, Lower case or Mixed case
#	D or N - Dot or No Dot (if there is no ".3" type suffix)
#	S or N - Semi-colon-One (;1) or Not
#	
#	The second parameter is the filename, and may have a leading path,
#	which will be converted to upper or lower case as necessary.
#	The filename is assumed to be in 8.3 format.
#
#	The translated filename is returned in $TR_FILE
#
###############################################################################

tr_file(){
	# Extract 1st, 2nd and 3rd characters.
	# sed is more generally available than cut.
	U_or_L=`echo $1 | sed -e 's/^\(.\).*/\1/'`
	D_or_N=`echo $1 | sed -e 's/^.\(.\).*/\1/'`
	S_or_N=`echo $1 | sed -e 's/^..\(.\).*/\1/'`

	TR_FILE=$2

	case $U_or_L in
	U)
		# Translate to upper case.
		TR_FILE=`echo $TR_FILE | toupper`
		;;
	M)
		# Mixed Case - do no translation
		;;
	*)
		# Translate to lower case (default).
		TR_FILE=`echo $TR_FILE | tolower`
		;;
	esac

	case $D_or_N in
	D)
		# Add a dot at the end, but only if there isn't already one.
		( basename $TR_FILE | grep '\.' > /dev/null ) || TR_FILE=${TR_FILE}'.'
		;;
	*)
		# Do not add a dot
		;;
	esac

	case $S_or_N in
	S)
		# Add a semi-colon-one (;1) at the end.
		TR_FILE=${TR_FILE}\;1
		;;
	*)
		# Do not add a semi-colon-one
		;;
	esac
}

###############################################################################
#
#	setCDidfile CDROOT
#
#	Tries to find a product ID file (pkgid) file in the given directory.
#
#	As a result of this, TR_FILE_KEY and TR_DIR_KEY will be set to values
#	suitable for the directory being examined.  This serves two purposes.
#	Firstly, it allows TR_XXXX_KEY to be set, with no previous information,
#	and secondly, it provides a translated filename (CDidfile) which contains
#	package and version information strings.
#
###############################################################################

setCDidfile()
{
	trialDIR=$1
	tr_file_key_list="UDS UDN UNS UNN LDS LDN LNS LNN MDS MDN MNS MNN"
	

	for key in $tr_file_key_list
	do
		tr_file $key "PkgId"
		if [ -r "$trialDIR/$TR_FILE" ]
		then
			# Make sure both file and directory keys are set.
			get_tr_key "$trialDIR/$TR_FILE"
			CDidfile=$trialDIR/$TR_FILE
			return
		fi
	done
	CDidfile=""
}

#!/bin/sh
###############################################################################
#
#	Name:		locale.sh
#
#	$Id$
#
#	Purpose:	Use the locale to apply the correct message file.
#
#	Copyright 2009-2015 Citrix Systems, Inc. All rights reserved.
#
###############################################################################

find_tr_file()
{
    if [ -n "$TR_FILE" -a -f "$TR_FILE" ]
    then
        if [ -n "$VERBOSE_FIND" ]
        then
            echo Found "$TR_FILE"
        fi
        true
        return
    fi
    false
}

###############################################################################
#
#       find_locale_file
#
#	Finds the file given as a parameter, by searching under the 
#       directory given as a parameter for the best fit to the LANG variable.
#       The file found is returned as the value of TR_FILE.
#
###############################################################################
find_locale_file()
{
    FIND_TOP_DIR=$1
    TARGET=$2

    if [ -z "$TR_DIR_KEY" ] 
    then
        echo TR_DIR_KEY is empty
        exit 1
    fi

    # Find components
    if [ -n "$VERBOSE_FIND" ]
    then
        echo LANG=$LANG
    fi
    if [ -z "$LANG" ]
    then
        LANG="en"
        # echo LANG missing, using en
    fi
    Lang=`echo $LANG | sed -n -e 's/[_\.].*//' -e 'p'`
    Territory=`echo $LANG | sed -n -e 's/\(.*\)\..*/\1/' -e 's/.*_\(.*\)/\1/p'`
    CodeSet=`echo $LANG | sed -n -e 's/.*\.\(.*\)/\1/p'`
    # echo Lang = $Lang
    # echo Territory = $Territory
    # echo CodeSet = $CodeSet
    if [ "$Lang" = "POSIX" -o "$Lang" = "C" ]
    then
        Lang="en"
        # echo new Lang = $Lang
    fi
    if [ "$CodeSet" = "utf8" -o "$CodeSet" = "UTF8" -o "$CodeSet" = "utf-8" ]
    then
        CodeSet="UTF-8"
        # echo new CodeSet = $CodeSet
    fi
    if [ -n "$CodeSet" ]
    then
        if [ -n "$Territory" ]
        then
            # echo Try ${Lang}/$Territory/$CodeSet
            tr_file "$TR_DIR_KEY" "$FIND_TOP_DIR/nls/${Lang}_$Territory.$CodeSet/$TARGET"
            if find_tr_file
            then
                return
            fi
        fi
        # echo Try ${Lang}.$CodeSet
        tr_file "$TR_DIR_KEY" "$FIND_TOP_DIR/nls/${Lang}.$CodeSet/$TARGET"
        if find_tr_file
        then
            return
        fi
    fi
    if [ -n "$Territory" ]
    then
        # echo Try ${Lang}_$Territory
        tr_file "$TR_DIR_KEY" "$FIND_TOP_DIR/nls/${Lang}_$Territory/$TARGET"
        if find_tr_file
        then
            return
        fi
    fi
    # echo Try $Lang
    tr_file "$TR_DIR_KEY" "$FIND_TOP_DIR/nls/${Lang}/$TARGET"
    if find_tr_file
    then
        return
    fi

    if [ "$CodeSet" != "utf8" -a "$CodeSet" != "UTF8" -a "$CodeSet" != "utf-8" ]
    then
        # echo Try forcing UTF-8
        if [ -n "$Territory" ]
        then
            # echo Try ${Lang}_$Territory.UTF-8
            tr_file "$TR_DIR_KEY" "$FIND_TOP_DIR/nls/${Lang}_$Territory.UTF-8/$TARGET"
            if find_tr_file
            then
                return
            fi
        fi
        # echo Try ${Lang}.UTF-8
        tr_file "$TR_DIR_KEY" "$FIND_TOP_DIR/nls/${Lang}.UTF-8/$TARGET"
        if find_tr_file
        then
            return
        fi
    fi

    if [ -z "$FIND_TARGET_MAY_BE_ABSENT" ]
    then
        echo No target "$TARGET" found under "$FIND_TOP_DIR" for $LANG 
        if [ ! "$Lang" = "en" ]
        then
            echo Trying English...
        fi
    fi
    if [ "$Lang" = "en" ]
    then
        false
        return
    fi
    if [ -n "$CodeSet" ]
    then
        # echo Try en.$CodeSet
        tr_file "$TR_DIR_KEY" "$FIND_TOP_DIR/nls/en.$CodeSet/$TARGET"
        if find_tr_file
        then
            return
        fi
    fi
    # echo Try en
    tr_file "$TR_DIR_KEY" "$FIND_TOP_DIR/nls/en/$TARGET"
    if find_tr_file
    then
        return
    fi
    if [ -z "$FIND_TARGET_MAY_BE_ABSENT" ]
    then
        echo Could not find "$TARGET" under "$FIND_TOP_DIR" for en
    fi
    echo

    false
}

###############################################################################
#
#       load_locale_messages
#
#	Loads the .msg file given as a parameter, by searching under the 
#       directory given as a parameter for the best fit to the LANG variable.
#
###############################################################################
load_locale_messages()
{
    LOAD_TOP_DIR=$1
    TARGET=$2

    if find_locale_file "$LOAD_TOP_DIR" "$TARGET"
    then
        . "$TR_FILE"
        true
        return
    else
        false
        return
    fi
}

#!/bin/sh
###############################################################################
#
#	Name: locale_links.sh
#
#	$Id$
#
#	Purpose: Install symbolic links based on the installation locale.
#
#	Copyright 2008-2016 Citrix Systems, Inc.  All Rights Reserved.
#
###############################################################################

# Make symbolic links in the installation tree, based on the current
# (installation time) locale.
# Current directory MUST be $INST_DIR.

make_locale_links()
{
    tr_file $TR_FILE_KEY eula.txt
    find_locale_file . $TR_FILE
    chmod 444 "$TR_FILE"
    ln -sf $TR_FILE "$INST_DIR/eula.txt"

    # Similarly, links for ini files go into the config directory

    cd config
    tr_file $TR_FILE_KEY module.ini
    find_locale_file .. $TR_FILE
    chmod 444 "$TR_FILE"
    ln -sf $TR_FILE "$INST_DIR/config/module.ini"

    tr_file $TR_FILE_KEY wfclient.template
    find_locale_file .. $TR_FILE
    chmod 444 "$TR_FILE"
    ln -sf $TR_FILE "$INST_DIR/config/wfclient.template"

    tr_file $TR_FILE_KEY appsrv.template
    find_locale_file .. $TR_FILE
    chmod 444 "$TR_FILE"
    ln -sf $TR_FILE "$INST_DIR/config/appsrv.template"

    tr_file $TR_FILE_KEY debug.ini
    FIND_TARGET_MAY_BE_ABSENT=yes
    if find_locale_file .. $TR_FILE
    then
        chmod 444 "$TR_FILE"
        ln -sf $TR_FILE "$INST_DIR/config/debug.ini"
        ln -sf "$INST_DIR/config/debug.ini" "$INST_DIR/debug.ini"
    fi
    FIND_TARGET_MAY_BE_ABSENT=

    # and a link for index.htm goes into the help directory

    cd ../help
    tr_file $TR_FILE_KEY index.htm
    find_locale_file .. $TR_FILE
    chmod 444 "$TR_FILE"
    ln -sf $TR_FILE "$INST_DIR/help/index.htm"

    cd ..

}

#!/bin/sh
###############################################################################
#
#  Name:		integ_ica.sh
#
#  $Id$
#
#  Purpose:	To integrate the ICA client with browsers and
#		desktop environments.
#
#  Copyright 2012-2014 Citrix Systems, Inc. All rights reserved.
#
###############################################################################

Installer=""
IcaDir=.ICAClient
CustomDimensionsFile=.customdimensions_sent
chrome_extension_id="dbdlmgpfijccjgnnpacnamgdfmljoeee"
# the id 'dbdlmgpfijccjgnnpacnamgdfmljoeee' belongs to following ext in chrome web store
# https://chrome.google.com/webstore/detail/citrix-workspace-continui/dbdlmgpfijccjgnnpacnamgdfmljoeee
edge_extension_id="pmdpflpcmcomdkocbehamllbfkdgnalf"
# the id 'pmdpflpcmcomdkocbehamllbfkdgnalf' belongs to following ext in MS edge web store
# https://microsoftedge.microsoft.com/addons/detail/citrix-workspace-web-exte/pmdpflpcmcomdkocbehamllbfkdgnalf

###############################################################################
#
#	delete_debug_symlink()
#		Removes debug.ini symbolic link in $ICAROOT directory

###############################################################################

delete_debug_symlink() {
	if [ -L "$ICAInstDir/debug.ini" ]
	then
		rm -rf $ICAInstDir/debug.ini
	fi
}

###############################################################################
#
#	delete_service_worker_cache()
#		Removes service worker cache/scripts left from last installation

###############################################################################

delete_service_worker_cache() {
	UserHome="$1"
	if [ -d "$UserHome/.cache/selfservice" ]
	then
		rm -rf $UserHome/.cache/selfservice
	fi

	if [ -d "$UserHome/.local/share/webkitgtk/localstorage" ]
	then
		rm -rf $UserHome/.local/share/webkitgtk/localstorage/*
	fi

	if [ -d "$UserHome/.local/share/webkitgtk/serviceworkers" ]
	then
		rm -rf $UserHome/.local/share/webkitgtk/serviceworkers/*
	fi
}

###############################################################################
#	delete_customdimensionssent_from_ICAHome()
#
#	Handles the removal of .customdimensions_sent file after uninstalling the
#   current workspace installation, so that when the user re-installs or upgrades,
#   the custom dimensions are again sent as the app version would have changed.
#
###############################################################################
delete_customdimensionssent_from_ICAHome()
{
	UserHome="$1"
    if [ "X${UserHome}" != "X" ]; then
            if [ -d "$UserHome/$IcaDir" ]; then
                    IcaHome="$UserHome/$IcaDir"
                    	if [ "X${IcaHome}" != "X" ]; then
					            if [ -f "$IcaHome/$CustomDimensionsFile" ]; then
					                    rm -rf "$IcaHome/$CustomDimensionsFile"

					                    if [ $? -ne 0 ]; then
					                    	user_echo "Failed to remove : $IcaHome/$CustomDimensionsFile"
					                    else
					                    	user_echo "Customdimensions_sent cleanup complete from : `dirname $IcaHome`"
					                	fi
					            fi
					    else
					            user_echo "Paramerter \$IcaHome is not provided"
					    fi
			fi
    else
            user_echo "Please provide the user \"$HOME\" as first argument"
    fi
}
###############################################################################
#   delete_InvalidConnectionLease()
#
#   Delete the invalid Connection leases when install or uninstall the CWAL, 
#   to avoid keeping the redundant leases in profile directory. 
#   If all the leases under the user folder are invalid(older than 30 days), 
#   go on removing the directory.
#	when the second parameter is set to 1, all the lease files will be removed anyway.

###############################################################################
delete_InvalidConnectionLease()
{
	userhome="$1"
	deleteAll=0
	if [ "x$2" = "x1" ]
	then
		deleteAll=1
	fi
	Connectionleasepath="$userhome/.ICAClient/cache/ConnectionLease"
	if [ -d "$Connectionleasepath" ];
	then
		leasefolder=$(find $Connectionleasepath -mindepth 3 -maxdepth 3 -type d | grep -e /leases)
		for eachleasefolder in $leasefolder
			do
				countAllfiles=$(find $eachleasefolder -type f | wc -l)
				countInvalidfiles=$countAllfiles
				if [ "$deleteAll" = "0" ]
				then
					countInvalidfiles=$(find $eachleasefolder -type f -mtime +30 | wc -l)
				fi
				if [ "$countAllfiles" = "$countInvalidfiles" ];
				then
					storedir=$(basename ${eachleasefolder%/*/*})
					if [ -d $Connectionleasepath/$storedir ];
					then
						rm -r $Connectionleasepath/$storedir
					fi
				fi
			done
	fi
}
check_InvalidConnectionLease(){
	deleteAll="$1"
	installer=`whoami`
    if [ "$installer" = root ]
    then
        for user_home in `awk -F: '{print $6}' /etc/passwd`
        do
        	delete_InvalidConnectionLease $user_home $deleteAll
        done
    else
        # a normal user, delete invalid leases of its own user dir
        user_home=`getent passwd "$installer" | awk -F: '{print $6}'`
        delete_InvalidConnectionLease $user_home $deleteAll
    fi
}

delete_leaselaunch_cache(){
	installer=`whoami`
	leaselaunch_path="/.ICAClient/cache/Stores/"
	leaselaunch_file_prefix="LeaseLaunch_Cache_"
	user_homes=""
    if [ "$installer" = root ]
    then
		user_homes=$(awk -F: '{print $6}' /etc/passwd)
    else
        # a normal user
        user_homes=`getent passwd "$installer" | awk -F: '{print $6}'`
    fi

	for user_home in ${user_homes}
	do
		leaselaunch_full_path="$user_home""$leaselaunch_path"
		if [ -d "${leaselaunch_full_path}" ]
		then
			for file_to_be_deleted in `find ${leaselaunch_full_path} | grep -e ${leaselaunch_file_prefix} 2>/dev/null`
			do
				if [ -f "$file_to_be_deleted" ]
				then
					rm -f "$file_to_be_deleted"
				fi
			done
		fi
	done
	
}


###############################################################################
#   delete_shieldkid()
#
#   Delete the shieldkid file when install or uninstall the CWAL, so make the
#   shield device key maintained properly and shield working as expected.
###############################################################################
delete_shieldkid()
{
    installer=`whoami`
    if [ "$installer" = root ]
    then
        # installer is root, delete ~/.ICAClient/ShieldKid for all users
        for user_home in `awk -F: '{print $6}' /etc/passwd`
        do
            file_to_be_deleted="$user_home/.ICAClient/ShieldKid"
            if [ -f "$file_to_be_deleted" ]
            then
                rm -f "$file_to_be_deleted"
            fi
        done

        # the original user may not appear in passwd, a domain user, e.g.
        # comment this because some distro don't have logname utility
        # so the domain user would become a corner case/known issue
        #original_user=`logname`
        #user_home=`getent passwd "$original_user" | awk -F: '{print $6}'`
        #file_to_be_deleted="$user_home/.ICAClient/ShieldKid"
        #if [ -f "$file_to_be_deleted" ]
        #then
        #    rm -f "$file_to_be_deleted"
        #fi
    else
        # a normal user, delete ShieldKid of itself
        user_home=`getent passwd "$installer" | awk -F: '{print $6}'`
        file_to_be_deleted="$user_home/.ICAClient/ShieldKid"
        if [ -f "$file_to_be_deleted" ]
        then
            rm -f "$file_to_be_deleted"
        fi
    fi
}

integrate_ICA_client() {
	# Integrate with netscape
	netscape_integrate

	# integrate with CDE
	case $PORT in
		dec|ibm|hp|sco|solaris|solx86)
			if [ -n "$SU_INSTALL" ]
			then
				CDE_integrate
			fi
		;;
	esac

	# integrate with GNOME/KDE
	case $PORT in
		linuxx86|linuxx64|linuxarm64|linuxarmhf|solaris|solx86)

			# Always create .desktop files.

			DT_FILE_DIR="${ICAInstDir}"/desktop
			DT_create_files
			DT_integrate
		;;
	esac

	# integrate with GStreamer
	case $PORT in
		linuxx86|linuxx64|linuxarm64|linuxarmhf)
			GST_FILE_DIR="${ICAInstDir}"/util
			GST_integrate
	esac

	# Kill icasessonmgr if present
	killall -q -9 icasessionmgr

	#Remove the .customdimensions_sent file
	Installer=`whoami`

	if [ $Installer = "root" ]; then
		for UserHome in `cat /etc/passwd |awk -F: '{print $6}'`
			do
				delete_service_worker_cache $UserHome
			done
		#Remove the service worker caches when it's domain user
		if command -v sudo &> /dev/null
		then
			UserHome=$(sudo -H -u `echo $USER` sh -c 'echo $HOME')
			delete_service_worker_cache $UserHome
		fi
	else
		for UserHome in `getent passwd $Installer | cut -d: -f6`
			do
				delete_service_worker_cache $UserHome
			done
	fi

	# Admin is doing the installation
    if [ $Installer  = "root" ]; then
            for UserHome in `cat /etc/passwd |awk -F: '{print $6}'`
                    do
                            delete_customdimensionssent_from_ICAHome $UserHome
                    done
    else
            # This is only for user installation and he should control his own directory
            delete_customdimensionssent_from_ICAHome  "$HOME"
    fi

	# Install app protection files
	if [ $Installer  = "root" ]; then
		if [ "$NON_INTERACTIVE" = false ] && [ "$PORT" != "linuxx86" ]; then
	    	user_prompt "$instsentry2"
	    	getyesno $INSTALLER_NO
	    	if [ "$ANSWER" = "$INSTALLER_YES" ]; then
				remove_sentrybay
				install -m  555 "$CDSourceDir/${PORT}/${PORT}.cor/util/AppProtection-service.tar.gz" "$ICAInstDir/util"
				install_AppProtection "$ICAInstDir"
	    	else
  				user_echo "$instsentry3"
	    	fi
		else
			if [ "$install_app_protection" = "yes" ]; then
				remove_sentrybay
				install_AppProtection "$ICAInstDir"
			fi
		fi

    else
        user_echo "$instsentry1"
    fi
	if [ $Installer  = "root" ]; then
		install_browser_extension "$ICAInstDir" 
	fi
	check_InvalidConnectionLease 1
    delete_shieldkid
	delete_leaselaunch_cache

	# Install multi-touch udev rules
	install_mtch "$ICAInstDir"

	# Install hid udev rules
	install_hid "$ICAInstDir"
}



disintegrate_ICA_client() {
	# disintegrate from netscape
	netscape_disintegrate
	delete_debug_symlink

	# disintegrate from CDE
	case $PORT in
		dec|ibm|hp|sco|solaris|solx86)
			if [ -n "$SU_INSTALL" ]
			then
				CDE_disintegrate
			fi
		;;
	esac

	# Disintegrate with GNOME/KDE.
	case $PORT in
		linuxx86|linuxx64|linuxarm64|linuxarmhf|solaris|solx86)
			DT_FILE_DIR="${ICAInstDir}"/desktop
			DT_remove_files
			DT_disintegrate
		;;
	esac

	# Disintegrate with GStreamer
	case $PORT in
		linuxx86|linuxx64|linuxarm64|linuxarmhf)
			GST_disintegrate
	esac

	# Disintegrate with LibSecret
	case $PORT in
		linuxx86|linuxx64|linuxarm64|linuxarmhf)
			LibSecret_disintegrate
	esac

	# Remove app protection files
	# if [ "$NON_INTERACTIVE" = false ]; then
		remove_AppProtection "$ICAInstDir"
		remove_sentrybay
	# fi
	uninstall_browser_extension
	# removed all lease folders when uninstalling cwa
	check_InvalidConnectionLease 1
    delete_shieldkid
	delete_leaselaunch_cache

	# remove multi-touch udev rules file
	remove_mtch

	# remove hid udev rules file
	remove_hid
}
#!/bin/sh
###############################################################################
#
#       Name:           icalicense.sh
#
#       Created:        26 May 2000
#
#       Version:        $Id$
#
#       Purpose:        Create /etc/icalicense/clientlicense for RPM install.
#
#       Copyright 2000-2011 Citrix Systems, Inc. All rights reserved.
#
###############################################################################

if test ! -d /etc/icalicense
then
    mkdir /etc/icalicense
fi
if test ! -f /etc/icalicense/clientlicense
then
    "$ICAROOT/util/echo_cmd" -l >/etc/icalicense/clientlicense
fi
chmod 444 /etc/icalicense/clientlicense
chmod 555 /etc/icalicense
#!/bin/sh
###############################################################################
#
#	Name:		Version.sh
#
#	Version:	$Id$
#
#	Purpose:	Sets shell variables for the version of Client
#			to be installed.
#
#	Copyright 1996-2001 Citrix Systems, Inc. All rights reserved.
#
###############################################################################


# Use lots of minor numbers, because a grep for 2.0 would pick up 2.0.1
ID_VERSION=24.8.0.98
# This one is for display of the version number only.
DISP_VERSION=24.8.0
CHANGE_NO=98
ID_PRODUCT_NAME=CitrixICAClient

#!/bin/sh
###############################################################################
#
#	Name:		ctxcwalogd_installer.sh
#
#	Version:	$Id$
#
#	Purpose:	Install ctxlog daemon
#                
#
#	Copyright 2020 Citrix Systems, Inc. All rights reserved.
#
###############################################################################

###############################################################################
# find_available_uid
#
# Function to find an available UID less than 1000
###############################################################################
find_available_uid() 
{
    local uid=800  # Start from 800
    local getent_path=$(which getent)

    # If getent is available, use it to check for the UID
    if [ -n "$getent_path" ]; then
        while $getent_path passwd $uid >/dev/null; do
            uid=$((uid + 1))
            if [ $uid -ge 1000 ]; then
                echo "No available UID less than 1000 found for citrixlog user."
                exit 1
            fi
        done
    else
        # If getent is not available, use grep
        while grep -q "^.*:x:$uid:" /etc/passwd; do
            uid=$((uid + 1))
            if [ $uid -ge 1000 ]; then
                echo "No available UID less than 1000 found for citrixlog user."
                exit 1
            fi
        done
    fi

    echo $uid
}

###############################################################################
# install_ctxcwalogd
#
# Installs and starts ctxcwalogd service
###############################################################################
install_ctxcwalogd() 
{
    local MyLocation=$1/${PORT}/${PORT}.cor
    local ICAROOT=$2
    local PkgInstall=$3
    local ServiceName="ctxcwalogd.service"
    local ServicePath=""
    local SystemdSupport=$(which systemctl 2>/dev/null)

    # Remove existing citrixlog user if it exists
    if id -u citrixlog > /dev/null 2>&1; then
        userdel citrixlog
    fi

    if [ ! -z "${SU_INSTALL}" ] || [ ! -z ${PKG_INSTALL} ]; then
        # Create citrixlog user if not exists
        local citrixlog_uid
        if ! id -u citrixlog > /dev/null 2>&1; then
            citrixlog_uid=$(find_available_uid)
            useradd -u $citrixlog_uid -d /var/log/citrix -s /sbin/nologin -g nogroup citrixlog
            mkdir -p /var/log/citrix
            chown -R citrixlog:nogroup /var/log/citrix
        fi
    fi

    if [ ! -z $SystemdSupport ]; then
        local Suffix=""
        local USER=""
        local CitrixUser=""
        local ServicePath=""
        if [ -z ${PkgInstall} ]; then
            if [ -z "$SU_INSTALL" ]; then
                ServicePath="${HOME}/.config/systemd/user/"
                Suffix=" --user"
                USER="default"
                mkdir -p "${ServicePath}"
            else
                CitrixUser="User=citrixlog"
                USER="multi-user"
                ServicePath="/lib/systemd/system"
            fi

            sed -e "s,###ICAROOT###,$ICAROOT,g" \
                -e "s,###USER###,$USER,g" \
                -e "s,###CitrixUser###,$CitrixUser,g" \
                    <"${MyLocation}/${ServiceName}" >"${ServicePath}/${ServiceName}"
        fi
        systemctl${Suffix} daemon-reload
        systemctl${Suffix} enable ${ServiceName}
        systemctl${Suffix} restart ${ServiceName}
    elif [ ! -z "${SU_INSTALL}" ] && [ -d "/etc/init.d" ]; then
        if [ -z ${PkgInstall} ]; then
            sed -e "s,^DAEMON=.*,DAEMON=\"$ICAROOT/util/ctxcwalogd\",g" \
                <"${MyLocation}/ctxcwalogd.rc" >/etc/init.d/ctxcwalogd
            chmod 700 /etc/init.d/ctxcwalogd
        fi
        integrateDaemon /etc/init.d/ctxcwalogd
        # Start the daemon
        startDaemon /etc/init.d/ctxcwalogd
    fi
}

###############################################################################
# remove_ctxcwalogd
#
# Removes and stops ctxcwalogd service
###############################################################################
remove_ctxcwalogd_common() 
{
    local ServiceName="$1.service"
    local DaemonName="/etc/init.d/$1"
    local LogConfigFileName=".ctxcwalogconf"

    if [ ! -z "${SU_INSTALL}" ]; then
    	rm -f /var/log/citrix/${LogConfigFileName}
    fi

    local Suffix=""
    if [ ! -z "${SU_INSTALL}" ] && [ -f $DaemonName ]; then
        # Stop the daemon
        stopDaemon $DaemonName
        # Remove the daemon from init
        removeDaemon $DaemonName
        rm $DaemonName
    fi
    if [ -z "$SU_INSTALL" ]; then
        Suffix=" --user"
        ServicePath="${HOME}/.config/systemd/user/"
        rm -f ${HOME}/.ICAClient/${LogConfigFileName}
    else
        ServicePath="/lib/systemd/system"
    fi

    if [ -f "${ServicePath}/${ServiceName}" ]; then
        rm ${ServicePath}/${ServiceName}
        systemctl${Suffix} disable ${ServiceName} > /dev/null &2>1
        systemctl${Suffix} stop  ${ServiceName} > /dev/null &2>1
        systemctl${Suffix} daemon-reload
    fi
}

remove_ctxcwalogd() 
{
    if [ ! -f /opt/Citrix/VDA/bin/ctxlogd ]; then
        # If LVDA ctxlogd is NOT found, remove ctxlogd which is renamed as ctxcwalogd.
        # The LVDA check is to make sure we don't stop and remove LVDA ctxlogd.
        remove_ctxcwalogd_common ctxlogd
    fi
    remove_ctxcwalogd_common ctxcwalogd

    # Remove citrixlog user if it exists
    if id -u citrixlog > /dev/null 2>&1; then
        userdel citrixlog
    fi
}
###############################################################################
#
#	Name:		daemon_util.sh
#
#	$Id:$
#
#	Purpose:	Provides utilities for use with init daemons.
#
#	Copyright 2009-2017 Citrix Systems, Inc. All rights reserved.
#
###############################################################################


###############################################################################
#
#  integrateDaemon is responsible for integrating a daemon into the init run
#  levels. It achieves this by progressing through a list of possible methods
#  in the following order LSB (Linux Standard Base) functions, update-rc.d
#  (Debian style), chkconfig (Redhat style).
#
#  Input: full-daemon-path , i.e. /etc/init.d/ctxusbd
#
#  Return: 0 if a means of integrating could be found, otherwise 1.
#
###############################################################################
integrateDaemon()
{
	DAEMONPATH=$1
	DAEMONNAME=`basename "$DAEMONPATH"`

	## Install the daemon into the init run levels of the system

	# Use LSB install_initd if possible
	if INIT_INSTALL_CMD="/usr/lib/lsb/install_initd" ; test -x "$INIT_INSTALL_CMD" ; then
		# Use LSB install_initd function
		"$INIT_INSTALL_CMD" "$DAEMONPATH"

	# Use update-rc.d in PATH if possible
	elif INIT_INSTALL_CMD="`which update-rc.d 2>/dev/null`" ; test -x "$INIT_INSTALL_CMD" ; then
		# update-rc.d is used on Debian based distros
		"$INIT_INSTALL_CMD" "$DAEMONNAME" defaults > /dev/null 2>&1

	# Use static location for update-rc.d if possible
	elif INIT_INSTALL_CMD="/usr/sbin/update-rc.d" ; test -x "$INIT_INSTALL_CMD" ; then
		"$INIT_INSTALL_CMD" "$DAEMONNAME" defaults > /dev/null 2>&1

	# Use chkconfig in PATH if possible
	elif INIT_INSTALL_CMD="`which chkconfig 2>/dev/null`" ; test -x "$INIT_INSTALL_CMD" ; then
		# chkconfig is used on Redhat based distros
		"$INIT_INSTALL_CMD" --add "$DAEMONNAME"

	# Use static location for chkconfig if possible
	elif INIT_INSTALL_CMD="/sbin/chkconfig" ; test -x "$INIT_INSTALL_CMD" ; then
		# chkconfig is used on Redhat based distros
		"$INIT_INSTALL_CMD" --add "$DAEMONNAME"
	
	# No means of integrating the daemon into the init run levels was found.
	# This probably means the system is very cut down.
	else
		return 1
	fi

	# Magic for the Asus EeePc which uses a custom init system
	if [ -f /usr/sbin/services.sh ]; then
		grep "$DAEMONNAME" /etc/fastservices 2>&1 > /dev/null || echo "$DAEMONNAME" >> /etc/fastservices
	fi

	return 0
}


###############################################################################
#
#  removeDaemon is responsible for removing a daemon from the init run
#  levels. It achieves this by progressing through a list of possible methods
#  in the following order LSB (Linux Standard Base) functions, update-rc.d
#  (Debian style), chkconfig (Redhat style).
#
#  Input: full-daemon-path , i.e. /etc/init.d/ctxusbd
#
#  Return: 0 if a means of removal could be found, otherwise 1.
#
###############################################################################
removeDaemon()
{
	DAEMONPATH=$1
	DAEMONNAME=`basename "$DAEMONPATH"`

	## Remove the daemon from init levels

	# Use LSB remove_initd if possible
	if INIT_REMOVE_CMD="/usr/lib/lsb/remove_initd" ; test -x "$INIT_REMOVE_CMD" ; then
		# Use LSB remove_initd function
		"$INIT_REMOVE_CMD" "$DAEMONPATH" > /dev/null 2>&1

	# Use update-rc.d in PATH if possible
	elif INIT_REMOVE_CMD="`which update-rc.d 2>/dev/null`" ; test -x "$INIT_REMOVE_CMD" ; then
		# update-rc.d is used on Debian based distros
		"$INIT_REMOVE_CMD" -f "$DAEMONNAME" remove > /dev/null 2>&1

	# Use static location for update-rc.d if possible
	elif INIT_REMOVE_CMD="/usr/sbin/update-rc.d" ; test -x "$INIT_REMOVE_CMD" ; then
		"$INIT_REMOVE_CMD" -f "$DAEMONNAME" remove > /dev/null 2>&1

	# Use chkconfig in PATH if possible
	elif INIT_REMOVE_CMD="`which chkconfig 2>/dev/null`" ; test -x "$INIT_REMOVE_CMD" ; then
		# chkconfig is used on Redhat based distros
		if [ -f $DAEMONPATH ] ; then
			"$INIT_REMOVE_CMD" --del "$DAEMONNAME"
		fi

	# Use static location for chkconfig if possible
	elif INIT_REMOVE_CMD="/sbin/chkconfig" ; test -x "$INIT_REMOVE_CMD" ; then
		   # chkconfig is used on Redhat based distros
		if [ -f $DAEMONPATH ]  ; then
			"$INIT_REMOVE_CMD" --del "$DAEMONNAME"
		fi
	
	# Use systemctl if possible
	elif INIT_REMOVE_CMD="systemctl" ; command -v "$INIT_REMOVE_CMD" >/dev/null 2>&1 ; then
		if [ -f $DAEMONPATH ]  ; then
			systemctl disable "$DAEMONNAME"
			rm $DAEMONPATH
			systemctl daemon-reload
			systemctl reset-failed
		fi
	
	# No means of de-integrating daemon from the init run levels was found.
	# This probably means the system is very cut down.
	else
		return 1
	fi
	
	# Magic to remove the Asus EeePc daemon
    if [ -f /etc/fastservices ]; then
        sed -e "/^$DAEMONNAME$/d" -i /etc/fastservices
    fi
	
	return 0
}


###############################################################################
#
#  startDaemon is responsible for starting a init deamon.
#
#  Input: full-daemon-path , i.e. /etc/init.d/ctxusbd
#
#  Return: 0
#
###############################################################################
startDaemon()
{
	DAEMONPATH=$1
	DAEMONNAME=`basename "$DAEMONPATH"`

	## Start the daemon
	if [ -x "`which invoke-rc.d 2>/dev/null`" ]
	then
		invoke-rc.d "$DAEMONNAME" start
	else
		if [ -x "`which systemctl 2>/dev/null`" ]; then
			systemctl daemon-reload
		fi
		"$DAEMONPATH" start
	fi
}	


###############################################################################
#
#  stopDaemon is responsible for stopping a init deamon.
#
#  Input: full-daemon-path , i.e. /etc/init.d/ctxusbd
#
#  Return: 0
#
###############################################################################
stopDaemon()
{
	DAEMONPATH=$1
	DAEMONNAME=`basename "$DAEMONPATH"`

	## Stop the daemon
	if [ -x "$DAEMONPATH" ]
	then
		if [ -x "`which invoke-rc.d 2>/dev/null`" ]
		then
			invoke-rc.d $DAEMONNAME stop
		else
			$DAEMONPATH stop
		fi
	fi
}

#!/bin/sh
###############################################################################
#
#       Name:           setup-DT.sh
#
#       $Id$
#
#       Purpose:        Handles static part of integration of the client
#                       with KDE and Gnome.
#
#       Copyright 2003-2017 Citrix Systems, Inc. All rights reserved.
#
###############################################################################

CITRIX_BRANDNAME="Citrix Workspace"
DT_PROGRAMS="selfservice:wfica:new_store:configmgr:conncenter:receiver:receiver_fido2:fido2_llt:logmgr:sendfeedback"
conncenter_NAME=$CITRIX_BRANDNAME' (conncenter)'
conncenter_EXTRAS_1='NoDisplay=true'
configmgr_NAME=$CITRIX_BRANDNAME' (configmgr)'
configmgr_EXTRAS_1='NoDisplay=true'
logmgr_NAME=$CITRIX_BRANDNAME' (logmgr)'
logmgr_EXTRAS_1='NoDisplay=true'
sendfeedback_NAME=$CITRIX_BRANDNAME' (sendfeedback)'
sendfeedback_EXTRAS_1='NoDisplay=true'
selfservice_NAME=$CITRIX_BRANDNAME 
wfica_NAME=$CITRIX_BRANDNAME' Engine'
wfica_EXTRAS_1='StartupWMClass=Wfica'
wfica_EXTRAS_2='NoDisplay=true'
wfica_EXTRAS_3='MimeType=application/x-ica;'
wfica_EXEC='%f'
new_store_NAME='Add Citrix Store'
new_store_EXTRAS_1='NoDisplay=true'
new_store_EXTRAS_2='MimeType=application/vnd.citrix.receiver.configure;'
new_store_EXEC='%f'
receiver_NAME=$CITRIX_BRANDNAME' Launcher' 
receiver_EXTRAS_1='StartupWMClass=Wfica'
receiver_EXTRAS_2='NoDisplay=true'
receiver_EXTRAS_3='MimeType=x-scheme-handler/receiver'
receiver_EXEC='%u'
receiver_fido2_NAME=$CITRIX_BRANDNAME' Launcher' 
receiver_fido2_EXTRAS_1='StartupWMClass=Wfica'
receiver_fido2_EXTRAS_2='NoDisplay=true'
receiver_fido2_EXTRAS_3='MimeType=x-scheme-handler/citrixauthwebviewdone'
receiver_fido2_EXEC='%u'
fido2_llt_NAME=$CITRIX_BRANDNAME' Launcher' 
fido2_llt_EXTRAS_1='StartupWMClass=Wfica'
fido2_llt_EXTRAS_2='NoDisplay=true'
fido2_llt_EXTRAS_3='MimeType=x-scheme-handler/ctxlinuxamloauth'
fido2_llt_EXEC='%u'

XDG_TARGET_DIR="/usr/share/applications"
TL_MENU_DIR="Vendor_menus/Applications"                    # TurboLinux
TL_TOP_DIR="/usr/share/applnk/"$TL_MENU_DIR
DEFAULT_GNOME_TARGET_DIRS="/etc/opt/gnome/SuSE/Internet"
DEFAULT_KDE_TARGET_DIRS="/usr/share/applnk-redhat/Internet \
                         /usr/share/applnk-mdk/Networking \
                         $TL_TOP_DIR/Internet \
			 /etc/opt/kde3/share/applnk/SuSE/Internet \
                         /usr/share/applnk/Applications"

DEFAULT_DT_TARGET_DIRS="$XDG_TARGET_DIR $DEFAULT_GNOME_TARGET_DIRS $DEFAULT_KDE_TARGET_DIRS"
DEFAULT_DT_FOLDER="Applications"

# This per-user directory is used by all modern KDE and GNOME versions.
DT_SHARED_USER_DIR=$HOME/.local/share/applications

# Escape a given string so that it is in valid ".desktop" file syntax
DesktopEscapeString()
{
    string="$1"

    echo "$string" | sed -e "s%\\\\%\\\\\\\\%g" -e "s% %\\\\s%g"
}

# Escape a given command string so that it is in valid ".desktop" file syntax
DesktopEscapeStringCommand()
{
    string="$1"

    echo "$string" | sed -e "s%\\\\%\\\\\\\\%g" -e "s% %\\\\ %g"
}

# Output the value of a variable like wfica_EXTRAS_1, only if defined.
echo_if_exists()
{
if eval [ \"\${${i}_${1}}\" ]
then
  eval echo \${${i}_${1}}
fi
}

# Output a line like "Name=$wfica_NAME", but only if defined.
set_if_exists()
{
if eval [ \"\${${i}_${1}}\" ]
then
  eval echo ${2}=\${${i}_${1}}
fi
}

DT_create_files()
{
    if [ ! -d "$DT_FILE_DIR" ]
    then
	mkdir "$DT_FILE_DIR"
    fi

    DesktopEscapedICAInstDir=`DesktopEscapeString "$ICAInstDir"`
    DesktopEscapedICAInstDirCommand=`DesktopEscapeStringCommand "$ICAInstDir"`

    # Use a sub-shell and change $IFS for colon-separated list (see DT_integrate() below).
    (
        IFS=:
	for i in $DT_PROGRAMS
	do
		{
		echo '[Desktop Entry]'
		echo 'Encoding=UTF-8'
		echo 'Version=1.0'
		echo 'Type=Application'
		set_if_exists NAME Name
		set_if_exists de_NAME Name[de]
		set_if_exists ja_NAME Name[ja]
		echo_if_exists EXTRAS_1
		echo_if_exists EXTRAS_2
		echo_if_exists EXTRAS_3
		echo 'Categories=Application;Network;X-Red-Hat-Base;X-SuSE-Core-Internet;'
		# the installation directory file path might contain non-UTF-8 characters,
		#   e.g. EUC-JP characters if installing on a Japanese machine;
		#   this will cause problems on KDE (see CPR 67106 Notes).

        if [ "$i" = "selfservice" ]
		then
		# Add "--icaroot"
            echo 'Icon='$DesktopEscapedICAInstDir'/icons/receiver.png'
			echo 'TryExec='$DesktopEscapedICAInstDir/$i
			printf "Exec=$DesktopEscapedICAInstDirCommand/$i --icaroot $DesktopEscapedICAInstDirCommand %s\n" `echo_if_exists EXEC`
		else
			if [ "$i" = "wfica" ]
			then
				# Add "-icaroot"
                echo 'Icon='$DesktopEscapedICAInstDir'/icons/receiver.png'
				echo 'TryExec='$DesktopEscapedICAInstDir/$i
				printf "Exec=$DesktopEscapedICAInstDirCommand/$i -icaroot $DesktopEscapedICAInstDirCommand %s\n" `echo_if_exists EXEC`
			else
				if [ "$i" = "receiver" ] || [ "$i" = "receiver_fido2" ] || [ "$i" = "fido2_llt" ]
				then
					#Added by J
                    echo 'Icon='$DesktopEscapedICAInstDir'/icons/receiver.png'
					echo 'TryExec='$DesktopEscapedICAInstDir'/util/ctxwebhelper'
					printf "Exec= $DesktopEscapedICAInstDir/util/ctxwebhelper %s\n" `echo_if_exists EXEC`
				else
				    # Add '/util' to path and "-icaroot"
                    if [ "$i" = "logmgr" ] || [ "$i" = "sendfeedback" ]
                    then
                        echo 'Icon='$DesktopEscapedICAInstDir'/icons/utility.png'
                        echo 'TryExec='$DesktopEscapedICAInstDir'/util'/$i
                    else
                        echo 'Icon='$DesktopEscapedICAInstDir'/icons/utility.png'
					    echo 'TryExec='$DesktopEscapedICAInstDir'/util'/$i
					    printf "Exec=$DesktopEscapedICAInstDirCommand/util/$i -icaroot $DesktopEscapedICAInstDirCommand %s\n" `echo_if_exists EXEC`
                    fi
				fi
			fi
		fi
		} > "$DT_FILE_DIR/$i.desktop"
	done
	)
}

DT_remove_files()
{
    if [ ! -d "$DT_FILE_DIR" ]
    then
        return
    fi

    # Use a sub-shell and change $IFS for colon-separated list (see DT_integrate() below).
    (
        IFS=:
        for i in $DT_PROGRAMS
        do
            rm -f "$DT_FILE_DIR/$i.desktop" 2> /dev/null
        done
    )

    rmdir "$DT_FILE_DIR" 2> /dev/null
}

###############################################################################
#
#	DT_integrate()
#
#	This function is called to integrate the client with Gnome and KDE.
#
###############################################################################

DT_integrate()
{
    DT_get_target_dirs

    (
        # By setting IFS to be :, we avoid space-separation issues.
        IFS=:

        INTEGRATE=-1

        for i in $DT_TARGET_DIRS
        do
            if [ -z "$SU_INSTALL" -a ! -d $i ] #If it is not sudo and the director does not exit, create dir
            then
                create_dir $i
            fi
            if [ -d $i ] #If directory exists make INTEGRATE=1
            then
                INTEGRATE=1
            fi
        done

        if [ $INTEGRATE -eq -1 ]   #If integrate fails ...
        then
            user_echo $integrate_DT1
            return -1
        fi

        return 0
    )

    if [ $? -ne 0 ]
    then
        return
    fi

    # Spray links into directories.

    FIRST_ONLY=-1
    DO_LINK=1
    if [ -n "$SU_INSTALL" -a -d $XDG_TARGET_DIR -a ! -d $TL_TOP_DIR ] #If sudo user and XDG_TARGET_DIR exists and TL_TOP_DIR does not exist
    then
	DESKTOP_FILE_COUNT=`ls -1 $XDG_TARGET_DIR/*.desktop 2> /dev/null | wc -c`
	if [ -n "$DESKTOP_FILE_COUNT" -a "$DESKTOP_FILE_COUNT" -ge 15 ]
	then
	    # If XDG_TARGET_DIR exists and looks well-populated
	    # assume it is the only directory we need to link from.
	    # Assume it is the first directory in the list $DT_TARGET_DIRS.

	    FIRST_ONLY=1
	fi
    fi

    (
        IFS=:
        for i in $DT_TARGET_DIRS
        do
            if [ -d $i -a -w $i ]
            then
                for prog in $DT_PROGRAMS
                do
                    link=$i/${prog}.desktop
                    $LNKTST0 $link && rm -f $link #if $link exists and is a symbolic link then remove the link
                    [ $DO_LINK -eq 1 ] && ln -s "$DT_FILE_DIR/${prog}.desktop" $link #Make a symbolic link
                done

		[ $FIRST_ONLY -ne -1 ] && DO_LINK=-1
            fi
        done
    )

    # Now that we have a wfica.desktop, make it the default ICA file handler
    # and similarly for the new_store script.

    DT_register_MIMEs;
}

###########################################################################
#
#	DT_disintegrate()
#
#	This function is called to disintegrate the client with GNOME and KDE.
#
###########################################################################

DT_disintegrate()
{
    DT_get_target_dirs

    # Remove dangling symbolic links to removed desktop files.

    (
        IFS=:
        for i in $DT_TARGET_DIRS
        do
            for prog in $DT_PROGRAMS
            do
                link=$i/${prog}.desktop
                SymlinkPointsToICAInst $link && rm -f $link
            done
        done
    )

    # The exit code at this point is quite likely to be non-zero (ie
    # false).  This can cause problems, particularly when uninstalling
    # the RPM.
    return 0
}

###########################################################################
#
#       DT_get_target_dirs()
#
#       This function is called to work out which window manager 
#       directories hold the ICA client information and put these
#       into the DT_TARGET_DIRS environment variable.
#
###########################################################################
DT_get_target_dirs()
{
    if [ -z "$DT_TARGET_DIRS" ]
    then
        if [ -n "$SU_INSTALL" ]
        then
            # Root install
            DT_TARGET_DIRS=`echo $DEFAULT_DT_TARGET_DIRS | sed -e "y/ /:/"`
        else
            # Non-privileged-user install
            # Check whether kde-config is present - run the command in a
            # subshell so that if it is not found, the error message from the
            # shell can be discarded
            if
               (kde-config -v > /dev/null) 2> /dev/null
            then
                USE_PARENT_DIR=""
                # Try more recent xdg option first. Otherwise fall back.
                DT_USER_DIR=`kde-config --path xdgdata-apps 2>/dev/null | cut -f 1 -d:`
                if [ -z "$DT_USER_DIR" ]
                then
                    DT_USER_DIR=`kde-config --path apps 2>/dev/null | cut -f 1 -d:`
                fi
	    else
                USE_PARENT_DIR=yes
		if [ -n "$KDEHOME" ]
                then
                    DT_USER_DIR=$KDEHOME/share
                else
                    DT_USER_DIR=$HOME/.kde/share
		fi
	    fi

	    if [ -d "$DT_USER_DIR/$TL_MENU_DIR" ]
            then
		DT_USER_DIR="$DT_USER_DIR/$TL_MENU_DIR"    # TurboLinux
	    fi

            # Create the per-user settings so that they appear in an
            # existing system folder, if possible.

            if [ -n "$DT_USER_DIR" ]
            then
                for i in $DEFAULT_KDE_TARGET_DIRS
                do
                    if [ -d $i ]
                    then
			if [ -n "$USE_PARENT_DIR" ]
			then
			    PARENT_DIR=`dirname $i`
			    DT_NEW_DIR=`basename $PARENT_DIR`/`basename $i`
			else
                            DT_NEW_DIR=`basename $i`
			fi
                        DT_TARGET_DIRS="$DT_USER_DIR/$DT_NEW_DIR"
                        break
                    fi
                done
                if [ -z "$DT_TARGET_DIRS" ]
                then
                    # None of the usual directories are present, put it somewhere
		    if [ -n "$USE_PARENT_DIR" ]
		    then
			(
			    # Make sure that spaces in users' home directories do not confuse the "for"
			    IFS=\0
			    for i in $DT_USER_DIR/applnk*
                            do
				if [ `basename $i` = 'applnk*' ]
				then
				    DT_USER_DIR=$DT_USER_DIR/applnk
				else
				    DT_USER_DIR=$i
				fi
				break;
			    done
			)
		    fi
                    DT_TARGET_DIRS="$DT_USER_DIR/$DEFAULT_DT_FOLDER"
                fi
            fi

	    # Make sure $HOME/.local/share/applications is in the list.

	    for i in DT_TARGET_DIRS
	    do
	        if [ "$i" = "$DT_SHARED_USER_DIR" ]
		then
		    return
		fi
	    done
	    DT_TARGET_DIRS=$DT_TARGET_DIRS:$DT_SHARED_USER_DIR
        fi
    fi
}


# commands to run the integrations

RunningFrom=INSTALLED
INSTALLER_YES=yes

CurrentIDNAME="$ID_PRODUCT_NAME"
CurrentIDVER="$ID_VERSION"
CurrentCHANGENO="$CHANGE_NO"
CurrentDISPNAME="$DISP_PRODUCT_NAME"
CurrentDISPVER="$DISP_VERSION"

set_install_mode

if [ -n "$SU_INSTALL" ]
then
    # Ensure that the configuration files are readable by world, but not
    # writeable.
    umask 022
else
    if [ -n "$HOME" ] ; then
        DefaultInstallDir=${HOME}/ICAClient/${PORT}
    fi
fi


setICAInstDirNoPrompt

# Add this here to simulate side-effects of calling install_packages() wrt the INST_DIR var
INST_DIR=$ICAInstDir

if [ x`basename $0` == xintegrate.sh ]
then
	# Make pkginf directory
	put_version_info "RPM_installation" "RPM_package" core $PORT

	# Make locale-dependent symbolic links

	lowercase_letters=abcdefghijklmnopqrstuvwxyz
	uppercase_letters=ABCDEFGHIJKLMNOPQRSTUVWXYZ
	get_tr_key $INST_DIR/pkginf/Ver.core.*
	cd $INST_DIR
	make_locale_links

	# Perform integrations
	install_ctxcwalogd "not_used" "not_used" "yes"
	integrate_ICA_client
elif [ x`basename $0` == xdisintegrate.sh ]
then
	# Remove pkginf directory
	rm -rf $INST_DIR/pkginf

	# Perform de-integration
	disintegrate_ICA_client

	# Clean up locale-specific links

	rm -f $INST_DIR/config/module.ini $INST_DIR/eula.txt
	rm -f $INST_DIR/config/wfclient.template $INST_DIR/config/appsrv.template
	rm -f $INST_DIR/config/debug.ini $INST_DIR/help/index.htm

	# Clean up generated scripts

	rm -f $INST_DIR/wfica.sh
	remove_ctxcwalogd
fi
exit 0
