#!/bin/bash
#############################################################
# Citrix Workspace app for Linux Check for dependencies #
#############################################################
# Copyright 2020-2020 Citrix Systems, Inc.  All Rights Reserved.

# Usage: ./workspacecheck.sh
#
# How to use: 1.) Get the Latest receiver Tar.gz archive from Citrix.com
#             2.) Extract the archive to /tmp/
#             3.) Run the script
#
# Note: If you wish to extract the client to a different folder, just change the
# 	$ClientPath variable below to wherever the client is.
#
# Output: Results are shown on stdout and a log file is produced
#
# Script dependencies(any version):
# bash
# awk
# curl
# ldd
#
# This script checks for the all the requirements for respective features

logfile=/tmp/logfile.txt
rm -rf /tmp/logfile.txt
clear


CitrixBrandName="Citrix Workspace"
ShieldAlias="Service Continuity"
FullPathForLib=""
DefaultRootClientPath=/opt/Citrix/ICAClient
DefaultUserClientPath=`echo $HOME/ICAClient/linux*`

#if ICAROOT is set use it, otherwise use a default client location
if [ ! "$ICAROOT" ]
then
    if [ -d "$DefaultRootClientPath" ]
    then
        ClientPath="$DefaultRootClientPath"
    elif [ -d "$DefaultUserClientPath" ]
    then
        ClientPath="$DefaultUserClientPath"
    else
        echo "Please export ICAROOT no default installation could be found." | tee -a $logfile
        exit 1
    fi
else
    if [ ! -d "$ICAROOT" ]
    then
        echo "Invalid ICAROOT - $ICAROOT: No such file or directory" | tee -a $logfile
        exit 1
    else
        ClientPath="$ICAROOT"
    fi
fi

HDXCheckScript="${ClientPath}/util/hdxcheck.sh"
LibPath="/usr/lib"
IncludePath="/usr/include"

initializeANSI()
{
  esc=""

  whitef="${esc}[37m"   blueb="${esc}[44m";
  boldon="${esc}[1m";    boldoff="${esc}[22m"
  italicson="${esc}[3m"; italicsoff="${esc}[23m"
  reset="${esc}[0m";    redb="${esc}[41m"
}

initializeANSI
versionCompare() {
  if [[ $1 == $2 ]]
    then
        return '0'
    fi
    local IFS=.
    local i ver1=($1) ver2=($2)
    # fill empty fields in ver1 with zeros
    for ((i=${#ver1[@]}; i<${#ver2[@]}; i++))
    do
        ver1[i]=0
    done
    for ((i=0; i<${#ver1[@]}; i++))
    do
        if [[ -z ${ver2[i]} ]]
        then
            # fill empty fields in ver2 with zeros
            ver2[i]=0
        fi
        if ((${ver1[i]} > ${ver2[i]}))
        then
            return '1'
        fi
        if ((${ver1[i]} < ${ver2[i]}))
        then
            return '0' #0 if ver2 is bigger
        fi
    done
    return '0'
}

check_lib()
{
    local libname="$1" goodmsg="$2" badmsg="$3"
    if [ "$archVer" = "x86_64" ];then
    #64 bit check
    libs=$(find /usr/lib /lib /usr/lib64 /lib64 -name '*'$libname'*' -and \( -type f -or -type l \) 2>/dev/null \
			| xargs file 2>/dev/null | grep -e 'ELF[[:space:]]*64-bit[[:space:]]*LSB[[:space:]]*shared[[:space:]]*object' | cut -d: -f1 )
    else
    #32 bit check(armel and armhf included)
    libs=$(find /usr/lib /lib -name '*'$libname'*' -and \( -type f -or -type l \) \
			| xargs file 2>/dev/null | grep -e 'ELF[[:space:]]*32-bit[[:space:]]*LSB[[:space:]]*shared[[:space:]]*object' | cut -d: -f1 )
    fi
    echo $libs >> $logfile
    if [ -n "$libs" ]
    then
        if [ "$goodmsg" ]; then echo "$goodmsg" | tee -a $logfile; fi
        Problem="0"
    else
        if [ "$badmsg"  ]; then echo "${redb}${boldon}${whitef}$badmsg${reset}"  | tee -a $logfile; fi
        Problem="1"
    fi
}

get_full_path_of_lib()
{
    local libname="$1"

    if [ "$archVer" = "x86_64" ];then
    #64 bit check
    FullPathForLib=$(find /usr/lib /lib /usr/lib64 /lib64 -name '*'$libname'*' -and \( -type f -or -type l \) 2>/dev/null \
                        | xargs file 2>/dev/null | grep -e 'ELF[[:space:]]*64-bit[[:space:]]*LSB[[:space:]]*shared[[:space:]]*object' | cut -d: -f1 )
    else
    #32 bit check(armel and armhf included)
    FullPathForLib=$(find /usr/lib /lib -name '*'$libname'*' -and \( -type f -or -type l \) \
                        | xargs file 2>/dev/null | grep -e 'ELF[[:space:]]*32-bit[[:space:]]*LSB[[:space:]]*shared[[:space:]]*object' | cut -d: -f1 )
    fi

    echo $FullPathForLib >> $logfile
}

# Gnome-keyring version must be 3.18.3 or later to support shield
check_version_for_gnome_keyring()
{
    local cmdToCheck="/usr/bin/gnome-keyring"

    if [ ! -f $cmdToCheck ]; then
        echo $cmdToCheck not exist >> $logfile
        cmdToCheck=`which gnome-keyring`
        echo $cmdToCheck get from which command >> $logfile
    fi

    echo `$cmdToCheck version` >> $logfile

    local gnomeKeyringMinVersion="3.18.3"
    local gnomeKeyringVersion=`$cmdToCheck version | cut -d':' -f2`

    versionCompare $gnomeKeyringMinVersion $gnomeKeyringVersion
    if [ $? -eq "0" ]
    then
        echo "Success! - A compatible version of Gnome-keyring is installed!" | tee -a $logfile
    else
        echo "Warning! - A compatible version of Gnome-keyring missing, ${ShieldAlias} will not be supported" | tee -a $logfile
    fi
}

check_dependencies_for_shield()
{
    echo "${blueb}${boldon}${whitef}---------------------------------${reset}"
    echo "${blueb}${boldon}${whitef}-- Checking for ${ShieldAlias} Support ... --${reset}"
    echo "${blueb}${boldon}${whitef}--------------------------------${reset}"

    check_lib "libsecret-1.so" "Success! - libsecret-1.so is installed." "Warning! - libsecret-1 missing, ${ShieldAlias} will not be supported"
    check_lib "libuuid.so" "Success! - libuuid.so is installed." "Warning! - libuuid missing, ${ShieldAlias} will not be supported"

    check_version_for_gnome_keyring

    get_full_path_of_lib "libwebkit2gtk-"
    local currentWebkit2GtkLibVersion=`echo $FullPathForLib | awk -F '-' '{print $NF}'`
    local webkit2GtkLibMinVersion="4.0.so.37.49.0"

    versionCompare $webkit2GtkLibMinVersion $currentWebkit2GtkLibVersion

    if [ $? -eq "0" ]
    then
        echo "Success! - A compatible version of libwebkit2gtk is installed!" | tee -a $logfile
    else
        echo "Warning! - A compatible version of libwebkit2gtk missing, ${ShieldAlias} will not be supported" | tee -a $logfile
    fi
}

if [ -x $HDXCheckScript ]
then
    source $HDXCheckScript
else
    echo "Could not find HDXCheck script"
fi


####################################################################
echo "${blueb}${boldon}${whitef}---------------------------------${reset}"
echo "${blueb}${boldon}${whitef}-- Checking for AML Support ... --${reset}"
echo "${blueb}${boldon}${whitef}--------------------------------${reset}"
check_lib "libsecret-1.so.0" "Success! - libsecret-1.so.0 is installed. AML support is present." "Warning! - libsecret-1.so.0 \
missing, AML will not be supported"
echo ""

####################################################################

check_dependencies_for_shield

echo "Logfile: $logfile"
