#!/bin/bash

set -e  # Exit immediately if a command exits with a non-zero status

# Function to print usage
print_usage() 
{
  echo "Usage: $0 [options]"
  echo "Options:"
  echo "  --store-url <URL>    Specify the store URL to be enumerated by ICAClient."
  echo "  -h, --help           Display this help message and exit."
}

# Function to log messages with timestamps
log() 
{
  echo "$(date +'%Y-%m-%d %H:%M:%S') - $1"
}

# Function to extract the first resource ID
extract_resource_id() 
{
  local url="$1"
  local enumerateOutput="$2"
  # Extract the first resource ID after the store URL
  echo "$enumerateOutput" | awk -v url="$url" -F"'" '
    BEGIN { flag=0 }
    $0 ~ url { flag=1; next }
    flag && /^'\''/ { print $2; flag=0 }
  '
}


# Function to add and enumerate store
add_and_enumerate_store() 
{
  local store_url=$1
  local storebrowse_bin=$2

  log "Adding store URL: $store_url"
  local storeAdded
  storeAdded=$("$storebrowse_bin" -a "$store_url")

  if [[ "$storeAdded" == *"Error adding store:"* ]]; then
    log "Error adding store: $storeAdded"
    exit 1
  else
    log "Store URL added successfully: $storeAdded"
  fi

  log "Enumerating added store"
  local enumerateOutput
  enumerateOutput=$("$storebrowse_bin" -E "$storeAdded" 2>&1)

  log "Enumerated Apps/Resources: $enumerateOutput"
  
  # Extract and log the first resource ID
  local resource_id
  resource_id=$(extract_resource_id "$store_url" "$enumerateOutput")
  if [[ -n "$resource_id" ]]; then
    log "First resource ID: $resource_id"
    local resourceLaunchDetails
    resourceLaunchDetails=$("$storebrowse_bin" -L "$resource_id" "$storeAdded" 2>&1)
  else
    log "No resource found."
  fi
}

# Function to prepare ICAClient configuration
configure_ICAClient_store() 
{
  local store_url=$1
  local user_profile="$HOME/.ICAClient"
  local icaroot_dir="/opt/Citrix/ICAClient"
  local storebrowse_bin="$icaroot_dir/util/storebrowse"

  mkdir -p "$user_profile"
  touch "$user_profile/.eula_accepted"

  add_and_enumerate_store "$store_url" "$storebrowse_bin"
}

# Parse command line arguments
STORE_URL=""
while [[ "$#" -gt 0 ]]; do
  case $1 in
    --store-url) STORE_URL="$2"; shift ;;
    -h|--help) print_usage; exit 0 ;;
    *) echo "Unknown parameter passed: $1"; print_usage; exit 1 ;;
  esac
  shift
done

# Ensure the store URL is provided
if [[ -z "$STORE_URL" ]]; then
  echo "Error: Store URL is required."
  print_usage
  exit 1
fi

# Prepare ICAClient configuration
configure_ICAClient_store "$STORE_URL"

log "Script execution completed successfully."

