#!/bin/bash

set -e  # Exit immediately if a command exits with a non-zero status

# Logging function
log() 
{
  echo "$(date +'%Y-%m-%d %H:%M:%S') - $1"
}

# Function to detect the package manager
detect_package_manager() {
  if command -v apt-get > /dev/null 2>&1; then
    echo "apt-get"
  elif command -v yum > /dev/null 2>&1; then
    echo "yum"
  elif command -v dnf > /dev/null 2>&1; then
    echo "dnf"
  else
    log "Unsupported package manager."
    exit 1
  fi
}

# Function to install a package
install_package() {
  local PACKAGE=$1
  local PACKAGE_MANAGER=$2

  if ! dpkg -l | grep -q "$PACKAGE"; then
    log "Installing $PACKAGE..."
    case "$PACKAGE_MANAGER" in
      apt-get)
        sudo apt-get install -y "$PACKAGE"
        ;;
      yum)
        sudo yum install -y "$PACKAGE"
        ;;
      dnf)
        sudo dnf install -y "$PACKAGE"
        ;;
    esac
  else
    log "$PACKAGE is already installed."
  fi
}

# Function to install python3 and pip3
install_python_and_pip() {
  local PACKAGE_MANAGER=$1

  case "$PACKAGE_MANAGER" in
    apt-get)
      if ! command -v python3 > /dev/null 2>&1; then
        log "Python 3 is not installed. Installing Python 3..."
        sudo apt-get update
        sudo apt-get install -y python3
      else
        log "Python 3 is already installed."
      fi

      if ! command -v pip3 > /dev/null 2>&1; then
        log "pip3 is not installed. Installing pip3..."
        sudo apt-get update
        sudo apt-get install -y python3-pip
      else
        log "pip3 is already installed."
      fi
      ;;
    yum)
      if ! command -v python3 > /dev/null 2>&1; then
        log "Python 3 is not installed. Installing Python 3..."
        sudo yum install -y python3
      else
        log "Python 3 is already installed."
      fi

      if ! command -v pip3 > /dev/null 2>&1; then
        log "pip3 is not installed. Installing pip3..."
        sudo yum install -y python3-pip
      else
        log "pip3 is already installed."
      fi
      ;;
    dnf)
      if ! command -v python3 > /dev/null 2>&1; then
        log "Python 3 is not installed. Installing Python 3..."
        sudo dnf install -y python3
      else
        log "Python 3 is already installed."
      fi

      if ! command -v pip3 > /dev/null 2>&1; then
        log "pip3 is not installed. Installing pip3..."
        sudo dnf install -y python3-pip
      else
        log "pip3 is already installed."
      fi
      ;;
  esac
}

# Function to install a Python package using pip3
install_python_package() {
  local PACKAGE=$1

  if ! pip3 show "$PACKAGE" > /dev/null 2>&1; then
    log "Installing Python package: $PACKAGE..."
    sudo pip3 install "$PACKAGE"
  else
    log "Python package $PACKAGE is already installed."
  fi
}

# Function to set up a system service
setup_service() {
  local SERVICE_FILE=$1
  local SERVICE_DIR=$2
  local SYSTEMCTL=$3

  log "Copying $SERVICE_FILE to $SERVICE_DIR..."
  sudo cp "/opt/Citrix/ICAClient/util/Fido2HIDBridge/$SERVICE_FILE" "$SERVICE_DIR"

  log "Reloading systemd daemon..."
  sudo $SYSTEMCTL daemon-reload

  log "Enabling $SERVICE_FILE to start at boot..."
  sudo $SYSTEMCTL enable "$SERVICE_FILE"

  log "Starting $SERVICE_FILE..."
  sudo $SYSTEMCTL start "$SERVICE_FILE"
}

# Main function
main() {
  if [[ "$EUID" -ne 0 ]]; then
    log "This script must be run as root."
    exit 1
  fi

  log "Starting FIDO2 service setup..."

  # Detect the package manager
  local PACKAGE_MANAGER
  PACKAGE_MANAGER=$(detect_package_manager)

  # Update package list
  log "Updating package list..."
  case "$PACKAGE_MANAGER" in
    apt-get)
      sudo apt-get update
      ;;
    yum)
      sudo yum update -y
      ;;
    dnf)
      sudo dnf check-update -y
      ;;
  esac

  # Install required packages
  install_package "swig" "$PACKAGE_MANAGER"
  install_package "libpcsclite-dev" "$PACKAGE_MANAGER"
  install_package "pcscd" "$PACKAGE_MANAGER"
  install_python_and_pip "$PACKAGE_MANAGER"
  install_python_package "pyscard"
  install_python_package "uhid"
  install_python_package "fido2"
  

  # Start pcscd service
  log "Starting pcscd service..."
  sudo systemctl start pcscd

  # Setup the service
  local SERVICE_DIR="/etc/systemd/system/"
  local SYSTEMCTL="systemctl"
  setup_service "fido2-hid-bridge.service" "$SERVICE_DIR" "$SYSTEMCTL"

  log "FIDO2 service setup completed."
}

# Execute main function
main "$@"
