#!/usr/bin/python3

# =============================================================
# This is the script to collect logs.
# Currently, it is designed to collect HDX Teams related logs.
# However, it can be used to collect any other components logs,
# just to add log and log path in logcollector.ini file, then 
# run the script. All logs specified in ini file will be stored.
#
# $Id$
# Copyright 2021 Citrix Systems, Inc.  All Rights Reserved.
#
# =============================================================
# -*- coding: utf-8 -*-

import os,sys
from os import path
import time, glob
from datetime import datetime
import tarfile
from configparser import ConfigParser
import argparse


def read_config(logini='/opt/Citrix/ICAClient/config/logcollector.ini'):
    log_dict = {}

    if not path.exists(logini) or not path.isfile(logini): 
        print('Error: log collector configuration file is missing: ', logini)
        return log_dict

    try:
        parser = ConfigParser()
        parser.read(logini)
    except configparser.ParsingError as err:
        print('Error: Could not parse: ', err)

    for section_name in parser:
        section = parser[section_name]
        templist = []

        for name in section:
            # expand wildcard and env variable in path
            if not glob.glob(path.expandvars(section[name])):
                print('Error: expand whildcard or variables got empty result: ' + section[name])
            else:
                expand_list = glob.glob(path.expandvars(section[name]))
                for item in expand_list:
                    opt = '{} = {}'.format(name, item)
                    templist.append(opt)

        log_dict[section_name] = templist

    #example:
    #log_dict = {'DEFAULT':[]',
    #            'common':['icaclient = /var/log/ICAClient.log', 'syslog = /var/log/syslog'],
    #            'hdxteams':['hdxrtcengine = "/tmp/hdxrtcengine"', 'webrpc = /tmp/webrpc', 'webrtc = /tmp/webrpc']}

    return log_dict


def gen_tarfile_name():
    now = datetime.now()
    file_name = 'cwalog.' + now.strftime("%Y_%d_%m_%H_%M_%S") +'.tar.gz'

    return file_name


def gen_tarfile(log_dict, tarpath='/tmp'):
    if not log_dict:
        print('Error: There is no content in log collector configuration file!')
        return

    file_name = gen_tarfile_name()
    print("INFO: Save log file in folder " + tarpath)
    full_path = path.join(tarpath, file_name)

    try:
        out = tarfile.open(full_path, mode='a')
    except Exception as e:
        print("Exception: ", e)
        return

    try:
        for seckey, optvalues in log_dict.items():

            if len(optvalues) > 0:
                for item in optvalues:
                    print("Info: handling " + item)
                    log = item.split("=")[-1]
                    if not log:
                        print("Error: Wrong format in logcollector.ini file, please check!")
                        break

                    try:
                        out.add(log.strip(" "), arcname=path.basename(log.strip(" ")))
                        print("Info: " + log + " is added to tar file")
                    except FileNotFoundError as err: 
                        print("Exception: " + log + " is not found, just skip", err)
                    except IOError as e:
                        print("IO Exception: ", e)
                    except Exception as exp:
                        print("Exception: ", exp)
                        raise
    finally:
        print('Debug: closing tar archive')
        out.close()


def remove_tmpfile(tmpfile):
    if os.path.exists(tmpfile):
        os.remove(tempfile)
    else:
        print("Error: The file does not exist!: " + tmpfile)



if __name__ == '__main__':
    parser = argparse.ArgumentParser()
    parser.add_argument("-c", "--config", help="The logcollector.ini path & file")
    parser.add_argument("-a", "--archive", help="The archive path & file")
    args = parser.parse_args()

    answer = input('Notice: This tool may collect syslog. Do you want to continue?:')
    if answer.lower().startswith("y"):
        print("ok, carry on then")
    elif answer.lower().startswith("n"):
        print("ok, sayonnara")
        exit()

    log_dict = {}
    try:
        icaroot = os.environ['ICAROOT']
    except KeyError:
        if args.config is None:
            print("INFO: ICAROOT is not set, just use default /opt/Citrix/ICAClient")
            icaroot = "/opt/Citrix/ICAClient"

    if args.config:
        log_dict = read_config(args.config)
    else:
        config = icaroot + "/config/logcollector.ini"
        log_dict = read_config(config)

    if args.archive:
        gen_tarfile(log_dict, args.archive)
    else:
        gen_tarfile(log_dict)

