<# Copyright © 2023. Cloud Software Group, Inc. All Rights Reserved. #>
# This function is used to check if all required XmlElement are defined. Using the .Item() method, because .name would return the name of the parent if element is missing. 
Function Test-CTXOERegistryInput ([Xml.XmlElement]$Params) {
    # 'Path' is the only property that is  required for all registry operations
    If ($Params.Item("path") -isnot [Xml.XmlElement]) {Throw "Required property 'Path' is missing"}; 
    
    # Test if 'Path' contains unsupported ":" character (PSdrive). ":" is supported as part of key name, but not as part of root drive (e.g. 'HKLM:\Software'), so we cannot use simple .Contains() and have to parse the first token instead
    If ($Params.Path.Split("\")[0][-1] -eq ":") {Throw "Registry path includes unsupported ':' character in root key path"};

    # DeleteKey requires only path, but doesn't care about value name. We don't need to validate any other parameters and can exit
    If ($Params.Value -eq "CTXOE_DeleteKey") {Return;}

    # "Name" is just an optional parameter. If it is not specified, Optimizer will just create empty registry key. However if 'Name' is provided, we need to make sure that also 'ValueType' and 'Value' are provided
    If ($Params.Item("name") -is [Xml.XmlElement]) {
        If ($Params.Item("value") -isnot [Xml.XmlElement]) {Throw "Required property Value is missing"};
        If ($Params.Item("valuetype") -isnot [Xml.XmlElement] -and $Params.value -ne "CTXOE_DeleteValue") {Throw "Required property ValueType is missing"};
    } Else {
        If ($Params.Value -eq "CTXOE_DeleteValue") {Throw "Required property Name is missing"};
    }
    
}

Function Invoke-CTXOERegistryAnalyze ([Xml.XmlElement]$Params) {
    Test-CtxOEREgistryInput -Params $Params

    # When modifications are made to default user, we need to load the registry hive file first
    # Since there are few different places where code can be returned in Test-CTXOERegistryValue function, we are handling hive loading here

    [Boolean]$m_IsDefaultUserProfile = $Params.Path -like "HKDU\*" -or $Params.Path -like "HKEY_USERS\DefaultUser\*";

    If ($m_IsDefaultUserProfile) {
            # Mount DefaultUser
            Reg Load "HKU\DefaultUser" "C:\Users\Default\NTUSER.DAT" | Out-Null;
            New-PSDrive -Name DefaultUser -PSProvider Registry -Root HKEY_USERS\DefaultUser | Out-Null;
    
            # Replace HKDU\ with actual registry path
            $Params.Path = $Params.Path -Replace "HKDU\\", "HKEY_USERS\DefaultUser\";
    }

    # Using psbase.InnerText because it's the only way how to handle (optional) XML elements. If we use $Params.Name and 'name' doesn't exist, parent element (params) is used by function instead.
    [Hashtable]$m_Result = CTXOE\Test-CTXOERegistryValue -Key $Params.Path -Name $Params.Item("name").psbase.InnerText -Value $Params.Item("value").psbase.InnerText
    $Global:CTXOE_Result = $m_Result.Result
    $Global:CTXOE_Details = $m_Result.Details

    If ($m_IsDefaultUserProfile) {
        # Unmount DefaultUser
        Remove-PSDrive -Name DefaultUser;
        [GC]::Collect();
        Reg Unload "HKU\DefaultUser" | Out-Null;
    }
    
}

Function Invoke-CTXOERegistryExecuteInternal ([Xml.XmlElement]$Params, [Boolean]$RollbackSupported = $False) {
    Test-CtxOEREgistryInput -Params $Params

    # When modifications are made to default user, we need to load the registry hive file first
    # Since there are few different places where code can be returned in Test-CTXOERegistryValue function, we are handling hive loading here

    [Boolean]$m_IsDefaultUserProfile = $Params.Path -Like "HKDU\*" -or $Params.Path -Like "HKEY_USERS\DefaultUser\*";

    If ($m_IsDefaultUserProfile) {
            # Mount DefaultUser
            Reg Load "HKU\DefaultUser" "C:\Users\Default\NTUSER.DAT" | Out-Null;
            New-PSDrive -Name DefaultUser -PSProvider Registry -Root HKEY_USERS\DefaultUser | Out-Null;
    
            # Replace HKDU\ with actual registry path
            $Params.Path = $Params.Path -Replace "HKDU\\","HKEY_USERS\DefaultUser\";
    }

    # Test if value is already configured or not
    [Hashtable]$m_Result = CTXOE\Test-CTXOERegistryValue -Key $Params.Path -Name $Params.Item("name").psbase.InnerText -Value $Params.Item("value").psbase.InnerText
    
    # If the value is not configured, change it
    If ($m_Result.Result -ne $true) {
        [Hashtable]$m_Result = CTXOE\Set-CTXOERegistryValue -Key $Params.Path -ValueType $Params.Item("valuetype").psbase.InnerText -Value $Params.Item("value").psbase.InnerText -Name $Params.Item("name").psbase.InnerText
        $Global:CTXOE_SystemChanged = $true;
        
        If ($RollbackSupported) {
            [Xml.XmlDocument]$m_RollbackElement = CTXOE\ConvertTo-CTXOERollbackElement -Element $Params
            $m_RollbackElement.rollbackparams.value = $m_Result.OriginalValue.ToString();
            $Global:CTXOE_ChangeRollbackParams = $m_RollbackElement
        }
    }

    $Global:CTXOE_Result = $m_Result.Result
    $Global:CTXOE_Details = $m_Result.Details

    If ($m_IsDefaultUserProfile) {
        # Unmount DefaultUser
        Remove-PSDrive -Name DefaultUser;
        [GC]::Collect();
        Reg Unload "HKU\DefaultUser" | Out-Null;
    }
}

Function Invoke-CTXOERegistryExecute ([Xml.XmlElement]$Params) {
    Invoke-CTXOERegistryExecuteInternal -Params $Params -RollbackSupported $True
}

Function Invoke-CTXOERegistryRollback ([Xml.XmlElement]$Params) {
    Invoke-CTXOERegistryExecuteInternal -Params $Params -RollbackSupported $False
}
# SIG # Begin signature block
# MIIoogYJKoZIhvcNAQcCoIIokzCCKI8CAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCAarL58lVsE8abG
# Xo1yqxdPtWFNwD8sOg63uTJndHORS6CCDcAwggawMIIEmKADAgECAhAIrUCyYNKc
# TJ9ezam9k67ZMA0GCSqGSIb3DQEBDAUAMGIxCzAJBgNVBAYTAlVTMRUwEwYDVQQK
# EwxEaWdpQ2VydCBJbmMxGTAXBgNVBAsTEHd3dy5kaWdpY2VydC5jb20xITAfBgNV
# BAMTGERpZ2lDZXJ0IFRydXN0ZWQgUm9vdCBHNDAeFw0yMTA0MjkwMDAwMDBaFw0z
# NjA0MjgyMzU5NTlaMGkxCzAJBgNVBAYTAlVTMRcwFQYDVQQKEw5EaWdpQ2VydCwg
# SW5jLjFBMD8GA1UEAxM4RGlnaUNlcnQgVHJ1c3RlZCBHNCBDb2RlIFNpZ25pbmcg
# UlNBNDA5NiBTSEEzODQgMjAyMSBDQTEwggIiMA0GCSqGSIb3DQEBAQUAA4ICDwAw
# ggIKAoICAQDVtC9C0CiteLdd1TlZG7GIQvUzjOs9gZdwxbvEhSYwn6SOaNhc9es0
# JAfhS0/TeEP0F9ce2vnS1WcaUk8OoVf8iJnBkcyBAz5NcCRks43iCH00fUyAVxJr
# Q5qZ8sU7H/Lvy0daE6ZMswEgJfMQ04uy+wjwiuCdCcBlp/qYgEk1hz1RGeiQIXhF
# LqGfLOEYwhrMxe6TSXBCMo/7xuoc82VokaJNTIIRSFJo3hC9FFdd6BgTZcV/sk+F
# LEikVoQ11vkunKoAFdE3/hoGlMJ8yOobMubKwvSnowMOdKWvObarYBLj6Na59zHh
# 3K3kGKDYwSNHR7OhD26jq22YBoMbt2pnLdK9RBqSEIGPsDsJ18ebMlrC/2pgVItJ
# wZPt4bRc4G/rJvmM1bL5OBDm6s6R9b7T+2+TYTRcvJNFKIM2KmYoX7BzzosmJQay
# g9Rc9hUZTO1i4F4z8ujo7AqnsAMrkbI2eb73rQgedaZlzLvjSFDzd5Ea/ttQokbI
# YViY9XwCFjyDKK05huzUtw1T0PhH5nUwjewwk3YUpltLXXRhTT8SkXbev1jLchAp
# QfDVxW0mdmgRQRNYmtwmKwH0iU1Z23jPgUo+QEdfyYFQc4UQIyFZYIpkVMHMIRro
# OBl8ZhzNeDhFMJlP/2NPTLuqDQhTQXxYPUez+rbsjDIJAsxsPAxWEQIDAQABo4IB
# WTCCAVUwEgYDVR0TAQH/BAgwBgEB/wIBADAdBgNVHQ4EFgQUaDfg67Y7+F8Rhvv+
# YXsIiGX0TkIwHwYDVR0jBBgwFoAU7NfjgtJxXWRM3y5nP+e6mK4cD08wDgYDVR0P
# AQH/BAQDAgGGMBMGA1UdJQQMMAoGCCsGAQUFBwMDMHcGCCsGAQUFBwEBBGswaTAk
# BggrBgEFBQcwAYYYaHR0cDovL29jc3AuZGlnaWNlcnQuY29tMEEGCCsGAQUFBzAC
# hjVodHRwOi8vY2FjZXJ0cy5kaWdpY2VydC5jb20vRGlnaUNlcnRUcnVzdGVkUm9v
# dEc0LmNydDBDBgNVHR8EPDA6MDigNqA0hjJodHRwOi8vY3JsMy5kaWdpY2VydC5j
# b20vRGlnaUNlcnRUcnVzdGVkUm9vdEc0LmNybDAcBgNVHSAEFTATMAcGBWeBDAED
# MAgGBmeBDAEEATANBgkqhkiG9w0BAQwFAAOCAgEAOiNEPY0Idu6PvDqZ01bgAhql
# +Eg08yy25nRm95RysQDKr2wwJxMSnpBEn0v9nqN8JtU3vDpdSG2V1T9J9Ce7FoFF
# UP2cvbaF4HZ+N3HLIvdaqpDP9ZNq4+sg0dVQeYiaiorBtr2hSBh+3NiAGhEZGM1h
# mYFW9snjdufE5BtfQ/g+lP92OT2e1JnPSt0o618moZVYSNUa/tcnP/2Q0XaG3Ryw
# YFzzDaju4ImhvTnhOE7abrs2nfvlIVNaw8rpavGiPttDuDPITzgUkpn13c5Ubdld
# AhQfQDN8A+KVssIhdXNSy0bYxDQcoqVLjc1vdjcshT8azibpGL6QB7BDf5WIIIJw
# 8MzK7/0pNVwfiThV9zeKiwmhywvpMRr/LhlcOXHhvpynCgbWJme3kuZOX956rEnP
# LqR0kq3bPKSchh/jwVYbKyP/j7XqiHtwa+aguv06P0WmxOgWkVKLQcBIhEuWTatE
# QOON8BUozu3xGFYHKi8QxAwIZDwzj64ojDzLj4gLDb879M4ee47vtevLt/B3E+bn
# KD+sEq6lLyJsQfmCXBVmzGwOysWGw/YmMwwHS6DTBwJqakAwSEs0qFEgu60bhQji
# WQ1tygVQK+pKHJ6l/aCnHwZ05/LWUpD9r4VIIflXO7ScA+2GRfS0YW6/aOImYIbq
# yK+p/pQd52MbOoZWeE4wggcIMIIE8KADAgECAhADWiAGLDpbT4qELERZjaX6MA0G
# CSqGSIb3DQEBCwUAMGkxCzAJBgNVBAYTAlVTMRcwFQYDVQQKEw5EaWdpQ2VydCwg
# SW5jLjFBMD8GA1UEAxM4RGlnaUNlcnQgVHJ1c3RlZCBHNCBDb2RlIFNpZ25pbmcg
# UlNBNDA5NiBTSEEzODQgMjAyMSBDQTEwHhcNMjUwMjA3MDAwMDAwWhcNMjYwMjA2
# MjM1OTU5WjCBjzELMAkGA1UEBhMCVVMxEDAOBgNVBAgTB0Zsb3JpZGExGDAWBgNV
# BAcTD0ZvcnQgTGF1ZGVyZGFsZTEdMBsGA1UEChMUQ2l0cml4IFN5c3RlbXMsIElu
# Yy4xFjAUBgNVBAsTDUNpdHJpeCBYZW5BcHAxHTAbBgNVBAMTFENpdHJpeCBTeXN0
# ZW1zLCBJbmMuMIIBojANBgkqhkiG9w0BAQEFAAOCAY8AMIIBigKCAYEAnBm4JMW+
# W2tRIJltyaJM/q5G6WRAN293DlUc71U/dg6HzXRtahZP2kvV6ii32orbgdNkCwig
# +bo0W7AKjBBHKJqP96nuTNTsHoz+aeEE6PfgaFzGTRq77+dGDCk5DuPbFaXJCn9s
# UjVdbJTG1YXAcll8ZU6Iarg3eOokp2CcejMcjeD2dO8y89o6y2W5sWj2oIA+QRE9
# iU5qNfDLUtXo5i017JTN+qs9RtuwD77svXoV29wmkPmGuUq625ZZYFtRa0/t/C7w
# /k00UOFjykbvNBPj/cT67i/J/Um8tOanuC3cYU18VToMsfpS2t4irTdtrBzHupr/
# MB2DzZTwr5x2m3UzgEsnrr9bYnCIpOuW+K/oExuTpHtZsk6fnpoteOfyP059dNMg
# i1Gj074k6JfaJG+6fwKW0i2Unf7NDBArHdoHA6eIYB/OivPt4cmusgzRr2PziAl4
# LpA/9VRcnR68CWjnoTTr7qhdDeGqMgk24cNtmg+6BDt65GDVDX2ycthpAgMBAAGj
# ggIDMIIB/zAfBgNVHSMEGDAWgBRoN+Drtjv4XxGG+/5hewiIZfROQjAdBgNVHQ4E
# FgQUEpWeJpU+v9Nt97sQvlQgm4DN5wMwPgYDVR0gBDcwNTAzBgZngQwBBAEwKTAn
# BggrBgEFBQcCARYbaHR0cDovL3d3dy5kaWdpY2VydC5jb20vQ1BTMA4GA1UdDwEB
# /wQEAwIHgDATBgNVHSUEDDAKBggrBgEFBQcDAzCBtQYDVR0fBIGtMIGqMFOgUaBP
# hk1odHRwOi8vY3JsMy5kaWdpY2VydC5jb20vRGlnaUNlcnRUcnVzdGVkRzRDb2Rl
# U2lnbmluZ1JTQTQwOTZTSEEzODQyMDIxQ0ExLmNybDBToFGgT4ZNaHR0cDovL2Ny
# bDQuZGlnaWNlcnQuY29tL0RpZ2lDZXJ0VHJ1c3RlZEc0Q29kZVNpZ25pbmdSU0E0
# MDk2U0hBMzg0MjAyMUNBMS5jcmwwgZQGCCsGAQUFBwEBBIGHMIGEMCQGCCsGAQUF
# BzABhhhodHRwOi8vb2NzcC5kaWdpY2VydC5jb20wXAYIKwYBBQUHMAKGUGh0dHA6
# Ly9jYWNlcnRzLmRpZ2ljZXJ0LmNvbS9EaWdpQ2VydFRydXN0ZWRHNENvZGVTaWdu
# aW5nUlNBNDA5NlNIQTM4NDIwMjFDQTEuY3J0MAkGA1UdEwQCMAAwDQYJKoZIhvcN
# AQELBQADggIBAMGu7yYWlIS96uBwpNvMWgRgVFRwDx0HBdxGRi2+XvrtKkooOglx
# eUl/IYzZb8WhEBs/vknAmZkxBnkbbdgKKqxDeIkT2JX1sNGLKIzBBLPPjGMd2wxH
# CAVssWrA9xMESJ5Zx+5jHyoeVPt4XdPAXNB6Ca1JCa68Ip8Bn+atqop5wOmhXK85
# 2V2PdCWXjQr3Yl0k0PY/OtDDxnQc0dClj9TYlfGsOwyVNNQh/eExH2wkVlYYxgcj
# Zn89SmO4L+9AHkfpj/wd+/pJ+BYSflAotC6d9l4dft6f6PZ6KkbAz56dVkBX20xz
# YKKIK9tCf5sa1wnjkBEm2yn1IDn6K1cwUGO0whhtzGgcv4vPfv5bkumCSD7AquQv
# 37LP/gOZT9v3CiZAlTbZAoOK105qFmawt7AKPFzNFq9EjbUHYVBeYFnETDoa9zLz
# KCCcF/xJsjfn7YqFM+b1zq3C1YJK1B6+f3b5acpXNHbyMxPSOeNDZ13wMAYKah17
# D/4GZe74bsVqFo0oHlB82Bocr0AnnrLnLvfEirc0KIon5CQ5cci3LolBE0aNhD+n
# UtInPAJN1uRFpvhSK1ucVO/afmTQVtUgp5a8a8Dthc0bqh2ere4hoY+9U08Bda5g
# mFOO9YU8xEU3Do0AbuJPUVVALAAs0I+38RjYtIfPPt/T1zrMFHv+CMsKMYIaODCC
# GjQCAQEwfTBpMQswCQYDVQQGEwJVUzEXMBUGA1UEChMORGlnaUNlcnQsIEluYy4x
# QTA/BgNVBAMTOERpZ2lDZXJ0IFRydXN0ZWQgRzQgQ29kZSBTaWduaW5nIFJTQTQw
# OTYgU0hBMzg0IDIwMjEgQ0ExAhADWiAGLDpbT4qELERZjaX6MA0GCWCGSAFlAwQC
# AQUAoIHQMBkGCSqGSIb3DQEJAzEMBgorBgEEAYI3AgEEMBwGCisGAQQBgjcCAQsx
# DjAMBgorBgEEAYI3AgEVMC8GCSqGSIb3DQEJBDEiBCCDrC5wQAfgE/qgkpg5UvGL
# FIt+hgigAv6zf/RPKRLvZDBkBgorBgEEAYI3AgEMMVYwVKA4gDYAQwBpAHQAcgBp
# AHgAIABTAHUAcABwAG8AcgB0AGEAYgBpAGwAaQB0AHkAIABUAG8AbwBsAHOhGIAW
# aHR0cDovL3d3dy5jaXRyaXguY29tIDANBgkqhkiG9w0BAQEFAASCAYBrptF6KRKH
# FvUtla2dZptjoKET0J+qY1KgzMxm0Y+4YFeAFWGTjzcGXDxJFMMam5YnkieNQFZ9
# gIE7tdgxTjRVyY3UVmgUDRGFhTg8WPSe8ksNSGkbzDT1NJMuqQOf0KVRzkkx6hAI
# cM5xIn87VtGr/W4PCBnfFfTliUvYWB8U2C5S7/B6sv5hxH8lzL2qOgtjfUu9YDWw
# cwzFKV5IF53T5uiQ9Q+Kl1xAbsPZO1TP9/DxwumAs2cSGqHEESNLnVZxxZmbNmG5
# siNhDThB2zcm8GBXJwu9fTXPYAN7is0yVngxAwpitipXQ6szJBOmiSgPSSqUQo/8
# efJNZJ3CumfLM4gNMeZ82kMVjjH7J8lq7LG3nwNcNPKHs/6CRfFjsoGsIv71wbIZ
# C0vpZ9j0d2WNUnv/vKCH3oAXXyduBWxtDTkM7W6Iw5KyO01AGFJDd9r1Y9d95KBw
# MR9SptJdIpiHg7ciQ/VizahnKvbIs7jNlzcSC7xtEkrDpFN168S36jGhghc5MIIX
# NQYKKwYBBAGCNwMDATGCFyUwghchBgkqhkiG9w0BBwKgghcSMIIXDgIBAzEPMA0G
# CWCGSAFlAwQCAQUAMHcGCyqGSIb3DQEJEAEEoGgEZjBkAgEBBglghkgBhv1sBwEw
# MTANBglghkgBZQMEAgEFAAQgr7FJ4Dig7T2l4O089FazM8UnmYNtMPQBvlvK41aD
# aDYCEGqiPoFi6tcglzUP+Es1lrcYDzIwMjUwMzExMDYyNTMxWqCCEwMwgga8MIIE
# pKADAgECAhALrma8Wrp/lYfG+ekE4zMEMA0GCSqGSIb3DQEBCwUAMGMxCzAJBgNV
# BAYTAlVTMRcwFQYDVQQKEw5EaWdpQ2VydCwgSW5jLjE7MDkGA1UEAxMyRGlnaUNl
# cnQgVHJ1c3RlZCBHNCBSU0E0MDk2IFNIQTI1NiBUaW1lU3RhbXBpbmcgQ0EwHhcN
# MjQwOTI2MDAwMDAwWhcNMzUxMTI1MjM1OTU5WjBCMQswCQYDVQQGEwJVUzERMA8G
# A1UEChMIRGlnaUNlcnQxIDAeBgNVBAMTF0RpZ2lDZXJ0IFRpbWVzdGFtcCAyMDI0
# MIICIjANBgkqhkiG9w0BAQEFAAOCAg8AMIICCgKCAgEAvmpzn/aVIauWMLpbbeZZ
# o7Xo/ZEfGMSIO2qZ46XB/QowIEMSvgjEdEZ3v4vrrTHleW1JWGErrjOL0J4L0HqV
# R1czSzvUQ5xF7z4IQmn7dHY7yijvoQ7ujm0u6yXF2v1CrzZopykD07/9fpAT4Bxp
# T9vJoJqAsP8YuhRvflJ9YeHjes4fduksTHulntq9WelRWY++TFPxzZrbILRYynyE
# y7rS1lHQKFpXvo2GePfsMRhNf1F41nyEg5h7iOXv+vjX0K8RhUisfqw3TTLHj1uh
# S66YX2LZPxS4oaf33rp9HlfqSBePejlYeEdU740GKQM7SaVSH3TbBL8R6HwX9QVp
# GnXPlKdE4fBIn5BBFnV+KwPxRNUNK6lYk2y1WSKour4hJN0SMkoaNV8hyyADiX1x
# uTxKaXN12HgR+8WulU2d6zhzXomJ2PleI9V2yfmfXSPGYanGgxzqI+ShoOGLomMd
# 3mJt92nm7Mheng/TBeSA2z4I78JpwGpTRHiT7yHqBiV2ngUIyCtd0pZ8zg3S7bk4
# QC4RrcnKJ3FbjyPAGogmoiZ33c1HG93Vp6lJ415ERcC7bFQMRbxqrMVANiav1k42
# 5zYyFMyLNyE1QulQSgDpW9rtvVcIH7WvG9sqYup9j8z9J1XqbBZPJ5XLln8mS8wW
# mdDLnBHXgYly/p1DhoQo5fkCAwEAAaOCAYswggGHMA4GA1UdDwEB/wQEAwIHgDAM
# BgNVHRMBAf8EAjAAMBYGA1UdJQEB/wQMMAoGCCsGAQUFBwMIMCAGA1UdIAQZMBcw
# CAYGZ4EMAQQCMAsGCWCGSAGG/WwHATAfBgNVHSMEGDAWgBS6FtltTYUvcyl2mi91
# jGogj57IbzAdBgNVHQ4EFgQUn1csA3cOKBWQZqVjXu5Pkh92oFswWgYDVR0fBFMw
# UTBPoE2gS4ZJaHR0cDovL2NybDMuZGlnaWNlcnQuY29tL0RpZ2lDZXJ0VHJ1c3Rl
# ZEc0UlNBNDA5NlNIQTI1NlRpbWVTdGFtcGluZ0NBLmNybDCBkAYIKwYBBQUHAQEE
# gYMwgYAwJAYIKwYBBQUHMAGGGGh0dHA6Ly9vY3NwLmRpZ2ljZXJ0LmNvbTBYBggr
# BgEFBQcwAoZMaHR0cDovL2NhY2VydHMuZGlnaWNlcnQuY29tL0RpZ2lDZXJ0VHJ1
# c3RlZEc0UlNBNDA5NlNIQTI1NlRpbWVTdGFtcGluZ0NBLmNydDANBgkqhkiG9w0B
# AQsFAAOCAgEAPa0eH3aZW+M4hBJH2UOR9hHbm04IHdEoT8/T3HuBSyZeq3jSi5GX
# eWP7xCKhVireKCnCs+8GZl2uVYFvQe+pPTScVJeCZSsMo1JCoZN2mMew/L4tpqVN
# bSpWO9QGFwfMEy60HofN6V51sMLMXNTLfhVqs+e8haupWiArSozyAmGH/6oMQAh0
# 78qRh6wvJNU6gnh5OruCP1QUAvVSu4kqVOcJVozZR5RRb/zPd++PGE3qF1P3xWvY
# ViUJLsxtvge/mzA75oBfFZSbdakHJe2BVDGIGVNVjOp8sNt70+kEoMF+T6tptMUN
# lehSR7vM+C13v9+9ZOUKzfRUAYSyyEmYtsnpltD/GWX8eM70ls1V6QG/ZOB6b6Yu
# m1HvIiulqJ1Elesj5TMHq8CWT/xrW7twipXTJ5/i5pkU5E16RSBAdOp12aw8IQhh
# A/vEbFkEiF2abhuFixUDobZaA0VhqAsMHOmaT3XThZDNi5U2zHKhUs5uHHdG6BoQ
# au75KiNbh0c+hatSF+02kULkftARjsyEpHKsF7u5zKRbt5oK5YGwFvgc4pEVUNyt
# mB3BpIiowOIIuDgP5M9WArHYSAR16gc0dP2XdkMEP5eBsX7bf/MGN4K3HP50v/01
# ZHo/Z5lGLvNwQ7XHBx1yomzLP8lx4Q1zZKDyHcp4VQJLu2kWTsKsOqQwggauMIIE
# lqADAgECAhAHNje3JFR82Ees/ShmKl5bMA0GCSqGSIb3DQEBCwUAMGIxCzAJBgNV
# BAYTAlVTMRUwEwYDVQQKEwxEaWdpQ2VydCBJbmMxGTAXBgNVBAsTEHd3dy5kaWdp
# Y2VydC5jb20xITAfBgNVBAMTGERpZ2lDZXJ0IFRydXN0ZWQgUm9vdCBHNDAeFw0y
# MjAzMjMwMDAwMDBaFw0zNzAzMjIyMzU5NTlaMGMxCzAJBgNVBAYTAlVTMRcwFQYD
# VQQKEw5EaWdpQ2VydCwgSW5jLjE7MDkGA1UEAxMyRGlnaUNlcnQgVHJ1c3RlZCBH
# NCBSU0E0MDk2IFNIQTI1NiBUaW1lU3RhbXBpbmcgQ0EwggIiMA0GCSqGSIb3DQEB
# AQUAA4ICDwAwggIKAoICAQDGhjUGSbPBPXJJUVXHJQPE8pE3qZdRodbSg9GeTKJt
# oLDMg/la9hGhRBVCX6SI82j6ffOciQt/nR+eDzMfUBMLJnOWbfhXqAJ9/UO0hNoR
# 8XOxs+4rgISKIhjf69o9xBd/qxkrPkLcZ47qUT3w1lbU5ygt69OxtXXnHwZljZQp
# 09nsad/ZkIdGAHvbREGJ3HxqV3rwN3mfXazL6IRktFLydkf3YYMZ3V+0VAshaG43
# IbtArF+y3kp9zvU5EmfvDqVjbOSmxR3NNg1c1eYbqMFkdECnwHLFuk4fsbVYTXn+
# 149zk6wsOeKlSNbwsDETqVcplicu9Yemj052FVUmcJgmf6AaRyBD40NjgHt1bicl
# kJg6OBGz9vae5jtb7IHeIhTZgirHkr+g3uM+onP65x9abJTyUpURK1h0QCirc0PO
# 30qhHGs4xSnzyqqWc0Jon7ZGs506o9UD4L/wojzKQtwYSH8UNM/STKvvmz3+Drhk
# Kvp1KCRB7UK/BZxmSVJQ9FHzNklNiyDSLFc1eSuo80VgvCONWPfcYd6T/jnA+bIw
# pUzX6ZhKWD7TA4j+s4/TXkt2ElGTyYwMO1uKIqjBJgj5FBASA31fI7tk42PgpuE+
# 9sJ0sj8eCXbsq11GdeJgo1gJASgADoRU7s7pXcheMBK9Rp6103a50g5rmQzSM7TN
# sQIDAQABo4IBXTCCAVkwEgYDVR0TAQH/BAgwBgEB/wIBADAdBgNVHQ4EFgQUuhbZ
# bU2FL3MpdpovdYxqII+eyG8wHwYDVR0jBBgwFoAU7NfjgtJxXWRM3y5nP+e6mK4c
# D08wDgYDVR0PAQH/BAQDAgGGMBMGA1UdJQQMMAoGCCsGAQUFBwMIMHcGCCsGAQUF
# BwEBBGswaTAkBggrBgEFBQcwAYYYaHR0cDovL29jc3AuZGlnaWNlcnQuY29tMEEG
# CCsGAQUFBzAChjVodHRwOi8vY2FjZXJ0cy5kaWdpY2VydC5jb20vRGlnaUNlcnRU
# cnVzdGVkUm9vdEc0LmNydDBDBgNVHR8EPDA6MDigNqA0hjJodHRwOi8vY3JsMy5k
# aWdpY2VydC5jb20vRGlnaUNlcnRUcnVzdGVkUm9vdEc0LmNybDAgBgNVHSAEGTAX
# MAgGBmeBDAEEAjALBglghkgBhv1sBwEwDQYJKoZIhvcNAQELBQADggIBAH1ZjsCT
# tm+YqUQiAX5m1tghQuGwGC4QTRPPMFPOvxj7x1Bd4ksp+3CKDaopafxpwc8dB+k+
# YMjYC+VcW9dth/qEICU0MWfNthKWb8RQTGIdDAiCqBa9qVbPFXONASIlzpVpP0d3
# +3J0FNf/q0+KLHqrhc1DX+1gtqpPkWaeLJ7giqzl/Yy8ZCaHbJK9nXzQcAp876i8
# dU+6WvepELJd6f8oVInw1YpxdmXazPByoyP6wCeCRK6ZJxurJB4mwbfeKuv2nrF5
# mYGjVoarCkXJ38SNoOeY+/umnXKvxMfBwWpx2cYTgAnEtp/Nh4cku0+jSbl3ZpHx
# cpzpSwJSpzd+k1OsOx0ISQ+UzTl63f8lY5knLD0/a6fxZsNBzU+2QJshIUDQtxMk
# zdwdeDrknq3lNHGS1yZr5Dhzq6YBT70/O3itTK37xJV77QpfMzmHQXh6OOmc4d0j
# /R0o08f56PGYX/sr2H7yRp11LB4nLCbbbxV7HhmLNriT1ObyF5lZynDwN7+YAN8g
# Fk8n+2BnFqFmut1VwDophrCYoCvtlUG3OtUVmDG0YgkPCr2B2RP+v6TR81fZvAT6
# gt4y3wSJ8ADNXcL50CN/AAvkdgIm2fBldkKmKYcJRyvmfxqkhQ/8mJb2VVQrH4D6
# wPIOK+XW+6kvRBVK5xMOHds3OBqhK/bt1nz8MIIFjTCCBHWgAwIBAgIQDpsYjvnQ
# Lefv21DiCEAYWjANBgkqhkiG9w0BAQwFADBlMQswCQYDVQQGEwJVUzEVMBMGA1UE
# ChMMRGlnaUNlcnQgSW5jMRkwFwYDVQQLExB3d3cuZGlnaWNlcnQuY29tMSQwIgYD
# VQQDExtEaWdpQ2VydCBBc3N1cmVkIElEIFJvb3QgQ0EwHhcNMjIwODAxMDAwMDAw
# WhcNMzExMTA5MjM1OTU5WjBiMQswCQYDVQQGEwJVUzEVMBMGA1UEChMMRGlnaUNl
# cnQgSW5jMRkwFwYDVQQLExB3d3cuZGlnaWNlcnQuY29tMSEwHwYDVQQDExhEaWdp
# Q2VydCBUcnVzdGVkIFJvb3QgRzQwggIiMA0GCSqGSIb3DQEBAQUAA4ICDwAwggIK
# AoICAQC/5pBzaN675F1KPDAiMGkz7MKnJS7JIT3yithZwuEppz1Yq3aaza57G4QN
# xDAf8xukOBbrVsaXbR2rsnnyyhHS5F/WBTxSD1Ifxp4VpX6+n6lXFllVcq9ok3DC
# srp1mWpzMpTREEQQLt+C8weE5nQ7bXHiLQwb7iDVySAdYyktzuxeTsiT+CFhmzTr
# BcZe7FsavOvJz82sNEBfsXpm7nfISKhmV1efVFiODCu3T6cw2Vbuyntd463JT17l
# Necxy9qTXtyOj4DatpGYQJB5w3jHtrHEtWoYOAMQjdjUN6QuBX2I9YI+EJFwq1WC
# QTLX2wRzKm6RAXwhTNS8rhsDdV14Ztk6MUSaM0C/CNdaSaTC5qmgZ92kJ7yhTzm1
# EVgX9yRcRo9k98FpiHaYdj1ZXUJ2h4mXaXpI8OCiEhtmmnTK3kse5w5jrubU75KS
# Op493ADkRSWJtppEGSt+wJS00mFt6zPZxd9LBADMfRyVw4/3IbKyEbe7f/LVjHAs
# QWCqsWMYRJUadmJ+9oCw++hkpjPRiQfhvbfmQ6QYuKZ3AeEPlAwhHbJUKSWJbOUO
# UlFHdL4mrLZBdd56rF+NP8m800ERElvlEFDrMcXKchYiCd98THU/Y+whX8QgUWtv
# sauGi0/C1kVfnSD8oR7FwI+isX4KJpn15GkvmB0t9dmpsh3lGwIDAQABo4IBOjCC
# ATYwDwYDVR0TAQH/BAUwAwEB/zAdBgNVHQ4EFgQU7NfjgtJxXWRM3y5nP+e6mK4c
# D08wHwYDVR0jBBgwFoAUReuir/SSy4IxLVGLp6chnfNtyA8wDgYDVR0PAQH/BAQD
# AgGGMHkGCCsGAQUFBwEBBG0wazAkBggrBgEFBQcwAYYYaHR0cDovL29jc3AuZGln
# aWNlcnQuY29tMEMGCCsGAQUFBzAChjdodHRwOi8vY2FjZXJ0cy5kaWdpY2VydC5j
# b20vRGlnaUNlcnRBc3N1cmVkSURSb290Q0EuY3J0MEUGA1UdHwQ+MDwwOqA4oDaG
# NGh0dHA6Ly9jcmwzLmRpZ2ljZXJ0LmNvbS9EaWdpQ2VydEFzc3VyZWRJRFJvb3RD
# QS5jcmwwEQYDVR0gBAowCDAGBgRVHSAAMA0GCSqGSIb3DQEBDAUAA4IBAQBwoL9D
# XFXnOF+go3QbPbYW1/e/Vwe9mqyhhyzshV6pGrsi+IcaaVQi7aSId229GhT0E0p6
# Ly23OO/0/4C5+KH38nLeJLxSA8hO0Cre+i1Wz/n096wwepqLsl7Uz9FDRJtDIeuW
# cqFItJnLnU+nBgMTdydE1Od/6Fmo8L8vC6bp8jQ87PcDx4eo0kxAGTVGamlUsLih
# Vo7spNU96LHc/RzY9HdaXFSMb++hUD38dglohJ9vytsgjTVgHAIDyyCwrFigDkBj
# xZgiwbJZ9VVrzyerbHbObyMt9H5xaiNrIv8SuFQtJ37YOtnwtoeW/VvRXKwYw02f
# c7cBqZ9Xql4o4rmUMYIDdjCCA3ICAQEwdzBjMQswCQYDVQQGEwJVUzEXMBUGA1UE
# ChMORGlnaUNlcnQsIEluYy4xOzA5BgNVBAMTMkRpZ2lDZXJ0IFRydXN0ZWQgRzQg
# UlNBNDA5NiBTSEEyNTYgVGltZVN0YW1waW5nIENBAhALrma8Wrp/lYfG+ekE4zME
# MA0GCWCGSAFlAwQCAQUAoIHRMBoGCSqGSIb3DQEJAzENBgsqhkiG9w0BCRABBDAc
# BgkqhkiG9w0BCQUxDxcNMjUwMzExMDYyNTMxWjArBgsqhkiG9w0BCRACDDEcMBow
# GDAWBBTb04XuYtvSPnvk9nFIUIck1YZbRTAvBgkqhkiG9w0BCQQxIgQgDZwag98/
# V+crx/VPN7XGNOJEHPmbYFWQDrVcVLm4kqswNwYLKoZIhvcNAQkQAi8xKDAmMCQw
# IgQgdnafqPJjLx9DCzojMK7WVnX+13PbBdZluQWTmEOPmtswDQYJKoZIhvcNAQEB
# BQAEggIAOmrWo+8y2CMNATDHOg2ROXgpz/ipkKf+exC0j9yy1G02rThn7n7IJmre
# 0HtdIIXq4TWC8xADNZmLztqzPj94r16nAi+GVPPfC8ZTbEfCYITASD1FB56lEKAf
# PPEjqWZeLFo4c8X5rMgDFFbOKF0rxlQEYLsiYr/DpvcecuedxkH74A17u/NyGCdg
# Zyezrt0q8hlgydSZVF0iCS+Ml7M9vfXk4zp/7BLIg8KvHFsJZ9ri+gcF9qXurlYo
# +X4O+j4UkeKSYf+Th0ZBu43R+U4zviRICzYNOpsIAGAgxvtjhmE6pZEnxNpkGpzk
# l7St5aKjuYrPqYre51hgMfEnqm93vKAHtP4XvPrhuPlAFM4STlj0Ij+ZBdB8z008
# +nn40hxYIq5lIXXaAR1NMswac+mHV1ublX6rP3NOMGQd1PTfWx5nOcYhhYM1w+IB
# a6xov7l3MAcTaOySltrW04qr7ppQXKsFN2N3inHntJUuapwa+T/ajSpiht0I83sJ
# h9IjonljEYRVjuRAv+hlZnugGhucdYBQj6w1otXC1WPAacMdZ9cnwR0voviUfvuP
# aaBOA1c3qqBIgaYBMUd0ZTV0unsV1JSy3vqSV91itnNzYHBA3j6bsIvFmwjaVNak
# +eqAAk+StztI5NrPNyo61IywWI256uGJVJFyQk4PBHdP2wrnfZc=
# SIG # End signature block
