<#
    Copyright (c) 2023. Cloud Software Group, Inc. All Rights Reserved.
.SYNOPSIS 
    Citrix Optimization Engine helps to optimize operating system to run better with XenApp or XenDesktop solutions.

.DESCRIPTION
    Citrix Optimization Engine helps to optimize operating system to run better with XenApp or XenDesktop solutions. This script can run in three different modes - Analyze, Execute and Rollback. Each execution will automatically generate an XML file with a list of performed actions (stored under .\Logs folder) that can be used to rollback the changes applied. 

.PARAMETER Source
    Source XML file that contains the required configuration. Typically located under .\Templates folder. This file provides instructions that CTXOE can process. Template can be specified with or without .xml extension and can be full path or just a filename. If you specify only filename, template must be located in .\Templates folder.
    If -Source or -Template is not specified, Optimizer will automatically try to detect the best suitable template. It will look in .\Templates folder for file called <templateprefix>_<OS>_<build>. See help for -templateprefix to learn more about using your own custom templates.

.PARAMETER TemplatePrefix
    When -Source or -Template parameter is not specified, Optimizer will try to find the best matching template automatically. By default, it is looking for templates that start with "Citrix_Windows" and are provided by Citrix as part of default Optimizer build. If you would like to use your own templates with auto-select, you can override the default templates prefix. 
    For example if your template is called My_Windows_10_1809.xml, use '-TemplatePrefix "My_Windows"' to automatically select your templates based on current operating system and build.

.PARAMETER Mode
    CTXOE supports three different modes:
        Analyze - Do not apply any changes, only show the recommended changes.
        Execute - Apply the changes to the operating system.
        Rollback - Revert the applied changes. Requires a valid XML backup from the previously run Execute phase. This file is usually called Execute_History.xml.

    WARNING: Rollback mode cannot restore applications that have been removed. If you are planning to remove UWP applications and want to be able to recover them, use snapshots instead of rollback mode.

.PARAMETER IgnoreConditions
    When you use -IgnoreConditions switch, all conditions are skipped and optimizations are applied without any environments tests. This is used mostly for troubleshooting and is not recommended for normal environments.

.PARAMETER Groups
    Array that allows you to specify which groups to process from a specified source file.

.PARAMETER OutputLogFolder
    The location where to save all generated log files. This will replace an automatically generated folder .\Logs and is typically used with ESD solutions like SCCM.

.PARAMETER OutputXml
    The location where the output XML should be saved. The XML with results is automatically saved under .\Logs folder, but you can optionally specify also other location. This argument can be used together with -OutputHtml.

.PARAMETER OutputHtml
    The location where the output HTML report should be saved. The HTML with results is automatically saved under .\Logs folder, but you can optionally specify another location. This argument can be used together with -OutputXml.

.PARAMETER OptimizerUI
    Parameter used by Citrix Optimizer Tool UI to retrieve information from optimization engine. For internal use only.

.EXAMPLE
    .\CtxOptimizerEngine.ps1 -Source C:\Temp\Win10.xml -Mode Analyze
    Process all entries in Win10.xml file and display the recommended changes. Changes are not applied to the system.

.EXAMPLE
    .\CtxOptimizerEngine.ps1 -Source C:\Temp\Win10.xml -Mode Execute
    Process all entries from Win10.xml file. These changes are applied to the operating system.

.EXAMPLE
    .\CtxOptimizerEngine.ps1 -Source C:\Temp\Win10.xml -Mode Execute -Groups "DisableServices", "RemoveApplications"
    Process entries from groups "Disable Services" and "Remove built-in applications" in Win10.xml file. These changes are applied to the operating system.

.EXAMPLE
    .\CtxOptimizerEngine.ps1 -Source C:\Temp\Win10.xml -Mode Execute -OutputXml C:\Temp\Rollback.xml
    Process all entries from Win10.xml file. These changes are applied to the operating system. Save the rollback instructions in the file rollback.xml.

.EXAMPLE
    .\CtxOptimizerEngine.ps1 -Source C:\Temp\Rollback.xml -Mode Rollback
    Revert all changes from the file rollback.xml.

.NOTES
    Author: Martin Zugec
    Date:   February 17, 2017

.LINK
    https://support.citrix.com/article/CTX224676
#>

#Requires -Version 2

Param (
    [Alias("Template")]
    [System.String]$Source,

    [ValidateSet('analyze','execute','rollback')]

    [System.String]$Mode = "Analyze",

    [Array]$Groups,

    [String]$OutputLogFolder,

    [String]$OutputHtml,

    [String]$OutputXml,

    [Switch]$OptimizerUI,

    [Switch]$IgnoreConditions,

    [String]$TemplatePrefix
)

[String]$EngineVersion = "2.9";
# Retrieve friendly OS name (e.g. Winodws 10 Pro)
[String]$m_OSName = (Get-WmiObject Win32_OperatingSystem).Caption;
# If available, retrieve a build number (yymm like 1808). This is used on Windows Server 2016 and Windows 10, but is not used on older operating systems and is optional
[String]$m_OSBuild = $(Get-ItemProperty 'HKLM:\SOFTWARE\Microsoft\Windows NT\CurrentVersion' -Name ReleaseID -ErrorAction SilentlyContinue | Select-Object -ExpandProperty ReleaseID);

Write-Host "------------------------------"
Write-Host "| Citrix Optimization Engine |"
Write-Host "| Version $EngineVersion                |"
Write-Host "------------------------------"
Write-Host

Write-Host "Running in " -NoNewline
Write-Host -ForegroundColor Yellow $Mode -NoNewLine
Write-Host " mode"

# Error handling. We want Citrix Optimizer Tool to abort on any error, so error action preference is set to "Stop".
# The problem with this approach is that if Optimizer is called from another script, "Stop" instruction will apply to that script as well, so failure in Optimizer engine will abort calling script(s).
# As a workaround, instead of terminating the script, Optimizer has a global error handling procedure that will restore previous setting of ErrorActionPreference and properly abort the execution.
$OriginalErrorActionPreferenceValue = $ErrorActionPreference;
$ErrorActionPreference = "Stop";

Trap {
    Write-Host "Citrix Optimizer Tool engine has encountered a problem and will now terminate";
    $ErrorActionPreference = $OriginalErrorActionPreferenceValue;
    Write-Error $_;

    # Update $Run_Status with error encountered and save output XML file.
    If ($Run_Status) {
        $Run_Status.run_successful = $False.ToString();
        $Run_Status.run_details = "Error: $_";
        $Run_Status.time_end = [DateTime]::Now.ToString('yyyy-MM-dd_HH-mm-ss') # Saving DateTime in predefined format. This is required, since we don't know the target localization settings and want to make sure that UI and engine can communicate in same language.
        $PackDefinitionXml.Save($ResultsXml);
    }

    Return $False;
}

# Create enumeration for PluginMode. Enumeration cannot be used in the param() section, as that would require a DynamicParam on a script level.
[String]$PluginMode = $Mode;

# Just in case if previous run failed, make sure that all modules are reloaded
Remove-Module CTXOE*;

# Create $CTXOE_Main variable that defines folder where the script is located. If code is executed manually (copy & paste to PowerShell window), current directory is being used
If ($MyInvocation.MyCommand.Path -is [Object]) {
    [string]$Global:CTXOE_Main = $(Split-Path -Parent $MyInvocation.MyCommand.Path);
} Else {
    [string]$Global:CTXOE_Main = $(Get-Location).Path;
}

# Create Logs folder if it doesn't exists
If ($OutputLogFolder.Length -eq 0) {
    $Global:CTXOE_LogFolder = "$CTXOE_Main\Logs\$([DateTime]::Now.ToString('yyyy-MM-dd_HH-mm-ss'))"
} Else {
    $Global:CTXOE_LogFolder = $OutputLogFolder;
}

If ($(Test-Path "$CTXOE_LogFolder") -eq $false) {
    Write-Host "Creating Logs folder $(Split-Path -Leaf $CTXOE_LogFolder)"
    MkDir $CTXOE_LogFolder | Out-Null
}

# Report the location of log folder to UI
If ($OptimizerUI) {
    $LogFolder = New-Object -TypeName PSObject
    $LogFolder.PSObject.TypeNames.Insert(0,"logfolder")
    $LogFolder | Add-Member -MemberType NoteProperty -Name Location -Value $CTXOE_LogFolder
    Write-Output $LogFolder
}

# Initialize debug log (transcript). PowerShell ISE doesn't support transcriptions at the moment.
# Previously, we tried to determine if current host supports transcription or not, however this functionality is broken since PowerShell 4.0. Using Try/Catch instead.
Write-Host "Starting session log"
Try {
    $CTXOE_DebugLog = "$CTXOE_LogFolder\Log_Debug_CTXOE.log"
    Start-Transcript -Append -Path "$CTXOE_DebugLog" | Out-Null
} Catch { Write-Host "An exception happened when starting transcription: $_" -ForegroundColor Red }

# Check if user is administrator
Write-Host "Checking permissions"
If (-not ([Security.Principal.WindowsPrincipal] [Security.Principal.WindowsIdentity]::GetCurrent()).IsInRole([Security.Principal.WindowsBuiltInRole] "Administrator")) {
    Throw "You must be administrator in order to execute this script"
}

# Check if template name has been provided. If not, try to detect proper template automatically
If ($Source.Length -eq 0 -or $Source -eq "AutoSelect") {
    Write-Host "Template not specified, turning on auto-select mode";
    
    # Multiple template prefixes can be used - users can have their own custom templates.
    [array]$m_TemplatePrefixes = @();
    If ($TemplatePrefix.Length -gt 0) {
        Write-Host "Custom template prefix detected: $TemplatePrefix";
        $m_TemplatePrefixes += $TemplatePrefix;
    }
    $m_TemplatePrefixes += "Citrix_Windows";
    
    # Strip the description, keep only numbers. Special processing is required to include "R2" versions. Result of this regex is friendly version number (7, 10 or '2008 R2' for example)
    [String]$m_TemplateNameOSVersion = $([regex]"([0-9])+\sR([0-9])+|[(0-9)]+").Match($m_OSName).Captures[0].Value.Replace(" ", "");
    
    # Go through all available template prefixes, starting with custom prefix. Default Citrix prefix is used as a last option
    ForEach ($m_TemplateNamePrefix in $m_TemplatePrefixes) {

        Write-Host "Trying to find matching templates for prefix $m_TemplateNamePrefix"

        # If this is server OS, include "Server" in the template name. If this is client, don't do anything. While we could include _Client in the template name, it just looks weird.
        If ((Get-ItemProperty 'HKLM:\SOFTWARE\Microsoft\Windows NT\CurrentVersion' -Name InstallationType).InstallationType -eq "Server") {
            $m_TemplateNamePrefix += "_Server";
            If ($TemplatePrefix.Length -gt 0) {$m_TemplateNameCustomPrefix += "_Server";}
        }

        # First, we try to find if template for current OS and build is available. If not, we tried to find last build for the same OS version. If that is not available, we finally check for generic version of template (not build specific)
        If (Test-Path -Path "$CTXOE_Main\Templates\$(($m_TemplateNamePrefix) + '_' + ($m_TemplateNameOSVersion) + '_' + ($m_OSBuild)).xml") {
            Write-Host "Template detected - using optimal template for current version and build";
            $Source = "$CTXOE_Main\Templates\$(($m_TemplateNamePrefix) + '_' + ($m_TemplateNameOSVersion) + '_' + ($m_OSBuild)).xml";
            Break;
        } Else {
            Write-Host "Preferred template $(($m_TemplateNamePrefix) + '_' + ($m_TemplateNameOSVersion) + '_' + ($m_OSBuild)) was not found, using fallback mode"
            [array]$m_PreviousBuilds = Get-ChildItem -Path "$CTXOE_Main\Templates" -Filter $($m_TemplateNamePrefix + '_' + ($m_TemplateNameOSVersion) + '_*');
            # Older versions of PowerShell (V2) will automatically delect object instead of initiating an empty array.
            If ($m_PreviousBuilds -isnot [Object] -or $m_PreviousBuilds.Count -eq 0) {
                If (Test-Path -Path "$CTXOE_Main\Templates\$(($m_TemplateNamePrefix) + '_' + ($m_TemplateNameOSVersion)).xml") {
                    Write-Host "Template detected - using generic template for OS version";
                    $Source = "$CTXOE_Main\Templates\$(($m_TemplateNamePrefix) + '_' + ($m_TemplateNameOSVersion)).xml";
                    Break;
                }
            } Else {
                Write-Host "Template detected - using previous OS build";
                $Source = "$CTXOE_Main\Templates\$($m_PreviousBuilds | Sort-Object Name | Select-Object -ExpandProperty Name -Last 1)";
                Break;
            }
        }
    
    }

    If ($Source.Length -eq 0 -or $Source -eq "AutoSelect")  {Throw "Auto-detection of template failed, no suitable template has been found"}

}

# Check if -Source is a fullpath or just name of the template. If it's just the name, expand to a fullpath.
If (-not $Source.Contains("\")) {
    If (-not $Source.ToLower().EndsWith(".xml")) {
         $Source = "$Source.xml";
    }

    $Source = "$CTXOE_Main\Templates\$Source";
}

# Specify the default location of output XML
[String]$ResultsXml = "$CTXOE_LogFolder\$($PluginMode)_History.xml"
If ($OutputHtml.Length -eq 0) {
    [String]$OutputHtml = "$CTXOE_LogFolder\$($PluginMode)_History.html"
}

Write-Host
Write-Host "Processing definition file $Source"
[Xml]$PackDefinitionXml = Get-Content $Source

# Try to find if this template has been executed before. If <runstatus /> is present, move it to history (<previousruns />). This is used to store all previous executions of this template.
If ($PackDefinitionXml.root.run_status) {
    # Check if <previousruns /> exists. If not, create a new one.
    If (-not $PackDefinitionXml.root.previousruns) {
        $PackDefinitionXml.root.AppendChild($PackDefinitionXml.CreateElement("previousruns")) | Out-Null;
    }

    $PackDefinitionXml.root.Item("previousruns").AppendChild($PackDefinitionXml.root.run_status) | Out-Null;
}

# Create new XML element to store status of the execution.
[System.Xml.XmlElement]$Run_Status = $PackDefinitionXml.root.AppendChild($PackDefinitionXml.ImportNode($([Xml]"<run_status><run_mode /><time_start /><time_end /><entries_total /><entries_success /><entries_failed /><run_successful /><run_details /><engineversion /><targetos /><targetcomputer /></run_status>").DocumentElement, $True));
$Run_Status.run_successful = $False.ToString();
$Run_Status.run_mode = $PluginMode;
$Run_Status_Default_Message = "Run started, but never finished";
$Run_Status.run_details = $Run_Status_Default_Message;
$Run_Status.time_start = [DateTime]::Now.ToString('yyyy-MM-dd_HH-mm-ss') # Saving DateTime in predefined format. This is required, since we don't know the target localization settings and want to make sure that UI and engine can communicate in same language.
$Run_Status.engineversion = $EngineVersion;
$Run_Status.targetcomputer =  $Env:ComputerName;

If ($m_OSBuild.Length -gt 0) {
    $Run_Status.targetos = $m_OSName + " build " + $m_OSBuild;
} Else {
    $Run_Status.targetos = $m_OSName;
}

$PackDefinitionXml.Save($ResultsXml);

# Create new variables for counting of successful/failed/skipped entries execution. This is used in run_status reporting.
$Run_Status.entries_total = $PackDefinitionXml.SelectNodes("//entry").Count.ToString();
[Int]$Run_Status_Success = 0;
[Int]$Run_Status_Failed = 0;

# Add CTXOE modules to PSModulePath variable. With this modules can be loaded dynamically based on the prefix.
Write-Host "Adding CTXOE modules"
$Global:CTXOE_Modules = "$CTXOE_Main\Modules"
$Env:PSModulePath = "$([Environment]::GetEnvironmentVariable("PSModulePath"));$($Global:CTXOE_Modules)"

# Older version of PowerShell cannot load modules on-demand. All modules are pre-loaded.
If ($Host.Version.Major -le 2) {
    Write-Host "Detected older version of PowerShell. Importing all modules manually."
    ForEach ($m_Module in $(Get-ChildItem -Path "$CTXOE_Main\Modules" -Recurse -Filter "*.psm1")) {
        Import-Module -Name $m_Module.FullName
    }
}

# If mode is rollback, check if definition file contains the required history elements
If ($PluginMode -eq "Rollback") {
    If ($PackDefinitionXml.SelectNodes("//rollbackparams").Count -eq 0) {
        Throw "You need to select a log file from execution for rollback. This is usually called execute_history.xml. The file specified doesn't include instructions for rollback"
    }
}

# Display metadata for selected template. This acts as a header information about template
$PackDefinitionXml.root.metadata.ChildNodes | Select-Object Name, InnerText | Format-Table -HideTableHeaders

# First version of templates organized groups in packs. This was never really used and < pack/> element was removed in schema version 2.0
# This code is used for backwards compatibility with older templates
If ($PackDefinitionXml.root.pack -is [System.Xml.XmlElement]) {
    Write-host "Old template format has been detected, you should migrate to newer format" -for Red;
    $GroupElements = $PackDefinitionXml.SelectNodes("/root/pack/group");
} Else {
    $GroupElements = $PackDefinitionXml.SelectNodes("/root/group");
}

# Check if template has any conditions to process. In rollback mode, we do not need to process conditions - they've been already resolved to $True in execute mode and we should be able to rollback all changes.
If ($PluginMode -ne "rollback" -and -not $IgnoreConditions -and $PackDefinitionXml.root.condition -is [Object]) {
    Write-Host
    Write-Host "Template condition detected"
    [Hashtable]$m_TemplateConditionResult = CTXOE\Test-CTXOECondition -Element $PackDefinitionXml.root.condition; 
    Write-Host "Template condition result: $($m_TemplateConditionResult.Result)"
    Write-Host "Template condition details: $($m_TemplateConditionResult.Details)"
    Write-Host
    If ($m_TemplateConditionResult.Result -eq $False) {
        $Run_Status.run_details = "Execution stopped by template condition: $($m_TemplateConditionResult.Details)";
    }
}

# Check if template supports requested mode. If not, abort execution and throw an error
If ($PackDefinitionXml.root.metadata.SelectSingleNode("$($PluginMode.ToLower())_not_supported") -is [System.Xml.XmlElement]) {
    $Run_Status.run_details = "This template does NOT support requested mode $($PluginMode)!";
}


# Process template
ForEach ($m_Group in $GroupElements) {
    Write-Host
    Write-Host "        Group: $($m_Group.DisplayName)"
    Write-Host "        Group ID: $($m_Group.ID)"

    # Proceed only if the current run status message has NOT been modified. This is used to detect scenarios where template is reporting that run should not proceed, e.g. when conditions are used or mode is unsupported.
    If ($Run_Status.run_details -ne $Run_Status_Default_Message) {
        Write-Host "        Template processing failed, skipping"
        Continue
    }

    If ($Groups.Count -gt 0 -and $Groups -notcontains $m_Group.ID) {
        Write-Host "        Group not included in the -Groups argument, skipping"
        Continue
    }

    If ($m_Group.Enabled -eq "0") {
        Write-Host "    This group is disabled, skipping" -ForegroundColor DarkGray
        Continue
    }
    
    # Check if group supports requested mode. If not, move to next group
    [Boolean]$m_GroupModeNotSupported = $m_Group.SelectSingleNode("$($PluginMode.ToLower())_not_supported") -is [System.Xml.XmlElement]

    If ($m_GroupModeNotSupported) {
        Write-Host "    This group does not support $($PluginMode.ToLower()) mode, skipping" -ForegroundColor DarkGray
    }

    # PowerShell does not have concept of loop scope. We need to clear all variables from previous group before we process next group.
    Remove-Variable m_GroupConditionResult -ErrorAction SilentlyContinue;

    # Check if group has any conditions to process. 
    If ($PluginMode -ne "rollback" -and -not $IgnoreConditions -and $m_Group.condition -is [Object]) {
        Write-Host
        Write-Host "        Group condition detected"
        [Hashtable]$m_GroupConditionResult = CTXOE\Test-CTXOECondition -Element $m_Group.condition; 
        Write-Host "        Group condition result: $($m_GroupConditionResult.Result)"
        Write-Host "        Group condition details: $($m_GroupConditionResult.Details)"
        Write-Host
    }

    ForEach ($m_Entry in $m_Group.SelectNodes("./entry")) {
        Write-Host "            $($m_Entry.Name) - " -NoNewline

        If ($m_Entry.Enabled -eq "0") {
            Write-Host "    This entry is disabled, skipping" -ForegroundColor DarkGray
            CTXOE\New-CTXOEHistoryElement -Element $m_Entry -SystemChanged $False -StartTime $([DateTime]::Now) -Result $False -Details "Entry is disabled"
            Continue
        }

        If ($m_Entry.Execute -eq "0") {
            Write-Host " Entry is not marked for execution, skipping" -ForegroundColor DarkGray
            CTXOE\New-CTXOEHistoryElement -Element $m_Entry -SystemChanged $False -StartTime $([DateTime]::Now) -Result $False -Details "Entry is not marked for execution, skipping"
            Continue
        }

        # Check if entry supports requested mode. If not, move to next entry. If parent group does not support this mode, all entries should be skipped.
        # We need to make sure that ALL entries that are skipped have "Execute" set to 0 - otherwise status summary will fail to properly determine if script run was successful or not
        If (($m_GroupModeNotSupported -eq $True) -or ($m_Entry.SelectSingleNode("$($PluginMode.ToLower())_not_supported") -is [System.Xml.XmlElement])) {
            Write-Host "    This entry does not support $($PluginMode.ToLower()) mode, skipping" -ForegroundColor DarkGray
            CTXOE\New-CTXOEHistoryElement -Element $m_Entry -SystemChanged $False -StartTime $([DateTime]::Now) -Result $False -Details "This entry does not support $($PluginMode.ToLower()) mode, skipping"
            $m_Entry.Execute = "0";
            Continue
        }

        # Check if entry supports requested mode. If not, move to next entry
        If ($m_Entry.SelectSingleNode("$($PluginMode.ToLower())_not_supported") -is [System.Xml.XmlElement]) {
            Write-Host "    This entry does not support $($PluginMode.ToLower()) mode, skipping" -ForegroundColor DarkGray
            CTXOE\New-CTXOEHistoryElement -Element $m_Entry -SystemChanged $False -StartTime $([DateTime]::Now) -Result $False -Details "This entry does not support $($PluginMode.ToLower()) mode, skipping"

            Continue
        }

        # Section to process (parent) group conditions and entry conditions. Can be skipped if -IgnoreConditions is used or in rollback mode
        If ($PluginMode -ne "Rollback" -and -not $IgnoreConditions ) {
            # Check if the group condition has failed. If yes, none of the entries should be processed
            If ($m_GroupConditionResult -is [object] -and $m_GroupConditionResult.Result -eq $False) {
                Write-Host "    This entry is disabled by group condition, skipping" -ForegroundColor DarkGray;
                CTXOE\New-CTXOEHistoryElement -Element $m_Entry -SystemChanged $False -StartTime $([DateTime]::Now) -Result $False -Details "FILTERED: $($m_GroupConditionResult.Details)";
                $m_Entry.Execute = "0";
                Continue
            }

            # PowerShell does not have concept of loop scope. We need to clear all variables from previous group before we process next group.
            Remove-Variable m_ItemConditionResult -ErrorAction SilentlyContinue;

            # Check if this item has any conditions to process. 
            If ($m_Entry.condition -is [Object]) {
                Write-Host
                Write-Host "            Entry condition detected"
                [Hashtable]$m_ItemConditionResult = CTXOE\Test-CTXOECondition -Element $m_Entry.condition; 
                Write-Host "            Entry condition result: $($m_ItemConditionResult.Result)"
                Write-Host "            Entry condition details: $($m_ItemConditionResult.Details)"
                Write-Host
                
                If ($m_ItemConditionResult.Result -eq $False) {
                    Write-Host "    This entry is disabled by condition, skipping" -ForegroundColor DarkGray;
                    CTXOE\New-CTXOEHistoryElement -Element $m_Entry -SystemChanged $False -StartTime $([DateTime]::Now) -Result $False -Details "FILTERED: $($m_ItemConditionResult.Details)";
                    $m_Entry.Execute = "0";
                    Continue;
                }
            }
        }

        $m_Action = $m_Entry.SelectSingleNode("./action")
        Write-Verbose "            Plugin: $($m_Action.Plugin)"

        # While some plugins can use only a single set of instructions to perform all the different operations (typically services or registry keys), this might not be always possible.

        # Good example is "PowerShell" plugin - different code can be used to analyze the action and execute the action (compare "Get-CurrentState -eq $True" for analyze to "Set-CurrentState -Mode Example -Setup Mode1" for execute mode).

        # In order to support this scenarios, it is possible to override the default <params /> element with a custom element for analyze and rollback phases. Default is still <params />. With this implementation, there can be an action that will implement all three elements (analyzeparams, rollbackparams and executeparams).

        [String]$m_ParamsElementName = "params"
        [String]$m_OverrideElement = "$($PluginMode.ToLower())$m_ParamsElementName"

        If ($m_Action.$m_OverrideElement -is [Object]) {
            Write-Verbose "Using custom <$($m_OverrideElement) /> element"
            $m_ParamsElementName = $m_OverrideElement
        }

        # To prevent any unexpected damage to the system, Rollback mode requires use of custom params object and cannot use the default one.
        If ($PluginMode -eq "Rollback" -and $m_Action.$m_OverrideElement -isnot [Object]) {
            If ($m_Entry.history.systemchanged -eq "0") {
                Write-Host "This entry has not changed, skip" -ForegroundColor DarkGray
                Continue
            } Else {
                Write-Host "Rollback mode requires custom instructions that are not available, skip" -ForegroundColor DarkGray
                Continue
            }
        }

        # Reset variables that are used to report the status
        [Boolean]$Global:CTXOE_Result = $False;
        $Global:CTXOE_Details = "No data returned by this entry (this is unexpected)";

        # Two variables used by rollback. First identify that this entry has modified the system. The second should contain information required for rollback of those changes (if possible). This is required only for "execute" mode.
        [Boolean]$Global:CTXOE_SystemChanged = $False

        $Global:CTXOE_ChangeRollbackParams = $Null

        [DateTime]$StartTime = Get-Date;
        CTXOE\Invoke-CTXOEPlugin -PluginName $($m_Action.Plugin) -Params $m_Action.$m_ParamsElementName -Mode $PluginMode -Verbose

        # Test if there is custom details message for current mode or general custom message. This allows you to display friendly message instead of generic error.
        # This can be either mode-specific or generic (message_analyze_true or message_true). Last token (true/false) is used to identify if custom message should be displayed for success or failure
        # If custom message is detected, output from previous function is ignored and CTXOE_Details is replaced with custom text
        [string]$m_OverrideOutputMessageMode = "message_$($PluginMode.ToLower())_$($Global:CTXOE_Result.ToString().ToLower())";
        [string]$m_OverrideOutputMessageGeneric = "message_$($Global:CTXOE_Result.ToString().ToLower())";

        If ($m_Entry.$m_OverrideOutputMessageMode -is [Object]) {
            $Global:CTXOE_Details = $($m_Entry.$m_OverrideOutputMessageMode);
        } ElseIf ($m_Entry.$m_OverrideOutputMessageGeneric -is [Object]) {
            $Global:CTXOE_Details = $($m_Entry.$m_OverrideOutputMessageGeneric);
        }

		# This code is added to have a situation where CTXOE_Result is set, but not to boolean value (for example to empty string). This will prevent engine from crashing and report which entry does not behave as expected.
		# We do this check here so following code does not need to check if returned value exists
		If ($Global:CTXOE_Result -isnot [Boolean]) {
			$Global:CTXOE_Result = $false;
			$Global:CTXOE_Details = "While processing $($m_Entry.Name) from group $($m_Group.ID), there was an error or code did not return expected result. This value should be boolean, while returned value is $($Global:CTXOE_Result.GetType().FullName)."; 
		}

        If ($Global:CTXOE_Result -eq $false) {
            $Run_Status_Failed += 1;
            Write-Host -ForegroundColor Red $CTXOE_Details
        } Else {
            $Run_Status_Success += 1;
            Write-Host -ForegroundColor Green $CTXOE_Details
        }

        # Save information about changes as an element
        CTXOE\New-CTXOEHistoryElement -Element $m_Entry -SystemChanged $CTXOE_SystemChanged -StartTime $StartTime -Result $CTXOE_Result -Details $CTXOE_Details -RollbackInstructions $CTXOE_ChangeRollbackParams

        If ($OptimizerUI) {
            $history = New-Object -TypeName PSObject
            $history.PSObject.TypeNames.Insert(0,"history")
            $history | Add-Member -MemberType NoteProperty -Name GroupID -Value $m_Group.ID
            $history | Add-Member -MemberType NoteProperty -Name EntryName -Value $m_Entry.Name
            $history | Add-Member -MemberType NoteProperty -Name SystemChanged -Value $m_Entry.SystemChanged
            $history | Add-Member -MemberType NoteProperty -Name StartTime -Value $m_Entry.History.StartTime
            $history | Add-Member -MemberType NoteProperty -Name EndTime -Value $m_Entry.History.EndTime
            $history | Add-Member -MemberType NoteProperty -Name Result -Value $m_Entry.History.Return.Result
            $history | Add-Member -MemberType NoteProperty -Name Details -Value $m_Entry.History.Return.Details

            Write-Output $history
        }
    }
}

#Region "Run status processing"
# Finish processing of run_status, save everything to return XML file
$Run_Status.time_end = [DateTime]::Now.ToString('yyyy-MM-dd_HH-mm-ss') # Saving DateTime in predefined format. This is required, since we don't know the target localization settings and want to make sure that UI and engine can communicate in same language.

$Run_Status.entries_success = $Run_Status_Success.ToString();
$Run_Status.entries_failed = $Run_Status_Failed.ToString();

# Run status should be determined ONLY if template has not aborted execution before.
If ($Run_Status.run_details -eq $Run_Status_Default_Message) {
    
    # Count all entries that were expected to execute (execute=1), but have not finished successfully (result!=1)
    [Int]$m_EntriesNotExecuted = $PackDefinitionXml.SelectNodes("//entry[execute=1 and not(history/return/result=1)]").Count

    # If we have entries that are not successful
    If ($m_EntriesNotExecuted -gt 0) {
        If ($m_EntriesNotExecuted -eq 1) {
            $Run_Status.run_details = "$m_EntriesNotExecuted entry has failed";
        } Else {
            $Run_Status.run_details = "$m_EntriesNotExecuted entries have failed";
        }
    # If anything is marked as failed
    } ElseIf ($Run_Status_Failed -gt 0) {
        If ($Run_Status_Failed -eq 1) {
            $Run_Status.run_details = "$Run_Status_Failed entry from this template failed";
        } Else {
            $Run_Status.run_details = "$Run_Status_Failed entries from this template failed";
        }
    # If nothing was actually executed
    } ElseIf ($Run_Status_Success -eq 0) {
        $Run_Status.run_details = "No entries from this template have been processed";
    # Nothing failed, something was successful = sounds good
    } ElseIf ($Run_Status_Success -gt 0 -and $Run_Status_Failed -eq 0) {
        $Run_Status.run_successful = $True.ToString();
        $Run_Status.run_details = "Template has been processed successfully";
    } Else {
        $Run_Status.run_details = "Unknown condition when evaluating run result";  
    }
}
#EndRegion

# Send the overall execute result for UI to show
If ($OptimizerUI) {
    $overallresult = New-Object -TypeName PSObject
    $overallresult.PSObject.TypeNames.Insert(0,"overallresult")
    $overallresult | Add-Member -MemberType NoteProperty -Name run_successful -Value $Run_Status.run_successful
    $overallresult | Add-Member -MemberType NoteProperty -Name run_details -Value $Run_Status.run_details
    $overallresult | Add-Member -MemberType NoteProperty -Name entries_success -Value $Run_Status.entries_success
    $overallresult | Add-Member -MemberType NoteProperty -Name entries_failed -Value $Run_Status.entries_failed

    Write-Output $overallresult
}
# end

# Save the output in XML format for further parsing\history
$PackDefinitionXml.Save($ResultsXml);

#Region "Registry status reporting"

# If mode is 'execute', then save registry status. If mode is 'rollback' (and registry status exists), remove it. No action required for 'analyze' mode

[String]$m_RegistryPath = "HKLM:\SOFTWARE\Citrix\Optimizer\" + $PackDefinitionXml.root.metadata.category;

If ($PluginMode -eq "execute") {
    # Check if registry key exists
    If ($(Test-Path $m_RegistryPath) -eq $False) {
        # If registry key doesn't exist, create it
        New-Item -Path $m_RegistryPath -Force | Out-Null;
    }

    # Save location of XML file that contains more details about execution
    New-ItemProperty -Path $m_RegistryPath -Name "log_path" -PropertyType "string" -Value $ResultsXml -Force | Out-Null;

    # Save all <metadata /> and <run_status />
    ForEach ($m_Node in $PackDefinitionXml.root.metadata.SelectNodes("*")) {
        New-ItemProperty -Path $m_RegistryPath -Name $m_Node.Name -PropertyType "string" -Value $m_Node.InnerText -Force | Out-Null;
    }
    ForEach ($m_Node in $PackDefinitionXml.root.run_status.SelectNodes("*")) {
        New-ItemProperty -Path $m_RegistryPath -Name $m_Node.Name -PropertyType "string" -Value $m_Node.InnerText -Force | Out-Null;
    }

} ElseIf ($PluginMode -eq "rollback") {
    # Check if registry key exists
    If ($(Test-Path $m_RegistryPath) -eq $True) {
        # If registry key exists, delete it
        Remove-Item -Path $m_RegistryPath -Force | Out-Null;
    }
}
#EndRegion

# Use transformation file to generate HTML report
$XSLT = New-Object System.Xml.Xsl.XslCompiledTransform;
$XSLT.Load("$CTXOE_Main\CtxOptimizerReport.xslt");
$XSLT.Transform($ResultsXml, $OutputHtml);

# If another location is requested, save the XML file here as well.
If ($OutputXml.Length -gt 0) {
    $PackDefinitionXml.Save($OutputXml);
}

# If the current host is transcribing, save the transcription
Try {
    Stop-Transcript | Out-Null
} Catch { Write-Host "An exception happened when stopping transcription: $_" -ForegroundColor Red }

# SIG # Begin signature block
# MIIoowYJKoZIhvcNAQcCoIIolDCCKJACAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCAEQ19w+DJ9tWYA
# rPE+d1bjxCgCYDnU1tMAtwlowHS5IKCCDcAwggawMIIEmKADAgECAhAIrUCyYNKc
# TJ9ezam9k67ZMA0GCSqGSIb3DQEBDAUAMGIxCzAJBgNVBAYTAlVTMRUwEwYDVQQK
# EwxEaWdpQ2VydCBJbmMxGTAXBgNVBAsTEHd3dy5kaWdpY2VydC5jb20xITAfBgNV
# BAMTGERpZ2lDZXJ0IFRydXN0ZWQgUm9vdCBHNDAeFw0yMTA0MjkwMDAwMDBaFw0z
# NjA0MjgyMzU5NTlaMGkxCzAJBgNVBAYTAlVTMRcwFQYDVQQKEw5EaWdpQ2VydCwg
# SW5jLjFBMD8GA1UEAxM4RGlnaUNlcnQgVHJ1c3RlZCBHNCBDb2RlIFNpZ25pbmcg
# UlNBNDA5NiBTSEEzODQgMjAyMSBDQTEwggIiMA0GCSqGSIb3DQEBAQUAA4ICDwAw
# ggIKAoICAQDVtC9C0CiteLdd1TlZG7GIQvUzjOs9gZdwxbvEhSYwn6SOaNhc9es0
# JAfhS0/TeEP0F9ce2vnS1WcaUk8OoVf8iJnBkcyBAz5NcCRks43iCH00fUyAVxJr
# Q5qZ8sU7H/Lvy0daE6ZMswEgJfMQ04uy+wjwiuCdCcBlp/qYgEk1hz1RGeiQIXhF
# LqGfLOEYwhrMxe6TSXBCMo/7xuoc82VokaJNTIIRSFJo3hC9FFdd6BgTZcV/sk+F
# LEikVoQ11vkunKoAFdE3/hoGlMJ8yOobMubKwvSnowMOdKWvObarYBLj6Na59zHh
# 3K3kGKDYwSNHR7OhD26jq22YBoMbt2pnLdK9RBqSEIGPsDsJ18ebMlrC/2pgVItJ
# wZPt4bRc4G/rJvmM1bL5OBDm6s6R9b7T+2+TYTRcvJNFKIM2KmYoX7BzzosmJQay
# g9Rc9hUZTO1i4F4z8ujo7AqnsAMrkbI2eb73rQgedaZlzLvjSFDzd5Ea/ttQokbI
# YViY9XwCFjyDKK05huzUtw1T0PhH5nUwjewwk3YUpltLXXRhTT8SkXbev1jLchAp
# QfDVxW0mdmgRQRNYmtwmKwH0iU1Z23jPgUo+QEdfyYFQc4UQIyFZYIpkVMHMIRro
# OBl8ZhzNeDhFMJlP/2NPTLuqDQhTQXxYPUez+rbsjDIJAsxsPAxWEQIDAQABo4IB
# WTCCAVUwEgYDVR0TAQH/BAgwBgEB/wIBADAdBgNVHQ4EFgQUaDfg67Y7+F8Rhvv+
# YXsIiGX0TkIwHwYDVR0jBBgwFoAU7NfjgtJxXWRM3y5nP+e6mK4cD08wDgYDVR0P
# AQH/BAQDAgGGMBMGA1UdJQQMMAoGCCsGAQUFBwMDMHcGCCsGAQUFBwEBBGswaTAk
# BggrBgEFBQcwAYYYaHR0cDovL29jc3AuZGlnaWNlcnQuY29tMEEGCCsGAQUFBzAC
# hjVodHRwOi8vY2FjZXJ0cy5kaWdpY2VydC5jb20vRGlnaUNlcnRUcnVzdGVkUm9v
# dEc0LmNydDBDBgNVHR8EPDA6MDigNqA0hjJodHRwOi8vY3JsMy5kaWdpY2VydC5j
# b20vRGlnaUNlcnRUcnVzdGVkUm9vdEc0LmNybDAcBgNVHSAEFTATMAcGBWeBDAED
# MAgGBmeBDAEEATANBgkqhkiG9w0BAQwFAAOCAgEAOiNEPY0Idu6PvDqZ01bgAhql
# +Eg08yy25nRm95RysQDKr2wwJxMSnpBEn0v9nqN8JtU3vDpdSG2V1T9J9Ce7FoFF
# UP2cvbaF4HZ+N3HLIvdaqpDP9ZNq4+sg0dVQeYiaiorBtr2hSBh+3NiAGhEZGM1h
# mYFW9snjdufE5BtfQ/g+lP92OT2e1JnPSt0o618moZVYSNUa/tcnP/2Q0XaG3Ryw
# YFzzDaju4ImhvTnhOE7abrs2nfvlIVNaw8rpavGiPttDuDPITzgUkpn13c5Ubdld
# AhQfQDN8A+KVssIhdXNSy0bYxDQcoqVLjc1vdjcshT8azibpGL6QB7BDf5WIIIJw
# 8MzK7/0pNVwfiThV9zeKiwmhywvpMRr/LhlcOXHhvpynCgbWJme3kuZOX956rEnP
# LqR0kq3bPKSchh/jwVYbKyP/j7XqiHtwa+aguv06P0WmxOgWkVKLQcBIhEuWTatE
# QOON8BUozu3xGFYHKi8QxAwIZDwzj64ojDzLj4gLDb879M4ee47vtevLt/B3E+bn
# KD+sEq6lLyJsQfmCXBVmzGwOysWGw/YmMwwHS6DTBwJqakAwSEs0qFEgu60bhQji
# WQ1tygVQK+pKHJ6l/aCnHwZ05/LWUpD9r4VIIflXO7ScA+2GRfS0YW6/aOImYIbq
# yK+p/pQd52MbOoZWeE4wggcIMIIE8KADAgECAhAExKVRXTiJBY6ZBfjtmlFwMA0G
# CSqGSIb3DQEBCwUAMGkxCzAJBgNVBAYTAlVTMRcwFQYDVQQKEw5EaWdpQ2VydCwg
# SW5jLjFBMD8GA1UEAxM4RGlnaUNlcnQgVHJ1c3RlZCBHNCBDb2RlIFNpZ25pbmcg
# UlNBNDA5NiBTSEEzODQgMjAyMSBDQTEwHhcNMjQwMzA4MDAwMDAwWhcNMjUwMzA3
# MjM1OTU5WjCBjzELMAkGA1UEBhMCVVMxEDAOBgNVBAgTB0Zsb3JpZGExGDAWBgNV
# BAcTD0ZvcnQgTGF1ZGVyZGFsZTEdMBsGA1UEChMUQ2l0cml4IFN5c3RlbXMsIElu
# Yy4xFjAUBgNVBAsTDUNpdHJpeCBYZW5BcHAxHTAbBgNVBAMTFENpdHJpeCBTeXN0
# ZW1zLCBJbmMuMIIBojANBgkqhkiG9w0BAQEFAAOCAY8AMIIBigKCAYEA+gJ84IK2
# t0aBgBuXB3yfqoC+riwIEohXTQHVqNvJ7v3xiiIYD8PvzYLhaQq2dG4euY1AOhXL
# 0Uc2rNLcQZKEB4PslYrrorg+x5iRJl2k9XltDEpDbqJJqkUELY6d7GEIpp4ChdGf
# NJuGU9GCAgiml88tzZMOjtAqcLIXSRSOugQgjUOR+EvfeLlMTLs/ADjyOi898+0m
# TtdE6Sy/a2bn/fgsJaU8O2Kn78YiHX1N6lklKKBzO5tYfRD2VZhOn2kx6PUI+n+R
# ckwE22o0hBhckBeBGNW2DUrABDLwYuNOEWZEGx1KC2UDO4p+eDmxANjC/+q9t53d
# Z4V1Y/qL9CRnin0JNowVcLT3zGqeebgIag2Pqsh6m3lvASiy7DpmDtyaTCwv/aLd
# 1x5TZvUPiKcfvMOSCRxTP+a7L4DdjuqcQxWENk1kNEZjZa/ZS0BDWalklp6Ji9ki
# JpUKxB5C6GD/Lns+hbdp/2XHnEsqGbeQBCiW7ftXO3ugffq+/Bz1CelPAgMBAAGj
# ggIDMIIB/zAfBgNVHSMEGDAWgBRoN+Drtjv4XxGG+/5hewiIZfROQjAdBgNVHQ4E
# FgQUh00+M0JEuzdBGq4PW6l3UIoBQsIwPgYDVR0gBDcwNTAzBgZngQwBBAEwKTAn
# BggrBgEFBQcCARYbaHR0cDovL3d3dy5kaWdpY2VydC5jb20vQ1BTMA4GA1UdDwEB
# /wQEAwIHgDATBgNVHSUEDDAKBggrBgEFBQcDAzCBtQYDVR0fBIGtMIGqMFOgUaBP
# hk1odHRwOi8vY3JsMy5kaWdpY2VydC5jb20vRGlnaUNlcnRUcnVzdGVkRzRDb2Rl
# U2lnbmluZ1JTQTQwOTZTSEEzODQyMDIxQ0ExLmNybDBToFGgT4ZNaHR0cDovL2Ny
# bDQuZGlnaWNlcnQuY29tL0RpZ2lDZXJ0VHJ1c3RlZEc0Q29kZVNpZ25pbmdSU0E0
# MDk2U0hBMzg0MjAyMUNBMS5jcmwwgZQGCCsGAQUFBwEBBIGHMIGEMCQGCCsGAQUF
# BzABhhhodHRwOi8vb2NzcC5kaWdpY2VydC5jb20wXAYIKwYBBQUHMAKGUGh0dHA6
# Ly9jYWNlcnRzLmRpZ2ljZXJ0LmNvbS9EaWdpQ2VydFRydXN0ZWRHNENvZGVTaWdu
# aW5nUlNBNDA5NlNIQTM4NDIwMjFDQTEuY3J0MAkGA1UdEwQCMAAwDQYJKoZIhvcN
# AQELBQADggIBAI1e2H/UCCob4N14i0h2Z1Yi7dInt7RICuwc9yiMcDeGvlvtgcJT
# zp/CyQwpDh58/WGxR0HX3TuLjLh0Foxzm6aSgTLnyvI65jugv24L7yu1Bl5n7WR1
# x4tqtvVI1DOCTWNN9BqokeBTuT7NVwDPyoLWxz8f3GqnERJ/xe4DJOv9RXre6sy0
# NF9a+d4oqvEJ5r03WOpI6jO3YG6hyxCF7611ZhVdvBuw0ZfVgXPH1n/l+kGb4Mdc
# s0kJ6ny/auQ3Cm0lpCxrjh+LxwrxYtk1s/iZMd9T+C5vjUzH1LWhqlPur0ur05yU
# cG61A8pPwKLE02FdW9b6miAPknpUI6o6EgdU1KRCCjtYoR1iBSagB75k0RZBcgor
# B6mQKcFF5EwRhw5rSJdn1r7rTjRgVTzgRjgHL7QBY27yshp9rTwqHzO9VUmfTxg1
# 4UT5dcZ3xGc8U4j+gMSjNXmJJEpRaNtuh1QDXoQCwC0fM/9r4BxIlS50nuWrhnBL
# zSbrdxbmDK+iW5UU2oCKs3zHbrPmqQgWd4CDi/g/kUEruRIL+HMS1eAdZlyZ8DCw
# H8Qwv6+T6we/iDDbHtxLy7TjGsYChqEgaVOT7gj++iIdoHq3+mb1bxtDgovELXFN
# EwOxXT5mt7qZlc4ApoUTdruNzS9aMofoIB2SIVo/P6FCryDE5YxVmv6zMYIaOTCC
# GjUCAQEwfTBpMQswCQYDVQQGEwJVUzEXMBUGA1UEChMORGlnaUNlcnQsIEluYy4x
# QTA/BgNVBAMTOERpZ2lDZXJ0IFRydXN0ZWQgRzQgQ29kZSBTaWduaW5nIFJTQTQw
# OTYgU0hBMzg0IDIwMjEgQ0ExAhAExKVRXTiJBY6ZBfjtmlFwMA0GCWCGSAFlAwQC
# AQUAoIHQMBkGCSqGSIb3DQEJAzEMBgorBgEEAYI3AgEEMBwGCisGAQQBgjcCAQsx
# DjAMBgorBgEEAYI3AgEVMC8GCSqGSIb3DQEJBDEiBCAeu5XMgOYfcmU2tR3obwEj
# E5ZMba3xf5wXVuOemV1zSzBkBgorBgEEAYI3AgEMMVYwVKA4gDYAQwBpAHQAcgBp
# AHgAIABTAHUAcABwAG8AcgB0AGEAYgBpAGwAaQB0AHkAIABUAG8AbwBsAHOhGIAW
# aHR0cDovL3d3dy5jaXRyaXguY29tIDANBgkqhkiG9w0BAQEFAASCAYBYQjcOIarD
# uisGU5WWHdkUjRO5tJzc9oU5D183Q+mjewrhPDMIoSjfMaD4XEiCnG1fpGh2Z6e4
# iwlwxsjAp2SUSzGnJSK3phEvRn3lw0zVOI6Q+WklgchsJKx7AvYG+nVI+Q6XIX6Z
# iSb6uSqRFYRbtc/PHcuJHZnOsAehNxwe7bWR1RpqI4S0jRMKSmqGAAUfstXIHHex
# bQxEPINEYJCfmvP85BGUnvRvFb0AhPOkzl8DJjyrRuqm6r33CV5jnqBOgYVz/w8g
# AireYrZtm0XOKeDGZNp3Lpf3Ay4wmrpHiM1tANUqxnXviiYzudQuF8pciUjQ5/Jb
# tE1u6etm2EjwpjwT3X4c0FxQgxxpLgeZwjjH+Std1KTwfybPUsoI1WWiFhygGQee
# oBHpxp7MNGg949ZWehKhIX8rZelltHErhNm29iTRuaAeq4Mf9a5TPFqPITQPJswt
# 85ug8lwwv2XnRXyzjWi+qhf4Gn2z8HYQWdS0BAeyfK3SqDpJhR4hj4Ohghc6MIIX
# NgYKKwYBBAGCNwMDATGCFyYwghciBgkqhkiG9w0BBwKgghcTMIIXDwIBAzEPMA0G
# CWCGSAFlAwQCAQUAMHgGCyqGSIb3DQEJEAEEoGkEZzBlAgEBBglghkgBhv1sBwEw
# MTANBglghkgBZQMEAgEFAAQgP14Tnh4jeNWA8cajLvYtcWQzQ4dS0H38Vc2KWE0+
# OBsCEQCkFdyq/opX5W6sYm3KNP91GA8yMDI0MTEyMjA1NTMyNFqgghMDMIIGvDCC
# BKSgAwIBAgIQC65mvFq6f5WHxvnpBOMzBDANBgkqhkiG9w0BAQsFADBjMQswCQYD
# VQQGEwJVUzEXMBUGA1UEChMORGlnaUNlcnQsIEluYy4xOzA5BgNVBAMTMkRpZ2lD
# ZXJ0IFRydXN0ZWQgRzQgUlNBNDA5NiBTSEEyNTYgVGltZVN0YW1waW5nIENBMB4X
# DTI0MDkyNjAwMDAwMFoXDTM1MTEyNTIzNTk1OVowQjELMAkGA1UEBhMCVVMxETAP
# BgNVBAoTCERpZ2lDZXJ0MSAwHgYDVQQDExdEaWdpQ2VydCBUaW1lc3RhbXAgMjAy
# NDCCAiIwDQYJKoZIhvcNAQEBBQADggIPADCCAgoCggIBAL5qc5/2lSGrljC6W23m
# WaO16P2RHxjEiDtqmeOlwf0KMCBDEr4IxHRGd7+L660x5XltSVhhK64zi9CeC9B6
# lUdXM0s71EOcRe8+CEJp+3R2O8oo76EO7o5tLuslxdr9Qq82aKcpA9O//X6QE+Ac
# aU/byaCagLD/GLoUb35SfWHh43rOH3bpLEx7pZ7avVnpUVmPvkxT8c2a2yC0WMp8
# hMu60tZR0ChaV76Nhnj37DEYTX9ReNZ8hIOYe4jl7/r419CvEYVIrH6sN00yx49b
# oUuumF9i2T8UuKGn9966fR5X6kgXj3o5WHhHVO+NBikDO0mlUh902wS/Eeh8F/UF
# aRp1z5SnROHwSJ+QQRZ1fisD8UTVDSupWJNstVkiqLq+ISTdEjJKGjVfIcsgA4l9
# cbk8Smlzddh4EfvFrpVNnes4c16Jidj5XiPVdsn5n10jxmGpxoMc6iPkoaDhi6Jj
# Hd5ibfdp5uzIXp4P0wXkgNs+CO/CacBqU0R4k+8h6gYldp4FCMgrXdKWfM4N0u25
# OEAuEa3JyidxW48jwBqIJqImd93NRxvd1aepSeNeREXAu2xUDEW8aqzFQDYmr9ZO
# Nuc2MhTMizchNULpUEoA6Vva7b1XCB+1rxvbKmLqfY/M/SdV6mwWTyeVy5Z/JkvM
# FpnQy5wR14GJcv6dQ4aEKOX5AgMBAAGjggGLMIIBhzAOBgNVHQ8BAf8EBAMCB4Aw
# DAYDVR0TAQH/BAIwADAWBgNVHSUBAf8EDDAKBggrBgEFBQcDCDAgBgNVHSAEGTAX
# MAgGBmeBDAEEAjALBglghkgBhv1sBwEwHwYDVR0jBBgwFoAUuhbZbU2FL3Mpdpov
# dYxqII+eyG8wHQYDVR0OBBYEFJ9XLAN3DigVkGalY17uT5IfdqBbMFoGA1UdHwRT
# MFEwT6BNoEuGSWh0dHA6Ly9jcmwzLmRpZ2ljZXJ0LmNvbS9EaWdpQ2VydFRydXN0
# ZWRHNFJTQTQwOTZTSEEyNTZUaW1lU3RhbXBpbmdDQS5jcmwwgZAGCCsGAQUFBwEB
# BIGDMIGAMCQGCCsGAQUFBzABhhhodHRwOi8vb2NzcC5kaWdpY2VydC5jb20wWAYI
# KwYBBQUHMAKGTGh0dHA6Ly9jYWNlcnRzLmRpZ2ljZXJ0LmNvbS9EaWdpQ2VydFRy
# dXN0ZWRHNFJTQTQwOTZTSEEyNTZUaW1lU3RhbXBpbmdDQS5jcnQwDQYJKoZIhvcN
# AQELBQADggIBAD2tHh92mVvjOIQSR9lDkfYR25tOCB3RKE/P09x7gUsmXqt40ouR
# l3lj+8QioVYq3igpwrPvBmZdrlWBb0HvqT00nFSXgmUrDKNSQqGTdpjHsPy+Laal
# TW0qVjvUBhcHzBMutB6HzeledbDCzFzUy34VarPnvIWrqVogK0qM8gJhh/+qDEAI
# dO/KkYesLyTVOoJ4eTq7gj9UFAL1UruJKlTnCVaM2UeUUW/8z3fvjxhN6hdT98Vr
# 2FYlCS7Mbb4Hv5swO+aAXxWUm3WpByXtgVQxiBlTVYzqfLDbe9PpBKDBfk+rabTF
# DZXoUke7zPgtd7/fvWTlCs30VAGEsshJmLbJ6ZbQ/xll/HjO9JbNVekBv2Tgem+m
# LptR7yIrpaidRJXrI+UzB6vAlk/8a1u7cIqV0yef4uaZFORNekUgQHTqddmsPCEI
# YQP7xGxZBIhdmm4bhYsVA6G2WgNFYagLDBzpmk9104WQzYuVNsxyoVLObhx3Ruga
# EGru+SojW4dHPoWrUhftNpFC5H7QEY7MhKRyrBe7ucykW7eaCuWBsBb4HOKRFVDc
# rZgdwaSIqMDiCLg4D+TPVgKx2EgEdeoHNHT9l3ZDBD+XgbF+23/zBjeCtxz+dL/9
# NWR6P2eZRi7zcEO1xwcdcqJsyz/JceENc2Sg8h3KeFUCS7tpFk7CrDqkMIIGrjCC
# BJagAwIBAgIQBzY3tyRUfNhHrP0oZipeWzANBgkqhkiG9w0BAQsFADBiMQswCQYD
# VQQGEwJVUzEVMBMGA1UEChMMRGlnaUNlcnQgSW5jMRkwFwYDVQQLExB3d3cuZGln
# aWNlcnQuY29tMSEwHwYDVQQDExhEaWdpQ2VydCBUcnVzdGVkIFJvb3QgRzQwHhcN
# MjIwMzIzMDAwMDAwWhcNMzcwMzIyMjM1OTU5WjBjMQswCQYDVQQGEwJVUzEXMBUG
# A1UEChMORGlnaUNlcnQsIEluYy4xOzA5BgNVBAMTMkRpZ2lDZXJ0IFRydXN0ZWQg
# RzQgUlNBNDA5NiBTSEEyNTYgVGltZVN0YW1waW5nIENBMIICIjANBgkqhkiG9w0B
# AQEFAAOCAg8AMIICCgKCAgEAxoY1BkmzwT1ySVFVxyUDxPKRN6mXUaHW0oPRnkyi
# baCwzIP5WvYRoUQVQl+kiPNo+n3znIkLf50fng8zH1ATCyZzlm34V6gCff1DtITa
# EfFzsbPuK4CEiiIY3+vaPcQXf6sZKz5C3GeO6lE98NZW1OcoLevTsbV15x8GZY2U
# KdPZ7Gnf2ZCHRgB720RBidx8ald68Dd5n12sy+iEZLRS8nZH92GDGd1ftFQLIWhu
# NyG7QKxfst5Kfc71ORJn7w6lY2zkpsUdzTYNXNXmG6jBZHRAp8ByxbpOH7G1WE15
# /tePc5OsLDnipUjW8LAxE6lXKZYnLvWHpo9OdhVVJnCYJn+gGkcgQ+NDY4B7dW4n
# JZCYOjgRs/b2nuY7W+yB3iIU2YIqx5K/oN7jPqJz+ucfWmyU8lKVEStYdEAoq3ND
# zt9KoRxrOMUp88qqlnNCaJ+2RrOdOqPVA+C/8KI8ykLcGEh/FDTP0kyr75s9/g64
# ZCr6dSgkQe1CvwWcZklSUPRR8zZJTYsg0ixXNXkrqPNFYLwjjVj33GHek/45wPmy
# MKVM1+mYSlg+0wOI/rOP015LdhJRk8mMDDtbiiKowSYI+RQQEgN9XyO7ZONj4Kbh
# PvbCdLI/Hgl27KtdRnXiYKNYCQEoAA6EVO7O6V3IXjASvUaetdN2udIOa5kM0jO0
# zbECAwEAAaOCAV0wggFZMBIGA1UdEwEB/wQIMAYBAf8CAQAwHQYDVR0OBBYEFLoW
# 2W1NhS9zKXaaL3WMaiCPnshvMB8GA1UdIwQYMBaAFOzX44LScV1kTN8uZz/nupiu
# HA9PMA4GA1UdDwEB/wQEAwIBhjATBgNVHSUEDDAKBggrBgEFBQcDCDB3BggrBgEF
# BQcBAQRrMGkwJAYIKwYBBQUHMAGGGGh0dHA6Ly9vY3NwLmRpZ2ljZXJ0LmNvbTBB
# BggrBgEFBQcwAoY1aHR0cDovL2NhY2VydHMuZGlnaWNlcnQuY29tL0RpZ2lDZXJ0
# VHJ1c3RlZFJvb3RHNC5jcnQwQwYDVR0fBDwwOjA4oDagNIYyaHR0cDovL2NybDMu
# ZGlnaWNlcnQuY29tL0RpZ2lDZXJ0VHJ1c3RlZFJvb3RHNC5jcmwwIAYDVR0gBBkw
# FzAIBgZngQwBBAIwCwYJYIZIAYb9bAcBMA0GCSqGSIb3DQEBCwUAA4ICAQB9WY7A
# k7ZvmKlEIgF+ZtbYIULhsBguEE0TzzBTzr8Y+8dQXeJLKftwig2qKWn8acHPHQfp
# PmDI2AvlXFvXbYf6hCAlNDFnzbYSlm/EUExiHQwIgqgWvalWzxVzjQEiJc6VaT9H
# d/tydBTX/6tPiix6q4XNQ1/tYLaqT5Fmniye4Iqs5f2MvGQmh2ySvZ180HAKfO+o
# vHVPulr3qRCyXen/KFSJ8NWKcXZl2szwcqMj+sAngkSumScbqyQeJsG33irr9p6x
# eZmBo1aGqwpFyd/EjaDnmPv7pp1yr8THwcFqcdnGE4AJxLafzYeHJLtPo0m5d2aR
# 8XKc6UsCUqc3fpNTrDsdCEkPlM05et3/JWOZJyw9P2un8WbDQc1PtkCbISFA0LcT
# JM3cHXg65J6t5TRxktcma+Q4c6umAU+9Pzt4rUyt+8SVe+0KXzM5h0F4ejjpnOHd
# I/0dKNPH+ejxmF/7K9h+8kaddSweJywm228Vex4Ziza4k9Tm8heZWcpw8De/mADf
# IBZPJ/tgZxahZrrdVcA6KYawmKAr7ZVBtzrVFZgxtGIJDwq9gdkT/r+k0fNX2bwE
# +oLeMt8EifAAzV3C+dAjfwAL5HYCJtnwZXZCpimHCUcr5n8apIUP/JiW9lVUKx+A
# +sDyDivl1vupL0QVSucTDh3bNzgaoSv27dZ8/DCCBY0wggR1oAMCAQICEA6bGI75
# 0C3n79tQ4ghAGFowDQYJKoZIhvcNAQEMBQAwZTELMAkGA1UEBhMCVVMxFTATBgNV
# BAoTDERpZ2lDZXJ0IEluYzEZMBcGA1UECxMQd3d3LmRpZ2ljZXJ0LmNvbTEkMCIG
# A1UEAxMbRGlnaUNlcnQgQXNzdXJlZCBJRCBSb290IENBMB4XDTIyMDgwMTAwMDAw
# MFoXDTMxMTEwOTIzNTk1OVowYjELMAkGA1UEBhMCVVMxFTATBgNVBAoTDERpZ2lD
# ZXJ0IEluYzEZMBcGA1UECxMQd3d3LmRpZ2ljZXJ0LmNvbTEhMB8GA1UEAxMYRGln
# aUNlcnQgVHJ1c3RlZCBSb290IEc0MIICIjANBgkqhkiG9w0BAQEFAAOCAg8AMIIC
# CgKCAgEAv+aQc2jeu+RdSjwwIjBpM+zCpyUuySE98orYWcLhKac9WKt2ms2uexuE
# DcQwH/MbpDgW61bGl20dq7J58soR0uRf1gU8Ug9SH8aeFaV+vp+pVxZZVXKvaJNw
# wrK6dZlqczKU0RBEEC7fgvMHhOZ0O21x4i0MG+4g1ckgHWMpLc7sXk7Ik/ghYZs0
# 6wXGXuxbGrzryc/NrDRAX7F6Zu53yEioZldXn1RYjgwrt0+nMNlW7sp7XeOtyU9e
# 5TXnMcvak17cjo+A2raRmECQecN4x7axxLVqGDgDEI3Y1DekLgV9iPWCPhCRcKtV
# gkEy19sEcypukQF8IUzUvK4bA3VdeGbZOjFEmjNAvwjXWkmkwuapoGfdpCe8oU85
# tRFYF/ckXEaPZPfBaYh2mHY9WV1CdoeJl2l6SPDgohIbZpp0yt5LHucOY67m1O+S
# kjqePdwA5EUlibaaRBkrfsCUtNJhbesz2cXfSwQAzH0clcOP9yGyshG3u3/y1Yxw
# LEFgqrFjGESVGnZifvaAsPvoZKYz0YkH4b235kOkGLimdwHhD5QMIR2yVCkliWzl
# DlJRR3S+Jqy2QXXeeqxfjT/JvNNBERJb5RBQ6zHFynIWIgnffEx1P2PsIV/EIFFr
# b7GrhotPwtZFX50g/KEexcCPorF+CiaZ9eRpL5gdLfXZqbId5RsCAwEAAaOCATow
# ggE2MA8GA1UdEwEB/wQFMAMBAf8wHQYDVR0OBBYEFOzX44LScV1kTN8uZz/nupiu
# HA9PMB8GA1UdIwQYMBaAFEXroq/0ksuCMS1Ri6enIZ3zbcgPMA4GA1UdDwEB/wQE
# AwIBhjB5BggrBgEFBQcBAQRtMGswJAYIKwYBBQUHMAGGGGh0dHA6Ly9vY3NwLmRp
# Z2ljZXJ0LmNvbTBDBggrBgEFBQcwAoY3aHR0cDovL2NhY2VydHMuZGlnaWNlcnQu
# Y29tL0RpZ2lDZXJ0QXNzdXJlZElEUm9vdENBLmNydDBFBgNVHR8EPjA8MDqgOKA2
# hjRodHRwOi8vY3JsMy5kaWdpY2VydC5jb20vRGlnaUNlcnRBc3N1cmVkSURSb290
# Q0EuY3JsMBEGA1UdIAQKMAgwBgYEVR0gADANBgkqhkiG9w0BAQwFAAOCAQEAcKC/
# Q1xV5zhfoKN0Gz22Ftf3v1cHvZqsoYcs7IVeqRq7IviHGmlUIu2kiHdtvRoU9BNK
# ei8ttzjv9P+Aufih9/Jy3iS8UgPITtAq3votVs/59PesMHqai7Je1M/RQ0SbQyHr
# lnKhSLSZy51PpwYDE3cnRNTnf+hZqPC/Lwum6fI0POz3A8eHqNJMQBk1RmppVLC4
# oVaO7KTVPeix3P0c2PR3WlxUjG/voVA9/HYJaISfb8rbII01YBwCA8sgsKxYoA5A
# Y8WYIsGyWfVVa88nq2x2zm8jLfR+cWojayL/ErhULSd+2DrZ8LaHlv1b0VysGMNN
# n3O3AamfV6peKOK5lDGCA3YwggNyAgEBMHcwYzELMAkGA1UEBhMCVVMxFzAVBgNV
# BAoTDkRpZ2lDZXJ0LCBJbmMuMTswOQYDVQQDEzJEaWdpQ2VydCBUcnVzdGVkIEc0
# IFJTQTQwOTYgU0hBMjU2IFRpbWVTdGFtcGluZyBDQQIQC65mvFq6f5WHxvnpBOMz
# BDANBglghkgBZQMEAgEFAKCB0TAaBgkqhkiG9w0BCQMxDQYLKoZIhvcNAQkQAQQw
# HAYJKoZIhvcNAQkFMQ8XDTI0MTEyMjA1NTMyNFowKwYLKoZIhvcNAQkQAgwxHDAa
# MBgwFgQU29OF7mLb0j575PZxSFCHJNWGW0UwLwYJKoZIhvcNAQkEMSIEIKpkPobq
# FcilEy8+7U68u3WJgYXGP7c5uVppBLmsEm7RMDcGCyqGSIb3DQEJEAIvMSgwJjAk
# MCIEIHZ2n6jyYy8fQws6IzCu1lZ1/tdz2wXWZbkFk5hDj5rbMA0GCSqGSIb3DQEB
# AQUABIICAJ8A+ZQR2NnF9J56e9W3L3X/CA/1gB8u4fGJqltUCcFyyDaFMcMgI6UK
# z4uej9JE8+dlP5Uz6OK/KDdwdRgvKAubADIUIE8mY1/iPHhUoY9XcE7GmJY4NCYX
# 3am4idhin1EgGoqZdcHWNh0wXaJ/9ld65Sv8S1EPRnuNC2QcvkS6for9ZoS1vlBy
# iWFSK+ojiGXZe9a2NUu3xYgeNoFWsW78ZCwWD1jx0P+dtxflXqVkvuJo1MECEEMD
# cAfCMJWbF3nxEk64bwqofpid0TePFIzAAXsRPSDbUwotrr0MonQw2dRjkNyNNifq
# VeKVuyeRN+4M2zzAgykrpmDm1qWrn93aDXyjYFEpLHNdtrAN3Ar11mJGY2dWG4+G
# T8MGqHiL/oT9C4smLXYcOc+gO360YPryNPLoDTJGrjNqKlgDHY9u7bi7w/nhvoaF
# yH6GPtYejSQIdcVQFuHBLrPzrtN2hLa6cJ1hUX0DhFb7BUKvChyj6RNQy72xSYu7
# BlP2NAMUeLQGkecReSSkJou570lPFWej3W/TYrlgZgFnJowE8i59+Z1VAYwes5FC
# zmTVCCNa0N1nr/cgFAToctUXV9ZEVLkB0Bxf8Ach1H4T7R3gSCBVCtmUSgh/JJkV
# QCkhSaKuAUi2XNGN3mOxcjoZGTggSGbHiSjX717S7rhGapb2Vo2K
# SIG # End signature block
