﻿<# Copyright © 2023. Cloud Software Group, Inc. All Rights Reserved. #>
# General function for execution of the plugins. 
Function Invoke-CTXOEPlugin ([String]$PluginName, [System.Xml.XmlElement]$Params, [String]$Mode) {

    [String]$m_FunctionName = "Invoke-CTXOE$($PluginName)$($Mode.ToString())"

    # First test if the required plugin and function is available 
    If ($(Get-Command "$m_FunctionName" -Module CTXOEP_$($PluginName) -ErrorAction SilentlyContinue) -isnot [System.Management.Automation.FunctionInfo]) {
        Throw "Failed to load the required plugin or required function has not been implemented.
        Module: CTXOEP_$($PluginName)
        Function: $m_FunctionName"
    }

    If ($Params -isnot [Object]) {
        Throw "<params /> element is invalid for current entry. Review the definition XML file."
    }

    # Run the plugin with arguments
    & (Get-ChildItem "Function:$m_FunctionName") -Params $Params

}

# Test if registry key (Key + Name) has the required value (Value). Returns a dictionary with two values - [Bool]Result and [String]Details. 
Function Test-CTXOERegistryValue ([String]$Key, [String]$Name, [String]$Value) {
    # Initialize $return object and always assume failure
    [Hashtable]$Return = @{}
    $Return.Result = $False

    [Boolean]$m_RegKeyExists = Test-Path Registry::$($Key)

    # If value is CTXOE_DeleteKey, check if key itself exists. We need to process this first, because DeleteKey does not require 'Name' parameter and next section would fail
    If ($Value -eq "CTXOE_DeleteKey") {
        $Return.Result = $m_RegKeyExists -eq $False;
        If ($Return.Result -eq $True) {
            $Return.Details = "Registry key does not exist";
        } Else {
            $Return.Details = "Registry key exists";
        }
        Return $Return;
    }

    # If value name ('name') is not defined, Optimizer will only test if key exists. This is used in scenarios where you only need to create registry key, but without any values.
    If ($Name.Length -eq 0) {
        $Return.Result = $m_RegKeyExists;
        If ($Return.Result -eq $True) {
            $Return.Details = "Registry key exists";
        } Else {
            $Return.Details = "Registry key does not exist";
        }
        Return $Return;
    }

    # Retrieve the registry item
    $m_RegObject = Get-ItemProperty -Path Registry::$($Key) -Name $Name -ErrorAction SilentlyContinue;

    # If value is CTXOE_DeleteValue (or legacy CTXOE_NoValue), check if value exists. This code doesn't care what is the actual value data, only if it exists or not.
    If (($Value -eq "CTXOE_NoValue") -or ($Value -eq "CTXOE_DeleteValue")) {
        $Return.Result = $m_RegObject -isnot [System.Management.Automation.PSCustomObject];
        If ($Return.Result -eq $True) {
            $Return.Details = "Registry value does not exist";
        } Else {
            $Return.Details = "Registry value exists";
        }
        Return $Return;
    }

    # Return false if registry value was not found
    If ($m_RegObject -isnot [System.Management.Automation.PSCustomObject]) {
        $Return.Details = "Registry value does not exists"
        Return $Return;
    }

    # Registry value can be different object types, for example byte array or integer. The problem is that PowerShell does not properly compare some object types, for example you cannot compare two byte arrays. 
    # When we force $m_Value to always be [String], we have more predictable comparison operation. For example [String]$([Byte[]]@(1,1,1)) -eq $([Byte[]]@(1,1,1)) will work as expected, but $([Byte[]]@(1,1,1)) -eq $([Byte[]]@(1,1,1)) will not
    [string]$m_Value = $m_RegObject.$Name; 

    # If value is binary array, we need to convert it to string first
    If ($m_RegObject.$Name -is [System.Byte[]]) {
        [Byte[]]$Value = $Value.Split(",");
    }

    # If value type is DWORD or QWORD, registry object returns decimal value, while template can use both decimal and hexadecimal. If hexa is used in template, convert to decimal before comparison
    If ($Value -like "0x*") {
        # $m_RegObject.$Name can be different types (Int32, UInt32, Int64, UInt64...). Instead of handling multiple If...Else..., just use convert as to make sure that we are comparing apples to apples
        $Value = $Value -as $m_RegObject.$Name.GetType();
    }
    
    # $m_Value is always [String], $Value can be [String] or [Byte[]] array
    If ($m_value -ne $Value) {
        $Return.Details = "Different value ($m_value instead of $Value)"
    } Else {
        $Return.Result = $True
        $Return.Details = "Requested value $Value is configured"
    }
    Return $Return
}

# Set value of a specified registry key. Returns a dictionary with two values - [Bool]Result and [String]Details.
# There are few special values - CTXOE_DeleteKey (delete whole registry key if present), CTXOE_DeleteValue (delete registry value if present) and LEGACY CTXOE_NoValue (use CTXOE_DeleteValue instead, this was original name)
Function Set-CTXOERegistryValue ([String]$Key, [String]$Name, [String]$Value, [String]$ValueType) {
    
    [Hashtable]$Return = @{"Result" = $False; "Details" = "Internal error in function"}; 

    [Boolean]$m_RegKeyExists = Test-Path Registry::$Key;

    # First we need to handle scenario where whole key should be deleted
    If ($Value -eq "CTXOE_DeleteKey") {
        If ($m_RegKeyExists -eq $True) {
            Remove-Item -Path Registry::$Key -Force -ErrorAction SilentlyContinue | Out-Null
        }

        # Test if registry key exists or not. We need to pass value, so test function understands that we do NOT expect to find anything at target location
        [Hashtable]$Return = Test-CTXOERegistryValue -Key $Key -Value $Value;

        # When we delete whole registry key, we cannot restore it (unless we completely export it before, which is not supported yet)
        $Return.OriginalValue = "CTXOE_DeleteKey";

        Return $Return;

    }
    
    # If parent registry key does not exists, create it
    If ($m_RegKeyExists -eq $False) {
        New-Item Registry::$Key -Force | Out-Null;
        $Return.OriginalValue = "CTXOE_DeleteKey";
    }

    # If 'Name' is not defined, we need to only create a key and not any values
    If ($Name.Length -eq 0) {
        [Hashtable]$Return = Test-CTXOERegistryValue -Key $Key;
        # We need to re-assign this value again - $Return is overwritten by function Test-CTXOERegistryValue
        If ($m_RegKeyExists -eq $False) {
            $Return.OriginalValue = "CTXOE_DeleteKey";
        }
        Return $Return;
    }

    # Now change the value
    $m_ExistingValue = Get-ItemProperty -Path Registry::$Key -Name $Name -ErrorAction SilentlyContinue
    Try {
        If (($Value -eq "CTXOE_NoValue") -or ($Value -eq "CTXOE_DeleteValue")) {
            Remove-ItemProperty -Path Registry::$Key -Name $Name -Force -ErrorAction SilentlyContinue | Out-Null
        } Else {
            # If value type is binary, we need to convert string to byte array first. If this method is used directly with -Value argument (one line instead of two), it fails with error "You cannot call a method on a null-valued expression."
            If ($ValueType -eq "Binary") {
                [Byte[]]$m_ByteArray = $Value.Split(","); #[System.Text.Encoding]::Unicode.GetBytes($Value);
                New-ItemProperty -Path Registry::$Key -Name $Name -PropertyType $ValueType -Value $m_ByteArray -Force | Out-Null
            } Else {
                New-ItemProperty -Path Registry::$Key -Name $Name -PropertyType $ValueType -Value $Value -Force | Out-Null
            }
        }
    } Catch {
        $Return.Details = $($_.Exception.Message); 
        $Return.OriginalValue = "CTXOE_DeleteValue";
        Return $Return; 
    }

    # Re-run the validation test again
    [Hashtable]$Return = Test-CTXOERegistryValue -Key $Key -Name $Name -Value $Value
    
    # Save previous value for rollback functionality
    If ($m_RegKeyExists -eq $True) {
        If ($m_ExistingValue -is [Object]) {
            $Return.OriginalValue = $m_ExistingValue.$Name
        } Else {
            $Return.OriginalValue = "CTXOE_DeleteValue"
        }
    } Else {
        # We need to set this again, since $Return is overwritten by Test-CTXOERegistryValue function
        $Return.OriginalValue = "CTXOE_DeleteKey";
    }
    
    Return $Return
}
Function ConvertTo-CTXOERollbackElement ([Xml.XmlElement]$Element) {
    # Convert the element to XmlDocument. 
    [Xml]$m_TempXmlDocument = New-Object Xml.XmlDocument

    # Change the <params /> (or <executeparams /> to <rollbackparams />. 
    [Xml.XmlElement]$m_TempRootElement = $m_TempXmlDocument.CreateElement("rollbackparams")
    $m_TempRootElement.InnerXml = $Element.InnerXml
    $m_TempXmlDocument.AppendChild($m_TempRootElement) | Out-Null

    # Rollback is based on <value /> element. If this element doesn't exist already (in $Element), create an empty one. If we don't create this empty element, other functions that are trying to assign data to property .value will fail
    If ($m_TempRootElement.Item("value") -isnot [Xml.XmlElement]) {
        $m_TempRootElement.AppendChild($m_TempXmlDocument.CreateElement("value")) | Out-Null; 
    }

    # Return object
    Return $m_TempXmlDocument
}
Function New-CTXOEHistoryElement ([Xml.XmlElement]$Element, [Boolean]$SystemChanged, [DateTime]$StartTime, [Boolean]$Result, [String]$Details, [Xml.XmlDocument]$RollbackInstructions) {
    # Delete any previous <history /> from $Element
    If ($Element.History -is [Object]) {
        $Element.RemoveChild($Element.History) | Out-Null; 
    }

    # Get the parente XML document of the target element
    [Xml.XmlDocument]$SourceXML = $Element.OwnerDocument

    # Generate new temporary XML document. This is easiest way how to construct more complex XML structures with minimal performance impact. 
    [Xml]$m_TempXmlDoc = "<history><systemchanged>$([Int]$SystemChanged)</systemchanged><starttime>$($StartTime.ToString())</starttime><endtime>$([DateTime]::Now.ToString())</endtime><return><result>$([Int]$Result)</result><details>$Details</details></return></history>"

    # Import temporary XML document (standalone) as an XML element to our existing document
    $m_TempNode = $SourceXML.ImportNode($m_TempXmlDoc.DocumentElement, $true)
    $Element.AppendChild($m_TempNode) | Out-Null; 

    # If $RollbackInstructions is provided, save it as a <rollackparams /> element
    If ($RollbackInstructions -is [Object]) {
        $Element.Action.AppendChild($SourceXML.ImportNode($RollbackInstructions.DocumentElement, $true)) | Out-Null
    }
}

# Function to validate conditions. Returns hashtable object with two properties - Result (boolean) and Details. Result should be $True
Function Test-CTXOECondition([Xml.XmlElement]$Element) {

    [Hashtable]$m_Result = @{}; 

    # Always assume that script will fail
    $m_Result.Result = $False;
    $m_Result.Details = "No condition message defined"

    # $CTXOE_Condition is variable that should be returned by code. Because it is global, we want to reset it first. Do NOT assign $Null to variable - it will not delete it, just create variable with $null value
    Remove-Variable -Force -Name CTXOE_Condition -ErrorAction SilentlyContinue -Scope Global;
    Remove-Variable -Force -Name CTXOE_ConditionMessage -ErrorAction SilentlyContinue -Scope Global;

    # Check if condition has all required information (code is most important)
    If ($Element.conditioncode -isnot [object]) {
        $m_Result.Details = "Invalid or missing condition code. Condition cannot be processed";
        Return $m_Result;
    }

    # Execute code. This code should always return $Global:CTXOE_Condition variable (required) and $Global:CTXOE_ConditionMessage (optional)
    Try {
        Invoke-Expression -Command $Element.conditioncode;
    } Catch {
        $m_Result.Details = "Unexpected failure while processing condition: $($_.Exception.Message)";
        Return $m_Result;
    }
    

    # Validate output

    # Test if variable exists
    If (-not $(Test-Path Variable:Global:CTXOE_Condition)) {
        $m_Result.Details = "Required variable (CTXOE_Condition) NOT returned by condition. Condition cannot be processed";
        Return $m_Result;
    }

    # Test if variable is boolean
    If ($Global:CTXOE_Condition -isnot [Boolean]) {
        $m_Result.Details = "Required variable (CTXOE_Condition) is NOT boolean ($True or $False), but $($Global:CTXOE_Condition.GetType().FullName). Condition cannot be processed";
        Return $m_Result;
    }

    # Assign value to variable
    $m_Result.Result = $Global:CTXOE_Condition;

    # If condition failed and failed message is specified in XML section for condition, assign it
    If ($Element.conditionfailedmessage -is [Object] -and $m_Result.Result -eq $False) {
        $m_Result.Details = $Element.conditionfailedmessage;
    }

    # If $CTXOE_ConditionMessage is returned by code, use it to override the failed message
    If ((Test-Path Variable:Global:CTXOE_ConditionMessage)) {
        $m_Result.Details = $Global:CTXOE_ConditionMessage
    }

    # Return object
    Return $m_Result;

}
# SIG # Begin signature block
# MIIoowYJKoZIhvcNAQcCoIIolDCCKJACAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCC0do/ckUyv1iEK
# MGyKF6Nz8Ti+Nm2HrGDwsAzImkuzxKCCDcAwggawMIIEmKADAgECAhAIrUCyYNKc
# TJ9ezam9k67ZMA0GCSqGSIb3DQEBDAUAMGIxCzAJBgNVBAYTAlVTMRUwEwYDVQQK
# EwxEaWdpQ2VydCBJbmMxGTAXBgNVBAsTEHd3dy5kaWdpY2VydC5jb20xITAfBgNV
# BAMTGERpZ2lDZXJ0IFRydXN0ZWQgUm9vdCBHNDAeFw0yMTA0MjkwMDAwMDBaFw0z
# NjA0MjgyMzU5NTlaMGkxCzAJBgNVBAYTAlVTMRcwFQYDVQQKEw5EaWdpQ2VydCwg
# SW5jLjFBMD8GA1UEAxM4RGlnaUNlcnQgVHJ1c3RlZCBHNCBDb2RlIFNpZ25pbmcg
# UlNBNDA5NiBTSEEzODQgMjAyMSBDQTEwggIiMA0GCSqGSIb3DQEBAQUAA4ICDwAw
# ggIKAoICAQDVtC9C0CiteLdd1TlZG7GIQvUzjOs9gZdwxbvEhSYwn6SOaNhc9es0
# JAfhS0/TeEP0F9ce2vnS1WcaUk8OoVf8iJnBkcyBAz5NcCRks43iCH00fUyAVxJr
# Q5qZ8sU7H/Lvy0daE6ZMswEgJfMQ04uy+wjwiuCdCcBlp/qYgEk1hz1RGeiQIXhF
# LqGfLOEYwhrMxe6TSXBCMo/7xuoc82VokaJNTIIRSFJo3hC9FFdd6BgTZcV/sk+F
# LEikVoQ11vkunKoAFdE3/hoGlMJ8yOobMubKwvSnowMOdKWvObarYBLj6Na59zHh
# 3K3kGKDYwSNHR7OhD26jq22YBoMbt2pnLdK9RBqSEIGPsDsJ18ebMlrC/2pgVItJ
# wZPt4bRc4G/rJvmM1bL5OBDm6s6R9b7T+2+TYTRcvJNFKIM2KmYoX7BzzosmJQay
# g9Rc9hUZTO1i4F4z8ujo7AqnsAMrkbI2eb73rQgedaZlzLvjSFDzd5Ea/ttQokbI
# YViY9XwCFjyDKK05huzUtw1T0PhH5nUwjewwk3YUpltLXXRhTT8SkXbev1jLchAp
# QfDVxW0mdmgRQRNYmtwmKwH0iU1Z23jPgUo+QEdfyYFQc4UQIyFZYIpkVMHMIRro
# OBl8ZhzNeDhFMJlP/2NPTLuqDQhTQXxYPUez+rbsjDIJAsxsPAxWEQIDAQABo4IB
# WTCCAVUwEgYDVR0TAQH/BAgwBgEB/wIBADAdBgNVHQ4EFgQUaDfg67Y7+F8Rhvv+
# YXsIiGX0TkIwHwYDVR0jBBgwFoAU7NfjgtJxXWRM3y5nP+e6mK4cD08wDgYDVR0P
# AQH/BAQDAgGGMBMGA1UdJQQMMAoGCCsGAQUFBwMDMHcGCCsGAQUFBwEBBGswaTAk
# BggrBgEFBQcwAYYYaHR0cDovL29jc3AuZGlnaWNlcnQuY29tMEEGCCsGAQUFBzAC
# hjVodHRwOi8vY2FjZXJ0cy5kaWdpY2VydC5jb20vRGlnaUNlcnRUcnVzdGVkUm9v
# dEc0LmNydDBDBgNVHR8EPDA6MDigNqA0hjJodHRwOi8vY3JsMy5kaWdpY2VydC5j
# b20vRGlnaUNlcnRUcnVzdGVkUm9vdEc0LmNybDAcBgNVHSAEFTATMAcGBWeBDAED
# MAgGBmeBDAEEATANBgkqhkiG9w0BAQwFAAOCAgEAOiNEPY0Idu6PvDqZ01bgAhql
# +Eg08yy25nRm95RysQDKr2wwJxMSnpBEn0v9nqN8JtU3vDpdSG2V1T9J9Ce7FoFF
# UP2cvbaF4HZ+N3HLIvdaqpDP9ZNq4+sg0dVQeYiaiorBtr2hSBh+3NiAGhEZGM1h
# mYFW9snjdufE5BtfQ/g+lP92OT2e1JnPSt0o618moZVYSNUa/tcnP/2Q0XaG3Ryw
# YFzzDaju4ImhvTnhOE7abrs2nfvlIVNaw8rpavGiPttDuDPITzgUkpn13c5Ubdld
# AhQfQDN8A+KVssIhdXNSy0bYxDQcoqVLjc1vdjcshT8azibpGL6QB7BDf5WIIIJw
# 8MzK7/0pNVwfiThV9zeKiwmhywvpMRr/LhlcOXHhvpynCgbWJme3kuZOX956rEnP
# LqR0kq3bPKSchh/jwVYbKyP/j7XqiHtwa+aguv06P0WmxOgWkVKLQcBIhEuWTatE
# QOON8BUozu3xGFYHKi8QxAwIZDwzj64ojDzLj4gLDb879M4ee47vtevLt/B3E+bn
# KD+sEq6lLyJsQfmCXBVmzGwOysWGw/YmMwwHS6DTBwJqakAwSEs0qFEgu60bhQji
# WQ1tygVQK+pKHJ6l/aCnHwZ05/LWUpD9r4VIIflXO7ScA+2GRfS0YW6/aOImYIbq
# yK+p/pQd52MbOoZWeE4wggcIMIIE8KADAgECAhAExKVRXTiJBY6ZBfjtmlFwMA0G
# CSqGSIb3DQEBCwUAMGkxCzAJBgNVBAYTAlVTMRcwFQYDVQQKEw5EaWdpQ2VydCwg
# SW5jLjFBMD8GA1UEAxM4RGlnaUNlcnQgVHJ1c3RlZCBHNCBDb2RlIFNpZ25pbmcg
# UlNBNDA5NiBTSEEzODQgMjAyMSBDQTEwHhcNMjQwMzA4MDAwMDAwWhcNMjUwMzA3
# MjM1OTU5WjCBjzELMAkGA1UEBhMCVVMxEDAOBgNVBAgTB0Zsb3JpZGExGDAWBgNV
# BAcTD0ZvcnQgTGF1ZGVyZGFsZTEdMBsGA1UEChMUQ2l0cml4IFN5c3RlbXMsIElu
# Yy4xFjAUBgNVBAsTDUNpdHJpeCBYZW5BcHAxHTAbBgNVBAMTFENpdHJpeCBTeXN0
# ZW1zLCBJbmMuMIIBojANBgkqhkiG9w0BAQEFAAOCAY8AMIIBigKCAYEA+gJ84IK2
# t0aBgBuXB3yfqoC+riwIEohXTQHVqNvJ7v3xiiIYD8PvzYLhaQq2dG4euY1AOhXL
# 0Uc2rNLcQZKEB4PslYrrorg+x5iRJl2k9XltDEpDbqJJqkUELY6d7GEIpp4ChdGf
# NJuGU9GCAgiml88tzZMOjtAqcLIXSRSOugQgjUOR+EvfeLlMTLs/ADjyOi898+0m
# TtdE6Sy/a2bn/fgsJaU8O2Kn78YiHX1N6lklKKBzO5tYfRD2VZhOn2kx6PUI+n+R
# ckwE22o0hBhckBeBGNW2DUrABDLwYuNOEWZEGx1KC2UDO4p+eDmxANjC/+q9t53d
# Z4V1Y/qL9CRnin0JNowVcLT3zGqeebgIag2Pqsh6m3lvASiy7DpmDtyaTCwv/aLd
# 1x5TZvUPiKcfvMOSCRxTP+a7L4DdjuqcQxWENk1kNEZjZa/ZS0BDWalklp6Ji9ki
# JpUKxB5C6GD/Lns+hbdp/2XHnEsqGbeQBCiW7ftXO3ugffq+/Bz1CelPAgMBAAGj
# ggIDMIIB/zAfBgNVHSMEGDAWgBRoN+Drtjv4XxGG+/5hewiIZfROQjAdBgNVHQ4E
# FgQUh00+M0JEuzdBGq4PW6l3UIoBQsIwPgYDVR0gBDcwNTAzBgZngQwBBAEwKTAn
# BggrBgEFBQcCARYbaHR0cDovL3d3dy5kaWdpY2VydC5jb20vQ1BTMA4GA1UdDwEB
# /wQEAwIHgDATBgNVHSUEDDAKBggrBgEFBQcDAzCBtQYDVR0fBIGtMIGqMFOgUaBP
# hk1odHRwOi8vY3JsMy5kaWdpY2VydC5jb20vRGlnaUNlcnRUcnVzdGVkRzRDb2Rl
# U2lnbmluZ1JTQTQwOTZTSEEzODQyMDIxQ0ExLmNybDBToFGgT4ZNaHR0cDovL2Ny
# bDQuZGlnaWNlcnQuY29tL0RpZ2lDZXJ0VHJ1c3RlZEc0Q29kZVNpZ25pbmdSU0E0
# MDk2U0hBMzg0MjAyMUNBMS5jcmwwgZQGCCsGAQUFBwEBBIGHMIGEMCQGCCsGAQUF
# BzABhhhodHRwOi8vb2NzcC5kaWdpY2VydC5jb20wXAYIKwYBBQUHMAKGUGh0dHA6
# Ly9jYWNlcnRzLmRpZ2ljZXJ0LmNvbS9EaWdpQ2VydFRydXN0ZWRHNENvZGVTaWdu
# aW5nUlNBNDA5NlNIQTM4NDIwMjFDQTEuY3J0MAkGA1UdEwQCMAAwDQYJKoZIhvcN
# AQELBQADggIBAI1e2H/UCCob4N14i0h2Z1Yi7dInt7RICuwc9yiMcDeGvlvtgcJT
# zp/CyQwpDh58/WGxR0HX3TuLjLh0Foxzm6aSgTLnyvI65jugv24L7yu1Bl5n7WR1
# x4tqtvVI1DOCTWNN9BqokeBTuT7NVwDPyoLWxz8f3GqnERJ/xe4DJOv9RXre6sy0
# NF9a+d4oqvEJ5r03WOpI6jO3YG6hyxCF7611ZhVdvBuw0ZfVgXPH1n/l+kGb4Mdc
# s0kJ6ny/auQ3Cm0lpCxrjh+LxwrxYtk1s/iZMd9T+C5vjUzH1LWhqlPur0ur05yU
# cG61A8pPwKLE02FdW9b6miAPknpUI6o6EgdU1KRCCjtYoR1iBSagB75k0RZBcgor
# B6mQKcFF5EwRhw5rSJdn1r7rTjRgVTzgRjgHL7QBY27yshp9rTwqHzO9VUmfTxg1
# 4UT5dcZ3xGc8U4j+gMSjNXmJJEpRaNtuh1QDXoQCwC0fM/9r4BxIlS50nuWrhnBL
# zSbrdxbmDK+iW5UU2oCKs3zHbrPmqQgWd4CDi/g/kUEruRIL+HMS1eAdZlyZ8DCw
# H8Qwv6+T6we/iDDbHtxLy7TjGsYChqEgaVOT7gj++iIdoHq3+mb1bxtDgovELXFN
# EwOxXT5mt7qZlc4ApoUTdruNzS9aMofoIB2SIVo/P6FCryDE5YxVmv6zMYIaOTCC
# GjUCAQEwfTBpMQswCQYDVQQGEwJVUzEXMBUGA1UEChMORGlnaUNlcnQsIEluYy4x
# QTA/BgNVBAMTOERpZ2lDZXJ0IFRydXN0ZWQgRzQgQ29kZSBTaWduaW5nIFJTQTQw
# OTYgU0hBMzg0IDIwMjEgQ0ExAhAExKVRXTiJBY6ZBfjtmlFwMA0GCWCGSAFlAwQC
# AQUAoIHQMBkGCSqGSIb3DQEJAzEMBgorBgEEAYI3AgEEMBwGCisGAQQBgjcCAQsx
# DjAMBgorBgEEAYI3AgEVMC8GCSqGSIb3DQEJBDEiBCD0nI9DGTTEG2qGZIuC7YbR
# NNIracZI1v8G9yw1bJDvKzBkBgorBgEEAYI3AgEMMVYwVKA4gDYAQwBpAHQAcgBp
# AHgAIABTAHUAcABwAG8AcgB0AGEAYgBpAGwAaQB0AHkAIABUAG8AbwBsAHOhGIAW
# aHR0cDovL3d3dy5jaXRyaXguY29tIDANBgkqhkiG9w0BAQEFAASCAYDB/Ff+W2R5
# WohrZ5DesAV0laB6kutxEOku06dIEdNhcBHnMm0X3C/s/8y6bunnJxLfuuxk19s8
# dSIfkZgX1xauQEJTXieWsJ45r2PaMgInNiHvklye3NbssBrL0ZswhnUrJxtter2D
# 2nDoFIpLjgUtDI/EQumItMc3IL+KrFgWB1ZW7I7I/UFLDYRFlc1Foypkpk8x9A1c
# OkOchDwpOCEE/XhfGAE5NrNeDri5b46OK17nno4kSx6f7CfDS7EiAFEFWgkLpVvx
# kANRpGNbsH6jf96TyM3vBvAV+u0mgraS9dIxsTiiW4EHk1BmAxKKc6ZBMzAKc9kH
# TdruGnuZLcOH0FlE7buPoWkbUP2JC0yeTU/cYTIyFzeUDuqnWruUfnYH7lZMuQQU
# XO4XKG8l7ZedVlqoQKYB2CLlCJ5GAaDgbm12tgbdrVhixWs1Xq4hDlDtsrIa8bRn
# ZP3mKG76gOF0iGFYHCc4CFfA4yIJInSgZtwTUK6JEdelZzgLqHBQuvehghc6MIIX
# NgYKKwYBBAGCNwMDATGCFyYwghciBgkqhkiG9w0BBwKgghcTMIIXDwIBAzEPMA0G
# CWCGSAFlAwQCAQUAMHgGCyqGSIb3DQEJEAEEoGkEZzBlAgEBBglghkgBhv1sBwEw
# MTANBglghkgBZQMEAgEFAAQgx/da+s+7LkoojtaVJyuGy7AU7jHPXP25ieN/mae0
# rdcCEQCblZzRjntMytE1/P9v6MkUGA8yMDI0MTEyMjA1NTQwNFqgghMDMIIGvDCC
# BKSgAwIBAgIQC65mvFq6f5WHxvnpBOMzBDANBgkqhkiG9w0BAQsFADBjMQswCQYD
# VQQGEwJVUzEXMBUGA1UEChMORGlnaUNlcnQsIEluYy4xOzA5BgNVBAMTMkRpZ2lD
# ZXJ0IFRydXN0ZWQgRzQgUlNBNDA5NiBTSEEyNTYgVGltZVN0YW1waW5nIENBMB4X
# DTI0MDkyNjAwMDAwMFoXDTM1MTEyNTIzNTk1OVowQjELMAkGA1UEBhMCVVMxETAP
# BgNVBAoTCERpZ2lDZXJ0MSAwHgYDVQQDExdEaWdpQ2VydCBUaW1lc3RhbXAgMjAy
# NDCCAiIwDQYJKoZIhvcNAQEBBQADggIPADCCAgoCggIBAL5qc5/2lSGrljC6W23m
# WaO16P2RHxjEiDtqmeOlwf0KMCBDEr4IxHRGd7+L660x5XltSVhhK64zi9CeC9B6
# lUdXM0s71EOcRe8+CEJp+3R2O8oo76EO7o5tLuslxdr9Qq82aKcpA9O//X6QE+Ac
# aU/byaCagLD/GLoUb35SfWHh43rOH3bpLEx7pZ7avVnpUVmPvkxT8c2a2yC0WMp8
# hMu60tZR0ChaV76Nhnj37DEYTX9ReNZ8hIOYe4jl7/r419CvEYVIrH6sN00yx49b
# oUuumF9i2T8UuKGn9966fR5X6kgXj3o5WHhHVO+NBikDO0mlUh902wS/Eeh8F/UF
# aRp1z5SnROHwSJ+QQRZ1fisD8UTVDSupWJNstVkiqLq+ISTdEjJKGjVfIcsgA4l9
# cbk8Smlzddh4EfvFrpVNnes4c16Jidj5XiPVdsn5n10jxmGpxoMc6iPkoaDhi6Jj
# Hd5ibfdp5uzIXp4P0wXkgNs+CO/CacBqU0R4k+8h6gYldp4FCMgrXdKWfM4N0u25
# OEAuEa3JyidxW48jwBqIJqImd93NRxvd1aepSeNeREXAu2xUDEW8aqzFQDYmr9ZO
# Nuc2MhTMizchNULpUEoA6Vva7b1XCB+1rxvbKmLqfY/M/SdV6mwWTyeVy5Z/JkvM
# FpnQy5wR14GJcv6dQ4aEKOX5AgMBAAGjggGLMIIBhzAOBgNVHQ8BAf8EBAMCB4Aw
# DAYDVR0TAQH/BAIwADAWBgNVHSUBAf8EDDAKBggrBgEFBQcDCDAgBgNVHSAEGTAX
# MAgGBmeBDAEEAjALBglghkgBhv1sBwEwHwYDVR0jBBgwFoAUuhbZbU2FL3Mpdpov
# dYxqII+eyG8wHQYDVR0OBBYEFJ9XLAN3DigVkGalY17uT5IfdqBbMFoGA1UdHwRT
# MFEwT6BNoEuGSWh0dHA6Ly9jcmwzLmRpZ2ljZXJ0LmNvbS9EaWdpQ2VydFRydXN0
# ZWRHNFJTQTQwOTZTSEEyNTZUaW1lU3RhbXBpbmdDQS5jcmwwgZAGCCsGAQUFBwEB
# BIGDMIGAMCQGCCsGAQUFBzABhhhodHRwOi8vb2NzcC5kaWdpY2VydC5jb20wWAYI
# KwYBBQUHMAKGTGh0dHA6Ly9jYWNlcnRzLmRpZ2ljZXJ0LmNvbS9EaWdpQ2VydFRy
# dXN0ZWRHNFJTQTQwOTZTSEEyNTZUaW1lU3RhbXBpbmdDQS5jcnQwDQYJKoZIhvcN
# AQELBQADggIBAD2tHh92mVvjOIQSR9lDkfYR25tOCB3RKE/P09x7gUsmXqt40ouR
# l3lj+8QioVYq3igpwrPvBmZdrlWBb0HvqT00nFSXgmUrDKNSQqGTdpjHsPy+Laal
# TW0qVjvUBhcHzBMutB6HzeledbDCzFzUy34VarPnvIWrqVogK0qM8gJhh/+qDEAI
# dO/KkYesLyTVOoJ4eTq7gj9UFAL1UruJKlTnCVaM2UeUUW/8z3fvjxhN6hdT98Vr
# 2FYlCS7Mbb4Hv5swO+aAXxWUm3WpByXtgVQxiBlTVYzqfLDbe9PpBKDBfk+rabTF
# DZXoUke7zPgtd7/fvWTlCs30VAGEsshJmLbJ6ZbQ/xll/HjO9JbNVekBv2Tgem+m
# LptR7yIrpaidRJXrI+UzB6vAlk/8a1u7cIqV0yef4uaZFORNekUgQHTqddmsPCEI
# YQP7xGxZBIhdmm4bhYsVA6G2WgNFYagLDBzpmk9104WQzYuVNsxyoVLObhx3Ruga
# EGru+SojW4dHPoWrUhftNpFC5H7QEY7MhKRyrBe7ucykW7eaCuWBsBb4HOKRFVDc
# rZgdwaSIqMDiCLg4D+TPVgKx2EgEdeoHNHT9l3ZDBD+XgbF+23/zBjeCtxz+dL/9
# NWR6P2eZRi7zcEO1xwcdcqJsyz/JceENc2Sg8h3KeFUCS7tpFk7CrDqkMIIGrjCC
# BJagAwIBAgIQBzY3tyRUfNhHrP0oZipeWzANBgkqhkiG9w0BAQsFADBiMQswCQYD
# VQQGEwJVUzEVMBMGA1UEChMMRGlnaUNlcnQgSW5jMRkwFwYDVQQLExB3d3cuZGln
# aWNlcnQuY29tMSEwHwYDVQQDExhEaWdpQ2VydCBUcnVzdGVkIFJvb3QgRzQwHhcN
# MjIwMzIzMDAwMDAwWhcNMzcwMzIyMjM1OTU5WjBjMQswCQYDVQQGEwJVUzEXMBUG
# A1UEChMORGlnaUNlcnQsIEluYy4xOzA5BgNVBAMTMkRpZ2lDZXJ0IFRydXN0ZWQg
# RzQgUlNBNDA5NiBTSEEyNTYgVGltZVN0YW1waW5nIENBMIICIjANBgkqhkiG9w0B
# AQEFAAOCAg8AMIICCgKCAgEAxoY1BkmzwT1ySVFVxyUDxPKRN6mXUaHW0oPRnkyi
# baCwzIP5WvYRoUQVQl+kiPNo+n3znIkLf50fng8zH1ATCyZzlm34V6gCff1DtITa
# EfFzsbPuK4CEiiIY3+vaPcQXf6sZKz5C3GeO6lE98NZW1OcoLevTsbV15x8GZY2U
# KdPZ7Gnf2ZCHRgB720RBidx8ald68Dd5n12sy+iEZLRS8nZH92GDGd1ftFQLIWhu
# NyG7QKxfst5Kfc71ORJn7w6lY2zkpsUdzTYNXNXmG6jBZHRAp8ByxbpOH7G1WE15
# /tePc5OsLDnipUjW8LAxE6lXKZYnLvWHpo9OdhVVJnCYJn+gGkcgQ+NDY4B7dW4n
# JZCYOjgRs/b2nuY7W+yB3iIU2YIqx5K/oN7jPqJz+ucfWmyU8lKVEStYdEAoq3ND
# zt9KoRxrOMUp88qqlnNCaJ+2RrOdOqPVA+C/8KI8ykLcGEh/FDTP0kyr75s9/g64
# ZCr6dSgkQe1CvwWcZklSUPRR8zZJTYsg0ixXNXkrqPNFYLwjjVj33GHek/45wPmy
# MKVM1+mYSlg+0wOI/rOP015LdhJRk8mMDDtbiiKowSYI+RQQEgN9XyO7ZONj4Kbh
# PvbCdLI/Hgl27KtdRnXiYKNYCQEoAA6EVO7O6V3IXjASvUaetdN2udIOa5kM0jO0
# zbECAwEAAaOCAV0wggFZMBIGA1UdEwEB/wQIMAYBAf8CAQAwHQYDVR0OBBYEFLoW
# 2W1NhS9zKXaaL3WMaiCPnshvMB8GA1UdIwQYMBaAFOzX44LScV1kTN8uZz/nupiu
# HA9PMA4GA1UdDwEB/wQEAwIBhjATBgNVHSUEDDAKBggrBgEFBQcDCDB3BggrBgEF
# BQcBAQRrMGkwJAYIKwYBBQUHMAGGGGh0dHA6Ly9vY3NwLmRpZ2ljZXJ0LmNvbTBB
# BggrBgEFBQcwAoY1aHR0cDovL2NhY2VydHMuZGlnaWNlcnQuY29tL0RpZ2lDZXJ0
# VHJ1c3RlZFJvb3RHNC5jcnQwQwYDVR0fBDwwOjA4oDagNIYyaHR0cDovL2NybDMu
# ZGlnaWNlcnQuY29tL0RpZ2lDZXJ0VHJ1c3RlZFJvb3RHNC5jcmwwIAYDVR0gBBkw
# FzAIBgZngQwBBAIwCwYJYIZIAYb9bAcBMA0GCSqGSIb3DQEBCwUAA4ICAQB9WY7A
# k7ZvmKlEIgF+ZtbYIULhsBguEE0TzzBTzr8Y+8dQXeJLKftwig2qKWn8acHPHQfp
# PmDI2AvlXFvXbYf6hCAlNDFnzbYSlm/EUExiHQwIgqgWvalWzxVzjQEiJc6VaT9H
# d/tydBTX/6tPiix6q4XNQ1/tYLaqT5Fmniye4Iqs5f2MvGQmh2ySvZ180HAKfO+o
# vHVPulr3qRCyXen/KFSJ8NWKcXZl2szwcqMj+sAngkSumScbqyQeJsG33irr9p6x
# eZmBo1aGqwpFyd/EjaDnmPv7pp1yr8THwcFqcdnGE4AJxLafzYeHJLtPo0m5d2aR
# 8XKc6UsCUqc3fpNTrDsdCEkPlM05et3/JWOZJyw9P2un8WbDQc1PtkCbISFA0LcT
# JM3cHXg65J6t5TRxktcma+Q4c6umAU+9Pzt4rUyt+8SVe+0KXzM5h0F4ejjpnOHd
# I/0dKNPH+ejxmF/7K9h+8kaddSweJywm228Vex4Ziza4k9Tm8heZWcpw8De/mADf
# IBZPJ/tgZxahZrrdVcA6KYawmKAr7ZVBtzrVFZgxtGIJDwq9gdkT/r+k0fNX2bwE
# +oLeMt8EifAAzV3C+dAjfwAL5HYCJtnwZXZCpimHCUcr5n8apIUP/JiW9lVUKx+A
# +sDyDivl1vupL0QVSucTDh3bNzgaoSv27dZ8/DCCBY0wggR1oAMCAQICEA6bGI75
# 0C3n79tQ4ghAGFowDQYJKoZIhvcNAQEMBQAwZTELMAkGA1UEBhMCVVMxFTATBgNV
# BAoTDERpZ2lDZXJ0IEluYzEZMBcGA1UECxMQd3d3LmRpZ2ljZXJ0LmNvbTEkMCIG
# A1UEAxMbRGlnaUNlcnQgQXNzdXJlZCBJRCBSb290IENBMB4XDTIyMDgwMTAwMDAw
# MFoXDTMxMTEwOTIzNTk1OVowYjELMAkGA1UEBhMCVVMxFTATBgNVBAoTDERpZ2lD
# ZXJ0IEluYzEZMBcGA1UECxMQd3d3LmRpZ2ljZXJ0LmNvbTEhMB8GA1UEAxMYRGln
# aUNlcnQgVHJ1c3RlZCBSb290IEc0MIICIjANBgkqhkiG9w0BAQEFAAOCAg8AMIIC
# CgKCAgEAv+aQc2jeu+RdSjwwIjBpM+zCpyUuySE98orYWcLhKac9WKt2ms2uexuE
# DcQwH/MbpDgW61bGl20dq7J58soR0uRf1gU8Ug9SH8aeFaV+vp+pVxZZVXKvaJNw
# wrK6dZlqczKU0RBEEC7fgvMHhOZ0O21x4i0MG+4g1ckgHWMpLc7sXk7Ik/ghYZs0
# 6wXGXuxbGrzryc/NrDRAX7F6Zu53yEioZldXn1RYjgwrt0+nMNlW7sp7XeOtyU9e
# 5TXnMcvak17cjo+A2raRmECQecN4x7axxLVqGDgDEI3Y1DekLgV9iPWCPhCRcKtV
# gkEy19sEcypukQF8IUzUvK4bA3VdeGbZOjFEmjNAvwjXWkmkwuapoGfdpCe8oU85
# tRFYF/ckXEaPZPfBaYh2mHY9WV1CdoeJl2l6SPDgohIbZpp0yt5LHucOY67m1O+S
# kjqePdwA5EUlibaaRBkrfsCUtNJhbesz2cXfSwQAzH0clcOP9yGyshG3u3/y1Yxw
# LEFgqrFjGESVGnZifvaAsPvoZKYz0YkH4b235kOkGLimdwHhD5QMIR2yVCkliWzl
# DlJRR3S+Jqy2QXXeeqxfjT/JvNNBERJb5RBQ6zHFynIWIgnffEx1P2PsIV/EIFFr
# b7GrhotPwtZFX50g/KEexcCPorF+CiaZ9eRpL5gdLfXZqbId5RsCAwEAAaOCATow
# ggE2MA8GA1UdEwEB/wQFMAMBAf8wHQYDVR0OBBYEFOzX44LScV1kTN8uZz/nupiu
# HA9PMB8GA1UdIwQYMBaAFEXroq/0ksuCMS1Ri6enIZ3zbcgPMA4GA1UdDwEB/wQE
# AwIBhjB5BggrBgEFBQcBAQRtMGswJAYIKwYBBQUHMAGGGGh0dHA6Ly9vY3NwLmRp
# Z2ljZXJ0LmNvbTBDBggrBgEFBQcwAoY3aHR0cDovL2NhY2VydHMuZGlnaWNlcnQu
# Y29tL0RpZ2lDZXJ0QXNzdXJlZElEUm9vdENBLmNydDBFBgNVHR8EPjA8MDqgOKA2
# hjRodHRwOi8vY3JsMy5kaWdpY2VydC5jb20vRGlnaUNlcnRBc3N1cmVkSURSb290
# Q0EuY3JsMBEGA1UdIAQKMAgwBgYEVR0gADANBgkqhkiG9w0BAQwFAAOCAQEAcKC/
# Q1xV5zhfoKN0Gz22Ftf3v1cHvZqsoYcs7IVeqRq7IviHGmlUIu2kiHdtvRoU9BNK
# ei8ttzjv9P+Aufih9/Jy3iS8UgPITtAq3votVs/59PesMHqai7Je1M/RQ0SbQyHr
# lnKhSLSZy51PpwYDE3cnRNTnf+hZqPC/Lwum6fI0POz3A8eHqNJMQBk1RmppVLC4
# oVaO7KTVPeix3P0c2PR3WlxUjG/voVA9/HYJaISfb8rbII01YBwCA8sgsKxYoA5A
# Y8WYIsGyWfVVa88nq2x2zm8jLfR+cWojayL/ErhULSd+2DrZ8LaHlv1b0VysGMNN
# n3O3AamfV6peKOK5lDGCA3YwggNyAgEBMHcwYzELMAkGA1UEBhMCVVMxFzAVBgNV
# BAoTDkRpZ2lDZXJ0LCBJbmMuMTswOQYDVQQDEzJEaWdpQ2VydCBUcnVzdGVkIEc0
# IFJTQTQwOTYgU0hBMjU2IFRpbWVTdGFtcGluZyBDQQIQC65mvFq6f5WHxvnpBOMz
# BDANBglghkgBZQMEAgEFAKCB0TAaBgkqhkiG9w0BCQMxDQYLKoZIhvcNAQkQAQQw
# HAYJKoZIhvcNAQkFMQ8XDTI0MTEyMjA1NTQwNFowKwYLKoZIhvcNAQkQAgwxHDAa
# MBgwFgQU29OF7mLb0j575PZxSFCHJNWGW0UwLwYJKoZIhvcNAQkEMSIEIFjjem3T
# LkkBs1D2ciGcDrJPzMOqhgGaqitclxqM2B5VMDcGCyqGSIb3DQEJEAIvMSgwJjAk
# MCIEIHZ2n6jyYy8fQws6IzCu1lZ1/tdz2wXWZbkFk5hDj5rbMA0GCSqGSIb3DQEB
# AQUABIICAHvQPOC56dWzNzRTLXo1SqWffk+2ZJG+zFMszqnvBPIHHxrUDV8d5Jvg
# Vomhjig48WwKRU4gopxK0pZHWousnbnAZ12vlD0CXfN6fos1nkZ3iTHrEqY4SD+d
# ind+vwuT/7vZS+UQyQTDC8L3Mb3sGDO4pPAfywYDhUCFqq506AsxCg6yq913TYHe
# AbHHC8VkSFPC2Y+O3f9RS1tLbaeT1j96Pa05hLVAeXjCu1BDy3zK9A8tQzqU+AvM
# I9S8FzuGakHWaa24Pp9nX46VBJR93lDgNWILUOF37zgwjLx04rWAkP/4OnJP2PBS
# Minhyu3OZufo1HgIFWSV7B/lGVYVBy4h4dl28ilsDK06Jq0lAyCxJcTuvo7tsrw4
# Ha5rIm3viJ7Z+2IAcBt4DBD5ViwFID5HkkDmSlP5yGIu2GewsFHDsRJ98cO28Jje
# qFUcwhURERQA2m5fKPa0/1e2rPapPSgD/vDG4l/z9/5fPbxUOXHKjnVMZcen52LJ
# k6hEMCADTC+QqtS/A2r7vQoKNIaLMmYT0hvFwiHX4gxFq1JoXsOSl44MWk72Pj+V
# fAoaAx8Kudn0jldCedRjg+cigHDy5yE0PRHVse3IX8r1HlG+ii20ig3CTgeliCF8
# j3E9x9FEu08IBi9Y8IPzF+wY0flfyHArkVm0X2lTcl27W+EbBSg2
# SIG # End signature block
