﻿<# Copyright © 2023. Cloud Software Group, Inc. All Rights Reserved. #>
# In order to support both Windows 7 and Windows 10, Get-ScheduledTask cannot be used. Original code was using schtasks, but since it's language specific, it had to be replaced with COM object Schedule.Service. 
$CTXOESchTasks_COMSchedule = New-Object –ComObject ("Schedule.Service")
$CTXOESchTasks_COMSchedule.Connect("localhost")

# Function to return all subfolders for scheduled tasks
Function Get-CTXOESchTasksSubfolders ($Folder) {
    [Array]$m_Subfolders = @()
    
    If ($Folder.Path -eq "\") {$m_Subfolders += $Folder}
    
    ForEach ($m_Folder in  $Folder.GetFolders(1)) {
        $m_Subfolders += $m_Folder;
        ForEach ($m_Subfolder in $(Get-CTXOESchTasksSubfolders -Folder $m_Folder)) {
            $m_Subfolders += $m_Subfolder;
        }
    }
    Return $m_Subfolders;
}

# function to return all scheduled tasks
Function Get-CTXOESchTasksTasks () {
    [Array]$m_Tasks = @()
    ForEach ($m_Subfolder in Get-CTXOESchTasksSubfolders -Folder $CTXOESchTasks_COMSchedule.GetFolder("\")) {
        $m_Tasks += $m_Subfolder.GetTasks(1);
    }
    Return $m_Tasks;
}

# Generate cache of all scheduled tasks
$CTXOESchTasks_Cache = Get-CTXOESchTasksTasks;
$CTXOESchTasks_IsCacheValid = $true;

# Check if leading and trailing "\" are available, if not append and return modified string
Function Test-CTXOESchTasksBackslashes ([string]$Path) {
    If ($Path -notlike "\*") {$Path = "\$Path"}

    If ($Path -notlike "*\") {$Path = "$Path\"}

    Return $Path
}

# Check if scheduled task exists
Function Test-CTXOESchTasksExist ([String]$Path) {
    Return $($CTXOESchTasks_Cache | Where-Object {$_.Path -eq "$Path"}) -is [Object]
}

# Check if scheduled task is enabled or disabled. Return $False for disabled, $True for enabled.
Function Test-CTXOESchTasksState ([String]$Path) {
    If ($CTXOESchTasks_IsCacheValid -eq $false) {$CTXOESchTasks_Cache = Get-CTXOESchTasksTasks};    
    Return [Boolean]$($CTXOESchTasks_Cache | Where-Object {$_.Path -eq $Path} | Select-Object -ExpandProperty Enabled)
}

Function Invoke-CTXOESchTasksExecuteInternal ([Xml.XmlElement]$Params, [Boolean]$RollbackSupported = $False) {


    [String]$m_Name = $Params.Name
    [String]$m_Path = Test-CTXOESchTasksBackslashes -Path $Params.Path
    [String]$m_State = $Params.Value
    [String]$m_FullPath = "$m_Path$m_Name"

    If ($m_State -ne "Disabled" -and $m_State -ne "Enabled") {Throw "Requested state is $m_State, which is not disabled or enabled"}

    [Boolean]$m_Exists = Test-CTXOESchTasksExist -Path $m_FullPath

    # If scheduled task does not exist, return $True if goal was to disable it, otherwise return $False
    If ($m_Exists -eq $False) {
        $Global:CTXOE_Result = $m_State -eq "Disabled"
        $Global:CTXOE_Details = "Scheduled task does not exist"
        Return
    }

    [Boolean]$m_CurrentState = Test-CTXOESchTasksState -Path $m_FullPath
    [Boolean]$m_DesiredState = $m_State -ne "Disabled"

    If ($m_DesiredState -eq $m_CurrentState) {
        $Global:CTXOE_Result = $True
        $Global:CTXOE_Details = "Scheduled Task already $($m_State)"
        Return
    } Else {

        $CTXOESchTasks_IsCacheValid = $false;
        [String]$m_RollbackState = ""

        If ($m_State -eq "Disabled") {
            schtasks /change /tn "$m_FullPath" /disable | Out-Null
            $CTXOESchTasks_IsCacheValid = $False;
            $m_RollbackState = "Enabled";
        } Else {
            schtasks /change /tn "$m_FullPath" /enable | Out-Null
            $CTXOESchTasks_IsCacheValid = $False;
            $m_RollbackState = "Disabled"
        }

        [Boolean]$m_CurrentState = Test-CTXOESchTasksState -Path $m_FullPath
    
        If ($m_DesiredState -eq $m_CurrentState) {
            $Global:CTXOE_Result = $True
            $Global:CTXOE_Details = "Scheduled Task has been $($m_State)"

            # System has been changed. Report it and generate a rollback element.
            $Global:CTXOE_SystemChanged = $true;
            If ($RollbackSupported) {
                [Xml.XmlDocument]$m_RollbackElement = CTXOE\ConvertTo-CTXOERollbackElement -Element $Params
                $m_RollbackElement.rollbackparams.value = $m_RollbackState
                $Global:CTXOE_ChangeRollbackParams = $m_RollbackElement
            }

            Return
        } Else {
            $Global:CTXOE_Result = $False
            $Global:CTXOE_Details = "Failed to set $($m_Name) to $($m_State) state"
            Return
        }
    }
    Return
}

Function Invoke-CTXOESchTasksAnalyze ([Xml.XmlElement]$Params) {

    [String]$m_Name = $Params.Name
    [String]$m_Path = Test-CTXOESchTasksBackslashes -Path $Params.Path
    [String]$m_State = $Params.Value
    [String]$m_FullPath = "$m_Path$m_Name"

    # If scheduled task does not exist, return $True if goal was to disable it, otherwise return $False
    If ($(Test-CTXOESchTasksExist -Path $m_FullPath) -eq $False) {
        $Global:CTXOE_Result = $m_State -eq "Disabled"
        $Global:CTXOE_Details = "Scheduled task does not exist"
        Return
    }

    [Boolean]$m_CurrentState = Test-CTXOESchTasksState -Path $m_FullPath

    If ($m_State -eq "Disabled" -and $m_CurrentState -eq $False) {
        $Global:CTXOE_Result = $True
        $Global:CTXOE_Details = "Scheduled Task is disabled"
    } ElseIf ($m_State -ne "Disabled" -and $m_CurrentState -ne $False) {
        $Global:CTXOE_Result = $True
        $Global:CTXOE_Details = "Scheduled Task is enabled"
    } Else {
        $Global:CTXOE_Result = $False
        $Global:CTXOE_Details = "Scheduled Task is not in $($m_State) state"
    }

    Return
}

Function Invoke-CTXOESchTasksExecute ([Xml.XmlElement]$Params) {
    Invoke-CTXOESchTasksExecuteInternal -Params $Params -RollbackSupported $true
}

Function Invoke-CTXOESchTasksRollback ([Xml.XmlElement]$Params) {
    Invoke-CTXOESchTasksExecuteInternal -Params $Params -RollbackSupported $false
}
# SIG # Begin signature block
# MIIoowYJKoZIhvcNAQcCoIIolDCCKJACAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCAcGAgN+nlATIWt
# 2HXDtEN6D6HNyL7l8rmdmkbcnoPUdqCCDcAwggawMIIEmKADAgECAhAIrUCyYNKc
# TJ9ezam9k67ZMA0GCSqGSIb3DQEBDAUAMGIxCzAJBgNVBAYTAlVTMRUwEwYDVQQK
# EwxEaWdpQ2VydCBJbmMxGTAXBgNVBAsTEHd3dy5kaWdpY2VydC5jb20xITAfBgNV
# BAMTGERpZ2lDZXJ0IFRydXN0ZWQgUm9vdCBHNDAeFw0yMTA0MjkwMDAwMDBaFw0z
# NjA0MjgyMzU5NTlaMGkxCzAJBgNVBAYTAlVTMRcwFQYDVQQKEw5EaWdpQ2VydCwg
# SW5jLjFBMD8GA1UEAxM4RGlnaUNlcnQgVHJ1c3RlZCBHNCBDb2RlIFNpZ25pbmcg
# UlNBNDA5NiBTSEEzODQgMjAyMSBDQTEwggIiMA0GCSqGSIb3DQEBAQUAA4ICDwAw
# ggIKAoICAQDVtC9C0CiteLdd1TlZG7GIQvUzjOs9gZdwxbvEhSYwn6SOaNhc9es0
# JAfhS0/TeEP0F9ce2vnS1WcaUk8OoVf8iJnBkcyBAz5NcCRks43iCH00fUyAVxJr
# Q5qZ8sU7H/Lvy0daE6ZMswEgJfMQ04uy+wjwiuCdCcBlp/qYgEk1hz1RGeiQIXhF
# LqGfLOEYwhrMxe6TSXBCMo/7xuoc82VokaJNTIIRSFJo3hC9FFdd6BgTZcV/sk+F
# LEikVoQ11vkunKoAFdE3/hoGlMJ8yOobMubKwvSnowMOdKWvObarYBLj6Na59zHh
# 3K3kGKDYwSNHR7OhD26jq22YBoMbt2pnLdK9RBqSEIGPsDsJ18ebMlrC/2pgVItJ
# wZPt4bRc4G/rJvmM1bL5OBDm6s6R9b7T+2+TYTRcvJNFKIM2KmYoX7BzzosmJQay
# g9Rc9hUZTO1i4F4z8ujo7AqnsAMrkbI2eb73rQgedaZlzLvjSFDzd5Ea/ttQokbI
# YViY9XwCFjyDKK05huzUtw1T0PhH5nUwjewwk3YUpltLXXRhTT8SkXbev1jLchAp
# QfDVxW0mdmgRQRNYmtwmKwH0iU1Z23jPgUo+QEdfyYFQc4UQIyFZYIpkVMHMIRro
# OBl8ZhzNeDhFMJlP/2NPTLuqDQhTQXxYPUez+rbsjDIJAsxsPAxWEQIDAQABo4IB
# WTCCAVUwEgYDVR0TAQH/BAgwBgEB/wIBADAdBgNVHQ4EFgQUaDfg67Y7+F8Rhvv+
# YXsIiGX0TkIwHwYDVR0jBBgwFoAU7NfjgtJxXWRM3y5nP+e6mK4cD08wDgYDVR0P
# AQH/BAQDAgGGMBMGA1UdJQQMMAoGCCsGAQUFBwMDMHcGCCsGAQUFBwEBBGswaTAk
# BggrBgEFBQcwAYYYaHR0cDovL29jc3AuZGlnaWNlcnQuY29tMEEGCCsGAQUFBzAC
# hjVodHRwOi8vY2FjZXJ0cy5kaWdpY2VydC5jb20vRGlnaUNlcnRUcnVzdGVkUm9v
# dEc0LmNydDBDBgNVHR8EPDA6MDigNqA0hjJodHRwOi8vY3JsMy5kaWdpY2VydC5j
# b20vRGlnaUNlcnRUcnVzdGVkUm9vdEc0LmNybDAcBgNVHSAEFTATMAcGBWeBDAED
# MAgGBmeBDAEEATANBgkqhkiG9w0BAQwFAAOCAgEAOiNEPY0Idu6PvDqZ01bgAhql
# +Eg08yy25nRm95RysQDKr2wwJxMSnpBEn0v9nqN8JtU3vDpdSG2V1T9J9Ce7FoFF
# UP2cvbaF4HZ+N3HLIvdaqpDP9ZNq4+sg0dVQeYiaiorBtr2hSBh+3NiAGhEZGM1h
# mYFW9snjdufE5BtfQ/g+lP92OT2e1JnPSt0o618moZVYSNUa/tcnP/2Q0XaG3Ryw
# YFzzDaju4ImhvTnhOE7abrs2nfvlIVNaw8rpavGiPttDuDPITzgUkpn13c5Ubdld
# AhQfQDN8A+KVssIhdXNSy0bYxDQcoqVLjc1vdjcshT8azibpGL6QB7BDf5WIIIJw
# 8MzK7/0pNVwfiThV9zeKiwmhywvpMRr/LhlcOXHhvpynCgbWJme3kuZOX956rEnP
# LqR0kq3bPKSchh/jwVYbKyP/j7XqiHtwa+aguv06P0WmxOgWkVKLQcBIhEuWTatE
# QOON8BUozu3xGFYHKi8QxAwIZDwzj64ojDzLj4gLDb879M4ee47vtevLt/B3E+bn
# KD+sEq6lLyJsQfmCXBVmzGwOysWGw/YmMwwHS6DTBwJqakAwSEs0qFEgu60bhQji
# WQ1tygVQK+pKHJ6l/aCnHwZ05/LWUpD9r4VIIflXO7ScA+2GRfS0YW6/aOImYIbq
# yK+p/pQd52MbOoZWeE4wggcIMIIE8KADAgECAhAExKVRXTiJBY6ZBfjtmlFwMA0G
# CSqGSIb3DQEBCwUAMGkxCzAJBgNVBAYTAlVTMRcwFQYDVQQKEw5EaWdpQ2VydCwg
# SW5jLjFBMD8GA1UEAxM4RGlnaUNlcnQgVHJ1c3RlZCBHNCBDb2RlIFNpZ25pbmcg
# UlNBNDA5NiBTSEEzODQgMjAyMSBDQTEwHhcNMjQwMzA4MDAwMDAwWhcNMjUwMzA3
# MjM1OTU5WjCBjzELMAkGA1UEBhMCVVMxEDAOBgNVBAgTB0Zsb3JpZGExGDAWBgNV
# BAcTD0ZvcnQgTGF1ZGVyZGFsZTEdMBsGA1UEChMUQ2l0cml4IFN5c3RlbXMsIElu
# Yy4xFjAUBgNVBAsTDUNpdHJpeCBYZW5BcHAxHTAbBgNVBAMTFENpdHJpeCBTeXN0
# ZW1zLCBJbmMuMIIBojANBgkqhkiG9w0BAQEFAAOCAY8AMIIBigKCAYEA+gJ84IK2
# t0aBgBuXB3yfqoC+riwIEohXTQHVqNvJ7v3xiiIYD8PvzYLhaQq2dG4euY1AOhXL
# 0Uc2rNLcQZKEB4PslYrrorg+x5iRJl2k9XltDEpDbqJJqkUELY6d7GEIpp4ChdGf
# NJuGU9GCAgiml88tzZMOjtAqcLIXSRSOugQgjUOR+EvfeLlMTLs/ADjyOi898+0m
# TtdE6Sy/a2bn/fgsJaU8O2Kn78YiHX1N6lklKKBzO5tYfRD2VZhOn2kx6PUI+n+R
# ckwE22o0hBhckBeBGNW2DUrABDLwYuNOEWZEGx1KC2UDO4p+eDmxANjC/+q9t53d
# Z4V1Y/qL9CRnin0JNowVcLT3zGqeebgIag2Pqsh6m3lvASiy7DpmDtyaTCwv/aLd
# 1x5TZvUPiKcfvMOSCRxTP+a7L4DdjuqcQxWENk1kNEZjZa/ZS0BDWalklp6Ji9ki
# JpUKxB5C6GD/Lns+hbdp/2XHnEsqGbeQBCiW7ftXO3ugffq+/Bz1CelPAgMBAAGj
# ggIDMIIB/zAfBgNVHSMEGDAWgBRoN+Drtjv4XxGG+/5hewiIZfROQjAdBgNVHQ4E
# FgQUh00+M0JEuzdBGq4PW6l3UIoBQsIwPgYDVR0gBDcwNTAzBgZngQwBBAEwKTAn
# BggrBgEFBQcCARYbaHR0cDovL3d3dy5kaWdpY2VydC5jb20vQ1BTMA4GA1UdDwEB
# /wQEAwIHgDATBgNVHSUEDDAKBggrBgEFBQcDAzCBtQYDVR0fBIGtMIGqMFOgUaBP
# hk1odHRwOi8vY3JsMy5kaWdpY2VydC5jb20vRGlnaUNlcnRUcnVzdGVkRzRDb2Rl
# U2lnbmluZ1JTQTQwOTZTSEEzODQyMDIxQ0ExLmNybDBToFGgT4ZNaHR0cDovL2Ny
# bDQuZGlnaWNlcnQuY29tL0RpZ2lDZXJ0VHJ1c3RlZEc0Q29kZVNpZ25pbmdSU0E0
# MDk2U0hBMzg0MjAyMUNBMS5jcmwwgZQGCCsGAQUFBwEBBIGHMIGEMCQGCCsGAQUF
# BzABhhhodHRwOi8vb2NzcC5kaWdpY2VydC5jb20wXAYIKwYBBQUHMAKGUGh0dHA6
# Ly9jYWNlcnRzLmRpZ2ljZXJ0LmNvbS9EaWdpQ2VydFRydXN0ZWRHNENvZGVTaWdu
# aW5nUlNBNDA5NlNIQTM4NDIwMjFDQTEuY3J0MAkGA1UdEwQCMAAwDQYJKoZIhvcN
# AQELBQADggIBAI1e2H/UCCob4N14i0h2Z1Yi7dInt7RICuwc9yiMcDeGvlvtgcJT
# zp/CyQwpDh58/WGxR0HX3TuLjLh0Foxzm6aSgTLnyvI65jugv24L7yu1Bl5n7WR1
# x4tqtvVI1DOCTWNN9BqokeBTuT7NVwDPyoLWxz8f3GqnERJ/xe4DJOv9RXre6sy0
# NF9a+d4oqvEJ5r03WOpI6jO3YG6hyxCF7611ZhVdvBuw0ZfVgXPH1n/l+kGb4Mdc
# s0kJ6ny/auQ3Cm0lpCxrjh+LxwrxYtk1s/iZMd9T+C5vjUzH1LWhqlPur0ur05yU
# cG61A8pPwKLE02FdW9b6miAPknpUI6o6EgdU1KRCCjtYoR1iBSagB75k0RZBcgor
# B6mQKcFF5EwRhw5rSJdn1r7rTjRgVTzgRjgHL7QBY27yshp9rTwqHzO9VUmfTxg1
# 4UT5dcZ3xGc8U4j+gMSjNXmJJEpRaNtuh1QDXoQCwC0fM/9r4BxIlS50nuWrhnBL
# zSbrdxbmDK+iW5UU2oCKs3zHbrPmqQgWd4CDi/g/kUEruRIL+HMS1eAdZlyZ8DCw
# H8Qwv6+T6we/iDDbHtxLy7TjGsYChqEgaVOT7gj++iIdoHq3+mb1bxtDgovELXFN
# EwOxXT5mt7qZlc4ApoUTdruNzS9aMofoIB2SIVo/P6FCryDE5YxVmv6zMYIaOTCC
# GjUCAQEwfTBpMQswCQYDVQQGEwJVUzEXMBUGA1UEChMORGlnaUNlcnQsIEluYy4x
# QTA/BgNVBAMTOERpZ2lDZXJ0IFRydXN0ZWQgRzQgQ29kZSBTaWduaW5nIFJTQTQw
# OTYgU0hBMzg0IDIwMjEgQ0ExAhAExKVRXTiJBY6ZBfjtmlFwMA0GCWCGSAFlAwQC
# AQUAoIHQMBkGCSqGSIb3DQEJAzEMBgorBgEEAYI3AgEEMBwGCisGAQQBgjcCAQsx
# DjAMBgorBgEEAYI3AgEVMC8GCSqGSIb3DQEJBDEiBCBhXEeCmPICJO/jqygjMI4Z
# MFb6hMUGk54nWhFlsMV2vjBkBgorBgEEAYI3AgEMMVYwVKA4gDYAQwBpAHQAcgBp
# AHgAIABTAHUAcABwAG8AcgB0AGEAYgBpAGwAaQB0AHkAIABUAG8AbwBsAHOhGIAW
# aHR0cDovL3d3dy5jaXRyaXguY29tIDANBgkqhkiG9w0BAQEFAASCAYDuLrLWb+In
# 7GQb5HEfMK92KkB09LLKey7NRHnu4zBrHqzKazz2gcWn1bIouVMqM/YrbzxesVVI
# tRr6t6pA2Z+ZicafDBYDHLzJ5ID+s5P94uZdDkrl0wda7rwWDe/k/an9Ef8tMSgg
# yZtti/n1o1Q+Zx6ThPZGQvtTQtxWAcIUig5S0+/5HZj2Nc6hbWlKL6DZuG1vb7lI
# gN99U6BH3xT+x52lghRY5WLWenXTxsnKIEk6RcKlDT0TwkyAWXihFM8a91LpLNzx
# upN8f4p7OKTVlTr/yP7jZHsF0VZC+C6cx8pXSpp7Aaqz1MJkYsg+L86DlgTkCz00
# mZ36AcnPXNBl2xCYxrpTXnLqiBnf5jLsADUeZuiD4x6yRjgTc/BiXQAQbAhzNPG9
# xCpwCVliV4cQPzcxsSdnGo5if/scllCQ0CMT5OKw6kSY38boUwieacdr0QScRaNg
# EKk4x3TXx1h/gXicia6Z64tBN1tgYuHV8PENZRcEbjVOeCqBCMPZHTahghc6MIIX
# NgYKKwYBBAGCNwMDATGCFyYwghciBgkqhkiG9w0BBwKgghcTMIIXDwIBAzEPMA0G
# CWCGSAFlAwQCAQUAMHgGCyqGSIb3DQEJEAEEoGkEZzBlAgEBBglghkgBhv1sBwEw
# MTANBglghkgBZQMEAgEFAAQgE3wUBzhsVlrkWNVZ0CDRM9hZCJNroZL+xyrETSL0
# 2BkCEQCo6bp5b3rqcbvgoxHvH9ctGA8yMDI0MTEyMjA1NTM1NFqgghMDMIIGvDCC
# BKSgAwIBAgIQC65mvFq6f5WHxvnpBOMzBDANBgkqhkiG9w0BAQsFADBjMQswCQYD
# VQQGEwJVUzEXMBUGA1UEChMORGlnaUNlcnQsIEluYy4xOzA5BgNVBAMTMkRpZ2lD
# ZXJ0IFRydXN0ZWQgRzQgUlNBNDA5NiBTSEEyNTYgVGltZVN0YW1waW5nIENBMB4X
# DTI0MDkyNjAwMDAwMFoXDTM1MTEyNTIzNTk1OVowQjELMAkGA1UEBhMCVVMxETAP
# BgNVBAoTCERpZ2lDZXJ0MSAwHgYDVQQDExdEaWdpQ2VydCBUaW1lc3RhbXAgMjAy
# NDCCAiIwDQYJKoZIhvcNAQEBBQADggIPADCCAgoCggIBAL5qc5/2lSGrljC6W23m
# WaO16P2RHxjEiDtqmeOlwf0KMCBDEr4IxHRGd7+L660x5XltSVhhK64zi9CeC9B6
# lUdXM0s71EOcRe8+CEJp+3R2O8oo76EO7o5tLuslxdr9Qq82aKcpA9O//X6QE+Ac
# aU/byaCagLD/GLoUb35SfWHh43rOH3bpLEx7pZ7avVnpUVmPvkxT8c2a2yC0WMp8
# hMu60tZR0ChaV76Nhnj37DEYTX9ReNZ8hIOYe4jl7/r419CvEYVIrH6sN00yx49b
# oUuumF9i2T8UuKGn9966fR5X6kgXj3o5WHhHVO+NBikDO0mlUh902wS/Eeh8F/UF
# aRp1z5SnROHwSJ+QQRZ1fisD8UTVDSupWJNstVkiqLq+ISTdEjJKGjVfIcsgA4l9
# cbk8Smlzddh4EfvFrpVNnes4c16Jidj5XiPVdsn5n10jxmGpxoMc6iPkoaDhi6Jj
# Hd5ibfdp5uzIXp4P0wXkgNs+CO/CacBqU0R4k+8h6gYldp4FCMgrXdKWfM4N0u25
# OEAuEa3JyidxW48jwBqIJqImd93NRxvd1aepSeNeREXAu2xUDEW8aqzFQDYmr9ZO
# Nuc2MhTMizchNULpUEoA6Vva7b1XCB+1rxvbKmLqfY/M/SdV6mwWTyeVy5Z/JkvM
# FpnQy5wR14GJcv6dQ4aEKOX5AgMBAAGjggGLMIIBhzAOBgNVHQ8BAf8EBAMCB4Aw
# DAYDVR0TAQH/BAIwADAWBgNVHSUBAf8EDDAKBggrBgEFBQcDCDAgBgNVHSAEGTAX
# MAgGBmeBDAEEAjALBglghkgBhv1sBwEwHwYDVR0jBBgwFoAUuhbZbU2FL3Mpdpov
# dYxqII+eyG8wHQYDVR0OBBYEFJ9XLAN3DigVkGalY17uT5IfdqBbMFoGA1UdHwRT
# MFEwT6BNoEuGSWh0dHA6Ly9jcmwzLmRpZ2ljZXJ0LmNvbS9EaWdpQ2VydFRydXN0
# ZWRHNFJTQTQwOTZTSEEyNTZUaW1lU3RhbXBpbmdDQS5jcmwwgZAGCCsGAQUFBwEB
# BIGDMIGAMCQGCCsGAQUFBzABhhhodHRwOi8vb2NzcC5kaWdpY2VydC5jb20wWAYI
# KwYBBQUHMAKGTGh0dHA6Ly9jYWNlcnRzLmRpZ2ljZXJ0LmNvbS9EaWdpQ2VydFRy
# dXN0ZWRHNFJTQTQwOTZTSEEyNTZUaW1lU3RhbXBpbmdDQS5jcnQwDQYJKoZIhvcN
# AQELBQADggIBAD2tHh92mVvjOIQSR9lDkfYR25tOCB3RKE/P09x7gUsmXqt40ouR
# l3lj+8QioVYq3igpwrPvBmZdrlWBb0HvqT00nFSXgmUrDKNSQqGTdpjHsPy+Laal
# TW0qVjvUBhcHzBMutB6HzeledbDCzFzUy34VarPnvIWrqVogK0qM8gJhh/+qDEAI
# dO/KkYesLyTVOoJ4eTq7gj9UFAL1UruJKlTnCVaM2UeUUW/8z3fvjxhN6hdT98Vr
# 2FYlCS7Mbb4Hv5swO+aAXxWUm3WpByXtgVQxiBlTVYzqfLDbe9PpBKDBfk+rabTF
# DZXoUke7zPgtd7/fvWTlCs30VAGEsshJmLbJ6ZbQ/xll/HjO9JbNVekBv2Tgem+m
# LptR7yIrpaidRJXrI+UzB6vAlk/8a1u7cIqV0yef4uaZFORNekUgQHTqddmsPCEI
# YQP7xGxZBIhdmm4bhYsVA6G2WgNFYagLDBzpmk9104WQzYuVNsxyoVLObhx3Ruga
# EGru+SojW4dHPoWrUhftNpFC5H7QEY7MhKRyrBe7ucykW7eaCuWBsBb4HOKRFVDc
# rZgdwaSIqMDiCLg4D+TPVgKx2EgEdeoHNHT9l3ZDBD+XgbF+23/zBjeCtxz+dL/9
# NWR6P2eZRi7zcEO1xwcdcqJsyz/JceENc2Sg8h3KeFUCS7tpFk7CrDqkMIIGrjCC
# BJagAwIBAgIQBzY3tyRUfNhHrP0oZipeWzANBgkqhkiG9w0BAQsFADBiMQswCQYD
# VQQGEwJVUzEVMBMGA1UEChMMRGlnaUNlcnQgSW5jMRkwFwYDVQQLExB3d3cuZGln
# aWNlcnQuY29tMSEwHwYDVQQDExhEaWdpQ2VydCBUcnVzdGVkIFJvb3QgRzQwHhcN
# MjIwMzIzMDAwMDAwWhcNMzcwMzIyMjM1OTU5WjBjMQswCQYDVQQGEwJVUzEXMBUG
# A1UEChMORGlnaUNlcnQsIEluYy4xOzA5BgNVBAMTMkRpZ2lDZXJ0IFRydXN0ZWQg
# RzQgUlNBNDA5NiBTSEEyNTYgVGltZVN0YW1waW5nIENBMIICIjANBgkqhkiG9w0B
# AQEFAAOCAg8AMIICCgKCAgEAxoY1BkmzwT1ySVFVxyUDxPKRN6mXUaHW0oPRnkyi
# baCwzIP5WvYRoUQVQl+kiPNo+n3znIkLf50fng8zH1ATCyZzlm34V6gCff1DtITa
# EfFzsbPuK4CEiiIY3+vaPcQXf6sZKz5C3GeO6lE98NZW1OcoLevTsbV15x8GZY2U
# KdPZ7Gnf2ZCHRgB720RBidx8ald68Dd5n12sy+iEZLRS8nZH92GDGd1ftFQLIWhu
# NyG7QKxfst5Kfc71ORJn7w6lY2zkpsUdzTYNXNXmG6jBZHRAp8ByxbpOH7G1WE15
# /tePc5OsLDnipUjW8LAxE6lXKZYnLvWHpo9OdhVVJnCYJn+gGkcgQ+NDY4B7dW4n
# JZCYOjgRs/b2nuY7W+yB3iIU2YIqx5K/oN7jPqJz+ucfWmyU8lKVEStYdEAoq3ND
# zt9KoRxrOMUp88qqlnNCaJ+2RrOdOqPVA+C/8KI8ykLcGEh/FDTP0kyr75s9/g64
# ZCr6dSgkQe1CvwWcZklSUPRR8zZJTYsg0ixXNXkrqPNFYLwjjVj33GHek/45wPmy
# MKVM1+mYSlg+0wOI/rOP015LdhJRk8mMDDtbiiKowSYI+RQQEgN9XyO7ZONj4Kbh
# PvbCdLI/Hgl27KtdRnXiYKNYCQEoAA6EVO7O6V3IXjASvUaetdN2udIOa5kM0jO0
# zbECAwEAAaOCAV0wggFZMBIGA1UdEwEB/wQIMAYBAf8CAQAwHQYDVR0OBBYEFLoW
# 2W1NhS9zKXaaL3WMaiCPnshvMB8GA1UdIwQYMBaAFOzX44LScV1kTN8uZz/nupiu
# HA9PMA4GA1UdDwEB/wQEAwIBhjATBgNVHSUEDDAKBggrBgEFBQcDCDB3BggrBgEF
# BQcBAQRrMGkwJAYIKwYBBQUHMAGGGGh0dHA6Ly9vY3NwLmRpZ2ljZXJ0LmNvbTBB
# BggrBgEFBQcwAoY1aHR0cDovL2NhY2VydHMuZGlnaWNlcnQuY29tL0RpZ2lDZXJ0
# VHJ1c3RlZFJvb3RHNC5jcnQwQwYDVR0fBDwwOjA4oDagNIYyaHR0cDovL2NybDMu
# ZGlnaWNlcnQuY29tL0RpZ2lDZXJ0VHJ1c3RlZFJvb3RHNC5jcmwwIAYDVR0gBBkw
# FzAIBgZngQwBBAIwCwYJYIZIAYb9bAcBMA0GCSqGSIb3DQEBCwUAA4ICAQB9WY7A
# k7ZvmKlEIgF+ZtbYIULhsBguEE0TzzBTzr8Y+8dQXeJLKftwig2qKWn8acHPHQfp
# PmDI2AvlXFvXbYf6hCAlNDFnzbYSlm/EUExiHQwIgqgWvalWzxVzjQEiJc6VaT9H
# d/tydBTX/6tPiix6q4XNQ1/tYLaqT5Fmniye4Iqs5f2MvGQmh2ySvZ180HAKfO+o
# vHVPulr3qRCyXen/KFSJ8NWKcXZl2szwcqMj+sAngkSumScbqyQeJsG33irr9p6x
# eZmBo1aGqwpFyd/EjaDnmPv7pp1yr8THwcFqcdnGE4AJxLafzYeHJLtPo0m5d2aR
# 8XKc6UsCUqc3fpNTrDsdCEkPlM05et3/JWOZJyw9P2un8WbDQc1PtkCbISFA0LcT
# JM3cHXg65J6t5TRxktcma+Q4c6umAU+9Pzt4rUyt+8SVe+0KXzM5h0F4ejjpnOHd
# I/0dKNPH+ejxmF/7K9h+8kaddSweJywm228Vex4Ziza4k9Tm8heZWcpw8De/mADf
# IBZPJ/tgZxahZrrdVcA6KYawmKAr7ZVBtzrVFZgxtGIJDwq9gdkT/r+k0fNX2bwE
# +oLeMt8EifAAzV3C+dAjfwAL5HYCJtnwZXZCpimHCUcr5n8apIUP/JiW9lVUKx+A
# +sDyDivl1vupL0QVSucTDh3bNzgaoSv27dZ8/DCCBY0wggR1oAMCAQICEA6bGI75
# 0C3n79tQ4ghAGFowDQYJKoZIhvcNAQEMBQAwZTELMAkGA1UEBhMCVVMxFTATBgNV
# BAoTDERpZ2lDZXJ0IEluYzEZMBcGA1UECxMQd3d3LmRpZ2ljZXJ0LmNvbTEkMCIG
# A1UEAxMbRGlnaUNlcnQgQXNzdXJlZCBJRCBSb290IENBMB4XDTIyMDgwMTAwMDAw
# MFoXDTMxMTEwOTIzNTk1OVowYjELMAkGA1UEBhMCVVMxFTATBgNVBAoTDERpZ2lD
# ZXJ0IEluYzEZMBcGA1UECxMQd3d3LmRpZ2ljZXJ0LmNvbTEhMB8GA1UEAxMYRGln
# aUNlcnQgVHJ1c3RlZCBSb290IEc0MIICIjANBgkqhkiG9w0BAQEFAAOCAg8AMIIC
# CgKCAgEAv+aQc2jeu+RdSjwwIjBpM+zCpyUuySE98orYWcLhKac9WKt2ms2uexuE
# DcQwH/MbpDgW61bGl20dq7J58soR0uRf1gU8Ug9SH8aeFaV+vp+pVxZZVXKvaJNw
# wrK6dZlqczKU0RBEEC7fgvMHhOZ0O21x4i0MG+4g1ckgHWMpLc7sXk7Ik/ghYZs0
# 6wXGXuxbGrzryc/NrDRAX7F6Zu53yEioZldXn1RYjgwrt0+nMNlW7sp7XeOtyU9e
# 5TXnMcvak17cjo+A2raRmECQecN4x7axxLVqGDgDEI3Y1DekLgV9iPWCPhCRcKtV
# gkEy19sEcypukQF8IUzUvK4bA3VdeGbZOjFEmjNAvwjXWkmkwuapoGfdpCe8oU85
# tRFYF/ckXEaPZPfBaYh2mHY9WV1CdoeJl2l6SPDgohIbZpp0yt5LHucOY67m1O+S
# kjqePdwA5EUlibaaRBkrfsCUtNJhbesz2cXfSwQAzH0clcOP9yGyshG3u3/y1Yxw
# LEFgqrFjGESVGnZifvaAsPvoZKYz0YkH4b235kOkGLimdwHhD5QMIR2yVCkliWzl
# DlJRR3S+Jqy2QXXeeqxfjT/JvNNBERJb5RBQ6zHFynIWIgnffEx1P2PsIV/EIFFr
# b7GrhotPwtZFX50g/KEexcCPorF+CiaZ9eRpL5gdLfXZqbId5RsCAwEAAaOCATow
# ggE2MA8GA1UdEwEB/wQFMAMBAf8wHQYDVR0OBBYEFOzX44LScV1kTN8uZz/nupiu
# HA9PMB8GA1UdIwQYMBaAFEXroq/0ksuCMS1Ri6enIZ3zbcgPMA4GA1UdDwEB/wQE
# AwIBhjB5BggrBgEFBQcBAQRtMGswJAYIKwYBBQUHMAGGGGh0dHA6Ly9vY3NwLmRp
# Z2ljZXJ0LmNvbTBDBggrBgEFBQcwAoY3aHR0cDovL2NhY2VydHMuZGlnaWNlcnQu
# Y29tL0RpZ2lDZXJ0QXNzdXJlZElEUm9vdENBLmNydDBFBgNVHR8EPjA8MDqgOKA2
# hjRodHRwOi8vY3JsMy5kaWdpY2VydC5jb20vRGlnaUNlcnRBc3N1cmVkSURSb290
# Q0EuY3JsMBEGA1UdIAQKMAgwBgYEVR0gADANBgkqhkiG9w0BAQwFAAOCAQEAcKC/
# Q1xV5zhfoKN0Gz22Ftf3v1cHvZqsoYcs7IVeqRq7IviHGmlUIu2kiHdtvRoU9BNK
# ei8ttzjv9P+Aufih9/Jy3iS8UgPITtAq3votVs/59PesMHqai7Je1M/RQ0SbQyHr
# lnKhSLSZy51PpwYDE3cnRNTnf+hZqPC/Lwum6fI0POz3A8eHqNJMQBk1RmppVLC4
# oVaO7KTVPeix3P0c2PR3WlxUjG/voVA9/HYJaISfb8rbII01YBwCA8sgsKxYoA5A
# Y8WYIsGyWfVVa88nq2x2zm8jLfR+cWojayL/ErhULSd+2DrZ8LaHlv1b0VysGMNN
# n3O3AamfV6peKOK5lDGCA3YwggNyAgEBMHcwYzELMAkGA1UEBhMCVVMxFzAVBgNV
# BAoTDkRpZ2lDZXJ0LCBJbmMuMTswOQYDVQQDEzJEaWdpQ2VydCBUcnVzdGVkIEc0
# IFJTQTQwOTYgU0hBMjU2IFRpbWVTdGFtcGluZyBDQQIQC65mvFq6f5WHxvnpBOMz
# BDANBglghkgBZQMEAgEFAKCB0TAaBgkqhkiG9w0BCQMxDQYLKoZIhvcNAQkQAQQw
# HAYJKoZIhvcNAQkFMQ8XDTI0MTEyMjA1NTM1NFowKwYLKoZIhvcNAQkQAgwxHDAa
# MBgwFgQU29OF7mLb0j575PZxSFCHJNWGW0UwLwYJKoZIhvcNAQkEMSIEIOJZ1Imw
# cPswHps+XVSzTtRHaTUrxYpzx/BOvfcqB2q8MDcGCyqGSIb3DQEJEAIvMSgwJjAk
# MCIEIHZ2n6jyYy8fQws6IzCu1lZ1/tdz2wXWZbkFk5hDj5rbMA0GCSqGSIb3DQEB
# AQUABIICAAKHlbGmZVXuvdHn/ZSw7xfxFrKO13Jf/rmlUB39uCbz+kY9mdIRoT08
# SfVmob19rWrtf/A/qL+3QdBR7ZDDKV7NBlgCdbqQ4kq0msnhFfDb2OA8iMKH0z1v
# YQM29i2YK3dLnH4Sl07VK9suvCKVLr1XMPwF2e1ofgj4oCnnnVA1CtLWvzJFZakq
# scoyfRnKnBo5WY9dzqSBZHolhS3z7xZQQLxvMpAXUSMWOZac01raa5TT0qIk5OrN
# xZ10ATGWs/5B9G4FqOo5XWD/YV3SeY2SK6RoN8k4CV/RIG1d+zpMx8M2+2HEBxRK
# T+sYvlYE2a0cONfYeo5J/HM9N+NO0VwAziG3zitik3NRpPMa6OTy+LsZHvOCE6LZ
# WXfBmikY2BV42mzmUCVELXGLPgQtf/xOM6oq+ATcTJMbx1IvzvPUYQyRwBzbOB6J
# w2PMQrfMMeBSAta0iuK7M8hosiOUElfce83OGcPcUz6hm2gsyKu5IyCk4Ma9FuxK
# JDZsIU9ltTbELoFUe+QYU2XL76/7fDdRgL7E9+k4bHCrkvMHBiM5NimSUlrVfEbL
# PagjgAv14JPQ6rXyWRf55WoAi7tNoRh4EK2AoK/W++ynzKHh/BqtPpm3rV2x6yoz
# N0NGrVPZjT4nPOtbWnoyUKKOrkFz8Xp3QVOovRhTU7DUGG0O7gjB
# SIG # End signature block
