#pragma once

#include <vector>
#include <string>
#include <iostream>
#include <type_traits>
#include <cwctype>
#include <algorithm>
#include <Unknwnbase.h>
#include "Logger.h"

#define EPSILON 0.001

template <typename T, typename = std::enable_if_t<std::is_base_of_v<IUnknown, T>>>
void SafeRelease(T* &rpT)
{
    if (rpT)
    {
        rpT->Release();
        rpT = nullptr;
    }
}

template <typename ... T>
void ReleaseEach(T&& ... arg)
{
    // dummy initializer allows to iterate through template parameter args, c++11 friendly
     int expander[] = { 0, (SafeRelease(std::forward<T>(arg)), 0) ... };
}

// data structures to temprary keep camera and resolution data
struct CameraInfo
{
    std::wstring friendlyName;
    std::wstring devicePath;
};

struct ResolutionInfo
{
    std::wstring colorSpace;  // could be also compression, like H264
    uint32_t width = 0;
    uint32_t height = 0;
    double fps = 0;
    bool isVideo1 = true;  // simple DS filter graph does not play input with FORMAT_VideoInfo2(CLSID_KsDataTypeHandlerVideo2) correctly, requires manual addition of overlay filter, we'll skip input with VideoInfo2 header for now
    bool operator ==(const ResolutionInfo& other) const
    {

        return ToUCase(colorSpace) == ToUCase(other.colorSpace) && width == other.width && height == other.height && std::abs(other.fps - fps) <= EPSILON && isVideo1 == other.isVideo1;
    }
private:
	std::wstring ToUCase(std::wstring str) const
	{
		std::transform(str.begin(), str.end(), str.begin(), [](wchar_t ch) { return std::towupper(ch); });
		return str;

	}
};

static void ErrLog(const char* message, HRESULT hr)
{
        LOG << message << " Error: 0x" << std::hex << hr << "\r\n";
}

static DWORD getDwordFromBytes(BYTE b[])
{
	return (b[0]) | (b[1] << 8) | (b[2] << 16) | (b[3] << 24);
}


static LONG  dumpRegistryKeyHelper(HKEY hRootKey, int tabs, std::wstringstream& wss)
{
	DWORD i = 0;
	wchar_t  name[255] = { L'\0' };
	DWORD nameSize = 255;
	int ctabs = tabs + 1;
	DWORD Type;
	DWORD dataSize = 16383;
	HKEY hKey;
	auto  Data = std::make_unique< BYTE[]>(dataSize+1);
	LSTATUS result = ERROR_SUCCESS;
	//Get all values
	while (RegEnumValueW(hRootKey, i++, name, &nameSize, NULL, &Type, Data.get(), &dataSize) == ERROR_SUCCESS)
	{
		//other types skipped,could be added,if necessary
		if (Type == REG_DWORD)
			wss << std::wstring(ctabs * 2, L' ') << name << ":   0x" << std::setfill(L'0') << std::setw(8) << std::hex << getDwordFromBytes(Data.get()) << "\r\n";
		else if (Type == REG_SZ || Type == REG_EXPAND_SZ)
			wss << std::wstring(ctabs * 2, L' ') << name << L":   " << reinterpret_cast<wchar_t*>(Data.get()) << "\r\n";
		nameSize = 255;
		dataSize = 16383;
	}

	i = 0;
	//Get all subkeys
	while (RegEnumKeyExW(hRootKey, i++, name, &nameSize, NULL, NULL, NULL, NULL) == ERROR_SUCCESS)
	{
		wss << std::wstring(ctabs * 2, L' ') << L"[" << name << "]\r\n";
		result = RegOpenKeyExW(hRootKey, name, 0, KEY_READ, &hKey);
		if (result == ERROR_SUCCESS)
		{
			result = dumpRegistryKeyHelper(hKey, ctabs, wss);
			result = RegCloseKey(hKey);
		}
		if (result != ERROR_SUCCESS)
			return result;
		nameSize = 255;

	}
	return ERROR_SUCCESS;
}

static std::wstring dumpRegistryKey(HKEY hRootKey, LPCWSTR subKey)
{

	
	int tabs = 0;
	HKEY hKey;
	std::wstringstream wss;
	wss <<  subKey << "\r\n";
	LONG lError = RegOpenKeyExW(hRootKey, subKey, 0, KEY_READ, &hKey);
	if (lError == ERROR_SUCCESS)
	{
		lError = dumpRegistryKeyHelper(hKey, tabs, wss);
		lError = RegCloseKey(hKey);
	}
	if (lError != ERROR_SUCCESS)
		return std::wstring(L"Failed to read registry.");
	return wss.str();
}