#pragma once
#include <vector>
#include <string>
#include <wmsdkidl.h>
#include <wmcodecdsp.h>
#include <uuids.h>
#include <sstream>
#include <stack>
#include <iomanip>


struct GMediaFormat {
    std::wstring name;
    GUID media_format;
};

//List of some usual color space/compressor guids 
const std::vector<GMediaFormat> GMediaFormats =
{
{ L"RGB8", MEDIASUBTYPE_RGB8 },
{ L"RGB32", MEDIASUBTYPE_RGB32 },
{ L"RGB24", MEDIASUBTYPE_RGB24 },
{ L"RGB565", MEDIASUBTYPE_RGB565 },
{ L"RGB555", MEDIASUBTYPE_RGB555 },
{ L"YUV420P", MEDIASUBTYPE_IYUV },
{ L"YUV422P", MEDIASUBTYPE_YUYV },
{ L"YUV411", MEDIASUBTYPE_Y411 },
{ L"YUV411P", MEDIASUBTYPE_Y41P },
{ L"YUV410P", MEDIASUBTYPE_YVU9 },
{ L"YUY2", MEDIASUBTYPE_YUY2 },
{ L"MJPG", MEDIASUBTYPE_MJPG },
{ L"UYVY", MEDIASUBTYPE_UYVY },
{ L"I420", WMMEDIASUBTYPE_I420 },
{ L"H264", MEDIASUBTYPE_H264},
{ L"h264", MEDIASUBTYPE_h264},
{ L"AVC1", MEDIASUBTYPE_AVC1},
{ L"NV12", MEDIASUBTYPE_NV12}
};


std::wstring  GetCSName(const GUID& subtype)
{
    std::wstring RetVal;
    auto idx = std::find_if(GMediaFormats.begin(), GMediaFormats.end(), [&subtype](const GMediaFormat& gMediaFormat) {return gMediaFormat.media_format == subtype; });
    if (idx != GMediaFormats.end())
    {
        RetVal = idx->name;
    }
    else  // if not found in known formats list, prints subtype GUID
    {
        LPOLESTR guidStr = nullptr;
        auto hr = StringFromCLSID(subtype, &guidStr);
        RetVal = std::wstring(guidStr);
        CoTaskMemFree(guidStr);
    }
    return RetVal;
}


static void DeleteMediaType(AM_MEDIA_TYPE* pmt)
{
    if (pmt != nullptr)
    {
        if (pmt->cbFormat != 0)
        {
            CoTaskMemFree((PVOID)pmt->pbFormat);
            pmt->cbFormat = 0;
            pmt->pbFormat = nullptr;
        }
        if (pmt->pUnk != nullptr)
        {
            // pUnk should not be used.
            pmt->pUnk->Release();
            pmt->pUnk = nullptr;
        }
        CoTaskMemFree(pmt);
    }
}

HRESULT AddFilterByCLSID(IGraphBuilder* pGraph, REFGUID clsid, IBaseFilter** ppFilter, LPCWSTR wszName)
{
    LOG << "Enter " << __FUNCTION__ << " Filter name: " << wszName << ", clsid: " << clsid  << "\r\n";
    *ppFilter = nullptr;
    IBaseFilter* pFilter = nullptr;
    auto hr = S_OK;
    auto done = [&]() {SafeRelease(pFilter); return hr; };
    if ((hr = CoCreateInstance(clsid, nullptr, CLSCTX_INPROC_SERVER, IID_PPV_ARGS(&pFilter)))<0)
    {
        return done();
    }
    if ((hr = pGraph->AddFilter(pFilter, wszName)) < 0)
    {
        return done();
    }
    *ppFilter = pFilter;
    (*ppFilter)->AddRef();
    LOG << "Leave " << __FUNCTION__ << "\r\n";
    return done();
}

//get text representation of list of filters selected for preview graph
std::wstring GetAddedFiltersList(IGraphBuilder* pGraphBuilder)
{
    std::wostringstream oss;
    IEnumFilters* pFilterEnum = nullptr;
    HRESULT hr = pGraphBuilder->EnumFilters(&pFilterEnum);
    std::stack <std::wstring> filters;
    if (SUCCEEDED(hr))
    {
        FILTER_INFO filterInfo;
        IBaseFilter* pFilter = nullptr;
        while (S_OK == pFilterEnum->Next(1, &pFilter, nullptr))
        {
            pFilter->QueryFilterInfo(&filterInfo);
            filters.emplace(filterInfo.achName);
            SafeRelease(pFilter);
        }
        SafeRelease(pFilterEnum);
    }
    while (!filters.empty())    
    {
        oss << filters.top() << L"\r\n";
        filters.pop();
    }
        return oss.str();
}

//get text representation of selected stream properties
std::wstring getMediaTypeDescription(const AM_MEDIA_TYPE* pmtConfig,  const  VIDEOINFOHEADER* pVidHeadr)
{
    std::wostringstream oss;

    oss << "Input Width:  " << pVidHeadr->bmiHeader.biWidth << "\r\n";
    oss << "Input Height: " << abs(pVidHeadr->bmiHeader.biHeight) << "\r\n";
    oss << "Color space/Compression: " << GetCSName(pmtConfig->subtype) << "\r\n";
    oss << "FPS: " << std::setprecision(2) << std::noshowpoint << 10000000.0 / (double)(pVidHeadr->AvgTimePerFrame) << "\r\n";
    oss << "BitRate: " << pVidHeadr->dwBitRate << "\r\n";
    oss << "Fixed Size Samples: " << pmtConfig->bFixedSizeSamples << "\r\n";
    oss << "Temporal Compression: " << pmtConfig->bTemporalCompression << "\r\n";
    oss << "Sample Size: " << pmtConfig->lSampleSize << "\r\n";

    return oss.str();

}

//Gets first  available pin from the filter. There could be more pins on capture
//filter, which complying with the given PIN_DIRECTION (like separate pins for YUV2,
//MJPEG for one and H264 for another on some WebCams) but commonly 
//YUV2,RGB24,I420,MJPEG are on the first pin.
HRESULT GetPin(IBaseFilter* pFilter, PIN_DIRECTION PinDir, IPin** ppPin)
{
    IEnumPins* pEnum = nullptr;
    IPin* pPin = nullptr;
    HRESULT    hr;

    if (ppPin == nullptr)
    {
        return E_POINTER;
    }
    hr = pFilter->EnumPins(&pEnum);
    if (FAILED(hr))
    {
        return hr;
    }
    while (pEnum->Next(1, &pPin, 0) == S_OK)
    {
        PIN_DIRECTION PinDirThis;
        hr = pPin->QueryDirection(&PinDirThis);
        if (FAILED(hr))
        {
            ReleaseEach(pPin, pEnum);
            return hr;
        }
        if (PinDir == PinDirThis)
        {
            *ppPin = pPin;
            (*ppPin)->AddRef();
            ReleaseEach(pPin, pEnum);
            return S_OK;
        }
        // Release the pin pointer to reuse.
        SafeRelease(pPin);
    }
    SafeRelease(pEnum);
    return E_FAIL;
}
