#include <wmcodecdsp.h>
#include <algorithm>
#include "DSWebCamPreview.h"
#include "DSUtils.h"
#include "PlayerWindow.h"


#pragma comment(lib, "strmiids")
#pragma comment(lib, "wmcodecdspuuid")

extern HWND hWnd;  // Preview window handle

DSWebCamPreview::~DSWebCamPreview()
{
    FilterCleanup();
    ReleaseEach(pGraphBuilder, pMoniker, pVideoWindow, pMediaControl, pMediaEvent);
    CoUninitialize();
    LOG << "DS WebcamPreview finished\r\n";
}


HRESULT DSWebCamPreview::ListWebCams()
{
    cameraInfos.clear();
    return ListDevices();
}

HRESULT DSWebCamPreview::SetWebCam(ULONG index)
{
    selectedCameraIndex = index;
    LOG << "Selected camera : " << cameraInfos[index].friendlyName << "\r\n\tDevice Path: " << cameraInfos[index].devicePath << "\r\n";
    return ActivateDevice(selectedCameraIndex);
}


HRESULT DSWebCamPreview::ListWebCamResolutions()
{
    resolutionInfos.clear();
    return ListCaptureFormats();
}





HRESULT DSWebCamPreview::Start(HANDLE& threadHandle, ULONG index)
{
    LOG << "Enter " << __FUNCTION__ << "\r\n";
    auto hr = S_OK;

    if ((hr = SetCaptureFormat(index)) != S_OK)
    {
        LOG << "Failed to set selected capture resolution" << "\r\n";
        return hr;
    }

    if ((hr = CreatePlayerWindow(resolutionInfos[index], threadHandle)) != S_OK)
        return hr;
    if ((hr = SetupVideoWindow()) != S_OK)
        return hr;
    // Set the window handle used to process graph events
    if ((hr = pMediaEvent->SetNotifyWindow((OAHWND)hWnd, WM_GRAPHNOTIFY, 0)) != S_OK)
        return hr;
    hr = pMediaControl->Run();
    LOG << "Leave " << __FUNCTION__ << "\r\n";
    return hr;
}


HRESULT DSWebCamPreview::Stop()
{
    LOG << "Enter " << __FUNCTION__ << "\r\n";
    HRESULT hr = S_OK;
    if((hr = pMediaControl->StopWhenReady())!=S_OK)
        LOG <<"Capture rendering not stopped. "<< hr << "\n";
    if (pMediaEvent)
    {
        if ((hr = pMediaEvent->SetNotifyWindow(NULL, WM_GRAPHNOTIFY, 0)) != S_OK)
            LOG << "Video Window notifying is not stopped. " << hr << "\n";
    }
    // Release ownership of the video window.
    if (pVideoWindow)
    {
        pVideoWindow->put_Visible(OAFALSE);
        hr = pVideoWindow->put_Owner(NULL);        
    }
    LOG << "Leave " << __FUNCTION__ << "\r\n";
    return hr;
}


HRESULT DSWebCamPreview::GetInterfaces()
{
    LOG << "Enter " << __FUNCTION__ << "\r\n";
    auto hr = S_OK;
    auto ReleaseAll = [&]() {ReleaseEach(pGraphBuilder, pMediaControl, pVideoWindow, pMediaEvent); };

    // Create the Filter Graph Manager.
    if ((hr = CoCreateInstance(CLSID_FilterGraphPrivateThread, nullptr, CLSCTX_INPROC, IID_IGraphBuilder, reinterpret_cast<LPVOID*>(&pGraphBuilder))) !=S_OK)
    {
        ReleaseAll();
        return hr;
    }

    // Obtain interfaces for media control,event and Video Window
    if ((hr = pGraphBuilder->QueryInterface(IID_IMediaControl, reinterpret_cast<LPVOID*>(&pMediaControl))) !=S_OK)
    {
        ReleaseAll();
        return hr;
    }
    if ((hr = pGraphBuilder->QueryInterface(IID_IVideoWindow, reinterpret_cast<LPVOID*>(&pVideoWindow))) !=S_OK)
    {
        ReleaseAll();
        return hr;
    }
    if ((hr = pGraphBuilder->QueryInterface(IID_IMediaEventEx, reinterpret_cast<LPVOID*>(&pMediaEvent))) !=S_OK)
    {
        ReleaseAll();
    }
    LOG << "Leave " << __FUNCTION__ << "\r\n";
    return hr;
}


HRESULT DSWebCamPreview::GetMonikerEnumerator(IEnumMoniker** ppMonikerEnum) const
{
    LOG << "Enter " << __FUNCTION__ << "\r\n";
    auto hr = S_OK;
    ICreateDevEnum* pDevEnum = nullptr;
    IEnumMoniker* pMonikerEnum = nullptr;
    auto onFail = [&](const char* str ) { ErrLog(str, hr); ReleaseEach(pDevEnum, pMonikerEnum); return hr; };
    if ((hr = CoCreateInstance(CLSID_SystemDeviceEnum, nullptr, CLSCTX_INPROC, IID_ICreateDevEnum, (void**)&pDevEnum)) !=S_OK)
        return onFail("Failed to create device enumerator");
    if ((hr = pDevEnum->CreateClassEnumerator(CLSID_VideoInputDeviceCategory, &pMonikerEnum, 0)) != S_OK)
        return onFail("Failed to create video device enumerator");
    *ppMonikerEnum = pMonikerEnum;
    (*ppMonikerEnum)->AddRef();
    ReleaseEach(pDevEnum, pMonikerEnum);
    LOG << "Leave " << __FUNCTION__ << "\r\n";
    return hr;
}


HRESULT DSWebCamPreview::ListDevices()
{
    LOG << "Enter " << __FUNCTION__ << "\r\n";
    HRESULT hr;
    IEnumMoniker* pMonikerEnum = nullptr;
    IMoniker* pMonikerL = nullptr;
    IPropertyBag* pPropBag = nullptr;
    auto onFail = [&](const wchar_t* str) { LOG << str<<L"\n"; ReleaseEach(pMonikerEnum, pMonikerL, pPropBag); return hr; };

    if ((hr = GetMonikerEnumerator(&pMonikerEnum)) != S_OK)
        return onFail(L"Failed to get Moniker Enumerator");
    while (S_OK == pMonikerEnum->Next(1, &pMonikerL, nullptr))
    {
        CameraInfo cameraInfo;
        hr = pMonikerL->BindToStorage(nullptr, nullptr, IID_IPropertyBag, reinterpret_cast<void**>(&pPropBag));
        if (FAILED(hr))
        {
            SafeRelease(pMonikerL);
            cameraInfo.friendlyName = L"WebCam is not recognized.";
            continue;  // maybe next webcam kindly gives us its properties?
        }
        VARIANT var;
        VariantInit(&var);

        hr = pPropBag->Read(L"FriendlyName", &var, nullptr);
        if (SUCCEEDED(hr))
        {
            cameraInfo.friendlyName = std::wstring(var.bstrVal, SysStringLen(var.bstrVal));
            VariantClear(&var);
        }
        hr = pPropBag->Read(L"DevicePath", &var, nullptr);
        if (SUCCEEDED(hr))
        {
            cameraInfo.devicePath = std::wstring(var.bstrVal, SysStringLen(var.bstrVal));
            VariantClear(&var);
        }
        cameraInfos.push_back(cameraInfo);
        SafeRelease(pPropBag);
        SafeRelease(pMonikerL);
    }
    ReleaseEach(pMonikerEnum, pMonikerL, pPropBag);
    LOG << "Leave " << __FUNCTION__ << "\r\n";
    return hr;
}

HRESULT DSWebCamPreview::ActivateDevice(ULONG index)
{
    LOG << "Enter " << __FUNCTION__ << "\r\n";
    auto hr = S_OK;
    FilterCleanup();  // remove all filters from previously built graph, if any
    IEnumMoniker* pMonikerEnum = nullptr;
    auto onFail = [&]() { SafeRelease(pMonikerEnum); return hr; };
    if ((hr = GetMonikerEnumerator(&pMonikerEnum)) != S_OK)
        return onFail();
    if ((hr = pMonikerEnum->Reset()) !=S_OK)
        return onFail();
    //pMonikerEnum->Skip(index) - not for VDA
    // so just crawling through
    for (ULONG i = 0; i < index+1; i++)
    {
        SafeRelease(pMoniker);
        pMonikerEnum->Next(1, &pMoniker, nullptr);
    }
    SafeRelease(pMonikerEnum);
    LOG << "Leave " << __FUNCTION__ << "\r\n";
    return hr;
}

HRESULT DSWebCamPreview::ListCaptureFormats()
{
    auto hr = S_OK;
    LOG << "Enter " << __FUNCTION__ << "\r\n";
    IBaseFilter* pCaptureFilter = nullptr;
    IPin* pOutPin = nullptr;
    IEnumMediaTypes* pEnum = nullptr;

    auto onFail = [&](const char* str) { ErrLog( str, hr); ReleaseEach(pCaptureFilter, pOutPin, pEnum, pMoniker ); return hr; };
    hr = pMoniker->BindToObject(0, 0, IID_IBaseFilter, (void**)&pCaptureFilter);
    if (FAILED(hr)) return hr;

    if ((hr = GetPin(pCaptureFilter, PINDIR_OUTPUT, &pOutPin)) !=S_OK)
    {
        return onFail("Failed to get capture filter output pin.");
    }

    if ((hr = pOutPin->EnumMediaTypes(&pEnum)) !=S_OK)
    {
        return onFail("Failed to get Webcam resolutions list.");
    }

    AM_MEDIA_TYPE* pmtConfig = nullptr;
    while (S_OK == pEnum->Next(1, &pmtConfig, 0))
    {
        ResolutionInfo resolutionInfo;
        if (pmtConfig->majortype == MEDIATYPE_Video)
        {
            auto pVidHeadr = reinterpret_cast<VIDEOINFOHEADER*>(pmtConfig->pbFormat);
            resolutionInfo.width = pVidHeadr->bmiHeader.biWidth;
            resolutionInfo.height = abs(pVidHeadr->bmiHeader.biHeight);
            resolutionInfo.colorSpace = GetCSName(pmtConfig->subtype);
            resolutionInfo.fps = 10000000.0 / (double)(pVidHeadr->AvgTimePerFrame);
            resolutionInfo.isVideo1 = pmtConfig->formattype != FORMAT_VideoInfo2;
        }
        DeleteMediaType(pmtConfig);
        resolutionInfos.push_back(resolutionInfo);
    }
    pmtConfig = nullptr;

    ReleaseEach(pCaptureFilter, pOutPin, pEnum);

    LOG << "Leave " << __FUNCTION__ << "\r\n";
    return hr;
}



HRESULT DSWebCamPreview::SetCaptureFormat(ULONG index)
{
    IBaseFilter* pCaptureFilter = nullptr;
    IPin* pCaptureOutPin = nullptr;
    IBaseFilter* pVideoRendererFilter = nullptr;
    IPin* pRenderInPin = nullptr;
    IAMStreamConfig* pConfig = nullptr;
    IBaseFilter* pVideoDecoderFilter = nullptr;
    IPin* pDecoderInPin = nullptr;
    IPin* pDecoderOutPin = nullptr;    
    IEnumMediaTypes* pEnum = nullptr;
    HRESULT hr;

    LOG << "Enter " << __FUNCTION__ << "\r\n";

    //set GraphBuilder pin connection logging
    HANDLE rPipe = nullptr;
    HANDLE wPipe = nullptr;
    SECURITY_ATTRIBUTES secattr;
    ZeroMemory(&secattr, sizeof(secattr));
    secattr.nLength = sizeof(secattr);
    secattr.bInheritHandle = TRUE;
    if (CreatePipe(&rPipe, &wPipe, &secattr, 8 * 1024) && wPipe != INVALID_HANDLE_VALUE && wPipe != nullptr) 
        pGraphBuilder->SetLogFile((DWORD_PTR)wPipe);


    // We do not need multiple filter chains at the same time,
    // so we removing from graph previous filter set, if it exists.
    // if we are playing same color space/compression input,
    // than stop,change resolution, start with same graph could be considered
    FilterCleanup();
    auto ReleaseAll = [&]()
    {
        ReleaseEach(pCaptureFilter,
                    pCaptureOutPin,
                    pVideoRendererFilter,
                    pRenderInPin,
                    pConfig,
                    pVideoDecoderFilter,
                    pDecoderInPin,
                    pDecoderOutPin,
                    pEnum);
        if (rPipe != nullptr)
            CloseHandle(rPipe);
        if (wPipe != nullptr)
            CloseHandle(wPipe);
        return hr;
    };
    auto onFail = [&](const char* str) { ErrLog(str, hr); return ReleaseAll(); };
    if (!pMoniker)
    {
        ReleaseAll();
        return E_NOT_SET;
    }
    if ((hr = pMoniker->BindToObject(nullptr, nullptr, IID_IBaseFilter, (void**)&pCaptureFilter)) !=S_OK)
    {
        ReleaseAll();
        return hr;
    }

    if ((hr = GetPin(pCaptureFilter, PINDIR_OUTPUT, &pCaptureOutPin)) !=S_OK)
    {
        return onFail("Failed to get output capture pin.");
    }
    if ((hr = pCaptureOutPin->EnumMediaTypes(& pEnum)) !=S_OK)
    {
        return onFail("Failed to enumerate output capture pin media types.");
    }


    if ((hr = pCaptureOutPin->QueryInterface(IID_IAMStreamConfig, (void**)&pConfig)) !=S_OK)
    {
        return onFail("Failed to get MediaStream Config interface.");
    }

    AM_MEDIA_TYPE* pMediaTypeSet = nullptr;
    if (index != 0)
        pEnum->Skip(index);
    if ((hr = pEnum->Next(1, &pMediaTypeSet, 0)) !=S_OK)
    {
        DeleteMediaType(pMediaTypeSet);
        return ReleaseAll();
    }

    const auto& colorSp = pMediaTypeSet->subtype;
    

    auto pVidHeadr = reinterpret_cast<VIDEOINFOHEADER*>(pMediaTypeSet->pbFormat);

    LOG << "\r\n-------------Webcam properties------------\r\n"
        << getMediaTypeDescription(pMediaTypeSet, pVidHeadr)
        << "------------------------------------------------------\r\n";

    if ((hr = pGraphBuilder->AddFilter(pCaptureFilter, L"Video Capture")) !=S_OK)
    {
        DeleteMediaType(pMediaTypeSet);
        return onFail("Failed to add Video Capture filter to the graph.");
    }

    // This part is only to support current DS Webcam redirection filter implementation on VDA.
    // Normally capture filter should accept IAMStreamConfig::SetFormat and graph should 
    // be built automatically after it.
    if (colorSp == MEDIASUBTYPE_YUY2 || colorSp == MEDIASUBTYPE_I420)
    {
        if ((hr = AddFilterByCLSID(pGraphBuilder, CLSID_AVIDec, &pVideoDecoderFilter, L"Avi Decoder")) !=S_OK)
        {
            DeleteMediaType(pMediaTypeSet);
            return onFail("Failed to add Avi Decoder filter to the graph.");
        }
    }
    else if (colorSp == MEDIASUBTYPE_MJPG)
    {
        if ((hr = AddFilterByCLSID(pGraphBuilder, CLSID_MjpegDec, &pVideoDecoderFilter, L"MJPEG Decoder")) !=S_OK)
        {
            DeleteMediaType(pMediaTypeSet);
            return onFail("Failed to add MJPEG Decoder filter to the graph.");
        }
    }
   
    
    if (pVideoDecoderFilter != nullptr)
    {
        if ((hr = GetPin(pVideoDecoderFilter, PINDIR_INPUT, &pDecoderInPin)) !=S_OK)
        {
            DeleteMediaType(pMediaTypeSet);
            return onFail("Failed to get decoder filter input pin.");
        }
        if ((hr = GetPin(pVideoDecoderFilter, PINDIR_OUTPUT, &pDecoderOutPin)) !=S_OK)
        {
            DeleteMediaType(pMediaTypeSet);
            return onFail("Failed to get decoder filter output pin.");
        }
        if ((hr = pGraphBuilder->ConnectDirect(pCaptureOutPin, pDecoderInPin, pMediaTypeSet)) !=S_OK)
        {
            DeleteMediaType(pMediaTypeSet);
            return onFail("Failed to connect capture and decoder filters.");
        }
    }   
    
    DeleteMediaType(pMediaTypeSet);

    static const  GUID h264guids[] = { MEDIASUBTYPE_AVC1, MEDIASUBTYPE_H264 , MEDIASUBTYPE_h264, MEDIASUBTYPE_X264, MEDIASUBTYPE_x264 }; 

    // Available renderers: CLSID_VideoRendererDefault, CLSID_VideoRenderer, CLSID_VideoMixingRenderer,
    // CLSID_VideoMixingRenderer9, CLSID_EnhancedVideoRenderer
    // CLSID_EnhancedVideoRenderer will be set only for  H264 format(H264 throws exception on default renderer)
    if (std::any_of(std::begin(h264guids), std::end(h264guids), [&](const GUID& guid) {return guid == colorSp; }))
    {
        if ((hr = AddFilterByCLSID(pGraphBuilder, CLSID_EnhancedVideoRenderer, &pVideoRendererFilter, L"Enhanced Video Renderer")) != S_OK)
        {
            return onFail("Failed to add Enhanced Video Renderer filter to the graph.");
        }
    }
    else
    {
        // CLSID_VideoRenderer plays RGB24 and I420, also YUY2 and MJPEG(after decoding) formats.
        // CLSID_VideoMixingRenderer and others does not play 
        // RGB24 with smart pin connect(at least in my case)
        if ((hr = AddFilterByCLSID(pGraphBuilder, CLSID_VideoRenderer, &pVideoRendererFilter, L"Video Renderer")) != S_OK)
        {
            return onFail("Failed to add Video Renderer filter to the graph.");
        }
    }


    if ((hr = GetPin(pVideoRendererFilter, PINDIR_INPUT, &pRenderInPin)) !=S_OK)
    {   
        return onFail("Failed to get Video Renderer input pin.");
    }

    // If you know exacly how you want to play specified capture input, you can add
    // more DS filters and connect them together with pGraphBuilder->ConnectDirect(.,.,.),
    // without relying on smart connection
    if (pDecoderOutPin != nullptr)
    {
        hr = pGraphBuilder->Connect(pDecoderOutPin, pRenderInPin);
    }
    else
    {
        hr = pGraphBuilder->Connect(pCaptureOutPin, pRenderInPin);        
    }   


    if (pGraphBuilder)
        pGraphBuilder->SetLogFile(NULL);
    if (wPipe != INVALID_HANDLE_VALUE && wPipe != nullptr)
    {
        CloseHandle(wPipe);
        wPipe = nullptr;
    }
    DWORD reDword;
    if (PeekNamedPipe(rPipe, nullptr, 0, nullptr, &reDword, nullptr))
    {
        std::wstring PinLogOutput(reDword, '\0');
        if (ReadFile(rPipe, (LPVOID)&PinLogOutput[0], reDword, &reDword, 0))
            LOG << "\r\n--------------Pin connection log-------------\r\n" 
            << PinLogOutput
            << "-------------------------------------------------------\r\n ";
    }
    if (rPipe != INVALID_HANDLE_VALUE && rPipe != nullptr)
    {
        CloseHandle(rPipe);
        rPipe = nullptr;
    }
    LOG << "\r\n--------------Filter chain-------------\r\n"
        << GetAddedFiltersList(pGraphBuilder)
        << "--------------------------------------------\r\n ";
    LOG << "Leave " << __FUNCTION__ << "\r\n";
    return ReleaseAll();
}

HRESULT DSWebCamPreview::SetupVideoWindow()
{
    LOG << "Enter " << __FUNCTION__ << "\r\n";
    HRESULT hr;
    // Set the video window to be a child of the main window
    if ((hr = pVideoWindow->put_Owner((OAHWND)hWnd)) !=S_OK)
        return hr;
    // Set video window style
    if ((hr = pVideoWindow->put_WindowStyle(WS_CHILD | WS_CLIPCHILDREN)) !=S_OK)
        return hr;
    RECT rcPW;  // Preview window client aria
    GetClientRect(hWnd, &rcPW);
    // move video window to adjust with preview window
    pVideoWindow->SetWindowPosition(0, 0, rcPW.right, rcPW.bottom);
    // Make the video window visible
    hr = pVideoWindow->put_Visible(OATRUE);
    LOG << "Leave " << __FUNCTION__ << "\r\n";
    return hr;
}

HRESULT DSWebCamPreview::FilterCleanup()
{
    LOG << "Enter " << __FUNCTION__ << "\r\n";
    IEnumFilters* pEnum = nullptr;
    HRESULT hr = pGraphBuilder->EnumFilters(&pEnum);
    if (SUCCEEDED(hr))
    {
        FILTER_INFO pInfo;
        IBaseFilter* pFilter = nullptr;
        while (S_OK == pEnum->Next(1, &pFilter, nullptr))
        {
            pFilter->QueryFilterInfo(&pInfo);
            pGraphBuilder->RemoveFilter(pFilter);
            pEnum->Reset();
            SafeRelease(pFilter);
        }       
    }
    SafeRelease(pEnum);
    LOG << "Leave " << __FUNCTION__ << "\r\n";
    return hr;
}

