#include "MFWebCamPreview.h"
#include "PlayerWindow.h"



#pragma comment(lib, "mf")
#pragma comment(lib, "mfplat")
#pragma comment(lib, "mfuuid")
#pragma comment(lib, "Strmiids")

extern HWND hWnd;  // Player window handle

MFWebCamPreview::~MFWebCamPreview()
{
    if (pSource)
        pSource->Shutdown();
    SafeRelease(pSource);
    SafeRelease(pSession);
    MFShutdown();
    CoUninitialize();
    LOG << "MF Webcam Preview Finished\r\n";
}

HRESULT MFWebCamPreview::ListWebCams()
{
    LOG << "Enter " << __FUNCTION__ << "\r\n";
    cameraInfos.clear();
    return ListDevices();
}

HRESULT MFWebCamPreview::SetWebCam(ULONG index)
{
    selectedCameraIndex = index;
    LOG << "Selected camera : " << cameraInfos[index].friendlyName << "\r\n\tDevice Path: " << cameraInfos[index].devicePath << "\r\n";
    return ActivateDevice(index);
}

HRESULT MFWebCamPreview::ListWebCamResolutions()
{
    resolutionInfos.clear();
    return ListCaptureFormats();
}



HRESULT MFWebCamPreview::Start(HANDLE& ThreadHandle,ULONG index)
{
    LOG << "Enter " << __FUNCTION__ << "\r\n";
    HRESULT hr;
    auto onFail = [&](const char* str) { ErrLog(str, hr);  return hr; };
    ActivateDevice(selectedCameraIndex);
    if ((hr = SetCaptureFormat(index)) != S_OK) 
        return onFail("Failed to set selected capture resolution");
    
    if ((hr = CreatePlayerWindow(resolutionInfos[index], ThreadHandle)) != S_OK)
        return onFail("Failed to initialise video window.");
    if ((hr = MFCreateMediaSession(nullptr, &pSession)) != S_OK)
        return onFail("Failed to create media session.");
    if ((hr = BuildTopology(hWnd)) != S_OK)
        return onFail("Failed to Build Topology");
    PROPVARIANT varStart;
    PropVariantInit(&varStart);
    hr = pSession->Start(&GUID_NULL, &varStart);
    PropVariantClear(&varStart);
    if (hr != S_OK)
        return onFail("Failed to start session");
    LOG << "Leave " << __FUNCTION__ << "\r\n";
    return hr;
}

HRESULT MFWebCamPreview::Stop()
{
    LOG << "Enter " << __FUNCTION__ << "\r\n";
    auto hr = S_OK;
    if (!pSession)
        return hr;
    if ((hr = pSession->Stop()) != S_OK)
        return hr;
    if ((hr = pSession->Shutdown()) != S_OK)
        return hr;
    if ((hr = pSource->Shutdown()) != S_OK)
        return hr;
    SafeRelease(pSession);
    SafeRelease(pSource);
    LOG << "Leave " << __FUNCTION__ << "\r\n";
    return S_OK;
}

HRESULT MFWebCamPreview::BuildTopology(HWND hWindow)
{
    LOG << "Enter " << __FUNCTION__ << "\r\n";
    IMFTopology* pTopology = nullptr;
    IMFActivate* pVideoActivate = nullptr;
    IMFTopologyNode* pVideoSourceNode = nullptr;
    IMFTopologyNode* pVideoSinkNode = nullptr;
    IMFPresentationDescriptor* pSourcePD = nullptr;
    IMFStreamDescriptor* pSourceSD = nullptr;
    BOOL fSelected = false;
    HRESULT hr;
    auto releaseAll = [&]() {ReleaseEach(pVideoActivate, pVideoSourceNode, pVideoSinkNode, pSourcePD, pSourceSD, pTopology); };
    auto onFail = [&](const char* str) { ErrLog( str, hr); releaseAll(); return hr; };
    if (!pSource)
        if ((hr=ActivateDevice(selectedCameraIndex)) != S_OK) //  just for case when we rerun same resolution without clicking on it, device was not activated
            return onFail("Failed to set and activate WebCam.");
    if ((hr = MFCreateTopology(&pTopology)) != S_OK)
        return onFail("Failed to create topology object.");
    // Presentation descriptor to select streams and to get information about the source content.
    if ((hr = pSource->CreatePresentationDescriptor(&pSourcePD)) != S_OK)
        return onFail("Failed to create presentation descriptor from source.");
    if ((hr = pSourcePD->GetStreamDescriptorByIndex(0, &fSelected, &pSourceSD)) != S_OK)
        return onFail("Failed to get stream descriptor from presentation descriptor.");
    if ((hr = AddSourceNode(pTopology, pSource, pSourcePD, pSourceSD, &pVideoSourceNode)) != S_OK)
        return onFail("Failed to add video source node");
    // Creates an activation object for the enhanced video renderer(EVR) media sink.
    // Can render: Reference stream : Progressive or interlaced YUV with no per-pixel alpha(such as NV12 or YUY2)
    // or progressive RGB. Substreams: Progressive YUV with per - pixel - alpha, such as AYUV or AI44.
    if ((hr = MFCreateVideoRendererActivate(hWindow, &pVideoActivate)) != S_OK)
        return onFail("Failed to create video renderer activate object.");
    if ((hr = AddOutputNode(pTopology, pVideoActivate, 0, &pVideoSinkNode)) != S_OK)
        return onFail("Failed to add video output node.");
    if ((hr = pVideoSourceNode->ConnectOutput(0, pVideoSinkNode, 0)) != S_OK)
        return onFail("Failed to connect video source and sink nodes.");
    if ((hr = pSession->SetTopology(1, pTopology)) != S_OK)
        return onFail("Failed to set topology for the session.");
    releaseAll();
    LOG << "Leave " << __FUNCTION__ << "\r\n";
    return hr;
}

 HRESULT MFWebCamPreview::AddSourceNode(IMFTopology* pTopology, IMFMediaSource* pMediaSource, IMFPresentationDescriptor* pPD, IMFStreamDescriptor* pSD, IMFTopologyNode** ppNode)
{
    LOG << "Enter " << __FUNCTION__ << "\r\n";
    IMFTopologyNode* pNode = nullptr;
    HRESULT hr;
    auto onFail = [&](const char* str, HRESULT hrl) { ErrLog( str, hrl); SafeRelease(pNode); return hrl; };
    if ((hr = MFCreateTopologyNode(MF_TOPOLOGY_SOURCESTREAM_NODE, &pNode)) != S_OK)
        return onFail("Failed to create topology node.", hr);
    if ((hr = pNode->SetUnknown(MF_TOPONODE_SOURCE, pMediaSource)) != S_OK)
        return onFail("Failed to set source on topology node.", hr);
    if ((hr = pNode->SetUnknown(MF_TOPONODE_PRESENTATION_DESCRIPTOR, pPD)) != S_OK)
        return onFail("Failed to set presentation descriptor on topology node.", hr);
    if ((hr = pNode->SetUnknown(MF_TOPONODE_STREAM_DESCRIPTOR, pSD)) != S_OK)
        return onFail("Failed to set stream descriptor on topology node.", hr);
    if ((hr = pTopology->AddNode(pNode)) != S_OK)
        return onFail("Failed to add source node to topology.", hr);

    // Return the pointer to the caller.
    *ppNode = pNode;
    (*ppNode)->AddRef();
    SafeRelease(pNode);
    LOG << "Leave " << __FUNCTION__ << "\r\n";
    return hr;
}

HRESULT MFWebCamPreview::AddOutputNode(IMFTopology* pTopology, IMFActivate* pActivate, DWORD dwId, IMFTopologyNode** ppNode)
{
    LOG << "Enter " << __FUNCTION__ << "\r\n";
    IMFTopologyNode* pNode = nullptr;
    HRESULT hr;
    auto onFail = [&](const char* str, HRESULT hrl) { ErrLog( str, hrl); SafeRelease(pNode); return hrl; };
    if ((hr = MFCreateTopologyNode(MF_TOPOLOGY_OUTPUT_NODE, &pNode)) != S_OK)
        return onFail("Failed to create topology node.", hr);
    if ((hr = pNode->SetObject(pActivate)) != S_OK)
        return onFail("Failed to set sink on topology node.", hr);
    if ((hr = pNode->SetUINT32(MF_TOPONODE_STREAMID, dwId)) != S_OK)
        return onFail("Failed to set stream ID on topology node.", hr);
    if ((hr = pNode->SetUINT32(MF_TOPONODE_NOSHUTDOWN_ON_REMOVE, FALSE)) != S_OK)
        return onFail("Failed to set no shutdown on topology node.", hr);
    if ((hr = pTopology->AddNode(pNode)) != S_OK)
        return onFail("Failed to add output node to the topology.", hr);

    // Return the pointer to the caller.
    *ppNode = pNode;
    (*ppNode)->AddRef();

    SafeRelease(pNode);
    LOG << "Leave " << __FUNCTION__ << "\r\n";
    return S_OK;
}

HRESULT MFWebCamPreview::ListDevices()
{
    LOG << "Enter " << __FUNCTION__ << "\r\n";
    IMFAttributes* pAttributes = nullptr;
    IMFActivate** ppDevices = nullptr;
    UINT32 count = 0;
    auto hr = S_OK;
    auto onFail = [&]() { LOG << std::hex << hr << "\n"; SafeRelease(pAttributes); return hr; };

    hr = MFCreateAttributes(&pAttributes, 1);
    if (hr != S_OK)
        return onFail();
    // We want video capture devices
    hr = pAttributes->SetGUID(MF_DEVSOURCE_ATTRIBUTE_SOURCE_TYPE, MF_DEVSOURCE_ATTRIBUTE_SOURCE_TYPE_VIDCAP_GUID);
    if (hr != S_OK)
        return onFail();
    // Enumerate devices.
    hr = MFEnumDeviceSources(pAttributes, &ppDevices, &count);
    if (hr != S_OK)
        return onFail();

    LPWSTR wName = nullptr;
    UINT32 stringLength;

    for (UINT32 i = 0; i < count; ++i)
    {
        CameraInfo ci;
        hr = ppDevices[i]->GetAllocatedString(MF_DEVSOURCE_ATTRIBUTE_FRIENDLY_NAME, &wName, &stringLength);
        if (hr != S_OK)
            onFail();
        ci.friendlyName = std::wstring(wName);
        CoTaskMemFree(wName);
        hr = ppDevices[i]->GetAllocatedString(MF_DEVSOURCE_ATTRIBUTE_SOURCE_TYPE_VIDCAP_SYMBOLIC_LINK, &wName, &stringLength);
        if (hr != S_OK)
            onFail();
        if (wName != nullptr)
            ci.devicePath = std::wstring(wName);
        CoTaskMemFree(wName);
        cameraInfos.push_back(ci);
    }
    for (DWORD i = 0; i < count; i++)
    {
        SafeRelease(ppDevices[i]);
    }
    CoTaskMemFree(ppDevices);
    SafeRelease(pAttributes);
    LOG << "Leave " << __FUNCTION__ << "\r\n";
    return hr;
}



HRESULT MFWebCamPreview::ActivateDevice(ULONG index)
{
    LOG << "Enter " << __FUNCTION__ << "\r\n";
    IMFAttributes* pAttributes = nullptr;
    IMFActivate** ppDevices = nullptr;
    UINT32 count = 0;
    auto hr = S_OK;
    auto onFail = [&]() {  LOG << std::hex << hr << "\r\n"; SafeRelease(pAttributes); if (pSource) pSource->Shutdown(); SafeRelease(pSource); return hr;  };

    hr = MFCreateAttributes(&pAttributes, 1);
    if (hr != S_OK)
        return onFail();
    // We want video capture devices
    hr = pAttributes->SetGUID(MF_DEVSOURCE_ATTRIBUTE_SOURCE_TYPE, MF_DEVSOURCE_ATTRIBUTE_SOURCE_TYPE_VIDCAP_GUID);
    if (hr != S_OK)
        return onFail();
    // Enumerate devices.
    hr = MFEnumDeviceSources(pAttributes, &ppDevices, &count);
    if (hr != S_OK)
        return onFail();
    if(pSource)
        pSource->Shutdown();
    SafeRelease(pSource);
    
    hr = ppDevices[index]->ActivateObject(IID_PPV_ARGS(&pSource));

    for (DWORD i = 0; i < count; i++)
    {
        SafeRelease(ppDevices[i]);
    }
    CoTaskMemFree(ppDevices);
    SafeRelease(pAttributes);
    LOG << "Leave " << __FUNCTION__ << "\r\n";
    return hr;
}

HRESULT MFWebCamPreview::GetMediaHandler(IMFMediaTypeHandler** ppHandler)
{
    LOG << "Enter " << __FUNCTION__ << "\r\n";
    IMFPresentationDescriptor* pPD = nullptr;
    IMFStreamDescriptor* pSD = nullptr;
    IMFMediaTypeHandler* pHandler = nullptr;
    DWORD sourceStreamCount = 0;
    HRESULT hr;
    auto ReleaseAll = [&]() {SafeRelease(pPD); SafeRelease(pSD); SafeRelease(pHandler); };
    auto onFail = [&](const char* str, HRESULT hrl) { ErrLog( str, hrl); ReleaseAll(); return hrl; };

    if ((hr = pSource->CreatePresentationDescriptor(&pPD)) != S_OK)
        return onFail("Failed to get media source's presentation descriptor.", hr);
    if ((hr = pPD->GetStreamDescriptorCount(&sourceStreamCount)) != S_OK)
        return onFail("Failed to get source stream count.", hr);
    if(sourceStreamCount<1)
        return onFail("No streams found.(req by SoQ)", hr);
    // Iterate over the available source streams and create renderers.
    for (DWORD i = 0; i < sourceStreamCount; i++)
    {
        BOOL fSelected = FALSE;
        GUID guidMajorType;

        if ((hr = pPD->GetStreamDescriptorByIndex(i, &fSelected, &pSD)) != S_OK)
            return onFail("Failed to get stream descriptor from presentation descriptor.", hr);
        if ((hr = pSD->GetMediaTypeHandler(&pHandler)) != S_OK)
            return onFail("Failed to create media type handler from presentation descriptor.", hr);
        if ((hr = pHandler->GetMajorType(&guidMajorType)) != S_OK)
            return onFail("Failed to get media type handler from source stream.", hr);
        if (guidMajorType == MFMediaType_Video && fSelected)  // we've got video handler!
        {
            *ppHandler = pHandler;
            (*ppHandler)->AddRef();
            break;
        }
    }
    ReleaseAll();
    LOG << "Leave " << __FUNCTION__ << "\r\n";
    return hr;
}
HRESULT MFWebCamPreview::ListCaptureFormats()
{
    LOG << "Enter " << __FUNCTION__ << "\r\n";
    IMFMediaTypeHandler* pHandler = nullptr;
    IMFMediaType* pType = nullptr;
    DWORD TypesCount = 0;
    UINT32 fpsNum = 0;
    UINT32 fpsDen = 0;
    auto ReleaseAll = [&]() {SafeRelease(pType); SafeRelease(pHandler); };
    auto onFail = [&](const char* str, HRESULT hr) { ErrLog( str, hr); ReleaseAll(); return hr; };
    auto hr = GetMediaHandler(&pHandler);
    if (FAILED(hr))
        return onFail("Cannot get MediaType Handler", hr);
    hr = pHandler->GetMediaTypeCount(&TypesCount);
    if (FAILED(hr))
        return onFail("Cannot get media types for the selected camera.Please reconnect and try again.", hr);
    for (DWORD i = 0; i < TypesCount; i++)
    {
        hr = pHandler->GetMediaTypeByIndex(i, &pType);
        if (FAILED(hr))
            return onFail("Incorrect media type index ", hr);
        ResolutionInfo rInfo;
        GUID subType;
        hr = pType->GetGUID(MF_MT_SUBTYPE, &subType);
        if (FAILED(hr))
        {
            LOG << "Failed to get video CS/encoder.\r\n";
        }
        std::wstring GUIDName(GetGUIDNameConst(subType));
        GUIDName = GUIDName.substr(GUIDName.find_last_of(L'_') + 1);
        rInfo.colorSpace = GUIDName;
        hr = MFGetAttributeSize(pType, MF_MT_FRAME_SIZE, &rInfo.width, &rInfo.height);
        if (FAILED(hr))
            LOG << "Failed to get frame size.\r\n";
        hr = MFGetAttributeRatio(pType, MF_MT_FRAME_RATE, &fpsNum, &fpsDen);
        if (FAILED(hr))
        {
            LOG << "Failed to get frame rate.\r\n";
            rInfo.fps = 0;
        }
        else
            rInfo.fps = fpsNum * 1.0 / fpsDen;
        resolutionInfos.push_back(rInfo);
        SafeRelease(pType);
    }
    ReleaseAll();
    LOG << "Leave " << __FUNCTION__ << "\r\n";
    return hr;
}

HRESULT MFWebCamPreview::SetCaptureFormat(ULONG index)
{
    LOG << "Enter " << __FUNCTION__ << "\r\n";
    IMFMediaTypeHandler* pHandler = nullptr;
    IMFMediaType* pType = nullptr;
    auto hr=E_FAIL;
    auto ReleaseAll = [&]() {SafeRelease(pType); SafeRelease(pHandler); };
    auto onFail = [&](const char* str) { ErrLog( str, hr); ReleaseAll(); return hr; };

    if ((hr = GetMediaHandler(&pHandler)) < 0)
    {
        return onFail("Cannot get MediaType Handler");
    }
    if ((hr = pHandler->GetMediaTypeByIndex(index, &pType)) !=S_OK)
    {
        return onFail("Cannot get MediaType");
    }
    if ((hr = pHandler->SetCurrentMediaType(pType))!=S_OK)
    {
        return onFail("Cannot set current media type");
    }

    LOG << "\r\n-------------Webcam properties------------\r\n"
        << GetMediaTypeDescription(pType)
        << "------------------------------------------------------\r\n";    
    ReleaseAll();
    LOG << "Leave " << __FUNCTION__ << "\r\n";
    return hr;
}
