
#include <memory>
#include <algorithm>
#include "resource.h"
#include "CTX.h"
#include "MFWebCamPreview.h"
#include "DSWebCamPreview.h"
#include "PlayerWindow.h"
#include <commctrl.h>


HWND hDlg = nullptr;

constexpr int MAX_LOADSTRING = 100;

enum class RenderEngine { DirectShow, MediaFoundation };
std::unique_ptr<WebCamPreview> webCamPreview = nullptr; 

// Global Variables:
WCHAR szTitle[MAX_LOADSTRING];        // The title bar text
WCHAR szWindowClass[MAX_LOADSTRING];  // the main window class name


struct MoveResizeControl
{
    int ID;
    bool doMoveResH; // allow move or resize horizontal
    bool doMoveResV;// allow move or resize vertical
    long rDist = 0;// distance between control element right and main window right
    long bDist = 0;// distance between control element bottom and main window bottom

    MoveResizeControl(int ID, bool doMoveResH, bool doMoveResV) :ID{ ID }, doMoveResH{ doMoveResH }, doMoveResV{ doMoveResV }
    {}

    void GetRelativeCtrlRect(HWND hWnd, RECT* elementRect) const
    {
        GetWindowRect(hWnd, elementRect);        
        ScreenToClient(hDlg, & ((LPPOINT)elementRect)[0]);
        ScreenToClient(hDlg, & ((LPPOINT)elementRect)[1]);
    }

    void SetRect()
    {
        HWND hwnditem;
        RECT hDlgClientRect;
        RECT controlRect;
        hwnditem = GetDlgItem(hDlg, ID);        
        GetRelativeCtrlRect(hwnditem, &controlRect);       
        GetClientRect(hDlg, &hDlgClientRect);
        if (doMoveResH && hDlgClientRect.right - controlRect.right >= 0)
            rDist = hDlgClientRect.right - controlRect.right;
        if (doMoveResV && hDlgClientRect.bottom - controlRect.bottom >= 0)
            bDist = hDlgClientRect.bottom - controlRect.bottom;
    }
    void stretch() const
    {
        RECT hDlgClientRect;
        RECT controlRect;
        GetClientRect(hDlg, &hDlgClientRect);  
        auto elementHwnd = GetDlgItem(hDlg, ID);
        GetRelativeCtrlRect(elementHwnd, &controlRect);
        auto width = doMoveResH ? hDlgClientRect.right - rDist - controlRect.left : controlRect.right - controlRect.left;
        auto  height = doMoveResV ? hDlgClientRect.bottom - bDist - controlRect.top : controlRect.bottom - controlRect.top;
        MoveWindow(elementHwnd, controlRect.left, controlRect.top, width, height, TRUE);

    }
    void move() const
     {
        RECT hDlgClientRect;
        RECT controlRect;
        GetClientRect(hDlg, &hDlgClientRect);
        auto elementHwnd = GetDlgItem(hDlg, ID);
        GetRelativeCtrlRect(elementHwnd, &controlRect);       
        auto width = controlRect.right - controlRect.left;
        auto  height = controlRect.bottom - controlRect.top;        
        auto left = doMoveResH ? hDlgClientRect.right - rDist - width : controlRect.left;
        auto top = doMoveResV ? hDlgClientRect.bottom - bDist - height : controlRect.top;       
        MoveWindow(elementHwnd, left, top, width, height, TRUE);

    }
};

std::vector<MoveResizeControl> controlsToResize
{
     MoveResizeControl(IDC_BUTTON_PREVIEW, true, true),
     MoveResizeControl(IDC_LIST_RES, false, true),
     MoveResizeControl(IDC_EDIT_INFO, true, true),
     MoveResizeControl(IDC_LOG_PROP, true, true)
};

void ErrorBoxMessage(LPCWSTR message)
{
    MessageBox(hDlg, message, TEXT("WebCamPreview"), MB_OK);
}

void SelectEngine(RenderEngine renderEngine)
{
    // CLEANUP
    HWND hwnditem;
    hwnditem = GetDlgItem(hDlg, IDC_LIST_CAMS);
    SendMessage(hwnditem, LB_RESETCONTENT, 0, 0);
    hwnditem = GetDlgItem(hDlg, IDC_LIST_RES);
    SendMessage(hwnditem, LB_RESETCONTENT, 0, 0);
    hwnditem = GetDlgItem(hDlg, IDC_EDIT_INFO);
    SendMessage(hwnditem, LB_RESETCONTENT, 0, 0);
    // close rendering engine,if one is running
    if (webCamPreview != nullptr)
        webCamPreview.reset(nullptr);
    // INIT new engine
    if (renderEngine == RenderEngine::MediaFoundation)
    {
        webCamPreview = std::make_unique<MFWebCamPreview>();
    }
    else if (renderEngine == RenderEngine::DirectShow)
    {
        webCamPreview = std::make_unique<DSWebCamPreview>();
    }
    // fill camera list
    hwnditem = GetDlgItem(hDlg, IDC_LIST_CAMS);
    if (webCamPreview->ListWebCams() != S_OK)
    {
        LOG<<"Failed to list webcams.\r\n";
        return;
    }
    auto cameras = webCamPreview->GetCameras();
    for (const auto& cam : cameras)
    {
        SendMessage(hwnditem, LB_ADDSTRING, 0, (LPARAM)cam.friendlyName.c_str());
    }
}




void SelectWebCam(ULONG camIndex)
{
    // CLEANUP
    auto hwndList = GetDlgItem(hDlg, IDC_LIST_RES);
    SendMessage(hwndList, LB_RESETCONTENT, 0, 0);
    auto hwndEdit = GetDlgItem(hDlg, IDC_EDIT_INFO);
    SendMessage(hwndEdit, LB_RESETCONTENT, 0, 0);
    if (webCamPreview->SetWebCam(camIndex) != S_OK)
    {
        LOG<<"Failed activate selected webcam\r\n";
        return;
    }
    if (webCamPreview->ListWebCamResolutions() != S_OK)
    {
        LOG << "Failed to get resolution list.\r\n";
        return;
    }
    const auto resolutions = webCamPreview->GetCameraResolutions();

    for (size_t i = 0; i < resolutions.size(); i++)
    {
        const auto& res = resolutions[i];
        if (resolutions[i].isVideo1)
        {
            std::wostringstream resolutionStr;
            resolutionStr << res.width << L"x" << res.height << L", " << res.colorSpace << L", " << std::setprecision(2) << std::noshowpoint << res.fps << L" fps";
            auto pos = (int)SendMessage(hwndList, LB_ADDSTRING, 0, (LPARAM)resolutionStr.str().c_str());
            SendMessage(hwndList, LB_SETITEMDATA, pos, (LPARAM)i);
        }
    }
}

bool IsListElementSelected(int nIDDlgItem)
{
    auto hwndList = GetDlgItem(hDlg, nIDDlgItem);
    return SendMessage(hwndList, LB_GETCURSEL, 0, 0) != LB_ERR;
}

bool StartPreview()
{
    auto hr = S_OK;
    HANDLE ThreadHandle;  // will not be used here, only for console

    //Check if we have selected engine, items in webcam and resolution lists
    if (!webCamPreview)
    {
        ErrorBoxMessage(TEXT("Please select Video engine."));
        return false;
    }
    if(!IsListElementSelected( IDC_LIST_CAMS))    
    {
        ErrorBoxMessage(TEXT("Please select WebCam."));
        return false;
    }   
    if (!IsListElementSelected(IDC_LIST_RES))
    {
        ErrorBoxMessage(TEXT("Please select capture resolution."));
        return false;

    }

    //Starting preview
    hr = webCamPreview->Start(ThreadHandle, webCamPreview->ResIndex());
    
    if (hr != S_OK && hr != S_FALSE)  // S_FALSE is OK too(for DS)
    {
        LOG << L"Faild to start Preview.Error: " << std::to_wstring(hr)<<"\r\n";
        return false;
    }
    return true;
}

void toggleControls(bool enable)
{
    int items[] = { IDC_RADIO_DS ,IDC_RADIO_MF,IDC_LIST_CAMS,IDC_LIST_RES,IDC_BUTTON_PREVIEW };
    std::for_each(std::begin(items), std::end(items), [&enable](int item) {HWND hwnditem = GetDlgItem(hDlg, item); EnableWindow(hwnditem, enable); });
}

/// <summary>
/// Alters Edit control default context menu.
/// </summary>
VOID CALLBACK ContextMenuProcCallback(HWINEVENTHOOK hWinEventHook, DWORD dwEvent, HWND hwnd, LONG idObject, LONG idChild, DWORD dwEventThread, DWORD dwmsEventTime)
{
    HMENU hMenu = (HMENU)SendMessage(hwnd, MN_GETHMENU, 0, 0);
    InsertMenu(hMenu, 6, MF_SEPARATOR | MF_BYPOSITION, 0, NULL);
    InsertMenu(hMenu, 7, MF_BYPOSITION, ID_CLEAR, L"Clear");
    InsertMenu(hMenu, 8, MF_BYPOSITION , ID_REGDUMP, L"CTX-Webcam registry info");   
    DeleteMenu(hMenu, WM_PASTE, 0);
    DeleteMenu(hMenu, WM_CUT, 0);
    return;
}

/// <summary>
/// Subclass hook to alter edit control menu and handle additional functionality
/// </summary>
LRESULT CALLBACK EditSubclassProc(HWND hWndEdit, UINT Msg, WPARAM wParam, LPARAM lParam, UINT_PTR uIDSubclass, DWORD_PTR dwRefData)
{
    LRESULT ret{};

    switch (Msg)
    {
    case WM_CONTEXTMENU:
    {
        HWINEVENTHOOK hEventHook = SetWinEventHook(EVENT_SYSTEM_MENUPOPUPSTART, EVENT_SYSTEM_MENUPOPUPSTART, 0,
            ContextMenuProcCallback, GetCurrentProcessId(), GetCurrentThreadId(), 0);
        DefSubclassProc(hWndEdit, WM_CONTEXTMENU, wParam, lParam);
        UnhookWinEvent(hEventHook);
        break;
    } 
    case ID_REGDUMP:
        LOG << "\r\n-------CTX webcam registry------\r\n"
            << dumpHdxRealTime()
            << "-----------------------------------------------\r\n";
        break;
    case ID_CLEAR:
        SetWindowText(hWndEdit, 0);
        
        break;
    default:
    {
        ret = DefSubclassProc(hWndEdit, Msg, wParam, lParam);
    }
    break;
    }

    return ret;
}


INT_PTR CALLBACK DlgProc(HWND hwnd, UINT msg, WPARAM wParam, LPARAM lParam)
{
    switch (msg)
    {
    case WM_INITDIALOG:
    {
        RECT DesktopRect{};
        RECT DialogRect{};
        GetWindowRect(GetDesktopWindow(), &DesktopRect);
        GetWindowRect(hwnd, &DialogRect);
        SetWindowPos(hwnd, HWND_TOP, (DesktopRect.right - DialogRect.right) / 2,
            (DesktopRect.bottom - DialogRect.bottom) / 2, 0, 0, SWP_NOSIZE);
        auto editControl = GetDlgItem(hwnd, IDC_EDIT_INFO);
        SetWindowSubclass(editControl, EditSubclassProc, 0, NULL);
        return TRUE;
    }
    case WM_SHOWWINDOW:
        for (auto& item : controlsToResize)
            item.SetRect();
        return TRUE;
    case WM_CLOSE:
        if(webCamPreview)
            webCamPreview->Stop();
        DestroyWindow(hwnd);
        return 0;
    case WM_CREATE:
    {        
        
        return TRUE;
    }
    case WM_DESTROY:
        PostQuitMessage(0);
        return FALSE;
    case WM_PREVIEW_COMPLETE:  
    {
        webCamPreview->Stop();
        toggleControls(true);
        return TRUE;
    }
    case WM_ENTERSIZEMOVE:
    {
        for (auto& item : controlsToResize)
            item.SetRect();
        return TRUE;
    }
    case WM_SIZE:
    {
        auto getControlElement = [](int ID) {return std::find_if(controlsToResize.begin(), controlsToResize.end(), [&ID](const MoveResizeControl& el) {return el.ID == ID; }); };
        getControlElement(IDC_BUTTON_PREVIEW)->move();
        getControlElement(IDC_EDIT_INFO)->stretch();
        getControlElement(IDC_LIST_RES)->stretch();
        getControlElement(IDC_LOG_PROP)->stretch();
        return TRUE;
    }
    case WM_COMMAND:
    {
        switch (LOWORD(wParam))
        {
       
        case IDCANCEL:
        {
            EndDialog(hwnd, LOWORD(wParam));
            return TRUE;
        }
        case IDC_BUTTON_PREVIEW:
        {
            if (StartPreview())
                toggleControls(false);
            return TRUE;
        }
        case IDC_RADIO_DS:
        {
            if (IsDlgButtonChecked(hwnd, IDC_RADIO_DS) != BST_CHECKED)
            {
                CheckRadioButton(hwnd, IDC_RADIO_DS, IDC_RADIO_MF, IDC_RADIO_DS);
                SelectEngine(RenderEngine::DirectShow);
            }

            return TRUE;
        }
        case IDC_RADIO_MF:
        {
            if (IsDlgButtonChecked(hwnd, IDC_RADIO_MF) != BST_CHECKED)
            {
                CheckRadioButton(hwnd, IDC_RADIO_DS, IDC_RADIO_MF, IDC_RADIO_MF);
                SelectEngine(RenderEngine::MediaFoundation);
            }
            return TRUE;
        }
        case IDC_CLEAR:
        {
            HWND hWndEdit = GetDlgItem(hwnd, IDC_EDIT_INFO);
            SetWindowText(hWndEdit, 0);
            return TRUE;
        }
        case IDC_REGDUMP:
        {
            LOG << "\r\n-------CTX webcam registry------\r\n"
                << dumpHdxRealTime()
                << "-----------------------------------------------\r\n";
            return TRUE;
        }

        case IDC_LIST_CAMS:
        {
            switch (HIWORD(wParam))
            {
            case LBN_SELCHANGE:
            {
                HWND hwndList = GetDlgItem(hwnd, IDC_LIST_CAMS);
                auto lbItem = (ULONG)SendMessage(hwndList, LB_GETCURSEL, 0, 0);                
                SelectWebCam(lbItem);
                return TRUE;
            }
            default:
                return TRUE;
            }
        }
        case IDC_LIST_RES:
        {
            switch (HIWORD(wParam))
            {
            case LBN_SELCHANGE:
            {
                HWND hwndList = GetDlgItem(hwnd, IDC_LIST_RES);
                auto lbItem = (ULONG)SendMessage(hwndList, LB_GETCURSEL, 0, 0);
                auto resIndex = (ULONG)SendMessage(hwndList, LB_GETITEMDATA, lbItem, 0);
                webCamPreview->ResIndex() = resIndex;
                return TRUE;
            }
            default:
                return TRUE;
            }
        }
        default:
            break;
        }
        break;
    }
    }
    return FALSE;
}


int APIENTRY wWinMain(_In_ HINSTANCE hInstance, _In_opt_ HINSTANCE hPrevInstance, _In_ LPWSTR    lpCmdLine, _In_ int nCmdShow)
{
    UNREFERENCED_PARAMETER(hPrevInstance);
    UNREFERENCED_PARAMETER(lpCmdLine);

    LoadStringW(hInstance, IDS_APP_TITLE, szTitle, MAX_LOADSTRING);
    LoadStringW(hInstance, IDC_WEBCAMPREVIEWUI, szWindowClass, MAX_LOADSTRING);

    hDlg = CreateDialog(hInstance, MAKEINTRESOURCE(IDD_DIALOG_MAIN), nullptr, DlgProc);
    ShowWindow(hDlg, SW_SHOW);
    UpdateWindow(hDlg);

    MSG msg;
    while (GetMessage(&msg, nullptr, 0, 0))
    {
        if (!IsDialogMessage(hDlg, &msg))
        {
            TranslateMessage(&msg);
            DispatchMessage(&msg);
        }
    }
    return 0;
}
